#include <asm/io.h>
#include <asm/cmos.h>
#include <asm/system.h>
#include <cnix/driver.h>
#include <cnix/kernel.h>

#define TIMER_MODE	0x43
#define TIMER0		0x40

#define TIMER_FREQ	1193182L
#define HZ		100
#define LATCH		(TIMER_FREQ / HZ)

#define LATCH_COUNT	0x00
#define TIMER_COUNT	((unsigned)(TIMER_FREQ / HZ))

struct bios_time{
	int second;
	int minute;
	int hour;
	int day;
	int month;
	int year;
	int century;
};

/* fn_t is defined in sched.h. */

extern int put_irq_handler(int irq, fn_t fn);
extern void schedule(void);

long volatile timer_count;
long volatile boot_time;

struct bios_time bios_time;

/* do some computing, and ..., struct regs_t */
static void do_timer(void)
{
	timer_count++;		

	/* Now do something really useful. */
	if(current != &(init_task.task)){
		if(current->counter)
			current->counter--;
		else{
			current->need_sched = 1;
		}
	}else{
		//printk("Idle is current!\n");
		if(run_queue->next != run_queue){
		//	printk("force to schedule!\n");
			/* It's so odd, it will be useless in kernel */
			current->need_sched = 1;
		}
	}
}

#define MINUTE	60
#define HOUR	(MINUTE * 60)
#define DAY	(HOUR * 24)
#define YEAR	(DAY * 365)

/* the months of leap year */
static int month[12] = {
	0,
	DAY * (31),
	DAY * (31 + 29),
	DAY * (31 + 29 + 31),
	DAY * (31 + 29 + 31 + 30),
	DAY * (31 + 29 + 31 + 30 + 31),
	DAY * (31 + 29 + 31 + 30 + 31 + 30),
	DAY * (31 + 29 + 31 + 30 + 31 + 30 + 31),
	DAY * (31 + 29 + 31 + 30 + 31 + 30 + 31 + 31),
	DAY * (31 + 29 + 31 + 30 + 31 + 30 + 31 + 31 + 30),
	DAY * (31 + 29 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31),
	DAY * (31 + 29 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31 + 30),
};

static long time_to_second(struct bios_time * time)
{
	long ret;
	int year;

	/* this century is 1 less than what we offen think of */
	year = time->year + time->century * 100 - 1970;

	/* 1972 % 4 == 0, 1973 - 1970 == 3, 3 + 1 == 4 */
	ret = year * YEAR + DAY * ((year + 1) / 4);	

	ret += month[time->month - 1];
	/* if it's not leap year */
	if(time->month > 2 && ((year + 2) % 4))
		ret -= DAY;
	ret += DAY * (time->day - 1);
	ret += HOUR * time->hour;
	ret += MINUTE * time->minute;
	ret += time->second;

	return ret;
}

void m_start(struct m_state * ms)
{
	ms->prev = 0;
	ms->now = 0;
}

/* delay time may be wrong with real time, but not at all. */
unsigned int m_elapsed(struct m_state * ms)
{
	unsigned int count;

	outb(LATCH_COUNT, TIMER_MODE);
	
	count = inb(TIMER0);
	count |= inb(TIMER0) << 8;
	
	/* count is decreasing, in two conditions count will be bigger than 
	 * ms->prev, 1 and 2 will set prev to appropriate value.
	 * 1: at first, ms->prev is 0
	 * 2: when accross 65535 to 0 
	 */
	ms->now += (count <= ms->prev ? (ms->prev - count) : 1);
	ms->prev = count;

	return (ms->now / (TIMER_FREQ / 1000));
}

void m_delay(unsigned int msec)
{
	struct m_state ms;

	ms.now = ms.prev = 0;

	while(m_elapsed(&ms) < msec);
}

void timer_init(void)
{
	bios_time.second = CMOS_READ(RTC_SECOND);
	bios_time.minute = CMOS_READ(RTC_MINUTE);
	bios_time.hour = CMOS_READ(RTC_HOUR);
	bios_time.day = CMOS_READ(RTC_DAY_OF_MONTH);
	bios_time.month = CMOS_READ(RTC_MONTH);
	bios_time.year = CMOS_READ(RTC_YEAR);
	bios_time.century = CMOS_READ(RTC_CENTURY);
	BCD_TO_BIN(bios_time.second);	BCD_TO_BIN(bios_time.minute);
	BCD_TO_BIN(bios_time.hour);	BCD_TO_BIN(bios_time.day);
	BCD_TO_BIN(bios_time.month);	BCD_TO_BIN(bios_time.year);
	BCD_TO_BIN(bios_time.century);

	boot_time = time_to_second(&bios_time);

	printk("Boot time: %u\n", boot_time);
	printk("Century %d Year %d Month %d Day %d Time: %d : %d : %d\n", 
			bios_time.century,
			bios_time.year, bios_time.month,
			bios_time.day, bios_time.hour, 
			bios_time.minute, bios_time.second);

	timer_count = 0;		/* initilize */

  	/* set timer rate */
	outb(0x34, TIMER_MODE);		/* binary, mode 2, LSB/MSB, ch 0 */
	outb(LATCH & 0xff, TIMER0);	/* LSB */
	outb(LATCH >> 8, TIMER0);	/* MSB */

	put_irq_handler(0x00, (fn_t)&do_timer);
}
