#include <asm/io.h>
#include <asm/system.h>
#include <cnix/wait.h>
#include <cnix/partition.h>
#include <cnix/fs.h>
#include <cnix/driver.h>
#include <cnix/kernel.h>

#define ide_io_read(port,buf,nr) \
__asm__("cld;rep;insw"::"d" (port),"D" (buf),"c" (nr))

#define ide_io_write(port,buf,nr) \
__asm__("cld;rep;outsw"::"d" (port),"S" (buf),"c" (nr))

#define ATOMIC_SET(var, value)\
	do{\
		unsigned int flags;\
		save_flags(flags);\
		cli();\
		var = value;\
		restore_flags(flags);\
	}while(0)

static unsigned int cyl;
static unsigned char head;
static unsigned int sec;
static unsigned int precomp;

#define MAX_ERRS	3
static int need_reset = 0, errs = 0;

static struct dev_info{
	int flag;
	unsigned int start_sec;
	unsigned int num_sec;
}dev_info[4];
		
static struct request{
	int sec;
	int nsec;
	int rw_mode;
	struct request * next;
	struct request * prev;
	struct buf_head * buf;
}reqq[REQQ_SIZE];

static struct request * cur_req = NULL, * free_req;
static struct wait_queue * req_wait = NULL;

struct systypes{
    unsigned char type;
    char *name;
}sys_types[] = {
    {0, "Empty"},
    {1, "DOS 12-bit FAT"},		/* Primary DOS with 12-bit FAT */
    {2, "XENIX /"},			/* XENIX / filesystem */
    {3, "XENIX /usr"},			/* XENIX /usr filesystem */
    {4, "DOS 16-bit FAT <32M"},		/* Primary DOS with 16-bit FAT */
    {5, "DOS Extended"},		/* DOS 3.3+ extended partition */
    {6, "DOS 16-bit FAT >=32M"},
    {7, "OS/2 IFS (e.g., HPFS) or NTFS or QNX2 or Advanced UNIX"},
    {8, "AIX boot or SplitDrive"},
    {9, "AIX data or Coherent"},
    {0x0a, "OS/2 Boot Manager or Coherent swap"},
    {0x0b, "Windows FAT32"},
    {0x0c, "Windows FAT32 (lba)"},
    {0x0d, "Windows FAT16(lba)"},
    {0x0e, "DOS (16-bit FAT), CHS-mapped"},
    {0x0f, "Ext. partition, CHS-mapped"},
    {0x10, "OPUS"},
    {0x11, "OS/2 BM: hidden DOS 12-bit FAT"},
    {0x12, "Compaq diagnostics"},
    {0x14, "OS/2 BM: hidden DOS 16-bit FAT <32M"},
    {0x16, "OS/2 BM: hidden DOS 16-bit FAT >=32M"},
    {0x17, "OS/2 BM: hidden IFS"},
    {0x18, "AST Windows swapfile"},
    {0x24, "NEC DOS"},
    {0x3c, "PartitionMagic recovery"},
    {0x40, "Venix 80286"},
    {0x41, "Linux/MINIX (sharing disk with DRDOS)"},
    {0x42, "SFS or Linux swap (sharing disk with DRDOS)"},
    {0x43, "Linux native (sharing disk with DRDOS)"},
    {0x50, "DM (disk manager)"},
    {0x51, "DM6 Aux1 (or Novell)"},
    {0x52, "CP/M or Microport SysV/AT"},
    {0x53, "DM6 Aux3"},
    {0x54, "DM6"},
    {0x55, "EZ-Drive (disk manager)"},
    {0x56, "Golden Bow (disk manager)"},
    {0x5c, "Priam Edisk (disk manager)"}, /* according to S. Widlake */
    {0x61, "SpeedStor"},
    {0x63, "GNU HURD or Mach or Sys V/386 (such as ISC UNIX)"},
    {0x64, "Novell Netware 286"},
    {0x65, "Novell Netware 386"},
    {0x70, "DiskSecure Multi-Boot"},
    {0x75, "PC/IX"},
    {0x77, "QNX4.x"},
    {0x78, "QNX4.x 2nd part"},
    {0x79, "QNX4.x 3rd part"},
    {0x80, "MINIX until 1.4a"},
    {0x81, "MINIX since 1.4b, early Linux, Mitac dmgr"},
    {0x82, "Linux swap"},
    {0x83, "Linux native"},
    {0x84, "OS/2 hidden C: drive"},
    {0x85, "Linux extended"},
    {0x86, "NTFS volume set??"},
    {0x87, "NTFS volume set??"},
    {0x93, "Amoeba"},
    {0x94, "Amoeba BBT"},		/* (bad block table) */
    {0xa0, "IBM Thinkpad hibernation"}, /* according to dan@fch.wimsey.bc.ca */
    {0xa5, "BSD/386"},			/* 386BSD */
    {0xa7, "NeXTSTEP 486"},
    {0xb7, "BSDI fs"},
    {0xb8, "BSDI swap"},
    {0xc1, "DRDOS/sec (FAT-12)"},
    {0xc4, "DRDOS/sec (FAT-16, < 32M)"},
    {0xc6, "DRDOS/sec (FAT-16, >= 32M)"},
    {0xc7, "Syrinx"},
    {0xdb, "CP/M or Concurrent CP/M or Concurrent DOS or CTOS"},
    {0xe1, "DOS access or SpeedStor 12-bit FAT extended partition"},
    {0xe3, "DOS R/O or SpeedStor"},
    {0xe4, "SpeedStor 16-bit FAT extended partition < 1024 cyl."},
    {0xf1, "SpeedStor"},
    {0xf2, "DOS 3.3+ secondary"},
    {0xf4, "SpeedStor large partition"},
    {0xfe, "SpeedStor >1024 cyl. or LANstep"},
    {0xff, "Xenix Bad Block Table"}
};

static int ide_waitfor(int, int);
static int finish_cmd(void);
static int ide_reset(void);
static void ide_status_ok(void);
static int free_cur_req(void);
static void ide_intr(void);
static int ide_cmd(int, int, unsigned int, unsigned int);
static struct request * get_req(void);
static void get_dev_info(void);

void ide_init(void);
int ll_rw_block(int, struct buf_head *);

static int ide_waitfor(int mask, int value)
{
	struct m_state ms;

	m_start(&ms);
	do{
		if((inb(IDE_STATUS) & mask) == value)
			return 1;
	}while(m_elapsed(&ms) < TIMEOUT);

	ATOMIC_SET(need_reset, 1);
	return 0;
}

/* for serious fault */
static int ide_reset(void)
{
	m_delay(500);

	outb(RESET_CTL, IDE_CTL);
	m_delay(1);
	outb(0, IDE_CTL);
	m_delay(1);

	if(!ide_waitfor(BUSY_STAT | READY_STAT, READY_STAT))
		return 0;
	
	ATOMIC_SET(need_reset, 0);
	return 1;
}

static void ide_status_ok(void)
{
      while(inb(IDE_STATUS) & BUSY_STAT){
	    m_delay(50);
      }
}

static int free_cur_req(void)
{
	struct request * tmp;

	tmp = cur_req;

	if(cur_req->next == cur_req){
		cur_req = NULL;

		/* insert tmp into free_req */
		tmp->next = NULL;
		if(NULL == free_req){
			free_req = tmp;
			wakeup(&req_wait);	
		}else{
			tmp->next = free_req;
			free_req = tmp;
		}

		return 0; /* say there are nothing left */
	}else{
		cur_req->prev->next = cur_req->next;
		cur_req->next->prev = cur_req->prev;

		cur_req = cur_req->next;
		
		/* insert tmp into free_req */
		tmp->next = NULL;
		if(NULL == free_req){
			free_req = tmp;
			wakeup(&req_wait);	
		}else{
			tmp->next = free_req;
			free_req = tmp;
		}

		return 1; /* say there are still some work to do */
	}		
}

/* set one watchdog to see interrupt have happend or not in time is better */
void ide_intr(void)
{
	while(inb(IDE_STATUS) & BUSY_STAT){ printk("looping in ide_intr\n"); }

	if(NULL == cur_req)
		panic("cur_req is NULL in ide_intr!!!\n");

	if(cur_req->rw_mode == READ){
		ide_io_read(IDE_DATA, 
			&(cur_req->buf->b_data[(2 - cur_req->nsec) * 512]), 
				256);
	}

	cur_req->nsec--;

	/* need to see if it is the end, and reset hard disk */
	if(cur_req->nsec == 0){
		/* 
		 * here is in intr handler, so i think no other process will run
		 */
		cur_req->buf->b_flags |= B_DONE;
		wakeup((struct wait_queue **)&(cur_req->buf->b_wait));

		if(!free_cur_req())
			return;
		
		finish_cmd();
	}
}

static int ide_cmd(int drive, int cmd, unsigned int sec, unsigned int nsec)
{
	int c, h, s;

	s = sec & 0xff;
	c = (sec & 0xffff00) >> 8;
	h = (sec & 0xf000000) >> 24;

	if((inb(IDE_STATUS) & BUSY_STAT) != 0 || !ide_waitfor(BUSY_STAT, 0)){
		printk("ide controller is not ready!\n");
			return 0;
	}
	
	outb(0xe0 | (drive << 4) | h, IDE_CURRENT);

	if((inb(IDE_STATUS) & BUSY_STAT) != 0 || !ide_waitfor(BUSY_STAT, 0)){
		printk("ide controller is not ready!\n");
			return 0;
	}
	
	outb(head >= 8 ? 8 : 0, IDE_CTL);
	outb(precomp, IDE_PRECOMP);

     	outb(nsec, IDE_NSECTOR);
	outb(s, IDE_SECTOR);
	outb(c, IDE_LCYL);
	outb(c >> 8, IDE_HCYL);
	
	outb(cmd, IDE_CMD);

	return 1;
}

/* I think if here happen some erorr, then process who call ll_rw_block
 * will can't be wakedup, assume will not have interrupt happen.
 */
static int finish_cmd(void)
{
	if(NULL == cur_req)
		panic("cur_req is NULL in finish_cmd!!!\n");

	if(cur_req->rw_mode == READ){
		if(!ide_cmd(0, READ_CMD, cur_req->sec, cur_req->nsec)){
			errs++;
			while(errs < MAX_ERRS){
				if(need_reset)
					if(!ide_reset())
						panic("ide_reset is useless!\n");
				if(ide_cmd(0, READ_CMD, 
					cur_req->sec, cur_req->nsec))
					break;

				errs++;
			}

			if(errs == MAX_ERRS){
				/* panic will be better, now we can't get back
				 * cur_req and buf, and we can't wakeup the 
				 * process.
				 */
				printk("So many times for error in finish_cmd!\n");

				cur_req->buf->b_flags |= B_ERROR;
				return 0;	
			}

			errs = 0;
		}
	}else if(cur_req->rw_mode == WRITE){
		if(!ide_cmd(0, WRITE_CMD, cur_req->sec, cur_req->nsec)){
			errs++;
			while(errs < MAX_ERRS){
				if(need_reset)
					if(!ide_reset())
						panic("ide_reset is useless!\n");
				if(ide_cmd(0, WRITE_CMD, 
					cur_req->sec, cur_req->nsec))
					break;

				errs++;
			}

			if(errs == MAX_ERRS){
				/* panic will be better, now we can't get back
				 * cur_req and buf, and we can't wakeup the 
				 * process.
				 */
				printk("So many times for error in finish_cmd!\n");

				cur_req->buf->b_flags |= B_ERROR;
				return 0;	
			}

			errs = 0;
		}

		ide_waitfor(BUSY_STAT, 0);
		while(inb(IDE_STATUS) & BUSY_STAT){ printk("looping in ide_intr\n"); }
		ide_io_write(IDE_DATA, cur_req->buf->b_data, 512);
	}else{
		panic("rw_mode %d out of value...\n", cur_req->rw_mode);
	}

	return 1;	
}

static struct request * get_req(void)
{
	unsigned long flags;
	struct request * req;

	save_flags(flags);
	cli();

go_on:
	if(free_req){
		req = free_req;		
		free_req = free_req->next;
		req->next = req;
		req->prev = req;
	}else{
		restore_flags(flags);
		
		/* wake up in ide_intr */
		sleep_on(&req_wait);
		goto go_on;
	}
		
	restore_flags(flags);

	return req;
}

int ll_rw_block(int rw_mode, struct buf_head * buf)
{
	unsigned long flags;
	struct request * req, * rptr;

	if(buf->b_dev < 0 || buf->b_dev > 3 || !dev_info[buf->b_dev].flag){
		buf->b_flags |= B_ERROR;
		printk("bad device ...!\n");
		return 0;
	}

	if((dev_info[buf->b_dev].num_sec / 2 - 1) < buf->b_blocknr){
		buf->b_flags = B_ERROR;
		printk("bad block ...!\n");
		return 0;
	}

	req = get_req();
	req->sec = dev_info[buf->b_dev].start_sec + buf->b_blocknr * 2;
	req->nsec = 2; /* always 2 */
	req->rw_mode = rw_mode;
	req->buf = buf;

	/* I think only to disable_irq(14) will be satisfied */
	save_flags(flags);
	cli();

	/* the queue is a double-linked ring, cur_req points to the current
	 * request being done with.
	 */
	if(NULL == cur_req){
		cur_req = req;	
		restore_flags(flags);
	}else{
		if(cur_req->next == cur_req){
			cur_req->prev = req;
			cur_req->next = req;
			req->prev = cur_req;
			req->next = cur_req;
		}else{
			rptr = cur_req;

			while(rptr->next != cur_req){
				/* rptr->next->sec >= here '=' will ... */
				if(rptr->sec < req->sec && 
					rptr->next->sec >= req->sec) 
					break;
				
				rptr = rptr->next;
			}
			
			/* Now insert req at the next position of rptr
			 */
			rptr->next->prev = req;
			req->next = rptr->next;
			req->prev = rptr;
			rptr->next = req;
		}

		/* allow interrupt again */
		restore_flags(flags);

		return 1;
	}

	if(cur_req->rw_mode == READ){
		if(!ide_cmd(0, READ_CMD, cur_req->sec, cur_req->nsec)){
			errs++;
			while(errs < MAX_ERRS){
				if(need_reset)
					if(!ide_reset())
						panic("ide_reset is useless!\n");
				if(ide_cmd(0, READ_CMD, 
					cur_req->sec, cur_req->nsec))
					break;

				errs++;
			}

			if(errs == MAX_ERRS){
				/* panic will be better, now we can't get back
				 * cur_req and make other things mess.
				 */
				printk("So many times for error in ll_rw_block!\n");
				cur_req->buf->b_flags |= B_ERROR;
				return 0;
			}

			errs = 0;
		}
	}else if(cur_req->rw_mode == WRITE){
		if(!ide_cmd(0, WRITE_CMD, cur_req->sec, cur_req->nsec)){
			errs++;
			while(errs < MAX_ERRS){
				if(need_reset)
					if(!ide_reset())
						panic("ide_reset is useless!\n");
				if(ide_cmd(0, WRITE_CMD, 
					cur_req->sec, cur_req->nsec))
					break;

				errs++;
			}

			if(errs == MAX_ERRS){
				/* panic will be better, now we can't get back
				 * cur_req and make other things mess.
				 */
				printk("So many times for error in ll_rw_block!\n");

				cur_req->buf->b_flags |= B_ERROR;
				return 0;
			}
			
			errs = 0;
		}

		ide_waitfor(BUSY_STAT, 0);
		while(inb(IDE_STATUS) & BUSY_STAT){ printk("looping in ide_intr\n"); }
		ide_io_write(IDE_DATA, cur_req->buf->b_data, 512);
	}else{
		panic("rw_mode %d out of value...\n", cur_req->rw_mode);
	}

	return 1;
}

static void get_dev_info(void)
{
	int i;
	unsigned char id[512]; 
        unsigned short *ver_add;
	unsigned char *disk_add;
	unsigned int lba_capacity;
	unsigned int ver_cs, ver_offset, ver_line;
	struct partition * par_ptr;

        outb(0xa0, IDE_CURRENT);
        outb(0xec, IDE_STATUS);
      
        ide_status_ok();

	ide_io_read(IDE_DATA, id, 256);

#define CAPACITY_OFF	120
	
	lba_capacity = *((unsigned int *)&id[CAPACITY_OFF]);
	
	/* bios C/H/S */
	ver_add = (unsigned short *)0x104;
	ver_offset = ver_add[0];
	ver_cs = ver_add[1];
	ver_line= ver_cs * 0x10 + ver_offset;
	disk_add = (unsigned char *)ver_line; 
	
	cyl =  *((unsigned short *)&disk_add[0]);
	head = disk_add[2];
	sec = disk_add[14];
	precomp = (*(unsigned short *)((&disk_add[5]))) >> 2;

	/* lba mode */
        cyl = lba_capacity / (head * sec) ;

	printk("Your disk C:%d, H:%d, S:%d, PRECOMP:%d, LBA:%d\n", 
			cyl, head, sec, precomp, lba_capacity);

	ide_cmd(0, READ_CMD, 0, 1);

	ide_status_ok();
	
	ide_io_read(IDE_DATA, id, 256);

	printk("Partition info:\n");
	par_ptr = (struct partition *)&id[0x1be];
	for(i = 0; i < 4; i++, par_ptr++){
		if(par_ptr->sys_ind == 0){
			dev_info[i].flag = 0;
			continue;
		}

		dev_info[i].flag = 1;
		dev_info[i].start_sec = par_ptr->start_sec;
		dev_info[i].num_sec = par_ptr->num_sec;

		printk("%d: %x %x %dM\n", i, par_ptr->sys_ind,
			par_ptr->start_sec, par_ptr->num_sec / 2048);
	}
}

void ide_init(void)
{
	int i;

	/* when in free_req, only use next field, and the field next of last one
	 * will be NULL.
	 */
	for(i = 0; i < REQQ_SIZE - 1; i++)
		reqq[i].next = &reqq[i + 1];
	reqq[REQQ_SIZE - 1].next = NULL;

	free_req = reqq;
	
	/* outb only for not causing intr in vmware */
	//outb(inb(0x21) | 0x04, 0x21);
	outb(inb(0xa1) | 0x40, 0xa1);
	get_dev_info();
	//outb(inb(0x21) & 0xfb, 0x21);
	outb(inb(0xa1) & 0xbf, 0xa1);
	put_irq_handler(0x0e, &ide_intr);
}
