#include <errno.h>
#include <string.h>
#include <asm/regs.h>
#include <cnix/elf.h>
#include <cnix/fs.h>
#include <cnix/mm.h>
#include <cnix/kernel.h>

#define invalidate() \
__asm__("movl %%eax, %%cr3"::"a"(current->pg_dir))

static unsigned long copy_param(char * filename, char ** argv, char ** envp,
	int * argc_ptr, int * argv_len_ptr, int * envc_ptr, int * envp_len_ptr)
{
	unsigned long param_page;
	int i, j, k, argc, argv_len, envc, envp_len;
	char ** argvptr, ** envpptr, * pg_ptr;

	if(!(param_page = get_one_page()))
		return 0;

	pg_ptr = (char *)param_page;

	argc = 0;
	argv_len = 0;
	argvptr = NULL;
	if(argv != NULL){
		/* 
		 * Note if argv end without NULL, if some error, how to free 
		 * param_page ??? so this code must be placed before, or add 
		 * some code to check the area.
		 */
		for(i = 0; argv[i]; i++)
			argv_len += strlen(argv[i]) + 1;

		argc = i;
		
		/* align them, and plus for NULL pointer at tail */
		argv_len = ((argv_len + 3) & (~3)) + 4;

		argvptr = (char **)(param_page + 12);
		for(j = 0, k = 12 + ((i + 1) * 4); j < i; j++){
			/* check out of loop */
			/* 
			if((k + strlen(argv[j]) + 1) > PAGE_SIZE){ 
			 	free_one_page(param_page); 
				return 0; 
			} 
			*/

			*((int *)&argvptr[j]) = k;
			strcpy(&pg_ptr[k], argv[j]);
			
			k += strlen(argv[j]) + 1;
		}

		argvptr[j] = NULL;
	}

	envc = 0;
	envpptr = NULL;
	envp_len = 0;
	if(envp != NULL){
		/* 
		 * Note if argv end without NULL, if some error, how to free 
		 * param_page ??? so this code must be placed before, or add 
		 * some code to check the area.
		 */
		for(i = 0; envp[i]; i++)
			envp_len += strlen(envp[i]) + 1;

		envc = i;
		
		/* align them */
		envp_len = ((envp_len + 3) & (~3)) + 4;

		envpptr = (char **)(param_page + 12 
			+ (argc + 1) * 4 + argv_len);
		for(j = 0, k = 12 + (argc + 1) * 4 + argv_len + (i + 1) * 4;
			j < i; j++){
			/* check out of loop */
			/* 
			if((k + strlen(envp[j]) + 1) > PAGE_SIZE){ 
			 	free_one_page(param_page); 
				return 0; 
			} 
			*/

			*((int *)&envpptr[j]) = k;
			strcpy(&pg_ptr[k], envp[j]);

			k += strlen(envp[j]) + 1;
		}

		envpptr[j] = NULL;
	}

	*argc_ptr = argc;
	*argv_len_ptr = argv_len;
	*envc_ptr = envc;
	*envp_len_ptr = envp_len;

	return param_page;
}

static void adjust_param(unsigned long stack, unsigned long param_page,
	int argc, int argv_len, int envc, int envp_len)
{
	int i;
	char ** stackptr;

	stackptr = (char **)(param_page + 12);
	for(i = 0; i < argc; i++)
		stackptr[i] = (char *)(stack - PAGE_SIZE + 
			*((int *)&stackptr[i]));

	stackptr = (char **)(param_page + 12 + (argc + 1) * 4 + argv_len);
	for(i = 0; i < envc; i++)
		stackptr[i] = (char *)(stack - PAGE_SIZE +
			*((int *)&stackptr[i]));

	stackptr = (char **)param_page;
	
	*((int *)&stackptr[0]) = argc;
	stackptr[1] = (char *)(stack - PAGE_SIZE + 12);
	stackptr[2] = (char *)(stack - PAGE_SIZE + 12 + (argc + 1) * 4 + argv_len);
}

int do_execve(char * filename, char ** argv, char ** envp, struct regs_t * regs)
{
	int i, phnum, phentsize, filesz, memsz, fd;
	unsigned char * p, * q;
	unsigned long * pg, entry, addr, stack, param_page;
	struct elfhdr * hdr;
	struct elf_phdr * phdr;

	int argc, argv_len, envc, envp_len;

	p = (unsigned char *)get_one_page();
	if(!p)
		return -EAGAIN;

	/* Note to share code page */
	/* check file name is NULL or ..., then copy it to kernel space */
	fd = open(filename, 0);
	if(fd < 0){
		free_one_page((unsigned long)p);

		return -ENFILE;
	}
	
	read(fd, p, sizeof(struct elfhdr));

	hdr = (struct elfhdr *)p;

	if(hdr->e_ident[0] != 0177 || hdr->e_ident[1] != 'E' ||
			hdr->e_ident[2] != 'L' || hdr->e_ident[3] != 'F'){
		free_one_page((unsigned long)p);
		close(fd);
			
		return -ENOEXEC;
	}

	if(hdr->e_type != 2){
		return -ENOEXEC;
		free_one_page((unsigned long)p);
		close(fd);
	}
	
	if(hdr->e_machine != 3){
		free_one_page((unsigned long)p);
		close(fd);

		return -ENOEXEC;
	}

	/* copy argv[0] ... argv[], copy envp[0] .. envp[] */
	if(!(param_page = copy_param(filename, argv, envp, &argc, &argv_len,
		&envc, &envp_len))){
		free_one_page((unsigned long)p);
		close(fd);
		
		return -EAGAIN;
	}

	free_page_tables(current->pg_dir, 16, 16);

	pg = (unsigned long *)get_one_page();

	/* assume 4M will enough */
	*((unsigned long *)(current->pg_dir + 4 * 16)) = (unsigned long)pg + 7;

	phnum = hdr->e_phnum;
	phentsize = hdr->e_phentsize;
	entry = hdr->e_entry;

	seek(fd, hdr->e_phoff, SEEK_SET);
	read(fd, p, phentsize * phnum);
	phdr = (struct elf_phdr *)p;		
	
	addr = 0;
	/* Notice!!! when phnum == 0 */
	for(i = 0; i < phnum; i++){
		if(phdr->p_type == PT_LOAD){
			seek(fd, phdr->p_offset, SEEK_SET);
		
			addr = phdr->p_vaddr & 0xfffff000;
			memsz = 0;
			if(memsz >= PAGE_SIZE)
				memsz = phdr->p_memsz 
					- (PAGE_SIZE - phdr->p_filesz % PAGE_SIZE);
			for(filesz = 0; filesz < phdr->p_filesz; 
					filesz += PAGE_SIZE){
				q = (unsigned char *)get_one_page();
				/* free other pages, but now random */
				if(!q)
					return -EAGAIN;

				/* if error ..., or read 0 byte */
				read(fd, q, PAGE_SIZE);

				if(phdr->p_flags == (PF_R | PF_X))
					pg[(addr >> 12) & 0x000001ff] = (unsigned long)q + 5;
				else /* if(phdr->p_flags == (PF_R | PF_W)) */
					pg[(addr >> 12) & 0x000001ff] = (unsigned long)q + 7;
				
				addr += PAGE_SIZE;	
			}

			for(; memsz > 0; memsz -= PAGE_SIZE){
				q = (unsigned char *)get_one_page();
				/* free other pages, but now random */
				if(!q)
					return -EAGAIN;

				if(phdr->p_flags == (PF_R | PF_X))
					pg[(addr >> 12) & 0x000001ff] = (unsigned long)q + 5;
				else /* if(phdr->p_flags == (PF_R | PF_W)) */
					pg[(addr >> 12) & 0x000001ff] = (unsigned long)q + 7;

				addr += PAGE_SIZE;
			}
		}

		phdr++;
	}

#define USER_ADDR	0x4000000
	/* if only got one page, then get another page as stack page! */
	if(addr - USER_ADDR <= PAGE_SIZE){
		stack = get_one_page();
		/* free other pages, but now random */
		if(!stack)
			return -EAGAIN;
		pg[(addr >> 12) & 0x000001ff] = (unsigned long)stack + 7;
		addr += PAGE_SIZE;
	}

	stack = addr + PAGE_SIZE;
	adjust_param(stack, param_page, argc, argv_len, envc, envp_len);

	/* for parameter and enviroment */
	pg[(addr >> 12) & 0x000001ff] = (unsigned long)param_page + 7;

	invalidate();

	/* ... from here... */
	regs->cs = 0x1b;
	regs->eip = entry;
	regs->ds = 0x23;
	regs->es = 0x23;
	regs->ss = 0x23;
	regs->esp = stack - PAGE_SIZE - 4;

	close(fd);

	free_one_page((unsigned long)p);

	return 0;
}

/* After exit, its item in task[] will be release by its father */
int do_exit(int exitcode)
{
	int i, pid, ppid;
	struct task_struct * p, * pp;

	for(i = 0; i < FILE_DESC_NUM; i++)
		close(i);

	__asm__("movl %%eax, %%cr3"::"a"((unsigned long)kp_dir));

	pid = current->pid;
	p = task[pid];

	ppid = current->ppid;
	pp = task[ppid];
	pp->exit_child_pid = pid;

	del_run(p);

	current->state = TASK_ZOMBIE;
	current->exit_code = exitcode;

	free_page_tables(p->pg_dir, 0, 32);
	free_one_page(p->pg_dir);
	
	wakeup(&pp->wait_for_child);

	schedule();

	return 0;
}

/* this is not standard, will be modified in future */
#define ENOCHILD	100	
int do_wait(int * status)
{
	int pid;
	struct task_struct * p;
	
	if(current->child_num == 0)
		return -ENOCHILD;

	if(current->exit_child_pid == 0)
		sleep_on(&current->wait_for_child);	

	current->child_num--;

	pid = current->exit_child_pid;
	current->exit_child_pid = 0;
	p = task[pid];

	*status = p->exit_code;

	task[pid] = NULL;
	free_one_page((unsigned long)p);
	
	return pid;
}
