#include <string.h>
#include <errno.h>
#include <cnix/fs.h>
#include <cnix/kernel.h>

#define FS_VERSION	"CNIX FILESYSTEM V1.0"
struct superblock sb;

#define INODE_NUM	(BLOCKSIZ * IBSIZE) / (sizeof(struct inode))
struct inode inodes[INODE_NUM];

#define DIRITEMNUM	(BLOCKSIZ / DIRITEMSIZE)
struct dir_item root_block[DIRITEMNUM];

#define DIR_FD		0
#define START_FD	1

/* from 0 block --> ... */
/* the bit before DATABLOCK is reserved */
unsigned char bitmap[BLOCKSIZ];

#define MAX_FILP_NUM	200
int filp_index;
struct filp filps[MAX_FILP_NUM];

void fs_init(void)
{
	int i;
	struct buf_head * bh;
	
	filp_index = 0;
	for(i = 0; i < MAX_FILP_NUM; i++)
		filps[i].filp_count = 0;

	bh = bread(DEFAULT_PART, SUPERBLOCK);
	memcpy(&sb, bh->b_data, sizeof(struct superblock));

	if(bh)
		brelse(bh);

	printk("VERSION: %s\n", sb.version);

	if(strcmp(sb.version, FS_VERSION))
		panic("Sorry for wrong version filesystem!\n");

	bh = bread(DEFAULT_PART, ROOTBLOCK);
	memcpy((char *)&root_block[0], bh->b_data, BLOCKSIZ);
	if(bh)
		brelse(bh);

	bh = bread(DEFAULT_PART, BITBLOCK);
	memcpy((char *)bitmap, bh->b_data, BLOCKSIZ);
	if(bh)
		brelse(bh);

	bh = bread(DEFAULT_PART, INODEBLOCK);
	memcpy((char *)&inodes[0], bh->b_data, BLOCKSIZ);
	if(bh)
		brelse(bh);

	bh = bread(DEFAULT_PART, INODEBLOCK + 1);
	memcpy((char *)&inodes[16], bh->b_data, BLOCKSIZ);
	if(bh)
		brelse(bh);
}

void flush_fs(void)
{
	struct buf_head * bh;
	
	bh = bread(DEFAULT_PART, SUPERBLOCK);
	memcpy(bh->b_data, &sb, sizeof(struct superblock));
	
	bwrite(bh);
	if(bh)
		brelse(bh);
	
	bh = bread(DEFAULT_PART, ROOTBLOCK);
	memcpy(bh->b_data, (char *)&root_block[0], BLOCKSIZ);

	bwrite(bh);
	if(bh)
		brelse(bh);

	bh = bread(DEFAULT_PART, BITBLOCK);
	memcpy(bh->b_data, (char *)bitmap, BLOCKSIZ);
	
	bwrite(bh);
	if(bh)
		brelse(bh);

	bh = bread(DEFAULT_PART, INODEBLOCK);
	memcpy(bh->b_data, (char *)&inodes[0], BLOCKSIZ);
	bwrite(bh);
	if(bh)
		brelse(bh);

	bh = bread(DEFAULT_PART, INODEBLOCK + 1);
	memcpy(bh->b_data, (char *)&inodes[16], BLOCKSIZ);

	bwrite(bh);
	if(bh)
		brelse(bh);
}

struct filp * get_filp(void)
{
	int i;
	struct filp * flptr;
	
	flptr = NULL;
	for(i = 0; i < MAX_FILP_NUM; i++){
		flptr = &filps[i];

		if(flptr->filp_count == 0){
			flptr->filp_count++;

			break;
		}
	}

	return flptr;
}

void free_filp(struct filp * flptr)
{
	if(flptr == NULL)
		return;

	flptr->filp_count--;
}

struct dir_item * get_diritem(void)
{
	int i;
	struct dir_item * diptr;

	diptr = NULL;
	for(i = 0; i < DIRITEMNUM; i++){
		diptr = &root_block[i];

		if(diptr->flag == FREE){
			diptr->flag = USED;
			break;
		}
	}

	return diptr;
}

struct dir_item * find_diritem(char * filename)
{
	int i;
	struct dir_item * diptr;

	diptr = NULL;
	for(i = 0; i < DIRITEMNUM; i++)
		if(root_block[i].flag == USED 
			&& strcmp(root_block[i].fname, filename) == 0){
			diptr = &root_block[i];

			break;
		}

	return diptr;
}

void free_diritem(struct dir_item * diptr)
{
	if(diptr == NULL)
		return;

	diptr->flag = FREE;
}

int get_inode(void)
{
	int i;
	struct inode * iptr;

	iptr = NULL;
	for(i = 0; i < INODE_NUM; i++){
		iptr = &inodes[i];
		if(iptr->attr == FREE){
			iptr->attr = USED;
			break;
		}
	}

	if(i == INODE_NUM)
		return -1;

	return i;
}

void free_inode(int index)
{
	struct inode * iptr;

	if(index > INODE_NUM - 1);
		return;

	iptr = &inodes[index];
	iptr->attr = FREE;
}

unsigned short get_block(void)
{
	unsigned short i;
	int index, offset;

	for(i = DATABLOCK; i < BLOCKSIZ * 8; i++){
		index = i / 8;
		offset = i % 8;
		if(!((bitmap[index] >> offset) & USED)){
			bitmap[index] |= USED << offset;
			break;
		}
	}

	if(i == BLOCKSIZ * 8)
		return 0;

	return i;
}

void free_block(unsigned short blocknr)
{
	int index, offset;

	if(blocknr > BLOCKSIZ * 8 - 1)
		return;

	index = blocknr / 8;
	offset = blocknr % 8;
	/* right ??? */
	bitmap[index] &= ~(USED << offset);
}

int open(char * filename, int mode)
{
	int i, j, inum;
	struct dir_item * diptr;
	struct filp * flptr;
	struct inode * iptr;
	struct file_desc * fdptr;

	if(mode & CREATE)
		mode |= RDWR;

	if(strcmp(filename, "tty0") == 0){
		current->tty = 0;
		
		return FILE_DESC_NUM;
	}else if(strcmp(filename, "tty1") == 0){
		current->tty = 1;

		return FILE_DESC_NUM;
	}else if(strcmp(filename, "/") == 0){
		fdptr = &current->file[DIR_FD];

		/* have been opened */
		if(fdptr->flag == USED)
			return -EAGAIN;
				
		flptr = get_filp();
		if(flptr == NULL)
			return -EAGAIN;
		flptr->filp_offset = 0;

		fdptr->flag = USED;
		fdptr->mode = mode;
		fdptr->flptr = flptr;

		return DIR_FD;	
	}

	flptr = get_filp();

	if(flptr == NULL)
		return -EAGAIN;

	for(i = START_FD; i < FILE_DESC_NUM; i++)
		if(current->file[i].flag == FREE)
			break;

	if(i == FILE_DESC_NUM){
		free_filp(flptr);

		return -EAGAIN;
	}

	diptr = find_diritem(filename);

	if(mode & CREATE && diptr == NULL){
		diptr = get_diritem();

		if(diptr == NULL){
			free_filp(flptr);

			return -EAGAIN;
		}

		inum = get_inode();
		if(inum == -1){
			free_diritem(diptr);
			free_filp(flptr);

			return -EAGAIN;
		}
		
		iptr = &inodes[inum];
		iptr->fsize = 0;	/* Initialate inode */
		/* 20 ... NOTICE !!! */
		for(j = 0; j < 20; j++)
			iptr->block[j] = 0;

		diptr->inum = inum;

		/* 
		 if(strlen(filename) > FNAME_BYTES - 1)
		 	return -ENFILE;
		*/
		strcpy(diptr->fname, filename);
	}else if(diptr != NULL){
		iptr = &inodes[diptr->inum];

		if(mode & CREATE)
			iptr->fsize = 0;
	}else{
		free_filp(flptr);

		return -ENFILE;
	}

	fdptr = &current->file[i];
	fdptr->flag = USED;
	fdptr->mode = mode;
	fdptr->flptr = flptr;

	flptr->iptr = iptr;
	flptr->filp_offset = 0;

	/* ... */
	if(mode & CREATE)
		flush_fs();

	return i;
}

int read(int fd, char * buffer, int size)
{
	struct buf_head * bh;
	struct filp * flptr;
	struct inode * iptr;
	unsigned short blocknr;
	int index, offset, fsize, end, offinblock, buffer_off, readbytes;

	int i;
	struct dir_item * diptr;
	struct dirent * dptr;
	
	if(fd > FILE_DESC_NUM - 1 || current->file[fd].flag == FREE)
		return -EBADF;

	if(fd == DIR_FD){
		flptr = current->file[fd].flptr;

		/* check whether it opened this file or not ??? */
		if(flptr->filp_count == 0)
			return -EBADF;

		if(flptr->filp_offset == DIRITEMNUM)
			return 0;
	
		diptr = NULL;
		for(i = flptr->filp_offset; i < DIRITEMNUM; i++)
			if(root_block[i].flag == USED){
				diptr = &root_block[i];

				break;
			}
		
		flptr->filp_offset = i + 1;
		if(diptr == NULL)
			return 0;

		dptr = (struct dirent *)buffer;
		dptr->d_ino = diptr->inum;
		dptr->d_off = i;
		dptr->d_reclen = strlen(diptr->fname);
		strcpy(dptr->d_name, diptr->fname);

		return 1;
	}

	flptr = current->file[fd].flptr;

	/* check whether it opened this file or not ??? */
	if(flptr->filp_count == 0)
		return -EBADF;

	/* check open mode */

	/* check buffer limit */
	/* modify head.S and exec.c */

	offset = flptr->filp_offset;

	iptr = flptr->iptr;

	fsize = iptr->fsize;
	end = offset + size;
	if(end > fsize - 1)
		end = fsize - 1;
	size = end - offset;

	if(size <= 0)
		return 0;

	readbytes = size;

	index = offset / BLOCKSIZ;
	offinblock = offset % BLOCKSIZ;

	blocknr = iptr->block[index];

	index++;

	buffer_off = 0;
	flptr->filp_offset += size;
	if(size <= BLOCKSIZ - offinblock){
		if(blocknr == 0)
			memset(buffer, '\0', size);
		else{
			bh = bread(DEFAULT_PART, blocknr);
			memcpy(buffer, (char *)&bh->b_data[offinblock], size);
			if(bh)
				brelse(bh);
		}
	}else{
		if(blocknr == 0)
			memset(buffer, '\0', BLOCKSIZ - offinblock);
		else{	
			bh = bread(DEFAULT_PART, blocknr);
			memcpy(buffer, (char *)&bh->b_data[offinblock], 
				BLOCKSIZ - offinblock);

			if(bh)
				brelse(bh);

			size -= BLOCKSIZ - offinblock;
			buffer_off += BLOCKSIZ - offinblock;
		}

		while(size > BLOCKSIZ){
			blocknr = iptr->block[index++];

			if(blocknr == 0)
				memset(&buffer[buffer_off], '\0', BLOCKSIZ);
			else{
				bh = bread(DEFAULT_PART, blocknr);
				memcpy((char *)&buffer[buffer_off], (char *)bh->b_data, BLOCKSIZ);
				if(bh)
					brelse(bh);
				size -= BLOCKSIZ;
				buffer_off += BLOCKSIZ;
			}
		}
		
		if(size > 0){
			blocknr = iptr->block[index++];

			if(blocknr == 0)
				memset(&buffer[buffer_off], '\0', size);
			else{
				bh = bread(DEFAULT_PART, blocknr);
				memcpy((char *)&buffer[buffer_off], (char *)bh->b_data, size);
				if(bh)
					brelse(bh);
			}
		}
	}

	return readbytes;
}

/* Note ... file's size could not > 20 * BLOCKSIZ */
/* maybe have another bug ..., check and check and check */
int write(int fd, char * buffer, int size)
{
	struct buf_head * bh;
	struct filp * flptr;
	struct inode * iptr;
	unsigned short blocknr;
	int index, offset, fsize, end, offinblock, buffer_off, writebytes;
	
	if(fd > FILE_DESC_NUM - 1 || current->file[fd].flag == FREE)
		return -EBADF;

	if(fd == DIR_FD)
		return -EBADF;

	flptr = current->file[fd].flptr;

	if(flptr->filp_count == 0)
		return -EBADF;

	if(!(current->file[fd].mode & WRONLY))
		return -EIO;	

	/* check buffer limit */
	/* modify head.S and exec.c */

	offset = flptr->filp_offset;

	iptr = flptr->iptr;

	fsize = iptr->fsize;
	end = offset + size;

	if(end > fsize)
		iptr->fsize = end;

	writebytes = size;

	index = offset / BLOCKSIZ;
	offinblock = offset % BLOCKSIZ;

	buffer_off = 0;
	flptr->filp_offset += size;

	if(size <= BLOCKSIZ - offinblock){
		if((index * BLOCKSIZ) > fsize){
			blocknr = get_block();
			iptr->block[index] = blocknr;
			bh = getblk(DEFAULT_PART, blocknr);

			if(bh == NULL)
				panic("getblk failed in write!");
		}else{
			blocknr = iptr->block[index];
		
			if(blocknr == 0){
				blocknr = get_block();
				iptr->block[index] = blocknr;
				bh = getblk(DEFAULT_PART, blocknr);

				if(bh == NULL)
					panic("getblk failed in write!");
			}else
				bh = bread(DEFAULT_PART, blocknr);
		}

		memcpy((char *)&bh->b_data[offinblock], buffer, size);
		bwrite(bh);

		if(bh)
			brelse(bh);
	}else{
		if((index * BLOCKSIZ) > fsize){
			blocknr = get_block();
			iptr->block[index] = blocknr;
			bh = getblk(DEFAULT_PART, blocknr);

			if(bh == NULL)
				panic("getblk failed in write!");
		}else{
			blocknr = iptr->block[index];

			if(blocknr == 0){
				blocknr = get_block();
				iptr->block[index] = blocknr;
				bh = getblk(DEFAULT_PART, blocknr);

				if(bh == NULL)
					panic("getblk failed in write!");
			}else
				bh = bread(DEFAULT_PART, blocknr);
		}

		memcpy((char *)&bh->b_data[offinblock], buffer, BLOCKSIZ - offinblock);
		bwrite(bh);

		if(bh)
			brelse(bh);

		size -= BLOCKSIZ - offinblock;
		buffer_off += BLOCKSIZ - offinblock;

		while(size > BLOCKSIZ){
			if((index * BLOCKSIZ) > fsize){
				blocknr = get_block();
				iptr->block[++index] = blocknr;
				bh = getblk(DEFAULT_PART, blocknr);

				if(bh == NULL)
					panic("getblk failed in write!");
			}else{
				blocknr = iptr->block[++index];

				if(blocknr == 0){
					blocknr = get_block();
					iptr->block[index] = blocknr;
					bh = getblk(DEFAULT_PART, blocknr);
	
					if(bh == NULL)
						panic("getblk failed in write!");
				}else
					bh = bread(DEFAULT_PART, blocknr);
			}

			memcpy((char *)&bh->b_data[offinblock], &buffer[buffer_off], BLOCKSIZ);
			bwrite(bh);

			if(bh)
				brelse(bh);

			size -= BLOCKSIZ;
			buffer_off += BLOCKSIZ;
		}
		
		if(size > 0){
			if((index * BLOCKSIZ) > fsize){
				blocknr = get_block();
				iptr->block[++index] = blocknr;
				bh = getblk(DEFAULT_PART, blocknr);

				if(bh == NULL)
					panic("getblk failed in write!");
			}else{
				blocknr = iptr->block[++index];

				if(blocknr == 0){
					blocknr = get_block();
					iptr->block[index] = blocknr;
					bh = getblk(DEFAULT_PART, blocknr);

					if(bh == NULL)
						panic("getblk failed in write!");
				}else
					bh = bread(DEFAULT_PART, blocknr);
			}

			memcpy((char *)&bh->b_data[offinblock], &buffer[buffer_off], BLOCKSIZ);
			bwrite(bh);

			if(bh)
				brelse(bh);
		}
	}

	flush_fs();

	return writebytes;
}

int seek(int fd, int offset, int mode)
{
	int tooff, fsize, index;
	struct filp * flptr;
	struct inode * iptr;

	if(fd > FILE_DESC_NUM - 1 || current->file[fd].flag == FREE)
		return -EBADF;

	flptr = current->file[fd].flptr;
	if(flptr->filp_count == 0)
		return -EBADF;

	fsize = flptr->iptr->fsize;
	iptr = flptr->iptr;

	if(mode == SEEK_SET)	
		tooff = flptr->filp_offset = offset;
	else if(mode == SEEK_CUR)
		tooff = flptr->filp_offset += offset;
	else if(mode == SEEK_END)
		tooff = flptr->filp_offset = fsize + offset;
	else
		return -EIO;

	if(tooff > fsize - 1){
		index = (fsize / BLOCKSIZ);

		/* Note ... file's size could not > 20 * BLOCKSIZ  */
		while(((index + 1) * BLOCKSIZ) - 1 <= tooff)
			iptr->block[index++] = 0;

		flptr->iptr->fsize = tooff;
		
		/* ... */
		flush_fs();
	}

	return 0;
}

int close(int fd)
{
	struct filp * flptr;

	if(fd > FILE_DESC_NUM - 1 || current->file[fd].flag == FREE)
		return -EBADF;

	flptr = current->file[fd].flptr;
	if(flptr->filp_count == 0)
		return -EBADF;

	current->file[fd].flag = FREE;

	free_filp(flptr);

	return 0;
}

/* NOTICE !!! competitive condition, here i care nothing */
int unlink(char * filename)
{
	int i;

	struct dir_item * diptr;
	struct inode * iptr;

	diptr = find_diritem(filename);
	if(diptr == NULL)
		return -EIO;

	iptr = &inodes[diptr->inum];
	/* NOTICE 20 ... */
	for(i = 0; i < 20; i++){
		free_block(iptr->block[i]);
		iptr->block[i] = 0;
	}

	free_inode(diptr->inum);

	free_diritem(diptr);

	flush_fs();	

	return 0;
}
