/*
 * straycats.c
 *
 * Copyright (C), 1994, Graeme W. Wilford. (Wilf.)
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * code to detect stray cats - somewhat tricky with the FSSTND. It Will find
 * stray cats for both FSSTND compliant man trees and old style/local user
 * type man trees. Will also do lots of other useful (?) things now.
 *
 * Tue May  3 21:24:51 BST 1994 Wilf. (G.Wilford@ee.surrey.ac.uk)
 */

#define MANPATH_MAIN    /* to not define *std_sections[] */

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "config.h"
#include "mydbm.h"
#include "straycats.h"
#include "manp.h"
#include "dbver.h"
#include "convert_name.h"

#ifdef HAS_GETOPT_LONG
#  include <getopt.h>
#endif /* HAS_GETOPT_LONG */

extern char *optarg;

void ver(void);
char *mkprogname(char *s);

char *prognam;
int debug;

#ifdef COMP_SRC
extern struct compress_ext decompress[];
#endif

static MYDBM_FILE dbf;
static short all;	/* check all man trees, even unwritable ones */

static short create_place;	/* create place-holders */
static short del_rel_stray;	/* delete TRUE straycats !! */
static short del_nonrel_stray;	/* delete non-rel straycats */
static short del_place;  	/* delete place-holders */
static short del_normal;	/* delete normal catfile */
static char *alt_system = NULL;	/* use alternate system man pages */
static int verbose = 2;		/* level of verbosity (0-4) */

void usage(void)
{
	printf("usage: %s [[-da] [-N|-C|-S|-R|-Z|-X] [-m system] [v level]] | [-h] | [-V]\n", prognam);
	printf("-d --debug                  produce debugging info.\n"
  	       "-a --all                    check all man trees regardless of write perms.\n"
	       "-m --systems system         also search alternate system(s) man directory.\n"
	       "-v --verbose level          verbosity of messages; 0: none, 2: default, 4: all.\n"
	       "-V --version                show version.\n"
	       "-h --help                   show this usage message.\n"
  	       "\n\tUse the following options with care, creation/deletion with your\n"
  	       "\t(root included) priveledges will occur. Read the man page.\n\n"
  	       "-N --normal                 default CREATION/DELETION, invokes -S -C.\n"
	       "-C --create-placeholders    CREATE place-holders.\n"
  	       "-S --delete-semi-straycats  DELETE straycats with a non-relative source.\n"
	       "-R --delete-placeholders    DELETE place-holders.\n"
	       "-Z --delete-cats            DELETE normal cat files that have a source.\n"
	       "-X --delete-true-straycats  DELETE straycats with no source!\n");
}

#ifdef HAS_GETOPT_LONG
static struct option long_options[] =
{
    {"debug",			  no_argument, 		0, 'd'},
    {"all", 			  no_argument, 		0, 'a'},
    {"normal",			  no_argument, 		0, 'N'},
    {"delete-semi-straycats",	  no_argument, 		0, 'S'},
    {"create-placeholders",	  no_argument, 		0, 'C'},
    {"delete-placeholders",	  no_argument, 		0, 'R'},
    {"delete-cats", 		  no_argument, 		0, 'Z'},
    {"delete-true-straycats",     no_argument, 		0, 'X'},
    {"verbose",			  optional_argument,	0, 'v'},
    {"help",			  no_argument, 		0, 'h'},
    {"version", 		  no_argument, 		0, 'V'},
    {"systems", 		  optional_argument, 	0, 'm'},
    {0, 0, 0, 0}
};
#endif /* HAS_GETOPT_LONG */

static char args[] = "daNSCRZXhVm:v:";
	
int main(int argc, char *argv[])
{
	char *c1, *path;
	int c;
	int option_index; /* not used, but required by getopt_long() */

	prognam = mkprogname (argv[0]);

#ifdef HAS_GETOPT_LONG

	while ((c = getopt_long (argc, argv, args,
				 long_options, &option_index)) != EOF){
#else /* HAS_GETOPT_LONG */

	while ((c = getopt (argc, argv, args)) != EOF){

#endif /* HAS_GETOPT_LONG */

		switch (c){

			case 'd':
				debug++;
				break;
			case 'a':
				all++;
				break;
			case 'N':
				del_nonrel_stray = 1;
				create_place = 1;
				break;
			case 'S':
				del_nonrel_stray = 1;
				break;	
			case 'C':
				create_place = 1;
				break;
			case 'R':
				del_place++;
				break;
			case 'Z':
				del_normal++;
				break;
			case 'X':
				del_rel_stray++;
				break;
			case 'V':
				ver();
				exit(0);
			case 'm':
				alt_system = strdup(optarg);
				break;
			case 'v':
				if (isdigit(*optarg))
					verbose = atoi(optarg);
				else {
					fprintf(stderr, "%s: (%s) invalid argument to option -- v\n",
					  prognam, optarg);
					usage();
					exit(1);
				}
				break;
			case 'h':
				usage();
				exit(0);
			default:
				usage();
				exit(1);
		}
	}
	
	if (del_rel_stray && create_place) {
		usage();
		fputs(
		  "\nWill not delete straycats AND touch relative src files: pointless",
		  stderr);
		exit(1);
	}
	if ( (create_place && del_place)
	  || (del_rel_stray && del_nonrel_stray) ) {
	  	usage();
	  	fputs("\npointless combination of options", stderr);
	  	exit(1);
	}
	
	if (!create_place && !del_rel_stray && !del_place 
	  && !del_nonrel_stray && !del_normal)
	  	if (verbose > 0)
			printf("** Only looking for straycats, use an option for creation/deletion. **\n");

	path = manpath(0, alt_system);

	free(alt_system);

	/* 
	 * should check write perms in directories first and only deal 
	 * with ours.
	 */

	while( (c1 = strrchr(path, ':')) != NULL){
		*c1 = '\0';

		compare_dirs(++c1);
	}
	compare_dirs(path);

	return 0;
}

void compare_dirs(char *path)
{
	char catdir[BUFSIZ], mandir[BUFSIZ], database[BUFSIZ];

	strcpy(mandir, path);

	/* 
	 * check for write perms
	 */

	if (!all && access(mandir, W_OK) == -1) {
		if (verbose > 0)
			printf("skipping %s: no write permission.\n",
			  path);
		return;
	}
	if (verbose > 0)
		printf("Looking at %s tree\n", path);

	/* 
	 * if it's MANDB_MAP'd, it'll be in the global database
	 * if not, it'll be in the users local database.
	 */
	 
	if (global_catpath(mandir) == NULL) {
		strcpy(database, mandir);
		strcat(database, LOCAL_DB);
	}
	else
		strcpy(database, GLOBAL_DB);

	if ( (dbf = MYDBM_REOPEN(database)) == NULL) {
		fprintf(stderr, "%s: compare_dirs: ", prognam);
		perror(database);
		exit(1);
	}

	dbver_rd(dbf);

	/* regardless of the FSSTND, we need to check the usual place */

	strcpy(catdir, mandir);
	open_dirs(mandir, catdir);

	/* if we're FSSTND, check the appropriate place if necessary */
	/* we should check to see if this is just a symlink, oh well... */
	
#ifdef FSSTND
	strcpy(mandir, path);

	if (global_catpath(mandir) == NULL) {
		MYDBM_CLOSE(dbf);
		return;
	}
	else if (strlen(mandir) < 9) { /* sizeof "/usr/man" */
		strcpy(catdir, CAT_ROOT);
	} else 
		strcpy(catdir, convert_name_sc(mandir));

	open_dirs(mandir, catdir);
#endif
	MYDBM_CLOSE(dbf);
}

void open_dirs(char *mandir, char *catdir)
{
	DIR *cdir;
	struct dirent *catlist;
	size_t catlen, manlen;
	char *t1;

	if ( (cdir = opendir(catdir)) == NULL){
		fprintf(stderr, "could not open dir for reading: ");
		perror(catdir);
		return;
	}
	strcat(catdir, "/");
	strcat(mandir, "/");
	catlen = strlen(catdir);
	manlen = strlen(mandir);
		
	while ((catlist = readdir(cdir)) != NULL){
		if (*catlist->d_name != 'c')
			continue;
		strcpy(catdir + catlen, catlist->d_name);
		strcpy(mandir + manlen, catlist->d_name);
		t1 = strrchr(mandir, '/');
		*(t1 + 1) = 'm';
		*(t1 + 3) = 'n';
		
		if (debug)
			fprintf(stderr, "catdir: %s\nmandir: %s\n",
			  catdir, mandir);
 
		touch_stray_src(catdir, mandir);
	}
	closedir(cdir);
}

void touch_stray_src(char *catdir, char *mandir)
{
	DIR *cdir;
	FILE *fd;
	datum key, content;
	struct dirent *catlist;
	struct stat buf;
#ifdef COMP_SRC
	struct compress_ext *comp;
#endif
	short found;
	size_t len, lencat;
	char *filename, *ext;
	
	if ( (cdir = opendir(catdir)) == NULL){
		fprintf(stderr, "could not open dir for reading: ");
		perror(catdir);
		return;
	}

	strcat(mandir, "/");
	strcat(catdir, "/");
	len = strlen(mandir);
	lencat = strlen(catdir);
		
	while ((catlist = readdir(cdir)) != NULL) {
		if (*catlist->d_name == '.' && strlen(catlist->d_name) < 3)
			continue;

		strcpy(mandir + len, catlist->d_name);
		strcpy(catdir + lencat, catlist->d_name);

		ext = strrchr(mandir, '.');
		if (ext == NULL) {
			if (verbose > 0) {
				printf("Warning: %s has no extension.\n", catdir);
				continue;
			}
		} else if (strcmp(ext, COMPRESS_EXT) == 0)
			*ext = '\0';

		ext = strrchr(mandir, '.');

		/* check for bogosity/wrong compression extension */
		
		if (ext == NULL || *(ext + 1) == '\0' || 
		  *(ext + 1) != *(mandir + len - 2)) {
		  	if (verbose > 0) {
			  	printf("Warning: %s is bogus (unsupported extension?)\n",
			  	  catdir);
			  	continue;
			}
		}
		 
		/* 
		 * now that we've stripped off the cat compression
		 * extension (if it has one), we can try some of ours.
		 */

		if (debug)
			fprintf(stderr, "Testing for existence: %s: ", mandir);

		found = 0;

		if (!stat(mandir, &buf)) 
			found++;
#ifdef COMP_SRC 
		else {
			len = strlen(mandir);
			for (comp = decompress; comp->ext; comp++) {
				strcpy(mandir + len, comp->ext);
				if (debug)
					fprintf(stderr, "Testing for existence: %s: ", mandir);

				if (!stat(mandir, &buf)) {
					found++;
					break;
				}
			}
	/*		*(mandir + len) = '\0'; */
		}
#endif
		/* test to see if we found the cats' source file */
		
		if (debug)
			fprintf(stderr, "%s, error=%d\n", found ? "found" : "unfound", errno);

		if (!found && errno == ENOENT) {

			/* RELATIVE source non existent */

			key.dptr = strdup(catlist->d_name);

			ext = strrchr(key.dptr, '.'); /* .Z ... */
			if (ext == NULL) {
				if (verbose > 0) {
					printf("Warning: %s has no extension.\n", catdir);
					continue;
				}
			} else if (strcmp(ext, COMPRESS_EXT) == 0)
				*ext = '\0';

			ext = strrchr(key.dptr, '.'); /* .# */
			if (ext == NULL) {
				if (verbose > 0) {
					printf("Warning: %s has no section.\n", catdir);
					continue;
				}
			} else *ext = '\0';

			if (debug)
				fprintf(stderr, "src not found, db key is %s\n", key.dptr);

			key.dsize = strlen(key.dptr) + 1;

			content = MYDBM_FETCH(dbf, key);
			free(key.dptr);

			/* 
			 * check the db, it might exist in another 
			 * heirarchy. (source tree)
			 * Make sure that what we've found belongs to
			 * the right section. Check all paths, not again...
			 */

			if ( (filename = exists(content.dptr, catlist->d_name)) 
			  == NULL) {

				/* 
				 * It absolutely doesn't exist. 
				 * We can either bin it, or be more
				 * friendly and create a place-holder 
				 * for it.
				 */
				 	
				if (verbose > 1)
					printf("%s: STRAYCAT\n", catdir);

				if (del_rel_stray) {
					if (remove(catdir) != 0) {
						fprintf(stderr,
						  "delete straycat: ");
						perror(catdir);
					} else
						if (verbose > 2)
							printf("%s: deleted straycat.\n",
							  catdir);
				}

				if (!create_place)
					continue;

				if ( (fd = fopen(mandir, "w")) == NULL){
					fprintf(stderr,
					  "%s: creation: ", prognam);
					perror(mandir);
					continue;
				}

				if (verbose > 2)
					printf("%s: created place holder\n",
					  mandir);
					  
				if (fclose(fd) != 0){
					fprintf(stderr,
					  "%s: could not close: ", prognam);
					perror(mandir);
					exit(1);
				}
			} else {
				/* 
				 * this is where it gets tricky.
				 * we have a stray cat, but we have the 
				 * source in a different man tree :-)
				 * 
			  	 * check to see if database file actually
			  	 * exists in reality. If not, db is out 
			  	 * of sync. die and inform user.
			  	 */ 

				/*
			  	 * if we're deleting place-holders, it's
			  	 * likely that the db will be out of sync
			  	 * continue regardless :-$
			  	 */

			  	if (del_place)
			  		continue;

				if (stat(filename, &buf) == -1 
				  && errno == ENOENT) {
				  	if (lstat(filename, &buf) == -1) {
				  		fputs("ERROR: Global database is out of date with the filesystem, run mandb first.\n", stderr);
				  		exit(0);
				  	} else {
				  		if (verbose > 0)
				  			printf("Warning: %s: hanging symbolic link\n",
				  			  filename);
				  		continue;
				  	}
				}
				
				if (verbose > 1)
			 		printf("%s: STRAYCAT, but has a NON RELATIVE source file: %s\n",
			 		  catdir, filename);

				if (del_nonrel_stray) {
				 	if (remove(catdir) != 0) {
				 		fputs("delete non-rel stray: ", 
				 		  stderr);
				 		perror(catdir);
				 	} else
				 		if (verbose > 2)
				 			printf("%s: deleted non-relative straycat.\n",
				 			  catdir);
				}
				MYDBM_FREE(content.dptr);
			}
			
		} else if (buf.st_size == 0) {
	
			/* we have a cat place holder */
	
			if (verbose > 2)
				printf("%s: has place-holder: %s\n", catdir, mandir);

			if (del_place) {
				if (remove(mandir) != 0) {
				 	fputs("delete place-holder ", stderr);
				 	perror(mandir);
				} else
					if (verbose > 2)
						printf("%s: deleted place-holder\n", 
						  mandir);
			}
		} else {

			/* we have a normal cat file */

			if (verbose > 3)
				printf("%s: cat file from %s\n", catdir, mandir);

			if (del_normal) {
				if (remove(catdir) != 0) {
					fputs("delete normal cat file ", stderr);
					perror(catdir);
				} else
					if (verbose > 2)
						printf("%s: deleted normal cat file.\n", 
						  catdir);
			}
		}
	}
	closedir(cdir);
}

#ifdef FSSTND

/*
 * Change name of form "/usr/.../man" to "/var/catman/..." 
 */

char *convert_name_sc(char *name)
{
	char *to_name, *newto_name;
	char *t2, *t1;

	to_name = strdup (name);

	t1 = strrchr (to_name, '/');
	*t1 = '\0';
	t2 = strrchr (to_name, '/');

	newto_name = (char *) malloc (sizeof CAT_ROOT + strlen(t2) + 2);
	strcpy(newto_name, CAT_ROOT);
	strcat(newto_name, t2);

	free(to_name);
	return newto_name;
}
#endif

char *exists(char *list, char *catname)
{
	char *t1;
	size_t len;

	if (list == NULL)
		return NULL;

	/* check to see if we have THE valid extension first */

	if (strcmp(strrchr(catname, '.'), COMPRESS_EXT) == 0)
		len = strlen(catname) - (sizeof COMPRESS_EXT - 1);  /* - 2; */
	else
		len = strlen(catname);

	while ( (t1 = strrchr(list, ':')) != NULL) {
		*t1 = '\0';
		if (strncmp(strrchr(++t1, '/') + 1, catname, len) == 0)
		  	return t1;
	}
	if (strncmp(strrchr(list, '/') + 1, catname, len) == 0)
		return list;

	return NULL;
}
