/*
 * mandb.c: used to create and initialise global man database.
 *  
 * Copyright (C), 1994, Graeme W. Wilford. (Wilf.)
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * Tue Apr 26 12:56:44 BST 1994  Wilf. (G.Wilford@ee.surrey.ac.uk) 
 */

#define MANPATH_MAIN    /* to not define *std_sections[] */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>

#include "config.h"
#include "mydbm.h"
#include "statdir.h"
#include "manp.h"
#include "manpath.h"
#include "util.h"
#include "gripes.h"
#include "dbver.h"

#ifdef HAS_GETOPT_LONG
#  include <getopt.h>
#endif /* HAS_GETOPT_LONG */

extern char *optarg;

/* Berkeley db routines emulate ndbm but don't add .dir & .pag, just .db! */
#ifdef _DB_H_ /* has Berkeley db.h been included? */     
#define BERKELEY_DB
#endif

void ver(void);
char *mkprogname(char *s);

char *prognam;
int debug = 0;
short quiet = 0;
short create_user = 0;
short dups = DUPS;
MYDBM_FILE dbf;
char *manp;
char *database;

void usage(void)
{
	printf("usage: %s [ -dqruV ] [ -m system ] [ -h ]\n", prognam);
	printf(	"-d --debug                  produce debugging info.\n"
		"-q --quiet                  work quietly, except for 'bogus' warning.\n"
		"-r --reverse-dups           reverse the 'duplicates' flag - default: %s.\n"
		"-u --user-db                create user databases only.\n"
		"-m --systems system         include alternate systems man pages.\n" 
		"-V --version                show version.\n"
		"-h --help                   show this usage message.\n",
			dups ? "don't add" : "add");
}

char *get_mandb_manpath(const char *systems)
{
	size_t len = 0;
	char *manpathlist;
	DIRLIST *dlp;
	extern DIRLIST list[]; /* mappings set up by manpath() */

	for (dlp = list; dlp->mandir[0] != '\0'; dlp++)
		if (dlp->mandatory == MANDB_MAP)
			len += strlen (dlp->mandir) + 1;

	/* see if we have anything at all */

	if (!len)
		return NULL;

	manpathlist = (char *) malloc (len + 1);
	if (manpathlist == NULL)
		gripe_alloc (len, "mandb manpathlist");

	*manpathlist = '\0';

	for (dlp = list; dlp->mandir[0] != '\0'; dlp++) {
		if (dlp->mandatory == MANDB_MAP) {
			strcat(manpathlist, ":");
			strcat(manpathlist, dlp->mandir);
		}
	}
	return add_system_manpath(systems, ++manpathlist);
}

short create_local(void)
{
	short amount;
	
	if (!quiet)
		printf("Creating user database: %s\n", database);

	if ( (dbf = MYDBM_OPEN(database)) == NULL){
		fprintf(stderr, "%s: create_local: ", prognam);
		perror(database);
		exit(1);
	}

	dbver_wr(dbf);
	
	MYDBM_CLOSE(dbf);

	amount = check_mandirs(0);
	return amount;
}

#ifdef HAS_GETOPT_LONG
static struct option long_options[] =
{
    {"debug",	     no_argument, 	0, 'd'},
    {"reverse-dups", no_argument, 	0, 'r'},
    {"user-db",      no_argument, 	0, 'u'},
    {"quiet",        no_argument, 	0, 'q'},
    {"help",         no_argument,	0, 'h'},
    {"version",      no_argument, 	0, 'V'},
    {"systems",      optional_argument, 0, 'm'},
    {0, 0, 0, 0}
};
#endif /* HAS_GETOPT_LONG */

static char args[] = "druqhVm:";
	
int main(int argc, char *argv[])
{
	int c;
	short amount = 0;
	char *c1, *user;
	char *alt_systems = NULL;
	int option_index; /* not used, but required by getopt_long() */
#ifdef NDBM
	size_t ndbm_db;
#endif

	prognam = mkprogname (argv[0]);

#ifdef HAS_GETOPT_LONG

	while ((c = getopt_long (argc, argv, args,
				 long_options, &option_index)) != EOF){
#else /* HAS_GETOPT_LONG */

	while ((c = getopt (argc, argv, args)) != EOF){

#endif /* HAS_GETOPT_LONG */

		switch (c){

			case 'd':
				debug++;
				break;
			case 'q':
				quiet++;
				break;
			case 'r':
				dups = !dups;
				break;
			case 'u':
				create_user++;
				break;
			case 'V':
				ver();
				exit(0);
			case 'h':
				usage();
				exit(0);
			case 'm':
				alt_systems = optarg;
				dups = 0;
				break;
			default:
				usage();
				exit(1);
		}
	}

	manp = manpath(quiet, alt_systems);

	user = strdup(GLOBAL_DB);
	*(strrchr(user, '/')) = '\0';

	/*
	 * Check access rights of invoking user. Would they be allowed to 
	 * either write to the database, or if it doesn't exist, create in 
	 * the directory that it lives?
	 */

	if (!create_user && (access(GLOBAL_DB, W_OK) == 0 || access(user, W_OK) == 0) ) {

#ifdef SECURE_MAN_UID
		struct passwd *user;
#endif		
		if ( (c1 = get_mandb_manpath(alt_systems)) != NULL)
			manp = c1;
		else
			fprintf(stderr, 
			  "Warning: No MANDB_MAP directives in " CONFIG_FILE ", using your manpath.\n");
	
#ifdef SECURE_MAN_UID
		if ( (user = getpwnam(MAN_OWNER)) == NULL) {
			fprintf(stderr, "The user " MAN_OWNER " does not exist");
			exit(1);
		}
#endif 
	
		if (!quiet) { 
			puts("Creating global database: " GLOBAL_DB);
			printf("Using %s as manpath\n", manp);
		}
		database = GLOBAL_DB;
		if ( (dbf = MYDBM_OPEN(database)) == NULL){
			fprintf(stderr, "%s: main: ", prognam);
			perror(database);
			exit(1);
		}

	/* give the db a version */

		dbver_wr(dbf);
		MYDBM_CLOSE(dbf);
		
		amount = check_mandirs(1);

	/* now we need to change the perms & group (if setuid) */

#ifdef NDBM
		ndbm_db = sizeof GLOBAL_DB - 1;
		database = (char *) malloc ( ndbm_db + 5 ); /* ".dir",".pag" */
		strcpy(database, GLOBAL_DB);
#ifdef BERKELEY_DB
		strcpy(database + ndbm_db, ".db");
#else /* not BERKELEY */
		strcpy(database + ndbm_db, ".dir");
#endif /* BERKELEY */

#ifdef SECURE_MAN_UID
		if (chown(database, user->pw_uid, -1) != 0) {
			perror(database);
			remove(database);
		}
#endif /* SECURE_MAN_UID */

		if (chmod(database, MODE) != 0) {
			perror(database);
			remove(database);
		}

#ifndef BERKELEY_DB
		strcpy(database + ndbm_db, ".pag");
		
#ifdef SECURE_MAN_UID
		if (chown(database, user->pw_uid, -1) != 0) {
			perror(database);
			remove(database);
			exit(1);
		}
#endif /* SECURE_MAN_UID */

		if (chmod(database, MODE) != 0) {
			perror(database);
			remove(database);
		}
#endif /* not BERKELEY */
		free(database);
#else /* NDBM (ie not NDBM) */

#ifdef SECURE_MAN_UID
		if (chown(database, user->pw_uid, -1) != 0) {
			perror(database);
			remove(database);
			exit(1);
		}
#endif /* SECURE_MAN_UID */

		if (chmod(database, MODE) != 0) {
			perror(database);
			remove(database);
		}
#endif /* NDBM */

	} else if ( (user = get_userdb_list()) != NULL) {
		while ( (c1 = strchr(user, ':')) != NULL) {
			*c1 = '\0';
			database = ++c1;
			amount += create_local();
		}
		database = user;
		amount += create_local();
		free(--user);
	} else {
		if (!quiet)
			puts("No access to global database and no user databases: nothing to do.");
		exit(0);
	}	

	if (!quiet)
		printf("%d man directories changed\n", amount);

	return 0;
}
