/*
 * check_mandirs.c
 *
 * Copyright (C), 1994, Graeme W. Wilford. (Wilf.)
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * routines used in statting dirs 
 *
 * Mon May  2 17:36:33 BST 1994  Wilf. (G.Wilford@ee.surrey.ac.uk)
 */

#define MANPATH_MAIN    /* to not define *std_sections[] */

#include <string.h>
#include <stdio.h>
#include <dirent.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"
#include "mydbm.h"
#include "order.h"
#include "util.h"
#include "convert_name.h"

extern char *prognam;
extern char *manpathlist[];
extern char *database;
extern int debug;
extern MYDBM_FILE dbf;
extern short quiet;
extern short dups;

#ifdef COMP_SRC
extern struct compress_ext decompress[];
#endif

int is_duplicate(char *oldpage, char *manpage)
{
	char *old, *man;
	int choice = 0;
	int status;
#ifdef COMP_SRC
	char *old_ext, *man_ext;
	struct compress_ext *comp;
#endif

	old = strrchr(oldpage, '/');
	man = strrchr(manpage, '/');
	if (old == NULL || man == NULL) {
		fprintf(stderr, "Warning - man file containing : found. This will cause problems.\n");
		return 0;
	}
	old = strdup(old + 1);
	man = strdup(man + 1);

#ifdef COMP_SRC
	/* do something here to check for compression extensions */

	old_ext = strrchr(old, '.');
	man_ext = strrchr(man, '.');
	
	for ( comp = decompress; comp->ext; comp++)
		if (strcmp(old_ext, comp->ext) == 0) {
			*old_ext = '\0';
			break;
		}

	for ( comp = decompress; comp->ext; comp++)
		if (strcmp(man_ext, comp->ext) == 0) {
			*man_ext = '\0';
			break;
		}
#endif
	/* need to do some checks on their time stamps and keep the newest */

	if (dups) { 	/* only if we're dup checking! */
		if (strcmp(old, man) == 0) {
			status = is_newer(oldpage, manpage);
	
			if (status & 2) 	/* oldpage is place-holder */
				choice -= 2;
			if (status & 4) 	/* manpage is place-holder */
				choice += 2;
			if (status & 1)		/* oldpage is newer */
				choice++;
			else			/* manpage is newer */
				choice--;
			
			if (choice >= 0) {
				if (!quiet)	
					printf("WARNING - duplicate: %s & (%s ignored)\n",
					  oldpage, manpage);
				choice = 1;
			} else {
				if (!quiet)
					printf("WARNING - duplicate: (%s ignored) & %s\n",
					  oldpage, manpage);
				choice = -1;
			}
		} else if (strncmp(old, man, 
		  (size_t) (strrchr(old, '.') - old + 2) ) == 0) 
			if (!quiet)
				printf("WARNING - possible duplicate: %s & (%s NOT ignored)\n",
				  oldpage, manpage);
	}

	free(old);
	free(man);
	return choice;
}

int dup_test(char *oldpage, char *manpage)
{
	char *t1, *newlist;
	int ret;

	*(newlist = (char *) malloc (strlen(oldpage) + 2)) = '\0'; 
	
	while( (t1 = strrchr(oldpage, ':')) != NULL){
		*(t1++) = '\0';

		if (strcmp(t1, manpage) == 0)
			ret = 1;
		else
			ret = is_duplicate(t1, manpage);

		if (ret == 1) {				/* 1 */
			free(newlist);
			return 1;
		} else if (ret == 0) {			/* 0 */
			strcat(newlist, t1);
			strcat(newlist, ":");
		} 					/* -1 */
	}

	if (strcmp(oldpage, manpage) == 0)
		ret = 1;
	else
		ret = is_duplicate(oldpage, manpage);

	if (ret == 1) {
		free(newlist);
		return 1;
	} else if (ret == 0) {
		strcat(newlist, oldpage);
		strcat(newlist, ":");
	}

	if ( (t1 = strrchr(newlist, ':')) != NULL)
		*t1 = '\0';

	strcpy(oldpage, newlist);
	free(newlist);
	return 0;
}

void add_dir_entrys(char *path, char *infile)
{
	char manpage[MAXPATHLEN];
	char keyc[80];
	char *oldpage, *t1, *t2;
	datum key, content;
	size_t len;
	struct dirent *newdir;
	DIR *dir;
#ifdef COMP_SRC
	struct compress_ext *comp;
#endif

	strcpy(manpage, path);
	strcat(manpage, "/");
	strcat(manpage, infile);
	strcat(manpage, "/");
	len = strlen(manpage);

	/*
	 * All filename entries in this dir should either be valid manpages
	 * or . files (such as current, parent dir).
	 */

	if ( (dir = opendir(infile)) == NULL){
                fprintf(stderr, "Could not open dir for reading: ");
                perror(manpage);
                return;
        }
        
	while ( (newdir = readdir(dir)) != NULL) {
		if ( *newdir->d_name == '.' && strlen(newdir->d_name) < 3)
			continue;
		else {
			strcpy(manpage + len, newdir->d_name);
			content.dptr = manpage;
	
			t1 = strrchr(manpage, '/');
			key.dptr = strcpy(keyc, t1 + 1); /* !don't optimise! */

#ifdef COMP_SRC
			/* Check file for compression extension */

			if ( (t2 = strrchr(keyc, '.')) != NULL) {
				for (comp = decompress; comp->ext; comp++)
					if (strcmp(comp->ext, t2) == 0)
						*t2 = '\0';
	
				t2 = strrchr(keyc, '.'); 
			}
#else
			t2 = strrchr(keyc, '.');
#endif
			/* 
			 * A bogus manpage is defined as having it's section
			 * number (alpha/num) != to the section it is
			 * stored under. (only the first char of the manpage
			 * section is compared with the last char of the 
			 * section it is under)
			 * ie. .../man5/shells.5.orig, .../man1/xast.man.
			 *
			 * also files without a period MUST be bogus!
			 *
			 * .../manavs/intro.avs WOULD be wrong. - 
			 * better to store in a separate man tree.
			 * 
			 * things like .../man1/ld.so.1 are fine and MUST 
			 * be accepted.
			 */
	
			if (t2 == NULL || *(--t1) != *(t2 + 1)){
				if (!quiet)
					printf("WARNING - bogus manpage:"
				  	  " %s ignored\n", manpage);
				continue;
			}
			*t2 = '\0'; 

			content.dsize = strlen(content.dptr) + 1;
			key.dsize = strlen(key.dptr) + 1;

	
			if (MYDBM_INSERT(dbf, key, content) != 0){
			
				/*
				 * this is where it gets a little bit tricky
				 * we already have a manpage registered 
				 * for this key.
				 *
				 * we need to check for duplicates, and 
				 * also maintain
				 * the appropriate order set out in 
				 *   char *std_sections[].
				 */
	
				oldpage = MYDBM_FETCH(dbf, key).dptr;
	
				/*
				 * check to see if it's a duplicate man page.  
				 * it's more subtle than I thought, need 
				 * to compare 
				 * with all of the other key pages, not 
				 * just the last
				 *
				 * dup_test() returns NULL if a duplicate was 
				 * found and the newer file was already in the
				 * db, else the replacement string is returned.
				 *
				 * This could either be an empty string, or 
				 * one or more pages in a list.
				 */
	
				if (dup_test(oldpage, manpage)) {
					MYDBM_FREE(oldpage);
					continue;	/* skip this page */
				}

				if (*oldpage != '\0')
					content.dptr = orderthem(manpage, oldpage);
				else
					content.dptr = manpage;

				/* contents of oldpage are now dead */
				MYDBM_FREE(oldpage); 

				content.dsize = strlen(content.dptr) + 1; 
				
				if (debug) 
					fprintf(stderr, "REPLACE: %s -> %s\n",
					  key.dptr, content.dptr);
	
				MYDBM_REPLACE(dbf, key, content);
			}
			else {
				if (debug) 
					fprintf(stderr, "INSERT: %s -> %s\n",
					  key.dptr, content.dptr);
			}
		}
	}
	closedir(dir);
}

/*
 * accepts the raw man dir tree eg. "/usr/man" and the time stored in the db
 * any dirs of the tree that have been modified (ie added to) will then be
 * scanned for new files, which are then added to the db.
 */
short testmandirs(char *path, time_t last)
{
	DIR *dir;
	struct dirent *mandir;
	struct stat stbuf;
	short amount = 0;

	if (debug)
		fprintf(stderr, "Testing %s for new files\n", path);

	if ( (dir = opendir(path)) == NULL){
		fprintf(stderr, "Could not open dir for reading: ");
		perror(path);
		return 0;
	}

	chdir(path); /* not a brilliant move! */

	while( (mandir = readdir(dir)) != NULL){
		if (*mandir->d_name != 'm' 
		  && *mandir->d_name != 'M')
		  /* these aren't man subdirs then */
			continue;
			
		if ( stat(mandir->d_name, &stbuf) == 0 
		  && S_ISDIR(stbuf.st_mode) && stbuf.st_mtime > last){

		  	/* 
		  	 * file exists, is a DIR, and has been 
		  	 * 'modified' since last full db update
		  	 * - could be a new manpage for us.
		  	 */

			/* (quiet) below, is supposed to be backwards */
			
			if (quiet)
				fputs("\rUpdating database(s). Wait...", stderr);

			if (debug)
				fprintf(stderr,
				  "\tsubdirectory %s has been 'modified'\n",
				  mandir->d_name);

			/*
			 * Maybe if opening the db in RW mode fails, we 
			 * should just chill out instead of bailing out?
			 */
			 
			while ( (dbf = MYDBM_RWOPEN(database)) == NULL) {
				static short err;

				if (err == 4) {
					fprintf(stderr,
					  "Could not open %s after 4 attempts.\n",
					  database);
					exit(1);
				} else if (errno == EAGAIN)
					sleep(++err);
				else {
					fprintf(stderr, 
					  "Could not open db for writing: ");
					perror(database);
					exit(1);
				}
			}

		  	add_dir_entrys(path, mandir->d_name);

		  	MYDBM_CLOSE(dbf);
		  	
		  	amount++;
		}
	}
	closedir(dir);
	return amount;
}

/* 
 * This routine is both wrong and out of date - needed for mandb 
 * at the moment. Mandb doesn't have access to *manpathlist[]. This will be
 * changed rsn. I'll move the code from man.c to manpath.c (where it 
 * should always have been).
 */ 
short check_mandirs(short globalman)
{
	datum key, content;
	char *c1;
	time_t now;
	short amount = 0;
	char *path;
	extern char *manp;
	
	key.dptr = KEY;
	key.dsize = sizeof KEY;

	if ( (dbf = MYDBM_REOPEN(database)) == NULL) {
		fprintf(stderr, "%s: check_mandirs: unable to reopen db: ",
		  prognam);
		perror(database);
		exit(1);
	}
	content = MYDBM_FETCH(dbf, key);
	MYDBM_CLOSE(dbf);

	if (content.dptr == NULL)
		now = 0;
	else
		now = atol(content.dptr);

	MYDBM_FREE(content.dptr);

	if (globalman){
		path = strdup(manp);
		while( (c1 = strrchr(path, ':')) != NULL){
			*c1 = '\0';
		/*	if (strncmp(++c1, MAN_ROOT, sizeof MAN_ROOT -1) == 0) */
			if (global_catpath(++c1) != NULL)
				amount += testmandirs(c1, now);
		}
		if (global_catpath(path) != NULL)
			amount += testmandirs(path, now);

	} else {
		/* strip the actual database name */
		path = strdup(database);
		*(strrchr(path, '/')) = '\0';
		amount += testmandirs(path, now);
	}

	free(path);

	if (amount){
		now = time(NULL);
		content.dptr = (char *) malloc(11); /* max long with '\0' */
		sprintf(content.dptr, "%ld", now);
		content.dsize = strlen(content.dptr) + 1;
		if ( (dbf = MYDBM_RWOPEN(database)) == NULL) {
			fprintf(stderr, "%s: check_mandirs: unable to rwopen db: ",
			  prognam);
			perror(database);
			exit(1);
		}
		MYDBM_REPLACE(dbf, key, content);
		MYDBM_CLOSE(dbf);
		free(content.dptr);
		fputs(" done\n", stderr);
	}
	return amount;
}
