/* install.c -- Install a binary distribution from a tar file
 * Created: Mon May  8 10:39:20 1995 by r.faith@ieee.org
 * Revised: Sat Oct 12 22:34:22 1996 by faith@cs.unc.edu
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: install.c,v 1.13 1996/10/13 03:31:04 faith Exp $
 * 
 */

#include "pm.h"
#include <fcntl.h>

				/* This stuff is in pmdb.h, but we don't
                                   include it here. */
#ifndef PMDB_DISP_INSTALLED
#define PMDB_DISP_INSTALLED 'I'
#endif

#ifndef PMDB_DISP_NOTINSTALLED
#define PMDB_DISP_NOTINSTALLED 'N'
#endif

typedef struct params {
   const char  *tarFile;
   int         tarFileFD;       /* open FD on tarFile, pre-filtered */
   int         compressed;
   List        list;		/* notes file */
   PmEntryList entries;
} Params;

static int install_source( const char *name, const char *tar )
{
   int        exitStatus;
   int        source     = 1;
   const char *tarFile = tar ?: pm_resolve_tarfile( name, &source );
   List       list;
   char       *tmp = tempnam( PmTmp, PgmName );
   FILE       *str;

   static void print( const char *line )
      {
	 PRINTF(PM_VERBOSE,("%s\n",line));
      }

   if (!source)
      pm_fatal( PMERR_INTERNAL, "Expected %s to be source\n", tarFile );

   list = pm_notes_read_tar( tarFile );
   pm_notes_parse( list );
   if (TEST(PM_INSTALL)) pm_notes_header( stdout, NULL );
   
   str = pm_file_create( tmp );
   fprintf( str, "#!%s\n", PmBourneShell );
   fprintf( str, "# This is an automatically generated script\n" );
   fprintf( str, "# Generated by %s on %s\n", pm_version(), PmTimeString );
   fprintf( str, "# Source package installation script for %s\n",
	    pm_gen_canonical( NULL ) );
   if (TEST(PM_VERBOSE)) fprintf( str, "set -x\n" );
   fprintf( str, "cd %s\n", PmTmp );
   fprintf( str, "mkdir %s.dir\n", tmp );
   fprintf( str, "cd %s.dir\n", tmp );

   if (pm_file_compressed( tarFile )) {
      if (TEST(PM_VERBOSE) || !PmQuiet)
	 fprintf( str, PM_UNTAR_COMPRESSED_VERBOSE, tarFile );
      else
	 fprintf( str, PM_UNTAR_COMPRESSED, tarFile );
   } else {
      if (TEST(PM_VERBOSE) || !PmQuiet)
	 fprintf( str, PM_UNTAR_VERBOSE, tarFile );
      else
	 fprintf( str, PM_UNTAR, tarFile );
   }
   fprintf( str, "\n" );

   pm_notes_make_hierarchy( str, PmNotesDir );
   fprintf( str, "for i in *%s *%s.gz ; do\n", PmNotesExt, PmNotesExt );
   fprintf( str, "  if [ -f $i ]; then\n");
   fprintf( str, "    mv -f $i %s\n", PmNotesDir );
   fprintf( str, "  fi\n");
   fprintf( str, "done\n");
   if (PmUseGroup && PmPkgGroup) {
      const char *partial = pm_file_path( 2, PmSourceDir, PmPkgGroup );
      pm_notes_make_hierarchy( str, partial );
      fprintf( str, "mv -f * %s\n", partial );
      xfree( (char *)partial );
   } else {
      pm_notes_make_hierarchy( str, PmSourceDir );
      fprintf( str, "mv -f * %s\n", PmSourceDir );
   }

   fprintf( str, "cd %s\n", PmTmp );
   fprintf( str, "rmdir %s.dir\n", tmp );
   fprintf( str, "exit 0\n" );
   fclose( str );
   pm_chmod( tmp, 0700 );
   if (!PmQuiet) printf( "Unpacking %s\n", tarFile );
   exitStatus = pm_shell( tmp );
   pm_unlink( tmp );

   xfree( tmp );
   pm_notes_free( list );
   xfree( (char *)tarFile );
   return exitStatus;
}

static void test_installation( PmEntryList entries )
{
   PmEntryList p;
   int         count = 0;
   
   if (PmForceInstall) return;
   pm_db_check_previous_version();

   for (p = entries; p && p->entry; p = p->next) {
      PmEntry e = p->entry;
      
      count += pm_db_check_previous_file( e->name );
   }

   if (count)
      pm_fatal( PMERR_INSTCONF,
		" (%d file%s already installed)\n",
		count, count == 1 ? "" : "s" );
}

static void pre_install_script( List list )
{
   char *tmp = tempnam( PmTmp, PgmName );
   int  exitStatus;

   if (pm_notes_script( list, PM_PREINSTALL, tmp, NULL )) {
      pm_chmod( tmp, 0700 );
      PRINTF(PM_VERBOSE,("(Running pre-installation script)\n"));
      exitStatus = pm_shell( tmp );
   }
   pm_unlink( tmp );
}

static int rename_ok( const char *filename )
{
   struct data {
      const char *name;
      int        tail;		/* else, check beginning */
   };
   struct data data[] = {
      { "/dev/zero",   0 },	/* Needed to load ELF binaries */
      { "ld.so",       1 },
      { NULL,          0 }
   };
   struct data *pt;
   char        *p;

   for (pt = data; pt->name; ++pt) {
      if (pt->tail) {
	 if ((p = strstr( filename, pt->name ))
	     && strlen( p ) == strlen( pt->name )) return 0; /* Don't rename */
      } else {
	 if (!strcmp( filename, pt->name )) return 0; /* Don't rename */
      }
   }
   return 1;			/* Rename ok */
}

static void rename_files( PmEntryList entries )
{
   PmEntryList p;
   char        *basename;
   char        buffer[MAXPATHLEN];

   for (p = entries; p && p->entry; p = p->next) {
      PmEntry e = p->entry;

      if (!(basename = strrchr( e->name, '/' ))) basename = e->name;
      else                                       ++basename;

				/* Check for shared libraries */
      if ((!strncmp( basename, "lib", 3 )
	   || !strncmp( basename, "ld-linux.so.", 12))
	  && strstr( basename, ".so." )) {
	 char        directory[MAXPATHLEN];
	 char        *pt;
	 struct stat sb;

	 strcpy( directory, e->name );
	 if ((pt = strrchr( directory, '/' ))) *pt        = '\0';
	 else                                  *directory = '\0';

	 if (!lstat( e->name, &sb )) {
	    if (S_ISLNK( sb.st_mode )) {
	       if (!PmQuiet) printf( "* Removing symlink: %s\n", e->name );
	       pm_unlink( e->name );
	    } else {
	       char major[MAXPATHLEN];
	       char majorBasename[MAXPATHLEN];
	    
	       if (!PmQuiet)
		  printf( "* Renaming shared library: %s\n", e->name );
	    
	       sprintf( buffer, "%s/%s-%s", directory, PM_OLD, basename );
	       strcpy( majorBasename, basename );
	       if ((pt = strchr( strstr( majorBasename, ".so." ) + 4, '.' ))) {
		  *pt = '\0';
		  sprintf( major, "%s/%s", directory, major );
		  PRINTF(PM_INSTALL,("(Rename %s to %s)\n",e->name,buffer));
		  pm_rename( e->name, buffer );
		  pm_unlink( major );
		  pm_symlink( buffer, major );
	       } else {
		  if (!PmQuiet)
		     printf( "* Cannot rename %s\n", e->name );
	       }
	    }
	 }
	 else printf( "** Cannot stat %s\n", e->name );
	 
	 ++PmNeedsLdconfig;
      } else if (rename_ok( e->name )) {
	 struct stat sb;
        
	 if (stat( e->name, &sb ) || !S_ISDIR( sb.st_mode )) {
	    /* Rename stale links, but don't rename directories.  */
	    sprintf( buffer, "%s.%s", e->name, PM_OLD );
	    PRINTF(PM_INSTALL,("(Rename %s to %s)\n",e->name,buffer));
	    pm_rename( e->name, buffer );
	 }
      } else {
	 PRINTF(PM_INSTALL,("(Do *NOT* rename %s)\n",e->name));
      }
   }
}

static int install_files( Params *p )
{
   int           totalFiles = pm_entry_list_count( p->entries );
   unsigned long totalBytes = pm_entry_list_size( p->entries );
   int           files      = 0;
   int           bytes      = 0;
   PmEntryList   pt         = p->entries;
   int           hashes     = 0;
   int           error      = 0;
   char          oldwd[MAXPATHLEN];
   
   static void printer( const char *line )
      {
	 int percent;

	 PRINTF(PM_INSTALL,("%s",line));

	 if (!strncmp( line, "tar: ", 5 )) {
	    printf( "%s", line );
	    ++error;
	    return;
	 }
	 
	 if (++files > totalFiles) return; /* finished */
	 bytes += pt->entry->size;

	 percent = +(bytes * 100) / totalBytes;
	 
	 if (!PmQuiet && PmHash) {
	    int i;
				/* Output some fraction of 50 hashes. */
	    
	    for (i = hashes; i < percent / 2; i++)
	       printf( "#" );
	    hashes = percent / 2;
	    
	 } else if (!PmQuiet) {
				/* A front-end can just parse 3
                                   space-delimited strings and forget about
                                   the rest. */
	    printf( "%3d%% %9d %s",
		    percent,
		    pt->entry->size,
		    pt->entry->name );
	    switch (pt->entry->kind) {
	    case PM_KIND_SYMLINK:
	       printf( " -> %s", pt->entry->link );
	       break;
	    case PM_KIND_CHAR:
	       printf( " (char special: %d, %d)",
		       major( pt->entry->device ),
		       minor( pt->entry->device ) );
	       break;
	    case PM_KIND_BLOCK:
	       printf( " (block special: %d, %d)",
		       major( pt->entry->device ),
		       minor( pt->entry->device ) );
	       break;
	    }
	    if (pt->entry->kind != PM_KIND_SYMLINK && pt->entry->link)
	       printf( " link to %s", pt->entry->link );
	    printf( "\n" );
	 }
	 if (pt->next) pt = pt->next;
	 fflush(stdout);
      }

   getcwd( oldwd, MAXPATHLEN );
   chdir( "/" );		/* We are in a chroot'd child! */
   if (!PmQuiet) {
      pm_notes_header( stdout, NULL );
      printf( "* %s: %d files, %lu bytes\n", /* FIXME!  CANONICAL */
	      pm_gen_canonical( NULL ), totalFiles, totalBytes );
   }
   if (pm_untar( p->tarFile, p->tarFileFD, p->compressed, printer ))
      pm_fatal( PMERR_UNTAR, "%s\n", p->tarFile );
   if (PmHash) printf( "\n" );
   chdir( oldwd );
   return error;
}

/* delete_old_files deals with the config files, and builds a fileList and
   dispList for later use by update_datebase. */

static void delete_old_files( PmEntryList entries,
			      List fileList,
			      List dispList )
{
   PmEntryList p;
   char        buffer[MAXPATHLEN];
   char        installedDisp[2]    = { PMDB_DISP_INSTALLED, '\0' };
   char        notinstalledDisp[2] = { PMDB_DISP_NOTINSTALLED, '\0' };

   pm_db_open( 1 );

   for (p = entries; p && p->entry; p = p->next) {
      PmEntry e = p->entry;

      sprintf( buffer, "%s.%s", e->name, PM_OLD );

      if (e->type == PM_TYPE_CONFIG && !access( buffer, R_OK )) {
	 const char *origCksum  = pm_db_get_checksum( e->name );
	 CheckSum   cksum       = pm_checksum_file( buffer );
	 const char *oldCksum   = pm_checksum_tostring( cksum );

	 PRINTF(PM_INSTALL,("(Orig = %s, Old = %s, New = %s)\n",
			    origCksum, oldCksum, e->checksum ));
	 
	 if (strcmp( origCksum, oldCksum )) {
				/* User modified file! */
	    if (!strcmp( origCksum, e->checksum )) {
				/* Original files are the same, so we can
                                   use the old file. */
	       
	       PRINTF(PM_INSTALL,("(Rename %s -> %s)\n",buffer,e->name));
	       pm_rename( buffer, e->name );
	       
	       pm_list_add( fileList, e->name );
	       pm_list_add( dispList, notinstalledDisp );
	    } else {
				/* Original files are differnet, so the
                                   default configuration has changed.  In
                                   this case, don't delete the .PM_OLD file,
                                   but note that the old file (with the new
                                   ".PM_OLD" name) is part of the previous
                                   package. */
	       pm_db_rename( e->name, buffer );
	       pm_list_add( fileList, e->name );
	       pm_list_add( dispList, installedDisp );
	    }
	 } else {
				/* User didn't modify it, so just replace
                                   the old with the new. */
	    pm_list_add( fileList, e->name );
	    pm_list_add( dispList, installedDisp );
	    
	    PRINTF(PM_INSTALL,("(Unlink %s)\n",buffer));
	    pm_unlink( buffer );
	    pm_db_delete_file( buffer ); /* In case this happened before. */
	 }

	 xfree( (char *)origCksum );
	 xfree( (char *)oldCksum );
	 pm_checksum_free( cksum );
	 
      } else {
	 pm_list_add( fileList, e->name );
	 pm_list_add( dispList, installedDisp );

	 PRINTF(PM_INSTALL,("(Unlink %s)\n",buffer));
	 pm_unlink( buffer );
	 pm_db_delete_file( buffer ); /* In case this happened before. */
      }
   }

   pm_db_close();
}

static void update_database( PmEntryList entries,
			     List fileList,
			     List dispList )
{
   pm_db_install_package( entries, fileList, dispList );
}

static void post_install_script( List list )
{
   char *tmp = tempnam( PmTmp, PgmName );
   int  exitStatus;
   
   if (pm_notes_script( list, PM_POSTINSTALL, tmp, NULL )) {
      pm_chmod( tmp, 0700 );
      PRINTF(PM_VERBOSE,("(Running post-installation script)\n"));
      exitStatus = pm_shell( tmp );
   }
   pm_unlink( tmp );
}

static int full_install( void *param )
{
   Params      *p       = (Params *)param;
   List        list     = p->list;
   PmEntryList entries  = p->entries;
   List        fileList = pm_list_create();
   List        dispList = pm_list_create();

   PRINTF(PM_TIME,("(Installation preparation: %ld mS)\n",pm_delta_time()));
   
   PRINTF(PM_INSTALL,("(Verifying ability to install)\n"));
   test_installation( entries );
   PRINTF(PM_TIME,("(Verified ability to install: %ld mS)\n",pm_delta_time()));

   if (PmTestFunction) return 0;
   
   PRINTF(PM_INSTALL,("(Checking for pre-installation script)\n"));
   pre_install_script( list );
   PRINTF(PM_TIME,("(Pre-installation script: %ld mS)\n",pm_delta_time()));

   PRINTF(PM_INSTALL,("(Renaming old files)\n"));
   rename_files( entries );
   PRINTF(PM_TIME,("(Renaming old files: %ld mS)\n",pm_delta_time()));

   PRINTF(PM_INSTALL,("(Installing new files)\n"));
   if (install_files( p )) return 1;
   PRINTF(PM_TIME,("(Installing new files: %ld mS)\n",pm_delta_time()));

   PRINTF(PM_INSTALL,("(Deleting old files)\n"));
   delete_old_files( entries, fileList, dispList );
   PRINTF(PM_TIME,("(Deleting old files: %ld mS)\n",pm_delta_time()));
   
   PRINTF(PM_INSTALL,("(Updating database)\n"));
   update_database( entries, fileList, dispList );
   PRINTF(PM_TIME,("(Updating database: %ld mS)\n",pm_delta_time()));
   
   PRINTF(PM_INSTALL,("(Checking for post-installation script)\n"));
   post_install_script( list );
   PRINTF(PM_TIME,("(Post-installation script: %ld mS)\n",pm_delta_time()));
   
   PRINTF(PM_INSTALL,("(Installation complete)\n"));

   pm_list_free( fileList );
   pm_list_free( dispList );

   return 0;
}

int install( const char *name )
{
   int         source;
   const char  *tarFile;
   int         exitStatus = 0;
   List        list;
   PmEntryList entries;
   Params      p;

   if (TEST(PM_TIME)) pm_reset_timer();
   tarFile = pm_resolve_tarfile( name, &source );
   if (source) return install_source( name, tarFile );

   list = pm_notes_read_tar( tarFile );
   pm_notes_parse( list );
   pm_notes_check( 0 );
   if (TEST(PM_INSTALL)) pm_notes_header( stdout, NULL );

   entries = pm_notes_files_dist( list );
   if (TEST(PM_INSTALL)) pm_entry_list_print( stdout, entries, 0 );

   if ((p.tarFileFD = open( tarFile, O_RDONLY )) < 0)
      pm_fatal( PMERR_READOPEN, "%s\n", tarFile );
   
   p.tarFile    = tarFile;
   p.compressed = pm_file_compressed( tarFile );
   p.list       = list;
   p.entries    = entries;
   if ((TEST(PM_NOCHROOT)) || !PmInstRoot || !strcmp(PmInstRoot, "/"))
      exitStatus = full_install( &p );
   else
      exitStatus = pm_chroot( PmInstRoot, full_install, &p );

   pm_notes_free( list );
   xfree( (char *)tarFile );
   pm_entry_list_free( entries );
   
   PRINTF(PM_TIME,("(Installation complete: %ld mS)\n",pm_total_time()));
   return exitStatus;
}
