/* resolve.c -- Resolve filenames
 * Created: Mon May  8 14:34:05 1995 by r.faith@ieee.org
 * Revised: Sun Jul 14 02:07:33 1996 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: resolve.c,v 1.9 1996/10/13 03:31:01 faith Exp $
 * 
 */

#include "pmlib.h"

static int check( const char *filename )
{
   if (!filename || !*filename) return 0;
   
   if (!access( filename, R_OK )) {
      PRINTF(PM_SEARCH,("(%s is readable)\n",filename));
      return 1;
   }
   PRINTF(PM_SEARCH,("(%s is not readable)\n",filename));
   return 0;
}

   static void do_strip( char *t, char *buffer )
      {
	 int        i;
	 char       *end;
	 const char *p;
	 int        len;
	 
	 for (end = buffer; *end; ++end); /* find end */

	 len = strlen( t );
	 if (end - buffer < len) return;

	 for (p = t, i = len; i; --i, ++p)
	    if (*p != end[-i]) return;
	 end[-len] = 0;
      }

static void strip( char *buffer, const char *tail )
{
   char       *tailBuf = alloca( strlen( tail ) + 5 );

   strcpy( tailBuf, tail );
   do_strip( tailBuf, buffer );
   strcat( tailBuf, ".gz" );
   do_strip( tailBuf, buffer );
}

/* Extract directories from a colon-separated list and perform
   substitutions if appropriate. */

const char *pm_get_partial( const char *paths )
{
   static const char *currentPaths = NULL;
   static char       pathsBuffer[MAXPATHLEN];
   static char       *pathsPt;
   static char       tmpBuffer[MAXPATHLEN];
   char              *retval;
   char              *pt;

tail_recurse:
   if (!currentPaths || currentPaths != paths) {
      if (!(currentPaths = paths)) return NULL;
      strcpy( pathsBuffer, currentPaths );
      pathsPt = pathsBuffer;
   }

   if (!(retval = pathsPt)) return NULL;
   
   if ((pt = strchr( pathsPt, ':' ))) {
      *pt = '\0';
      pathsPt = pt + 1;
   } else
      pathsPt = NULL;

   if (*retval == '$') {
      char *target, *replacement;

      if (!PmCurrentNotesDir) goto tail_recurse;

      target = retval + 1;
      if (!(pt = strchr( target, '$' ))) goto tail_recurse;
      *pt = '\0';
      
      replacement = pt + 1;
      if (!(pt = strchr( replacement, '$' ))) goto tail_recurse;
      *pt = '\0';

      PRINTF(PM_SEARCH,("(Replacing %s with %s in %s)\n",
			target,replacement,PmCurrentNotesDir));
      if (!(pt = strstr( PmCurrentNotesDir, target ))) goto tail_recurse;
      
      strncpy( tmpBuffer, PmCurrentNotesDir, pt - PmCurrentNotesDir );
      tmpBuffer[ pt - PmCurrentNotesDir ] = '\0'; 
      strcat( tmpBuffer, replacement );
      strcat( tmpBuffer, pt + strlen( target ) );
      
      return tmpBuffer;
   }

   return retval;
}


/* Resolve a partial notes file name into a complete path to a notes file. */

const char *pm_resolve_notesfile( const char *name )
{
   static char buffer[MAXPATHLEN] = { 0 };
   static char *cache = NULL;
   const char  *path;
   char        *end;
   const char  *basename;
   const char  *retval;
   const char  *partial;

   static const char *lcheck( void )
      {
	 if (check( buffer )) return xstrdup( buffer );
	 
				/* temporarily add extensions */
	 for (end = buffer; *end; ++end);
	 strcat( buffer, PmNotesExt );
	 if (check( buffer )) return xstrdup( buffer );
	 strcat( buffer, ".gz" );
	 if (check( buffer )) return xstrdup( buffer );
	 *end = '\0';
	 
	 strcat( buffer, ".gz" );
	 if (check( buffer )) return xstrdup( buffer );
	 *end = '\0';

	 return 0;
      }
   
   if ((basename = strrchr( name, '/' )))               ++basename;
   if (!basename && (basename = strrchr( name, ' ' )))  ++basename;
   if (!basename && (basename = strrchr( name, '\t' ))) ++basename;
   if (!basename) basename = name;
   
   if (!cache) {
      cache = xstrdup( basename );
   } else {
      if (!strcmp( cache, basename ) && *buffer) {
	 PRINTF(PM_SEARCH,("(cached %s for %s)\n",
			   buffer,basename));
	 return xstrdup( buffer );
      }
      xfree( cache );
      cache = xstrdup( basename );
      buffer[0] = '\0';
   }
   PRINTF(PM_SEARCH,("(searching for %s)\n",basename));
   
				/* Look where it is supposed to be */
   pm_get_partial( NULL );
   while ((partial = pm_get_partial( PmNotesDir ))) {
      path = pm_file_path( 2, partial, basename );
      strcpy( buffer, path );
      strip( buffer, PmBinExt );
      strip( buffer, PmSrcExt );
      xfree( (char *)path );
      if ((retval = lcheck())) {
	 if (PmCurrentNotesDir) xfree( (char *)PmCurrentNotesDir );
	 PmCurrentNotesDir = xstrdup( partial );
	 return retval;
      }
   }

   pm_fatal( PMERR_NONOTES, "%s\n", name );
   return NULL;			/* not reached */
}

const char *pm_resolve_tarfile( const char *name, int *source )
{
   static char buffer[MAXPATHLEN] = { 0 };
   static char *cache = NULL;
   const char  *path;
   char        *end;
   const char  *basename;
   const char  *retval;
   const char  *partial;

   static const char *lcheck( void )
      {
	 if (check( buffer )) return xstrdup( buffer );
	 
				/* temporarily add extensions */
	 for (end = buffer; *end; ++end);
	 
	 strcat( buffer, PmBinExt );
	 if (check( buffer )) return xstrdup( buffer );
	 *end = '\0';
	 
	 strcat( buffer, PM_BINARY_EXT );
	 if (check( buffer )) return xstrdup( buffer );
	 *end = '\0';
	 
	 strcat( buffer, PmSrcExt );
	 if (check( buffer )) return xstrdup( buffer );
	 *end = '\0';
	 
	 strcat( buffer, PM_SOURCE_EXT );
	 if (check( buffer )) return xstrdup( buffer );
	 *end = '\0';

	 return 0;
      }

   static void analyze( const char *b )
      {
	 if (strstr( b, PmSrcExt )) *source = 1;
	 else                       *source = 0;
      }

   if ((basename = strrchr( name, '/' )))               ++basename;
   if (!basename && (basename = strrchr( name, ' ' )))  ++basename;
   if (!basename && (basename = strrchr( name, '\t' ))) ++basename;
   if (!basename) basename = name;
   
   if (!cache) {
      cache = xstrdup( basename );
   } else {
      if (!strcmp( cache, basename ) && *buffer) {
	 PRINTF(PM_SEARCH,("(cached %s for %s)\n",
			   buffer,basename));
	 return xstrdup( buffer );
      }
      xfree( cache );
      cache = xstrdup( basename );
      buffer[0] = '\0';
   }
   PRINTF(PM_SEARCH,("(searching for %s)\n",basename));

				/* Look at the absolute path */
   strcpy( buffer, name );
   strip( buffer, PmNotesExt );
   if ((retval = lcheck())) {
      analyze( buffer );
      return retval;
   }
	  
   
				/* Look in the distribution directories */
   pm_get_partial( NULL );
   while ((partial = pm_get_partial( PmBinDistDir ))) {
      path = pm_file_path( 2, partial, basename );
      strcpy( buffer, path );
      strip( buffer, PmNotesExt );
      xfree( (char *)path );
      if ((retval = lcheck())) {
	 analyze( buffer );
	 return retval;
      }
   }
   
   pm_get_partial( NULL );
   while ((partial = pm_get_partial( PmSrcDistDir ))) {
      path = pm_file_path( 2, partial, basename );
      strcpy( buffer, path );
      strip( buffer, PmNotesExt );
      xfree( (char *)path );
      if ((retval = lcheck())) {
	 analyze( buffer );
	 return retval;
      }
   }

   pm_fatal( PMERR_NOTAR, "%s\n", basename );
   return NULL;			/* not reached */
}

const char *pm_resolve_patchfile( const char *name )
{
   static char buffer[MAXPATHLEN] = { 0 };
   static char *cache = NULL;
   const char  *path;
   const char  *basename;
   const char  *retval;
   const char  *partial;

   static const char *lcheck( void )
      {
	 char *end;
	 
	 strip( buffer, ".gz" );
	 if (check( buffer )) return xstrdup( buffer );
	 strip( buffer, ".Z" );
	 if (check( buffer )) return xstrdup( buffer );
	 strip( buffer, ".z" );
	 if (check( buffer )) return xstrdup( buffer );
	 
				/* temporarily add extensions */
	 for (end = buffer; *end; ++end);
	 
	 strcat( buffer, ".gz" );
	 if (check( buffer )) return xstrdup( buffer );
	 *end = '\0';

	 strcat( buffer, PmPatchExt );
	 if (check( buffer )) return xstrdup( buffer );
	 strcat( buffer, ".gz" );
	 if (check( buffer )) return xstrdup( buffer );
	 *end = '\0';
	 
	 return 0;
      }
   
   if (!name) pm_fatal( PMERR_NOPATCH, "\"\"\n" );

   if ((basename = strrchr( name, '/' )))               ++basename;
   if (!basename && (basename = strrchr( name, ' ' )))  ++basename;
   if (!basename && (basename = strrchr( name, '\t' ))) ++basename;
   if (!basename) basename = name;

   if (!cache) {
      cache = xstrdup( basename );
   } else {
      if (!strcmp( cache, basename ) && *buffer) {
	 PRINTF(PM_SEARCH,("(cached %s for %s)\n",
			   buffer,basename));
	 return xstrdup( buffer );
      }
      xfree( cache );
      cache = xstrdup( basename );
      buffer[0] = '\0';
   }
   PRINTF(PM_SEARCH,("(searching for %s)\n",basename));

   if (PmCurrentNotesDir) {
      path = pm_file_path( 2, PmCurrentNotesDir, basename );
      strcpy( buffer, path );
      xfree( (char *)path );
      if ((retval = lcheck())) return retval;
   }
   
   pm_get_partial( NULL );
   while ((partial = pm_get_partial( PmSourceDir ))) {
      if (PmPkgGroup && PmUseGroup) {
	 path = pm_file_path( 3, partial, PmPkgGroup, basename );
	 strcpy( buffer, path );
	 xfree( (char *)path );
	 if ((retval = lcheck())) return retval;
      }
      
      path = pm_file_path( 2, partial, basename );
      strcpy( buffer, path );
      xfree( (char *)path );
      if ((retval = lcheck())) return retval;
      
      if (PmPkgGroup && !PmUseGroup) {
	 path = pm_file_path( 3, partial, PmPkgGroup, basename );
	 strcpy( buffer, path );
	 xfree( (char *)path );
	 if ((retval = lcheck())) return retval;
      }
   }
   
   pm_fatal( PMERR_NOPATCH, "%s\n", basename );
   return NULL;			/* not reached */
}

const char *pm_resolve_sourcefile( const char *name )
{
   static char buffer[MAXPATHLEN] = { 0 };
   static char *cache = NULL;
   const char  *path;
   const char  *basename;
   const char  *retval;
   const char  *partial;

   static const char *lcheck( void )
      {
	 strip( buffer, ".gz" );
	 if (check( buffer )) return xstrdup( buffer );
	 strip( buffer, ".Z" );
	 if (check( buffer )) return xstrdup( buffer );
	 strip( buffer, ".z" );
	 if (check( buffer )) return xstrdup( buffer );

	 strcat( buffer, ".gz" );
	 if (check( buffer )) return xstrdup( buffer );
	 strip( buffer, ".gz" );
	 if (check( buffer )) return xstrdup( buffer );
	 strcat( buffer, ".Z" );
	 if (check( buffer )) return xstrdup( buffer );
	 strcat( buffer, ".z" );
	 if (check( buffer )) return xstrdup( buffer );

	 return 0;
      }

   if (!name) pm_fatal( PMERR_NOSOURCE, "\"\"\n" );

				/* FIXME! This fails if whitespace
				   is at the end of the name!!!!!! */
   if ((basename = strrchr( name, '/' )))               ++basename;
   if (!basename && (basename = strrchr( name, ' ' )))  ++basename;
   if (!basename && (basename = strrchr( name, '\t' ))) ++basename;
   if (!basename) basename = name;

   if (!cache) {
      cache = xstrdup( basename );
   } else {
      if (!strcmp( cache, basename ) && *buffer) {
	 PRINTF(PM_SEARCH,("(cached %s for %s)\n",
			   buffer,basename));
	 return xstrdup( buffer );
      }
      xfree( cache );
      cache = xstrdup( basename );
      buffer[0] = '\0';
   }
   PRINTF(PM_SEARCH,("(searching for %s)\n",basename));

   
   pm_get_partial( NULL );
   while ((partial = pm_get_partial( PmSourceDir ))) {
      if (PmPkgGroup && PmUseGroup) {
	 path = pm_file_path( 3, partial, PmPkgGroup, basename );
	 strcpy( buffer, path );
	 xfree( (char *)path );
	 if ((retval = lcheck())) return retval;
      }
      
      path = pm_file_path( 2, partial, basename );
      strcpy( buffer, path );
      xfree( (char *)path );
      if ((retval = lcheck())) return retval;

      if (PmPkgGroup && !PmUseGroup) {
	 path = pm_file_path( 3, partial, PmPkgGroup, basename );
	 strcpy( buffer, path );
	 xfree( (char *)path );
	 if ((retval = lcheck())) return retval;
      }
   }

   pm_fatal( PMERR_NOSOURCE, "%s\n", basename );
   return NULL;			/* not reached */
}

const char *pm_create_distpath( const char *path, const char *name )
{
   pm_get_partial( NULL );
   return pm_file_path( 2, pm_get_partial( path ), name );
}
