/* notes.c -- Notes file support
 * Created: Thu May  4 08:29:39 1995 by r.faith@ieee.org
 * Revised: Sat Oct 12 23:01:03 1996 by faith@cs.unc.edu
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: notes.c,v 1.12 1996/10/13 03:31:00 faith Exp $
 * 
 */

#include "pmlib.h"
#include "notes.h"

/* Ftp is the same as a Var, except that only the basename will be used
   when environment variables are created.

    0 = Not required
    1 = Required, always
    2 = Overwrite at build time -- these are all special cased below
   -2 = Must *not* have at build time

   Also, %distfiles is used explicitly in notes-copy.c. */


Macros macros[] = {
   { "Description:",  1, Var, 1, &PmPkgDescription                           },
   { "Name:",         1, Var, 1, &PmPkgName                                  },
   { "Version:",      1, Var, 1, &PmPkgVersion                               },
   { "Copyright:",    1, Var, 0, &PmPkgCopyright                             },
   { "Release:",      0, Var, 1, &PmPkgRelease                               },
   { "Subname:",     -2, Var, 1, &PmPkgSubName                               },
   { "Compiler:",     2, Var, 0, &PmPkgCompiler                              },
   { "Libraries:",    2, Var, 0, &PmPkgLibraries                             },
   { "CompiledBy:",   2, Var, 0, &PmPkgCompiledBy                            },
   { "DateCompiled:", 2, Var, 0, &PmPkgDateCompiled                          },
   { "Distribution:", 2, Var, 0, &PmPkgDistribution                          },
   { "Source",        0, Ftp, 1, PmPkgSource, 0, 0, 0, ":", PM_MAXSOURCE     },
   { "Patch",         0, Ftp, 1, PmPkgPatch,  0, 0, 0, ":", PM_MAXPATCH      },
   { "Group:",        0, Var, 1, &PmPkgGroup                                 },
   { "%interpreter",  0, Var, 0, &PmPkgInterpreter                           },
   { "%prep",         0, Sect,0,0, PM_PREP                                   },
   { "%build",        0, Sect,0,0, PM_BUILD,      0,0,0,0, 1                 },
   { "%install",      0, Sect,0,0, PM_INST,       0,0,0,0, 1                 },
   { "%package",      0, Sect,0,0, PM_PACK                                   },
   { "%pre",          0, Sect,0,0, PM_PREINSTALL                             },
   { "%post",         0, Sect,0,0, PM_POSTINSTALL                            },
   { "%preun",        0, Sect,0,0, PM_PREDELETE                              },
   { "%postun",       0, Sect,0,0, PM_POSTDELETE                             },
   { "%files",        1, Sect,0,&PmPkgHaveFiles,     PM_FILES                },
   { "%distfiles",   -2, Sect,0,&PmPkgHaveDistFiles, PM_DISTFILES            },
   { "%setup",        0, Macro, 0, 0, 0, pm_macro_setup                      },
   { "%patch",        0, Macro, 0, 0, 0, pm_macro_patch                      },
   { "%doc",          0, Macro, 0, 0, 0, pm_macro_doc,  pm_macro_doc_files   },
				/* Warning.  %config is hardcoded in
                                   notes-script.c:pm_move_config_files. */
   { "%config",       0, Macro, 0, 0, 0, 0,             pm_macro_config      },
   { "%addinfo",      0, Macro, 0, 0, 0, pm_macro_addinfo                    },
   { "%delinfo",      0, Macro, 0, 0, 0, pm_macro_delinfo                    },
   { "%addline",      0, Macro, 0, 0, 0, pm_macro_addline                    },
   { "%delline",      0, Macro, 0, 0, 0, pm_macro_delline                    },
   { 0, 0, 0 }
};

/* Basically, this routine finds macros from the above list.  If length and
   idx are not NULL, and if the type of the macro is Ftp, then idx returns
   the number of the macro.  So, "Source4:" would set idx to 4 and length
   to the length, but to the ':'.  This will vary, for example, between
   "Source2:" and "Source10:". */

Macros *pm_find_macro( const char *line, int *length, int *idx )
{
   char       buffer[PM_BUFFERSIZE];
   const char *s;
   char       *d;
   Macros     *pt;
   
   for (s = line, d = buffer; *s && *s != ' ' && *s != '\t';) *d++ = *s++;
   *d = '\0';
   for (pt = macros; pt->name; pt++) {
      if (pt->type == Ftp) {
	 int len = strlen( pt->name );
	 
	 if (!strncmp( buffer, pt->name, len )) {
	    char *p = buffer + len;

	    if (length) *length = len;
	    if (idx)    *idx    = atoi( p );
	    
	    while (*p && isdigit( *p )) {
	       if (length) ++*length;
	       ++p;
	    }
	    if (*p && !strcmp( p, pt->suffix )) {
	       if (length) *length += strlen( pt->suffix );
	       return pt;
	    }
	 }
      } else if (!strcmp( buffer, pt->name )) {
	 if (length) *length = strlen( pt->name );
	 return pt;
      }
   }
   return NULL;
}


/* Print the header information to the file stream str. */

void pm_notes_header( FILE *str, const char *subName )
{
   Macros *pt;

   for (pt = macros; pt->name; pt++) {
      if (pt->type == Var) {
	 if (pt->var == &PmPkgSubName && subName && *subName)
	    fprintf( str, "%s %s\n", pt->name, subName );
	 else if (*pt->var)
	    fprintf( str, "%s %s\n", pt->name, *pt->var );
      } else if (pt->type == Ftp && pt->var) {
	 int i;

	 for (i = 0; i < pt->max; i++) {
	    if (pt->var[i]) {
	       if (i)
		  fprintf( str, "%s%d%s %s\n", pt->name, i, pt->suffix,
			   pt->var[i] );
	       else
		  fprintf( str, "%s%s %s\n", pt->name, pt->suffix,
			   pt->var[i] );
	    }
	 }
      }
   }
}

/* Read a notes file into a List structure. */

List pm_notes_read( const char *filename )
{
   return pm_list_file( filename );
}

/* Read a notes file from a tar file into a List structure.  We take the
   first file with the extension PmNotesExt and hope it is the right
   one. . . */

List pm_notes_read_tar( const char *filename )
{
   FILE      *str   = pm_file_read( filename );
   TarHeader header = pm_tar_alloc_header();
   int       extLen;
   List      list   = NULL;
   int       retcode;

   if (!PmNotesExt)
      pm_fatal( PMERR_INTERNAL, "PmNotesExt is NULL\n" );
   extLen = strlen( PmNotesExt );

   while (!(retcode = pm_tar_next_header( str, header ))) {
      const char *name = pm_tar_filename( header );
      const char *end;
      int        len;
      
      for (end = name, len = 0; *end; ++end, ++len); /* find end, len */

      if ((len >= extLen && !strcmp( end - extLen, PmNotesExt ))
	  || (len >= extLen + 3
	      && !strncmp( end - extLen - 3, PmNotesExt, extLen )
	      && !strcmp( end - 3, ".gz" ))) {
	 list = pm_tar_listify( str, header );
	 break;
      }
   }

   pm_file_close( str );
   pm_tar_free_header( header );

   if (!list) switch (retcode) {
   case 0:  pm_fatal( PMERR_TAR, "%s, corrupt tar file\n", filename ); break;
   case 1:  pm_fatal( PMERR_TAR, "%s, bad checksum\n", filename );     break;
   case 2:
   case EOF:
      pm_fatal( PMERR_TAR, "%s, no %s file found\n", filename, PmNotesExt );
      break;
   default: pm_fatal( PMERR_TAR, "%s, error %d\n", retcode );          break;
   }

   return list;
}

/* Parse the header information from the notes file. */

void pm_notes_parse( List list )
{
   Macros *previous = NULL;
   
   static int process( const char *line )
      {
	 int        lineLen = strlen( line );
	 int        nameLen;
	 int        newLen;
	 const char *tail;
	 Macros     *m;
	 int        idx;
	 
	 if (!(m = pm_find_macro( line, &nameLen, &idx ))) {
	    if (previous && isspace( *line )) {	/* Allow continuations */
	       for (tail = line; isspace( *tail ); tail++);
	       if (tail) {	/* Skip blank lines */
		  char *new = xmalloc( lineLen + strlen(*previous->var) + 2 );

		  strcpy( new, *previous->var );
		  strcat( new, "\n" );
		  strcat( new, line );
		  xfree( (char *)*previous->var );
		  *previous->var = new;
	       }
	    } else {
	       if (*line == '%')
		  pm_warning( PMERR_INVMACRO, "%s\n", line );
	    }
	    return 0;
	 }

	 newLen  = lineLen - nameLen;

	 assert( newLen >= 0 );
   
				/* Find tail, after whitespace */
	 for (tail = line + nameLen; isspace( *tail ); tail++);

	 switch (m->type) {
	 case Sect:
	    if (!m->var) break;
	    /* fall through */
	 case Var:
	    if (*m->var) xfree( (char *)*m->var );
	    if (newLen) *m->var = xstrdup( tail );
	    else        *m->var = xstrdup( "" ); /* For pm_notes_check() */
	    break;
	 case Ftp:
	    if (m->var[idx]) xfree( (char *)m->var[idx] );
	    if (newLen) m->var[idx] = xstrdup( tail );
	    else        m->var[idx] = NULL;
	    break;
	 case Macro:
	    break;
	 }
	 if (m->type == Var) previous = m;
	 else                previous = NULL; /* continuation not allowed */
	 return 0;
      }

   pm_list_iterate( list, process );
}

/* Free everything in the list */

void pm_notes_free( List list )
{
   Macros *pt;

   for (pt = macros; pt->name; pt++)
      if (pt->type == Ftp && pt->var) {
	 int i;

	 for (i = 0; i < pt->max; i++) if (pt->var[i]) {
	    xfree( (char *)pt->var[i] );
	    pt->var[i] = NULL;
	 }
      } else if (pt->var && *pt->var) {
	 xfree( (char *)*pt->var );
	 *pt->var = NULL;
      }
   pm_list_free( list );
}

void pm_notes_check( int flag )
{
   Macros *pt;

   for (pt = macros; pt->name; pt++) {
      switch (pt->required) {
      case 1:
	 if (pt->var && !*pt->var)
	    pm_fatal( PMERR_NOTES, "Missing %s field (required)\n", pt->name );
	 break;
      case 2:
	 if (flag == 2) {
	    if (pt->var == &PmPkgDateCompiled) {
	       if (*pt->var) xfree( (char *)*pt->var );
	       *pt->var = xstrdup( PmTimeString );
	    } else if (pt->var == &PmPkgDistribution) {
	       if (*pt->var) xfree( (char *)*pt->var );
	       if (PmDistribution) *pt->var = xstrdup( PmDistribution );
	       else                *pt->var = xstrdup( "unknown" );
	    } else {
	       char *env = pm_notes_env_name( pt->name );
	       char *value;
	       
	       if ((value = getenv( env ))) {
		  if (*pt->var) xfree( (char *)*pt->var );
		  *pt->var = xstrdup( value );
	       } else if (pt->var == &PmPkgCompiledBy) {
		  if (*pt->var) xfree( (char *)*pt->var );
		  *pt->var = xstrdup( pm_get_email_name() );
	       }

	       xfree( env );
	    }
	 }
	 break;
      case -2:
	 if (flag == 2 && pt->var && *pt->var) {
	    pm_fatal( PMERR_NOTES,
		      "%s field is invalid during a build\n",
		      pt->name );
	 }
	 break;
      }
   }
}

void pm_notes_build_ancillary( List type, List field )
{
   Macros *pt;

   for (pt = macros; pt->name; pt++) {
      if (pt->type == Var && *pt->var && *pt->var[0]) {
	 pm_list_add( type, pt->name );
	 pm_list_add( field, *pt->var );
      } else if (pt->type == Ftp && pt->var) {
	 int i;

	 for (i = 0; i < pt->max; i++) {
	    if (pt->var[i]) {
	       char buffer[BUFSIZ];
	       
	       if (i)
		  sprintf( buffer, "%s%d%s", pt->name, i, pt->suffix );
	       else
		  sprintf( buffer, "%s%s", pt->name, pt->suffix );
	       pm_list_add( type, buffer );
	       pm_list_add( field, pt->var[i] );
	    }
	 }
      }
   }
}
