/* notes-script.c -- Create a script from a notes file
 * Created: Mon May  8 14:43:38 1995 by r.faith@ieee.org
 * Revised: Mon Nov 11 21:37:31 1996 by faith@cs.unc.edu
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: notes-script.c,v 1.21 1996/11/12 02:58:21 faith Exp $
 * 
 */

#include "pmlib.h"
#include "notes.h"

				/* Most modern Unices support mkdir -p */
#define USE_MKDIR_P

static int macroDocFound;

static const char *script_type( unsigned int level )
{
   static char buffer[BUFSIZ];
   char        *p;

   buffer[0] = '\0';
   
   if (level & PM_PREP)        strcat( buffer, "Prep/"   );
   if (level & PM_BUILD)       strcat( buffer, "Build/"   );
   if (level & PM_INST)        strcat( buffer, "Install/" );
   if (level & PM_PACK)        strcat( buffer, "Package/" );
   if (level & PM_PREINSTALL)  strcat( buffer, "Pre-install/" );
   if (level & PM_POSTINSTALL) strcat( buffer, "Post-install/" );
   if (level & PM_PREDELETE)   strcat( buffer, "Pre-uninstall/" );
   if (level & PM_POSTDELETE)  strcat( buffer, "Post-uninstall/" );
   if (level & PM_FILES)       strcat( buffer, "Files/" );
   if (level & PM_DISTFILES)   strcat( buffer, "DistFiles/" );

   for (p = buffer; *p; p++);	/* find end */
   sprintf( p, "Level-%08x", level );
   
   return buffer;
}

static void process_macro( Macros *m, const char *line, FILE *str )
{
   if (!m->script)
      pm_fatal( PMERR_INTERNAL, "No function for \"%s\" macro\n", m->name );
   m->script( m, line, str );
}

static void untar( FILE *str, const char *name )
{
   const char *path;

   path = pm_resolve_sourcefile( name );
   PRINTF(PM_SEARCH,("(Using %s as source tar file)\n",path));

   switch (pm_file_compressed( path )) {
   case PM_COMPRESSED:
      if (TEST(PM_VERBOSE)) fprintf( str, PM_UNTAR_COMPRESSED_VERBOSE, path );
      else                  fprintf( str, PM_UNTAR_COMPRESSED, path );
      break;
   case PM_ZIPPED:
      if (TEST(PM_VERBOSE)) fprintf( str, PM_UNZIP_VERBOSE, path );
      else                  fprintf( str, PM_UNZIP, path );
      break;
   default:
      if (TEST(PM_VERBOSE)) fprintf( str, PM_UNTAR_VERBOSE, path );
      else                  fprintf( str, PM_UNTAR, path );
      break;
   }
   fprintf( str, "\n" );
   xfree( (char *)path );
}

static void patch( FILE *str, const char *name, int strip, const char *filter,
		   const char *options )
{
   const char *path;

   path = pm_resolve_patchfile( name );

   if (pm_file_compressed( path )) {
      if (filter) fprintf( str,
			   PM_PATCH_COMPRESSED_FILTER,
			   path, filter, options, strip );
      else        fprintf( str, PM_PATCH_COMPRESSED, path, options, strip );
   } else {
      if (filter) fprintf( str, PM_PATCH_FILTER,
			   filter, path, options, strip );
      else        fprintf( str, PM_PATCH, options, strip, path );
   }
   fprintf( str, "\n" );
   xfree( (char *)path );
}

/* Output the shell commands necessary to guarantee a hierarchy exists. */

void pm_notes_make_hierarchy( FILE *str, const char *path )
{
#ifndef USE_MKDIR_P
   char       *buffer = alloca( strlen( path ) + 1 );
   const char *s;
   char       *d;
#endif

#ifdef USE_MKDIR_P
   fprintf( str, "mkdir -p %s;\n", path );
#else
   for (s = path, d = buffer; *s;) {
      if (*s == '/' && s != path) {
	 *d = '\0';
	 fprintf( str,
		  "if [ ! -d %s ]; then\n"
		  "   mkdir %s;\n"
		  "fi\n",
		  buffer,
		  buffer );
      }
      *d++ = *s++;
   }
   fprintf( str,
	    "if [ ! -d %s ]; then\n"
	    "   mkdir %s;\n"
	    "fi\n",
	    path,
	    path );
#endif
}

/* Make an environment variable name.  This is simplistic, especially if it
   isn't used to do anything fancy, but it does provide a lot of
   flexibility for the future. */

char *pm_notes_env_name( const char *name )
{
   char *p;
   char *env = xmalloc( strlen( name ) + 4 );
   char *pt  = env;

   *env = '\0';

#if 0
				/* Be sure to change config.c if you change
                                   this. */
   strcpy( env, "PM_" );
   pt = env + 3;
#endif
   
   strcat( pt, name );
   for (p = pt; *p; *p++ = toupper( *p ));
   if (pt[ strlen( pt ) - 1 ] == ':') pt[ strlen( pt ) - 1 ] = '\0';
   if (!isalpha(*pt)) memmove( pt, pt+1, strlen( pt+1 ) );

   return env;
}

static void pm_notes_env( FILE *str, int resolve_sources_flag )
{
   Macros     *pt;
   char       buffer[BUFSIZ];
   char       *slash, *env;
   const char *path;

   for (pt = macros; pt->name; pt++) {
      if (!pt->export) continue;
      if (pt->type == Ftp && pt->var) {
	 int i;
	 
	 env = pm_notes_env_name( pt->name );
	 for (i = 0; i < pt->max; i++) {
	    if (pt->var[i]) {
	       if ((slash = strrchr( pt->var[i], '/' ))
		   || (slash = strrchr( pt->var[i], ' '))
		   || (slash = strrchr( pt->var[i], '\t'))) {
		  fprintf( str, "# Original from: %s\n", pt->var[i] );
		  strcpy( buffer, slash + 1 );
	       } else
		  strcpy( buffer, pt->var[i] );
	       
	       path = NULL;
	       if (resolve_sources_flag) {
		  if (pt->var == PmPkgSource)
		     path = pm_resolve_sourcefile( buffer );
		  else if (pt->var == PmPkgPatch)
		     path = pm_resolve_patchfile( buffer );
	       }
	       
	       if (i) {
		  fprintf( str, "export %s%d=\"%s\"\n", env, i, buffer );
		  if (path) {
		     fprintf( str, "export %s%dPATH=\"%s\"\n", env, i, path );
		  }
	       } else {
		  fprintf( str, "export %s=\"%s\"\n", env, buffer );
		  if (path) {
		     fprintf( str, "export %sPATH=\"%s\"\n", env, path );
		  }
	       }
	       if (path) xfree( (char *)path );
	    }
	 }
	 xfree( env );
      } else if (pt->type == Var) {
	 env = pm_notes_env_name( pt->name );
	 strcpy( buffer, env );
	 xfree( env );
	 strcat( buffer, "=\"" );

				/* Give variables default values, since we
                                   often assume that they exist and the
                                   mandatory variables list has not been
                                   decided at this time. */
	 if (*pt->var) strcat( buffer, *pt->var );
	 else          strcat( buffer, "unknown" );
	 strcat( buffer, "\"" );
	 fprintf( str, "export %s\n", buffer );
      }
   }
}

static void set_buildsubdir( List list )
{
   static char *PmBuildSubDirSave = (char *)NULL;
    
   static int search( const char *line )
      {
	 Macros *m;

	 if ((m = pm_find_macro( line, NULL, NULL ))) {
	    if (m->type == Macro && m->script == pm_macro_setup) {
	       List args = pm_argify( line );
	       int  c;

	       optarg = NULL;
	       optind = 0;
	       while ((c = getopt( args->count,
				   args->lines, "cn:DTb:a:" )) != EOF)
		  if (c == 'n') {
		     if (PmBuildSubDir) xfree( (char *)PmBuildSubDir );
		     PmBuildSubDir = xstrdup( optarg );
		  }
	       pm_list_free( args );
	    }
	 }
	 return 0;
      }

   if (!PmBuildSubDirSave) {
      PmBuildSubDirSave = xstrdup( PmBuildSubDir );
   } else {
      if (PmBuildSubDir) xfree( (char *)PmBuildSubDir );
      PmBuildSubDir = xstrdup( PmBuildSubDirSave );
   }
   pm_list_iterate( list, search );
}

/* This implements the %setup macro:

   -n name ........... Set the name of the build directory (BUILDDIR) to
                       name.  This is used by --clean.  The default is
                       $NAME-$VERSION.  Other possibilities include $NAME,
                       ${NAME}${VERSION}, or whatever else the main tar
                       file uses.

   -c ................ Create the and cd to the named directory before
                       doing the untar.

   -b # .............. Untar Source# _before_ cd'ing into the directory.
                       This doesn't make sense with -c.

   -a # .............. Untar Source# _after_ cd'ing into the directory.

   -T ................ The default action is to untar "Source" (i.e.,
                       Source0), either before the cd (without -c), or
                       after the cd (with -c).  This flag inhibits this
                       default action and will require a "-b 0" or "-a 0"
                       to get the main source file untar'ed.

   -D ................ Do _not_ delete the directory before unpacking.
                       This is only useful if there are more than one
                       %setup command and this is _not_ the first of those
                       command.  Of course, multiple %setup commands should
                       be very, very uncommon.

   -C ................ Do _not_ recursively change owner and mode.  This is
                       useful if there are mutiple %setup commands.

 */

void pm_macro_setup( Macros *m, const char *line, FILE *str )
{
   List args       = pm_argify( line );
   int  c;
   int  createFlag = 0;
   int  deleteFlag = 1;
   int  tarFlag    = 1;
   int  cleanFlag  = 1;
   List before     = pm_list_create();
   List after      = pm_list_create();

   static int do_untar( const char *which )
      {
	 int value = atoi( which );

	 if (!value || value >= PM_MAXSOURCE)
	    pm_fatal( PMERR_INVSETUP, "\"%s\"\n", line );
	 untar( str, PmPkgSource[value] );
	 return 0;
      }

   optarg = NULL;
   optind = 0;
   while ((c = getopt( args->count, args->lines, "cn:DTCb:a:" )) != EOF)
      switch (c) {
      case 'c': ++createFlag;                      break;
      case 'n':
	 if (PmBuildSubDir) xfree( (char *)PmBuildSubDir );
	 PmBuildSubDir = xstrdup( optarg );
	 break;
      case 'C': cleanFlag = 0;                     break;
      case 'D': deleteFlag = 0;                    break;
      case 'T': tarFlag = 0;                       break;
      case 'b': pm_list_add( before, optarg );     break;
      case 'a': pm_list_add( after, optarg );      break;
      case '?':
	 pm_fatal( PMERR_INVSETUP, "`%c'\n", c );
      }
   
   fprintf( str, "\n# Setup macro (line = %s)\n", line );
   fprintf( str, "# SubDirectory = %s\n", PmBuildSubDir );
   if (deleteFlag) fprintf( str, "# Will remove old directory\n" );
   if (createFlag) fprintf( str, "# Will create directory and cd to it\n" );
   if (tarFlag)    fprintf( str, "# Will untar default SOURCE file\n" );

   pm_notes_make_hierarchy( str, PmBuildDir );
				/* If using groups, BUILDDIR != PmBuildDir */
#ifdef USE_MKDIR_P
   fprintf( str, "mkdir -p $BUILDDIR;\n" );
#else
   fprintf( str,
	    "if [ ! -d $BUILDDIR ]; then\n"
	    "   mkdir $BUILDDIR;\n"
	    "fi\n" );
#endif
   fprintf( str, "cd $BUILDDIR\n" );
   
   if (deleteFlag) fprintf( str, "rm -rf %s\n", PmBuildSubDir );
   
   if (createFlag) {
#ifdef USE_MKDIR_P
      fprintf( str, "mkdir -p %s;\n", PmBuildSubDir );
#else
      fprintf( str,
               "if [ ! -d %s ]; then\n"
               "   mkdir %s;\n"
               "fi\n",
               PmBuildSubDir,
	       PmBuildSubDir );
#endif
      fprintf( str, "cd %s\n", PmBuildSubDir );
   }
   
   if (!createFlag && tarFlag) untar( str, PmPkgSource[0] );
   pm_list_iterate( before, do_untar );
   if (!createFlag) fprintf( str, "cd %s\n", PmBuildSubDir );
   if (createFlag && tarFlag) untar( str, PmPkgSource[0] );
   pm_list_iterate( after, do_untar );
   fprintf( str, "\n" );

   fprintf( str, "cd $BUILDDIR/%s\n", PmBuildSubDir );
   if (cleanFlag) {
      fprintf( str,
	       "chown -R %s:%s .\n", pm_get_user_name(), pm_get_group_name() );
      fprintf( str, "chmod -R a+rX,g-w,o-w .\n" );
   }
   
   pm_list_free( args );
   pm_list_free( before );
   pm_list_free( after );
}

static void pm_macro_doc_init( void )
{
   macroDocFound = 0;
}

void pm_macro_doc( Macros *m, const char *line, FILE *str )
{
   List       files = pm_argify( line );
   int        first = 1;
   const char *path;

   static int copy( const char *filename )
      {
	 if (first) {
	    first = 0;
	    return 0;
	 }

	 if (strchr( filename, '/' )) {
	    char *buf = (char *)pm_file_path( 2, path, filename );
	    char *pt  = strrchr( buf, '/' );

	    if (pt) *pt = '\0';
	    pm_notes_make_hierarchy( str, buf );
	    fprintf( str, "cp -rp %s %s\n", filename, buf );
	    xfree( buf );
	 } else
	    fprintf( str, "cp -rp %s %s\n", filename, path );
	 return 0;
      }
   
   fprintf( str, "\n# Doc macro #%d (line = %s)\n", macroDocFound + 1, line );

   if (PmUseGroup) {
      path = pm_file_path( 3, PmDocDir, PmPkgGroup,
			   pm_gen_canonical( NULL ) );
      if (!macroDocFound) {
	 const char *partial = pm_file_path( 2, PmDocDir, PmPkgGroup );
	 pm_notes_make_hierarchy( str, partial );
	 xfree( (char *)partial );
      }
   } else {
      path = pm_file_path( 2, PmDocDir, pm_gen_canonical( NULL ) );
      if (!macroDocFound) pm_notes_make_hierarchy( str, PmDocDir );
   }
   
   if (!macroDocFound++) {
      fprintf( str, "rm -rf %s\n", path );
      fprintf( str, "mkdir %s\n", path );
   }

   pm_list_iterate( files, copy );
   fprintf( str,
	    "chown -R %s:%s %s\n",
	    pm_get_user_name(), pm_get_group_name(), path );
   fprintf( str, "chmod -R a+rX,g-w,o-w %s\n", path );
   fprintf( str, "\n" );
   
   xfree( (char *)path );
}

/* This implements the %patch macro:

   # ................. Apply Patch# as a patch.

   -p # .............. Specifies the number of directories to strip for the
                       patch(1) command.  This is usually 0 (the default)
                       or 1, but could be larger for brain-dead diff files.

   -P ................ The default action is to apply "Patch" (i.e.,
                       Patch0).  This flag inhibits this default action and
                       will require a "0" to get the main source file
                       untar'ed.  This option would be useful for a second
                       %patch macro that required a different "-p #" from
                       the first macro.

   -f filter ......... Specify a filter to preprocess the patch.  Using
                                  -f "sed 's/^- -/-/'"
		       is useful for filtering PGP signed email.

   -E ................ Pass the -E flag to patch(1).

 */
   

void pm_macro_patch( Macros *m, const char *line, FILE *str )
{
   List args      = pm_argify( line );
   int  c;
   int  strip     = 0;
   int  i;
   int  value;
   int  patchFlag = 1;
   int  eFlag     = 0;
   char *filter   = NULL;

   optarg = NULL;
   optind = 0;
   while ((c = getopt( args->count, args->lines, "p:Pf:e" )) != EOF)
      switch (c) {
      case 'p': strip = atoi( optarg ); break;
      case 'P': patchFlag = 0;          break;
      case 'f': filter = optarg;        break;
      case 'e': ++eFlag;                break;
      case '?':
	 pm_fatal( PMERR_INVPATCH, "`%c'\n", c );
      }
   
   fprintf( str, "\n# Patch macro (line = %s)\n", line );

   if (patchFlag) patch( str, PmPkgPatch[0], strip, filter,
			 eFlag ? "-E" : "" );
   
   for (i = optind; i < args->count; i++) {
       errno = 0;
       value = strtol( args->lines[i], (char **)NULL, 10);
      
       if (errno || value >= PM_MAXPATCH)
	    pm_fatal( PMERR_INVPATCH, "\"%s\"\n", args->lines[i] );
       patch( str, PmPkgPatch[value], strip, filter,
	      eFlag ? "-E" : "" );
   }
   fprintf( str, "\n" );

   pm_list_free( args );
}

void pm_macro_addinfo( Macros *m, const char *line, FILE *str )
{
   List list = pm_argify( line );
   
   fprintf( str, "\n# Addinfo macro (line = %s)\n", line );
   fprintf( str,
	    "if test -w /usr/info/dir"
	    " && ! fgrep '* %s' %s/dir > /dev/null 2>&1; then\n"
	    "    echo \"* %-29s %s\" >> %s/dir;\n"
	    "fi\n",
	    list->lines[1], PmInfoDir,
	    list->lines[1], list->lines[2], PmInfoDir );

   pm_list_free( list );
}

void pm_macro_delinfo( Macros *m, const char *line, FILE *str )
{
   List list = pm_argify( line );
   
   fprintf( str, "\n# Delinfo macro (line = %s)\n", line );
   fprintf( str,
	    "if test -w %s/dir"
	    " && ! fgrep '* %s' %s/dir > /dev/null 2>&1; then\n"
	    "   cp %s/dir /tmp/dir.$$\n"
	    "   fgrep -v '* %s' /tmp/dir.$$ > %s/dir\n"
	    "   rm -f /tmp/dir.$$\n"
	    "fi\n",
	    PmInfoDir,
	    list->lines[1], PmInfoDir,
	    PmInfoDir,
	    list->lines[1], PmInfoDir );
   
   pm_list_free( list );
}

void pm_macro_addline( Macros *m, const char *line, FILE *str )
{
   List       list = pm_argify( line );
   
   fprintf( str, "\n# Addline macro (line = %s)\n", line );
   fprintf( str,
	    "if test -w %s; then\n"
	    "    if ! grep '^%s$' %s 2>&1 > /dev/null; then\n"
	    "        echo '%s' >> %s;\n"
	    "    fi\n"
	    "fi\n",
	    list->lines[1],
	    list->lines[2], list->lines[1],
	    list->lines[2], list->lines[1] );
   
   pm_list_free( list );
}

void pm_macro_delline( Macros *m, const char *line, FILE *str )
{
   List list = pm_argify( line );
   
   fprintf( str, "\n# Delline macro (line = %s)\n", line );
   fprintf( str,
	    "if test -w %s; then\n"
	    "    cp %s /tmp/lines.$$;\n"
	    "    grep -v '^%s$' /tmp/lines.$$ > %s;\n"
	    "    rm -f /tmp/lines.$$;\n"
	    "fi\n",
	    list->lines[1],
	    list->lines[1],
	    list->lines[2], list->lines[1] );
   
   pm_list_free( list );
}

static void pm_move_config_files( List list, FILE *str, int direction )
{
   static int write_command( const char *line )
      {
	 if (direction) {
	    fprintf( str, "# Restore pre-install configuration files\n" );
	    fprintf( str,
		     "if test -f %s -a -f %s.PmSave; then\n"
		     "    mv -f %s %s.PmNew\n"
		     "    mv -f %s.PmSave %s\n"
		     "fi\n",
		     line+1, line+1, line+1,
		     line+1, line+1, line+1 );
	 } else {
	    fprintf( str, "# Move current configuration files for install\n" );
	    fprintf( str,
		     "if test -f %s; then\n"
		     "    mv -f %s %s.PmSave\n"
		     "fi\n",
		     line+1, line+1, line+1 );
	 }
	 return 0;
      }
   
   static int seek_config( const char *line )
      {
	 if (!strncmp( line, "%config ", 8 )) {
	    List files = pm_list_create();
	    
	    pm_macro_config( NULL, line, files );
	    pm_list_iterate( files, write_command );
	    pm_list_free( files );
	 }
	 return 0;
      }
   
   pm_list_iterate( list, seek_config );
}

/* Extract a script from the notes file in the specified list structure.
   Return the count of the sections actually copied.  This is useful for
   non-existant pre- and post- installation scripts. */

int pm_notes_script( List list,
		     unsigned int level,
		     const char *filename,
		     const char *packCommand )
{
   FILE   *str;
   int    copy        = 0;
   int    fakeLevel   = level;	/* To fake re-prep script */
   int    haveSection = 0;
   int    stageExists = 0;

   static int write_script( const char *line )
      {
	 Macros *m;

	 if ((m = pm_find_macro( line, NULL, NULL ))) {
	    if (m->type == Sect) {
	       if (m->stage & fakeLevel) {
		  copy = 1;
		  ++haveSection;
		  fprintf( str, "\n# \"%s\" commands:\n", m->name );
		  if (m->cd) {
		     fprintf( str, "# First, cd to top build directory\n" );
		     fprintf( str, "cd $BUILDDIR/%s\n\n", PmBuildSubDir );
		  }
	       }
	       else copy = 0;
	       PRINTF(PM_NOTES,
		      ("Stage = %08x, Level = %08x, copy = %d\n",
		       m->stage, fakeLevel, copy ));
	    } else if (copy && m->type == Macro) process_macro( m, line, str );
	 } else if (copy) fprintf( str, "%s\n", line );
	 
	 return 0;
      }

   static int check_script( const char *line )
      {
	 Macros *m;
	 
	 if ((m = pm_find_macro( line, NULL, NULL ))
	     && m->type == Sect
	     && m->stage & fakeLevel) {
	    ++stageExists;
	    return 1;
	 }
	 return 0;
      }

   pm_list_iterate( list, check_script );
   if (!stageExists) return 0;	/* Abort if non-existent */
   
   pm_macro_doc_init();		/* Reset flag to allow multiple %doc lines */
   
   str = pm_file_create( filename );

   if (PmPkgInterpreter) fprintf( str, "%s\n", PmPkgInterpreter );
   fprintf( str, "#!%s -e\n", PmBourneShell );
   fprintf( str, "# This is an automatically generated script\n" );
   fprintf( str, "# Generated by %s on %s\n", pm_version(), PmTimeString );
   fprintf( str, "# \"%s\" script for %s\n",
	    script_type( level ),
	    pm_gen_canonical( NULL ) );

   set_buildsubdir( list );

   fprintf( str, "\n# Package environment variables:\n" );
				/* Write notes environment variables */
   pm_notes_env( str,
		 (level & PM_PREP) || (level & PM_REPREP)
		 || (level & PM_BUILD) || (level & PM_PACKSOURCE) ); 
   
   fprintf( str, "\n# Configuration environment variables:\n" );
   pm_config_env( str );	/* Write config environment variables */

   if (TEST(PM_VERBOSE)) fprintf( str, "set -x\n" );
   
   fprintf( str, "\n# Script from %s file:\n", PmNotesExt );
   if (level & PM_INST) {
      fakeLevel = level & ~PM_INST;
      pm_list_iterate( list, write_script );
      fakeLevel = PM_PREINSTALL;
      pm_list_iterate( list, write_script );
      
      fakeLevel = PM_INST;
      pm_move_config_files( list, str, 0 ); /* Save %config files */
      pm_list_iterate( list, write_script );
      
      fakeLevel = PM_POSTINSTALL;
      pm_list_iterate( list, write_script );
   } else
      pm_list_iterate( list, write_script );

   if (level & PM_PACK) {
      fprintf( str, "\n# Pack binary distribution\n" );
      if (packCommand) fprintf( str, "%s\n", packCommand );
      else             fprintf( str, "# A pack command should go here\n" );
   }

   if (level & PM_INST)
      pm_move_config_files( list, str, 1 ); /* Restore %config files */

   if (level & PM_CLEAN) {
      fprintf( str, "\n# Clean up build directory\n" );
      fprintf( str, "cd $BUILDDIR\n" );
      fprintf( str, "rm -rf %s\n", PmBuildSubDir );
   }

   if (level & PM_REPREP) {
      fprintf( str, "\n# Re-prepare sources after build\n" );
      fakeLevel = PM_PREP;
      pm_list_iterate( list, write_script );
   }
   
   fprintf( str, "exit 0\n" );
   fclose( str );

   return haveSection;
}
