/* notes-files.c -- Support for file lists in notes files
 * Created: Mon May  8 15:07:15 1995 by r.faith@ieee.org
 * Revised: Mon Nov 11 21:00:08 1996 by faith@cs.unc.edu
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: notes-files.c,v 1.5 1996/11/12 02:58:20 faith Exp $
 * 
 */

#include "pmlib.h"
#include "notes.h"

void pm_macro_config( Macros *m, const char *line, List list )
{
   List files = pm_argify( line );
   int  first = 1;
   char buffer[MAXPATHLEN];

   static int append( const char *filename )
      {
	 if (first) {		/* skip macro name */
	    first = 0;
	    return 0;
	 }

	 buffer[0] = PM_CONFIG_FLAG;
	 strcpy( buffer + 1, filename );
	 pm_list_add( list, buffer );
	 PRINTF(PM_FILE,("Added \"%s\"\n",buffer));
	 return 0;
      }

   pm_list_iterate( files, append );
   pm_list_free( files );
}

void pm_macro_doc_files( Macros *m, const char *line, List list )
{
   List files = pm_argify( line );
   int first  = 1;

   static int expand( const char *filename )
      {
	 const char *path;
	 int        errval;
	 const char *pt;
	 
	 if (first) {		/* skip macro name */
	    first = 0;
	    return 0;
	 }

	 pt = filename;		/* Don't truncate any more */

	 if (PmUseGroup)
	    path = pm_file_path( 4, PmDocDir, PmPkgGroup,
				 pm_gen_canonical( NULL ), pt );
	 else
	    path = pm_file_path( 3, PmDocDir, pm_gen_canonical( NULL ), pt );
	 if ((errval = pm_file_expand( list, path, ' ' )))
	    pm_fatal( errval, "%s\n", path );
	 xfree( (char *)path );
	 return 0;
      }

   pm_list_iterate( files, expand );
   pm_list_free( files );
}

List pm_notes_files( List list, const char *subName )
{
   List       files           = pm_list_create();
   int        in_file_section = 0;

   static int find_files( const char *line )
      {
	 Macros     *m;
	 int        errval;
	 const char *target;

	 if ((m = pm_find_macro( line, NULL, NULL ))) {
	    if (m->type == Sect) {
	       List args = pm_argify( line );

	       if ((m->stage & PM_FILES)
		   && ((subName[0] == '\0' && args->count == 1)
		       || (args->count >= 2
			   && !strcmp( subName, args->lines[1] ))))
		  in_file_section = 1;
	       else
		  in_file_section = 0;
	       pm_list_free( args );
	    } else {
	       if (m->type == Macro && m->files) {
				/* Put the %doc files in all of the tar
                                   files.  This is reasonable, as they
                                   contain copyright and licensing
                                   information that should not be separated
                                   from any of the binaries.  If you want
                                   to separate them, then only do this
                                   macro-expansion when subName[0] ==
                                   '\0'. */
		  
		  if (m->files == pm_macro_doc_files)
		     m->files( m, line, files );
		  else if (in_file_section)
		     m->files( m, line, files );
	       }
	    }
	 } else if (in_file_section) {
	    char *pt;

				/* Remove trailing spaces */
	    for (pt = (char *)line; *pt; ++pt);
	    while (pt > line && isspace( pt[-1] )) --pt;
	    *pt = '\0';
	    
	    if (line[0] && line[0] != '/') {
	       char *tmp = alloca( strlen( line ) + 2 );
	       strcpy( tmp, "/" );
	       strcat( tmp, line );
	       target = tmp;
	    } else target = line;

	    if (*target) {	/* Skip blank lines */
	       PRINTF(PM_NOTES,("Expanding \"%s\"\n",target));
	       if ((errval = pm_file_expand( files, target, 0 )))
		  pm_fatal( errval, "%s\n", target );
	    }
	 }
	 return 0;
      }

   pm_list_iterate( list, find_files );
   return files;
}

List pm_notes_files_sections( List list )
{
   List       sections = pm_list_create();

   static int find_sections( const char *line )
      {
	 Macros *m;

	 if ((m = pm_find_macro( line, NULL, NULL ))) {
	    if (m->type == Sect) {
	       if (m->stage & PM_FILES) {
		  List args = pm_argify( line );

		  switch (args->count) {
		  case 1: pm_list_add( sections, "" );             break;
		  case 2: pm_list_add( sections, args->lines[1] ); break;
		  default:
		     pm_fatal( PMERR_FILESMAC, "line = %s\n", line );
		  }
		  pm_list_free( args );
	       }
	    }
	 }
	 return 0;
      }

   pm_list_iterate( list, find_sections );
   return sections;
}

PmEntryList pm_notes_files_dist( List list )
{
   PmEntryList entries = pm_entry_list_create();
   int         copy    = 0;

   static int find_entries( const char *line )
      {
	 Macros *m;

	 if ((m = pm_find_macro( line, NULL, NULL ))) {
	    if (m->type == Sect) {
	       if (m->stage & PM_DISTFILES) copy = 1;
	       else                         copy = 0;
	    }
	 } else 
	    if (copy)
	       pm_entry_list_add( entries, pm_entry_parse( line ) );
	 return 0;
      }

   pm_list_iterate( list, find_entries );
   return entries;
}
