/* file.c -- File testing support
 * Created: Thu May  4 18:51:52 1995 by r.faith@ieee.org
 * Revised: Wed Nov  1 16:15:56 1995 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 * This program comes with ABSOLUTELY NO WARRANTY.
 * 
 * $Id: file.c,v 1.5 1995/12/19 03:11:22 faith Exp $
 * 
 */

#include "pmlib.h"
#include <stdarg.h>
#include <glob.h>
#include <fcntl.h>

/* These ifdef's are derived from GNU autoconf documentation. */

#if HAVE_DIRENT_H
# ifdef __linux__
				/* Don't warn about shadowed variable */
#  define __select pm_some_arbitrary_name
# endif
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

/* These magic values are from GNU gzip 1.2.4 (gzip.h and lzw.h) */
#define PACK_MAGIC     "\037\036" /* Magic header for packed files */
#define GZIP_MAGIC     "\037\213" /* Magic header for gzip files, 1F 8B */
#define OLD_GZIP_MAGIC "\037\236" /* Magic header for gzip 0.5 = freeze 1.x */
#define LZH_MAGIC      "\037\240" /* Magic header for SCO LZH Compress files*/
#define LZW_MAGIC      "\037\235" /* Magic header for lzw files, 1F 9D */
#define PKZIP_MAGIC    "\120\113\003\004" /* Magic header for pkzip files */
 

int pm_buffer_compressed( const char *buf )
{
   if (!memcmp( PACK_MAGIC,     buf, 2 )) return PM_COMPRESSED;
   if (!memcmp( GZIP_MAGIC,     buf, 2 )) return PM_COMPRESSED;
   if (!memcmp( OLD_GZIP_MAGIC, buf, 2 )) return PM_COMPRESSED;
   if (!memcmp( LZH_MAGIC,      buf, 2 )) return PM_COMPRESSED;
   if (!memcmp( LZW_MAGIC,      buf, 2 )) return PM_COMPRESSED;
   if (!memcmp( PKZIP_MAGIC,    buf, 4 )) return PM_ZIPPED;
   
   return PM_UNCOMPRESSED;
}

int pm_file_compressed( const char *filename )
{
   FILE *str;
   char buf[4];
   int  i;
   int  retval = 0;

   if (!(str = fopen( filename, "r" ))) return 0;
   for (i = 0; i < 4; i++)
      if (fread( &buf[i], 1, 1, str ) != 1) {
	 fclose( str );
	 return 0;
      }
   fclose( str );

   retval = pm_buffer_compressed( buf );

   PRINTF(PM_SEARCH,("(%s %s compressed)\n",filename,retval?"is":"is not"));
   
   return retval;
}

int pm_file_nonempty_dir( const char *filename )
{
   struct stat   sb;
   DIR           *dir;
   struct dirent *dp;

   if (lstat( filename, &sb )) return 0;
   if (!S_ISDIR( sb.st_mode )) return 0;
   if (!(dir = opendir( filename ))) return 0;
   while ((dp = readdir( dir ))) {
				/* . and .. are present in empty dirs */
      if (dp->d_name[0] == '.'
	  && (NAMLEN( dp ) == 1
	      || (dp->d_name[1] == '.' && NAMLEN( dp ) == 2))) continue;
      closedir( dir );
      return 1;			/* nonempty */
   }
   closedir( dir );
   return 0;
}

const char *pm_file_path( int count, ... )
{
   va_list    ap;
   char       buffer[BUFSIZ];
   char       *end  = buffer;
   const char *next;

   va_start( ap, count );

   while (count--) {
      next = va_arg( ap, char * );
      *end = '\0';
      if (next) {
	 strcat( buffer, next );
	 for (; *end; end++); /* move to end */
	 if (end != buffer && end[-1] != '/') {
	    *end++ = '/';
	    *end = '\0';
	 }
      }
   }
   if (end != buffer && end[-1] == '/') end[-1] = '\0';

   PRINTF(PM_FILE,("Path is \"%s\"\n",buffer));
   return xstrdup( buffer );
}

const char *pm_file_readable( int count, ... )
{
   va_list    ap;
   char       buffer[BUFSIZ];
   char       *end  = buffer;
   const char *next;
   const char *retval = NULL;

   va_start( ap, count );

   while (count--) {
      next = va_arg( ap, char * );
      *end = '\0';
      if (next) {
	 strcat( buffer, next );
	 for (; *end; end++); /* move to end */
	 if (end != buffer && end[-1] != '/') {
	    *end++ = '/';
	    *end = '\0';
	 }
      }
   }
   if (end != buffer && end[-1] == '/') end[-1] = '\0';

   PRINTF(PM_FILE,("Testing \"%s\"\n",buffer));
   if (!access( buffer, R_OK )) retval = xstrdup( buffer );
   PRINTF(PM_SEARCH,("(%s %s readable)\n",buffer,retval?"is":"is not"));

   return retval;
}

/* We would use ftw(3) here, but ftw is so broken, it's easier to write our
   own tree walker.  Apparently, on some OS's d_name may not be NULL
   terminated.  Some techniques to handle this were derived from io/ftw.c
   in glibc-1.09. */

static void recurse( const char *filename, void (*f)( const char * ) )
{
   struct stat   sb;
   DIR           *dir;
   struct dirent *dp;
   int           len = strlen( filename );
   int           newLen;
   char          *buf;

   if (lstat( filename, &sb ))
      pm_fatal( PMERR_LSTAT, "%s\n", filename );

   if (!S_ISDIR( sb.st_mode )) {
      f( filename );
      return;
   }

   if (!(dir = opendir( filename )))
      pm_fatal( PMERR_DIROPEN, "%s\n", filename );

   while ((dp = readdir( dir ))) {
      assert( NAMLEN( dp ) > 0 );
      if (dp->d_name[0] == '.'
	  && (NAMLEN( dp ) == 1
	      || (dp->d_name[1] == '.' && NAMLEN( dp ) == 2))) continue;
      newLen = NAMLEN( dp ) + len + 1;
      if (newLen > MAXPATHLEN)
	 pm_fatal( PMERR_TOOLONG, "%s/%s\n", filename, dp->d_name );
      
				/* Use the heap to prevent stack overflow */
      buf = malloc( newLen + 1 );
      strcpy( buf, filename );
      buf[ len ] = '/';
      memcpy( buf + len + 1, dp->d_name, NAMLEN( dp ) );
      buf[ newLen ] = '\0';
      recurse( buf, f );
      xfree( buf );
   }

   closedir( dir );
}

int pm_file_expand( List list, const char *filename, int flag )
{
   int    flags = GLOB_PERIOD;
   glob_t globbuf;
   int    i;
   char   buffer[MAXPATHLEN];

   static void adder( const char *path )
      {
	 buffer[0] = flag ?: PM_DEFAULT_FLAG;
	 strcpy( buffer + 1, path );
	 pm_list_add( list, buffer );
	 PRINTF(PM_FILE,("Added \"%s\"\n",buffer));
      }

   while (isspace(*filename)) ++filename;
   PRINTF(PM_FILE,("Searching \"%s\"\n",filename));
   if (*filename == '\0'
       || (*filename == '/' && filename[1] == '\0')) return PMERR_NOMATCH;

   if (glob( filename, flags, NULL, &globbuf ))
      return PMERR_NOMATCH;
   for (i = 0; i < globbuf.gl_pathc; i++) {
      struct stat sb;
      char        *pt;

      if (!(pt = strrchr( globbuf.gl_pathv[i], '/' )))
	 pt = globbuf.gl_pathv[i];

      if (*pt && (!strcmp( pt + 1, "." ) || !strcmp( pt + 1, ".." ))) continue;
      
      adder( globbuf.gl_pathv[i] );
      if (lstat( globbuf.gl_pathv[i], &sb ))
	 pm_fatal( PMERR_LSTAT, "%s\n", globbuf.gl_pathv[i] );
      if (S_ISDIR( sb.st_mode ))
	 recurse( globbuf.gl_pathv[i], adder );
   }
   globfree( &globbuf );
   return 0;
}

/* Build a directory hierarchy right now. Don't mkdir for final name unless
   makeFinal is true, since it is probably a target filename, and not a
   directory name. */

void pm_file_hierarchy( const char *path, int makeFinal )
{
   char       *buffer = alloca( strlen( path ) + 1 );
   const char *s;
   char       *d;
   
   for (s = path, d = buffer; *s;) {
      if (*s == '/' && s != path) {
	 *d = '\0';
	 if (access( buffer, R_OK )) pm_mkdir( buffer, ~PmUmask & 0777 );
      }
      *d++ = *s++;
   }
   if (makeFinal) {
      if (access( path, R_OK )) pm_mkdir( path, ~PmUmask & 0777 );
   }
}

FILE *pm_file_create( const char *filename )
{
   FILE *str;

   pm_file_hierarchy( filename, 0 );
   if (!access( filename, R_OK )) pm_unlink( filename );
   if (!(str = fopen( filename, "w" )))
      pm_fatal( PMERR_WRITEOPEN, "%s\n", filename );

   PRINTF(PM_SEARCH,("(%s open for write)\n",filename));

   return str;
}

FILE *pm_file_read( const char *filename )
{
   FILE *str;

   if (pm_file_compressed( filename )) {
      int file;
      int fd;

      pm_child_clear();
      if ((file = open( filename, O_RDONLY )) < 0)
	 pm_fatal( PMERR_READOPEN, "%s\n", filename );
      fd = pm_gunzip_filter( file );
      if (!(str = fdopen( fd, "r" )))
	 pm_fatal( PMERR_READOPEN, "%s (fdopen)\n", filename );
   } else {
      if (!(str = fopen( filename, "r" )))
	 pm_fatal( PMERR_READOPEN, "%s\n", filename );
   }
   return str;
}

void pm_file_close( FILE *str )
{
   fclose( str );
   pm_child_wait();
}
