/* entry.c -- Handle database entry interface
 * Created: Wed May 10 11:34:20 1995 by r.faith@ieee.org
 * Revised: Tue Oct 17 09:08:55 1995 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: entry.c,v 1.2 1995/10/17 13:17:59 faith Exp $
 *
 * Note: Some of the hard link detection code was written while looking at
 * GNU Tar (prep.ai.mit.edu:/pub/gnu/tar-1.11.2.tar.gz), Copyright (C)
 * 1985, 1992, 1993 Free Software Foundation (Written 25 Aug 1985 by John
 * Gilmore, ihnp4!hoptoad!gnu).  GNU Tar may be distributed under the terms
 * of the GNU General Public License.
 *
 */

#include "pmlib.h"

struct link {
   char        *name;
   dev_t       device;
   ino_t       inode;
   struct link *next;
};

static int         initialized = 0;
static struct link *linklist   = NULL;

void pm_entry_init( void )
{
   if (initialized)
      pm_fatal( PMERR_INTERNAL, "pm_entry_init called twice\n" );
   ++initialized;
}

void pm_entry_shutdown( void )
{
   if (!initialized)
      pm_fatal( PMERR_INTERNAL,
		"pm_entry_shutdown called before pm_entry_init\n" );
   initialized = 0;
   while (linklist) {
      struct link *next = linklist->next;
      xfree( linklist->name );
      xfree( linklist );
      linklist = next;
   }
}

PmEntry pm_entry_create( const char *filename )
{
   const char    *truename = filename + 1;
   struct stat   sb;
   struct passwd *pw;
   struct group  *gr;
   CheckSum      cksum;
   PmEntry       entry = xmalloc( sizeof( struct PmEntry ) );
   char          buffer[MAXPATHLEN];
   int           size;

   if (lstat( truename, &sb ))
      pm_fatal( PMERR_LSTAT, "%s\n", truename );

   entry->name = xstrdup( truename );

   if (*filename == PM_CONFIG_FLAG) entry->type = PM_TYPE_CONFIG;
   else if (S_ISDIR( sb.st_mode ))  entry->type = PM_TYPE_DIRECTORY;
   else                             entry->type = PM_TYPE_REGULAR;

   /* We only handle the common things.  Stuff we don't know how to deal
      with is as follows:

      S_ISHIDDEN    Hidden directory (AIX)
      S_ISCDF       Context dependent files (HP/UX)
      S_ISNWK       Network special (HP/UX)
      S_ISCTG       Contiguous (Masscomp)

      If any of the weirder stuff is needed, the post-installation script
      should deal with it, not the archive program.
   */
      
   entry->kind = PM_KIND_UNKNOWN;
   if (S_ISREG( sb.st_mode ))  entry->kind = PM_KIND_REGULAR;
#ifdef S_ISLNK
   if (S_ISLNK( sb.st_mode ))  entry->kind = PM_KIND_SYMLINK;
#endif
#ifdef S_ISDIR
   if (S_ISDIR( sb.st_mode ))  entry->kind = PM_KIND_DIRECTORY;
#endif
#ifdef S_ISCHR
   if (S_ISCHR( sb.st_mode ))  entry->kind = PM_KIND_CHAR;
#endif
#ifdef S_ISBLK
   if (S_ISBLK( sb.st_mode ))  entry->kind = PM_KIND_BLOCK;
#endif
#ifdef S_ISFIFO
   if (S_ISFIFO( sb.st_mode )) entry->kind = PM_KIND_FIFO;
#endif
#ifdef S_ISSOCK
   if (S_ISSOCK( sb.st_mode )) entry->kind = PM_KIND_SOCKET;
#endif

   entry->size   = sb.st_size;
   entry->mode   = sb.st_mode;
   entry->mtime  = sb.st_mtime;
   entry->device = sb.st_rdev;

   if ((pw = getpwuid( sb.st_uid )))
      entry->user = xstrdup( pw->pw_name );
   else
      entry->user = xstrdup( "0" );

   if ((gr = getgrgid( sb.st_gid )))
      entry->group = xstrdup( gr->gr_name );
   else
      entry->group = xstrdup( "0" );

   if (entry->kind == PM_KIND_REGULAR) {
      cksum = pm_checksum_file( truename );
      entry->checksum = (char *)pm_checksum_tostring( cksum );
      pm_checksum_free( cksum );
   } else entry->checksum = NULL;

   entry->link = NULL;		/* because we free it later */
   
   switch (entry->kind) {
   case PM_KIND_SYMLINK:
      if ((size = readlink( truename, buffer, MAXPATHLEN )) < 0)
	 pm_fatal( PMERR_READLINK, "%s\n", truename );
      buffer[ size ] = '\0';
      entry->link = xstrdup( buffer );
      break;
   case PM_KIND_REGULAR:
   case PM_KIND_CHAR:
   case PM_KIND_BLOCK:
   case PM_KIND_FIFO:
   case PM_KIND_SOCKET:
      if (sb.st_nlink > 1) {
	 struct link *p;
				/* Both tar and cpio use a serial search of
                                   a linked list.  If speed is a concern,
                                   put a hash table here. */
	 
	 for (p = linklist; p; p = p->next) {
	    if (p->inode == sb.st_ino && p->device == sb.st_dev) {
	       entry->link = xstrdup( p->name );
	       goto end;	/* or just return entry here */
	    }
	 }
				/* link not found, so add one */
	 p = xmalloc( sizeof( struct link ) );
	 p->inode  = sb.st_ino;
	 p->device = sb.st_dev;
	 p->name   = xstrdup( entry->name );
	 p->next   = linklist;
	 linklist  = p;
      }
      break;
   }

end:
   return entry;
}

void pm_entry_free( PmEntry entry )
{
   xfree( entry->name );
   xfree( entry->user );
   xfree( entry->group );
   if (entry->checksum) xfree( entry->checksum );
   if (entry->link) xfree( entry->link );
   xfree( entry );
}

void pm_entry_write( FILE *str, PmEntry entry )
{
   fprintf( str,
	    "%s %c%c %lu 0%o %lu ",
	    entry->name,
	    entry->type,
	    entry->kind,
	    (unsigned long)entry->size,
	    entry->mode,
	    (unsigned long)entry->mtime );
   
   if (entry->kind == PM_KIND_BLOCK || PM_KIND_CHAR)
      fprintf( str, "%d,%d ", major( entry->device ), minor( entry->device ) );
   else
      fprintf( str, "0,0 " );

   fprintf( str, "%s %s ", entry->user, entry->group );

   if (entry->checksum) fprintf( str, "%s ", entry->checksum );
   else                 fprintf( str, "0 " );

   if (entry->link) fprintf( str, "%s\n", entry->link );
   else             fprintf( str, "\n" );
}

PmEntry pm_entry_parse( const char *line )
{
   PmEntry entry = xmalloc( sizeof( struct PmEntry ) );
   char    *buffer = alloca( strlen( line ) + 1 );
   char    *p;
   char    *end;
   int     devMajor;
   int     devMinor;

   strcpy( buffer, line );
   
   if ((p = strtok( buffer, " " ))) entry->name = xstrdup( p );
   else                             entry->name = NULL;
   
   if ((p = strtok( NULL, " " ))) {
      entry->type = p[0];
      entry->kind = p[1];
   }

   if ((p = strtok( NULL, " " ))) entry->size = strtol( p, NULL, 10 );
   if ((p = strtok( NULL, " " ))) entry->mode = strtol( p, NULL, 8 );
   if ((p = strtok( NULL, " " ))) entry->mtime = strtol( p, NULL, 10 );

   if ((p = strtok( NULL, " " ))) {
      devMajor = strtol( p, &end, 10 );
      devMinor = strtol( end + 1, NULL, 10 );
      entry->device = makedev( devMajor, devMinor );
   }

   if ((p = strtok( NULL, " " ))) entry->user = xstrdup( p );
   if ((p = strtok( NULL, " " ))) entry->group = xstrdup( p );
   if ((p = strtok( NULL, " " ))) entry->checksum = xstrdup( p );
   else                           entry->checksum = NULL;

   if ((p = strtok( NULL, " " ))) entry->link = xstrdup( p );
   else                           entry->link = NULL;

   return entry;
}

/* This should be done better, but the struct is < 16 bytes and 16 bytes is
   usually allocated from each call to malloc anyway, so there wouldn't be
   much memory savings and there wouldn't be any performance savings. */

PmEntryList pm_entry_list_create( void )
{
   PmEntryList list = xmalloc( sizeof( struct PmEntryList ) );

   list->entry = NULL;
   list->next  = NULL;
   list->tail  = NULL;
   return list;
}

void pm_entry_list_add( PmEntryList list, PmEntry entry )
{
   if (!list)
      pm_fatal( PMERR_INTERNAL, "pm_entry_list_add: list is NULL\n" );

   if (list->tail) {
      PmEntryList new = xmalloc( sizeof( struct PmEntryList ) );

      new->entry = entry;
      new->next  = NULL;
      new->tail  = NULL;

      list->tail->next = new;
      list->tail       = new;
   } else {			/* overwrite, since it's empty */
      list->entry = entry;
      list->next  = NULL;
      list->tail  = list;
   }
}

void pm_entry_list_free( PmEntryList list )
{
   while (list) {
      PmEntryList next = list->next;
      if (list->entry) pm_entry_free( list->entry );
      xfree( list );
      list = next;
   }
}

void pm_entry_list_print( FILE *str, PmEntryList list, int verbosity )
{
   while (list && list->entry) {
      switch (verbosity) {
      case 1:
	 fprintf( str, "%s\n", list->entry->name );
	 break;
      case 2:
	 fprintf( str,
		  "%s %lu\n",
		  list->entry->name,
		  (unsigned long)list->entry->size );
	 break;
      default:
	 pm_entry_write( str, list->entry );
	 break;
      }
      list = list->next;
   }
}

int pm_entry_list_count( PmEntryList list )
{
   int count = 0;
   
   while (list && list->entry) {
      ++count;
      list = list->next;
   }

   return count;
}

unsigned long pm_entry_list_size( PmEntryList list )
{
   unsigned int size = 0;
   
   while (list && list->entry) {
      size += list->entry->size;
      list = list->next;
   }

   return size;
}
