/* checksum.c -- Support for checksums
 * Created: Wed May  3 10:26:08 1995 by r.faith@ieee.org
 * Revised: Tue Oct 17 08:51:14 1995 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * $Id: checksum.c,v 1.3 1995/10/17 13:17:27 faith Exp $
 *
 * This code provides a few wrappers for Colin Plumb's Public Domain
 * implementation of the MD5 message-digest algorithm.  This implementation
 * is widely available on the Internet.  For more information on the MD5
 * algorithm, see Bruce Schneier's APPLIED CRYPTOGRAPHY: PROTOCOLS,
 * ALGORITHMS, AND SOURCE CODE IN C (New York: John Wiley & Sons, 1994,
 * pages 329--333).
 *
 * Note, however, that these routines provide a generic interface to a
 * checksum utility.  MD5 could be replaced by another algorithm.
 *
 */

#include "pmlib.h"
#include <fcntl.h>

#define CKSUMLEN 16

static CheckSum pm_checksum_malloc( void )
{
   return xmalloc( CKSUMLEN );
}

void pm_checksum_free( CheckSum cksum )
{
   xfree( cksum );
}

#if 0
CheckSum pm_checksum_stream( FILE *str )
{
   int               count;
   char              buffer[BUFSIZ];
   struct MD5Context ctx;
   CheckSum          cksum = pm_checksum_malloc();
   
   MD5Init( &ctx );
   while ((count = fread( buffer, 1, BUFSIZ, str )) > 0)
      MD5Update( &ctx, buffer, count );
   MD5Final( cksum, &ctx );

   return cksum;
}
#endif

CheckSum pm_checksum_file( const char *filename )
{
   int               fd;
   int               count;
   char              buffer[BUFSIZ*4];
   struct MD5Context ctx;
   CheckSum          cksum = pm_checksum_malloc();

   if (!(fd = open( filename, O_RDONLY )))
      pm_fatal( PMERR_READOPEN, "%s\n", filename );
   
   MD5Init( &ctx );
   while ((count = read( fd, buffer, BUFSIZ*4 )) > 0)
      MD5Update( &ctx, buffer, count );
   MD5Final( cksum, &ctx );

   close( fd );
   return cksum;
}

CheckSum pm_checksum_buffer( const char *buffer, unsigned length )
{
   struct MD5Context ctx;
   CheckSum          cksum = pm_checksum_malloc();
   
   MD5Init( &ctx );
   MD5Update( &ctx, buffer, length );
   MD5Final( cksum, &ctx );

   return cksum;
}

void pm_checksum_print( FILE *str, CheckSum cksum )
{
   int                 i;
   const unsigned char *p = (const unsigned char *)cksum;

   for (i = 0; i < CKSUMLEN; i++) fprintf( str, "%02x", *p++ );
}

const char *pm_checksum_tostring( CheckSum cksum )
{
   int                 i;
   const unsigned char *p     = (const unsigned char *)cksum;
   char                *buf   = xmalloc( CKSUMLEN * 2 + 1 );
   char                *bufpt = buf;

   for (i = 0; i < CKSUMLEN; i++) {
      sprintf( bufpt, "%02x", *p++ );
      bufpt += 2;
   }

   return buf;
}

static int hex( int c )
{
   if (!isxdigit( c )) return -1;

   if (c >= '0' && c <= '9') return c - '0';
   else if (islower( c )) return c - 'a' + 10;
   else return c - 'A' + 10;
}

CheckSum pm_checksum_fromstring( const char *string )
{
   int           i;
   CheckSum      cksum = pm_checksum_malloc();
   unsigned char *p    = (unsigned char *)cksum;
   const char    *pt   = string;
   int           d1;
   int           d2;

   for (i = 0; i < CKSUMLEN * 2; i++) {
      if ((d1 = hex( *pt++ )) == -1) return NULL;
      if ((d2 = hex( *pt++ )) == -1) return NULL;
      *p++ = d1 * 16 + d2;
   }

   return cksum;
}

int pm_checksum_compare( CheckSum cksum1, CheckSum cksum2 )
{
   int i;
   const unsigned char *p1 = (const unsigned char *)cksum1;
   const unsigned char *p2 = (const unsigned char *)cksum2;

   for (i = 0; i < CKSUMLEN; i++) if (*p1++ != *p2++) return 1;
   return 0;
}
