/* pmdb_delete_file.c, Copyright (c) 1995, Doug Hoffman
 * 
 * Program: pmdb_delete_file.c
 * Author : Doug Hoffman, hoffman@cybernetics.com
 * Purpose: c function to write file records to the pm database.
 * Created: Wed May  3 16:14:35 1995 by hoffman
 * Revised: Wed Nov  8 12:49:20 1995 by r.faith@ieee.org
 * 
 * Disclaimer:
 * 
 *   Copyright (c) 1995, Doug Hoffman
 *   This program comes with ABSOLUTELY NO WARRANTY.
 * 
 *   This program is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 * 
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 * 
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Abstract:
 * 
 * Description:
 * 
 * Modification History:
 * 
 * $Log: delete_file.c,v $
 * Revision 1.4  1995/11/08 17:57:04  faith
 * Use PMDB_DISP_* consistently
 *
 * Revision 1.3  1995/10/28  01:20:40  faith
 * Performance analysis and improvements
 *
 * Revision 1.2  1995/05/13  20:57:16  faith
 * Added const.  Added fields to records.  Added options to pmdb.
 *
 * Revision 1.1  1995/05/13  18:06:59  hoffman
 * Initial revision
 *
 * 
 */

#include "pmdb.h"

extern int	 PmdbDeleteCount;

/* =============== pmdb_delete_file
 *
 *  Function to close the datafiles for the pmdb
 *  database system. 
 *
 */

int pmdb_delete_file( const char *keyStr )
{
   datum	data;
   datum	key;
   datum	pkey;
   
   PMDBFileRec		record;   
   PMDBPackageRec	package;
   
   int		i;
   
   /* ----- Make sure the database is open */

   if (PMDBFile[PMDB_FILES].gdbmFile == NULL) return -1;

   /* ----- get key string */

   key.dsize = strlen( keyStr);
   key.dptr  = (char *)keyStr;

   /* ----- Make sure the package record exists
    *
    *  A file record must have a previously registered package record.
    *  The record is read and the file added to the packages file/disp
    *  list.
    *
    *  The first thing to do is get the file record.
    */

   data = gdbm_fetch( PMDBFile[PMDB_FILES].gdbmFile, key);
   if (NULL == data.dptr) {
      return gdbm_errno;
   }
   unpack_file_record( keyStr, &data, &record);
   free( data.dptr);

   /* -- find out which package it belongs to... */
   if ((record.field[PMDB_FR_PACKAGE] == NULL) ||
       (0 == (i = strlen( record.field[PMDB_FR_PACKAGE])))) return -2;

   pkey.dsize = i;
   pkey.dptr  = (char *)record.field[PMDB_FR_PACKAGE];
   
   data = gdbm_fetch( PMDBFile[PMDB_PACKAGES].gdbmFile, pkey);
   if (NULL == data.dptr) {
      return gdbm_errno;
   }

   /* -- update the file/disp list in the package record */
   unpack_package_record( record.field[PMDB_FR_PACKAGE], &data, &package);   
   free( data.dptr);
   i = mark_file_in_package( record.field[PMDB_FR_KEY],
			     &package, PMDB_DISP_DELETED);
   if (i == -1) {
      add_file_to_package( record.field[PMDB_FR_KEY], &package);
   }
   if (i != 0) {
      pack_package_record( &package, &data);
      gdbm_delete( PMDBFile[PMDB_PACKAGES].gdbmFile, pkey);
      PmdbDeleteCount += data.dsize;
      if (0 != (gdbm_store( PMDBFile[PMDB_PACKAGES].gdbmFile,
			    pkey, data, GDBM_REPLACE))) {
	 return gdbm_errno;
      }
   }
   /* -- free all the temporary storage */
   pmdb_free_package_rec( &package);   
   pmdb_free_file_rec( &record);
   free( data.dptr);

   /* ----- delete the file record.
    *
    *  At this point the package record has be updated to reflect the fact
    *  that the file has been deleted, if necessary, and all the temporary
    *  storage has been freed. All that remains is to actually delete the
    *  file record and exit.
    */

   gdbm_delete( PMDBFile[PMDB_FILES].gdbmFile, key);

   /* ----- see if we should compress the database.
    *
    *  Because rewrites cause ballooning database file sizes we need
    *  to occasionally call pmdb_reorganize to reclaim deleted record
    *  space.  Note that we only track deletes from the packages file
    *  because it is the main offender.  This is because of the large
    *  size of the records in this file. The other files will get reorged
    *  when ever the packages file gets it, so they are taken care of as
    *  well.
    *
    */

   if (PmdbDeleteCount > PMDB_DELETE_COUNT) {
      pmdb_reorganize();
      PmdbDeleteCount = 0;      
   }
   
   return 0;
} /* end pmdb_delete_file */

