/*
 * Bibliography:
 *
 * [1] _82365SL PC Card Interface Controller (PCIC)_, Intel
 * Corporation, Order Number 290423-002
 *
 * [2] PCMCIA Standards
 */

static const char *pcmcia_drv_c_rcsid = "$Id: pcmcia_drv.c,v 1.2 1993/09/07 20:53:37 bjaspan Exp $";

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>

#define REALLY_SLOW_IO
#define extern
#include <asm/io.h>
#undef extern

#include "pcmcia.h"

#define DEBUG

#define PHYS_TUPLE_SIZE 4096

#ifdef USER_DRV
extern int mem_fd;
#define static
#endif

struct pcic_sock pcic_socks[MAX_SOCKS];
static u_long phys_tuple_addr = 0;

void pcic_init(struct pcic_sock *);
void pcic_print_sock(struct pcic_sock *);
void pcic_print_cfent(struct pcmcia_cfentry *);
static void pcic_map_attr(struct pcic_sock *, u_long, int);
static void pcic_power(struct pcic_sock *, int);
static void pcic_reset(struct pcic_sock *);
static inline u_char pcic_readreg(struct pcic_sock *, u_short);
static inline void pcic_writereg(struct pcic_sock *, u_short, u_char);
static inline void pcic_writereg16(struct pcic_sock *, u_short, u_short);
static inline u_short pcic_readreg16(struct pcic_sock *, u_short);

int pcmcia_init(u_long mem_start, u_long mem_stop)
{
     int i;

     pcic_init(pcic_socks);

     phys_tuple_addr = mem_start;
     mem_start += PHYS_TUPLE_SIZE;

     for (i = 0; i < MAX_SOCKS; i++) {
	  if (pcic_socks[i].cardp) {
#ifdef USER_DRV
	       if (lseek(mem_fd, phys_tuple_addr, SEEK_SET) < 0) {
		    perror("seeking to tuple memory");
		    exit(1);
	       }
#endif /* USER_DRV */

	       pcic_power(&pcic_socks[i], 1);
	       pcic_reset(&pcic_socks[i]);
	       pcic_map_attr(&pcic_socks[i], phys_tuple_addr, 1);
	       parse_tuples(&pcic_socks[i], 0);
	       pcic_map_attr(&pcic_socks[i], phys_tuple_addr, 0);
#ifdef DEBUG
	       pcic_print_sock(&pcic_socks[i]);
#endif
	  }
     }

     return mem_start;
}

void pcic_init(struct pcic_sock *socks)
{
     int i;
     int base, off, id;

     memset((char *) socks, 0, sizeof(struct pcic_sock)*MAX_SOCKS);
     
     /* Magic numbers from [1], pp. 16-17 */
     for (i = 0; i < MAX_SOCKS; i++) {
	  socks[i].base = PCIC_BASE + ((i / 4) * 2);
	  socks[i].ofs = (i % 4)*0x40;	
	  
	  id = pcic_readreg(&socks[i], PCIC_ID_REV);
	  switch (id) {
	  case PCIC_INTEL0:
	  case PCIC_INTEL1:
	  case PCIC_IBM1:
	  case PCIC_IBM2:
	       break;
	  default:
	       continue;
	  }
	  socks[i].valid = id;
	  
	  /* Check for socketed card, just for kicks */
	  socks[i].cardp =
	       (pcic_readreg(&socks[i],PCIC_STATUS)&PCIC_CD)==PCIC_CD;
	  
#ifdef notdef
	  printf("socket %d found at %#3x, %scard inserted\n", i,
		 socks[i].base, socks[i].cardp ? "" : "no ");
#endif
     }
}

void pcmcia_select_cfentry(struct pcic_sock *sock, int idx)
{
     int i;
     char c;
     
     /* Enable attribute memory window */
     pcic_map_attr(sock, phys_tuple_addr, 1);

     /* power on card */
     pcic_power(sock, 1);

     /* reset card */
     pcic_reset(sock);
     
     c = idx & 0x3f;
     
#ifdef USER_DRV
     /* write config table entry index */
     if (lseek(mem_fd, phys_tuple_addr + sock->base_addr, SEEK_SET) < 0) {
	  perror("seeking to config register 0");
	  exit(1);
     }
     if (write(mem_fd, &c, 1) != 1) {
	  perror("writing config entry index to register 0");
	  exit(1);
     }

#else
     *((char *) phys_tuple_addr + sock->base_addr) = c;
#endif

     pcic_map_attr(sock, phys_tuple_addr, 0);

     /* wait for initialization XXX bit? */
     usleep(500);
}

static void pcic_reset(struct pcic_sock *sock)
{
     u_char i;
     
     i = pcic_readreg(sock, PCIC_INT_GEN);
     pcic_writereg(sock, PCIC_INT_GEN, 0xbf & i);
     usleep(100);
     pcic_writereg(sock, PCIC_INT_GEN, 0x40 | i);
}
		       
static void pcic_power(struct pcic_sock *sock, int on)
{
     if (on) {
	  on = PCIC_DISRST | PCIC_PCPWRE; /* | PCIC_APSENA */
	  pcic_writereg(sock, PCIC_POWER, on);
	  usleep(1000);    /* wait for power to come up.. XXX bit? */
	  on |= PCIC_OUTENA;
	  pcic_writereg(sock, PCIC_POWER, on);
	  usleep(1000); /* XXX probably not necessary */
     }
}

static void pcic_map_attr(struct pcic_sock *sock, u_long phys_base, int map)
{
     u_char wine;
     
     /* memory window 4 probably isn't used very often */

     phys_base >>= 12; 

     if (map) {
#ifdef notdef
	  printf("card mem offset %#x\n", (-phys_base & 0x3fff));
#endif
	  pcic_writereg16(sock, PCIC_SM4_STL, phys_base);
	  pcic_writereg16(sock, PCIC_SM4_SPL, phys_base);
	  pcic_writereg16(sock, PCIC_CM4_L, (-phys_base & 0x3fff) |
			  (PCIC_REG << 8));
     } else {
	  pcic_writereg16(sock, PCIC_SM4_STL, 0);
	  pcic_writereg16(sock, PCIC_SM4_SPL, 0);
	  pcic_writereg16(sock, PCIC_CM4_L, 0);
     }
     
     /* enable/disable window */
     wine = pcic_readreg(sock, PCIC_ADDRWINE);
     wine = (map ? (wine | PCIC_SM4_EN) : (wine & (~PCIC_SM4_EN)));
     pcic_writereg(sock, PCIC_ADDRWINE, wine);
}

void pcic_print_sock(struct pcic_sock *sock)
{
     int i;
     
     printf("PCIC socket %#x + %#x\n", sock->base, sock->ofs);
     if (!sock->valid)
	  printf("\tnot present\n");
     if (!sock->cardp)
	  printf("\tno card present\n");
     printf("\tVersion: %s, %s, %s, %s\n", sock->manufacturer,
	    sock->prod_name, sock->addl_info1, sock->addl_info2);
     printf("\tRegister offset %#x, mask 0x%04x%04x%04x%04x\n",
	    sock->base_addr, sock->regmask[3], sock->regmask[2],
	    sock->regmask[1], sock->regmask[0]);
#ifdef DEBUG
     printf("\tMax cfent entry %d\n", sock->config_midx);
#endif

     i = 0;
     while (i < MAX_CONFIGS) {
	  pcic_print_cfent(&sock->configs[i]);
	  if (sock->config_midx == sock->configs[i].idx)
	       break;
	  i++;
     }
     if (sock->config_midx != sock->configs[i].idx)
	  printf("\tLast config entry not present!\n");
}

void pcic_print_cfent(struct pcmcia_cfentry *cfent)
{
     int i;
     printf("\tConfig entry %d:\n", cfent->idx);
     printf("\t\tIRQ %d, mask %#x\n", cfent->irq, cfent->irq_mask);
     for (i = 0; i < cfent->ios; i++)
	  printf("\t\tI/O #%d: %#x->%#x (%d)\n", i, cfent->io_addrs[i],
		 cfent->io_addrs[i] + cfent->io_lens[i],
		 cfent->io_lens[i]);
     for (i = 0; i < cfent->mems; i++)
	  printf("\t\tMem #%d: %#x->%#x (%d) at %#x\n", i,
		 cfent->mem_caddrs[i],
		 cfent->mem_caddrs[i] + cfent->mem_lens[i], 
		 cfent->mem_lens[i], cfent->mem_haddrs[i]);
}

static inline void pcic_writereg(struct pcic_sock *sock, u_short addr,
				 u_char val)
{
#ifdef notdef
     printf("writing %#x to register %#x\n", val, sock->ofs + addr);
#endif
     outb_p(sock->ofs + addr, sock->base);
     outb_p(val, sock->base+1);
}

static inline u_char pcic_readreg(struct pcic_sock *sock, u_short addr)
{
     outb_p(sock->ofs + addr, sock->base);
     return inb_p(sock->base+1);
}

static inline void pcic_writereg16(struct pcic_sock *sock, u_short
				   addr, u_short val)
{
     pcic_writereg(sock, addr, val & 0xff);
     pcic_writereg(sock, addr+1, (val >> 8) & 0xff);
}

static inline u_short pcic_readreg16(struct pcic_sock *sock, u_short addr)
{
     return pcic_readreg(sock, addr) | (pcic_readreg(sock, addr+1) << 8);
}
