/* Ethercard enabler for the Databook TCIC/2 PCMCIA interface chip. */
/* Written 1994 by Donald Becker. */

#include <unistd.h>
#include <stdio.h>
#include <sys/file.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <asm/system.h>
#include <asm/io.h>

/* Default base address of the Databook TCIC/2 chip. */

#define	TCIC_BASE	(0x240)

/* Offsets of TCIC/2 registers from the TCIC/2 base address. */

enum tcic_offsets {
	TCIC_DATA = 0, TCIC_ADDR = 2, TCIC_SCTRL = 6, TCIC_SSTAT = 7,
	TCIC_MODE = 8, TCIC_PWR  = 9, TCIC_EDC = 10, TCIC_ICSR = 12,
	TCIC_IENA = 13, TCIC_AUX = 14};

/* Codes put into the top three bits of the TCIC_MODE register
   to select which auxiliary register is visiable at TCIC_AUX. */
enum auxreg_offsets {
	AUX_TCTL = 0 << 5, AUX_PCTL = 1 << 5, AUX_WCTL = 2 << 5,
	AUX_EXTERN = 3 << 5, AUX_PDATA = 4 << 5, AUX_SYSCFG = 5 << 5,
	AUX_ILOCK = 6 << 5, AUX_TEXT = 7 << 5 };

/* Mark that TCIC_ADDR points to internal registers (rather than into the
   card address space). */
#define	ADDR_H_INDIRECT		0x08000000

/* Bit definitions for selected fields (like just those that we use). */
#define	SSTAT_CARD		(0x80)	/* Card installed. */

/* Socket control register, TCIC_SCTRL. */
#define SCTRL_AUTOINC	0x18			/* Autoincrement after access. */
#define	SCTRL_ENB		(0x01)	/* Enable card access to selected soc */

/* Power control register */
#define	PWR_CURRENTL		(0x40)	/* Enable current limiting */
#define	PWR_VCCSEL0		(0x01)	/* 5 Volt supply control for sock 0 */
#define	PWR_VCCSEL1		(0x02)	/* 5 Volt supply control for sock 1 */

/* I/O map control register */
#define	IOCTL_EN		(0x8000) /* Enable this map */
#define	IOCTL_8BIT		(0x0800)
#define	IOCTL_16BIT		(0x0400)
#define	IOCTL_QUIET		(0x0040) /* Make the buffers quieter */
#define	IOCTL_ONEK		(0x0080) /* This map is 1k or less */

/* Interrupt control/status register */

#define	ICSR_WRITEALL		(3)	/* Write all bits 7:2 in CSR */

/* Interrupt enable register */

#define	IENA_SSTAT		(0x40)	/* Interrupt on any changed to SSTAT*/
#define	IENA_OPENDRAIN		(0x01)	/* Make STKIRQ output open drain */

/* Mode register */

#define	MODE_WORD		(0x10)	/* Mode register, word access */

/* Memory map control register */

#define	MCTL_EN			(0x8000) /* Mem map ctl reg, enable */
#define	MCTL_QUIET		(0x0040) /* Make accesses use quiet mode */

/* Memory map map register. */
#define	MMAP_ATTRIBUTE		(0x8000) /* Map this to card attribute space */

/* System configuration register 1 */
#define	SCF1_STATIO		(0x0800)

int mem_fd;

int main(int argc, char *argv[])
{
    /* This will probe for a TCIC/2 at the standard location. */
	int card_addr = 0x300;				/* Adaptor card I/O base. */
	int card_length = 16;
	int socket = 0;						/* Which socket to use. */
	int tcic = TCIC_BASE;				/* TCIC chip I/O base. */
	int irq = 10;
	int i;

	/* The 0x80 location is for the delay in the *_p() functions. */
	if (ioperm(tcic, 16, 1)				/* TCIC/2 locations. */
		|| ioperm(0x80, 1, 1)
		|| ioperm(card_addr, card_length, 1)) {
		perror("ioperm");
		fprintf(stderr, "This program must be run with root priveledges.\n");
		return 1;
	}
	if ((mem_fd = open("/dev/mem", O_RDWR, 0)) < 0) {
		perror("Opening /dev/mem");
		exit(1);
	}

	/* Verify that *something* is at the putative TCIC address. */
	outw(0xf00d, tcic + TCIC_ADDR);
	outw(0x0110, tcic + TCIC_ADDR + 2);
	if (inw(tcic + TCIC_ADDR) != 0xf00d) {
		printf("TCIC/2 not found at %#x.\n", tcic);
		return ENODEV;
	}

	if (inb(tcic + TCIC_SSTAT) & SSTAT_CARD) {
		printf("Found card in TCIC/2 (0x%x) in socket 0.\n", tcic);
		socket = 0;
	} else {
		printf("Found TCIC/2 (0x%x), but no card.\n", tcic);
		return ENODEV;
	}

	/* Select socket 0. */
	outl(0x0000, tcic + TCIC_ADDR);

#ifdef notdef
	outb(0x80, tcic + TCIC_SCTRL);    
	usleep(1000);
	outb(0x00, tcic + TCIC_SCTRL);    
	usleep(50000);

	printf("Turning off the power... ");
	/* Shut down, then turn on the card */
	outb(PWR_CURRENTL, tcic + TCIC_PWR);
	usleep(1000);
	printf("turning on power %2.2x.\n", PWR_VCCSEL0);
	outb(/*PWR_CURRENTL |*/ PWR_VCCSEL0, tcic + TCIC_PWR);
	usleep(20000);
#endif

	/* Disable the current socket and set autoincrement on data accesses. */
	outb(SCTRL_AUTOINC + 0x01, tcic + TCIC_SCTRL);

	/* Map the I/O space starting at 'card_addr' to the
	   socket specified by 'socket'.  Use 8-bit mapping, quiet mode, wait
	   state value of 7. */
	outl(0x08000200, tcic + TCIC_ADDR);
	outw(card_addr|(card_length >> 1), tcic + TCIC_DATA);
	outw(0x8447, tcic + TCIC_DATA);			/* Load I/O control register. */

	/* Load the system configuration auxiliary register*/
	outb(AUX_WCTL, tcic + TCIC_MODE);
	outw(0x0067, tcic + TCIC_AUX);
	outb(AUX_SYSCFG, tcic + TCIC_MODE);
	outw(0x4a00, tcic + TCIC_AUX);
	outb(AUX_ILOCK, tcic + TCIC_MODE);
#if defined(notdef) || 1
	outb(0xdc, tcic + TCIC_AUX);
	usleep(10000);
#endif
	outb(0xd8, tcic + TCIC_AUX);
	/* Give the chip 50 msecs. to reinitialize. */
	usleep(50000);

	outb(SCTRL_AUTOINC + 0x01, tcic + TCIC_SCTRL);

	/* Point to the socket configuration registers, and load them. */
	/* IR_SCF1 for socket 0. */
	outl(0x08000000, tcic + TCIC_ADDR);
	outw(0x7800 | irq, tcic + TCIC_DATA);
	outw(0x007b, tcic + TCIC_DATA);

	/* IR_SCF1 for socket 1. */
	outl(0x08000008, tcic + TCIC_ADDR);
	outw(0x0000, tcic + TCIC_DATA);
	outw(0x0079, tcic + TCIC_DATA);

	/* Map the attribute memory into 0xd0000. */
	/* Point to WR_MBASE_i */
	outl(0x08000100, tcic + TCIC_ADDR);
	outw(0, tcic);
	outw(0x4000 | 0x0d0, tcic);
	outw(0x8000 | ((-0x0d0)&0x0fff), tcic);
	outw(0x8007, tcic);

	outb(PWR_VCCSEL0, tcic + TCIC_PWR);
	usleep(10000);
	outb(SCTRL_AUTOINC + 0x01, tcic + TCIC_SCTRL);    
	outl(0x80000000, tcic + TCIC_ADDR);
	printf("Memory values are:");
	for (i = 0; i < 10; i++) {
		printf(" %4.4x", inw(tcic));
	}
	printf(" %#x.\n", inl(tcic + TCIC_ADDR));

	outb(PWR_VCCSEL0, tcic + TCIC_PWR);
	usleep(100000);

	/* Write the enable byte to the card. */
	outl(0x80002000, tcic + TCIC_ADDR);
	outw(0x0041, tcic);
	/* Keep the autoincrement from happening, so we can observer the IRQ
	   register. */
	outb(0x01, tcic + TCIC_SCTRL);    
	printf(" Interrupt value is %4.4x.\n", inw(tcic));
	/*outb(0x00, tcic + TCIC_PWR);*/

	printf("Card values are: ");
	for (i = 0; i < 10; i++) {
		printf(" %2.2x", inb(card_addr + i));
	}
	printf(".\n");

	lseek(mem_fd, 0xd0000, SEEK_SET);

	for (i = 0; i <32; i++) {
		unsigned char c[2];
		if ((i & 0xf) == 0)
			printf("\n%3.3x:", i);
		read(mem_fd, &c, 2);
		printf(" %2.2x", c[0]);
	}
	printf("\n");

	printf("Card values are: ");
	for (i = 0; i < card_length ; i+=2) {
		printf(" %4.4x", inw(card_addr + i));
	}
	printf(".\n");

	return 0;
}

/*
 * Local variables:
 *  compile-command: "cc -O -o dbether dbether.c -N -Wall"
 *  c-indent-level: 4
 *  tab-width: 4
 * End:
 */
