/*
 * ldconfig - update shared library symlinks
 *
 * usage: ldconfig [-Dvn] dir ...
 *        ldconfig -l [-Dv] lib ...
 *        -D: debug mode, don't update links
 *        -v: verbose mode, print things as we go
 *        -n: don't process standard directories
 *        -l: library mode, manually link libraries
 *        dir ...: directories to process
 *        lib ...: libraries to link
 *
 * Copyright 1993, David Engel
 *
 * This program may be used for any purpose as long as this
 * copyright notice is kept.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <getopt.h>
#include <dirent.h>
#include <unistd.h>
#include <a.out.h>
#include <sys/stat.h>
#include "config.h"

#define EXIT_OK    0
#define EXIT_FATAL 128

char *prog = NULL;
int debug = 0;			/* debug mode */
int verbose = 0;		/* verbose mode */
int libmode = 0;		/* library mode */

void warn(char *fmt, ...)
{
    va_list ap;

    fprintf(stderr, "%s: warning: ", prog);

    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);

    fprintf(stderr, "\n");

    return;
}

void error(char *fmt, ...)
{
    va_list ap;

    fprintf(stderr, "%s: ", prog);

    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);

    fprintf(stderr, "\n");

    exit(EXIT_FATAL);
}

void *xmalloc(size_t size)
{
    void *ptr;
    if ((ptr = malloc(size)) == NULL)
	error("out of memory");
    return ptr;
}

char *xstrdup(char *str)
{
    char *ptr;
    if ((ptr = strdup(str)) == NULL)
	error("out of memory");
    return ptr;
}

/* if shared library, return length of base name, else return -1 */
int is_shlib(char *dir, char *name)
{
    int good = 0;
    char *cp = name;
    FILE *file;
    struct exec exec;
    char buff[1024];

    /* see if name is of the form libZ.so.M.N[.P] */
    if (strncmp(name, "lib", 3) == 0 && (cp = strstr(name, ".so."))
	&& (cp = strchr(cp + 4, '.')))
    {
	/* construct the full path name */
	sprintf(buff, "%s%s%s", dir, (*dir && strcmp(dir, "/")) ?
		"/" : "", name);

	/* and then try opening it */
	if (!(file = fopen(buff, "rb")))
	    warn("can't open %s (%s), skipping", buff, strerror(errno));
	else
	{
	    /* now make sure it's ZMAGIC with a non-zero address */
	    if (fread(&exec, sizeof exec, 1, file) < 1)
		warn("can't read header from %s, skipping", buff);
	    else if (N_MAGIC(exec) != ZMAGIC)
		warn("%s is not a ZMAGIC file, skipping", buff);
	    else if (!exec.a_entry)
		warn("%s has a zero load address, skipping", buff);
	    else
		good = 1; /* looks good */

	    fclose(file);
	}
    }

    return good ? cp - name : -1;
}

/* update the symlink to new library */
void link_shlib(char *dir, char *file, int len)
{
    int nochange = 0;
    char libname[1024];
    char linkname[1024];
    struct stat libstat;
    struct stat linkstat;

    /* construct the full path names */
    sprintf(libname, "%s%s%s", dir, (*dir && strcmp(dir, "/")) ?
	    "/" : "", file);
    sprintf(linkname, "%s%s%.*s", dir, (*dir && strcmp(dir, "/"))
	    ? "/" : "", len, file);

    /* see if a link already exists */
    if (!stat(linkname, &linkstat))
    {
	/* now see if it's the one we want */
	if (stat(libname, &libstat))
	    error("can't stat library %s (%s)", libname, strerror(errno));
	if (libstat.st_dev == linkstat.st_dev &&
	    libstat.st_ino == linkstat.st_ino)
	    nochange = 1;
    }

    /* then update the link, if required */
    if (!debug && !nochange)
    {
	if (!lstat(linkname, &linkstat) && remove(linkname))
	    error("can't unlink %s (%s)", linkname, strerror(errno));
	if (symlink(file, linkname))
	    error("can't link %s to %s (%s)", linkname, file, strerror(errno));
    }

    /* some people like to know what we're doing */
    if (verbose)
	printf("\t%.*s => %s%s\n", len, file, file,
	       nochange ? " (no change)" : "");

    return;
}

struct lib
{
    char *name;			/* name of a library */
    int len;			/* length of base name */
    struct lib *next;		/* next library in list */
};

/* update all shared library links in a directory */
void scan_dir(char *name)
{
    DIR *dir;
    struct dirent *ent;
    int len;
    struct lib *lp, *libs = NULL;

    /* let 'em know what's going on */
    if (verbose)
	printf("%s:\n", name);

    /* if we can't open it, we can't do anything */
    if ((dir = opendir(name)) == NULL)
    {
	warn("can't process %s (%s), skipping", name, strerror(errno));
	return;
    }

    /* yes, we have to look at every single file */
    while ((ent = readdir(dir)) != NULL)
    {
	/* if it's not a shared library, don't bother */
	if ((len = is_shlib(name, ent->d_name)) < 0)
	    continue;

	/* have we already seen one with the same base name? */
	for (lp = libs; lp; lp = lp->next)
	{
	    if (strncmp(ent->d_name, lp->name, len+1) == 0)
	    {
		/* we have, which one do we want to use? */
		if (strcmp(ent->d_name, lp->name) > 0)
		{
		    /* let's use the new one */
		    free(lp->name);
		    lp->name = xstrdup(ent->d_name);
		}
		break;
	    }
	}

	/* congratulations, you're the first one we've seen */
	if (!lp)
	{
	    lp = xmalloc(sizeof *lp);
	    lp->name = xstrdup(ent->d_name);
	    lp->len = len;
	    lp->next = libs;
	    libs = lp;
	}
    }

    /* don't need this any more */
    closedir(dir);

    /* now we have all the latest libs, update the links */
    for (lp = libs; lp; lp = lp->next)
	link_shlib(name, lp->name, lp->len);

    /* always try to clean up after ourselves */
    while (libs)
    {
	lp = libs->next;
	free(libs->name);
	free(libs);
	libs = lp;
    }

    return;
}

/* return the list of system-specific directories */
char *get_extpath(void)
{
    char *cp = NULL;
    FILE *file;
    struct stat stat;

    if ((file = fopen(LDSO_CONF, "r")) != NULL)
    {
	fstat(fileno(file), &stat);
	cp = xmalloc(stat.st_size + 1);
	fread(cp, 1, stat.st_size, file);
	fclose(file);
	cp[stat.st_size] = '\0';
    }

    return cp;
}

int main(int argc, char **argv)
{
    int i, c, len;
    int nodefault = 0;
    char *cp, *dir;
    char *extpath;
    char defpath[] = DEFAULT_PATH;

    prog = argv[0];
    opterr = 0;

    while ((c = getopt(argc, argv, "Dvnl")) != EOF)
	switch (c)
	{
	case 'D':
	    debug = 1;		/* debug mode */
	    /* fall through */
	case 'v':
	    verbose = 1;	/* verbose mode */
	    break;
	case 'n':
	    nodefault = 1;	/* no default dirs */
	    break;
	case 'l':
	    libmode = 1;	/* library mode */
	    break;
	default:
	    error("invalid option -%c", optopt);
	}

    /* these don't make sense together */
    if (nodefault && libmode)
	error("-n and -l options are mutually exclusive");

    /* I me to introduce myself, hi, my name is ... */
    if (verbose)
	printf("%s: version %s\n", argv[0], VERSION);

    if (libmode)
    {
	/* so you want to do things manually, eh? */

	/* if you're so smart, which libraries do we link? */
	for (i = optind; i < argc; i++)
	{
	    /* split into directory and file parts */
	    if (!(cp = strrchr(argv[i], '/')))
	    {
		dir = "";	/* no dir, only a filename */
		cp = argv[i];
	    }
	    else
	    {
		if (cp == argv[i])
		    dir = "/";	/* file in root directory */
		else
		    dir = argv[i];
		*cp++ = '\0';	/* neither of the above */
	    }

	    /* we'd better do a little bit of checking */
	    if ((len = is_shlib(dir, cp)) < 0)
		error("%s%s%s is not a shared library", dir,
		      (*dir && strcmp(dir, "/")) ? "/" : "", cp);

	    /* so far, so good, maybe he knows what he's doing */
	    link_shlib(dir, cp, len);
	}
    }
    else
    {
	/* the lazy bum want's us to do all the work for him */

	/* OK, which directories should we do? */
	for (i = optind; i < argc; i++)
	    scan_dir(argv[i]);

	/* look ma, no defaults */
	if (!nodefault)
	{
	    /* I guess the defaults aren't good enough */
	    if ((extpath = get_extpath()))
	    {
		for (cp = strtok(extpath, DIR_SEP); cp;
		     cp = strtok(NULL, DIR_SEP))
		    scan_dir(cp);
		free(extpath);
	    }

	    /* everybody needs these, don't they? */
	    for (cp = strtok(defpath, DIR_SEP); cp;
		 cp = strtok(NULL, DIR_SEP))
		scan_dir(cp);
	}
    }

    exit(EXIT_OK);
}
