/* 
 * script.h --
 *
 *  This file contains string formatted TCL/TK scripts for animating 
 *  DLX pipeline configurations.
 *
 *  This file is part of DLXview.  It was written by Yinong Zhang 
 *  (yinong@ecn.purdue.edu).
 *
 */



char mkConfigure_tcl[] = "\
proc mkConfigure {{w .configure}} {\n\
  global config\n\
  global numWords\n\
  global configureEnabled\n\
  set configOrig $config\n\
  set numWordsOrig $numWords\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Configure\"\n\
  wm iconname $w \"configure\"\n\
  wm geometry $w +0+0\n\
  set oldFocus [focus]\n\
  frame $w.config  \n\
  frame $w.mem\n\
  frame $w.button\n\
  pack $w.config $w.mem $w.button \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  radiobutton $w.config.basicpipe \\\n\
    -relief flat \\\n\
    -text \"Basic Pipeline\" \\\n\
    -variable config \\\n\
    -value 0\n\
  radiobutton $w.config.tomasulo \\\n\
    -relief flat \\\n\
    -text \"Tomasulo\" \\\n\
    -variable config \\\n\
    -value 1\n\
  radiobutton $w.config.scoreboarding \\\n\
    -relief flat \\\n\
    -text \"Scoreboarding\" \\\n\
    -variable config \\\n\
    -value 2\n\
  pack $w.config.basicpipe $w.config.tomasulo $w.config.scoreboarding \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 2\n\
  scale $w.mem.size \\\n\
    -label \"memory size (words)\" \\\n\
    -from 2048 \\\n\
    -to 16384 \\\n\
    -tickinterval 14336 \\\n\
    -command ShowMemSize\n\
  pack $w.mem.size \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  button $w.button.ok \\\n\
    -text Ok \\\n\
    -state disabled \\\n\
    -command \"ConfigureOk $oldFocus\"\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \"destroy $w\n\
      focus $oldFocus\n\
      set config $configOrig\n\
      set numWords $numWordsOrig\"\n\
  pack $w.button.ok $w.button.cancel \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  trace variable config w SubConfigure\n\
  $w.mem.size set $numWords\n\
 \n\
  if {$configureEnabled == \"0\"} {\n\
    $w.mem.size configure -state disabled\n\
    $w.mem.size configure -troughcolor lightgrey\n\
    $w.mem.size configure -activebackground lightgrey\n\
    $w.button.ok configure -state disabled\n\
    switch $config {\n\
      0 {\n\
          $w.config.tomasulo configure -state disabled\n\
          $w.config.scoreboarding configure -state disabled\n\
        }\n\
      1 {\n\
          $w.config.basicpipe configure -state disabled\n\
          $w.config.scoreboarding configure -state disabled\n\
        }\n\
      2 {\n\
          $w.config.basicpipe configure -state disabled\n\
          $w.config.tomasulo configure -state disabled\n\
        }\n\
    }\n\
  }\n\
   \n\
  grab $w\n\
  focus $w\n\
}\n\
   ";






char ShowMemSize_tcl[] = "\
proc ShowMemSize {size {w .configure}} {\n\
  global numWords\n\
  set size [expr ($size + 1024)/2048*2048]\n\
  $w.mem.size set $size\n\
  set numWords $size\n\
}\n\
  \n\
   ";






char SubConfigure_tcl[] = "\
proc SubConfigure {name element op} {\n\
  global configureEnabled\n\
  upvar $name configure\n\
  if {[winfo exists .configure]} {\n\
    if {$configureEnabled != \"0\" && $configure != \"3\"} {\n\
      .configure.button.ok configure -state normal\n\
    }\n\
    switch $configure {\n\
      0 {\n\
          BasConfigure\n\
        }\n\
      1 {\n\
          TomConfigure\n\
        }\n\
      2 {\n\
          ScoConfigure\n\
        }\n\
      3 {}\n\
    }\n\
  }\n\
}\n\
   ";






char BasConfigure_tcl[] = "\
proc BasConfigure {{w .subconfigure}} {\n\
  global num_int_units \n\
  global num_add_units fp_add_latency add_ful_pipe\n\
  global num_mul_units fp_mul_latency mul_ful_pipe\n\
  global num_div_units fp_div_latency div_ful_pipe\n\
  global configureEnabled\n\
  set num_add_unitsOrig $num_add_units\n\
  set fp_add_latencyOrig $fp_add_latency \n\
  set num_mul_unitsOrig $num_mul_units\n\
  set fp_mul_latencyOrig $fp_mul_latency\n\
  set num_div_unitsOrig $num_div_units\n\
  set fp_div_latencyOrig $fp_div_latency\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Basic Pipeline Configuration\"\n\
  wm iconname $w \"subconfigure\"\n\
  wm geometry $w +0+0\n\
  frame $w.add\n\
  frame $w.mul\n\
  frame $w.div\n\
  frame $w.button  \n\
  set oldFocus [focus]\n\
  if {$configureEnabled} {\n\
    set scaletroco skyblue\n\
    set scaleactbg MediumOrchid\n\
  } else {\n\
    set scaletroco lightgrey\n\
    set scaleactbg lightgrey\n\
  }\n\
  pack $w.add $w.mul $w.div $w.button \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes  \n\
  scale $w.add.number \\\n\
    -tickinterval 7 \\\n\
    -label \"FP adder number\" \\\n\
    -command \"ScaleValue num_add_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.add.latency \\\n\
    -to 5  \\\n\
    -tickinterval 4 \\\n\
    -label \"FP adder latency (cycles)\" \\\n\
    -command \"ScaleValue fp_add_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg \n\
  radiobutton $w.add.fulpipe \\\n\
    -relief flat \\\n\
    -text \"fully piped\" \\\n\
    -variable add_ful_pipe \\\n\
    -value 1 \\\n\
    -command \"BasRadioCommand add inactive\"\n\
  radiobutton $w.add.notpipe \\\n\
    -relief flat \\\n\
    -text \"not piped\" \\\n\
    -variable add_ful_pipe \\\n\
    -value 0 \\\n\
    -command \"BasRadioCommand add active\"\n\
  pack $w.add.number $w.add.latency $w.add.fulpipe $w.add.notpipe \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.mul.number \\\n\
    -tickinterval 7 \\\n\
    -label \"FP multiplier number\" \\\n\
    -command \"ScaleValue num_mul_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.mul.latency \\\n\
    -to 10  \\\n\
    -tickinterval 9 \\\n\
    -label \"FP multiplier latency (cycles)\" \\\n\
    -command \"ScaleValue fp_mul_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  radiobutton $w.mul.fulpipe \\\n\
    -relief flat \\\n\
    -text \"fully piped\" \\\n\
    -variable mul_ful_pipe \\\n\
    -value 1 \\\n\
    -command \"BasRadioCommand mul inactive\"\n\
  radiobutton $w.mul.notpipe \\\n\
    -relief flat \\\n\
    -text \"not piped\" \\\n\
    -variable mul_ful_pipe \\\n\
    -value 0 \\\n\
    -command \"BasRadioCommand mul active\"\n\
  pack $w.mul.number $w.mul.latency $w.mul.fulpipe $w.mul.notpipe\\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.div.number \\\n\
    -from 0 \\\n\
    -tickinterval 8 \\\n\
    -label \"FP divider number\" \\\n\
    -command \"ScaleValue num_div_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.div.latency \\\n\
    -to 50  \\\n\
    -tickinterval 49 \\\n\
    -label \"FP divider latency (cycles)\" \\\n\
    -command \"ScaleValue fp_div_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  radiobutton $w.div.fulpipe \\\n\
    -relief flat \\\n\
    -text \"fully piped\" \\\n\
    -variable div_ful_pipe \\\n\
    -value 1 \\\n\
    -command \"BasRadioCommand div inactive\"\n\
  radiobutton $w.div.notpipe \\\n\
    -relief flat \\\n\
    -text \"not piped\" \\\n\
    -variable div_ful_pipe \\\n\
    -value 0 \\\n\
    -command \"BasRadioCommand div active\"\n\
  pack $w.div.number $w.div.latency $w.div.fulpipe $w.div.notpipe \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  button $w.button.ok \\\n\
    -text Ok \\\n\
    -command \"SubConfigureOk \n\
       destroy $w\n\
       focus $oldFocus\"\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \" set num_add_units $num_add_unitsOrig\n\
      set fp_add_latency $fp_add_latencyOrig \n\
      set num_mul_units $num_mul_unitsOrig\n\
      set fp_mul_latency $fp_mul_latencyOrig\n\
      set num_div_units $num_div_unitsOrig\n\
      set fp_div_latency $fp_div_latencyOrig\n\
      destroy $w\n\
      focus $oldFocus\"\n\
  pack $w.button.ok $w.button.cancel\\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  $w.add.number set $num_add_units\n\
  $w.add.latency set $fp_add_latency\n\
  $w.mul.number set $num_mul_units\n\
  $w.mul.latency set $fp_mul_latency\n\
  $w.div.number set $num_div_units\n\
  $w.div.latency set $fp_div_latency\n\
  if {$configureEnabled == \"0\"} {\n\
    $w.button.ok configure -state disabled\n\
    $w.add.number configure -state disabled\n\
    $w.add.latency configure -state disabled\n\
    $w.add.fulpipe configure -state disabled\n\
    $w.add.notpipe configure -state disabled\n\
    $w.mul.number configure -state disabled\n\
    $w.mul.latency configure -state disabled\n\
    $w.mul.fulpipe configure -state disabled\n\
    $w.mul.notpipe configure -state disabled\n\
    $w.div.number configure -state disabled\n\
    $w.div.latency configure -state disabled\n\
    $w.div.fulpipe configure -state disabled\n\
    $w.div.notpipe configure -state disabled\n\
  }\n\
  if {$add_ful_pipe} {\n\
    $w.add.fulpipe invoke\n\
  } else {\n\
    $w.add.notpipe invoke\n\
  }\n\
  if {$mul_ful_pipe} {\n\
    $w.mul.fulpipe invoke\n\
  } else {\n\
    $w.mul.notpipe invoke\n\
  }\n\
  if {$div_ful_pipe} {\n\
    $w.div.fulpipe invoke\n\
  } else {\n\
    $w.div.notpipe invoke\n\
  }\n\
  tkwait visibility $w\n\
  grab $w\n\
  focus $w\n\
}\n\
   ";






char TomConfigure_tcl[] = "\
proc TomConfigure {{w .subconfigure}} {\n\
  global num_int_units int_latency\n\
  global num_add_units fp_add_latency\n\
  global num_mul_units fp_mul_latency\n\
  global num_div_units fp_div_latency\n\
  global num_load_bufs load_buf_latency\n\
  global num_store_bufs store_buf_latency\n\
  global ld_st_exist fp_div_exist\n\
  global configureEnabled\n\
  set num_int_unitsOrig $num_int_units\n\
  set int_latencyOrig $int_latency\n\
  set num_add_unitsOrig $num_add_units\n\
  set fp_add_latencyOrig $fp_add_latency \n\
  set num_mul_unitsOrig $num_mul_units\n\
  set fp_mul_latencyOrig $fp_mul_latency\n\
  set num_div_unitsOrig $num_div_units\n\
  set fp_div_latencyOrig $fp_div_latency\n\
  set num_load_bufsOrig $num_load_bufs\n\
  set load_buf_latencyOrig $load_buf_latency\n\
  set num_store_bufsOrig $num_store_bufs\n\
  set store_buf_latencyOrig $store_buf_latency\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Tomasulo Algorithm Configuration\"\n\
  wm iconname $w \"subconfigure\"\n\
  wm geometry $w +0+0\n\
  frame $w.int\n\
  frame $w.add\n\
  frame $w.mul\n\
  frame $w.div\n\
  frame $w.load\n\
  frame $w.store\n\
  frame $w.button  \n\
  set oldFocus [focus]\n\
  if {$configureEnabled} {\n\
    set scaletroco skyblue\n\
    set scaleactbg MediumOrchid\n\
  } else {\n\
    set scaletroco lightgrey\n\
    set scaleactbg lightgrey\n\
  }\n\
  pack  $w.int $w.add $w.mul $w.div  $w.load $w.store $w.button \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes  \n\
  scale $w.int.number \\\n\
    -tickinterval 7 \\\n\
    -label \"integer reservation station number\" \\\n\
    -command \"ScaleValue num_int_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg \n\
  scale $w.int.latency \\\n\
    -to 3  \\\n\
    -tickinterval 2 \\\n\
    -label \"integer op latency (cycles)\" \\\n\
    -command \"ScaleValue int_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  pack $w.int.number $w.int.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.add.number \\\n\
    -tickinterval 7 \\\n\
    -label \"FP add reservation station number\" \\\n\
    -command \"ScaleValue num_add_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.add.latency \\\n\
    -to 5  \\\n\
    -tickinterval 4 \\\n\
    -label \"FP add latency (cycles)\" \\\n\
    -command \"ScaleValue fp_add_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg \n\
  pack $w.add.number $w.add.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.mul.number \\\n\
    -tickinterval 7 \\\n\
    -label \"FP multiply reservation station number\" \\\n\
    -command \"ScaleValue num_mul_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.mul.latency \\\n\
    -to 10  \\\n\
    -tickinterval 9 \\\n\
    -label \"FP multiply latency (cycles)\" \\\n\
    -command \"ScaleValue fp_mul_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  pack $w.mul.number $w.mul.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.div.number \\\n\
    -from 0 \\\n\
    -tickinterval 8 \\\n\
    -label \"FP divide reservation station number\" \\\n\
    -command \"ScaleValue num_div_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.div.latency \\\n\
    -to 50  \\\n\
    -tickinterval 49 \\\n\
    -label \"FP divide latency (cycles)\" \\\n\
    -command \"ScaleValue fp_div_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  pack $w.div.number $w.div.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.load.number \\\n\
    -from 0 \\\n\
    -tickinterval 8 \\\n\
    -label \"load buffer number\" \\\n\
    -command \"ScaleValue num_load_bufs\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.load.latency \\\n\
    -to 20  \\\n\
    -tickinterval 19 \\\n\
    -label \"load buffer latency (cycles)\" \\\n\
    -command \"ScaleValue load_buf_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  pack $w.load.number $w.load.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.store.number \\\n\
    -from 0 \\\n\
    -tickinterval 8 \\\n\
    -label \"store buffer number\" \\\n\
    -command \"ScaleValue num_store_bufs\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.store.latency \\\n\
    -to 20  \\\n\
    -tickinterval 19 \\\n\
    -label \"store buffer latency (cycles)\" \\\n\
    -command \"ScaleValue  store_buf_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  pack $w.store.number $w.store.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  button $w.button.ok \\\n\
    -text Ok \\\n\
    -command \"SubConfigureOk \n\
       destroy $w\n\
       focus $oldFocus\"\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \"set num_int_units $num_int_unitsOrig\n\
      set int_latency $int_latencyOrig\n\
      set num_add_units $num_add_unitsOrig\n\
      set fp_add_latency $fp_add_latencyOrig \n\
      set num_mul_units $num_mul_unitsOrig\n\
      set fp_mul_latency $fp_mul_latencyOrig\n\
      set num_div_units $num_div_unitsOrig\n\
      set fp_div_latency $fp_div_latencyOrig\n\
      set num_load_bufs $num_load_bufsOrig\n\
      set load_buf_latency $load_buf_latencyOrig\n\
      set num_store_bufs $num_store_bufsOrig\n\
      set store_buf_latency $store_buf_latencyOrig\n\
      destroy $w\n\
      focus $oldFocus\"\n\
  pack $w.button.ok $w.button.cancel\\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  $w.int.number set $num_int_units\n\
  $w.int.latency set $int_latency\n\
  $w.add.number set $num_add_units\n\
  $w.add.latency set $fp_add_latency\n\
  $w.mul.number set $num_mul_units\n\
  $w.mul.latency set $fp_mul_latency\n\
  $w.div.number set $num_div_units\n\
  $w.div.latency set $fp_div_latency\n\
  $w.load.number set $num_load_bufs\n\
  $w.load.latency set $load_buf_latency\n\
  $w.store.number set $num_store_bufs\n\
  $w.store.latency set $store_buf_latency\n\
  if {$configureEnabled == \"0\"} {\n\
    $w.button.ok configure -state disabled\n\
    $w.int.number configure -state disabled\n\
    $w.int.latency configure -state disabled\n\
    $w.add.number configure -state disabled\n\
    $w.add.latency configure -state disabled\n\
    $w.mul.number configure -state disabled\n\
    $w.mul.latency configure -state disabled\n\
    $w.div.number configure -state disabled\n\
    $w.div.latency configure -state disabled\n\
    $w.load.number configure -state disabled\n\
    $w.load.latency configure -state disabled\n\
    $w.store.number configure -state disabled\n\
    $w.store.latency configure -state disabled\n\
  }\n\
  tkwait visibility $w\n\
  grab $w\n\
  focus $w\n\
}\n\
   ";






char ScoConfigure_tcl[] = "\
proc ScoConfigure {{w .subconfigure}} {\n\
  global num_int_units int_latency\n\
  global num_add_units fp_add_latency\n\
  global num_mul_units fp_mul_latency\n\
  global num_div_units fp_div_latency\n\
  global fp_div_exist\n\
  global configureEnabled\n\
  set num_int_unitsOrig $num_int_units\n\
  set int_latencyOrig $int_latency\n\
  set num_add_unitsOrig $num_add_units\n\
  set fp_add_latencyOrig $fp_add_latency \n\
  set num_mul_unitsOrig $num_mul_units\n\
  set fp_mul_latencyOrig $fp_mul_latency\n\
  set num_div_unitsOrig $num_div_units\n\
  set fp_div_latencyOrig $fp_div_latency\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Scoreboarding Configuration\"\n\
  wm iconname $w \"subconfigure\"\n\
  wm geometry $w +0+0\n\
  frame $w.int\n\
  frame $w.add\n\
  frame $w.mul\n\
  frame $w.div\n\
  frame $w.button  \n\
  set oldFocus [focus]\n\
  if {$configureEnabled} {\n\
    set scaletroco skyblue\n\
    set scaleactbg MediumOrchid\n\
  } else {\n\
    set scaletroco lightgrey\n\
    set scaleactbg lightgrey\n\
  }\n\
  pack  $w.int $w.add $w.mul $w.div $w.button \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes  \n\
  scale $w.int.number \\\n\
    -tickinterval 7 \\\n\
    -label \"integer unit number\" \\\n\
    -command \"ScaleValue num_int_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg \n\
  scale $w.int.latency \\\n\
    -to 3  \\\n\
    -tickinterval 2 \\\n\
    -label \"integer unit latency (cycles)\" \\\n\
    -command \"ScaleValue int_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  pack $w.int.number $w.int.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.add.number \\\n\
    -tickinterval 7 \\\n\
    -label \"FP add reservation station number\" \\\n\
    -command \"ScaleValue num_add_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.add.latency \\\n\
    -to 5  \\\n\
    -tickinterval 4 \\\n\
    -label \"FP add latency (cycles)\" \\\n\
    -command \"ScaleValue fp_add_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg \n\
  pack $w.add.number $w.add.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.mul.number \\\n\
    -tickinterval 7 \\\n\
    -label \"FP multiply reservation station number\" \\\n\
    -command \"ScaleValue num_mul_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.mul.latency \\\n\
    -to 10  \\\n\
    -tickinterval 9 \\\n\
    -label \"FP multiply latency (cycles)\" \\\n\
    -command \"ScaleValue fp_mul_latency\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  pack $w.mul.number $w.mul.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  scale $w.div.number \\\n\
    -from 0 \\\n\
    -tickinterval 8 \\\n\
    -label \"FP divide reservation station number\" \\\n\
    -command \"ScaleValue num_div_units\" \\\n\
    -troughcolor $scaletroco \\\n\
    -activebackground $scaleactbg\n\
  scale $w.div.latency \\\n\
    -to 50  \\\n\
    -tickinterval 49 \\\n\
    -label \"FP divide latency (cycles)\" \\\n\
    -command \"ScaleValue fp_div_latency\" \\\n\
    -troughcolor $scaletroco       \\\n\
    -activebackground $scaleactbg           \n\
  pack $w.div.number $w.div.latency \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes \\\n\
    -padx 5 \\\n\
    -pady 10\n\
  button $w.button.ok \\\n\
    -text Ok \\\n\
    -command  \"SubConfigureOk \n\
       destroy $w\n\
       focus $oldFocus\"\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \"set num_int_units $num_int_unitsOrig\n\
      set int_latency $int_latencyOrig\n\
      set num_add_units $num_add_unitsOrig\n\
      set fp_add_latency $fp_add_latencyOrig \n\
      set num_mul_units $num_mul_unitsOrig\n\
      set fp_mul_latency $fp_mul_latencyOrig\n\
      set num_div_units $num_div_unitsOrig\n\
      set fp_div_latency $fp_div_latencyOrig\n\
      destroy $w\n\
      focus $oldFocus\"\n\
  pack $w.button.ok $w.button.cancel\\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  $w.int.number set $num_int_units\n\
  $w.int.latency set $int_latency\n\
  $w.add.number set $num_add_units\n\
  $w.add.latency set $fp_add_latency\n\
  $w.mul.number set $num_mul_units\n\
  $w.mul.latency set $fp_mul_latency\n\
  $w.div.number set $num_div_units\n\
  $w.div.latency set $fp_div_latency\n\
  if {$configureEnabled == \"0\"} {\n\
    $w.button.ok configure -state disabled\n\
    $w.int.number configure -state disabled\n\
    $w.int.latency configure -state disabled\n\
    $w.add.number configure -state disabled\n\
    $w.add.latency configure -state disabled\n\
    $w.mul.number configure -state disabled\n\
    $w.mul.latency configure -state disabled\n\
    $w.div.number configure -state disabled\n\
    $w.div.latency configure -state disabled\n\
  }\n\
  tkwait visibility $w\n\
  grab $w\n\
  focus $w\n\
}\n\
   ";






char ConfigureOk_tcl[] = "\
proc ConfigureOk {oldFocus {w .configure}} {\n\
  global config configureEnabled \n\
  switch $config {\n\
    0 {mkBasicPipe}\n\
    1 {mkTomasulo}\n\
    2 {mkScoreboard}\n\
  }\n\
  .button.top.load configure -state normal\n\
  set configureEnabled 0\n\
  destroy $w\n\
  focus $oldFocus\n\
}\n\
   ";






char SubConfigureOk_tcl[] = "\
proc SubConfigureOk {} {\n\
  global num_int_units ld_st_exist fp_div_exist \n\
  global num_load_bufs num_div_units\n\
  global config configureEnabled \n\
  if {$config == \"0\"} {\n\
    set num_int_units 2\n\
    set cycleDisplayCount 0\n\
  }\n\
  if {$config == \"1\" && $num_load_bufs > \"0\"} {\n\
    set ld_st_exist 1\n\
  } else {\n\
    set ld_st_exist 0\n\
  }\n\
  if {$num_div_units > \"0\"} {\n\
    set fp_div_exist 1\n\
  } else {\n\
    set fp_div_exist 0\n\
  }\n\
}\n\
   ";






char ScaleValue_tcl[] = "\
proc ScaleValue {whichScale value} {\n\
  global $whichScale\n\
  set $whichScale $value\n\
}\n\
   ";






char BasRadioCommand_tcl[] = "\
proc BasRadioCommand {unittype state {w .subconfigure}} {\n\
  global num_add_units num_mul_units num_div_units\n\
  if {$state == \"inactive\"} {\n\
    $w.$unittype.latency configure -label \"FP $unittype stages (cycles)\"\n\
    $w.$unittype.number set 1\n\
    $w.$unittype.number configure -state disabled\n\
    $w.$unittype.number configure -troughcolor lightgrey\n\
    $w.$unittype.number configure -activebackground lightgrey\n\
  } else {\n\
    $w.$unittype.latency configure -label \"FP $unittype latency (cycles)\"\n\
    $w.$unittype.number configure -state normal\n\
    $w.$unittype.number configure -troughcolor skyblue\n\
    $w.$unittype.number configure -activebackground MediumOrchid\n\
  }\n\
}\n\
   ";






char DISC_tcl[] = "\
set auto_path \"[pwd] $auto_path\"\n\
set config 3\n\
set numWords 16384\n\
set num_int_units 1    \n\
set int_latency 1\n\
set num_add_units 2\n\
set fp_add_latency 2  \n\
set add_ful_pipe 1\n\
set num_mul_units 2\n\
set fp_mul_latency 5  \n\
set mul_ful_pipe 1\n\
set num_div_units 0\n\
set fp_div_latency 19  \n\
set div_ful_pipe 0\n\
set num_load_bufs 0\n\
set load_buf_latency 1\n\
set num_store_bufs 0\n\
set store_buf_latency 1\n\
set ld_st_exist 0\n\
set fp_div_exist 1\n\
set fullNameList {}\n\
set startAddress {}\n\
set configureEnabled 1\n\
set fastmode 0\n\
global msg\n\
mkDisc\n\
   ";






char mkHelp_tcl[] = "\
proc mkHelp {{w .help}} {\n\
  global help_magenta help_blue help_green help_black help_italic\n\
  global help_section help_title\n\
  \n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Help\"\n\
  wm iconname $w \"help\"\n\
  wm geometry $w +200+200\n\
  canvas $w.c\n\
  frame $w.text\n\
  frame $w.button\n\
  pack $w.text $w.button \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  text $w.text.t \\\n\
    -relief raised \\\n\
    -bd 2 \\\n\
    -yscrollcommand \"$w.text.s set\" \\\n\
    -setgrid true \\\n\
    -width 80 \\\n\
    -height 40 \\\n\
    -font \"-Adobe-Helvetica-Bold-R-Normal-*-120-*\"\n\
  DisableTextWidget $w.text.t\n\
  scrollbar $w.text.s \\\n\
    -relief flat -command \"$w.text.t yview\"\n\
  pack $w.text.s \\\n\
    -side right \\\n\
    -fill y\n\
  pack $w.text.t \\\n\
    -expand yes \\\n\
    -fill both\n\
  button $w.button.ok \\\n\
    -text Ok \\\n\
    -command HelpOk\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \"destroy $w\"\n\
  pack $w.button.ok $w.button.cancel \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  set help_magenta \"-foreground magenta\"\n\
  set help_blue \"-foreground blue\"\n\
  set help_black \"-foreground black -font -Adobe-Courier-Medium-R-Normal-*-120-*\"\n\
  set help_green \"-foreground SeaGreen -font -Adobe-Courier-Medium-R-Normal-*-120-*\"\n\
  set help_italic \"-font -Adobe-Courier-Bold-R-Normal-*-120-*\"\n\
  set help_section \"-font -Adobe-Courier-Bold-R-Normal-*-140-*\"\n\
  set help_title \"-font -Adobe-Times-Bold-R-Normal-*-180-* -foreground SeaGreen\"\n\
  bind $w <Any-Enter> \"focus $w.text.t\"\n\
  HelpGeneral\n\
}\n\
   ";






char HelpGeneral_tcl[] = "\
proc HelpGeneral {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_blac help_italic\n\
  global help_section help_title\n\
  set help_cmd_stack \"HelpGeneral\"\n\
  $t delete 1.0 end\n\
HelpIn help_title {\n\
                       DLXview: Interactive Animated DLX Pipeline
		       Visualization\n\
}\n\
HelpIn HelpInfo {\n\
1. General Information\n\
}\n\
HelpIn HelpStart {\n\
2. Starting dlxview\n\
}\n\
HelpIn HelpControl {\n\
3. Control Panel Discription\n\
}\n\
HelpIn HelpMode {\n\
4. Pipelining Modes Discription\n\
}\n\
HelpIn HelpMisc {\n\
5. Miscellaneous\n\
}\n\
}\n\
   ";






char HelpInfo_tcl[] = "\
proc HelpInfo {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpInfo]\n\
  $t delete 1.0 end\n\
HelpIn help_section {\n\
Purpose:}\n\
HelpIn help_black {\n\
DLXview is an interactive pipelining simulator using the DLX instruction\n\
set, which is described in }\n\
HelpIn help_italic {Computer Architecture, A Quantitative Approach }\n\
HelpIn help_black {\n\
by Hennessy and Patterson.  The main goal of dlxview is to provide an\n\
animated environment where the internal operations of a processor are\n\
much easier to understand than by text description only.  In addition\n\
to its pedagogical purpose, dlxview also serves to be a handy tool in DLX\n\
instruction set understanding, debugging, and in processor performance\n\
evaluation. \n\
}\n\
HelpIn help_section {\n\
History and Future:}\n\
HelpIn help_black {\n\
DLXview is an extension and modification of DLXsim, a generic non-graphical\n\
DLX pipeline simulator, which was in turn modified form a MIPS simulator.\n\
This is not surprising given the similarities between DLX and MIPS \n\
instruction set.  the simple pipelining  model in DLXsim has been \n\
enhanced greatly to support the execution of basic pipelining, scoreboarding, \n\
and Tomasulo Algorithm.  The graphical interface built on top of Tcl/Tk \n\
tries to expose as much nitty-gritty detail as possible explained by the \n\
Hennessy and Patterson textbook in a }\n\
HelpIn help_italic {cycle-by-cycle }\n\
HelpIn help_black {base. \n\
}\n\
HelpIn help_black {\n\
The current version of dlxview is a branch of the CASLE }\n\
HelpIn help_italic {(KIWI Is What If) }\n\
HelpIn help_black {\n\
project at Purdue University, which aims to provide a comprehensive \n\
pedagogical tool for learning architectural concepts,  compiler \n\
technologies, and the interactions between them.  Based on the experience\n\
in the course of developing dlxview,  we plan to build a more powerful simulator\n\
which will cover the general microparallel execution models such as\n\
superscalar and VLIW with different configurations. This future tool\n\
will also target at some popular commercial RISC processors, for example, \n\
Alpha series and PowerPC series.\n\
}\n\
HelpIn help_section {\n\
Advice and Errors:}\n\
HelpIn help_black {\n\
DLXview is a project in progress.  For bug reports and further information,\n\
please contact yinong@ecn.purdue.edu.\n\
}\n\
}\n\
   ";






char HelpStart_tcl[] = "\
proc HelpStart {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpStart]\n\
  $t delete 1.0 end\n\
HelpIn help_section {\n\
Invoking DLXview:}\n\
HelpIn help_black {\n\
Once in the correct directory or a correct path has been set, you should\n\
be able to run the simulator by typing: }\n\
HelpIn help_italic {\n\
   disc \n\
}\n\
HelpIn help_black {\n\
Additional startup file can be used to save the time of a series of \n\
interactive configuration steps.  suppose your startup file is called\n\
\"foo\",  there are two ways of having dlxview read it:\n\
1. Treat your start up file a command line option, type }\n\
HelpIn help_italic {\n\
   disc -f foo }\n\
HelpIn help_black {\n\
   to begin simulation.  The disadvantage of this approach is that your \n\
   input terminal will no longer accept other key strokes during the \n\
   simulation session.  Fortunately, this feature is irrelevant to most \n\
   users who do not require extensive debugging of DLX assemblies and dlxview \n\
   internals.\n\
2. Start dlxview session first, then type }\n\
HelpIn help_italic {\n\
   source foo }\n\
HelpIn help_black {\n\
   after the \"(disc)\" prompt.\n\
}\n\
  \n\
HelpIn help_section {\n\
Format of a Startup File:}\n\
HelpIn help_black {\n\
The startup file contains a sequence of parameter assignments, each\n\
assignment is a variable-value pair with the following Tcl syntax:}\n\
HelpIn help_green {\n\
   set varName ?value? }\n\
HelpIn help_black { \n\
If there are more than one items in a value, for example, multiple\n\
assembly code files and data files are required to be loaded before\n\
execution, then these items should be wrapped by a pair of double\n\
quotes or curly parenthesis.  There is no ordering of assignments,\n\
however, the end of assignments should be followed by a \n\
}\n\
HelpIn help_green {\n\
   Start }\n\
HelpIn help_black {\n\
command.  The configurable variables are listed in the following\n\
Table1.  All the variables and commands are case sensitive, make sure\n\
you type them correctly.  Some variables are only meaningful in\n\
certain pipelining modes, Table1 gives the applicable mode to each\n\
variable in the last column.  Unpredictable results or core dumps\n\
will appear if a variable is assigned under a wrong mode. A \"must\"\n\
means the corresponding must be set no matter what pipelining mode has\n\
been chosen.  If a variable is not specified in the startup file, it\n\
will take the default value.  Table2 gives the domain of each\n\
variable and there default values.\n\
}  \n\
HelpIn help_black {\n\
}\n\
HelpIn help_green {\n\
            Table1.   Configurable Variables and Their Meanings\n\
VARIABLE           MEANING			   PIPELINE MODE    \n\
pipeline_mode      pipelining mode                 must\n\
input_files        code/data/initialize file name   must\n\
start_address      start address in the code       BasicPipe,ScoreBoard,Tomasulo\n\
numWords           memory size		           BasicPipe,ScoreBoard,Tomasulo\n\
num_int_units      number of integer units         ScoreBoard,Tomasulo\n\
num_add_units      number of FP adders             BasicPipe,ScoreBoard,Tomasulo\n\
num_mul_units      number of FP multipliers        BasicPipe,ScoreBoard,Tomasulo\n\
num_div_units      number of FP dividers           BasicPipe,ScoreBoard,Tomasulo\n\
int_latency        integer unit latency            BasicPipe,ScoreBoard,Tomasulo\n\
fp_add_latency     FP adder latency                BasicPipe,ScoreBoard,Tomasulo\n\
fp_mul_latency     FP multiplier latency           BasicPipe,ScoreBoard,Tomasulo\n\
fp_div_latency     FP divider latency              BasicPipe,ScoreBoard,Tomasulo\n\
add_ful_pipe       FP adder fully pipelined?       BasicPipe      \n\
mul_ful_pipe       FP multiplier fully pipelined?  BasicPipe      \n\
div_ful_pipe       FP divider fully pipelined?     BasicPipe \n\
num_load_bufs      number of load buffers          Tomasulo\n\
num_store_bufs     number of store buffers         Tomasulo\n\
load_buf_latency   load buffer latency             Tomasulo\n\
store_buf_latency  store buffer latency            Tomasulo\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_green {\n\
        Table2.   Domain and Default Values for Configurable Variables\n\
VARIABLE           DOMAIN			   DEFAULT   \n\
pipeline_mode      BasicPipe,ScoreBoard,Tomasulo    none\n\
input_files        files with .s .d .i extension   none\n\
start_address      any legal address or symbol     0x1000\n\
numWords           [1, 16384]                      16384\n\
num_int_units      1                               1\n\
num_add_units      [1, 8]                          2     \n\
num_mul_units      [1, 8]                          2     \n\
num_div_units      [0, 8]                          2   \n\
int_latency        [1, )                           1\n\
fp_add_latency     [1, )                           2\n\
fp_mul_latency     [1, )                           5    \n\
fp_div_latency     [1, )                           19    \n\
add_ful_pipe       0 for no, 1 for yes             1     \n\
mul_ful_pipe       0 for no, 1 for yes             1             \n\
div_ful_pipe       0 for no, 1 for yes             0\n\
num_load_bufs      [0, 8]                          0\n\
num_store_bufs     [0, 8]                          0   \n\
load_buf_latency   [1, )                           1\n\
store_buf_latency  [1, )                           1\n\
}\n\
HelpIn help_section {\n\
A Sample Startup File:}\n\
HelpIn help_black {\n\
foo:}\n\
HelpIn help_green {\n\
   set pipeline_mode BasicPipe\n\
   set fp_add_latency 3\n\
   set num_add_units 1\n\
   set mul_ful_pipe 0\n\
   set fp_mul_latency 10\n\
   set num_div_units 1\n\
   set num_mul_units 1\n\
   set fp_div_latency 24\n\
   set input_files {f347.i f347.d f347.s}\n\
   Start\n\
}\n\
}\n\
   ";






char HelpControl_tcl[] = "\
proc HelpControl {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpControl]\n\
  $t delete 1.0 end\n\
HelpIn help_black {\n\
If a button is enabled, the inside text is black; If it is disabled, the \n\
inside text is gray. \n\
} \n\
HelpIn help_section {\n\
Configure:}\n\
HelpIn help_black {\n\
When dlxview is invoked without using a startup file, or right after the processor\n\
state reset, you need to press this button to configure the machine to be \n\
simulated. In the popped mode configuration window, there are three radio \n\
buttons on the top.  Among them, click the pipelining mode you want to\n\
simulate,  another larger parameter configuration window will\n\
appear.  You can configure your machine by slider dragging or radio\n\
button clicking.  Once complete, you can exit the configuration windows by\n\
pressing the \"Ok\" button in the parameter configuration window first, and\n\
then pressing the \"Ok\" button in the mode configuration window.\n\
}\n\
HelpIn help_black {\n\
In the middle of a simulation session, the \"configure\" button on the control\n\
panel can be pressed again to check the current configurations. However, \n\
the sliders and buttons on the configuration windows are disabled, i.e.,\n\
you cannot change your processor configuration once it has started.  To \n\
abort the current configuration, press \"reset\" button on the control\n\
panel first, and then configure your machine from there.\n\
}\n\
HelpIn help_black {\n\
It is worth to mention that in the text book, the latency of a functional unit\n\
is defined as the number of intervening cycles between an instruction that \n\
produces a result and an instruction that uses the result (See textbook Page \n\
189). For example, the integer unit will have latency \"0\", which is somewhat\n\
unintuitive.  In dlxview, he latency of a functional unit is defined as the total number of clock cycles an instruction will stay in this functional unit.  \n\
Therefore,  the \"latency\" you chose is one clock cycle more than the \n\
specification from the textbook.\n\
}\n\
HelpIn help_black {\n\
For more information on the configurable parameters, please refer to\n\
tables in the }\n\
HelpIn HelpStart {starting DISC }\n\
HelpIn help_black {session.\n\
}\n\
HelpIn help_section {\n\
Load:}\n\
HelpIn help_black {\n\
DLXview forces you to configure the processor before file loading. A file\n\
selection window will pop up after you press the enabled \"load\"\n\
button.  DLXview accepts three types of files before a simulation session\n\
begins.  The *.s files are assembly code file, *.d files are data\n\
file.  It is not necessary to have a separate data file.   Code file and\n\
data file can be merged into a single *.s file.  *.i files contain\n\
commands to initialize the registers to be used. It is possible to add\n\
a few instructions in the assembly code to do the same job, the catch\n\
is that the code segment you are interested in won't start at clock\n\
cycle one.  Register initialization work are unavoidable in some\n\
situations. For instance, you want to examine the execution of a loop\n\
in which some instructions have have pre-specified operands.\n\
Letting this loop starting at clock cycle one, as most of the examples\n\
and exercises in the textbook did, will make it easier to focus on the\n\
loop instructions themselves without worrying about other minor facts\n\
such as the dependency between an loop instruction in the first\n\
iteration of a loop and an instruction outside the loop.  It is much\n\
handier to write a separate register initialization file under these\n\
case.\n\
}\n\
HelpIn HelpCodeFormat {\n\
    Format of code file and data File.}\n\
HelpIn HelpInitFormat {\n\
    Format of register initialization file.}\n\
HelpIn HelpInputSample {\n\
    Sample input files.\n\
}\n\
HelpIn help_black {\n\
When you choose to load related input files at the same time, clicking\n\
the left mouse button on the file selection window for any one of the\n\
files you intended to load will highlight these files as a group,\n\
given the related files have the same base name.  Press \"Load\" button\n\
in the Load window to load these files.  When you choose to load a\n\
single file, the contents of the clicked file will appear to the right\n\
of the file selection window, so that you can check whether you select\n\
the correct file.  You can load as many file as you can and switch\n\
between related and single load mode. When all the files you are\n\
interested are loaded, press \"Done\" button to close load window. Note,\n\
if you choose the files and then press \"Done\" directly without pressing\n\
\"Load\" first, no file is actually loaded.\n\
}\n\
HelpIn help_section {\n\
Edit:}\n\
HelpIn help_black {\n\
\"edit\"  button can be invoked any time during a simulation session to \n\
revise current DLX code and  edit new input files.  When the edit window\n\
pops up, the assembly code executed by the current simulation session will\n\
be automatically copied into the edit window.  If a totally new \n\
assembly file is going to be edited, simply press \"Clean\" button\n\
in the edit window to get rid of the old files.   When editing is completed, \n\
the new file can be directly loaded into the dlxview to start another simulation\n\
session by pressing \"Load\" button in the edit window.  Before the new\n\
file is actually loaded, a dialog box will ask if any other data file\n\
needs to be loaded first.  If the answer is \"No\", then the *d file used in the\n\
current simulation session will be loaded first by default. Once a newly\n\
edited file has been loaded into the simulator through the edit window, \n\
the current simulation session is interrupted, and the processor is \n\
reset in order to run the latest loaded file.  With the help of code editor,\n\
you can change the assembly code being executed at any time and tries the\n\
another simulation without leaving the dlxview environment.\n\
}\n\
HelpIn help_section {\n\
Step forward:}\n\
HelpIn help_black {\n\
If no instruction in the loaded code has entered the pipeline yet,\n\
pressing \"step forward\" will pop up a window to let you enter the\n\
starting address.  Otherwise, the simulation will continue from\n\
wherever execution previously stopped and run the next instruction.The \n\
clock cycle will always advance to the first stage of the newly processed \n\
instruction.  \n\
}\n\
HelpIn help_section {\n\
Step forward:}\n\
HelpIn help_black {\n\
If no instruction in the loaded code has entered the pipeline yet,\n\
pressing \"next cycle\" will pop up a window to let you enter the starting\n\
address.  Otherwise, the simulation will advance by exact one clock cycle.\n\
The simulator will not necessary proceed to the next instruction, because\n\
of the stalls of the pipeline.  Pressing \"next cycle\" continuously will\n\
show you every detail happened in a pipeline.\n\
}\n\
HelpIn help_section {\n\
Go:}\n\
HelpIn help_black {\n\
If no instruction in the loaded code has entered the pipeline yet,\n\
pressing \"go \" will pop up a window to let you enter the starting address.  \n\
Otherwise, the simulation will continue from wherever execution\n\
previously stopped and will not stop until the execution terminates\n\
naturally. Do not forget to put a \"trap #0\" instruction at the end of\n\
your assembly code, otherwise, the simulator will continuously fetch \n\
new \"nop\" instructions from the folowing zeroed memory.\n\
}\n\
HelpIn help_section {\n\
Step back:}\n\
HelpIn help_black {\n\
This button provides you the flexibility to back step one instruction and\n\
view the state of the processor at the clock cycle when the last instruction\n\
entered the pipeline.  If more than 1K clock cycles has already been\n\
simulated,  it will take a little while to move the pipeline to an earlier\n\
state.  Be patient!\n\
}\n\
HelpIn help_section {\n\
Previous cycle:}\n\
HelpIn help_black {\n\
The button provides you the flexibility to wrap back the state of \n\
the processor for exact one clock cycle.   If more than 1K clock cycles has \n\
already been simulated,  it will take a little while to move the pipeline \n\
to an earlier state.  Be patient!\n\
}\n\
HelpIn help_section {\n\
Reset:}\n\
HelpIn help_black {\n\
Whenever you want to change the pipelining mode, the configuration of\n\
the processor, or simulate a new piece of code, \"reset\" button is the right\n\
place to go.  The choices provides by the reset window are quit self-\n\
explainable.\n\
}\n\
HelpIn help_section {\n\
Trace:}\n\
HelpIn help_black {\n\
\"trace\" button provides the capability to produce an instruction trace\n\
or a memory reference trace.  You can feed it into a trace-driven simulator\n\
or dinero cache simulator later to do further analysis.  Enter the file name\n\
in which you want to save the trace, the trace will be collected starting from \n\
the current instruction.  Normally, the trace file is not available until\n\
the current simulation session ends, i.e, the simulation stops naturally or\n\
is interrupted by a reset.   However, when you press any of  the \"step back\"\n\
or \"previous cycle\" button during the simulation, the tracing process is \n\
stopped automatically to avoid producing garbage wrapped around trace.  \n\
In this case, a trace file containing the information from when you started \n\
the tracing up to the last instruction will be generated. \n\
}\n\
HelpIn help_section {\n\
Quit:}\n\
HelpIn help_black {\n\
Lets you say good-bye to dlxview.\n\
}\n\
}\n\
   ";






char HelpCodeFormat_tcl[] = "\
proc HelpCodeFormat {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpCodeFormat]\n\
  $t delete 1.0 end\n\
HelpIn help_black {\n\
1. Comments are started with a semicolon, and continue to the end of line.\n\
2. Constants can be entered either with or without a preceding number sign.\n\
3. The format of instructions and their operands are as shown in the \n\
   Computer Architecture book.\n\
4. Labels are defined by a group of non-blank characters starting with \n\
   either a letter, an underscore, or a dollar sign, and followed immediately \n\
   by a colon.  They are associated with the address immediately\n\
   following the last block of information stored.  This has the bad\n\
   effect that if you have code following a label following a block of\n\
   data that does not end on a word boundary (multiple of 4), the label\n\
   will not point to the first instruction in the code, but instead to 1\n\
   to 3 bytes before (since the address is only rounded when it is\n\
   necessary to correctly align data).  This is done so that if a label\n\
   is found in the middle of a data section, it will point to the start\n\
   of the next section of data without the data having to be aligned to a\n\
   word boundary.  The work-around for this is to use the }\n\
HelpIn help_italic {.align }\n\
HelpIn help_black {(see below).\n\
   directive before labels that will not not be aligned with the data \n\
   following them.  Labels can be accessed anywhere else within that file,\n\
   and in files loaded after that if the label is declared as }\n\
HelpIn help_italic {.global }\n\
HelpIn help_black {\n\
   (see below).\n\
5. The text (code) is ended by a \"trap #0\" instruction.\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
While the assembler is processing an assembly file, the data and instructions\n\
it assembles are placed in memory based on either a text (code) or data\n\
pointer. Which pointer is used is selected not by the type of information, \n\
but by whether the most recent directive was }\n\
HelpIn help_italic {.data }\n\
HelpIn help_black {or }\n\
HelpIn help_italic {.text.  }\n\
HelpIn help_black {The program\n\
initially loads into the text segment.\n\
The assembler supports several directives which affect how it loads the DLX's \n\
memory.  These should be entered in the place where you would normally place\n\
the instruction and its arguments.  The directives currently supported by \n\
dlxview are:\n\
}\n\
HelpIn help_italic {\n\
.align }\n\
HelpIn help_black {n  Cause the next higher address with the lower n bits\n\
          zeroed (the next closest address greater that or equal to the\n\
          current address that is a multiple of 2^n).\n\
}\n\
HelpIn help_italic {\n\
.ascii }\n\
HelpIn help_black {\"string1\", \"string2\", ...\n\
          Store the strings listed on the line in memory as a list of \n\
          characters.  The strings are not terminated by 0 byte.\n\
}\n\
HelpIn help_italic {\n\
.asciiz }\n\
HelpIn help_black {\"string1\", \"string2\", ...\n\
          similar to }\n\
HelpIn help_italic {.ascii, }\n\
HelpIn help_black {except each string is followed by\n\
          a 0 byte (like C strings).\n\
}\n\
HelpIn help_italic {\n\
.byte }\n\
HelpIn help_black {\"byte1\", \"byte1\", ...\n\
          Store the bytes listed on the line sequentially in memory.\n\
}\n\
HelpIn help_italic {\n\
.data }\n\
HelpIn help_black {[address]\n\
          Cause the following code and data to be stored in the data area.\n\
          If an address was supplied, the data will be loaded starting at\n\
          that address, otherwise, the last value for the data pointer will\n\
          be used.  If we were just reading code based on the text (code)\n\
          pointer, store that address so that we can continue form there\n\
          later (on a }\n\
HelpIn help_italic {.text } \n\
HelpIn help_black {directive).\n\
}\n\
HelpIn help_italic {\n\
.double }\n\
HelpIn help_black {\"number1\", \"number2\", ...\n\
          Store the numbers listed on the line sequentially in memory as \n\
          double precision floating point numbers. \n\
}\n\
HelpIn help_italic {\n\
.float }\n\
HelpIn help_black {\"number1\", \"number2\", ...\n\
          Store the numbers listed on the line sequentially in memory as \n\
          single precision floating point numbers. \n\
}\n\
HelpIn help_italic {\n\
.global }\n\
HelpIn help_black {label\n\
          Make the label available for reference by code found in files\n\
          loaded after this file.\n\
}\n\
HelpIn help_italic {\n\
.space }\n\
HelpIn help_black {size\n\
          Move the current storage pointer forwarded by size bytes (to leave\n\
          some empty space in memory).\n\
}\n\
HelpIn help_italic {\n\
.text }\n\
HelpIn help_black {[address]\n\
          Cause the following code and data to be stored in the text (code) \n\
          area. If the address was supplied, the data will be loaded starting \n\
          at that address, otherwise, the last value for the text pointer will\n\
          be used.  If we were just reading code based on the data pointer\n\
          store that address so that we can continue form there later (on a \n\
}\n\
HelpIn help_italic {.data } \n\
HelpIn help_black {directive).\n\
}\n\
HelpIn help_italic {\n\
.word }\n\
HelpIn help_black {\"word1\", \"word1\", ...\n\
          Store the words listed on the line sequentially in memory.\n\
DLXview allows the user access to a few simple C library functions through\n\
the use of the }\n\
HelpIn help_italic {TRAP }\n\
HelpIn help_black {operation.  Currently supported functions are: \n\
}\n\
HelpIn help_italic {open ()}\n\
HelpIn help_black {(trap #1), }\n\
HelpIn help_italic {close ()}\n\
HelpIn help_black {(trap #2), }\n\
HelpIn help_italic {read ()}\n\
HelpIn help_black {(trap #3), }\n\
HelpIn help_italic {write ()}\n\
HelpIn help_black {(trap #4), \n\
}\n\
HelpIn help_italic {printf ()}\n\
HelpIn help_black {(trap #5), When the appropriate trap is invoked, the\n\
first argument should be located in the word starting at the address in r14,\n\
with the following arguments (as seen in a C statement calling the function)\n\
in words above that (r14+4, r14+8, ...).  The result from the function call \n\
will be placed in r1 (this means there is currently no support for the \n\
library functions that return floating point values).  If a double precision\n\
floating point value is to b e passed to a library function, it will occupy\n\
two adjacent words with the lower word the value of the even valued floating \n\
point register, and the higher word containing the value of the odd valued\n\
floating point register (f0 in 0r(14), F1 in 4(r14)).\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
* The file loader of dlxview is completely inherited from DLXsim, and this\n\
help session is directly copied from the DLXsim user's manual.\n\
}\n\
}\n\
   ";






char HelpInitFormat_tcl[] = "\
proc HelpInitFormat {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpInitFormat]\n\
  $t delete 1.0 end\n\
HelpIn help_black {\n\
Two types of register initialization commands exist.  The first type\n\
is used to initialize integer registers, and the second type is used to \n\
initialize floating point registers. \n\
} \n\
HelpIn help_black {\n\
For register designators, any of\n\
the names \"r0\" through \"r31\" and \"f0\" through \"f31\" may be used.  The names \n\
\"$0\" through \"$31\"  may also be used (instead of \"r0\" through \"r31\"), \n\
however, it is safer to use \"r\" instead of \"$\" to avoid the confusion with\n\
Tcl variables.  The name \"pc\" may be used to refer to the program counter.\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_italic {put }\n\
HelpIn help_black {reg number\n\
           Store number in the register given by reg.  To store floating\n\
           point numbers (single or double precision), use the }\n\
HelpIn help_italic {fput }\n\
HelpIn help_black {command.\n\
}\n\
HelpIn help_italic {fput } \n\
HelpIn help_black {reg number [precision]\n\
           Store number in the register given by reg.  If precision is }\n\
HelpIn help_italic {d, }\n\
HelpIn help_black {\n\
           the number is stored a double precision floating point number (in\n\
           two words).  If precision is }\n\
 \n\
HelpIn help_italic {f, }\n\
HelpIn help_black { or no precision is given, \n\
           the number is stored as a single precision floating point number.\n\
}\n\
}\n\
   ";






char HelpInputSample_tcl[] = "\
proc HelpInputSample {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpInputSample]\n\
  $t delete 1.0 end\n\
HelpIn help_black {\n\
f347.s }\n\
HelpIn help_green {\n\
LL:ld      f2, 0(r1)\n\
   multd   f4, f2, f0\n\
   ld      f6, 0(r2)\n\
   addd    f6, f4, f6\n\
   sd      0(r2), f6\n\
   addi    r1, r1, 8\n\
   addi    r2, r2, 8\n\
   sgti    r3, r1, done\n\
   beqz    r3, LL\n\
   nop\n\
   trap    #0\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_black {\n\
f347.i }\n\
HelpIn help_green {\n\
   put     r1      0x1000\n\
   put     r2      0x2000\n\
   fput    f0      3.14    d\n\
}\n\
HelpIn help_black {\n\
f347.d }\n\
HelpIn help_green {\n\
   .data   0x1000\n\
   .double 1.00, 1.01, 1.02, 1.03, 1.04, 1.05, 1.06, 1.07, 1.08, 1.09 \n\
   .double 1.10, 1.11, 1.12, 1.13, 1.14, 1.15, 1.16, 1.17, 1.18, 1.19 \n\
   .double 1.20, 1.21, 1.22, 1.23, 1.24, 1.25, 1.26, 1.27, 1.28, 1.29 \n\
   .double 1.30, 1.31, 1.32, 1.33, 1.34, 1.35, 1.36, 1.37, 1.38, 1.39 \n\
   .double 1.40, 1.41, 1.42, 1.43, 1.44, 1.45, 1.46, 1.47, 1.48, 1.49 \n\
   .double 1.50, 1.51, 1.52, 1.53, 1.54, 1.55, 1.56, 1.57, 1.58, 1.59 \n\
   .double 1.60, 1.61, 1.62, 1.63, 1.64, 1.65, 1.66, 1.67, 1.68, 1.69 \n\
   .double 1.70, 1.71, 1.72, 1.73, 1.74, 1.75, 1.76, 1.77, 1.78, 1.79 \n\
   .double 1.80, 1.81, 1.82, 1.83, 1.84, 1.85, 1.86, 1.87, 1.88, 1.89 \n\
   .double 1.90, 1.91, 1.92, 1.93, 1.94, 1.95, 1.96, 1.97, 1.98, 1.99 \n\
   .global done\n\
   done:    .double 0\n\
}\n\
HelpIn help_black {\n\
}\n\
HelpIn help_green {\n\
   .data   0x2000\n\
   .double 2.00, 2.01, 2.02, 2.03, 2.04, 2.05, 2.06, 2.07, 2.08, 2.09 \n\
   .double 2.10, 2.11, 2.12, 2.13, 2.14, 2.15, 2.16, 2.17, 2.18, 2.19 \n\
   .double 2.20, 2.21, 2.22, 2.23, 2.24, 2.25, 2.26, 2.27, 2.28, 2.29 \n\
   .double 2.30, 2.31, 2.32, 2.33, 2.34, 2.35, 2.36, 2.37, 2.38, 2.39 \n\
   .double 2.40, 2.41, 2.42, 2.43, 2.44, 2.45, 2.46, 2.47, 2.48, 2.49 \n\
   .double 2.50, 2.51, 2.52, 2.53, 2.54, 2.55, 2.56, 2.57, 2.58, 2.59 \n\
   .double 2.60, 2.61, 2.62, 2.63, 2.64, 2.65, 2.66, 2.67, 2.68, 2.69 \n\
   .double 2.70, 2.71, 2.72, 2.73, 2.74, 2.75, 2.76, 2.77, 2.78, 2.79 \n\
   .double 2.80, 2.81, 2.82, 2.83, 2.84, 2.85, 2.86, 2.87, 2.88, 2.89 \n\
   .double 2.90, 2.91, 2.92, 2.93, 2.94, 2.95, 2.96, 2.97, 2.98, 2.99 \n\
}\n\
}\n\
   ";






char HelpMode_tcl[] = "\
proc HelpMode {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpMode]\n\
  $t delete 1.0 end\n\
HelpIn HelpBasicPipe {\n\
1. Basic Pipelie.\n\
}\n\
HelpIn HelpTomasulo {\n\
2. Tomasulo Algorithm.\n\
}\n\
HelpIn HelpScoreBoard {\n\
3. Scoreboarding.\n\
}\n\
}\n\
   ";






char HelpBasicPipe_tcl[] = "\
proc HelpBasicPipe {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpBasicPipe]\n\
  $t delete 1.0 end\n\
HelpIn help_black { \n\
In the visualization window for the basic\n\
pipeline, the top half is a instruction and clock cycle table, and the\n\
bottom half can be switched between the block diagram of the DLX\n\
pipeline (see Figure 3.44 in the textbook) and the datapath of the }\n\
HelpIn help_italic {\n\
integer }\n\
HelpIn help_black {unit (see Figure 3.22 in the textbook).  Each instruction \n\
entering the pipeline is assigned a unique color, which matches\n\
between the top half and bottom half of the visualization window.  The\n\
window exhibits the pipeline state for one particular clock cycle. This \n\
clock cycle number is displayed at the lower-right conner of the \n\
visualization window, and is also highlighted by a gray strip in the clock \n\
cycle table.\n\
}\n\
HelpIn help_black { \n\
The instruction and clock cycle table is capable to display the latest\n\
five instruction simulated by dlxview, with the last row showing the\n\
newest instruction entering the pipeline.  From the highlighted gray\n\
strip, you can determine which instruction is in which pipeline stage.\n\
There might be more than five instructions staying in the pipeline at\n\
a time, however, the earlier instructions are wrapped off in this\n\
table.  If you want to find exactly how many instructions are in the\n\
pipeline at present, check the number of colored boxes in the bottom\n\
pipeline block diagram.\n\
}\n\
HelpIn help_black {\n\
In the pipeline block diagram, if a floating point functional unit is\n\
fully pipelined, the EX pipe stages for this functional unit will be\n\
drawn as separated boxes.  On the other hand, the multicycle EX stage\n\
of a unpipelined floating point unit is shown as a long box with\n\
internal lines.  The block diagram as a whole is also convenient in\n\
figuring out which instruction is at which pipeline stage.  For a\n\
pipeline that supports multiple outstanding FP operations, there might\n\
be multiple instructions entering the MEM or WB stage simultaneously.\n\
(It can be proved that for the particular DLX machine described in the\n\
section 3.7 of the Hennessy and Patterson textbook, at most two\n\
instructions will go to MEM or WB stage at the same time.  Exercise\n\
for you?)  In this case, multiple colors will occupy MEM and WB boxes\n\
(they should be split in a real implementation).\n\
}\n\
HelpIn help_black {\n\
As said by its name, the integer datapath is only responsible for\n\
showing you what is happening in the integer datapath.  Nevertheless,\n\
floating point operations are exhibited here also when they are in the\n\
IF and ID stages.  At a particular clock cycle, there might be some\n\
floating point instruction in execution or writing their results back\n\
to the register file, don't be surprise if you can not find them in\n\
the integer datapath!\n\
}  \n\
HelpIn help_black {\n\
When a mux is used in operand selecting, the line connecting the\n\
output and the selected input will tell you where does the operand come\n\
from.  For example, by examining the muxes before ALU, you can\n\
determine whether operands come from the register file, the forwarding\n\
path, or the immediate number stored in the ID/EX stage-register.  If\n\
an operand is forwarded, it is also easy to trace back to the source\n\
of forwarding from the highlighted wires.\n\
}\n\
HelpIn help_black {\n\
The following is a brief description of the features of the DLX\n\
pipeline defined in the textbook that are supported by dlxview.\n\
1. Instructions are fetched and decoded sequentially. \n\
2. Multiple floating point functional units, either fully pipelined or not,\n\
   can be configured to your processor. However, only one integer unit\n\
   is allowed.  \n\
3. Branches are solved at the ID stage. There is one branch delay slot\n\
   after the branch instruciton, and branch penalties are reduced by the \n\
   delayed-branch scheme.   It is up to the compiler or user, whoever \n\
   provides the DLX code, to make sure the successor instruction of a branch \n\
   is valid and useful.  If no useful instructions can be found, a \"nop\" \n\
   instruction should be inserted after the branch in your assembly code. \n\
4. There is only one write port to the FP register file.\n\
5. Four kinds of checking are accomplished by the ID stage:\n\
   a) Structural hazard checking:\n\
   b) WAR hazard checking.\n\
   c) WAW hazard checking.\n\
   d) FP register file write port conflict checking. \n\
}\n\
HelpIn help_black {\n\
The moving of branch handling logic from MEM stage to ID stage might\n\
cause some additional stalls in the pipeline due to data dependence.\n\
In other words, \"load stall\" is not the only type of data hazard\n\
leading to a  stall for DLX pipeline, as stated by the textbook.  Since this\n\
hazard is not discussed in the textbook, the implementation used in\n\
dlxview is briefly listed here. You can view the hazard by writing a\n\
proper sequence of code and testing it with dlxview.  \n\
1. If the operand of a branch is generated by the immediate previous\n\
   integer ALU instruction, the pipeline will stall for one cycle. After\n\
   that, the operand is forwarded from the EX/MEM stage-register. The stall \n\
   might last longer for FP branches in this case, depending on the latency \n\
   of FP instruction.  Same arguments applies to the following discussion.\n\
2. If the  operand of a branch is generated by the immediate previous\n\
   load instruction,  the pipeline will stall for two cycle. After\n\
   that, the operand is forwarded from the MEM/WB stage-register.\n\
3. If the  operand of a branch is generated by the second immediate previous\n\
   load instruction,  the pipeline will stall for one cycle. After\n\
   that, the operand is forwarded from the MEM/WB stage-register.\n\
}\n\
}\n\
   ";






char HelpTomasulo_tcl[] = "\
proc HelpTomasulo {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpTomasulo]\n\
  HelpPackCanvas \n\
  $t delete 1.0 end\n\
  $t insert 0.0 {\n\
To be added ...\n\
  }\n\
}\n\
   ";






char HelpScoreBoard_tcl[] = "\
proc HelpScoreBoard {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpScoreboard]\n\
  HelpPackCanvas \n\
  $t delete 1.0 end\n\
  $t insert 0.0 {\n\
To be added ...\n\
  }\n\
}\n\
   ";






char HelpMisc_tcl[] = "\
proc HelpMisc {{t .help.text.t}} {\n\
  global help_cmd_stack\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  set help_cmd_stack [linsert $help_cmd_stack 0 HelpScoreMisc]\n\
  $t delete 1.0 end\n\
  $t insert 0.0 {\n\
To be added ...\n\
  }\n\
}\n\
   ";






char HelpOk_tcl[] = "\
proc HelpOk {{w .help}} {\n\
  global help_cmd_stack\n\
  if {[llength $help_cmd_stack] > \"1\"} {\n\
    set last_command [lindex $help_cmd_stack 1]\n\
    set help_cmd_stack [lrange $help_cmd_stack 2 end]\n\
    pack forget $w.c\n\
    $w.text.t configure -height 40\n\
    $last_command\n\
  }\n\
}\n\
   ";






char HelpPackCanvas_tcl[] = "\
proc HelpPackCanvas {{w .help}} {\n\
  pack forget $w.text \n\
  pack forget $w.button\n\
  pack $w.c $w.text $w.button \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  $w.text.t configure -height 23\n\
}\n\
   ";






char HelpIn_tcl[] = "\
proc HelpIn {newtags text} {\n\
  global help_magenta help_magenta help_blue help_green help_black help_italic\n\
  global help_section help_title\n\
set help_title\n\
  set w .help.text.t\n\
  set start [$w index insert]\n\
  $w insert insert $text\n\
  foreach tag [$w tag names $start] {\n\
    $w tag remove $tag $start insert\n\
  }\n\
  foreach i $newtags {\n\
    $w tag add $i $start insert\n\
    if {[string compare [string range $i 0 4] \"help_\"] == \"0\"} {\n\
      eval $w tag configure $i [set $i]\n\
    } else {\n\
      eval $w tag configure $i $help_blue\n\
      $w tag bind $i <Any-Enter> \"$w tag configure $i $help_magenta\"\n\
      $w tag bind $i <Any-Leave> \"$w tag configure $i $help_blue\"\n\
      $w tag bind $i <1> \"$i\"\n\
    }\n\
  }\n\
}\n\
   ";






char mkBasicPipe_tcl[] = "\
proc mkBasicPipe {{w .bas}} {\n\
  global num_add_units fp_add_latency add_ful_pipe\n\
  global num_mul_units fp_mul_latency mul_ful_pipe\n\
  global num_div_units fp_div_latency div_ful_pipe\n\
  global fastmode\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Basic DLX Pipeline\"\n\
  wm iconname $w \"basic pipe\"\n\
  wm geometry $w +0+0\n\
  frame $w.up \\\n\
    -bd 0\n\
  canvas $w.block \\\n\
    -width 850 \\\n\
    -height 500 \\\n\
    -bd 0 \\\n\
    -scrollregion {0 0 850 500} \n\
  canvas $w.datapath \\\n\
    -width 850 \\\n\
    -height 500 \\\n\
    -bd 0 \\\n\
    -scrollregion {0 0 850 500} \n\
  pack $w.up $w.block \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes   \n\
  canvas $w.up.code \\\n\
    -width 150 \\\n\
    -height 180 \\\n\
    -bd 0 \\\n\
    -scrollregion {0 0 165 180} \n\
  canvas $w.up.cycle \\\n\
    -width 650 \\\n\
    -height 180 \\\n\
    -bd 0 \\\n\
    -scrollregion {165 0 3000 180} \\\n\
    -xscrollcommand \"$w.up.s set\"\n\
  for {set i 0} {$i < 58} {incr i} {\n\
    set t strip$i\n\
    $w.up.cycle create rectangle [expr 155+$i*50] 0 [expr 205+$i*50] 180 \\\n\
      -outline \"\" \\\n\
      -tags \"$t strip\"\n\
  }\n\
  scrollbar $w.up.s \\\n\
    -command \"$w.up.cycle xview\" \\\n\
    -relief raised \\\n\
    -orient horizontal\n\
  pack $w.up.code \\\n\
    -side left \\\n\
    -fill y \\\n\
    -expand yes\n\
  pack $w.up.cycle \\\n\
    -side top \n\
  pack $w.up.s \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes  \n\
  set bigfont -Adobe-Courier-medium-R-Normal-*-120-*\n\
  set insfont -Adobe-Courier-bold-R-Normal-*-120-*\n\
  set hugfont -Adobe-Courier-bold-R-Normal-*-140-*\n\
  set desfont -Adobe-Courier-medium-R-Normal-*-100-*\n\
  set subfont -Adobe-Courier-medium-R-Normal-*-80-*\n\
  set labfont -adobe-helvetica-medium-r-normal--24-240-75-75-p-130-iso8859-1\n\
  set c $w.block\n\
  $c create rectangle 50 220 110 280 \\\n\
    -tags {RectIF rect}\n\
  $c create text 80 250 -font $bigfont -text \"IF\" \n\
  $c create rectangle 109 220 170 280 \\\n\
    -tags {RectID rect}\n\
  $c create text 140 250 -font $bigfont -text \"ID\"\n\
  $c create rectangle 600 220 660 280 \n\
  $c create rectangle 601 221 659 250 -outline \"\" \\\n\
    -tags {RectMEM.1 rect}\n\
  $c create rectangle 601 250 659 279 -outline \"\" \\\n\
    -tags {RectMEM.2 rect}\n\
  $c create text 630 250 -font $bigfont -text \"MEM\"\n\
  $c create rectangle 659 220 720 280 \n\
  $c create rectangle 660 221 719 250 -outline \"\" \\\n\
    -tags {RectWB.1 rect}\n\
  $c create rectangle 660 250 719 279 -outline \"\" \\\n\
    -tags {RectWB.2 rect}\n\
  $c create text 690 250 -font $bigfont -text \"WB\"\n\
  set total_units [expr 1+$num_add_units+$num_mul_units+$num_div_units]\n\
  \n\
  set step_large [expr 500.0/($total_units+1)]\n\
  set step_small [expr 60.0/($total_units+1)]\n\
  for {set i 1} {$i <= $total_units} {incr i} {\n\
    $c create line 170 [expr 220+$step_small*$i] 210 [expr $step_large*$i]\n\
    $c create line 560 [expr $step_large*$i] 600 [expr 220+$step_small*$i] \\\n\
      -arrow last\n\
  }\n\
  \n\
  set intervals 0\n\
  if {$add_ful_pipe} {\n\
    set intervals [expr $fp_add_latency*3-1]\n\
  }\n\
  if {$mul_ful_pipe} {\n\
    if {$intervals <= [expr $fp_mul_latency*3-1]} {\n\
      set intervals [expr $fp_mul_latency*3-1]\n\
    }\n\
  }\n\
  if {$div_ful_pipe} {\n\
    if {$intervals <= [expr $fp_div_latency*3-1]} {\n\
      set intervals [expr $fp_div_latency*3-1]\n\
    }\n\
  }\n\
  if {$intervals} {\n\
    set width [expr 330.0/$intervals]\n\
  } else {\n\
    set width 165\n\
  }\n\
  set height [expr $step_large/3]\n\
 \n\
  $c create rectangle [expr 385-$width] [expr $step_large-$height] \\\n\
    [expr 385+$width] [expr $step_large+$height] \\\n\
    -tags {RectEXInt rect}\n\
  $c create line 210 $step_large [expr 385-$width] $step_large \\\n\
    -arrow last\n\
  $c create line [expr 385+$width] $step_large 560 $step_large\n\
  $c create text 385 [expr $step_large-$height-10]  \\\n\
    -font $bigfont -text \"Integer unit\"\n\
  $c create text 385 $step_large  -font $bigfont -text \"EX\"\n\
  set i 2\n\
  for {set j 0} {$j < $num_mul_units} {incr j} {\n\
    set y [expr $step_large*$i]\n\
    $c create text 385 [expr $y-$height-10]  \\\n\
      -font $bigfont -text \"FP/integer multiply\"\n\
    if {$mul_ful_pipe} {\n\
      set temp_length [expr (350-($fp_mul_latency*3-1)*$width)/2]\n\
      $c create line 210 $y [expr 210+$temp_length] $y \\\n\
        -arrow last\n\
      $c create line [expr 560-$temp_length] $y 560 $y\n\
      for {set k 1} {$k <= $fp_mul_latency} {incr k} {\n\
        set x [expr 210+$temp_length+(3*$k-2)*$width]\n\
        $c create rectangle [expr $x-$width] [expr $y-$height] \\\n\
          [expr $x+$width] [expr $y+$height] \\\n\
          -tags \"RectEXMul$j.$k rect\"\n\
        $c create text $x $y -font $bigfont -text \"M$k\"\n\
        if {$k != $fp_mul_latency} {\n\
          $c create line [expr $x+$width] $y [expr $x+$width*2] $y \\\n\
            -arrow last\n\
        }\n\
      } \n\
    } else {\n\
      set distance [expr 330.0/($fp_mul_latency*2)]\n\
      $c create line 210 $y 220 $y \\\n\
        -arrow last\n\
      $c create line 550 $y 560 $y\n\
      for {set k 1} {$k <= $fp_mul_latency} {incr k} {\n\
        set x [expr 220+(2*$k-1)*$distance]\n\
        $c create rectangle [expr $x-$distance] [expr $y-$height] \\\n\
          [expr $x+$distance+1] [expr $y+$height] \\\n\
          -tags \"RectEXMul$j.$k rect\"\n\
      }\n\
      $c create text 385 $y -font $bigfont -text \"MUL\"\n\
    }    \n\
    incr i\n\
  }\n\
  \n\
  for {set j 0} {$j < $num_add_units} {incr j} {\n\
    set y [expr $step_large*$i]\n\
    $c create text 385 [expr $y-$height-10]  \\\n\
      -font $bigfont -text \"FP adder\"\n\
    if {$add_ful_pipe} {\n\
      set temp_length [expr (350-($fp_add_latency*3-1)*$width)/2]\n\
      $c create line 210 $y [expr 210+$temp_length] $y \\\n\
        -arrow last\n\
      $c create line [expr 560-$temp_length] $y 560 $y\n\
      for {set k 1} {$k <= $fp_add_latency} {incr k} {\n\
        set x [expr 210+$temp_length+(3*$k-2)*$width]\n\
        $c create rectangle [expr $x-$width] [expr $y-$height] \\\n\
          [expr $x+$width] [expr $y+$height] \\\n\
          -tags \"RectEXAdd$j.$k rect\"\n\
        $c create text $x $y -font $bigfont -text \"A$k\"\n\
        if {$k != $fp_add_latency} {\n\
          $c create line [expr $x+$width] $y [expr $x+$width*2] $y \\\n\
            -arrow last\n\
        }\n\
      } \n\
    } else {\n\
      set distance [expr 330.0/($fp_add_latency*2)]\n\
      $c create line 210 $y 220 $y \\\n\
        -arrow last\n\
      $c create line 550 $y 560 $y\n\
      for {set k 1} {$k <= $fp_add_latency} {incr k} {\n\
        set x [expr 220+(2*$k-1)*$distance]\n\
        $c create rectangle [expr $x-$distance] [expr $y-$height] \\\n\
          [expr $x+$distance+1] [expr $y+$height] \\\n\
          -tags \"RectEXAdd$j.$k rect\"\n\
      }\n\
      $c create text 385 $y -font $bigfont -text \"ADD\"\n\
    }    \n\
    incr i\n\
  }\n\
  \n\
  for {set j 0} {$j < $num_div_units} {incr j} {\n\
    set y [expr $step_large*$i]\n\
    $c create text 385 [expr $y-$height-10]  \\\n\
      -font $bigfont -text \"FP/integer divider\"\n\
    if {$div_ful_pipe} {\n\
      set temp_length [expr (350-($fp_div_latency*3-1)*$width)/2]\n\
      $c create line 210 $y [expr 210+$temp_length] $y \\\n\
        -arrow last\n\
      $c create line [expr 560-$temp_length] $y 560 $y\n\
      for {set k 1} {$k <= $fp_div_latency} {incr k} {\n\
        set x [expr 210+$temp_length+(3*$k-2)*$width]\n\
        $c create rectangle [expr $x-$width] [expr $y-$height] \\\n\
          [expr $x+$width] [expr $y+$height] \\\n\
          -tags \"RectEXDiv$j.$k rect\" \n\
        $c create text $x $y -font $bigfont -text \"D$k\"\n\
        if {$k != $fp_div_latency} {\n\
          $c create line [expr $x+$width] $y [expr $x+$width*2] $y \\\n\
            -arrow last\n\
        }\n\
      } \n\
    } else {\n\
      set distance [expr 330.0/($fp_div_latency*2)]\n\
      $c create line 210 $y 220 $y \\\n\
        -arrow last\n\
      $c create line 550 $y 560 $y\n\
      for {set k 1} {$k <= $fp_div_latency} {incr k} {\n\
        set x [expr 220+(2*$k-1)*$distance]\n\
        $c create rectangle [expr $x-$distance] [expr $y-$height] \\\n\
          [expr $x+$distance+1] [expr $y+$height] \\\n\
          -tags \"RectEXDiv$j.$k rect\" \n\
      }\n\
      $c create text 385 $y -font $bigfont -text \"DIV\"\n\
    }    \n\
    incr i\n\
  }\n\
  \n\
  button $c.button \\\n\
    -text \"view integer datapath\" \\\n\
    -width 20 \\\n\
    -command \"pack forget .bas.block \n\
              pack .bas.datapath\"\n\
  $c create window 80 480 \\\n\
    -window $c.button\n\
  $c create text 665 425 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-* \\\n\
    -text \"IF Cycle of Current Instruction\" \\\n\
    -tag cycleTitle\n\
  label $c.counter \\\n\
    -font $labfont\n\
  $c create window 665 445 \\\n\
    -window $c.counter\n\
  \n\
  \n\
  set c $w.datapath\n\
  $c create text 375 450 -font $hugfont -text \"INTEGER datapath of pipeline\"\n\
  $c create rectangle 30 225 50 275\n\
  $c create text 40 250 -font $desfont -text \"PC\"\n\
  $c create line 50 250 80 250 -arrow last \\\n\
    -tags {StageIF line}\n\
  $c create line 57 250 57 208 70 208 -arrow last \\\n\
    -tags {StageIF line}\n\
  create_dot $c 57 250\n\
  $c create rectangle 80 240 130 290\n\
  $c create text 105 260 -font $desfont -text \"Instr\"\n\
  $c create text 105 270 -font $desfont -text \"memory\"\n\
  $c create line 130 265 165 265 -arrow last \\\n\
    -tags {StageIF line}\n\
  \n\
  $c create line 70 220 100 202 100 173 70 155 70 180 80 187 70 195 70 220\n\
  $c create text 90 187 -font $desfont -text \"ADD\"\n\
  $c create line 130 265 165 265 -arrow last \\\n\
    -tags {StageIF line}\n\
  $c create text 145 255 -font $desfont -text \"IR\"\n\
  $c create line 100 195 125 195 -arrow last \\\n\
    -tags {StageIF line}\n\
  $c create line 57 167 70 167 -arrow last \\\n\
    -tags {StageIF line}\n\
  $c create text 53 167 -font $desfont -text \"4\"\n\
  $c create rectangle 125 160 145 210 \n\
  $c create text 135 175 -font $desfont -text \"M\"\n\
  $c create text 135 185 -font $desfont -text \"U\"\n\
  $c create text 135 195 -font $desfont -text \"X\"\n\
  $c create line 145 185 165 185 -arrow last \\\n\
    -tags {StageIF line}\n\
  $c create line 150 185 150 135 15 135 15 250 30 250 -arrow last \\\n\
    -tags {StageIF line}\n\
  create_dot $c 150 185\n\
  $c create text 90 20 -font $insfont -text \"\" \\\n\
    -tags {InstrIF text}\n\
  $c create rectangle 165 145 185 380 -fill gray75 \n\
  $c create text 175 135 -font $desfont -text \"IF/ID\"\n\
  $c create line 260 270 290 250 -fill lavender -width 3 \\\n\
    -tags {RegAluDest choise}\n\
  $c create line 260 290 290 250 -fill lavender -width 3 \\\n\
    -tags {RegMemDest choise}\n\
  $c create rectangle 260 190 320 310 \n\
  $c create text 290 250 -font $desfont -text \"Registers\"\n\
  $c create text 290 210 -font $insfont -text \"\" \\\n\
    -tags {RegA text}\n\
  $c create text 290 230 -font $insfont -text \"\" \\\n\
    -tags {RegB text}\n\
  $c create text 290 280 -font $insfont -text \"\" \\\n\
    -tags {RegDest text}\n\
  $c create line 320 235 380 235 -arrow last \\\n\
    -tags {StageID line}\n\
  $c create line 320 260 380 260 -arrow last \\\n\
    -tags {StageID line}\n\
  $c create line 335 235 335 179 345 179 -arrow last \\\n\
    -tags {StageID line}\n\
  create_dot $c 335 235\n\
  $c create oval 240 313 280 363 \n\
  $c create text 260 330 -font $desfont -text \"Sign\"\n\
  $c create text 260 340 -font $desfont -text \"extend\"\n\
  $c create line 280 338 380 338 -arrow last  \\\n\
    -tags {StageID line}\n\
  $c create line 292 333 298 343\n\
  $c create text 298 330 -font $desfont -text \"32\"\n\
  $c create line 327 338 327 180 260 180 260 102 280 102 -arrow last \\\n\
    -tags {StageID line}\n\
  create_dot $c 327 338\n\
  $c create line 345 155 375 167 -fill lavender -width 3 \\\n\
    -tags {BranchRes1 BranchRes2 choise}\n\
  $c create line 345 163 375 167 -fill lavender -width 3 \\\n\
    -tags {BranchRes3 choise}\n\
  $c create line 345 171 375 167 -fill lavender -width 3 \\\n\
    -tags {BranchMem1 choise}\n\
  $c create line 345 179 375 167 -fill lavender -width 3 \\\n\
    -tags {BranchReg choise}\n\
  $c create rectangle 345 180 375 155\n\
  $c create text 360 167 -font $desfont -text \"Zero?\"\n\
  $c create line 375 167 380 167 \\\n\
    -tags {StageIDBranch line}\n\
  $c create line 400 167 420 167 420 50 135 50 135 160 \\\n\
    -tags {StageIDBranch line}\n\
  $c create line 280 155 310 137 310 108 280 90 280 115 290 122 280 130 280 155\n\
  $c create text 300 122 -font $desfont -text \"ADD\"\n\
  $c create line 310 123 350 123 350 148 380 148 \\\n\
    -tags {StageIDBranch line}\n\
  $c create line 400 148 412 148 412 58 110 58 110 175 125 175 -arrow last \\\n\
    -tags {StageIDBranch line}\n\
  $c create line 185 265 196 265 196 365 380 365 -arrow last \\\n\
    -tags {StageID line}\n\
  $c create line 196 265 196 210 260 210 -arrow last \\\n\
    -tags {StageID line}\n\
  create_dot $c 196 265 \n\
  $c create text 216 202 -font $desfont -text \"IR\"\n\
  $c create text 240 205 -font $desfont -text \"6..10\"\n\
  $c create line 196 230 260 230 -arrow last \\\n\
    -tags {StageID line}\n\
  create_dot $c 196 230 \n\
  $c create text 216 222 -font $desfont -text \"IR\"\n\
  $c create text 240 225 -font $desfont -text \"11..15\"\n\
  $c create line 196 338 240 338 -arrow last  \\\n\
    -tags {StageID line}\n\
  create_dot $c 196 338 \n\
  $c create line 222 333 232 343\n\
  $c create text 230 330 -font $desfont -text \"16\"\n\
  $c create line 185 185 240 185 240 140 280 140 -arrow last \\\n\
    -tags {StageID line}\n\
  $c create text 280 20 -font $insfont -text \"\" \\\n\
    -tags {InstrID text}\n\
  $c create rectangle 380 145 400 380 -fill gray75 \n\
  $c create text 390 135 -font $desfont -text \"ID/EX\"\n\
  $c create line 490 300 530 270 530 230 490 200 490 240 500 250 490 260 490 300\n\
  $c create text 510 250 -font $desfont -text \"ALU\" \n\
  $c create line 530 250 550 250 -arrow last \\\n\
    -tags {StageEX line}\n\
  $c create line 450 205 470 220 -fill lavender -width 3 \\\n\
    -tags {AluARes1 choise}\n\
  $c create line 450 215 470 220 -fill lavender -width 3 \\\n\
    -tags {AluARes2 choise}\n\
  $c create line 450 225 470 220 -fill lavender -width 3 \\\n\
    -tags {AluAMem1 choise}\n\
  $c create line 450 235 470 220 -fill lavender -width 3 \\\n\
    -tags {AluAReg choise}\n\
  $c create rectangle 450 195 470 245\n\
  $c create text 460 210 -font $desfont -text \"M\"\n\
  $c create text 460 220 -font $desfont -text \"U\"\n\
  $c create text 460 230 -font $desfont -text \"X\"\n\
  $c create line 470 220 490 220 -arrow last \\\n\
    -tags {StageEX line}\n\
  $c create line 450 260 470 280 -fill lavender -width 3 \\\n\
    -tags {AluBReg choise}\n\
  $c create line 450 270 470 280 -fill lavender -width 3 \\\n\
    -tags {AluBImme choise}\n\
  $c create line 450 280 470 280 -fill lavender -width 3 \\\n\
    -tags {AluBReg1 choise}\n\
  $c create line 450 290 470 280 -fill lavender -width 3 \\\n\
    -tags {AluBReg2 choise}\n\
  $c create line 450 300 470 280 -fill lavender -width 3 \\\n\
    -tags {AluBMem1 choise}\n\
  $c create rectangle 450 255 470 305\n\
  $c create text 460 270 -font $desfont -text \"M\"\n\
  $c create text 460 280 -font $desfont -text \"U\"\n\
  $c create text 460 290 -font $desfont -text \"X\"\n\
  $c create line 470 280 490 280 -arrow last  \\\n\
    -tags {StageEX line}\n\
  $c create line 400 235 450 235 -arrow last \\\n\
    -tags {StageEX line}\n\
  $c create line 400 260 450 260 -arrow last \\\n\
    -tags {StageEX line}\n\
  $c create line 408 260 408 310 550 310 -arrow last \\\n\
    -tags {StageEX line}\n\
  create_dot $c 408 260\n\
  $c create line 400 338 416 338 416 270 450 270 -arrow last \\\n\
    -tags {StageEX line}\n\
  $c create line 400 365 550 365 -arrow last \\\n\
    -tags {StageEX line}\n\
  $c create text 470 20 -font $insfont -text \"\" \\\n\
    -tags {InstrEX text}\n\
  $c create rectangle 550 145 570 380 -fill gray75 \n\
  $c create text 560 135 -font $desfont -text \"EX/MEM\"\n\
  $c create line 620 220 650 280 -fill lavender -width 3 \\\n\
    -tags {MemRes2 choise}\n\
  $c create line 620 250 650 280 -fill lavender -width 3 \\\n\
    -tags {MemRes1 choise}\n\
  $c create line 620 280 650 280 -fill lavender -width 3 \\\n\
    -tags {MemMem0 choise}\n\
  $c create line 620 333 650 280 -fill lavender -width 3 \\\n\
    -tags {MemMem1 choise}\n\
  $c create line 620 310 650 280 -fill lavender -width 3 \\\n\
    -tags {MemReg choise}\n\
  $c create rectangle 620 215 680 335\n\
  $c create text 650 275 -font $desfont -text \"Data\"\n\
  $c create text 650 285 -font $desfont -text \"memory\"\n\
  $c create line 680 250 720 250 -arrow last \\\n\
    -tags {StageMEM line}\n\
  $c create line 570 250 590 250 \\\n\
    -tags {StageMEM line}\n\
  $c create line 590 250 590 350 720 350 -arrow last \\\n\
    -tags {StageMEM line}\n\
  create_dot $c 590 250\n\
  $c create line 590 350 590 392 424 392 424 280 450 280 -arrow last \\\n\
    -tags {StageMEM line}\n\
  create_dot $c 590 350\n\
  $c create line 590 250 590 100 530 100 530 190 424 190 424 205 450 205 -arrow last \\\n\
    -tags {StageMEM  line}\n\
  $c create line 530 100 335 100 335 155 345 155 -arrow last \\\n\
    -tags {StageMEM line}\n\
  create_dot $c 530 100\n\
  $c create line 740 365 750 365 750 400 204 400 204 250 260 250 -arrow last \\\n\
    -tags {StageWB line}\n\
  $c create text 231 242 -font $desfont -text \"MEM/WB.IR\"\n\
  $c create line 740 350 758 350 758 408 212 408 212 270 260 270 -arrow last \\\n\
    -tags {StageWB line}\n\
  $c create line 432 408 432 290 450 290 -arrow last \\\n\
    -tags {StageWB line}\n\
  create_dot $c 432 408\n\
  $c create line 758 350 758 92 522 92 522 182 416 182 416 215 450 215 -arrow last \\\n\
    -tags {StageWB line}\n\
  create_dot $c 758 350\n\
  $c create line 522 92 327 92 327 163 345 163 -arrow last \\\n\
    -tags {StageWB line}\n\
  create_dot $c 522 92\n\
  $c create line 598 408 598 250 620 250 -arrow last \\\n\
    -tags {StageWB line}\n\
  create_dot $c 598 408\n\
  $c create line 740 250 766 250 766 416 220 416 220 290 260 290 -arrow last \\\n\
    -tags {StageWB line}\n\
  $c create line 440 416 440 300 450 300 -arrow last \\\n\
    -tags {StageWB line}\n\
  create_dot $c 440 416 \n\
  $c create line 766 250 766 84 514 84 514 174 408 174 408 225 450 225 -arrow last \\\n\
    -tags {StageWB line}\n\
  create_dot $c 766 250\n\
  $c create line 514 84 319 84 319 171 345 171 -arrow last \\\n\
    -tags {StageWB line}\n\
  create_dot $c 514 84\n\
  $c create line 606 416 606 280 620 280 -arrow last \\\n\
    -tags {StageWB line}\n\
  create_dot $c 606 416\n\
  $c create line 570 310 620 310 -arrow last \\\n\
    -tags {StageMEM line}\n\
  $c create line 570 365 720 365 -arrow last \\\n\
    -tags {StageMEM line}\n\
  $c create text 640 20 -font $insfont -text \"\" \\\n\
    -tags {InstrMEM text}\n\
  $c create rectangle 720 145 740 380 -fill gray75 \n\
  $c create text 730 135 -font $desfont -text \"MEM/WB\"\n\
  $c create text 780 20 -font $insfont -text \"\" \\\n\
    -tags {InstrWB text}\n\
  button $c.button \\\n\
    -text \"view block diagram\" \\\n\
    -width 20 \\\n\
    -command \"pack forget .bas.datapath \n\
              pack .bas.block\"\n\
  $c create window 80 480 \\\n\
    -window $c.button\n\
  $c create text 665 425 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-* \\\n\
    -text \"IF Cycle of Current Instruction\" \\\n\
    -tag cycleTitle\n\
  label $c.counter \\\n\
    -font $labfont\n\
  $c create window 665 475 \\\n\
    -window $c.counter\n\
  trace variable fastmode w BasClean\n\
}\n\
   ";






char create_dot_tcl[] = "\
proc create_dot {c x y} {\n\
  $c create oval [expr $x-2] [expr $y-2] [expr $x+4] [expr $y+4] -fill black\n\
}\n\
   ";






char BasVisual_tcl[] = "\
proc BasVisual {linelist linecolorlist textlist textcolorlist\n\
    rectlist rectcolorlist} {\n\
  global fastmode\n\
  if {$fastmode} {\n\
    return\n\
  }\n\
  set c .bas.datapath\n\
  $c itemconfigure line -fill black -width 1\n\
  $c itemconfigure choise -fill lavender\n\
  $c itemconfigure text -text \"\"\n\
  set i 0\n\
  foreach line $linelist {\n\
    $c itemconfigure $line -fill  [lindex $linecolorlist $i] -width 3\n\
    incr i\n\
  }\n\
  set i 0\n\
  foreach text $textlist {\n\
    set color_content [lindex $textcolorlist $i]\n\
    $c itemconfigure $text \\\n\
      -fill [lindex $color_content 0] \\\n\
      -text [lrange $color_content 1 end] \n\
    incr i\n\
  }  \n\
  set c .bas.block\n\
  $c itemconfigure rect -fill lavender  \n\
  set i 0\n\
  foreach rect $rectlist {\n\
    set target $rect\n\
    if {$rect == \"RectMEM\" || $rect == \"RectWB\" } { \n\
      set target $rect.1\n\
      set origColor [lindex [$c itemconfigure $target -fill] 4]\n\
      if {$origColor == \"lavender\"} {\n\
        $c itemconfigure $target -fill  [lindex $rectcolorlist $i] \n\
      }\n\
      set target $rect.2\n\
    } \n\
    $c itemconfigure $target -fill  [lindex $rectcolorlist $i] \n\
    incr i\n\
  }\n\
}\n\
   ";






char BasGenTable_tcl[] = "\
proc BasGenTable {head tail IF args} {\n\
  global insCount fastmode\n\
  if {$fastmode} {\n\
    return\n\
  }\n\
  set xstart 180\n\
  set ystart 60\n\
  set c .bas.up\n\
 \n\
  $c.code delete Stage\n\
  $c.cycle delete Stage\n\
  $c.cycle itemconfigure strip -fill lavender \n\
  $c.cycle itemconfigure strip[expr $IF-$head] -fill gray75\n\
  for {set j $head} {$j <= $tail} {incr j} {\n\
    $c.cycle create text [expr $xstart+50*($j-$head)] [expr $ystart-25] \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -text $j \\\n\
      -tags Stage\n\
  }\n\
  for {set j 0} {$j < 5} {incr j} {\n\
    set instr [lindex $args $j]\n\
    set len [llength  $instr]\n\
    set instrStage [lrange $instr 0 [expr $len - 2]]\n\
    set instrColor [lindex [lindex $instr [expr $len - 1]] 0]\n\
    set instrContent [lrange [lindex $instr [expr $len - 1]] 1 end]\n\
    set i 0\n\
      $c.code create text 10 [expr $ystart+$j*25] \\\n\
        -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
        -text $instrContent \\\n\
        -anchor w \\\n\
        -fill $instrColor \\\n\
        -tags Stage \n\
      foreach stage $instrStage {\n\
        if {$stage == \"nonop\"} {\n\
	  set stage \"\"\n\
	}\n\
        $c.cycle create text [expr $xstart+$i*50] [expr $ystart+$j*25] \\\n\
          -font -Adobe-Courier-bold-R-Normal-*-140-* \\\n\
          -text $stage \\\n\
          -tags Stage \n\
        incr i \n\
      } \n\
  }\n\
}  \n\
   ";






char BasClean_tcl[] = "\
proc BasClean {name element op} {\n\
  upvar $name mode\n\
  if {$mode} {\n\
    set w .bas\n\
    $w.up.code delete Stage\n\
    $w.up.cycle delete Stage\n\
    $w.up.cycle itemconfigure strip -fill lavender \n\
    $w.datapath itemconfigure line -fill black -width 1\n\
    $w.datapath itemconfigure choise -fill lavender\n\
    $w.datapath itemconfigure text -text \"\"\n\
    $w.block itemconfigure rect -fill lavender\n\
  }\n\
}\n\
   ";






char BasChangeStrip_tcl[] = "\
proc BasChangeStrip {id} {\n\
  .bas.up.cycle itemconfigure strip -fill lavender \n\
  .bas.up.cycle itemconfigure strip$id -fill gray75\n\
}\n\
   ";






char mkEdit_tcl[] = "\
proc mkEdit {{w .edit}} {\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Edit\"\n\
  wm iconname $w \"edit\"\n\
  wm geometry $w +0+0\n\
  set oldFocus [focus]\n\
  option add Tk$w*Relief raised\n\
  option add Tk$w*Text.Setgrid yes\n\
  option add Tk$w*Text.Height 20\n\
  option add Tk$w*Text.Width 40\n\
  frame $w.text\n\
  frame $w.button  \n\
  pack $w.text \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  pack $w.button \\\n\
    -side bottom \\\n\
    -fill x \\\n\
    -expand yes\n\
  text $w.text.t \\\n\
    -yscrollcommand \"$w.text.s set\"\n\
  scrollbar $w.text.s \\\n\
    -command \"$w.text.t yview\"\n\
  pack $w.text.s \\\n\
    -side right \\\n\
    -fill y \n\
  pack $w.text.t \\\n\
    -expand yes \\\n\
    -fill both\n\
  button $w.button.clean \\\n\
    -text Clean \\\n\
    -command \"$w.text.t delete 1.0 end\"\n\
  button $w.button.load \\\n\
    -text Load \\\n\
    -command \"EditLoad $oldFocus\"\n\
  button $w.button.save \\\n\
    -text \"Save As\" \\\n\
    -command EditSave\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \"destroy $w\n\
      focus $oldFocus\"\n\
  pack $w.button.clean $w.button.load $w.button.save $w.button.cancel\\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes   \n\
  set totalLine [.code.t index end]\n\
  for {set i 1} {$i < $totalLine} {incr i} {\n\
    $w.text.t insert end [.code.t get $i.26 $i.1000]\n\
    $w.text.t insert end \"\\n\"\n\
  }\n\
  focus $w.text.t\n\
  grab $w\n\
}\n\
   ";






char EditLoad_tcl[] = "\
proc EditLoad {oldFocus {w .edit}} {\n\
  global msg fullNameList\n\
  ResetMode 0\n\
  set backNameList $fullNameList\n\
  set fullNameList {}\n\
  set mode [tk_dialog .editload {Load} {Do you want to load some data files before \\\n\
    you load the code?} questhead 0 Yes No]\n\
  if {$mode == \"0\"} {\n\
    mkLoad\n\
    tkwait window .load\n\
  } else {\n\
    foreach file $backNameList {\n\
      if {[string match *.s $file] == \"0\"} {\n\
        Load $file\n\
      }\n\
    }\n\
  }\n\
  \n\
  set f [open /tmp/dlx[pid].s w]\n\
  puts $f [$w.text.t get 1.0 end]\n\
  close $f\n\
  Load  /tmp/dlx[pid].s\n\
  destroy $w\n\
  focus $oldFocus\n\
}\n\
   ";






char EditSave_tcl[] = "\
proc EditSave {{w .editsave}} {\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Save\"\n\
  wm iconname $w \"save\"\n\
  wm geometry $w +300+300\n\
  set oldFocus [focus]\n\
  option add Tk$w*Relief raised\n\
  option add Tk$w*Label.Relief flat\n\
  option add Tk$w*Label.Width 16\n\
  option add Tk$w*Label.Anchor w\n\
  option add Tk$w*Entry.Width 40\n\
  frame $w.entry\n\
  frame $w.button \n\
  pack $w.entry $w.button\\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  frame $w.entry.up \n\
  frame $w.entry.down\n\
  pack $w.entry.up $w.entry.down \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes \n\
  label $w.entry.up.dirlabel \\\n\
    -text \"directory name:\" \n\
  entry $w.entry.up.direntry \n\
  bind $w.entry.up.direntry <Return> \"focus $w.entry.down.fileentry\"\n\
  \n\
  pack $w.entry.up.dirlabel \\\n\
    -side left\n\
  pack $w.entry.up.direntry \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  label $w.entry.down.filelabel \\\n\
    -text \"file name:\"\n\
  entry $w.entry.down.fileentry\n\
  bind $w.entry.down.fileentry <Return> \"EditSaveOk $oldFocus\"\n\
   \n\
  pack $w.entry.down.filelabel \\\n\
    -side left\n\
  pack $w.entry.down.fileentry \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  button $w.button.ok \\\n\
    -text Ok \\\n\
    -command \"EditSaveOk $oldFocus\"\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \"destroy $w\n\
      focus $oldFocus\"\n\
  pack $w.button.ok $w.button.cancel\\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes   \n\
  $w.entry.up.direntry configure -width [string length [pwd]]\n\
  $w.entry.up.direntry insert 0 [pwd]\n\
  focus $w.entry.down.fileentry\n\
  grab $w\n\
}\n\
   ";






char EditSaveOk_tcl[] = "\
proc EditSaveOk {oldFocus {w .editsave}} {\n\
  if {[$w.entry.down.fileentry get] != {}} {\n\
    set filename [$w.entry.up.direntry get]/[$w.entry.down.fileentry get]\n\
    set f [open $filename w]\n\
    puts $f [.edit.text.t get 1.0 end]\n\
    close $f\n\
  }\n\
  destroy $w\n\
  focus $oldFocus\n\
  grab .edit\n\
} \n\
  \n\
   ";






char mkDisc_tcl[] = "\
proc mkDisc {} {\n\
  wm title . \"DLX Visual Simulator\"\n\
  wm iconname . \"dlxview\"\n\
  wm geometry . +0+0\n\
  global backInsCount errInfo config\n\
  set backInsCount 2147483647\n\
  option add Tk.*Background lavender\n\
  option add Tk.*Frame.Relief raised\n\
  option add Tk.*Frame.borderWidth 1\n\
  option add Tk.*Frame.highlightThickness 0\n\
  option add Tk.*Canvas.highlightThickness 0\n\
  option add Tk.*Button.Width 12\n\
  option add Tk.*Button.borderWidth 3\n\
  option add Tk.*Button.foreground black\n\
  option add Tk.*Button.Background skyblue\n\
  option add Tk.*Button.activeForeground white\n\
  option add Tk.*Button.activeBackground MediumOrchid\n\
  option add Tk.*Button.padX 1\n\
  option add Tk.*Button.padY 1\n\
  option add Tk.*Menu.activeBackground MediumOrchid\n\
  option add Tk.*Menubutton.activeBackground MediumOrchid\n\
  option add Tk.*Radiobutton.Background skyblue\n\
  option add Tk.*Radiobutton.activeBackground MediumOrchid\n\
  option add Tk.*Radiobutton.selectColor MediumOrchid\n\
  option add Tk.*Label.Font -Adobe-Helvetica-Bold-R-Normal-*-140-* \n\
  option add Tk.*Label.foreground black\n\
  option add Tk.*Listbox.selectBackground MediumOrchid\n\
  option add Tk.*Text.Relief raised\n\
  option add Tk.*Text.borderWidth 1\n\
  option add Tk.*Text.Font -Adobe-Courier-medium-R-Normal-*-120-*\n\
  option add Tk.*Text.setgrid true\n\
  option add Tk.*Text.height 10 \n\
  option add Tk.*Text.foreground black\n\
  option add Tk.*Scrollbar.Relief raised\n\
  option add Tk.*Scrollbar.borderWidth 1\n\
  option add Tk.*Scrollbar.Background skyblue\n\
  option add Tk.*Scrollbar.activeBackground MediumOrchid \n\
  option add Tk.*Scrollbar.troughColor lavender\n\
  option add Tk.*Scrollbar.highlightThickness 0\n\
  option add Tk.*Scale.from 1\n\
  option add Tk.*Scale.to 8\n\
  option add Tk.*Scale.length 250\n\
  option add Tk.*Scale.orient horizontal\n\
  option add Tk.*Scale.troughColor skyblue\n\
  option add Tk.*Scale.activeBackground MediumOrchid \n\
  frame .button  \n\
  frame .code\n\
  frame .menu \n\
  pack .button .code .menu \\\n\
    -side bottom \\\n\
    -expand yes \\\n\
    -fill both \n\
  frame .button.top \\\n\
    -relief flat\n\
  frame .button.bot \\\n\
    -relief flat \n\
  pack .button.top .button.bot \\\n\
    -side top \\\n\
    -expand yes \\\n\
    -fill both \\\n\
    -padx 10 \n\
  button .button.top.configure \\\n\
    -text configure  \\\n\
    -command \"mkConfigure;  wm iconify .configure ; wm deiconify .configure\" \n\
  button .button.top.load \\\n\
    -text load \\\n\
    -command mkLoad\n\
  bind .button.top.load <Button-2> \"Load [pwd]/f347.i\n\
    Load [pwd]/f347.d\n\
    Load [pwd]/f347.s\n\
    .button.top.load configure -state disabled\n\
    .button.top.stepforward configure -state normal\n\
    .button.bot.stepbackward configure -state normal\n\
    .button.top.nextcycle configure -state normal\n\
    .button.bot.previouscycle configure -state normal\n\
    .button.top.go configure -state normal\n\
    .button.top.reset configure -state normal\n\
    \"\n\
  button .button.top.stepforward \\\n\
    -text \"step forward\" \\\n\
    -command \"mkGoStep Step\"\n\
  button .button.top.nextcycle \\\n\
    -text \"next cycle\" \\\n\
    -command \"mkGoStep Cycle\"\n\
  button .button.top.go \\\n\
    -text go \\\n\
    -command \"mkGoStep Go\"\n\
  button .button.top.reset  \\\n\
    -text reset \\\n\
    -command mkReset \\\n\
    -padx 1 \\\n\
    -pady 1	\n\
  button .button.top.quit \\\n\
    -text quit	\\\n\
    -command exit\n\
  pack .button.top.configure .button.top.load .button.top.stepforward \\\n\
   .button.top.nextcycle .button.top.go .button.top.reset .button.top.quit \\\n\
    -side left \\\n\
    -expand yes \\\n\
    -padx 2 \\\n\
    -pady 5	\n\
  button .button.bot.edit \\\n\
    -text edit \\\n\
    -command mkEdit\n\
  button .button.bot.put \\\n\
    -text \"put & get\" \\\n\
    -command \"PUT\"\n\
  button .button.bot.stepbackward \\\n\
    -text \"step back\" \\\n\
    -command mkStepBack\n\
  button .button.bot.previouscycle \\\n\
    -text \"previous cycle\" \\\n\
    -command mkPreviousCycle\n\
  button .button.bot.stop \\\n\
    -text stop  \\\n\
    -command mkStop\n\
  button .button.bot.stat \\\n\
    -text stat \\\n\
    -command \"STAT\"\n\
  button .button.bot.trace \\\n\
    -text trace  \\\n\
    -command \"mkTrace\"\n\
  pack .button.bot.edit .button.bot.put .button.bot.stepbackward \\\n\
    .button.bot.previouscycle .button.bot.stop .button.bot.stat .button.bot.trace \\\n\
    -side left \\\n\
    -expand yes \\\n\
    -padx 2 \\\n\
    -pady 5\n\
  \n\
  menubutton .menu.b \\\n\
    -text Help \\\n\
    -menu .menu.b.m\n\
 \n\
  menu .menu.b.m \n\
  .menu.b.m add command -label \"Tcl/Tk\" -command mkHelp\n\
  .menu.b.m add command -label \"HTML\" -command \"exec netscape http://yara.ecn.purdue.edu/~teamaaa/disc/help.html &\"\n\
  text .code.t \\\n\
    -yscrollcommand \".code.s set\" \n\
  DisableTextWidget .code.t\n\
  scrollbar .code.s \\\n\
    -command \".code.t yview\" \n\
  pack .menu.b \\\n\
    -side right\n\
  pack .code.s \\\n\
    -side right \\\n\
    -fill y\n\
 \n\
  pack .code.t \\\n\
    -expand yes \\\n\
    -fill both\n\
  label .purdue -bitmap purdue\n\
  place .purdue \\\n\
    -relx 0.88 \\\n\
    -rely 0.40\n\
 .button.top.load configure -state disabled\n\
 .button.top.stepforward configure -state disabled\n\
 .button.bot.stepbackward configure -state disabled\n\
 .button.top.nextcycle configure -state disabled\n\
 .button.bot.previouscycle configure -state disabled\n\
 .button.top.go configure -state disabled\n\
 .button.top.reset configure -state disabled\n\
 .button.bot.put configure -state disabled\n\
 .button.bot.stop configure -state disabled\n\
 .button.bot.stat configure -state disabled\n\
 \n\
}\n\
   ";






char mkStepBack_tcl[] = "\
proc mkStepBack {} {\n\
  global startAddress backInsCount insCount config fastmode\n\
  if {$insCount == 1} {\n\
    tk_dialog .warning {Warning} {You've already reached the beginning of your \\\n\
      program, no further back step can be excecuted.} warning 0 OK\n\
    return\n\
  }\n\
  if {$config} {\n\
    set backInsCount [expr $insCount-1]\n\
    ResetMode 3\n\
    Go $startAddress\n\
  } else {\n\
    if {$insCount == \"2\"} {\n\
      set backInsCount 0\n\
      ResetMode 3\n\
      Step $startAddress\n\
    } else {\n\
      set backInsCount [expr $insCount-2]\n\
      ResetMode 3\n\
      Go $startAddress\n\
      set fastmode 0\n\
      step\n\
    }\n\
  }\n\
}\n\
   ";






char mkPreviousCycle_tcl[] = "\
proc mkPreviousCycle {} {\n\
  global startAddress backInsCount insCount cycleCount cycleDisplayCount \n\
  global config msg fastmode\n\
    switch $config {\n\
      0 {set counter .bas.datapath.counter;  set startCycle 0}\n\
      1 {set counter .tom.c.counter;  set startCycle 1}\n\
      2 { set counter .sco.c.counter;  set startCycle 1}\n\
    }\n\
 \n\
   set backCycle [lindex [$counter configure -text] 4]\n\
  if {$backCycle == $startCycle} {\n\
    tk_dialog .warning {Warning} {You've already reached the beginning of your \\\n\
      program, no previous cycle can be reached.} warning 0 OK\n\
    return\n\
  }\n\
  if {$insCount > 3} {\n\
    set backInsCount [expr $insCount-2]\n\
    ResetMode 3\n\
    Go $startAddress\n\
  } else {\n\
    ResetMode 3\n\
    Cycle $startAddress\n\
  }\n\
  set fastmode 0\n\
  while {$cycleDisplayCount < $backCycle} {\n\
    cycle\n\
  }\n\
}\n\
   ";






char DeleteAndInsert_tcl[] = "\
proc DeleteAndInsert {w content} {\n\
  $w delete 1.0 end\n\
  $w insert 1.0 $content\n\
}\n\
   ";






char InsertAndHighLight_tcl[] = "\
proc InsertAndHighLight {w content} {\n\
  global highLightList\n\
  $w tag delete gray\n\
  $w delete 1.0 end\n\
  $w insert 1.0 $content\n\
  $w tag add toggle 1.0 end\n\
  $w tag configure toggle -foreground white -background black\n\
  lappend highLightList $w\n\
}\n\
   ";






char UnHighLight_tcl[] = "\
proc UnHighLight {} {\n\
  global highLightList \n\
  foreach w $highLightList {\n\
    $w tag configure toggle -foreground black -background gainsboro\n\
  }\n\
  set highLightList {}\n\
}\n\
   ";






char AddToEnd_tcl[] = "\
proc AddToEnd {w content} {\n\
  \n\
  $w insert end $content\n\
  $w insert end \"\n\
\"\n\
  $w yview -pickplace \"end -1 lines\"\n\
}\n\
    \n\
   ";






char HighLightCodeLine_tcl[] = "\
proc HighLightCodeLine {lineNum} {\n\
  .code.t tag remove toggle 1.0 end\n\
  .code.t tag add toggle $lineNum.0 [expr $lineNum+1].0\n\
  .code.t tag configure toggle -foreground white -background black\n\
  .code.t yview -pickplace $lineNum.0\n\
}\n\
   ";






char Trap0_tcl[] = "\
proc Trap0 {} {\n\
 .button.top.stepforward configure -state disabled\n\
 .button.top.nextcycle configure -state disabled\n\
 .button.top.go configure -state disabled\n\
}\n\
   ";






char Err_tcl[] = "\
proc Err {errMsg} {\n\
  set line \"Exception: $errMsg. Edit your program, and run it again\"\n\
  tk_dialog .warning {Warning} $line warning 0 OK\n\
  ResetMode 3\n\
  update\n\
}\n\
   ";






char ChangeCycleTitle_tcl[] = "\
proc ChangeCycleTitle {cycleMode} {\n\
  global config\n\
  grab set .code.t\n\
  switch $config {\n\
    0 {\n\
        if {$cycleMode} {\n\
          .bas.datapath itemconfigure cycleTitle -text \"Current Cycle\"\n\
          .bas.block itemconfigure cycleTitle -text \"Current Cycle\"\n\
        } else {\n\
          .bas.datapath itemconfigure cycleTitle -text \"IF Cycle of Current Instruction\"\n\
          .bas.block itemconfigure cycleTitle -text \"IF Cycle of Current Instruction\"\n\
          .bas.datapath.counter configure -bitmap hourglass\n\
          .bas.block.counter configure -bitmap hourglass\n\
        }\n\
      }\n\
    1 {\n\
        if {$cycleMode} {\n\
          .tom.c itemconfigure cycleTitle -text \"Current Cycle\"\n\
        } else {\n\
          .tom.c itemconfigure cycleTitle -text \"Issue Cycle of Current Instruction\"\n\
          .tom.c.counter configure -bitmap hourglass\n\
        }\n\
        catch {destroy .equation}\n\
      }\n\
    2 {\n\
        if {$cycleMode} {\n\
          .bas.c itemconfigure cycleTitle -text \"Current Cycle\"\n\
        } else {\n\
          .bas.c itemconfigure cycleTitle -text \"Issue Cycle of Current Instruction\"\n\
          .bas.c.counter configure -bitmap hourglass\n\
        }\n\
        catch {destroy .equation}\n\
      }\n\
  }\n\
  update\n\
} \n\
   ";






char ChangeCycleIcon_tcl[] = "\
proc ChangeCycleIcon {} {\n\
  global config \n\
  switch $config {\n\
    0 {\n\
        .bas.datapath.counter configure -bitmap \"\"\n\
        .bas.block.counter configure -bitmap \"\"\n\
      }\n\
    1 {\n\
        .tom.c.counter configure -bitmap \"\"\n\
      }\n\
    2 {\n\
        .sco.c.counter configure -bitmap \"\"\n\
      }\n\
  }\n\
  grab release .code.t\n\
}\n\
   ";






char Start_tcl[] = "\
proc Start {} {\n\
  global pipeline_mode input_files start_address\n\
  global config configureEnabled\n\
  switch $pipeline_mode {\n\
    BasicPipe {\n\
      set config 0\n\
      mkBasicPipe\n\
    }\n\
    Tomasulo {\n\
      set config 1\n\
      mkTomasulo\n\
    }\n\
    ScoreBoard {\n\
      set config 2\n\
      mkScoreBoard\n\
    }\n\
    default {\n\
      puts \"Unknown Pipelining Mode\"\n\
      exit\n\
    }\n\
  }\n\
  foreach f $input_files {\n\
    if {[string compare [string range $f 0 0] \"/\"] == \"0\"} {\n\
      Load $f\n\
    } else {\n\
      Load [pwd]/$f\n\
    }\n\
  }\n\
  \n\
  SubConfigureOk\n\
  set configureEnabled 0\n\
  .button.top.load configure -state disabled\n\
  .button.top.stepforward configure -state normal\n\
  .button.bot.stepbackward configure -state normal\n\
  .button.top.go configure -state normal\n\
  .button.top.nextcycle configure -state normal\n\
  .button.bot.previouscycle configure -state normal\n\
  .button.top.reset configure -state normal\n\
  if {[info exists start_address]} {\n\
      Step $start_address\n\
  } else {\n\
      Step 256\n\
  }\n\
}\n\
   ";






char DisableTextWidget_tcl[] = "\
proc DisableTextWidget {w} {\n\
  bind $w <Key> {break}\n\
  bind $w <Button-1> {break}\n\
  bind $w <Double-Button-1> {break}\n\
  bind $w <Triple-Button-1> {break}\n\
  bind $w <B1-Motion> {break}\n\
  bind $w <B1-ButtonRelease> {break}\n\
}\n\
   ";






char mkGoStep_tcl[] = "\
proc mkGoStep {choice {w .gostep}} {\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w $choice\n\
  wm iconname $w $choice\n\
  wm geometry $w +0+0\n\
  set oldFocus [focus]\n\
  option add Tk$w*Relief raised\n\
  option add Tk$w*Button.borderWidth 3\n\
  label $w.label \\\n\
    -relief flat \\\n\
    -font -Adobe-Courier-Bold-R-Normal-*-140-* \\\n\
    -text \"$choice from:\"\n\
  frame $w.entryok\n\
  frame $w.button\n\
  pack $w.label $w.entryok \\\n\
    -side top \\\n\
    -padx 10 \\\n\
    -pady 4\n\
 \n\
  pack $w.button \\\n\
    -side bottom \\\n\
    -expand yes \\\n\
    -fill x \\\n\
    -padx 10 \\\n\
    -pady 4\n\
  entry $w.entryok.entry \\\n\
    -width 20 \n\
  bind $w.entryok.entry <Return> \"$choice \\[$w.entryok.entry get \\]\n\
    destroy $w\n\
    focus $oldFocus\"\n\
 \n\
  frame $w.entryok.frame \\\n\
    -width 10\n\
  button $w.entryok.ok \\\n\
    -text Ok \\\n\
    -command \"$choice \\[$w.entryok.entry get \\]\n\
      destroy $w\n\
      focus $oldFocus\"\n\
  pack $w.entryok.entry $w.entryok.frame $w.entryok.ok \\\n\
    -side left \n\
  button $w.button.default \\\n\
    -text Default \\\n\
    -command \"destroy $w\n\
      focus $oldFocus\n\
      $choice 256\"\n\
 \n\
  button $w.button.main \\\n\
    -text Main \\\n\
    -command \"destroy $w\n\
      focus $oldFocus\n\
      $choice _main\"\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \"destroy $w\n\
      focus $oldFocus\"\n\
 \n\
  pack $w.button.default $w.button.main $w.button.cancel \\\n\
    -side left \\\n\
    -expand yes \\\n\
    -fill x \n\
  focus $w.entryok.entry\n\
}\n\
   ";






char Go_tcl[] = "\
proc Go {addr} {\n\
  global startAddress msg highLightList background cycleCount\n\
  global fastmode\n\
  if {$startAddress == \"\"} {\n\
    set startAddress $addr\n\
  }\n\
  go $addr\n\
  .button.top.go configure -command go\n\
  .button.top.stepforward configure -command step\n\
  .button.top.nextcycle configure -command cycle\n\
}\n\
   ";






char Step_tcl[] = "\
proc Step {addr} {\n\
  global startAddress msg highLightList background cycleCount\n\
  if {$startAddress == \"\"} {\n\
    set startAddress $addr\n\
  }\n\
  step $addr \n\
  .button.top.go configure -command go\n\
  .button.top.stepforward configure -command step\n\
  .button.top.nextcycle configure -command cycle\n\
}\n\
   ";






char Cycle_tcl[] = "\
proc Cycle {addr} {\n\
  global startAddress msg highLightList background cycleCount\n\
  if {$startAddress == \"\"} {\n\
    set startAddress $addr\n\
  }\n\
  cycle $addr\n\
  .button.top.go configure -command go\n\
  .button.top.stepforward configure -command step\n\
  .button.top.nextcycle configure -command cycle\n\
}\n\
   ";






char demo_tcl[] = "\
proc demo {op} {\n\
  global startAddress msg highLightList background cycleCount\n\
  if {$op == \"start\"} {\n\
      proc demoact {} [info body demoactbody]\n\
  } else {\n\
      proc demoact {} {}\n\
  }\n\
  demoact\n\
}\n\
   ";






char demoactbody_tcl[] = "\
proc demoactbody {} {\n\
  global startAddress msg highLightList background cycleCount\n\
  if {[lindex [.button.top.stepforward configure -state] 4] == \"normal\"} {\n\
    step \n\
    update\n\
      after 1000 demoact\n\
  }\n\
}\n\
   ";






char mkLoad_tcl[] = "\
proc mkLoad {{w .load}} {\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Load DLX Program, Data, and Initialize Registers\"\n\
  wm iconname $w \"load\"\n\
  wm geometry $w +0+0\n\
  set oldFocus [focus]\n\
  option add Tk$w*Relief raised\n\
  option add Tk$w*Listbox.Setgrid yes\n\
  option add Tk$w*Label.Relief flat\n\
  option add Tk$w*Label.Width 16\n\
  option add Tk$w*Label.Anchor w\n\
  option add Tk$w*Entry.Width 40\n\
  \n\
  frame $w.list \n\
  frame $w.entry  \n\
  label $w.label\n\
  frame $w.button  \n\
  pack $w.list \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
    \n\
  pack $w.entry \\\n\
    -side top \\\n\
    -pady 5 \\\n\
    -fill x \\\n\
    -expand yes\n\
  pack $w.label $w.button \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  listbox $w.list.dirlist \\\n\
    -yscrollcommand \"$w.list.dirscroll set\"       \n\
  bind $w.list.dirlist <Button-1> {\n\
    %W selection clear 0 end\n\
    .load.button.load configure -state normal\n\
    set num [%W nearest %y]\n\
    %W selection set $num\n\
    LoadOpen [%W get $num]\n\
  }\n\
  bind $w.list.dirlist <Button1-Motion> {%W selection clear 0 end\n\
    set num [%W nearest %y]\n\
    %W selection set $num\n\
    set fHeaderExt [%W get $num]\n\
    .load.button.load configure -state normal\n\
    set length [string length $fHeaderExt]\n\
    set fHeader [string range $fHeaderExt 0 [expr $length-3]]\n\
    if {[string match *.d $fHeaderExt]} {\n\
      if {[string compare $fHeader.i [%W get [expr $num+1]]] == \"0\"} {\n\
        %W selection set [expr $num+1]\n\
        if {[string compare $fHeader.s [%W get [expr $num+2]]] == \"0\"} {\n\
          %W selection set [expr $num+2]\n\
        }\n\
      } elseif {[string compare $fHeader.s [%W get [expr $num+1]]] == \"0\"} {\n\
        %W selection set [expr $num+1]\n\
      }     \n\
    }\n\
    if {[string match *.i $fHeaderExt]} {\n\
      if {[string compare $fHeader.d [%W get [expr $num-1]]] == \"0\"} {\n\
        %W selection set [expr $num-1]\n\
      }    \n\
      if {[string compare $fHeader.s [%W get [expr $num+1]]] == \"0\"} {\n\
        %W selection set [expr $num+1]\n\
      }       \n\
    }\n\
    if {[string match *.s $fHeaderExt]} {\n\
      if {[string compare $fHeader.i [%W get [expr $num-1]]] == \"0\"} {\n\
        %W selection set [expr $num-1]\n\
        if {[string compare $fHeader.d [%W get [expr $num-2]]] == \"0\"} {\n\
         %W selection set [expr $num-2]\n\
        }           \n\
      } elseif {[string compare $fHeader.d [%W get [expr $num-1]]] == \"0\"} {\n\
        %W selection set [expr $num-1]\n\
      }           \n\
    }\n\
    set fileList {}\n\
    foreach fileNum [%W curselection] {\n\
      set fileTail [%W get $fileNum]\n\
      lappend fileList $fileTail\n\
    }\n\
    LoadOpen $fileList\n\
  }\n\
  scrollbar $w.list.dirscroll \\\n\
    -command \"$w.list.dirlist yview\"\n\
  listbox $w.list.filelist \\\n\
    -yscrollcommand \"$w.list.filescroll set\"\n\
  scrollbar $w.list.filescroll \\\n\
    -command \"$w.list.filelist yview\"   \n\
 \n\
  pack $w.list.dirlist $w.list.dirscroll \\\n\
    -side left \\\n\
    -fill y \n\
  pack $w.list.filelist \\\n\
    -side left \\\n\
    -fill both \\\n\
    -expand yes\n\
  pack $w.list.filescroll \\\n\
    -side right \\\n\
    -fill y \n\
  frame $w.entry.up \n\
  frame $w.entry.down\n\
  pack $w.entry.up $w.entry.down \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
    \n\
  label $w.entry.up.dirlabel \\\n\
    -text \"directory name:\" \n\
  entry $w.entry.up.direntry \n\
  bind $w.entry.up.direntry <Return> \"focus $w.entry.down.fileentry\n\
    LoadOpen \\[$w.entry.up.direntry get \\]\"\n\
  \n\
  pack $w.entry.up.dirlabel \\\n\
    -side left\n\
  pack $w.entry.up.direntry \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  label $w.entry.down.filelabel \\\n\
    -text \"file name:\"\n\
  entry $w.entry.down.fileentry\n\
  bind $w.entry.down.fileentry <Return> FileEntryReturn  \n\
  pack $w.entry.down.filelabel \\\n\
    -side left\n\
  pack $w.entry.down.fileentry \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  button $w.button.load \\\n\
    -text Load \\\n\
    -command LoadFile\n\
  button $w.button.done \\\n\
    -text Done \\\n\
    -command LoadDone\n\
  button $w.button.cancel \\\n\
    -text Cancel \\\n\
    -command \"LoadCancel $oldFocus\"\n\
  pack $w.button.load $w.button.done $w.button.cancel\\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes   \n\
  global curDir env\n\
  if {[info exists env(DISC_INPUT)] && [file isdirectory $env(DISC_INPUT)]} {\n\
    set curDir $env(DISC_INPUT)\n\
  } else {\n\
    set curDir [pwd]\n\
  }\n\
  LoadRefresh\n\
  focus $w.entry.down.fileentry\n\
  grab $w\n\
}\n\
   ";






char LoadOpen_tcl[] = "\
proc LoadOpen {fname {w .load}} {\n\
  global curDir\n\
  if {[llength $fname] > \"1\"} {\n\
    $w.entry.down.fileentry delete 0 end\n\
    $w.entry.down.fileentry insert 0 \"Multiple files are selected\"\n\
    $w.list.filelist delete 0 end\n\
  } elseif {[file isdirectory $fname]} {\n\
    cd $fname\n\
    set curDir [pwd]\n\
    LoadRefresh\n\
  } else {\n\
    set thisfile [open $fname r]\n\
    $w.entry.down.fileentry delete 0 end\n\
    $w.entry.down.fileentry insert 0 $fname\n\
    $w.list.filelist delete 0 end\n\
    for {set i 1} {$i < 50} {incr i} {\n\
      $w.list.filelist insert end [gets $thisfile]\n\
    }\n\
    update\n\
    close $thisfile\n\
  }\n\
}\n\
   ";






char LoadRefresh_tcl[] = "\
proc LoadRefresh {{w .load}} {\n\
  global curDir \n\
  set list [exec ls -a]\n\
  $w.button.load configure -state disabled\n\
  $w.entry.up.direntry delete 0 end\n\
  $w.entry.up.direntry insert 0 $curDir\n\
  $w.entry.up.direntry config -width [string length $curDir]\n\
  $w.entry.down.fileentry delete 0 end\n\
  $w.list.filelist delete 0 end\n\
  $w.list.dirlist delete 0 end\n\
  foreach i $list {\n\
    if {[string match *.d $i] || [string match *.i $i] || \n\
        [string match *.s $i] || [file isdirectory $i]} {\n\
      $w.list.dirlist insert end $i\n\
    }\n\
  }\n\
  update\n\
}\n\
   ";






char LoadFile_tcl[] = "\
proc LoadFile {{w .load}} {\n\
  global curDir \n\
  set fileNumList [$w.list.dirlist curselection]\n\
  set fileList {}\n\
  foreach fileNum $fileNumList {\n\
    set fileTail [$w.list.dirlist get $fileNum]\n\
    lappend fileList $fileTail\n\
    set fullName $curDir/$fileTail\n\
    Load $fullName\n\
  }\n\
  if {[llength $fileList] == \"1\"} {\n\
    $w.label configure -text \"$fileList has been loaded.\"\n\
  } else {\n\
    $w.label configure -text \"$fileList have been loaded.\"\n\
  }\n\
} \n\
   ";






char Load_tcl[] = "\
proc Load {args} {\n\
  global msg fullNameList\n\
  foreach file $args {\n\
    if {[lsearch -exact $fullNameList $file] == -1} {\n\
      set fullNameList [concat $fullNameList $file] \n\
      if {[string match *.i $file]} {\n\
        source $file\n\
      } elseif {[string match *.d $file]} {\n\
        load $file\n\
      }  elseif {[string match *.s $file]} {\n\
        load $file\n\
      }\n\
    }\n\
  }\n\
} \n\
   ";






char LoadDone_tcl[] = "\
proc LoadDone {} {\n\
  global fullNameList config\n\
  set w .load\n\
  if {$fullNameList != {}} {\n\
    .button.top.load configure -state disabled\n\
    .button.top.stepforward configure -state normal\n\
    .button.bot.stepbackward configure -state normal\n\
    .button.top.nextcycle configure -state normal\n\
    .button.bot.previouscycle configure -state normal\n\
    .button.top.go configure -state normal\n\
  }\n\
  .button.top.reset configure -state normal\n\
  destroy $w\n\
}\n\
   ";






char LoadCancel_tcl[] = "\
proc LoadCancel {oldFocus {w .load}} {\n\
  global ld_st_exist fp_div_exist fullNameList\n\
  if {$fullNameList == {}} {\n\
    destroy $w\n\
    focus $oldFocus\n\
    return\n\
  }\n\
  set mode [tk_dialog .loadcancel {Cancel} {This will force all the files \\\n\
    (instructions and data) loaded previously being moved out.} \\\n\
    warning 0 Ok Cancel]\n\
 \n\
  switch $mode {\n\
    0 {\n\
      set backLd_st_exist $ld_st_exist\n\
      set backFp_div_exist $fp_div_exist\n\
      init\n\
      .code.t delete 1.0 end\n\
      set ld_st_exist $backLd_st_exist\n\
      set fp_div_exist $backFp_div_exist\n\
      set fullNameList {}\n\
      destroy $w\n\
      focus $oldFocus\n\
    }\n\
  }\n\
}\n\
   ";






char FileEntryReturn_tcl[] = "\
proc FileEntryReturn {} {\n\
  set w .load\n\
  set fileTail [$w.entry.down.fileentry get]\n\
  .load.button.load configure -state normal\n\
  if {[file exists $fileTail]} {\n\
    LoadOpen $fileTail\n\
    Load [pwd]/$fileTail\n\
    $w.label config -text \"$fileTail has been loaded\"\n\
  } else {\n\
    $w.label config -text \"$fileTail doesn't exist\n\
  }\n\
}\n\
   ";






char mkReset_tcl[] = "\
proc mkReset {} {\n\
  global fullNameList \n\
  \n\
  option add Tk.reset*Button.Width 25\n\
  \n\
  set mode [tk_dialog .reset {Reset DLX Machine} {Reset your machine with} \\\n\
    warning 1 \\\n\
    {Same Config & New Program} {New Config & Same Program} \\\n\
    {New Config & New Program} {Same Config & Same Program} {Cancel}]\n\
 \n\
  if {$mode == \"4\"} {\n\
    return\n\
  }\n\
  ResetMode $mode\n\
  if {$mode == \"0\"} {\n\
    set fullNameList {}\n\
    while {$fullNameList == {}} {\n\
      mkLoad\n\
      tkwait window .load\n\
    }\n\
  }\n\
}\n\
   ";






char ResetMode_tcl[] = "\
proc ResetMode {mode} {\n\
  global config highLightList \n\
  global fullNameList \n\
  global num_int_units num_add_units num_mul_units \n\
  global num_div_units num_load_bufs num_store_bufs\n\
  global fp_div_exist ld_st_exist\n\
  global configureEnabled \n\
  set highLightList {}\n\
  \n\
  if {$mode == \"4\"} {\n\
    return\n\
  }\n\
  init\n\
  .code.t delete 1.0 end\n\
  .button.top.go configure -command \"mkGoStep Go\"\n\
  .button.top.stepforward configure -command \"mkGoStep Step\"\n\
  .button.top.nextcycle configure -command \"mkGoStep Cycle\"\n\
  switch $mode {\n\
    0 {\n\
      switch $config {\n\
        0 {\n\
          .bas.up.code delete Stage\n\
          .bas.up.cycle delete Stage\n\
          .bas.up.cycle itemconfigure strip -fill lavender \n\
          .bas.datapath itemconfigure line -fill black -width 1\n\
	  .bas.datapath itemconfigure choise -fill lavender\n\
	  .bas.datapath itemconfigure text -text \"\"\n\
          .bas.datapath.counter configure -text {}\n\
          .bas.block itemconfigure rect -fill lavender\n\
          .bas.block.counter configure -text {}\n\
        }\n\
        1 {\n\
          set c .tom.c\n\
          $c delete thickLine\n\
          for {set i 0} {$i < 64} {incr i} {\n\
            $c.reg$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_int_units} {incr i} {\n\
            $c.int$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_add_units} {incr i} {\n\
            $c.add$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_mul_units} {incr i} {\n\
            $c.mul$i delete 1.0 end\n\
          }  \n\
          $c.instr delete 1.0 end  \n\
          if {$num_div_units > \"0\"} {\n\
            for {set i 0} {$i < $num_div_units} {incr i} {\n\
              $c.div$i delete 1.0 end\n\
            } \n\
            set fp_div_exist 1\n\
          } else {\n\
            set fp_div_exist 0\n\
          }\n\
          if {$num_load_bufs > \"0\"} {\n\
            for {set i 0} {$i < $num_load_bufs} {incr i} {\n\
              $c.load$i delete 1.0 end\n\
            } \n\
            for {set i 0} {$i < $num_store_bufs} {incr i} {\n\
              $c.store$i delete 1.0 end\n\
            } \n\
            set ld_st_exist 1\n\
          } else {\n\
            set ld_st_exist 0\n\
          }\n\
          $c.frame.table delete 1.0 end    \n\
          $c.counter configure -text {}    \n\
        }\n\
        2 {\n\
          set c .sco.c\n\
          $c delete thickLine\n\
          for {set i 0} {$i < 64} {incr i} {\n\
            $c.reg$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_int_units} {incr i} {\n\
            $c.int$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_add_units} {incr i} {\n\
            $c.add$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_mul_units} {incr i} {\n\
            $c.mul$i delete 1.0 end\n\
          }    \n\
          if {$num_div_units > \"0\"} {\n\
            for {set i 0} {$i < $num_div_units} {incr i} {\n\
              $c.div$i delete 1.0 end\n\
            } \n\
            set fp_div_exist 1\n\
          } else {\n\
            set fp_div_exist 0\n\
          }\n\
          $c.frame.table delete 1.0 end \n\
          $c.counter configure -text {}       \n\
        }\n\
      } \n\
    }\n\
    1 {\n\
      if {[winfo exists .bas]} {\n\
        destroy .bas\n\
      }\n\
      if {[winfo exists .tom]} {\n\
        destroy .tom\n\
      }\n\
      if {[winfo exists .sco]} {\n\
        destroy .sco\n\
      }\n\
      set backFullNameList $fullNameList\n\
      set fullNameList {}\n\
      foreach file $backFullNameList {\n\
        Load $file\n\
      }\n\
      set configureEnabled 1\n\
      while {![winfo exists .bas] && ![winfo exists .tom] && ![winfo exists .sco]} {\n\
     	.button.top.load configure -state disabled\n\
        .button.top.stepforward configure -state normal\n\
        .button.bot.stepbackward configure -state normal\n\
        .button.top.nextcycle configure -state normal\n\
        .button.bot.previouscycle configure -state normal\n\
        .button.top.go configure -state normal\n\
        mkConfigure\n\
        tkwait window .configure\n\
      } \n\
    }\n\
    2 {\n\
      if {[winfo exists .bas]} {\n\
        destroy .bas\n\
      }\n\
      if {[winfo exists .tom]} {\n\
        destroy .tom\n\
      }\n\
      if {[winfo exists .sco]} {\n\
        destroy .sco\n\
      }\n\
      set fullNameList {}\n\
      set configureEnabled 1\n\
      .button.top.load configure -state disabled\n\
      .button.top.stepforward configure -state disabled\n\
      .button.bot.stepbackward configure -state disabled\n\
      .button.top.nextcycle configure -state disabled\n\
      .button.bot.previouscycle configure -state disabled\n\
      .button.top.go configure -state disabled\n\
      .button.bot.stop configure -state disabled\n\
      .button.top.reset configure -state disabled\n\
    }\n\
    3 {\n\
      switch $config {\n\
        0 {\n\
          .bas.up.code delete Stage\n\
          .bas.up.cycle delete Stage\n\
          .bas.up.cycle itemconfigure strip -fill lavender \n\
          .bas.datapath itemconfigure line -fill black -width 1\n\
	  .bas.datapath itemconfigure choise -fill lavender\n\
	  .bas.datapath itemconfigure text -text \"\"\n\
          .bas.datapath.counter configure -text {}\n\
          .bas.block itemconfigure rect -fill lavender\n\
          .bas.block.counter configure -text {}\n\
        }\n\
        1 {\n\
          set c .tom.c\n\
          $c delete thickLine\n\
          for {set i 0} {$i < 64} {incr i} {\n\
            $c.reg$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_int_units} {incr i} {\n\
            $c.int$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_add_units} {incr i} {\n\
            $c.add$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_mul_units} {incr i} {\n\
            $c.mul$i delete 1.0 end\n\
          }  \n\
          $c.instr delete 1.0 end  \n\
          if {$num_div_units > \"0\"} {\n\
            for {set i 0} {$i < $num_div_units} {incr i} {\n\
              $c.div$i delete 1.0 end\n\
            } \n\
            set fp_div_exist 1\n\
          } else {\n\
            set fp_div_exist 0\n\
          }\n\
          if {$num_load_bufs > \"0\"} {\n\
            for {set i 0} {$i < $num_load_bufs} {incr i} {\n\
              $c.load$i delete 1.0 end\n\
            } \n\
            for {set i 0} {$i < $num_store_bufs} {incr i} {\n\
              $c.store$i delete 1.0 end\n\
            } \n\
            set ld_st_exist 1\n\
          } else {\n\
            set ld_st_exist 0\n\
          }\n\
          $c.frame.table delete 1.0 end    \n\
          $c.counter configure -text {}    \n\
        }\n\
        2 {\n\
          set c .sco.c\n\
          $c delete thickLine\n\
          for {set i 0} {$i < 64} {incr i} {\n\
            $c.reg$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_int_units} {incr i} {\n\
            $c.int$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_add_units} {incr i} {\n\
            $c.add$i delete 1.0 end\n\
          }\n\
          for {set i 0} {$i < $num_mul_units} {incr i} {\n\
            $c.mul$i delete 1.0 end\n\
          }    \n\
          if {$num_div_units > \"0\"} {\n\
            for {set i 0} {$i < $num_div_units} {incr i} {\n\
              $c.div$i delete 1.0 end\n\
            } \n\
            set fp_div_exist 1\n\
          } else {\n\
            set fp_div_exist 0\n\
          }\n\
          $c.frame.table delete 1.0 end \n\
          $c.counter configure -text {}       \n\
        }\n\
      } \n\
      set backFullNameList $fullNameList\n\
      set fullNameList {}\n\
      foreach file $backFullNameList {\n\
        Load $file\n\
      }\n\
      .button.top.stepforward configure -state normal\n\
      .button.top.go configure -state normal\n\
      .button.top.nextcycle configure -state normal\n\
    }\n\
  }\n\
}\n\
   ";






char mkScoreboard_tcl[] = "\
proc mkScoreboard {{w .sco}} {\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Scoreboard Algorithm on DLX\"\n\
  wm iconname $w \"scoreboard\"\n\
  wm geometry $w +0+0\n\
  set c $w.c\n\
  option add Tk$w*Text.Background gainsboro\n\
  canvas $c \\\n\
    -width 800 \\\n\
    -height 670 \\\n\
    -relief raised\n\
  pack $c\n\
  global num_int_units num_add_units num_mul_units \n\
  global num_div_units num_load_bufs num_store_bufs\n\
  global fp_div_exist ld_st_exist\n\
  for {set i 0} {$i < 64} {incr i} {\n\
    text $c.reg$i \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -setgrid true \\\n\
      -height 1 \\\n\
      -width 7 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
    DisableTextWidget $c.reg$i \n\
    $c create window [expr 50+$i%2*42] [expr 50+$i/2*13] \\\n\
      -anchor nw \\\n\
      -window $c.reg$i\n\
    if {$i%2 == 0} {\n\
      if {$i < 32} {\n\
        set reg R$i\n\
      } else {\n\
        set reg F[expr $i-32]\n\
      }\n\
      $c create text 25 [expr 57+$i/2*13] \\\n\
        -anchor w \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
        -text $reg\n\
    }\n\
  }\n\
  $c create text 100 40 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"Registers\"\n\
  set top [expr 102-$num_int_units*5]\n\
  for {set i 0} {$i < $num_int_units} {incr i} {\n\
    $c create rectangle 250 [expr $top+$i*10]  290 [expr $top+$i*10+9]\n\
  }\n\
  $c create text 330 112 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"Integer Unit\"\n\
  $c create line 139 98 250 98 \\\n\
    -arrow last\n\
  $c create line 139 106 250 106 \\\n\
    -arrow last\n\
  $c create line 290 102 325 102 325 56 139 56 \\\n\
    -arrow last\n\
  set bot [expr 102+$num_int_units*5-1]\n\
  \n\
  set top [expr 206-$num_add_units*5]\n\
  $c create line 270 $top 270 $bot \\\n\
    -arrow last\n\
  for {set i 0} {$i < $num_add_units} {incr i} {\n\
    $c create rectangle 250 [expr $top+$i*10]  290 [expr $top+$i*10+9]\n\
  }\n\
  $c create text 330 216 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"FP add\"\n\
  $c create text 200 150 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"Data buses\"\n\
  $c create line 139 202 250 202 \\\n\
    -arrow last\n\
  $c create line 139 210 250 210 \\\n\
    -arrow last\n\
  $c create line 290 206 325 206 325 160 139 160 \\\n\
    -arrow last\n\
  set bot [expr 206+$num_add_units*5-1]\n\
  set top [expr 310-$num_mul_units*5]\n\
  $c create line 270 $top 270 $bot \\\n\
    -arrow last\n\
  for {set i 0} {$i < $num_mul_units} {incr i} {\n\
    $c create rectangle 250 [expr $top+$i*10]  290 [expr $top+$i*10+9]\n\
  }\n\
  $c create text 330 320 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"FP mul\"\n\
  $c create line 139 306 250 306 \\\n\
    -arrow last\n\
  $c create line 139 314 250 314 \\\n\
    -arrow last\n\
  $c create line 290 310 325 310 325 264 139 264 \\\n\
    -arrow last\n\
  set bot [expr 310+$num_mul_units*5-1]\n\
  if {$fp_div_exist} {\n\
    set top [expr 414-$num_div_units*5]\n\
    $c create line 270 $top 270 $bot \\\n\
      -arrow last\n\
    for {set i 0} {$i < $num_div_units} {incr i} {\n\
      $c create rectangle 250 [expr $top+$i*10]  290 [expr $top+$i*10+9]\n\
    }\n\
    $c create text 330 424 \\\n\
      -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
      -text \"FP divide\"\n\
    $c create line 139 410 250 410 \\\n\
      -arrow last\n\
    $c create line 139 418 250 418 \\\n\
      -arrow last\n\
    $c create line 290 414 325 414 325 368 139 368 \\\n\
      -arrow last\n\
    set bot [expr 414+$num_div_units*5-1]\n\
  }\n\
  $c create line 270 $bot 270 480 600 480  \\\n\
    -arrow first\n\
  $c create text 320 470 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"Control/status\"\n\
  set top [expr 258-($num_int_units+$num_add_units+$num_mul_units+$num_div_units+1)*6.5]\n\
  $c create line 600 $top 600 30 60 30 60 50 \\\n\
    -arrow both\n\
  $c create text 315 20 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"Control/status\"\n\
  text $c.title \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -setgrid true \\\n\
    -height 1 \\\n\
    -width 53 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
  DisableTextWidget $c.title\n\
  $c create window 440 $top \\\n\
    -anchor nw \\\n\
    -window $c.title\n\
  $c.title insert end \"Busy Op       Fi   Fj   Fk     Qj      Qk    Rj  Rk\"\n\
  set top [expr $top+13]\n\
  for {set i 0} {$i < $num_int_units} {incr i} {\n\
    text $c.int$i \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -setgrid true \\\n\
      -height 1 \\\n\
      -width 53 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
    DisableTextWidget $c.int$i\n\
    $c create window 440 [expr $top+$i*13] \\\n\
      -anchor nw \\\n\
      -window $c.int$i\n\
    bind $c.int$i <Button-1> \"mkEquation Int $i\"\n\
    $c create text 380 [expr $top+7+$i*13 ] \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -anchor w \\\n\
      -text Integer[expr $i+1]\n\
  }  \n\
  set top [expr $top+$num_int_units*13]\n\
  for {set i 0} {$i < $num_add_units} {incr i} {\n\
    text $c.add$i \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -setgrid true \\\n\
      -height 1 \\\n\
      -width 53 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
    DisableTextWidget $c.add$i\n\
    $c create window 440 [expr $top+$i*13] \\\n\
      -anchor nw \\\n\
      -window $c.add$i\n\
    bind $c.add$i <Button-1> \"mkEquation Add [expr $i+8]\"\n\
    $c create text 380 [expr $top+7+$i*13 ] \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -anchor w \\\n\
      -text Add[expr $i+1]\n\
  }  \n\
  set top [expr $top+$num_add_units*13]\n\
  for {set i 0} {$i < $num_mul_units} {incr i} {\n\
    text $c.mul$i \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -setgrid true \\\n\
      -height 1 \\\n\
      -width 53 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
    DisableTextWidget $c.mul$i\n\
    $c create window 440 [expr $top+$i*13] \\\n\
      -anchor nw \\\n\
      -window $c.mul$i\n\
    bind $c.mul$i <Button-1> \"mkEquation Mul [expr $i+24]\"\n\
    $c create text 380 [expr $top+7+$i*13 ] \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -anchor w \\\n\
      -text Mult[expr $i+1]\n\
  }  \n\
  set top [expr $top+$num_mul_units*13]\n\
  if {$fp_div_exist} {\n\
    for {set i 0} {$i < $num_div_units} {incr i} {\n\
      text $c.div$i \\\n\
        -relief raised \\\n\
        -borderwidth 1 \\\n\
        -setgrid true \\\n\
        -height 1 \\\n\
        -width 53 \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
      DisableTextWidget $c.div$i\n\
      $c create window 440 [expr $top+$i*13] \\\n\
        -anchor nw \\\n\
        -window $c.div$i\n\
      bind $c.div$i <Button-1> \"mkEquation Div [expr $i+16]\"\n\
      $c create text 380 [expr $top+7+$i*13 ] \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
        -anchor w \\\n\
        -text Divide[expr $i+1]\n\
    }  \n\
    set top [expr $top+$num_div_units*13]\n\
  }\n\
  $c create line 600 $top 600 480 \\\n\
    -arrow first\n\
  frame $c.frame\n\
  text $c.frame.table \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -height 9 \\\n\
    -width 70 \\\n\
    -background lavender \\\n\
    -yscrollcommand \"$c.frame.s set\" \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \n\
  DisableTextWidget $c.frame.table\n\
  label $c.frame.l1 \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"                           Instruction status                        \"\n\
  label $c.frame.l2 \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"  Instruction           Issue  Read operands  Excute  Write result   \"\n\
  scrollbar $c.frame.s \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -command \"$c.frame.table yview\"\n\
  pack $c.frame.s \\\n\
    -side right \\\n\
    -fill y\n\
  pack $c.frame.l1 $c.frame.l2 \\\n\
    -side top \\\n\
    -expand yes \\\n\
    -fill x\n\
  pack $c.frame.table \\\n\
    -side bottom\n\
  $c create window 25 500 \\\n\
    -anchor nw \\\n\
    -window $c.frame\n\
  label $c.counter \\\n\
    -font -adobe-times-bold-i-normal-*-60-400-75-*-*-170-iso8859-1 \n\
  $c create window 670 560 \\\n\
    -window $c.counter\n\
  $c create text 670 600 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-* \\\n\
    -text \"Issue Cycle of Current Instruction\" \\\n\
    -tag cycleTitle\n\
}\n\
   ";






char mkStop_tcl[] = "\
proc mkStop {{w .stop}} {\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Stop\"\n\
  wm iconname $w \"stop\"\n\
  wm geometry $w +200+200\n\
  set oldFocus [focus]\n\
  global msg stops\n\
  option add Tk$w*Relief raised\n\
  option add Tk$w*Label.Anchor w\n\
  option add Tk$w*Listbox.Setgrid yes\n\
  option add Tk$w*Label.Relief flat\n\
  option add Tk$w*Entry.Width 10\n\
  frame $w.entry\n\
  frame $w.list\n\
  frame $w.button\n\
  pack $w.entry \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  pack $w.list \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  pack $w.button \\\n\
    -side bottom \\\n\
    -fill x \\\n\
    -expand yes\n\
  label $w.entry.l \\\n\
    -text \"Stop at\"\n\
  entry $w.entry.e\n\
  bind $w.entry.e <Return> \"stop at \\[$w.entry.e get \\]\n\
    $w.list.list delete 0 end\n\
    stop info\"\n\
  pack $w.entry.l \\\n\
    -side left\n\
  pack $w.entry.e \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  label $w.list.label \\\n\
    -text \"Current Stop Information\"\n\
  listbox $w.list.list \\\n\
    -yscrollcommand \"$w.list.scroll set\"\n\
  \n\
  scrollbar $w.list.scroll \\\n\
    -command \"$w.list.list yview\"   \n\
  pack $w.list.label \\\n\
    -side top\n\
 \n\
  pack $w.list.list $w.list.scroll \\\n\
    -side left \\\n\
    -fill both \\\n\
    -expand yes\n\
  button $w.button.delete \\\n\
    -text \"Delete Stop\" \\\n\
  button $w.button.ok \\\n\
    -text Ok \\\n\
    -command \"destroy $w;  focus $oldFocus\"\n\
  pack $w.button.delete $w.button.ok \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes   \n\
  stop info\n\
  focus $w.entry.e\n\
  grab $w\n\
}\n\
   ";






char mkTomasulo_tcl[] = "\
proc mkTomasulo {{w .tom}} {\n\
  catch {destroy $w}\n\
  toplevel $w \n\
  wm title $w \"Tomasulo Algorithm on DLX\"\n\
  wm iconname $w \"tomasulo\"\n\
  wm geometry $w +0+0\n\
  set c $w.c\n\
  option add Tk$w*Text.Background gainsboro\n\
  canvas $c \\\n\
    -width 1030 \\\n\
    -height 600 \\\n\
    -relief raised \\\n\
    -scrollregion {30 0 1060 600}\n\
  pack $c\n\
  global num_int_units num_add_units num_mul_units \n\
  global num_div_units num_load_bufs num_store_bufs\n\
  global fp_div_exist ld_st_exist cycleCount \n\
  if {$ld_st_exist} {\n\
    text $c.load \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -height 1 \\\n\
      -width 12 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-* \n\
    DisableTextWidget $c.load\n\
    $c create window 50 51 \\\n\
      -anchor sw \\\n\
      -window $c.load\n\
    $c.load insert end \"  address\"\n\
    for {set i [expr $num_load_bufs-1]} {$i >=0} {incr i -1} {\n\
      text $c.load$i \\\n\
        -relief raised \\\n\
        -borderwidth 1 \\\n\
        -setgrid true \\\n\
        -height 1 \\\n\
        -width 12 \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
      DisableTextWidget $c.load$i \n\
      $c create window 50 [expr 50+($num_load_bufs-1-$i)*13] \\\n\
        -anchor nw \\\n\
        -window $c.load$i\n\
      bind $c.load$i <Button-1> \"mkEquation Load [expr $i+32]\"\n\
      $c create text 42 [expr 57+($num_load_bufs-1-$i)*13] \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
        -text [expr $i+1]\n\
    }\n\
    $c create line 90 [expr 50+$num_load_bufs*13] 90 170 35 170 35 410 160 410 \\\n\
      -width 4 \n\
    $c create line 80 15 80 36 \\\n\
      -arrow last\n\
    $c create text 92 25 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -text \"Load Buffers\"\n\
  }\n\
  text $c.instr \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -setgrid true \\\n\
    -background ivory \\\n\
    -height 1 \\\n\
    -width 40 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
  DisableTextWidget $c.instr\n\
  $c create window 160 50 \\\n\
    -anchor sw \\\n\
    -window $c.instr\n\
 \n\
  text $c.reg \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -height 1 \\\n\
    -width 56 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-* \n\
  DisableTextWidget $c.reg\n\
  $c create window 430 51 \\\n\
    -anchor sw \\\n\
    -window $c.reg\n\
  $c.reg insert end \"  Qi     Qi     Qi     Qi     Qi     Qi     Qi     Qi\"\n\
  for {set i 0} {$i < 64} {incr i} {\n\
    text $c.reg$i \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -setgrid true \\\n\
      -height 1 \\\n\
      -width 7 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
    DisableTextWidget $c.reg$i\n\
    $c create window [expr 430+$i%8*42] [expr 50+$i/8*13] \\\n\
      -anchor nw \\\n\
      -window $c.reg$i\n\
    if {$i%8 == 0} {\n\
      if {$i < 32} {\n\
        set reg R$i\n\
      } else {\n\
        set reg F[expr $i-32]\n\
      }\n\
      $c create text 405 [expr 57+$i/8*13] \\\n\
        -anchor w \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
        -text $reg\n\
    }\n\
  }\n\
  if {$ld_st_exist} {\n\
    text $c.store \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -height 1 \\\n\
      -width 28 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-* \n\
    DisableTextWidget $c.store\n\
    $c create window 845 51 \\\n\
      -anchor sw \\\n\
      -window $c.store\n\
    $c.store insert end \"   Vi      Qi    address\"\n\
    for {set i [expr $num_store_bufs-1]} {$i >=0} {incr i -1} {\n\
      text $c.store$i \\\n\
        -relief raised \\\n\
        -borderwidth 1 \\\n\
        -setgrid true \\\n\
        -height 1 \\\n\
        -width 28 \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
      DisableTextWidget $c.store$i\n\
      $c create window 845 [expr 50+($num_store_bufs-1-$i)*13] \\\n\
        -anchor nw \\\n\
        -window $c.store$i\n\
      bind $c.store$i <Button-1> \"mkEquation Store [expr $i+40]\"\n\
      $c create text 837 [expr 57+($num_store_bufs-1-$i)*13] \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
        -text [expr $i+1]\n\
    }\n\
    $c create line 935 15 935 37 \\\n\
      -arrow last\n\
    $c create line 1000 [expr 50+$num_store_bufs*13] 1000 [expr 70+$num_store_bufs*13] \\\n\
      -arrow last\n\
    $c create line 885 180 885 [expr 50+$num_store_bufs*13] \\\n\
      -arrow last\n\
    $c create text 980 [expr 75+$num_store_bufs*13] \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -text \"To Memory\"\n\
    $c create text 885 25 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -text \"Store buffers\"\n\
  }\n\
  text $c.int \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -height 1 \\\n\
    -width 37 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-* \n\
  DisableTextWidget $c.int\n\
  $c create window 50 251 \\\n\
    -anchor sw \\\n\
    -window $c.int\n\
  $c.int insert end \" op      Vj      Qj      Vk      Qk\"\n\
  for {set i [expr $num_int_units-1]} {$i >= 0} {incr i -1} {\n\
    text $c.int$i \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -setgrid true \\\n\
      -height 1 \\\n\
      -width 37 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
    DisableTextWidget $c.int$i\n\
    $c create window 50 [expr 250+($num_int_units-1-$i)*13] \\\n\
      -anchor nw \\\n\
      -window $c.int$i\n\
    bind $c.int$i <Button-1> \"mkEquation Int $i\"\n\
    $c create text 42 [expr 257+($num_int_units-1-$i)*13] \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -text [expr $i+1]\n\
  }\n\
  set yshift [expr $num_int_units*13]\n\
  $c create line 135 [expr 250+$yshift] 135 371 \\\n\
    -arrow last\n\
  $c create line 185 [expr 250+$yshift] 185 371 \\\n\
    -arrow last\n\
  $c create line 160 389 160 408 \\\n\
    -arrow last\n\
  label $c.intl \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -text \"Integer units\"\n\
  $c create window 160 380 \\\n\
    -window $c.intl   \n\
  text $c.add \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -height 1 \\\n\
    -width 37 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-* \n\
  DisableTextWidget $c.add\n\
  $c create window 300 251 \\\n\
    -anchor sw \\\n\
    -window $c.add\n\
  $c.add insert end \" op      Vj      Qj      Vk      Qk\"\n\
  for {set i [expr $num_add_units-1]} {$i >= 0} {incr i -1} {\n\
    text $c.add$i \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -setgrid true \\\n\
      -height 1 \\\n\
      -width 37 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
    DisableTextWidget $c.add$i\n\
    $c create window 300 [expr 250+($num_add_units-1-$i)*13] \\\n\
      -anchor nw \\\n\
      -window $c.add$i\n\
    bind $c.add$i <Button-1> \"mkEquation Add [expr $i+8]\"\n\
    $c create text 292 [expr 257+($num_add_units-1-$i)*13] \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -text [expr $i+1]\n\
  }\n\
  set yshift [expr $num_add_units*13]\n\
  $c create line 385 [expr 250+$yshift] 385 371 \\\n\
    -arrow last\n\
  $c create line 435 [expr 250+$yshift] 435 371 \\\n\
    -arrow last\n\
  $c create line 410 389 410 408 \\\n\
    -arrow last\n\
  label $c.addl \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -text \"FP adders\"\n\
  $c create window 410 380 \\\n\
    -window $c.addl   \n\
  text $c.mul \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -height 1 \\\n\
    -width 37 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-* \n\
  DisableTextWidget $c.mul\n\
  $c create window 550 251 \\\n\
    -anchor sw \\\n\
    -window $c.mul\n\
  $c.mul insert end \" op      Vj      Qj      Vk      Qk\"\n\
  for {set i [expr $num_mul_units-1]} {$i >= 0} {incr i -1} {\n\
    text $c.mul$i \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -setgrid true \\\n\
      -height 1 \\\n\
      -width 37 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
    DisableTextWidget $c.mul$i\n\
    $c create window 550 [expr 250+($num_mul_units-1-$i)*13] \\\n\
      -anchor nw \\\n\
      -window $c.mul$i\n\
    bind $c.mul$i <Button-1> \"mkEquation Mul [expr $i+24]\"\n\
    $c create text 542 [expr 257+($num_mul_units-1-$i)*13] \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
      -text [expr $i+1]\n\
  }\n\
  set yshift [expr $num_mul_units*13]\n\
  $c create line 635 [expr 250+$yshift] 635 371 \\\n\
    -arrow last\n\
  $c create line 685 [expr 250+$yshift] 685 371 \\\n\
    -arrow last\n\
  $c create line 660 389 660 408 \\\n\
    -arrow last\n\
  label $c.mull \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -text \"FP multipliers\"\n\
  $c create window 660 380 \\\n\
    -window $c.mull   \n\
  if {$fp_div_exist} {\n\
    text $c.div \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -height 1 \\\n\
      -width 37 \\\n\
      -font -Adobe-Courier-medium-R-Normal-*-100-* \n\
    DisableTextWidget $c.div\n\
    $c create window 800 251 \\\n\
      -anchor sw \\\n\
      -window $c.div\n\
    $c.div insert end \" op      Vj      Qj      Vk      Qk\"\n\
    for {set i [expr $num_div_units-1]} {$i >= 0} {incr i -1} {\n\
      text $c.div$i \\\n\
        -relief raised \\\n\
        -borderwidth 1 \\\n\
        -setgrid true \\\n\
        -height 1 \\\n\
        -width 37 \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-100-*\n\
      DisableTextWidget $c.div$i\n\
      $c create window 800 [expr 250+($num_div_units-1-$i)*13] \\\n\
        -anchor nw \\\n\
        -window $c.div$i\n\
      bind $c.div$i <Button-1> \"mkEquation Div [expr $i+16]\"\n\
      $c create text 792 [expr 257+($num_div_units-1-$i)*13] \\\n\
        -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
        -text [expr $i+1]\n\
    }\n\
    set yshift [expr $num_div_units*13]\n\
    $c create line 885 [expr 250+$yshift] 885 371 \\\n\
      -arrow last\n\
    $c create line 935 [expr 250+$yshift] 935 371 \\\n\
      -arrow last\n\
    $c create line 910 389 910 408 \\\n\
      -arrow last\n\
    label $c.divl \\\n\
      -relief raised \\\n\
      -borderwidth 1 \\\n\
      -text \"FP dividers\"\n\
    $c create window 910 380 \\\n\
      -window $c.divl\n\
    $c create line 565 200 815 200 815 236 \\\n\
      -arrow last\n\
    $c create line 635 180 885 180 885 236 \\\n\
      -arrow last\n\
    $c create line 735 190 985 190 985 236 \\\n\
      -arrow last\n\
    $c create oval 563 198 569 204 \\\n\
      -fill black\n\
    $c create oval 633 178 639 184 \\\n\
      -fill black\n\
    $c create oval 733 188 739 194 \\\n\
      -fill black\n\
    $c create oval 883 212 889 218 \\\n\
      -fill black\n\
    $c create oval 983 212 989 218 \\\n\
      -fill black\n\
 }\n\
  frame $c.frame\n\
  text $c.frame.table \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -height 9 \\\n\
    -width 60 \\\n\
    -background lavender \\\n\
    -yscrollcommand \"$c.frame.s set\" \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \n\
  DisableTextWidget $c.frame.table\n\
  label $c.frame.l1 \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"                      Instruction status                    \"\n\
  label $c.frame.l2 \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -font -Adobe-Courier-bold-R-Normal-*-120-* \\\n\
    -text \"  Instruction             Issue   Excute  Write result      \"\n\
  scrollbar $c.frame.s \\\n\
    -relief raised \\\n\
    -borderwidth 1 \\\n\
    -command \"$c.frame.table yview\"\n\
  pack $c.frame.s \\\n\
    -side right \\\n\
    -fill y\n\
  pack $c.frame.l1 $c.frame.l2 \\\n\
    -side top \\\n\
    -expand yes \\\n\
    -fill x\n\
  pack $c.frame.table \\\n\
    -side bottom\n\
  $c create window 30 430 \\\n\
    -anchor nw \\\n\
    -window $c.frame\n\
  label $c.counter \\\n\
    -font -adobe-times-bold-i-normal-*-60-400-75-*-*-170-iso8859-1 \n\
  $c create window 750 500 \\\n\
    -window $c.counter\n\
  $c create text 750 550 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-100-* \\\n\
    -text \"Issue Cycle of Current Instruction\" \\\n\
    -tag cycleTitle\n\
  $c create line 320 15 320 36 \\\n\
    -arrow last\n\
  $c create line 280 50 280 200 65 200 65 236 \\\n\
    -arrow last\n\
  $c create line 280 200 315 200 315 236 \\\n\
    -arrow last\n\
  $c create line 315 200 565 200 565 236 \\\n\
    -arrow last\n\
  $c create line 520 154 520 180 135 180 135 236 \\\n\
    -arrow last\n\
  $c create line 385 180 385 236 \\\n\
    -arrow last\n\
  $c create line 520 180 635 180 635 236 \\\n\
    -arrow last\n\
  $c create line 550 154 550 190 485 190 485 236 \\\n\
    -arrow last \n\
  $c create line 485 190 235 190 235 236 \\\n\
    -arrow last\n\
  $c create line 550 190 735 190 735 236 \\\n\
    -arrow last\n\
  $c create line 160 410 1032 410 1032 15 620 15 620 37 \\\n\
    -width 4 \\\n\
    -arrow last \n\
  $c create line 1032 215 135 215 \\\n\
    -width 4\n\
  $c create oval 278 198 284 204 \\\n\
    -fill black\n\
  $c create oval 313 198 319 204 \\\n\
    -fill black\n\
  $c create oval 383 178 389 184 \\\n\
    -fill black\n\
  $c create oval 518 178 524 184 \\\n\
    -fill black\n\
  $c create oval 548 188 554 194 \\\n\
    -fill black\n\
  $c create oval 483 188 489 194 \\\n\
    -fill black\n\
  $c create oval 133 212 139 218 \\\n\
    -fill black\n\
  $c create oval 233 212 239 218 \\\n\
    -fill black\n\
  $c create oval 383 212 389 218 \\\n\
    -fill black\n\
  $c create oval 483 212 489 218 \\\n\
    -fill black\n\
  $c create oval 633 212 639 218 \\\n\
    -fill black\n\
  $c create oval 733 212 739 218 \\\n\
    -fill black\n\
  $c create text 320 10 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
    -text \"From instruction unit\"\n\
  $c create text 220 25 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
    -text \"Current instruction\"\n\
  $c create text 540 25 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
    -text \"Register file\"\n\
  $c create text 230 140 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
    -text \"Operation bus\"\n\
  $c create text 478 165 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
    -text \"Operand bus\"\n\
  $c create text 305 228 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
    -text \"Reservation stations\"\n\
  $c create text 285 400 \\\n\
    -font -Adobe-Courier-medium-R-Normal-*-120-* \\\n\
    -text \"Common data bus (CDB)\"\n\
}\n\
   ";






char ShowCdbLineBAK_tcl[] = "\
proc ShowCdbLineBAK {w} {\n\
  set c .tom.c\n\
  set start [string index $w 7]\n\
  if {$start == \"l\"} {\n\
    for {set y 60} {$y < 170} {incr y 10} {\n\
      $c create line 90 $y 90 [expr $y+10] -tag thickLine -width 10 -fill red\n\
      \n\
    }\n\
    for {set x 95} {$x > 25} {incr x -10} {\n\
      $c create line $x 170 [expr $x-10] 170 -tag thickLine -width 10 -fill red\n\
      \n\
    }\n\
    for {set y 170} {$y < 410} {incr y 10} {\n\
      $c create line 30 $y 30 [expr $y+10] -tag thickLine -width 10 -fill red\n\
      \n\
    }\n\
    for {set x 25} {$x < 166} {incr x +10} {\n\
      $c create line $x 410 [expr $x+10] 410 -tag thickLine -width 10 -fill red\n\
      \n\
    }\n\
  }\n\
  if {$start == \"l\" || $start == \"i\"} {\n\
    for {set x 160} {$x < 410} {incr x +10} {\n\
      $c create line $x 410 [expr $x+10] 410 -tag thickLine -width 10 -fill red\n\
      \n\
    }\n\
  }\n\
  if {$start == \"l\" || $start == \"i\" || $start == \"a\"} {\n\
    for {set x 410} {$x < 660} {incr x +10} {\n\
      $c create line $x 410 [expr $x+10] 410 -tag thickLine -width 10 -fill red\n\
      \n\
    }\n\
  }\n\
  if {$start == \"l\" || $start == \"i\" || $start == \"a\" || $start == \"m\"} {\n\
    for {set x 660} {$x < 910} {incr x +10} {\n\
      $c create line $x 410 [expr $x+10] 410 -tag thickLine -width 10 -fill red\n\
      \n\
    }\n\
  }\n\
    \n\
  for {set x 910} {$x < 1040} {incr x +10} {\n\
    $c create line $x 410 [expr $x+10] 410 -tag thickLine -width 10 -fill red\n\
    \n\
  }\n\
  for {set y 415} {$y > 215} {incr y -10} {\n\
    $c create line 1040 $y 1040 [expr $y-10] -tag thickLine -width 10 -fill red\n\
    \n\
  }\n\
  for {set i 0} {$i < 21} {incr i} {\n\
    $c create line 1040 [expr 220-$i*10] 1040 [expr 210-$i*10] -tag thickLine \\\n\
      -width 10 -fill red\n\
    $c create line [expr 1040-$i*10] 215 [expr 1030-$i*10] 215 -tag thickLine \\\n\
      -width 10 -fill red \n\
    \n\
  }\n\
  for {set i 0} {$i < 42} {incr i} {\n\
    $c create line [expr 1040-$i*10] 15 [expr 1030-$i*10] 15 -tag thickLine \\\n\
      -width 10 -fill red\n\
    $c create line [expr 840-$i*10] 215 [expr 830-$i*10] 215 -tag thickLine \\\n\
      -width 10 -fill red\n\
    \n\
  }\n\
  for {set i 0} {$i < 4} {incr i} {\n\
    $c create line 620 [expr 10+$i*10] 620 [expr 20+$i*10] -tag thickLine \\\n\
      -width 10 -fill red\n\
    $c create line [expr 460-$i*10] 215 [expr 460-$i*10] 215 -tag thickLine \\\n\
      -width 10 -fill red \n\
    \n\
  }\n\
  for {set x 420} {$x > 140} {incr x -10} {\n\
    $c create line $x 215 [expr $x-10] 215 -tag thickLine -width 10 -fill red\n\
    \n\
  }\n\
  $c create line 140 215 135 215 -tag thickLine -width 10 -fill red\n\
  HighLightWrite $w\n\
}\n\
   ";






char ShowCdbLine_tcl[] = "\
proc ShowCdbLine {w} {\n\
  set c .tom.c\n\
  set start [string index $w 7]\n\
  if {$start == \"l\"} {\n\
    $c create line 90 60 90 170 35 170 35 410 160 410 \\\n\
      -tag thickLine -width 10 -joinstyle miter -fill red\n\
  }\n\
  if {$start == \"l\" || $start == \"i\"} {\n\
    $c create line 160 410 410 410 -tag thickLine -width 10 -joinstyle miter -fill red\n\
  }\n\
  if {$start == \"l\" || $start == \"i\" || $start == \"a\"} {\n\
    $c create line 410 410 660 410 -tag thickLine -width 10 -joinstyle miter -fill red\n\
  }\n\
  if {$start == \"l\" || $start == \"i\" || $start == \"a\" || $start == \"m\"} {\n\
    $c create line 660 410 910 410 -tag thickLine -width 10 -joinstyle miter -fill red\n\
  }\n\
   \n\
  $c create line 910 410 1032 410 1032 15 620 15 620 37 \\\n\
    -tag thickLine -width 10 -joinstyle miter -fill red\n\
    \n\
  $c create line 1032 215 132 215 -tag thickLine -width 10 -joinstyle miter -fill red\n\
  HighLightWrite $w\n\
}\n\
   ";






char HighLightWrite_tcl[] = "\
proc HighLightWrite {w} {\n\
  $w tag add gray 1.0 end\n\
  $w tag configure gray -background gray\n\
}\n\
   ";






char ShowInsLine_tcl[] = "\
proc ShowInsLine {w} {\n\
  set c .tom.c\n\
  set end [string index $w 7]\n\
  $c create line 280 50 280 204 -tag thickLine -width 8 -fill ForestGreen\n\
  if {$end == \"i\"} {\n\
    $c create line 280 200 65 200 65 236 -tag thickLine -width 8 -fill ForestGreen \\\n\
       -joinstyle miter\n\
  }\n\
  if {$end == \"a\"} {\n\
    $c create line 280 200 315 200 315 236 -tag thickLine -width 8 -fill ForestGreen \\\n\
       -joinstyle miter\n\
  }\n\
  if {$end == \"m\"} {\n\
    $c create line 280 200 565 200 565 236 -tag thickLine -width 8 -fill ForestGreen \\\n\
       -joinstyle miter\n\
  }\n\
  if {$end == \"d\"} {\n\
    $c create line 280 200 815 200 815 236 -tag thickLine -width 6 -fill ForestGreen \\\n\
       -joinstyle miter\n\
  }\n\
}\n\
   ";






char ShowResLine_tcl[] = "\
proc ShowResLine {w} {\n\
  global num_int_units num_add_units num_mul_units num_div_units\n\
  set c [string range $w 1 6]\n\
  set start [string range $w 0 2]\n\
  set end [string index $w 8]\n\
  if {$start == \"1.t\" && $end == \"i\"} {\n\
    $c create line 520 154 520 180 135 180 135 236 -tag thickLine -width 8 \\\n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"1.t\" && $end == \"a\"} {\n\
    $c create line 520 154 520 180 385 180 385 236 -tag thickLine -width 8 \\\n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"1.t\" && $end == \"m\"} {\n\
    $c create line 520 154 520 180 635 180 635 236 -tag thickLine -width 8 \\\n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"1.t\" && $end == \"d\"} {\n\
    $c create line 520 154 520 180 885 180 885 236 -tag thickLine -width 8 \\ \n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"2.t\" && $end == \"i\"} {\n\
    $c create line 550 154 550 190 235 190 235 236 -tag thickLine -width 8 \\\n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"2.t\" && $end == \"a\"} {\n\
    $c create line 550 154 550 190 485 190 485 236 -tag thickLine -width 8 \\\n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"2.t\" && $end == \"m\"} {\n\
    $c create line 550 154 550 190 735 190 735 236 -tag thickLine -width 8 \\\n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"2.t\" && $end == \"d\"} {\n\
    $c create line 550 154 550 190 985 190 985 236 -tag thickLine -width 8 \\\n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"2.t\" && $end == \"s\"} {\n\
    $c create line 550 154 550 190 885 190 885 50 -tag thickLine -width 8 \\\n\
       -fill blue  -joinstyle miter\n\
  } elseif {$start == \"i.t\" && $end == \"i\"} {\n\
    $c create line 280 50 280 200 65 200 65 236 -tag thickLine -width 8 \\\n\
       -fill ForestGreen -joinstyle miter\n\
  } elseif {$start == \"i.t\" && $end == \"a\"} {\n\
    $c create line 280 50 280 200 315 200 315 236 -tag thickLine -width 8 \\\n\
       -fill ForestGreen -joinstyle miter\n\
  } elseif {$start == \"i.t\" && $end == \"m\"} {\n\
    $c create line 280 50 280 200 565 200 565 236 -tag thickLine -width 8 \\\n\
       -fill ForestGreen -joinstyle miter\n\
  } elseif {$start == \"i.t\" && $end == \"d\"} {\n\
    $c create line 280 50 280 200 815 200 815 236 -tag thickLine -width 8 \\\n\
       -fill ForestGreen -joinstyle miter\n\
  } elseif {$start == \"i.t\"} {\n\
    $c create line 280 50 280 204 -tag thickLine -width 8 \\\n\
       -fill ForestGreen -joinstyle miter\n\
  } elseif {$start == \"1.s\" && $end == \"i\"} {\n\
    $c create line 139 98 250 98 -tag thickLine -width 7 -fill blue \n\
    set y [expr 107-$num_int_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill blue\n\
  } elseif {$start == \"1.s\" && $end == \"a\"} {\n\
    $c create line 139 202 250 202 -tag thickLine -width 7 -fill blue\n\
    set y [expr 211-$num_add_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill blue\n\
  } elseif {$start == \"1.s\" && $end == \"m\"} {\n\
    $c create line 139 306 250 306 -tag thickLine -width 7 -fill blue \n\
    set y [expr 315-$num_mul_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill blue\n\
  } elseif {$start == \"1.s\" && $end == \"d\"} {\n\
    $c create line 139 410 250 410 -tag thickLine -width 7 -fill blue\n\
    set y [expr 419-$num_div_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill blue\n\
  } elseif {$start == \"2.s\" && $end == \"i\"} {\n\
    $c create line 139 106 250 106 -tag thickLine -width 7 -fill blue \n\
    set y [expr 107-$num_int_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill blue\n\
  } elseif {$start == \"2.s\" && $end == \"a\"} {\n\
    $c create line 139 210 250 210 -tag thickLine -width 7 -fill blue\n\
    set y [expr 211-$num_add_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill blue\n\
  } elseif {$start == \"2.s\" && $end == \"m\"} {\n\
    $c create line 139 314 250 314 -tag thickLine -width 7 -fill blue \n\
    set y [expr 315-$num_mul_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill blue\n\
  } elseif {$start == \"2.s\" && $end == \"d\"} {\n\
    $c create line 139 418 250 418 -tag thickLine -width 7 -fill blue\n\
    set y [expr 419-$num_div_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill blue\n\
  } elseif {$start == \"r.s\" && $end == \"i\"} {\n\
    $c create line 290 102 325 102 325 56 139 56 -tag thickLine -width 7 \\\n\
      -fill red -joinstyle miter\n\
    set y [expr 107-$num_int_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill red\n\
  } elseif {$start == \"r.s\" && $end == \"a\"} {\n\
    $c create line 290 206 325 206 325 160 139 160 -tag thickLine -width 7 \\\n\
      -fill red -joinstyle miter\n\
    set y [expr 211-$num_add_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill red\n\
  } elseif {$start == \"r.s\" && $end == \"m\"} {\n\
    $c create line 290 310 325 310 325 264 139 264 -tag thickLine -width 7 \\\n\
      -fill red -joinstyle miter\n\
    set y [expr 315-$num_mul_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill red\n\
  } elseif {$start == \"r.s\" && $end == \"d\"} {\n\
    $c create line 290 414 325 414 325 368 139 368 -tag thickLine -width 7 \\\n\
      -fill red -joinstyle miter\n\
    set y [expr 419-$num_div_units*5+[string index $w 11]*10]\n\
    $c create line 250 $y 290 $y -tag thickLine -width 8 -fill red\n\
  }\n\
   \n\
}\n\
   ";






char mkEquation_tcl[] = "\
proc mkEquation {res unit {w .equation}} {\n\
  global config\n\
  set name $res[expr $unit%8+1]\n\
  switch $config {\n\
    1 {\n\
        set title \"reservation station\"\n\
        set widget .tom.c.[string tolower $res][expr $unit%8]\n\
        set cycle [lindex [.tom.c.counter configure -text] 4]\n\
      }\n\
    2 {\n\
        set title \"functional unit\"\n\
        set widget .sco.c.[string tolower $res][expr $unit%8]\n\
        set cycle [lindex [.sco.c.counter configure -text] 4]\n\
      }\n\
  }\n\
  catch {destroy $w}\n\
  toplevel $w \n\
  wm title $w \"Action or bookeeping in $title $name\"\n\
  wm iconname $w \"equation\"\n\
  wm geometry $w +0+0\n\
  text $w.t \\\n\
    -relief raised \\\n\
    -setgrid true \\\n\
    -width 75 \\\n\
    -height 11\n\
  DisableTextWidget $w.t\n\
  button $w.ok \\\n\
    -text OK \\\n\
    -command \"destroy $w\"\n\
  pack $w.t \\\n\
    -side top\n\
  pack $w.ok \\\n\
    -side top \\\n\
    -fill x\n\
  $w.t tag configure bold -font -Adobe-Courier-Bold-O-Normal-*-120-*\n\
  $w.t tag configure normal -font -Adobe-Courier-Bold-R-Normal-*-120-*\n\
  $w.t tag configure red -foreground red\n\
  $w.t tag configure blue -foreground blue\n\
  $w.t tag configure underline -underline on\n\
  if {[$widget get 1.0 end] == \"\"} {\n\
    InsertWithTags $w.t \"No action for $title \" normal\n\
    InsertWithTags $w.t \"$name \" red \n\
    InsertWithTags $w.t \"at cycle \" normal\n\
    InsertWithTags $w.t \"$cycle\" red\n\
    InsertWithTags $w.t \".\" normal\n\
  } elseif {$config == \"1\"} {\n\
    if {[lindex [$widget tag configure toggle -background] 4] == \"black\"} {\n\
      set regs [equation $unit $cycle issue]\n\
      set rs1 [lindex $regs 0]\n\
      if {$rs1 != \"\"} {\n\
        InsertWithTags $w.t \"if (Register\\[\" normal\n\
        InsertWithTags $w.t \"$rs1\" red\n\
        InsertWithTags $w.t \"\\].Qi != 0) \\n   \" normal\n\
        set Qj [lindex $regs 1]\n\
        if {$Qj} {\n\
          InsertWithTags $w.t \"\\{RS\\[\" normal underline\n\
          InsertWithTags $w.t \"$name\" red underline\n\
          InsertWithTags $w.t \"\\].Qj<-Register\\[\" normal underline\n\
          InsertWithTags $w.t \"$rs1\" red underline\n\
          InsertWithTags $w.t \"\\].Qi\\}\" normal underline\n\
          if {$res != \"Load\"} {\n\
            InsertWithTags $w.t \"         (RS\\[$name\\].Qj = \" blue bold\n\
            if {$res != \"Store\"} {\n\
              InsertWithTags $w.t \"[lindex [$widget get 1.0 end] 1])\" blue bold\n\
            } else {\n\
              InsertWithTags $w.t \"[lindex [$widget get 1.0 end] 0])\" blue bold\n\
            }\n\
          }\n\
          InsertWithTags $w.t \"\\n\" \n\
          InsertWithTags $w.t \"else \" normal\n\
          InsertWithTags $w.t \"\\{RS\\[\" normal\n\
          InsertWithTags $w.t \"$name\" red\n\
          InsertWithTags $w.t \"\\].Vj<-\" normal\n\
          InsertWithTags $w.t \"$rs1\" red\n\
          InsertWithTags $w.t \"; RS\\[\" normal\n\
          InsertWithTags $w.t \"$name\" red\n\
          InsertWithTags $w.t \"\\].Qj<-0\\}\\n\\n\" normal\n\
        } else {\n\
          InsertWithTags $w.t \"\\{RS\\[\" normal \n\
          InsertWithTags $w.t \"$name\" red \n\
          InsertWithTags $w.t \"\\].Qj<-Register\\[\" normal\n\
          InsertWithTags $w.t \"$rs1\" red underline\n\
          InsertWithTags $w.t \"\\].Qi\\}\\nelse\" normal \n\
          InsertWithTags $w.t \"\\{RS\\[\" normal underline\n\
          InsertWithTags $w.t \"$name\" red underline\n\
          InsertWithTags $w.t \"\\].Vj<-\" normal underline\n\
          InsertWithTags $w.t \"$rs1\" red underline\n\
          InsertWithTags $w.t \"; RS\\[\" normal underline\n\
          InsertWithTags $w.t \"$name\" red underline\n\
          InsertWithTags $w.t \"\\].Qj<-0\\}\\n\\n\" normal underline\n\
        }\n\
      }\n\
      set rs2 [lindex $regs 2]\n\
      if {$rs2 != \"\"} {\n\
        InsertWithTags $w.t \"if (Register\\[\" normal\n\
        InsertWithTags $w.t \"$rs2\" red\n\
        InsertWithTags $w.t \"\\].Qi != 0) \\n   \" normal\n\
        set Qk [lindex $regs 3]\n\
        if {$Qk} {\n\
          InsertWithTags $w.t \"\\{RS\\[\" normal underline\n\
          InsertWithTags $w.t \"$name\" red underline\n\
          InsertWithTags $w.t \"\\].Qk<-Register\\[\" normal underline\n\
          InsertWithTags $w.t \"$rs2\" red underline\n\
          InsertWithTags $w.t \"\\].Qi\\}\" normal underline\n\
          if {$res != \"Load\"} {\n\
            InsertWithTags $w.t \"         (RS\\[$name\\].Qk = \" blue bold\n\
            if {$res != \"Store\"} {\n\
              InsertWithTags $w.t \"[lindex [$widget get 1.0 end] 1])\" blue bold\n\
            } else {\n\
              InsertWithTags $w.t \"[lindex [$widget get 1.0 end] 0])\" blue bold\n\
            }\n\
          }\n\
          InsertWithTags $w.t \"\\n\" \n\
          InsertWithTags $w.t \"else \" normal\n\
          InsertWithTags $w.t \"\\{RS\\[\" normal\n\
          InsertWithTags $w.t \"$name\" red\n\
          InsertWithTags $w.t \"\\].Vj<-\" normal\n\
          InsertWithTags $w.t \"$rs2\" red\n\
          InsertWithTags $w.t \"; RS\\[\" normal\n\
          InsertWithTags $w.t \"$name\" red\n\
          InsertWithTags $w.t \"\\].Qk<-0\\}\\n\\n\" normal\n\
        } else {\n\
          InsertWithTags $w.t \"\\{RS\\[\" normal \n\
          InsertWithTags $w.t \"$name\" red \n\
          InsertWithTags $w.t \"\\].Qk<-Register\\[\" normal\n\
          InsertWithTags $w.t \"$rs2\" red underline\n\
          InsertWithTags $w.t \"\\].Qi\\}\\nelse\" normal \n\
          InsertWithTags $w.t \"\\{RS\\[\" normal underline\n\
          InsertWithTags $w.t \"$name\" red underline\n\
          InsertWithTags $w.t \"\\].Vj<-\" normal underline\n\
          InsertWithTags $w.t \"$rs2\" red underline\n\
          InsertWithTags $w.t \"; RS\\[\" normal underline\n\
          InsertWithTags $w.t \"$name\" red underline\n\
          InsertWithTags $w.t \"\\].Qk<-0\\}\\n\\n\" normal underline\n\
        }\n\
      } \n\
      set rd [lindex $regs 4]\n\
      if {$rd != \"\"} {\n\
        InsertWithTags $w.t \"RS\\[\" normal underline\n\
        InsertWithTags $w.t \"$name\" red underline\n\
        InsertWithTags $w.t \"\\].Busy<-yes;\\nRegister\\[\" normal underline\n\
        InsertWithTags $w.t \"$rd\" red underline\n\
        InsertWithTags $w.t \"\\].Qi=\" normal underline\n\
        InsertWithTags $w.t \"$name\" red underline\n\
        InsertWithTags $w.t \";\" normal underline\n\
      }    \n\
    } elseif {[lindex [$widget tag configure gray -background] 4] == \"gray\"} {\n\
      set info [equation $unit $cycle write]\n\
      set r [lindex $info 0]\n\
      if {$r != \"\"} {\n\
        InsertWithTags $w.t \"for any x (if (Register\\[x\\].Qi=\" normal \n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \")     \" normal\n\
        InsertWithTags $w.t \"\\{x<-result;\\n\" normal underline\n\
        InsertWithTags $w.t \"Register\\[x].Qi<-0\\}\" normal underline \n\
        InsertWithTags $w.t \");\\}\\n\" normal\n\
        InsertWithTags $w.t \"x = $r\\n\\n\" blue bold\n\
      }\n\
      set i 1\n\
      if {[lindex $info $i] == \"1\"} {\n\
        InsertWithTags $w.t \"for any x (if (RS\\[x\\].Qj=\" normal \n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \")     \" normal\n\
        InsertWithTags $w.t \"\\{RS\\[x\\].Vj<-result;\\n\" normal underline\n\
        InsertWithTags $w.t \"RS\\[x].Qj<-0\\}\" normal underline \n\
        InsertWithTags $w.t \");\\}\\n\" normal\n\
        InsertWithTags $w.t \"x = \\{\" blue bold\n\
        for {} {[lindex $info $i] == \"1\"} {incr i} {\n\
          InsertWithTags $w.t \"[lindex $info [incr i]] \" blue bold\n\
        }\n\
        InsertWithTags $w.t \"\\}\\n\\n\" blue bold\n\
      }\n\
      if {[lindex $info $i] == \"2\"} {\n\
        InsertWithTags $w.t \"for any x (if (RS\\[x\\].Qk=\" normal \n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \")     \" normal\n\
        InsertWithTags $w.t \"\\{RS\\[x\\].Vk<-result;\\n\" normal underline\n\
        InsertWithTags $w.t \"RS\\[x].Qk<-0\\}\" normal underline \n\
        InsertWithTags $w.t \");\\}\\n\" normal\n\
        InsertWithTags $w.t \"x = \\{\" blue bold\n\
        for {} {[lindex $info $i] == \"2\"} {incr i} {\n\
          InsertWithTags $w.t \"[lindex $info [incr i]] \" blue bold\n\
        }\n\
        InsertWithTags $w.t \"\\}\\n\\n\" blue bold\n\
      }\n\
 \n\
      InsertWithTags $w.t \"RS\\[\" normal underline\n\
      InsertWithTags $w.t \"$name\" red\n\
      InsertWithTags $w.t \"\\].Busy<-No\" normal        \n\
    } else {\n\
      InsertWithTags $w.t \"Instruction at reservation station \" normal\n\
      InsertWithTags $w.t \"$name\" red\n\
      InsertWithTags $w.t \" is now excuting.\" normal\n\
    }\n\
  }  elseif {$config == \"2\"} {\n\
    if {[lindex [$widget tag configure toggle -background] 4] == \"black\"} {\n\
      InsertWithTags $w.t \"Busy(\" normal \n\
      InsertWithTags $w.t \"$name\" red \n\
      InsertWithTags $w.t \")<-yes\" normal\n\
      set regs [equation $unit $cycle issue]\n\
      set rs1 [lindex $regs 0]\n\
      set Qj [lindex $regs 1]\n\
      set rs2 [lindex $regs 2]\n\
      set Qk [lindex $regs 3]\n\
      set rd [lindex $regs 4]\n\
      if {$rd != \"\"} {\n\
        InsertWithTags $w.t \";  Result(\" normal\n\
        InsertWithTags $w.t \"$rd\" red\n\
        InsertWithTags $w.t \")<-\" normal\n\
        InsertWithTags $w.t \"$name\" red\n\
      }\n\
      \n\
      set op [lindex [$widget get 1.0 end] 1]\n\
      InsertWithTags $w.t \";  Op(\" normal\n\
      InsertWithTags $w.t \"$name\" red\n\
      InsertWithTags $w.t \")<-\" normal\n\
      InsertWithTags $w.t \"$op\" red\n\
      InsertWithTags $w.t \";\\n\"\n\
      if {$rd != \"\"} {\n\
        InsertWithTags $w.t \"Fi(\" normal\n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \")<-\" normal\n\
        InsertWithTags $w.t \"$rd\" red\n\
        InsertWithTags $w.t \";  \" normal\n\
      }\n\
      if {$rs1 != \"\"} {\n\
        InsertWithTags $w.t \"Fj(\" normal\n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \")<-\" normal\n\
        InsertWithTags $w.t \"$rs1\" red\n\
        InsertWithTags $w.t \";  \" normal\n\
      }\n\
      if {$rs2 != \"\"} {\n\
        InsertWithTags $w.t \"Fk(\" normal\n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \")<-\" normal\n\
        InsertWithTags $w.t \"$rs2\" red\n\
        InsertWithTags $w.t \";  \" normal\n\
      }\n\
      if {$rs1 != \"\"} {\n\
        InsertWithTags $w.t \"\\nQj<-Result(\" normal\n\
        InsertWithTags $w.t \"$rs1\" red\n\
        InsertWithTags $w.t \"); \" normal\n\
        InsertWithTags $w.t \"[$widget get 1.30 1.35]     \" blue\n\
      }\n\
      if {$rs2 != \"\"} {\n\
        if {$rs1 == \"\"} {\n\
          InsertWithTags $w.t \"\\n\" normal\n\
        }\n\
        InsertWithTags $w.t \"Qk<-Result(\" normal\n\
        InsertWithTags $w.t \"$rs2\" red\n\
        InsertWithTags $w.t \");\" normal\n\
        InsertWithTags $w.t \"[$widget get 1.38 1.43]\" blue\n\
      }\n\
      if {$rs1 != \"\"} {\n\
        InsertWithTags $w.t \"\\nRj<-not Qj; \" normal\n\
        InsertWithTags $w.t \"[$widget get 1.45 1.47]         \" blue\n\
      }\n\
      if {$rs2 != \"\"} {\n\
        if {$rs1 == \"\"} {\n\
          InsertWithTags $w.t \"\\n\" normal\n\
        }\n\
        InsertWithTags $w.t \"Rk<-not Qk;\" normal\n\
        InsertWithTags $w.t \"[$widget get 1.49 1.52]\" blue\n\
      }\n\
    } elseif {[lindex [$widget tag configure gray -background] 4] == \"gray\"} {\n\
      set info [equation $unit $cycle write]\n\
      \n\
      set i 1\n\
      if {[lindex $info $i] == \"1\"} {\n\
        InsertWithTags $w.t \"for any f (if (Qj(f)=\" normal \n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \" then Rj(f)<-Yes)\\n\" normal\n\
        InsertWithTags $w.t \"f = \\{\" blue bold\n\
        for {} {[lindex $info $i] == \"1\"} {incr i} {\n\
          InsertWithTags $w.t \"[lindex $info [incr i]] \" blue bold\n\
        }\n\
        InsertWithTags $w.t \"\\}\\n\\n\" blue bold\n\
      }\n\
    \n\
      if {[lindex $info $i] == \"2\"} {\n\
        InsertWithTags $w.t \"for any f (if (Qk(f)=\" normal \n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \" then Rk(f)<-Yes)\\n\" normal\n\
        InsertWithTags $w.t \"f = \\{\" blue bold\n\
        for {} {[lindex $info $i] == \"2\"} {incr i} {\n\
          InsertWithTags $w.t \"[lindex $info [incr i]] \" blue bold\n\
        }\n\
        InsertWithTags $w.t \"\\}\\n\\n\" blue bold\n\
      }\n\
      set r [lindex $info 0]\n\
      if {$r != \"\"} {\n\
        InsertWithTags $w.t \"Result(\" normal\n\
        InsertWithTags $w.t \"$r\" red\n\
        InsertWithTags $w.t \")<-Clear;  \" normal\n\
      }\n\
      InsertWithTags $w.t \"Busy(\" normal\n\
      InsertWithTags $w.t \"$name\" red\n\
      InsertWithTags $w.t \")<-No\" normal      \n\
    } else {\n\
      if {[$widget get 1.4 1.7] == \"DIV\"} {\n\
        set info [equation $unit $cycle inused]\n\
      } else {\n\
        set info [equation $unit $cycle inuse]\n\
      }\n\
      if {$info == \"read\"} {\n\
        InsertWithTags $w.t \"Rj<-No;  Rk<-No\" normal\n\
      } elseif {$info == \"exec\"} {\n\
        InsertWithTags $w.t \"Instruction at functional unit \" normal\n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \" is now excuting.\" normal\n\
      } elseif {$info == \"RAW\"} {\n\
        InsertWithTags $w.t \"Instruction at functional unit \" normal\n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \" is waiting for operand(s).\" normal\n\
      } elseif {$info == \"WAR\"} {\n\
        InsertWithTags $w.t \"Instruction at functional unit \" normal\n\
        InsertWithTags $w.t \"$name\" red\n\
        InsertWithTags $w.t \" is waiting because of WAR hazard.\" normal\n\
      }\n\
    }\n\
  }\n\
}\n\
   ";






char InsertWithTags_tcl[] = "\
proc InsertWithTags {w text args} {\n\
    set start [$w index insert]\n\
    $w insert insert $text\n\
    foreach tag [$w tag names $start] {\n\
	$w tag remove $tag $start insert\n\
    }\n\
    foreach i $args {\n\
	$w tag add $i $start insert\n\
    }\n\
}\n\
   ";






char mkTrace_tcl[] = "\
proc mkTrace {{w .trace}} {\n\
  catch {destroy $w}\n\
  toplevel $w\n\
  wm title $w \"Trace\"\n\
  wm iconname $w \"trace\"\n\
  wm geometry $w +200+200\n\
  set oldFocus [focus]\n\
  option add Tk$w*Relief raised\n\
  option add Tk$w*Label.Relief flat\n\
  option add Tk$w*Label.Width 16\n\
  option add Tk$w*Label.Anchor w\n\
  option add Tk$w*Entry.Width 40\n\
  frame $w.entry\n\
  frame $w.button\n\
    \n\
  pack $w.entry \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
  pack $w.button \\\n\
    -side bottom \\\n\
    -fill x \\\n\
    -expand yes\n\
  frame $w.entry.up \n\
  frame $w.entry.down\n\
  pack $w.entry.up $w.entry.down \\\n\
    -side top \\\n\
    -fill x \\\n\
    -expand yes\n\
    \n\
  label $w.entry.up.dirlabel \\\n\
    -text \"directory name:\" \n\
  entry $w.entry.up.direntry \n\
  bind $w.entry.up.direntry <Return> \"focus $w.entry.down.fileentry\"\n\
  \n\
  pack $w.entry.up.dirlabel \\\n\
    -side left\n\
  pack $w.entry.up.direntry \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  label $w.entry.down.filelabel \\\n\
    -text \"file name:\"\n\
  entry $w.entry.down.fileentry\n\
  \n\
  pack $w.entry.down.filelabel \\\n\
    -side left\n\
  pack $w.entry.down.fileentry \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes\n\
  button $w.button.itrace \\\n\
    -text \"Instruction Trace\" \\\n\
    -command \"itrace on \\[$w.entry.up.direntry get\\]/\\[$w.entry.down.fileentry get\\]\n\
       destroy $w;\n\
       focus $oldFocus\"\n\
  button $w.button.mtrace \\\n\
    -text \"Dinero Trace\" \\\n\
    -command \"mtrace on \\[$w.entry.up.direntry get\\]/\\[$w.entry.down.fileentry get\\]\n\
       destroy $w;\n\
       focus $oldFocus\"\n\
  button $w.button.cancel \\\n\
    -text \"Cancel\" \\\n\
    -command \"destroy $w;\n\
       focus $oldFocus\"\n\
  pack $w.button.itrace $w.button.mtrace $w.button.cancel \\\n\
    -side left \\\n\
    -fill x \\\n\
    -expand yes   \n\
  $w.entry.up.direntry insert 0 [pwd]\n\
  focus $w.entry.down.fileentry\n\
  grab $w\n\
}\n\
   ";






