// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/new_tab_page/new_tab_page_handler.h"

#include "base/base64.h"
#include "base/bind.h"
#include "base/i18n/rtl.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/search/background/ntp_background_service.h"
#include "chrome/browser/search/background/ntp_background_service_factory.h"
#include "chrome/browser/search/chrome_colors/chrome_colors_factory.h"
#include "chrome/browser/search/chrome_colors/chrome_colors_service.h"
#include "chrome/browser/search/instant_service.h"
#include "chrome/browser/search/instant_service_factory.h"
#include "chrome/browser/search_provider_logos/logo_service_factory.h"
#include "chrome/browser/ui/search/omnibox_utils.h"
#include "chrome/common/search/generated_colors_info.h"
#include "chrome/common/search/instant_types.h"
#include "chrome/common/themes/autogenerated_theme_util.h"
#include "components/search_provider_logos/logo_service.h"
#include "components/search_provider_logos/switches.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/gfx/color_utils.h"

namespace {

new_tab_page::mojom::ThemePtr MakeTheme(const NtpTheme& ntp_theme) {
  auto theme = new_tab_page::mojom::Theme::New();
  if (ntp_theme.using_default_theme) {
    theme->type = new_tab_page::mojom::ThemeType::DEFAULT;
    // TODO(crbug.com/1040682): This info has no meaning for the default theme
    // and shouldn't be used. We set it here to prevent a crash where mojo is
    // complaing about an unset info. However, we cannot make the field optional
    // as that is crashing JS. Once the JS crash is solved remove this line.
    theme->info = new_tab_page::mojom::ThemeInfo::NewChromeThemeId(-1);
  } else if (ntp_theme.color_id == -1) {
    theme->type = new_tab_page::mojom::ThemeType::THIRD_PARTY;
    auto info = new_tab_page::mojom::ThirdPartyThemeInfo::New();
    info->id = ntp_theme.theme_id;
    info->name = ntp_theme.theme_name;
    theme->info =
        new_tab_page::mojom::ThemeInfo::NewThirdPartyThemeInfo(std::move(info));
  } else if (ntp_theme.color_id == 0) {
    theme->type = new_tab_page::mojom::ThemeType::AUTOGENERATED;
    auto theme_colors = new_tab_page::mojom::ThemeColors::New();
    theme_colors->frame = ntp_theme.color_dark;
    theme_colors->active_tab = ntp_theme.color_light;
    theme->info = new_tab_page::mojom::ThemeInfo::NewAutogeneratedThemeColors(
        std::move(theme_colors));
  } else {
    theme->type = new_tab_page::mojom::ThemeType::CHROME;
    theme->info =
        new_tab_page::mojom::ThemeInfo::NewChromeThemeId(ntp_theme.color_id);
  }
  theme->background_color = ntp_theme.background_color;
  theme->shortcut_background_color = ntp_theme.shortcut_color;
  theme->shortcut_text_color = ntp_theme.text_color;
  theme->is_dark = !color_utils::IsDark(ntp_theme.text_color);
  if (ntp_theme.logo_alternate) {
    theme->logo_color = ntp_theme.logo_color;
  }
  if (!ntp_theme.custom_background_url.is_empty()) {
    theme->background_image_url = ntp_theme.custom_background_url;
  }
  if (!ntp_theme.custom_background_attribution_line_1.empty()) {
    theme->background_image_attribution_1 =
        ntp_theme.custom_background_attribution_line_1;
  }
  if (!ntp_theme.custom_background_attribution_line_2.empty()) {
    theme->background_image_attribution_2 =
        ntp_theme.custom_background_attribution_line_2;
  }
  if (!ntp_theme.custom_background_attribution_action_url.is_empty()) {
    theme->background_image_attribution_url =
        ntp_theme.custom_background_attribution_action_url;
  }
  return theme;
}

}  // namespace

NewTabPageHandler::NewTabPageHandler(
    mojo::PendingReceiver<new_tab_page::mojom::PageHandler>
        pending_page_handler,
    mojo::PendingRemote<new_tab_page::mojom::Page> pending_page,
    Profile* profile,
    content::WebContents* web_contents)
    : chrome_colors_service_(
          chrome_colors::ChromeColorsFactory::GetForProfile(profile)),
      instant_service_(InstantServiceFactory::GetForProfile(profile)),
      ntp_background_service_(
          NtpBackgroundServiceFactory::GetForProfile(profile)),
      logo_service_(LogoServiceFactory::GetForProfile(profile)),
      page_{std::move(pending_page)},
      receiver_{this, std::move(pending_page_handler)},
      web_contents_(web_contents) {
  CHECK(instant_service_);
  CHECK(ntp_background_service_);
  CHECK(logo_service_);
  CHECK(web_contents_);
  instant_service_->AddObserver(this);
  ntp_background_service_->AddObserver(this);
  instant_service_->UpdateNtpTheme();
  OmniboxTabHelper::CreateForWebContents(web_contents);
  OmniboxTabHelper::FromWebContents(web_contents_)->AddObserver(this);
}

NewTabPageHandler::~NewTabPageHandler() {
  instant_service_->RemoveObserver(this);
  ntp_background_service_->RemoveObserver(this);
  if (auto* helper = OmniboxTabHelper::FromWebContents(web_contents_)) {
    helper->RemoveObserver(this);
  }
}

void NewTabPageHandler::AddMostVisitedTile(
    const GURL& url,
    const std::string& title,
    AddMostVisitedTileCallback callback) {
  bool success = instant_service_->AddCustomLink(url, title);
  std::move(callback).Run(success);
}

void NewTabPageHandler::DeleteMostVisitedTile(const GURL& url) {
  if (instant_service_->IsCustomLinksEnabled()) {
    instant_service_->DeleteCustomLink(url);
  } else {
    instant_service_->DeleteMostVisitedItem(url);
    last_blacklisted_ = url;
  }
}

void NewTabPageHandler::RestoreMostVisitedDefaults() {
  if (instant_service_->IsCustomLinksEnabled()) {
    instant_service_->ResetCustomLinks();
  } else {
    instant_service_->UndoAllMostVisitedDeletions();
  }
}

void NewTabPageHandler::ReorderMostVisitedTile(const GURL& url,
                                               uint8_t new_pos) {
  instant_service_->ReorderCustomLink(url, new_pos);
}

void NewTabPageHandler::SetMostVisitedSettings(bool custom_links_enabled,
                                               bool visible) {
  auto pair = instant_service_->GetCurrentShortcutSettings();
  // The first of the pair is true if most-visited tiles are being used.
  bool old_custom_links_enabled = !pair.first;
  bool old_visible = pair.second;
  // |ToggleMostVisitedOrCustomLinks()| always notifies observers. Since we only
  // want to notify once, we need to call |ToggleShortcutsVisibility()| with
  // false if we are also going to call |ToggleMostVisitedOrCustomLinks()|.
  bool toggleCustomLinksEnabled =
      old_custom_links_enabled != custom_links_enabled;
  if (old_visible != visible) {
    instant_service_->ToggleShortcutsVisibility(
        /* do_notify= */ !toggleCustomLinksEnabled);
  }
  if (toggleCustomLinksEnabled) {
    instant_service_->ToggleMostVisitedOrCustomLinks();
  }
}

void NewTabPageHandler::UndoMostVisitedTileAction() {
  if (instant_service_->IsCustomLinksEnabled()) {
    instant_service_->UndoCustomLinkAction();
  } else if (last_blacklisted_.is_valid()) {
    instant_service_->UndoMostVisitedDeletion(last_blacklisted_);
    last_blacklisted_ = GURL();
  }
}

void NewTabPageHandler::GetChromeThemes(GetChromeThemesCallback callback) {
  std::vector<new_tab_page::mojom::ChromeThemePtr> themes;
  for (const auto& color_info : chrome_colors::kGeneratedColorsInfo) {
    auto theme_colors = GetAutogeneratedThemeColors(color_info.color);
    auto theme = new_tab_page::mojom::ChromeTheme::New();
    theme->id = color_info.id;
    theme->label = l10n_util::GetStringUTF8(color_info.label_id);
    auto colors = new_tab_page::mojom::ThemeColors::New();
    colors->frame = theme_colors.frame_color;
    colors->active_tab = theme_colors.active_tab_color;
    theme->colors = std::move(colors);
    themes.push_back(std::move(theme));
  }
  std::move(callback).Run(std::move(themes));
}

void NewTabPageHandler::ApplyDefaultTheme() {
  chrome_colors_service_->ApplyDefaultTheme(web_contents_);
}

void NewTabPageHandler::ApplyAutogeneratedTheme(const SkColor& frame_color) {
  chrome_colors_service_->ApplyAutogeneratedTheme(frame_color, web_contents_);
}

void NewTabPageHandler::ApplyChromeTheme(int32_t id) {
  auto* begin = std::begin(chrome_colors::kGeneratedColorsInfo);
  auto* end = std::end(chrome_colors::kGeneratedColorsInfo);
  auto* result = std::find_if(begin, end,
                              [id](const chrome_colors::ColorInfo& color_info) {
                                return color_info.id == id;
                              });
  if (result == end) {
    return;
  }
  chrome_colors_service_->ApplyAutogeneratedTheme(result->color, web_contents_);
}

void NewTabPageHandler::ConfirmThemeChanges() {
  chrome_colors_service_->ConfirmThemeChanges();
}

void NewTabPageHandler::RevertThemeChanges() {
  chrome_colors_service_->RevertThemeChanges();
}

void NewTabPageHandler::UpdateMostVisitedInfo() {
  instant_service_->UpdateMostVisitedInfo();
}

void NewTabPageHandler::UpdateMostVisitedTile(
    const GURL& url,
    const GURL& new_url,
    const std::string& new_title,
    UpdateMostVisitedTileCallback callback) {
  bool success = instant_service_->UpdateCustomLink(
      url, new_url != url ? new_url : GURL(), new_title);
  std::move(callback).Run(success);
}

void NewTabPageHandler::GetBackgroundCollections(
    GetBackgroundCollectionsCallback callback) {
  if (!ntp_background_service_ || background_collections_callback_) {
    std::move(callback).Run(
        std::vector<new_tab_page::mojom::BackgroundCollectionPtr>());
    return;
  }
  background_collections_callback_ = std::move(callback);
  ntp_background_service_->FetchCollectionInfo();
}

void NewTabPageHandler::GetBackgroundImages(
    const std::string& collection_id,
    GetBackgroundImagesCallback callback) {
  if (background_images_callback_) {
    std::move(background_images_callback_)
        .Run(std::vector<new_tab_page::mojom::BackgroundImagePtr>());
  }
  if (!ntp_background_service_) {
    std::move(callback).Run(
        std::vector<new_tab_page::mojom::BackgroundImagePtr>());
    return;
  }
  images_request_collection_id_ = collection_id;
  background_images_callback_ = std::move(callback);
  ntp_background_service_->FetchCollectionImageInfo(collection_id);
}

void NewTabPageHandler::FocusOmnibox() {
  search::FocusOmnibox(true, web_contents_);
}

void NewTabPageHandler::PasteIntoOmnibox(const std::string& text) {
  search::PasteIntoOmnibox(base::UTF8ToUTF16(text), web_contents_);
}

void NewTabPageHandler::GetDoodle(GetDoodleCallback callback) {
  search_provider_logos::LogoCallbacks callbacks;
  std::string fresh_doodle_param;
  if (net::GetValueForKeyInQuery(web_contents_->GetLastCommittedURL(),
                                 "fresh-doodle", &fresh_doodle_param) &&
      fresh_doodle_param == "1") {
    // In fresh-doodle mode, wait for the desired doodle to be downloaded.
    callbacks.on_fresh_encoded_logo_available =
        base::BindOnce(&NewTabPageHandler::OnLogoAvailable,
                       weak_ptr_factory_.GetWeakPtr(), std::move(callback));
  } else {
    // In regular mode, return cached doodle as it is available faster.
    callbacks.on_cached_encoded_logo_available =
        base::BindOnce(&NewTabPageHandler::OnLogoAvailable,
                       weak_ptr_factory_.GetWeakPtr(), std::move(callback));
  }
  // This will trigger re-downloading the doodle and caching it. This means that
  // in regular mode a new doodle will be returned on subsequent NTP loads.
  logo_service_->GetLogo(std::move(callbacks), /*for_webui_ntp=*/true);
}

void NewTabPageHandler::NtpThemeChanged(const NtpTheme& ntp_theme) {
  page_->SetTheme(MakeTheme(ntp_theme));
}

void NewTabPageHandler::MostVisitedInfoChanged(
    const InstantMostVisitedInfo& info) {
  std::vector<new_tab_page::mojom::MostVisitedTilePtr> list;
  auto result = new_tab_page::mojom::MostVisitedInfo::New();
  for (auto& tile : info.items) {
    auto value = new_tab_page::mojom::MostVisitedTile::New();
    if (tile.title.empty()) {
      value->title = tile.url.spec();
      value->title_direction = base::i18n::LEFT_TO_RIGHT;
    } else {
      value->title = base::UTF16ToUTF8(tile.title);
      value->title_direction =
          base::i18n::GetFirstStrongCharacterDirection(tile.title);
    }
    value->url = tile.url;
    list.push_back(std::move(value));
  }
  result->custom_links_enabled = !info.use_most_visited;
  result->tiles = std::move(list);
  result->visible = info.is_visible;
  page_->SetMostVisitedInfo(std::move(result));
}

void NewTabPageHandler::OnCollectionInfoAvailable() {
  if (!background_collections_callback_) {
    return;
  }

  std::vector<new_tab_page::mojom::BackgroundCollectionPtr> collections;
  for (const auto& info : ntp_background_service_->collection_info()) {
    auto collection = new_tab_page::mojom::BackgroundCollection::New();
    collection->id = info.collection_id;
    collection->label = info.collection_name;
    collection->preview_image_url = GURL(info.preview_image_url);
    collections.push_back(std::move(collection));
  }
  std::move(background_collections_callback_).Run(std::move(collections));
}

void NewTabPageHandler::OnCollectionImagesAvailable() {
  if (!background_images_callback_) {
    return;
  }
  std::vector<new_tab_page::mojom::BackgroundImagePtr> images;
  if (ntp_background_service_->collection_images().empty()) {
    std::move(background_images_callback_).Run(std::move(images));
  }
  auto collection_id =
      ntp_background_service_->collection_images()[0].collection_id;
  for (const auto& info : ntp_background_service_->collection_images()) {
    DCHECK(info.collection_id == collection_id);
    auto image = new_tab_page::mojom::BackgroundImage::New();
    image->preview_image_url = GURL(info.thumbnail_image_url);
    image->label = !info.attribution.empty() ? info.attribution[0] : "";
    images.push_back(std::move(image));
  }
  std::move(background_images_callback_).Run(std::move(images));
}

void NewTabPageHandler::OnNextCollectionImageAvailable() {}

void NewTabPageHandler::OnNtpBackgroundServiceShuttingDown() {
  ntp_background_service_->RemoveObserver(this);
  ntp_background_service_ = nullptr;
}

void NewTabPageHandler::OnOmniboxInputStateChanged() {
  // This handler was added for the local NTP to show the fakebox when pressing
  // escape while the omnibox has focus. The WebUI NTP only shows the fakebox
  // when blurring the omnibox. Thus, we do nothing here.
}

void NewTabPageHandler::OnOmniboxFocusChanged(OmniboxFocusState state,
                                              OmniboxFocusChangeReason reason) {
  page_->SetFakeboxFocused(state == OMNIBOX_FOCUS_INVISIBLE);
  // Don't make fakebox re-appear for a short moment before navigating away.
  if (web_contents_->GetController().GetPendingEntry() == nullptr) {
    page_->SetFakeboxVisible(reason != OMNIBOX_FOCUS_CHANGE_TYPING);
  }
}

void NewTabPageHandler::OnLogoAvailable(
    GetDoodleCallback callback,
    search_provider_logos::LogoCallbackReason type,
    const base::Optional<search_provider_logos::EncodedLogo>& logo) {
  if (!logo) {
    std::move(callback).Run(nullptr);
    return;
  }
  auto doodle = new_tab_page::mojom::Doodle::New();
  switch (logo->metadata.type) {
    case search_provider_logos::LogoType::SIMPLE:
    case search_provider_logos::LogoType::ANIMATED: {
      if (!logo->encoded_image) {
        std::move(callback).Run(nullptr);
        return;
      }
      std::string base64;
      base::Base64Encode(logo->encoded_image->data(), &base64);
      auto data_url =
          base::StringPrintf("data:%s;base64,%s",
                             logo->metadata.mime_type.c_str(), base64.c_str());
      doodle->content = new_tab_page::mojom::DoodleContent::NewImage(data_url);
    } break;
    case search_provider_logos::LogoType::INTERACTIVE:
      doodle->content = new_tab_page::mojom::DoodleContent::NewUrl(
          logo->metadata.full_page_url);
      break;
    default:
      std::move(callback).Run(nullptr);
      return;
  }
  std::move(callback).Run(std::move(doodle));
}
