/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/core/schematics/migrations/renderer-to-renderer2/helpers", ["require", "exports", "typescript"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    const ts = require("typescript");
    /** Gets the string representation of a helper function. */
    function getHelper(name, sourceFile, printer) {
        const helperDeclaration = getHelperDeclaration(name);
        return '\n' + printer.printNode(ts.EmitHint.Unspecified, helperDeclaration, sourceFile) + '\n';
    }
    exports.getHelper = getHelper;
    /** Creates a function declaration for the specified helper name. */
    function getHelperDeclaration(name) {
        switch (name) {
            case "AnyDuringRendererMigration" /* any */:
                return createAnyTypeHelper();
            case "__ngRendererCreateElementHelper" /* createElement */:
                return getCreateElementHelper();
            case "__ngRendererCreateTextHelper" /* createText */:
                return getCreateTextHelper();
            case "__ngRendererCreateTemplateAnchorHelper" /* createTemplateAnchor */:
                return getCreateTemplateAnchorHelper();
            case "__ngRendererProjectNodesHelper" /* projectNodes */:
                return getProjectNodesHelper();
            case "__ngRendererAnimateHelper" /* animate */:
                return getAnimateHelper();
            case "__ngRendererDestroyViewHelper" /* destroyView */:
                return getDestroyViewHelper();
            case "__ngRendererDetachViewHelper" /* detachView */:
                return getDetachViewHelper();
            case "__ngRendererAttachViewAfterHelper" /* attachViewAfter */:
                return getAttachViewAfterHelper();
            case "__ngRendererSetElementAttributeHelper" /* setElementAttribute */:
                return getSetElementAttributeHelper();
            case "__ngRendererSplitNamespaceHelper" /* splitNamespace */:
                return getSplitNamespaceHelper();
        }
        throw new Error(`Unsupported helper called "${name}".`);
    }
    /** Creates a helper for a custom `any` type during the migration. */
    function createAnyTypeHelper() {
        // type AnyDuringRendererMigration = any;
        return ts.createTypeAliasDeclaration([], [], "AnyDuringRendererMigration" /* any */, [], ts.createKeywordTypeNode(ts.SyntaxKind.AnyKeyword));
    }
    /** Creates a function parameter that is typed as `any`. */
    function getAnyTypedParameter(parameterName, isRequired = true) {
        // Declare the parameter as `any` so we don't have to add extra logic to ensure that the
        // generated code will pass type checking. Use our custom `any` type so people have an incentive
        // to clean it up afterwards and to avoid potentially introducing lint warnings in G3.
        const type = ts.createTypeReferenceNode("AnyDuringRendererMigration" /* any */, []);
        return ts.createParameter([], [], undefined, parameterName, isRequired ? undefined : ts.createToken(ts.SyntaxKind.QuestionToken), type);
    }
    /** Creates a helper for `createElement`. */
    function getCreateElementHelper() {
        const renderer = ts.createIdentifier('renderer');
        const parent = ts.createIdentifier('parent');
        const namespaceAndName = ts.createIdentifier('namespaceAndName');
        const name = ts.createIdentifier('name');
        const namespace = ts.createIdentifier('namespace');
        // [namespace, name] = splitNamespace(namespaceAndName);
        const namespaceAndNameVariable = ts.createVariableDeclaration(ts.createArrayBindingPattern([namespace, name].map(id => ts.createBindingElement(undefined, undefined, id))), undefined, ts.createCall(ts.createIdentifier("__ngRendererSplitNamespaceHelper" /* splitNamespace */), [], [namespaceAndName]));
        // `renderer.createElement(name, namespace)`.
        const creationCall = ts.createCall(ts.createPropertyAccess(renderer, 'createElement'), [], [name, namespace]);
        return getCreationHelper("__ngRendererCreateElementHelper" /* createElement */, creationCall, renderer, parent, [namespaceAndName], [ts.createVariableStatement(undefined, ts.createVariableDeclarationList([namespaceAndNameVariable], ts.NodeFlags.Const))]);
    }
    /** Creates a helper for `createText`. */
    function getCreateTextHelper() {
        const renderer = ts.createIdentifier('renderer');
        const parent = ts.createIdentifier('parent');
        const value = ts.createIdentifier('value');
        // `renderer.createText(value)`.
        const creationCall = ts.createCall(ts.createPropertyAccess(renderer, 'createText'), [], [value]);
        return getCreationHelper("__ngRendererCreateTextHelper" /* createText */, creationCall, renderer, parent, [value]);
    }
    /** Creates a helper for `createTemplateAnchor`. */
    function getCreateTemplateAnchorHelper() {
        const renderer = ts.createIdentifier('renderer');
        const parent = ts.createIdentifier('parent');
        // `renderer.createComment('')`.
        const creationCall = ts.createCall(ts.createPropertyAccess(renderer, 'createComment'), [], [ts.createStringLiteral('')]);
        return getCreationHelper("__ngRendererCreateTemplateAnchorHelper" /* createTemplateAnchor */, creationCall, renderer, parent);
    }
    /**
     * Gets the function declaration for a creation helper. This is reused between `createElement`,
     * `createText` and `createTemplateAnchor` which follow a very similar pattern.
     * @param functionName Function that the helper should have.
     * @param creationCall Expression that is used to create a node inside the function.
     * @param rendererParameter Parameter for the `renderer`.
     * @param parentParameter Parameter for the `parent` inside the function.
     * @param extraParameters Extra parameters to be added to the end.
     * @param precedingVariables Extra variables to be added before the one that creates the `node`.
     */
    function getCreationHelper(functionName, creationCall, renderer, parent, extraParameters = [], precedingVariables = []) {
        const node = ts.createIdentifier('node');
        // `const node = {{creationCall}}`.
        const nodeVariableStatement = ts.createVariableStatement(undefined, ts.createVariableDeclarationList([ts.createVariableDeclaration(node, undefined, creationCall)], ts.NodeFlags.Const));
        // `if (parent) { renderer.appendChild(parent, node) }`.
        const guardedAppendChildCall = ts.createIf(parent, ts.createBlock([ts.createExpressionStatement(ts.createCall(ts.createPropertyAccess(renderer, 'appendChild'), [], [parent, node]))], true));
        return ts.createFunctionDeclaration([], [], undefined, functionName, [], [renderer, parent, ...extraParameters].map(name => getAnyTypedParameter(name)), undefined, ts.createBlock([
            ...precedingVariables, nodeVariableStatement, guardedAppendChildCall,
            ts.createReturn(node)
        ], true));
    }
    /** Creates a helper for `projectNodes`. */
    function getProjectNodesHelper() {
        const renderer = ts.createIdentifier('renderer');
        const parent = ts.createIdentifier('parent');
        const nodes = ts.createIdentifier('nodes');
        const incrementor = ts.createIdentifier('i');
        // for (let i = 0; i < nodes.length; i++) {
        //   renderer.appendChild(parent, nodes[i]);
        // }
        const loopInitializer = ts.createVariableDeclarationList([ts.createVariableDeclaration(incrementor, undefined, ts.createNumericLiteral('0'))], ts.NodeFlags.Let);
        const loopCondition = ts.createBinary(incrementor, ts.SyntaxKind.LessThanToken, ts.createPropertyAccess(nodes, ts.createIdentifier('length')));
        const appendStatement = ts.createExpressionStatement(ts.createCall(ts.createPropertyAccess(renderer, 'appendChild'), [], [parent, ts.createElementAccess(nodes, incrementor)]));
        const loop = ts.createFor(loopInitializer, loopCondition, ts.createPostfix(incrementor, ts.SyntaxKind.PlusPlusToken), ts.createBlock([appendStatement]));
        return ts.createFunctionDeclaration([], [], undefined, "__ngRendererProjectNodesHelper" /* projectNodes */, [], [renderer, parent, nodes].map(name => getAnyTypedParameter(name)), undefined, ts.createBlock([loop], true));
    }
    /** Creates a helper for `animate`. */
    function getAnimateHelper() {
        // throw new Error('...');
        const throwStatement = ts.createThrow(ts.createNew(ts.createIdentifier('Error'), [], [ts.createStringLiteral('Renderer.animate is no longer supported!')]));
        return ts.createFunctionDeclaration([], [], undefined, "__ngRendererAnimateHelper" /* animate */, [], [], undefined, ts.createBlock([throwStatement], true));
    }
    /** Creates a helper for `destroyView`. */
    function getDestroyViewHelper() {
        const renderer = ts.createIdentifier('renderer');
        const allNodes = ts.createIdentifier('allNodes');
        const incrementor = ts.createIdentifier('i');
        // for (let i = 0; i < allNodes.length; i++) {
        //   renderer.destroyNode(allNodes[i]);
        // }
        const loopInitializer = ts.createVariableDeclarationList([ts.createVariableDeclaration(incrementor, undefined, ts.createNumericLiteral('0'))], ts.NodeFlags.Let);
        const loopCondition = ts.createBinary(incrementor, ts.SyntaxKind.LessThanToken, ts.createPropertyAccess(allNodes, ts.createIdentifier('length')));
        const destroyStatement = ts.createExpressionStatement(ts.createCall(ts.createPropertyAccess(renderer, 'destroyNode'), [], [ts.createElementAccess(allNodes, incrementor)]));
        const loop = ts.createFor(loopInitializer, loopCondition, ts.createPostfix(incrementor, ts.SyntaxKind.PlusPlusToken), ts.createBlock([destroyStatement]));
        return ts.createFunctionDeclaration([], [], undefined, "__ngRendererDestroyViewHelper" /* destroyView */, [], [renderer, allNodes].map(name => getAnyTypedParameter(name)), undefined, ts.createBlock([loop], true));
    }
    /** Creates a helper for `detachView`. */
    function getDetachViewHelper() {
        const renderer = ts.createIdentifier('renderer');
        const rootNodes = ts.createIdentifier('rootNodes');
        const incrementor = ts.createIdentifier('i');
        const node = ts.createIdentifier('node');
        // for (let i = 0; i < rootNodes.length; i++) {
        //   const node = rootNodes[i];
        //   renderer.removeChild(renderer.parentNode(node), node);
        // }
        const loopInitializer = ts.createVariableDeclarationList([ts.createVariableDeclaration(incrementor, undefined, ts.createNumericLiteral('0'))], ts.NodeFlags.Let);
        const loopCondition = ts.createBinary(incrementor, ts.SyntaxKind.LessThanToken, ts.createPropertyAccess(rootNodes, ts.createIdentifier('length')));
        // const node = rootNodes[i];
        const nodeVariableStatement = ts.createVariableStatement(undefined, ts.createVariableDeclarationList([ts.createVariableDeclaration(node, undefined, ts.createElementAccess(rootNodes, incrementor))], ts.NodeFlags.Const));
        // renderer.removeChild(renderer.parentNode(node), node);
        const removeCall = ts.createCall(ts.createPropertyAccess(renderer, 'removeChild'), [], [ts.createCall(ts.createPropertyAccess(renderer, 'parentNode'), [], [node]), node]);
        const loop = ts.createFor(loopInitializer, loopCondition, ts.createPostfix(incrementor, ts.SyntaxKind.PlusPlusToken), ts.createBlock([nodeVariableStatement, ts.createExpressionStatement(removeCall)]));
        return ts.createFunctionDeclaration([], [], undefined, "__ngRendererDetachViewHelper" /* detachView */, [], [renderer, rootNodes].map(name => getAnyTypedParameter(name)), undefined, ts.createBlock([loop], true));
    }
    /** Creates a helper for `attachViewAfter` */
    function getAttachViewAfterHelper() {
        const renderer = ts.createIdentifier('renderer');
        const node = ts.createIdentifier('node');
        const rootNodes = ts.createIdentifier('rootNodes');
        const parent = ts.createIdentifier('parent');
        const nextSibling = ts.createIdentifier('nextSibling');
        const incrementor = ts.createIdentifier('i');
        const createConstWithMethodCallInitializer = (constName, methodToCall) => {
            return ts.createVariableStatement(undefined, ts.createVariableDeclarationList([ts.createVariableDeclaration(constName, undefined, ts.createCall(ts.createPropertyAccess(renderer, methodToCall), [], [node]))], ts.NodeFlags.Const));
        };
        // const parent = renderer.parentNode(node);
        const parentVariableStatement = createConstWithMethodCallInitializer(parent, 'parentNode');
        // const nextSibling = renderer.nextSibling(node);
        const nextSiblingVariableStatement = createConstWithMethodCallInitializer(nextSibling, 'nextSibling');
        // for (let i = 0; i < rootNodes.length; i++) {
        //   renderer.insertBefore(parentElement, rootNodes[i], nextSibling);
        // }
        const loopInitializer = ts.createVariableDeclarationList([ts.createVariableDeclaration(incrementor, undefined, ts.createNumericLiteral('0'))], ts.NodeFlags.Let);
        const loopCondition = ts.createBinary(incrementor, ts.SyntaxKind.LessThanToken, ts.createPropertyAccess(rootNodes, ts.createIdentifier('length')));
        const insertBeforeCall = ts.createCall(ts.createPropertyAccess(renderer, 'insertBefore'), [], [parent, ts.createElementAccess(rootNodes, incrementor), nextSibling]);
        const loop = ts.createFor(loopInitializer, loopCondition, ts.createPostfix(incrementor, ts.SyntaxKind.PlusPlusToken), ts.createBlock([ts.createExpressionStatement(insertBeforeCall)]));
        return ts.createFunctionDeclaration([], [], undefined, "__ngRendererAttachViewAfterHelper" /* attachViewAfter */, [], [renderer, node, rootNodes].map(name => getAnyTypedParameter(name)), undefined, ts.createBlock([parentVariableStatement, nextSiblingVariableStatement, loop], true));
    }
    /** Creates a helper for `setElementAttribute` */
    function getSetElementAttributeHelper() {
        const renderer = ts.createIdentifier('renderer');
        const element = ts.createIdentifier('element');
        const namespaceAndName = ts.createIdentifier('namespaceAndName');
        const value = ts.createIdentifier('value');
        const name = ts.createIdentifier('name');
        const namespace = ts.createIdentifier('namespace');
        // [namespace, name] = splitNamespace(namespaceAndName);
        const namespaceAndNameVariable = ts.createVariableDeclaration(ts.createArrayBindingPattern([namespace, name].map(id => ts.createBindingElement(undefined, undefined, id))), undefined, ts.createCall(ts.createIdentifier("__ngRendererSplitNamespaceHelper" /* splitNamespace */), [], [namespaceAndName]));
        // renderer.setAttribute(element, name, value, namespace);
        const setCall = ts.createCall(ts.createPropertyAccess(renderer, 'setAttribute'), [], [element, name, value, namespace]);
        // renderer.removeAttribute(element, name, namespace);
        const removeCall = ts.createCall(ts.createPropertyAccess(renderer, 'removeAttribute'), [], [element, name, namespace]);
        // if (value != null) { setCall() } else { removeCall }
        const ifStatement = ts.createIf(ts.createBinary(value, ts.SyntaxKind.ExclamationEqualsToken, ts.createNull()), ts.createBlock([ts.createExpressionStatement(setCall)], true), ts.createBlock([ts.createExpressionStatement(removeCall)], true));
        const functionBody = ts.createBlock([
            ts.createVariableStatement(undefined, ts.createVariableDeclarationList([namespaceAndNameVariable], ts.NodeFlags.Const)),
            ifStatement
        ], true);
        return ts.createFunctionDeclaration([], [], undefined, "__ngRendererSetElementAttributeHelper" /* setElementAttribute */, [], [
            getAnyTypedParameter(renderer), getAnyTypedParameter(element),
            getAnyTypedParameter(namespaceAndName), getAnyTypedParameter(value, false)
        ], undefined, functionBody);
    }
    /** Creates a helper for splitting a name that might contain a namespace. */
    function getSplitNamespaceHelper() {
        const name = ts.createIdentifier('name');
        const match = ts.createIdentifier('match');
        const regex = ts.createRegularExpressionLiteral('/^:([^:]+):(.+)$/');
        const matchCall = ts.createCall(ts.createPropertyAccess(name, 'match'), [], [regex]);
        // const match = name.split(regex);
        const matchVariable = ts.createVariableDeclarationList([ts.createVariableDeclaration(match, undefined, matchCall)], ts.NodeFlags.Const);
        // return [match[1], match[2]];
        const matchReturn = ts.createReturn(ts.createArrayLiteral([ts.createElementAccess(match, 1), ts.createElementAccess(match, 2)]));
        // if (name[0] === ':') { const match = ...; return ...; }
        const ifStatement = ts.createIf(ts.createBinary(ts.createElementAccess(name, 0), ts.SyntaxKind.EqualsEqualsEqualsToken, ts.createStringLiteral(':')), ts.createBlock([ts.createVariableStatement([], matchVariable), matchReturn], true));
        // return ['', name];
        const elseReturn = ts.createReturn(ts.createArrayLiteral([ts.createStringLiteral(''), name]));
        return ts.createFunctionDeclaration([], [], undefined, "__ngRendererSplitNamespaceHelper" /* splitNamespace */, [], [getAnyTypedParameter(name)], undefined, ts.createBlock([ifStatement, elseReturn], true));
    }
});
//# sourceMappingURL=data:application/json;base64,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