// SPDX-License-Identifier: GPL-2.0+
/*
 * USB specification data
 *
 * Copyright (C) 2013 Tom Gundersen <teg@jklm.no>
 */

#ifndef _USB_SPEC_H
#define _USB_SPEC_H

/* ---------------------------------------------------------------------- */

struct audioterminal {
       struct audioterminal *next;
       const uint16_t termt;
       const char *name;
};

struct videoterminal {
       struct videoterminal *next;
       const uint16_t termt;
       const char *name;
};

struct genericstrtable {
       struct genericstrtable *next;
       const unsigned int num;
       const char *name;
};

/* ---------------------------------------------------------------------- */

static struct audioterminal audioterminals[] =
{
        { NULL, 0x0100, "USB Undefined" },
        { NULL, 0x0101, "USB Streaming" },
        { NULL, 0x01ff, "USB Vendor Specific" },
        { NULL, 0x0200, "Input Undefined" },
        { NULL, 0x0201, "Microphone" },
        { NULL, 0x0202, "Desktop Microphone" },
        { NULL, 0x0203, "Personal Microphone" },
        { NULL, 0x0204, "Omni-directional Microphone" },
        { NULL, 0x0205, "Microphone Array" },
        { NULL, 0x0206, "Processing Microphone Array" },
        { NULL, 0x0300, "Output Undefined" },
        { NULL, 0x0301, "Speaker" },
        { NULL, 0x0302, "Headphones" },
        { NULL, 0x0303, "Head Mounted Display Audio" },
        { NULL, 0x0304, "Desktop Speaker" },
        { NULL, 0x0305, "Room Speaker" },
        { NULL, 0x0306, "Communication Speaker" },
        { NULL, 0x0307, "Low Frequency Effects Speaker" },
        { NULL, 0x0400, "Bidirectional Undefined" },
        { NULL, 0x0401, "Handset" },
        { NULL, 0x0402, "Headset" },
        { NULL, 0x0403, "Speakerphone, no echo reduction" },
        { NULL, 0x0404, "Echo-suppressing speakerphone" },
        { NULL, 0x0405, "Echo-canceling speakerphone" },
        { NULL, 0x0500, "Telephony Undefined" },
        { NULL, 0x0501, "Phone line" },
        { NULL, 0x0502, "Telephone" },
        { NULL, 0x0503, "Down Line Phone" },
        { NULL, 0x0600, "External Undefined" },
        { NULL, 0x0601, "Analog Connector" },
        { NULL, 0x0602, "Digital Audio Interface" },
        { NULL, 0x0603, "Line Connector" },
        { NULL, 0x0604, "Legacy Audio Connector" },
        { NULL, 0x0605, "SPDIF interface" },
        { NULL, 0x0606, "1394 DA stream" },
        { NULL, 0x0607, "1394 DV stream soundtrack" },
        { NULL, 0x0700, "Embedded Undefined" },
        { NULL, 0x0701, "Level Calibration Noise Source" },
        { NULL, 0x0702, "Equalization Noise" },
        { NULL, 0x0703, "CD Player" },
        { NULL, 0x0704, "DAT" },
        { NULL, 0x0705, "DCC" },
        { NULL, 0x0706, "MiniDisc" },
        { NULL, 0x0707, "Analog Tape" },
        { NULL, 0x0708, "Phonograph" },
        { NULL, 0x0709, "VCR Audio" },
        { NULL, 0x070a, "Video Disc Audio" },
        { NULL, 0x070b, "DVD Audio" },
        { NULL, 0x070c, "TV Tuner Audio" },
        { NULL, 0x070d, "Satellite Receiver Audio" },
        { NULL, 0x070e, "Cable Tuner Audio" },
        { NULL, 0x070f, "DSS Audio" },
        { NULL, 0x0710, "Radio Receiver" },
        { NULL, 0x0711, "Radio Transmitter" },
        { NULL, 0x0712, "Multitrack Recorder" },
        { NULL, 0x0713, "Synthesizer" },
	{ NULL },
};

static struct videoterminal videoterminals[] =
{
        { NULL, 0x0100, "USB Vendor Specific" },
        { NULL, 0x0101, "USB Streaming" },
        { NULL, 0x0200, "Input Vendor Specific" },
        { NULL, 0x0201, "Camera Sensor" },
        { NULL, 0x0202, "Sequential Media" },
        { NULL, 0x0300, "Output Vendor Specific" },
        { NULL, 0x0301, "Generic Display" },
        { NULL, 0x0302, "Sequential Media" },
        { NULL, 0x0400, "External Vendor Specific" },
        { NULL, 0x0401, "Composite Video" },
        { NULL, 0x0402, "S-Video" },
        { NULL, 0x0403, "Component Video" },
	{ NULL },
};

static struct genericstrtable hiddescriptors[] =
{
        { NULL, 0x21, "HID" },
        { NULL, 0x22, "Report" },
        { NULL, 0x23, "Physical" },
	{ NULL },
};

static struct genericstrtable reports[] =
{
        { NULL, 0x04, "Usage Page" },
        { NULL, 0x08, "Usage" },
        { NULL, 0x14, "Logical Minimum" },
        { NULL, 0x18, "Usage Minimum" },
        { NULL, 0x24, "Logical Maximum" },
        { NULL, 0x28, "Usage Maximum" },
        { NULL, 0x34, "Physical Minimum" },
        { NULL, 0x38, "Designator Index" },
        { NULL, 0x44, "Physical Maximum" },
        { NULL, 0x48, "Designator Minimum" },
        { NULL, 0x54, "Unit Exponent" },
        { NULL, 0x58, "Designator Maximum" },
        { NULL, 0x64, "Unit" },
        { NULL, 0x74, "Report Size" },
        { NULL, 0x78, "String Index" },
        { NULL, 0x80, "Input" },
        { NULL, 0x84, "Report ID" },
        { NULL, 0x88, "String Minimum" },
        { NULL, 0x90, "Output" },
        { NULL, 0x94, "Report Count" },
        { NULL, 0x98, "String Maximum" },
        { NULL, 0xa0, "Collection" },
        { NULL, 0xa4, "Push" },
        { NULL, 0xa8, "Delimiter" },
        { NULL, 0xb0, "Feature" },
        { NULL, 0xb4, "Pop" },
        { NULL, 0xc0, "End Collection" },
	{ NULL },
};

static struct genericstrtable huts[] =
{
        { NULL, 0x00, "Undefined" },
        { NULL, 0x01, "Generic Desktop Controls" },
        { NULL, 0x02, "Simulation Controls" },
        { NULL, 0x03, "VR Controls" },
        { NULL, 0x04, "Sport Controls" },
        { NULL, 0x05, "Game Controls" },
        { NULL, 0x07, "Keyboard" },
        { NULL, 0x08, "LEDs" },
        { NULL, 0x09, "Buttons" },
        { NULL, 0x0a, "Ordinal" },
        { NULL, 0x0b, "Telephony" },
        { NULL, 0x0c, "Consumer" },
        { NULL, 0x0d, "Digitizer" },
        { NULL, 0x0f, "PID Page" },
        { NULL, 0x10, "Unicode" },
        { NULL, 0x14, "Alphanumeric Display" },
        { NULL, 0x80, "USB Monitor" },
        { NULL, 0x81, "USB Monitor Enumerated Values" },
        { NULL, 0x82, "Monitor VESA Virtual Controls" },
        { NULL, 0x84, "Power Device Page" },
        { NULL, 0x85, "Battery System Page" },
        { NULL, 0x86, "Power Pages" },
        { NULL, 0x87, "Power Pages" },
        { NULL, 0x8c, "Bar Code Scanner Page (POS)" },
        { NULL, 0x8d, "Scale Page (POS)" },
        { NULL, 0x90, "Camera Control Page" },
        { NULL, 0x91, "Arcade Control Page" },
        { NULL, 0xf0, "Cash Device" },
        { NULL, 0xff, "Vendor Specific" },
	{ NULL },
};

static struct genericstrtable biass[] =
{
        { NULL, 0x0, "Not Applicable" },
        { NULL, 0x1, "Right Hand" },
        { NULL, 0x2, "Left Hand" },
        { NULL, 0x3, "Both Hands" },
        { NULL, 0x4, "Either Hand" },
	{ NULL },
};

static struct genericstrtable physdess[] =
{
        { NULL, 0x00, "None" },
        { NULL, 0x01, "Hand" },
        { NULL, 0x02, "Eyeball" },
        { NULL, 0x03, "Eyebrow" },
        { NULL, 0x04, "Eyelid" },
        { NULL, 0x05, "Ear" },
        { NULL, 0x06, "Nose" },
        { NULL, 0x07, "Mouth" },
        { NULL, 0x08, "Upper Lip" },
        { NULL, 0x09, "Lower Lip" },
        { NULL, 0x0a, "Jaw" },
        { NULL, 0x0b, "Neck" },
        { NULL, 0x0c, "Upper Arm" },
        { NULL, 0x0d, "Elbow" },
        { NULL, 0x0e, "Forearm" },
        { NULL, 0x0f, "Wrist" },
        { NULL, 0x10, "Palm" },
        { NULL, 0x11, "Thumb" },
        { NULL, 0x12, "Index Finger" },
        { NULL, 0x13, "Middle Finger" },
        { NULL, 0x14, "Ring Finger" },
        { NULL, 0x15, "Little Finger" },
        { NULL, 0x16, "Head" },
        { NULL, 0x17, "Shoulder" },
        { NULL, 0x18, "Hip" },
        { NULL, 0x19, "Waist" },
        { NULL, 0x1a, "Thigh" },
        { NULL, 0x1b, "Knee" },
        { NULL, 0x1c, "calf" },
        { NULL, 0x1d, "Ankle" },
        { NULL, 0x1e, "Foot" },
        { NULL, 0x1f, "Heel" },
        { NULL, 0x20, "Ball of Foot" },
        { NULL, 0x21, "Big Toe" },
        { NULL, 0x22, "Second Toe" },
        { NULL, 0x23, "Third Toe" },
        { NULL, 0x24, "Fourth Toe" },
        { NULL, 0x25, "Fifth Toe" },
        { NULL, 0x26, "Brow" },
        { NULL, 0x27, "Cheek" },
	{ NULL },
};

/* HUT usage specs are represented as:
 *	(huttype << 16) + hutus
 */
static struct genericstrtable hutus[] =
{
	{ NULL, (0x01 << 16) + 0x000, "Undefined" },
	{ NULL, (0x01 << 16) + 0x001, "Pointer" },
	{ NULL, (0x01 << 16) + 0x002, "Mouse" },
	{ NULL, (0x01 << 16) + 0x004, "Joystick" },
	{ NULL, (0x01 << 16) + 0x005, "Gamepad" },
	{ NULL, (0x01 << 16) + 0x006, "Keyboard" },
	{ NULL, (0x01 << 16) + 0x007, "Keypad" },
	{ NULL, (0x01 << 16) + 0x008, "Multi-Axis Controller" },
	{ NULL, (0x01 << 16) + 0x030, "Direction-X" },
	{ NULL, (0x01 << 16) + 0x031, "Direction-Y" },
	{ NULL, (0x01 << 16) + 0x032, "Direction-Z" },
	{ NULL, (0x01 << 16) + 0x033, "Rotate-X" },
	{ NULL, (0x01 << 16) + 0x034, "Rotate-Y" },
	{ NULL, (0x01 << 16) + 0x035, "Rotate-Z" },
	{ NULL, (0x01 << 16) + 0x036, "Slider" },
	{ NULL, (0x01 << 16) + 0x037, "Dial" },
	{ NULL, (0x01 << 16) + 0x038, "Wheel" },
	{ NULL, (0x01 << 16) + 0x039, "Hat Switch" },
	{ NULL, (0x01 << 16) + 0x03a, "Counted Buffer" },
	{ NULL, (0x01 << 16) + 0x03b, "Byte Count" },
	{ NULL, (0x01 << 16) + 0x03c, "Motion Wakeup" },
	{ NULL, (0x01 << 16) + 0x03d, "Start" },
	{ NULL, (0x01 << 16) + 0x03e, "Select" },
	{ NULL, (0x01 << 16) + 0x040, "Vector-X" },
	{ NULL, (0x01 << 16) + 0x041, "Vector-Y" },
	{ NULL, (0x01 << 16) + 0x042, "Vector-Z" },
	{ NULL, (0x01 << 16) + 0x043, "Vector-X relative Body" },
	{ NULL, (0x01 << 16) + 0x044, "Vector-Y relative Body" },
	{ NULL, (0x01 << 16) + 0x045, "Vector-Z relative Body" },
	{ NULL, (0x01 << 16) + 0x046, "Vector" },
	{ NULL, (0x01 << 16) + 0x080, "System Control" },
	{ NULL, (0x01 << 16) + 0x081, "System Power Down" },
	{ NULL, (0x01 << 16) + 0x082, "System Sleep" },
	{ NULL, (0x01 << 16) + 0x083, "System Wake Up" },
	{ NULL, (0x01 << 16) + 0x084, "System Context Menu" },
	{ NULL, (0x01 << 16) + 0x085, "System Main Menu" },
	{ NULL, (0x01 << 16) + 0x086, "System App Menu" },
	{ NULL, (0x01 << 16) + 0x087, "System Menu Help" },
	{ NULL, (0x01 << 16) + 0x088, "System Menu Exit" },
	{ NULL, (0x01 << 16) + 0x089, "System Menu Select" },
	{ NULL, (0x01 << 16) + 0x08a, "System Menu Right" },
	{ NULL, (0x01 << 16) + 0x08b, "System Menu Left" },
	{ NULL, (0x01 << 16) + 0x08c, "System Menu Up" },
	{ NULL, (0x01 << 16) + 0x08d, "System Menu Down" },
	{ NULL, (0x01 << 16) + 0x090, "Direction Pad Up" },
	{ NULL, (0x01 << 16) + 0x091, "Direction Pad Down" },
	{ NULL, (0x01 << 16) + 0x092, "Direction Pad Right" },
	{ NULL, (0x01 << 16) + 0x093, "Direction Pad Left" },
	{ NULL, (0x02 << 16) + 0x000, "Undefined" },
	{ NULL, (0x02 << 16) + 0x001, "Flight Simulation Device" },
	{ NULL, (0x02 << 16) + 0x002, "Automobile Simulation Device" },
	{ NULL, (0x02 << 16) + 0x003, "Tank Simulation Device" },
	{ NULL, (0x02 << 16) + 0x004, "Spaceship Simulation Device" },
	{ NULL, (0x02 << 16) + 0x005, "Submarine Simulation Device" },
	{ NULL, (0x02 << 16) + 0x006, "Sailing Simulation Device" },
	{ NULL, (0x02 << 16) + 0x007, "Motorcycle Simulation Device" },
	{ NULL, (0x02 << 16) + 0x008, "Sports Simulation Device" },
	{ NULL, (0x02 << 16) + 0x009, "Airplane Simualtion Device" },
	{ NULL, (0x02 << 16) + 0x00a, "Helicopter Simulation Device" },
	{ NULL, (0x02 << 16) + 0x00b, "Magic Carpet Simulation Device" },
	{ NULL, (0x02 << 16) + 0x00c, "Bicycle Simulation Device" },
	{ NULL, (0x02 << 16) + 0x020, "Flight Control Stick" },
	{ NULL, (0x02 << 16) + 0x021, "Flight Stick" },
	{ NULL, (0x02 << 16) + 0x022, "Cyclic Control" },
	{ NULL, (0x02 << 16) + 0x023, "Cyclic Trim" },
	{ NULL, (0x02 << 16) + 0x024, "Flight Yoke" },
	{ NULL, (0x02 << 16) + 0x025, "Track Control" },
	{ NULL, (0x02 << 16) + 0x0b0, "Aileron" },
	{ NULL, (0x02 << 16) + 0x0b1, "Aileron Trim" },
	{ NULL, (0x02 << 16) + 0x0b2, "Anti-Torque Control" },
	{ NULL, (0x02 << 16) + 0x0b3, "Autopilot Enable" },
	{ NULL, (0x02 << 16) + 0x0b4, "Chaff Release" },
	{ NULL, (0x02 << 16) + 0x0b5, "Collective Control" },
	{ NULL, (0x02 << 16) + 0x0b6, "Dive Break" },
	{ NULL, (0x02 << 16) + 0x0b7, "Electronic Countermeasures" },
	{ NULL, (0x02 << 16) + 0x0b8, "Elevator" },
	{ NULL, (0x02 << 16) + 0x0b9, "Elevator Trim" },
	{ NULL, (0x02 << 16) + 0x0ba, "Rudder" },
	{ NULL, (0x02 << 16) + 0x0bb, "Throttle" },
	{ NULL, (0x02 << 16) + 0x0bc, "Flight COmmunications" },
	{ NULL, (0x02 << 16) + 0x0bd, "Flare Release" },
	{ NULL, (0x02 << 16) + 0x0be, "Landing Gear" },
	{ NULL, (0x02 << 16) + 0x0bf, "Toe Break" },
	{ NULL, (0x02 << 16) + 0x0c0, "Trigger" },
	{ NULL, (0x02 << 16) + 0x0c1, "Weapon Arm" },
	{ NULL, (0x02 << 16) + 0x0c2, "Weapons Select" },
	{ NULL, (0x02 << 16) + 0x0c3, "Wing Flaps" },
	{ NULL, (0x02 << 16) + 0x0c4, "Accelerator" },
	{ NULL, (0x02 << 16) + 0x0c5, "Brake" },
	{ NULL, (0x02 << 16) + 0x0c6, "Clutch" },
	{ NULL, (0x02 << 16) + 0x0c7, "Shifter" },
	{ NULL, (0x02 << 16) + 0x0c8, "Steering" },
	{ NULL, (0x02 << 16) + 0x0c9, "Turret Direction" },
	{ NULL, (0x02 << 16) + 0x0ca, "Barrel Elevation" },
	{ NULL, (0x02 << 16) + 0x0cb, "Drive Plane" },
	{ NULL, (0x02 << 16) + 0x0cc, "Ballast" },
	{ NULL, (0x02 << 16) + 0x0cd, "Bicylce Crank" },
	{ NULL, (0x02 << 16) + 0x0ce, "Handle Bars" },
	{ NULL, (0x02 << 16) + 0x0cf, "Front Brake" },
	{ NULL, (0x02 << 16) + 0x0d0, "Rear Brake" },
	{ NULL, (0x03 << 16) + 0x000, "Unidentified" },
	{ NULL, (0x03 << 16) + 0x001, "Belt" },
	{ NULL, (0x03 << 16) + 0x002, "Body Suit" },
	{ NULL, (0x03 << 16) + 0x003, "Flexor" },
	{ NULL, (0x03 << 16) + 0x004, "Glove" },
	{ NULL, (0x03 << 16) + 0x005, "Head Tracker" },
	{ NULL, (0x03 << 16) + 0x006, "Head Mounted Display" },
	{ NULL, (0x03 << 16) + 0x007, "Hand Tracker" },
	{ NULL, (0x03 << 16) + 0x008, "Oculometer" },
	{ NULL, (0x03 << 16) + 0x009, "Vest" },
	{ NULL, (0x03 << 16) + 0x00a, "Animatronic Device" },
	{ NULL, (0x03 << 16) + 0x020, "Stereo Enable" },
	{ NULL, (0x03 << 16) + 0x021, "Display Enable" },
	{ NULL, (0x04 << 16) + 0x000, "Unidentified" },
	{ NULL, (0x04 << 16) + 0x001, "Baseball Bat" },
	{ NULL, (0x04 << 16) + 0x002, "Golf Club" },
	{ NULL, (0x04 << 16) + 0x003, "Rowing Machine" },
	{ NULL, (0x04 << 16) + 0x004, "Treadmill" },
	{ NULL, (0x04 << 16) + 0x030, "Oar" },
	{ NULL, (0x04 << 16) + 0x031, "Slope" },
	{ NULL, (0x04 << 16) + 0x032, "Rate" },
	{ NULL, (0x04 << 16) + 0x033, "Stick Speed" },
	{ NULL, (0x04 << 16) + 0x034, "Stick Face Angle" },
	{ NULL, (0x04 << 16) + 0x035, "Stick Heel/Toe" },
	{ NULL, (0x04 << 16) + 0x036, "Stick Follow Through" },
	{ NULL, (0x04 << 16) + 0x038, "Stick Type" },
	{ NULL, (0x04 << 16) + 0x039, "Stick Height" },
	{ NULL, (0x04 << 16) + 0x047, "Stick Temp" },
	{ NULL, (0x04 << 16) + 0x050, "Putter" },
	{ NULL, (0x04 << 16) + 0x051, "1 Iron" },
	{ NULL, (0x04 << 16) + 0x052, "2 Iron" },
	{ NULL, (0x04 << 16) + 0x053, "3 Iron" },
	{ NULL, (0x04 << 16) + 0x054, "4 Iron" },
	{ NULL, (0x04 << 16) + 0x055, "5 Iron" },
	{ NULL, (0x04 << 16) + 0x056, "6 Iron" },
	{ NULL, (0x04 << 16) + 0x057, "7 Iron" },
	{ NULL, (0x04 << 16) + 0x058, "8 Iron" },
	{ NULL, (0x04 << 16) + 0x059, "9 Iron" },
	{ NULL, (0x04 << 16) + 0x05a, "10 Iron" },
	{ NULL, (0x04 << 16) + 0x05b, "11 Iron" },
	{ NULL, (0x04 << 16) + 0x05c, "Sand Wedge" },
	{ NULL, (0x04 << 16) + 0x05d, "Loft Wedge" },
	{ NULL, (0x04 << 16) + 0x05e, "Power Wedge" },
	{ NULL, (0x04 << 16) + 0x05f, "1 Wood" },
	{ NULL, (0x04 << 16) + 0x060, "3 Wood" },
	{ NULL, (0x04 << 16) + 0x061, "5 Wood" },
	{ NULL, (0x04 << 16) + 0x062, "7 Wood" },
	{ NULL, (0x04 << 16) + 0x063, "9 Wood" },
	{ NULL, (0x05 << 16) + 0x000, "Undefined" },
	{ NULL, (0x05 << 16) + 0x001, "3D Game Controller" },
	{ NULL, (0x05 << 16) + 0x002, "Pinball Device" },
	{ NULL, (0x05 << 16) + 0x003, "Gun Device" },
	{ NULL, (0x05 << 16) + 0x020, "Point Of View" },
	{ NULL, (0x05 << 16) + 0x021, "Turn Right/Left" },
	{ NULL, (0x05 << 16) + 0x022, "Pitch Right/Left" },
	{ NULL, (0x05 << 16) + 0x023, "Roll Forward/Backward" },
	{ NULL, (0x05 << 16) + 0x024, "Move Right/Left" },
	{ NULL, (0x05 << 16) + 0x025, "Move Forward/Backward" },
	{ NULL, (0x05 << 16) + 0x026, "Move Up/Down" },
	{ NULL, (0x05 << 16) + 0x027, "Lean Right/Left" },
	{ NULL, (0x05 << 16) + 0x028, "Lean Forward/Backward" },
	{ NULL, (0x05 << 16) + 0x029, "Height of POV" },
	{ NULL, (0x05 << 16) + 0x02a, "Flipper" },
	{ NULL, (0x05 << 16) + 0x02b, "Secondary Flipper" },
	{ NULL, (0x05 << 16) + 0x02c, "Bump" },
	{ NULL, (0x05 << 16) + 0x02d, "New Game" },
	{ NULL, (0x05 << 16) + 0x02e, "Shoot Ball" },
	{ NULL, (0x05 << 16) + 0x02f, "Player" },
	{ NULL, (0x05 << 16) + 0x030, "Gun Bolt" },
	{ NULL, (0x05 << 16) + 0x031, "Gun Clip" },
	{ NULL, (0x05 << 16) + 0x032, "Gun Selector" },
	{ NULL, (0x05 << 16) + 0x033, "Gun Single Shot" },
	{ NULL, (0x05 << 16) + 0x034, "Gun Burst" },
	{ NULL, (0x05 << 16) + 0x035, "Gun Automatic" },
	{ NULL, (0x05 << 16) + 0x036, "Gun Safety" },
	{ NULL, (0x05 << 16) + 0x037, "Gamepad Fire/Jump" },
	{ NULL, (0x05 << 16) + 0x038, "Gamepad Fun" },
	{ NULL, (0x05 << 16) + 0x039, "Gamepad Trigger" },
	{ NULL, (0x07 << 16) + 0x000, "No Event" },
	{ NULL, (0x07 << 16) + 0x001, "Keyboard ErrorRollOver" },
	{ NULL, (0x07 << 16) + 0x002, "Keyboard POSTfail" },
	{ NULL, (0x07 << 16) + 0x003, "Keyboard Error Undefined" },
	{ NULL, (0x07 << 16) + 0x004, "A" },
	{ NULL, (0x07 << 16) + 0x005, "B" },
	{ NULL, (0x07 << 16) + 0x006, "C" },
	{ NULL, (0x07 << 16) + 0x007, "D" },
	{ NULL, (0x07 << 16) + 0x008, "E" },
	{ NULL, (0x07 << 16) + 0x009, "F" },
	{ NULL, (0x07 << 16) + 0x00a, "G" },
	{ NULL, (0x07 << 16) + 0x00b, "H" },
	{ NULL, (0x07 << 16) + 0x00c, "I" },
	{ NULL, (0x07 << 16) + 0x00d, "J" },
	{ NULL, (0x07 << 16) + 0x00e, "K" },
	{ NULL, (0x07 << 16) + 0x00f, "L" },
	{ NULL, (0x07 << 16) + 0x010, "M" },
	{ NULL, (0x07 << 16) + 0x011, "N" },
	{ NULL, (0x07 << 16) + 0x012, "O" },
	{ NULL, (0x07 << 16) + 0x013, "P" },
	{ NULL, (0x07 << 16) + 0x014, "Q" },
	{ NULL, (0x07 << 16) + 0x015, "R" },
	{ NULL, (0x07 << 16) + 0x016, "S" },
	{ NULL, (0x07 << 16) + 0x017, "T" },
	{ NULL, (0x07 << 16) + 0x018, "U" },
	{ NULL, (0x07 << 16) + 0x019, "V" },
	{ NULL, (0x07 << 16) + 0x01a, "W" },
	{ NULL, (0x07 << 16) + 0x01b, "X" },
	{ NULL, (0x07 << 16) + 0x01c, "Y" },
	{ NULL, (0x07 << 16) + 0x01d, "Z" },
	{ NULL, (0x07 << 16) + 0x01e, "1 and ! (One and Exclamation)" },
	{ NULL, (0x07 << 16) + 0x01f, "2 and @ (2 and at)" },
	{ NULL, (0x07 << 16) + 0x020, "3 and # (3 and Hash)" },
	{ NULL, (0x07 << 16) + 0x021, "4 and $ (4 and Dollar Sign)" },
	{ NULL, (0x07 << 16) + 0x022, "5 and % (5 and Percent Sign)" },
	{ NULL, (0x07 << 16) + 0x023, "6 and ^ (6 and circumflex)" },
	{ NULL, (0x07 << 16) + 0x024, "7 and & (Seven and Ampersand)" },
	{ NULL, (0x07 << 16) + 0x025, "8 and * (Eight and asterisk)" },
	{ NULL, (0x07 << 16) + 0x026, "9 and ( (Nine and Parenthesis Left)" },
	{ NULL, (0x07 << 16) + 0x027, "0 and ) (Zero and Parenthesis Right)" },
	{ NULL, (0x07 << 16) + 0x028, "Return (Enter)" },
	{ NULL, (0x07 << 16) + 0x029, "Escape" },
	{ NULL, (0x07 << 16) + 0x02a, "Delete (Backspace)" },
	{ NULL, (0x07 << 16) + 0x02b, "Tab" },
	{ NULL, (0x07 << 16) + 0x02c, "Space Bar" },
	{ NULL, (0x07 << 16) + 0x02d, "- and _ (Minus and underscore)" },
	{ NULL, (0x07 << 16) + 0x02e, "= and + (Equal and Plus)" },
	{ NULL, (0x07 << 16) + 0x02f, "[ and { (Bracket and Braces Left)" },
	{ NULL, (0x07 << 16) + 0x030, "] and } (Bracket and Braces Right)" },
	{ NULL, (0x07 << 16) + 0x031, "\\ and | (Backslash and Bar)" },
	{ NULL, (0x07 << 16) + 0x032, "# and ~ (Hash and Tilde, Non-US Keyboard near right shift)" },
	{ NULL, (0x07 << 16) + 0x033, "; and : (Semicolon and Colon)" },
	{ NULL, (0x07 << 16) + 0x034, "´ and \" (Accent Acute and Double Quotes)" },
	{ NULL, (0x07 << 16) + 0x035, "` and ~ (Accent Grace and Tilde)" },
	{ NULL, (0x07 << 16) + 0x036, ", and < (Comma and Less)" },
	{ NULL, (0x07 << 16) + 0x037, ". and > (Period and Greater)" },
	{ NULL, (0x07 << 16) + 0x038, "/ and ? (Slash and Question Mark)" },
	{ NULL, (0x07 << 16) + 0x039, "Caps Lock" },
	{ NULL, (0x07 << 16) + 0x03a, "F1" },
	{ NULL, (0x07 << 16) + 0x03b, "F2" },
	{ NULL, (0x07 << 16) + 0x03c, "F3" },
	{ NULL, (0x07 << 16) + 0x03d, "F4" },
	{ NULL, (0x07 << 16) + 0x03e, "F5" },
	{ NULL, (0x07 << 16) + 0x03f, "F6" },
	{ NULL, (0x07 << 16) + 0x040, "F7" },
	{ NULL, (0x07 << 16) + 0x041, "F8" },
	{ NULL, (0x07 << 16) + 0x042, "F9" },
	{ NULL, (0x07 << 16) + 0x043, "F10" },
	{ NULL, (0x07 << 16) + 0x044, "F11" },
	{ NULL, (0x07 << 16) + 0x045, "F12" },
	{ NULL, (0x07 << 16) + 0x046, "Print Screen" },
	{ NULL, (0x07 << 16) + 0x047, "Scroll Lock" },
	{ NULL, (0x07 << 16) + 0x048, "Pause" },
	{ NULL, (0x07 << 16) + 0x049, "Insert" },
	{ NULL, (0x07 << 16) + 0x04a, "Home" },
	{ NULL, (0x07 << 16) + 0x04b, "Page Up" },
	{ NULL, (0x07 << 16) + 0x04c, "Delete Forward (without Changing Position)" },
	{ NULL, (0x07 << 16) + 0x04d, "End" },
	{ NULL, (0x07 << 16) + 0x04e, "Page Down" },
	{ NULL, (0x07 << 16) + 0x04f, "Right Arrow" },
	{ NULL, (0x07 << 16) + 0x050, "Left Arrow" },
	{ NULL, (0x07 << 16) + 0x051, "Down Arrow" },
	{ NULL, (0x07 << 16) + 0x052, "Up Arrow" },
	{ NULL, (0x07 << 16) + 0x053, "Num Lock and Clear" },
	{ NULL, (0x07 << 16) + 0x054, "Keypad / (Division Sign)" },
	{ NULL, (0x07 << 16) + 0x055, "Keypad * (Multiplication Sign)" },
	{ NULL, (0x07 << 16) + 0x056, "Keypad - (Subtraction Sign)" },
	{ NULL, (0x07 << 16) + 0x057, "Keypad + (Addition Sign)" },
	{ NULL, (0x07 << 16) + 0x058, "Keypad Enter" },
	{ NULL, (0x07 << 16) + 0x059, "Keypad 1 and END" },
	{ NULL, (0x07 << 16) + 0x05a, "Keypad 2 and Down Arrow" },
	{ NULL, (0x07 << 16) + 0x05b, "Keypad 3 and Page Down" },
	{ NULL, (0x07 << 16) + 0x05c, "Keypad 4 and Left Arrow" },
	{ NULL, (0x07 << 16) + 0x05d, "Keypad 5 (Tactilei Raised)" },
	{ NULL, (0x07 << 16) + 0x05f, "Keypad 6 and Right Arrow" },
	{ NULL, (0x07 << 16) + 0x060, "Keypad 7 and Home" },
	{ NULL, (0x07 << 16) + 0x061, "Keypad 8 and Up Arrow" },
	{ NULL, (0x07 << 16) + 0x062, "Keypad 8 and Page Up" },
	{ NULL, (0x07 << 16) + 0x063, "Keypad . (decimal delimiter) and Delete" },
	{ NULL, (0x07 << 16) + 0x064, "\\ and | (Backslash and Bar, UK and Non-US Keyboard near left shift)" },
	{ NULL, (0x07 << 16) + 0x065, "Keyboard Application (Windows Key for Win95 or Compose)" },
	{ NULL, (0x07 << 16) + 0x066, "Power (not a key)" },
	{ NULL, (0x07 << 16) + 0x067, "Keypad = (Equal Sign)" },
	{ NULL, (0x07 << 16) + 0x068, "F13" },
	{ NULL, (0x07 << 16) + 0x069, "F14" },
	{ NULL, (0x07 << 16) + 0x06a, "F15" },
	{ NULL, (0x07 << 16) + 0x06b, "F16" },
	{ NULL, (0x07 << 16) + 0x06c, "F17" },
	{ NULL, (0x07 << 16) + 0x06d, "F18" },
	{ NULL, (0x07 << 16) + 0x06e, "F19" },
	{ NULL, (0x07 << 16) + 0x06f, "F20" },
	{ NULL, (0x07 << 16) + 0x070, "F21" },
	{ NULL, (0x07 << 16) + 0x071, "F22" },
	{ NULL, (0x07 << 16) + 0x072, "F23" },
	{ NULL, (0x07 << 16) + 0x073, "F24" },
	{ NULL, (0x07 << 16) + 0x074, "Execute" },
	{ NULL, (0x07 << 16) + 0x075, "Help" },
	{ NULL, (0x07 << 16) + 0x076, "Menu" },
	{ NULL, (0x07 << 16) + 0x077, "Select" },
	{ NULL, (0x07 << 16) + 0x078, "Stop" },
	{ NULL, (0x07 << 16) + 0x079, "Again" },
	{ NULL, (0x07 << 16) + 0x07a, "Undo" },
	{ NULL, (0x07 << 16) + 0x07b, "Cut" },
	{ NULL, (0x07 << 16) + 0x07c, "Copy" },
	{ NULL, (0x07 << 16) + 0x07d, "Paste" },
	{ NULL, (0x07 << 16) + 0x07e, "Find" },
	{ NULL, (0x07 << 16) + 0x07f, "Mute" },
	{ NULL, (0x07 << 16) + 0x080, "Volume Up" },
	{ NULL, (0x07 << 16) + 0x081, "Volume Down" },
	{ NULL, (0x07 << 16) + 0x082, "Locking Caps Lock" },
	{ NULL, (0x07 << 16) + 0x083, "Locking Num Lock" },
	{ NULL, (0x07 << 16) + 0x084, "Locking Scroll Lock" },
	{ NULL, (0x07 << 16) + 0x085, "Keypad Comma" },
	{ NULL, (0x07 << 16) + 0x086, "Keypad Equal Sign (AS/400)" },
	{ NULL, (0x07 << 16) + 0x087, "International 1 (PC98)" },
	{ NULL, (0x07 << 16) + 0x088, "International 2 (PC98)" },
	{ NULL, (0x07 << 16) + 0x089, "International 3 (PC98)" },
	{ NULL, (0x07 << 16) + 0x08a, "International 4 (PC98)" },
	{ NULL, (0x07 << 16) + 0x08b, "International 5 (PC98)" },
	{ NULL, (0x07 << 16) + 0x08c, "International 6 (PC98)" },
	{ NULL, (0x07 << 16) + 0x08d, "International 7 (Toggle Single/Double Byte Mode)" },
	{ NULL, (0x07 << 16) + 0x08e, "International 8" },
	{ NULL, (0x07 << 16) + 0x08f, "International 9" },
	{ NULL, (0x07 << 16) + 0x090, "LANG 1 (Hangul/English Toggle, Korea)" },
	{ NULL, (0x07 << 16) + 0x091, "LANG 2 (Hanja Conversion, Korea)" },
	{ NULL, (0x07 << 16) + 0x092, "LANG 3 (Katakana, Japan)" },
	{ NULL, (0x07 << 16) + 0x093, "LANG 4 (Hiragana, Japan)" },
	{ NULL, (0x07 << 16) + 0x094, "LANG 5 (Zenkaku/Hankaku, Japan)" },
	{ NULL, (0x07 << 16) + 0x095, "LANG 6" },
	{ NULL, (0x07 << 16) + 0x096, "LANG 7" },
	{ NULL, (0x07 << 16) + 0x097, "LANG 8" },
	{ NULL, (0x07 << 16) + 0x098, "LANG 9" },
	{ NULL, (0x07 << 16) + 0x099, "Alternate Erase" },
	{ NULL, (0x07 << 16) + 0x09a, "SysReq/Attention" },
	{ NULL, (0x07 << 16) + 0x09b, "Cancel" },
	{ NULL, (0x07 << 16) + 0x09c, "Clear" },
	{ NULL, (0x07 << 16) + 0x09d, "Prior" },
	{ NULL, (0x07 << 16) + 0x09e, "Return" },
	{ NULL, (0x07 << 16) + 0x09f, "Separator" },
	{ NULL, (0x07 << 16) + 0x0a0, "Out" },
	{ NULL, (0x07 << 16) + 0x0a1, "Open" },
	{ NULL, (0x07 << 16) + 0x0a2, "Clear/Again" },
	{ NULL, (0x07 << 16) + 0x0a3, "CrSel/Props" },
	{ NULL, (0x07 << 16) + 0x0a4, "ExSel" },
	{ NULL, (0x07 << 16) + 0x0e0, "Control Left" },
	{ NULL, (0x07 << 16) + 0x0e1, "Shift Left" },
	{ NULL, (0x07 << 16) + 0x0e2, "Alt Left" },
	{ NULL, (0x07 << 16) + 0x0e3, "GUI Left" },
	{ NULL, (0x07 << 16) + 0x0e4, "Control Right" },
	{ NULL, (0x07 << 16) + 0x0e5, "Shift Right" },
	{ NULL, (0x07 << 16) + 0x0e6, "Alt Rigth" },
	{ NULL, (0x07 << 16) + 0x0e7, "GUI Right" },
	{ NULL, (0x08 << 16) + 0x000, "Undefined" },
	{ NULL, (0x08 << 16) + 0x001, "NumLock" },
	{ NULL, (0x08 << 16) + 0x002, "CapsLock" },
	{ NULL, (0x08 << 16) + 0x003, "Scroll Lock" },
	{ NULL, (0x08 << 16) + 0x004, "Compose" },
	{ NULL, (0x08 << 16) + 0x005, "Kana" },
	{ NULL, (0x08 << 16) + 0x006, "Power" },
	{ NULL, (0x08 << 16) + 0x007, "Shift" },
	{ NULL, (0x08 << 16) + 0x008, "Do not disturb" },
	{ NULL, (0x08 << 16) + 0x009, "Mute" },
	{ NULL, (0x08 << 16) + 0x00a, "Tone Enabke" },
	{ NULL, (0x08 << 16) + 0x00b, "High Cut Filter" },
	{ NULL, (0x08 << 16) + 0x00c, "Low Cut Filter" },
	{ NULL, (0x08 << 16) + 0x00d, "Equalizer Enable" },
	{ NULL, (0x08 << 16) + 0x00e, "Sound Field ON" },
	{ NULL, (0x08 << 16) + 0x00f, "Surround On" },
	{ NULL, (0x08 << 16) + 0x010, "Repeat" },
	{ NULL, (0x08 << 16) + 0x011, "Stereo" },
	{ NULL, (0x08 << 16) + 0x012, "Sampling Rate Detect" },
	{ NULL, (0x08 << 16) + 0x013, "Spinning" },
	{ NULL, (0x08 << 16) + 0x014, "CAV" },
	{ NULL, (0x08 << 16) + 0x015, "CLV" },
	{ NULL, (0x08 << 16) + 0x016, "Recording Format Detect" },
	{ NULL, (0x08 << 16) + 0x017, "Off-Hook" },
	{ NULL, (0x08 << 16) + 0x018, "Ring" },
	{ NULL, (0x08 << 16) + 0x019, "Message Waiting" },
	{ NULL, (0x08 << 16) + 0x01a, "Data Mode" },
	{ NULL, (0x08 << 16) + 0x01b, "Battery Operation" },
	{ NULL, (0x08 << 16) + 0x01c, "Battery OK" },
	{ NULL, (0x08 << 16) + 0x01d, "Battery Low" },
	{ NULL, (0x08 << 16) + 0x01e, "Speaker" },
	{ NULL, (0x08 << 16) + 0x01f, "Head Set" },
	{ NULL, (0x08 << 16) + 0x020, "Hold" },
	{ NULL, (0x08 << 16) + 0x021, "Microphone" },
	{ NULL, (0x08 << 16) + 0x022, "Coverage" },
	{ NULL, (0x08 << 16) + 0x023, "Night Mode" },
	{ NULL, (0x08 << 16) + 0x024, "Send Calls" },
	{ NULL, (0x08 << 16) + 0x025, "Call Pickup" },
	{ NULL, (0x08 << 16) + 0x026, "Conference" },
	{ NULL, (0x08 << 16) + 0x027, "Stand-by" },
	{ NULL, (0x08 << 16) + 0x028, "Camera On" },
	{ NULL, (0x08 << 16) + 0x029, "Camera Off" },
	{ NULL, (0x08 << 16) + 0x02a, "On-Line" },
	{ NULL, (0x08 << 16) + 0x02b, "Off-Line" },
	{ NULL, (0x08 << 16) + 0x02c, "Busy" },
	{ NULL, (0x08 << 16) + 0x02d, "Ready" },
	{ NULL, (0x08 << 16) + 0x02e, "Paper-Out" },
	{ NULL, (0x08 << 16) + 0x02f, "Paper-Jam" },
	{ NULL, (0x08 << 16) + 0x030, "Remote" },
	{ NULL, (0x08 << 16) + 0x031, "Forward" },
	{ NULL, (0x08 << 16) + 0x032, "Reverse" },
	{ NULL, (0x08 << 16) + 0x033, "Stop" },
	{ NULL, (0x08 << 16) + 0x034, "Rewind" },
	{ NULL, (0x08 << 16) + 0x035, "Fast Forward" },
	{ NULL, (0x08 << 16) + 0x036, "Play" },
	{ NULL, (0x08 << 16) + 0x037, "Pause" },
	{ NULL, (0x08 << 16) + 0x038, "Record" },
	{ NULL, (0x08 << 16) + 0x039, "Error" },
	{ NULL, (0x08 << 16) + 0x03a, "Usage Selected Indicator" },
	{ NULL, (0x08 << 16) + 0x03b, "Usage In Use Indicator" },
	{ NULL, (0x08 << 16) + 0x03c, "Usage Multi Indicator" },
	{ NULL, (0x08 << 16) + 0x03d, "Indicator On" },
	{ NULL, (0x08 << 16) + 0x03e, "Indicator Flash" },
	{ NULL, (0x08 << 16) + 0x03f, "Indicator Slow Blink" },
	{ NULL, (0x08 << 16) + 0x040, "Indicator Fast Blink" },
	{ NULL, (0x08 << 16) + 0x041, "Indicator Off" },
	{ NULL, (0x08 << 16) + 0x042, "Flash On Time" },
	{ NULL, (0x08 << 16) + 0x043, "Slow Blink On Time" },
	{ NULL, (0x08 << 16) + 0x044, "Slow Blink Off Time" },
	{ NULL, (0x08 << 16) + 0x045, "Fast Blink On Time" },
	{ NULL, (0x08 << 16) + 0x046, "Fast Blink Off Time" },
	{ NULL, (0x08 << 16) + 0x047, "Usage Color Indicator" },
	{ NULL, (0x08 << 16) + 0x048, "Indicator Red" },
	{ NULL, (0x08 << 16) + 0x049, "Indicator Green" },
	{ NULL, (0x08 << 16) + 0x04a, "Indicator Amber" },
	{ NULL, (0x08 << 16) + 0x04b, "Generic Indicator" },
	{ NULL, (0x08 << 16) + 0x04c, "System Suspend" },
	{ NULL, (0x08 << 16) + 0x04d, "External Power Connected" },
	{ NULL, (0x09 << 16) + 0x000, "No Button Pressed" },
	{ NULL, (0x09 << 16) + 0x001, "Button 1 (Primary)" },
	{ NULL, (0x09 << 16) + 0x002, "Button 2 (Secondary)" },
	{ NULL, (0x09 << 16) + 0x003, "Button 3 (Tertiary)" },
	{ NULL, (0x09 << 16) + 0x004, "Button 4" },
	{ NULL, (0x09 << 16) + 0x005, "Button 5" },
	{ NULL, (0x0a << 16) + 0x001, "Instance 1" },
	{ NULL, (0x0a << 16) + 0x002, "Instance 2" },
	{ NULL, (0x0a << 16) + 0x003, "Instance 3" },
	{ NULL, (0x0b << 16) + 0x000, "Unassigned" },
	{ NULL, (0x0b << 16) + 0x001, "Phone" },
	{ NULL, (0x0b << 16) + 0x002, "Answering Machine" },
	{ NULL, (0x0b << 16) + 0x003, "Message Controls" },
	{ NULL, (0x0b << 16) + 0x004, "Handset" },
	{ NULL, (0x0b << 16) + 0x005, "Headset" },
	{ NULL, (0x0b << 16) + 0x006, "Telephony Key Pad" },
	{ NULL, (0x0b << 16) + 0x007, "Programmable Button" },
	{ NULL, (0x0b << 16) + 0x020, "Hook Switch" },
	{ NULL, (0x0b << 16) + 0x021, "Flash" },
	{ NULL, (0x0b << 16) + 0x022, "Feature" },
	{ NULL, (0x0b << 16) + 0x023, "Hold" },
	{ NULL, (0x0b << 16) + 0x024, "Redial" },
	{ NULL, (0x0b << 16) + 0x025, "Transfer" },
	{ NULL, (0x0b << 16) + 0x026, "Drop" },
	{ NULL, (0x0b << 16) + 0x027, "Park" },
	{ NULL, (0x0b << 16) + 0x028, "Forward Calls" },
	{ NULL, (0x0b << 16) + 0x029, "Alternate Function" },
	{ NULL, (0x0b << 16) + 0x02a, "Line" },
	{ NULL, (0x0b << 16) + 0x02b, "Speaker Phone" },
	{ NULL, (0x0b << 16) + 0x02c, "Conference" },
	{ NULL, (0x0b << 16) + 0x02d, "Ring Enable" },
	{ NULL, (0x0b << 16) + 0x02e, "Ring Select" },
	{ NULL, (0x0b << 16) + 0x02f, "Phone Mute" },
	{ NULL, (0x0b << 16) + 0x030, "Caller ID" },
	{ NULL, (0x0b << 16) + 0x050, "Speed Dial" },
	{ NULL, (0x0b << 16) + 0x051, "Store Number" },
	{ NULL, (0x0b << 16) + 0x052, "Recall Number" },
	{ NULL, (0x0b << 16) + 0x053, "Phone Directory" },
	{ NULL, (0x0b << 16) + 0x070, "Voice Mail" },
	{ NULL, (0x0b << 16) + 0x071, "Screen Calls" },
	{ NULL, (0x0b << 16) + 0x072, "Do Not Disturb" },
	{ NULL, (0x0b << 16) + 0x073, "Message" },
	{ NULL, (0x0b << 16) + 0x074, "Answer On/Offf" },
	{ NULL, (0x0b << 16) + 0x090, "Inside Dial Tone" },
	{ NULL, (0x0b << 16) + 0x091, "Outside Dial Tone" },
	{ NULL, (0x0b << 16) + 0x092, "Inside Ring Tone" },
	{ NULL, (0x0b << 16) + 0x093, "Outside Ring Tone" },
	{ NULL, (0x0b << 16) + 0x094, "Priority Ring Tone" },
	{ NULL, (0x0b << 16) + 0x095, "Inside Ringback" },
	{ NULL, (0x0b << 16) + 0x096, "Priority Ringback" },
	{ NULL, (0x0b << 16) + 0x097, "Line Busy Tone" },
	{ NULL, (0x0b << 16) + 0x098, "Recorder Tone" },
	{ NULL, (0x0b << 16) + 0x099, "Call Waiting Tone" },
	{ NULL, (0x0b << 16) + 0x09a, "Confirmation Tone 1" },
	{ NULL, (0x0b << 16) + 0x09b, "Confirmation Tone 2" },
	{ NULL, (0x0b << 16) + 0x09c, "Tones Off" },
	{ NULL, (0x0b << 16) + 0x09d, "Outside Ringback" },
	{ NULL, (0x0b << 16) + 0x0b0, "Key 1" },
	{ NULL, (0x0b << 16) + 0x0b1, "Key 2" },
	{ NULL, (0x0b << 16) + 0x0b3, "Key 3" },
	{ NULL, (0x0b << 16) + 0x0b4, "Key 4" },
	{ NULL, (0x0b << 16) + 0x0b5, "Key 5" },
	{ NULL, (0x0b << 16) + 0x0b6, "Key 6" },
	{ NULL, (0x0b << 16) + 0x0b7, "Key 7" },
	{ NULL, (0x0b << 16) + 0x0b8, "Key 8" },
	{ NULL, (0x0b << 16) + 0x0b9, "Key 9" },
	{ NULL, (0x0b << 16) + 0x0ba, "Key Star" },
	{ NULL, (0x0b << 16) + 0x0bb, "Key Pound" },
	{ NULL, (0x0b << 16) + 0x0bc, "Key A" },
	{ NULL, (0x0b << 16) + 0x0bd, "Key B" },
	{ NULL, (0x0b << 16) + 0x0be, "Key C" },
	{ NULL, (0x0b << 16) + 0x0bf, "Key D" },
	{ NULL, (0x0c << 16) + 0x000, "Unassigned" },
	{ NULL, (0x0c << 16) + 0x001, "Consumer Control" },
	{ NULL, (0x0c << 16) + 0x002, "Numeric Key Pad" },
	{ NULL, (0x0c << 16) + 0x003, "Programmable Buttons" },
	{ NULL, (0x0c << 16) + 0x020, "+10" },
	{ NULL, (0x0c << 16) + 0x021, "+100" },
	{ NULL, (0x0c << 16) + 0x022, "AM/PM" },
	{ NULL, (0x0c << 16) + 0x030, "Power" },
	{ NULL, (0x0c << 16) + 0x031, "Reset" },
	{ NULL, (0x0c << 16) + 0x032, "Sleep" },
	{ NULL, (0x0c << 16) + 0x033, "Sleep After" },
	{ NULL, (0x0c << 16) + 0x034, "Sleep Mode" },
	{ NULL, (0x0c << 16) + 0x035, "Illumination" },
	{ NULL, (0x0c << 16) + 0x036, "Function Buttons" },
	{ NULL, (0x0c << 16) + 0x040, "Menu" },
	{ NULL, (0x0c << 16) + 0x041, "Menu Pick" },
	{ NULL, (0x0c << 16) + 0x042, "Menu Up" },
	{ NULL, (0x0c << 16) + 0x043, "Menu Down" },
	{ NULL, (0x0c << 16) + 0x044, "Menu Left" },
	{ NULL, (0x0c << 16) + 0x045, "Menu Right" },
	{ NULL, (0x0c << 16) + 0x046, "Menu Escape" },
	{ NULL, (0x0c << 16) + 0x047, "Menu Value Increase" },
	{ NULL, (0x0c << 16) + 0x048, "Menu Value Decrease" },
	{ NULL, (0x0c << 16) + 0x060, "Data on Screen" },
	{ NULL, (0x0c << 16) + 0x061, "Closed Caption" },
	{ NULL, (0x0c << 16) + 0x062, "Closed Caption Select" },
	{ NULL, (0x0c << 16) + 0x063, "VCR/TV" },
	{ NULL, (0x0c << 16) + 0x064, "Broadcast Mode" },
	{ NULL, (0x0c << 16) + 0x065, "Snapshot" },
	{ NULL, (0x0c << 16) + 0x066, "Still" },
	{ NULL, (0x0c << 16) + 0x080, "Selection" },
	{ NULL, (0x0c << 16) + 0x081, "Assign Selection" },
	{ NULL, (0x0c << 16) + 0x082, "Mode Step" },
	{ NULL, (0x0c << 16) + 0x083, "Recall Last" },
	{ NULL, (0x0c << 16) + 0x084, "Enter Channel" },
	{ NULL, (0x0c << 16) + 0x085, "Order Movie" },
	{ NULL, (0x0c << 16) + 0x086, "Channel" },
	{ NULL, (0x0c << 16) + 0x087, "Media Selection" },
	{ NULL, (0x0c << 16) + 0x088, "Media Select Computer" },
	{ NULL, (0x0c << 16) + 0x089, "Media Select TV" },
	{ NULL, (0x0c << 16) + 0x08a, "Media Select WWW" },
	{ NULL, (0x0c << 16) + 0x08b, "Media Select DVD" },
	{ NULL, (0x0c << 16) + 0x08c, "Media Select Telephone" },
	{ NULL, (0x0c << 16) + 0x08d, "Media Select Program Guide" },
	{ NULL, (0x0c << 16) + 0x08e, "Media Select Video Phone" },
	{ NULL, (0x0c << 16) + 0x08f, "Media Select Games" },
	{ NULL, (0x0c << 16) + 0x090, "Media Select Messages" },
	{ NULL, (0x0c << 16) + 0x091, "Media Select CD" },
	{ NULL, (0x0c << 16) + 0x092, "Media Select VCR" },
	{ NULL, (0x0c << 16) + 0x093, "Media Select Tuner" },
	{ NULL, (0x0c << 16) + 0x094, "Quit" },
	{ NULL, (0x0c << 16) + 0x095, "Help" },
	{ NULL, (0x0c << 16) + 0x096, "Media Select Tape" },
	{ NULL, (0x0c << 16) + 0x097, "Media Select Cable" },
	{ NULL, (0x0c << 16) + 0x098, "Media Select Satellite" },
	{ NULL, (0x0c << 16) + 0x099, "Media Select Security" },
	{ NULL, (0x0c << 16) + 0x09a, "Media Select Home" },
	{ NULL, (0x0c << 16) + 0x09b, "Media Select Call" },
	{ NULL, (0x0c << 16) + 0x09c, "Channel Increment" },
	{ NULL, (0x0c << 16) + 0x09d, "Channel Decrement" },
	{ NULL, (0x0c << 16) + 0x09e, "Media Select SAP" },
	{ NULL, (0x0c << 16) + 0x0a0, "VCR Plus" },
	{ NULL, (0x0c << 16) + 0x0a1, "Once" },
	{ NULL, (0x0c << 16) + 0x0a2, "Daily" },
	{ NULL, (0x0c << 16) + 0x0a3, "Weekly" },
	{ NULL, (0x0c << 16) + 0x0a4, "Monthly" },
	{ NULL, (0x0c << 16) + 0x0b0, "Play" },
	{ NULL, (0x0c << 16) + 0x0b1, "Pause" },
	{ NULL, (0x0c << 16) + 0x0b2, "Record" },
	{ NULL, (0x0c << 16) + 0x0b3, "Fast Forward" },
	{ NULL, (0x0c << 16) + 0x0b4, "Rewind" },
	{ NULL, (0x0c << 16) + 0x0b5, "Scan Next Track" },
	{ NULL, (0x0c << 16) + 0x0b6, "Scan Previous Track" },
	{ NULL, (0x0c << 16) + 0x0b7, "Stop" },
	{ NULL, (0x0c << 16) + 0x0b8, "Eject" },
	{ NULL, (0x0c << 16) + 0x0b9, "Random Play" },
	{ NULL, (0x0c << 16) + 0x0ba, "Select Disc" },
	{ NULL, (0x0c << 16) + 0x0bb, "Enter Disc" },
	{ NULL, (0x0c << 16) + 0x0bc, "Repeat" },
	{ NULL, (0x0c << 16) + 0x0bd, "Tracking" },
	{ NULL, (0x0c << 16) + 0x0be, "Track Normal" },
	{ NULL, (0x0c << 16) + 0x0bf, "Slow Tracking" },
	{ NULL, (0x0c << 16) + 0x0c0, "Frame Forward" },
	{ NULL, (0x0c << 16) + 0x0c1, "Frame Back" },
	{ NULL, (0x0c << 16) + 0x0c2, "Mark" },
	{ NULL, (0x0c << 16) + 0x0c3, "Clear Mark" },
	{ NULL, (0x0c << 16) + 0x0c4, "Repeat from Mark" },
	{ NULL, (0x0c << 16) + 0x0c5, "Return to Mark" },
	{ NULL, (0x0c << 16) + 0x0c6, "Search Mark Forward" },
	{ NULL, (0x0c << 16) + 0x0c7, "Search Mark Backward" },
	{ NULL, (0x0c << 16) + 0x0c8, "Counter Reset" },
	{ NULL, (0x0c << 16) + 0x0c9, "Show Counter" },
	{ NULL, (0x0c << 16) + 0x0ca, "Tracking Increment" },
	{ NULL, (0x0c << 16) + 0x0cb, "Tracking Decrement" },
	{ NULL, (0x0c << 16) + 0x0cc, "Stop/Eject" },
	{ NULL, (0x0c << 16) + 0x0cd, "Play/Pause" },
	{ NULL, (0x0c << 16) + 0x0ce, "Play/Skip" },
	{ NULL, (0x0c << 16) + 0x0e0, "Volume" },
	{ NULL, (0x0c << 16) + 0x0e1, "Balance" },
	{ NULL, (0x0c << 16) + 0x0e2, "Mute" },
	{ NULL, (0x0c << 16) + 0x0e3, "Bass" },
	{ NULL, (0x0c << 16) + 0x0e4, "Treble" },
	{ NULL, (0x0c << 16) + 0x0e5, "Bass Boost" },
	{ NULL, (0x0c << 16) + 0x0e6, "Surround Mode" },
	{ NULL, (0x0c << 16) + 0x0e7, "Loudness" },
	{ NULL, (0x0c << 16) + 0x0e8, "MPX" },
	{ NULL, (0x0c << 16) + 0x0e9, "Volume Increment" },
	{ NULL, (0x0c << 16) + 0x0ea, "Volume Decrement" },
	{ NULL, (0x0c << 16) + 0x0f0, "Speed Select" },
	{ NULL, (0x0c << 16) + 0x0f1, "Playback Speed" },
	{ NULL, (0x0c << 16) + 0x0f2, "Standard Play" },
	{ NULL, (0x0c << 16) + 0x0f3, "Long Play" },
	{ NULL, (0x0c << 16) + 0x0f4, "Extended Play" },
	{ NULL, (0x0c << 16) + 0x0f5, "Slow" },
	{ NULL, (0x0c << 16) + 0x100, "Fan Enable" },
	{ NULL, (0x0c << 16) + 0x101, "Fan Speed" },
	{ NULL, (0x0c << 16) + 0x102, "Light Enable" },
	{ NULL, (0x0c << 16) + 0x103, "Light Illumination Level" },
	{ NULL, (0x0c << 16) + 0x104, "Climate Control Enable" },
	{ NULL, (0x0c << 16) + 0x105, "Room Temperature" },
	{ NULL, (0x0c << 16) + 0x106, "Security Enable" },
	{ NULL, (0x0c << 16) + 0x107, "Fire Alarm" },
	{ NULL, (0x0c << 16) + 0x108, "Police Alarm" },
	{ NULL, (0x0c << 16) + 0x150, "Balance Right" },
	{ NULL, (0x0c << 16) + 0x151, "Balance Left" },
	{ NULL, (0x0c << 16) + 0x152, "Bass Increment" },
	{ NULL, (0x0c << 16) + 0x153, "Bass Decrement" },
	{ NULL, (0x0c << 16) + 0x154, "Treble Increment" },
	{ NULL, (0x0c << 16) + 0x155, "Treble Decrement" },
	{ NULL, (0x0c << 16) + 0x160, "Speaker System" },
	{ NULL, (0x0c << 16) + 0x161, "Channel Left" },
	{ NULL, (0x0c << 16) + 0x162, "Channel Right" },
	{ NULL, (0x0c << 16) + 0x163, "Channel Center" },
	{ NULL, (0x0c << 16) + 0x164, "Channel Front" },
	{ NULL, (0x0c << 16) + 0x165, "Channel Center Front" },
	{ NULL, (0x0c << 16) + 0x166, "Channel Side" },
	{ NULL, (0x0c << 16) + 0x167, "Channel Surround" },
	{ NULL, (0x0c << 16) + 0x168, "Channel Low Frequency Enhancement" },
	{ NULL, (0x0c << 16) + 0x169, "Channel Top" },
	{ NULL, (0x0c << 16) + 0x16a, "Channel Unknown" },
	{ NULL, (0x0c << 16) + 0x170, "Sub-Channel" },
	{ NULL, (0x0c << 16) + 0x171, "Sub-Channel Increment" },
	{ NULL, (0x0c << 16) + 0x172, "Sub-Channel Decrement" },
	{ NULL, (0x0c << 16) + 0x173, "Alternative Audio Increment" },
	{ NULL, (0x0c << 16) + 0x174, "Alternative Audio Decrement" },
	{ NULL, (0x0c << 16) + 0x180, "Application Launch Buttons" },
	{ NULL, (0x0c << 16) + 0x181, "AL Launch Button Configuration Tool" },
	{ NULL, (0x0c << 16) + 0x182, "AL Launch Button Configuration" },
	{ NULL, (0x0c << 16) + 0x183, "AL Consumer Control Configuration" },
	{ NULL, (0x0c << 16) + 0x184, "AL Word Processor" },
	{ NULL, (0x0c << 16) + 0x185, "AL Text Editor" },
	{ NULL, (0x0c << 16) + 0x186, "AL Spreadsheet" },
	{ NULL, (0x0c << 16) + 0x187, "AL Graphics Editor" },
	{ NULL, (0x0c << 16) + 0x188, "AL Presentation App" },
	{ NULL, (0x0c << 16) + 0x189, "AL Database App" },
	{ NULL, (0x0c << 16) + 0x18a, "AL Email Reader" },
	{ NULL, (0x0c << 16) + 0x18b, "AL Newsreader" },
	{ NULL, (0x0c << 16) + 0x18c, "AL Voicemail" },
	{ NULL, (0x0c << 16) + 0x18d, "AL Contacts/Address Book" },
	{ NULL, (0x0c << 16) + 0x18e, "AL Calendar/Schedule" },
	{ NULL, (0x0c << 16) + 0x18f, "AL Task/Project Manager" },
	{ NULL, (0x0c << 16) + 0x190, "AL Log/Jounal/Timecard" },
	{ NULL, (0x0c << 16) + 0x191, "AL Checkbook/Finance" },
	{ NULL, (0x0c << 16) + 0x192, "AL Calculator" },
	{ NULL, (0x0c << 16) + 0x193, "AL A/V Capture/Playback" },
	{ NULL, (0x0c << 16) + 0x194, "AL Local Machine Browser" },
	{ NULL, (0x0c << 16) + 0x195, "AL LAN/Wan Browser" },
	{ NULL, (0x0c << 16) + 0x196, "AL Internet Browser" },
	{ NULL, (0x0c << 16) + 0x197, "AL Remote Networking/ISP Connect" },
	{ NULL, (0x0c << 16) + 0x198, "AL Network Conference" },
	{ NULL, (0x0c << 16) + 0x199, "AL Network Chat" },
	{ NULL, (0x0c << 16) + 0x19a, "AL Telephony/Dialer" },
	{ NULL, (0x0c << 16) + 0x19b, "AL Logon" },
	{ NULL, (0x0c << 16) + 0x19c, "AL Logoff" },
	{ NULL, (0x0c << 16) + 0x19d, "AL Logon/Logoff" },
	{ NULL, (0x0c << 16) + 0x19e, "AL Terminal Local/Screensaver" },
	{ NULL, (0x0c << 16) + 0x19f, "AL Control Panel" },
	{ NULL, (0x0c << 16) + 0x1a0, "AL Command Line Processor/Run" },
	{ NULL, (0x0c << 16) + 0x1a1, "AL Process/Task Manager" },
	{ NULL, (0x0c << 16) + 0x1a2, "AL Select Task/Application" },
	{ NULL, (0x0c << 16) + 0x1a3, "AL Next Task/Application" },
	{ NULL, (0x0c << 16) + 0x1a4, "AL Previous Task/Application" },
	{ NULL, (0x0c << 16) + 0x1a5, "AL Preemptive Halt Task/Application" },
	{ NULL, (0x0c << 16) + 0x200, "Generic GUI Application Controls" },
	{ NULL, (0x0c << 16) + 0x201, "AC New" },
	{ NULL, (0x0c << 16) + 0x202, "AC Open" },
	{ NULL, (0x0c << 16) + 0x203, "AC CLose" },
	{ NULL, (0x0c << 16) + 0x204, "AC Exit" },
	{ NULL, (0x0c << 16) + 0x205, "AC Maximize" },
	{ NULL, (0x0c << 16) + 0x206, "AC Minimize" },
	{ NULL, (0x0c << 16) + 0x207, "AC Save" },
	{ NULL, (0x0c << 16) + 0x208, "AC Print" },
	{ NULL, (0x0c << 16) + 0x209, "AC Properties" },
	{ NULL, (0x0c << 16) + 0x21a, "AC Undo" },
	{ NULL, (0x0c << 16) + 0x21b, "AC Copy" },
	{ NULL, (0x0c << 16) + 0x21c, "AC Cut" },
	{ NULL, (0x0c << 16) + 0x21d, "AC Paste" },
	{ NULL, (0x0c << 16) + 0x21e, "AC Select All" },
	{ NULL, (0x0c << 16) + 0x21f, "AC Find" },
	{ NULL, (0x0c << 16) + 0x220, "AC Find and Replace" },
	{ NULL, (0x0c << 16) + 0x221, "AC Search" },
	{ NULL, (0x0c << 16) + 0x222, "AC Go To" },
	{ NULL, (0x0c << 16) + 0x223, "AC Home" },
	{ NULL, (0x0c << 16) + 0x224, "AC Back" },
	{ NULL, (0x0c << 16) + 0x225, "AC Forward" },
	{ NULL, (0x0c << 16) + 0x226, "AC Stop" },
	{ NULL, (0x0c << 16) + 0x227, "AC Refresh" },
	{ NULL, (0x0c << 16) + 0x228, "AC Previous Link" },
	{ NULL, (0x0c << 16) + 0x229, "AC Next Link" },
	{ NULL, (0x0c << 16) + 0x22b, "AC History" },
	{ NULL, (0x0c << 16) + 0x22c, "AC Subscriptions" },
	{ NULL, (0x0c << 16) + 0x22d, "AC Zoom In" },
	{ NULL, (0x0c << 16) + 0x22e, "AC Zoom Out" },
	{ NULL, (0x0c << 16) + 0x22f, "AC Zoom" },
	{ NULL, (0x0c << 16) + 0x230, "AC Full Screen View" },
	{ NULL, (0x0c << 16) + 0x231, "AC Normal View" },
	{ NULL, (0x0c << 16) + 0x232, "AC View Toggle" },
	{ NULL, (0x0c << 16) + 0x233, "AC Scroll Up" },
	{ NULL, (0x0c << 16) + 0x234, "AC Scroll Down" },
	{ NULL, (0x0c << 16) + 0x235, "AC Scroll" },
	{ NULL, (0x0c << 16) + 0x236, "AC Pan Left" },
	{ NULL, (0x0c << 16) + 0x237, "AC Pan Right" },
	{ NULL, (0x0c << 16) + 0x238, "AC Pan" },
	{ NULL, (0x0c << 16) + 0x239, "AC New Window" },
	{ NULL, (0x0c << 16) + 0x23a, "AC Tile Horizontally" },
	{ NULL, (0x0c << 16) + 0x23b, "AC Tile Vertically" },
	{ NULL, (0x0c << 16) + 0x23c, "AC Format" },
	{ NULL, (0x0d << 16) + 0x000, "Undefined" },
	{ NULL, (0x0d << 16) + 0x001, "Digitizer" },
	{ NULL, (0x0d << 16) + 0x002, "Pen" },
	{ NULL, (0x0d << 16) + 0x003, "Light Pen" },
	{ NULL, (0x0d << 16) + 0x004, "Touch Screen" },
	{ NULL, (0x0d << 16) + 0x005, "Touch Pad" },
	{ NULL, (0x0d << 16) + 0x006, "White Board" },
	{ NULL, (0x0d << 16) + 0x007, "Coordinate Measuring Machine" },
	{ NULL, (0x0d << 16) + 0x008, "3D Digitizer" },
	{ NULL, (0x0d << 16) + 0x009, "Stereo Plotter" },
	{ NULL, (0x0d << 16) + 0x00a, "Articulated Arm" },
	{ NULL, (0x0d << 16) + 0x00b, "Armature" },
	{ NULL, (0x0d << 16) + 0x00c, "Multiple Point Digitizer" },
	{ NULL, (0x0d << 16) + 0x00d, "Free Space Wand" },
	{ NULL, (0x0d << 16) + 0x020, "Stylus" },
	{ NULL, (0x0d << 16) + 0x021, "Puck" },
	{ NULL, (0x0d << 16) + 0x022, "Finger" },
	{ NULL, (0x0d << 16) + 0x030, "Tip Pressure" },
	{ NULL, (0x0d << 16) + 0x031, "Barrel Pressure" },
	{ NULL, (0x0d << 16) + 0x032, "In Range" },
	{ NULL, (0x0d << 16) + 0x033, "Touch" },
	{ NULL, (0x0d << 16) + 0x034, "Untouch" },
	{ NULL, (0x0d << 16) + 0x035, "Tap" },
	{ NULL, (0x0d << 16) + 0x036, "Quality" },
	{ NULL, (0x0d << 16) + 0x037, "Data Valid" },
	{ NULL, (0x0d << 16) + 0x038, "Transducer Index" },
	{ NULL, (0x0d << 16) + 0x039, "Tablet Function Keys" },
	{ NULL, (0x0d << 16) + 0x03a, "Program Change Keys" },
	{ NULL, (0x0d << 16) + 0x03b, "Battery Strength" },
	{ NULL, (0x0d << 16) + 0x03c, "Invert" },
	{ NULL, (0x0d << 16) + 0x03d, "X Tilt" },
	{ NULL, (0x0d << 16) + 0x03e, "Y Tilt" },
	{ NULL, (0x0d << 16) + 0x03f, "Azimuth" },
	{ NULL, (0x0d << 16) + 0x040, "Altitude" },
	{ NULL, (0x0d << 16) + 0x041, "Twist" },
	{ NULL, (0x0d << 16) + 0x042, "Tip Switch" },
	{ NULL, (0x0d << 16) + 0x043, "Secondary Tip Switch" },
	{ NULL, (0x0d << 16) + 0x044, "Barrel Switch" },
	{ NULL, (0x0d << 16) + 0x045, "Eraser" },
	{ NULL, (0x0d << 16) + 0x046, "Tablet Pick" },
	{ NULL, (0x0d << 16) + 0x047, "Confidence" },
	{ NULL, (0x0d << 16) + 0x048, "Width" },
	{ NULL, (0x0d << 16) + 0x049, "Height" },
	{ NULL, (0x0d << 16) + 0x051, "Contact ID" },
	{ NULL, (0x0d << 16) + 0x052, "Input Mode" },
	{ NULL, (0x0d << 16) + 0x053, "Device Index" },
	{ NULL, (0x0d << 16) + 0x054, "Contact Count" },
	{ NULL, (0x0d << 16) + 0x055, "Maximum Contact Number" },
	{ NULL, (0x0f << 16) + 0x000, "Undefined" },
	{ NULL, (0x0f << 16) + 0x001, "Physical Interface Device" },
	{ NULL, (0x0f << 16) + 0x020, "Normal" },
	{ NULL, (0x0f << 16) + 0x021, "Set Effect Report" },
	{ NULL, (0x0f << 16) + 0x022, "Effect Block Index" },
	{ NULL, (0x0f << 16) + 0x023, "Parameter Block Offset" },
	{ NULL, (0x0f << 16) + 0x024, "ROM Flag" },
	{ NULL, (0x0f << 16) + 0x025, "Effect Type" },
	{ NULL, (0x0f << 16) + 0x026, "ET Constant Force" },
	{ NULL, (0x0f << 16) + 0x027, "ET Ramp" },
	{ NULL, (0x0f << 16) + 0x028, "ET Custom Force Data" },
	{ NULL, (0x0f << 16) + 0x030, "ET Square" },
	{ NULL, (0x0f << 16) + 0x031, "ET Sine" },
	{ NULL, (0x0f << 16) + 0x032, "ET Triangle" },
	{ NULL, (0x0f << 16) + 0x033, "ET Sawtooth Up" },
	{ NULL, (0x0f << 16) + 0x034, "ET Sawtooth Down" },
	{ NULL, (0x0f << 16) + 0x040, "ET Spring" },
	{ NULL, (0x0f << 16) + 0x041, "ET Damper" },
	{ NULL, (0x0f << 16) + 0x042, "ET Inertia" },
	{ NULL, (0x0f << 16) + 0x043, "ET Friction" },
	{ NULL, (0x0f << 16) + 0x050, "Duration" },
	{ NULL, (0x0f << 16) + 0x051, "Sample Period" },
	{ NULL, (0x0f << 16) + 0x052, "Gain" },
	{ NULL, (0x0f << 16) + 0x053, "Trigger Button" },
	{ NULL, (0x0f << 16) + 0x054, "Trigger Repeat Interval" },
	{ NULL, (0x0f << 16) + 0x055, "Axes Enable" },
	{ NULL, (0x0f << 16) + 0x056, "Direction Enable" },
	{ NULL, (0x0f << 16) + 0x057, "Direction" },
	{ NULL, (0x0f << 16) + 0x058, "Type Specific Block Offset" },
	{ NULL, (0x0f << 16) + 0x059, "Block Type" },
	{ NULL, (0x0f << 16) + 0x05A, "Set Envelope Report" },
	{ NULL, (0x0f << 16) + 0x05B, "Attack Level" },
	{ NULL, (0x0f << 16) + 0x05C, "Attack Time" },
	{ NULL, (0x0f << 16) + 0x05D, "Fade Level" },
	{ NULL, (0x0f << 16) + 0x05E, "Fade Time" },
	{ NULL, (0x0f << 16) + 0x05F, "Set Condition Report" },
	{ NULL, (0x0f << 16) + 0x060, "CP Offset" },
	{ NULL, (0x0f << 16) + 0x061, "Positive Coefficient" },
	{ NULL, (0x0f << 16) + 0x062, "Negative Coefficient" },
	{ NULL, (0x0f << 16) + 0x063, "Positive Saturation" },
	{ NULL, (0x0f << 16) + 0x064, "Negative Saturation" },
	{ NULL, (0x0f << 16) + 0x065, "Dead Band" },
	{ NULL, (0x0f << 16) + 0x066, "Download Force Sample" },
	{ NULL, (0x0f << 16) + 0x067, "Isoch Custom Force Enable" },
	{ NULL, (0x0f << 16) + 0x068, "Custom Force Data Report" },
	{ NULL, (0x0f << 16) + 0x069, "Custom Force Data" },
	{ NULL, (0x0f << 16) + 0x06A, "Custom Force Vendor Defined Data" },
	{ NULL, (0x0f << 16) + 0x06B, "Set Custom Force Report" },
	{ NULL, (0x0f << 16) + 0x06C, "Custom Force Data Offset" },
	{ NULL, (0x0f << 16) + 0x06D, "Sample Count" },
	{ NULL, (0x0f << 16) + 0x06E, "Set Periodic Report" },
	{ NULL, (0x0f << 16) + 0x06F, "Offset" },
	{ NULL, (0x0f << 16) + 0x070, "Magnitude" },
	{ NULL, (0x0f << 16) + 0x071, "Phase" },
	{ NULL, (0x0f << 16) + 0x072, "Period" },
	{ NULL, (0x0f << 16) + 0x073, "Set Constant Force Report" },
	{ NULL, (0x0f << 16) + 0x074, "Set Ramp Force Report" },
	{ NULL, (0x0f << 16) + 0x075, "Ramp Start" },
	{ NULL, (0x0f << 16) + 0x076, "Ramp End" },
	{ NULL, (0x0f << 16) + 0x077, "Effect Operation Report" },
	{ NULL, (0x0f << 16) + 0x078, "Effect Operation" },
	{ NULL, (0x0f << 16) + 0x079, "Op Effect Start" },
	{ NULL, (0x0f << 16) + 0x07A, "Op Effect Start Solo" },
	{ NULL, (0x0f << 16) + 0x07B, "Op Effect Stop" },
	{ NULL, (0x0f << 16) + 0x07C, "Loop Count" },
	{ NULL, (0x0f << 16) + 0x07D, "Device Gain Report" },
	{ NULL, (0x0f << 16) + 0x07E, "Device Gain" },
	{ NULL, (0x0f << 16) + 0x07F, "PID Pool Report" },
	{ NULL, (0x0f << 16) + 0x080, "RAM Pool Size" },
	{ NULL, (0x0f << 16) + 0x081, "ROM Pool Size" },
	{ NULL, (0x0f << 16) + 0x082, "ROM Effect Block Count" },
	{ NULL, (0x0f << 16) + 0x083, "Simultaneous Effects Max" },
	{ NULL, (0x0f << 16) + 0x084, "Pool Alignment" },
	{ NULL, (0x0f << 16) + 0x085, "PID Pool Move Report" },
	{ NULL, (0x0f << 16) + 0x086, "Move Source" },
	{ NULL, (0x0f << 16) + 0x087, "Move Destination" },
	{ NULL, (0x0f << 16) + 0x088, "Move Length" },
	{ NULL, (0x0f << 16) + 0x089, "PID Block Load Report" },
	{ NULL, (0x0f << 16) + 0x08B, "Block Load Status" },
	{ NULL, (0x0f << 16) + 0x08C, "Block Load Success" },
	{ NULL, (0x0f << 16) + 0x08D, "Block Load Full" },
	{ NULL, (0x0f << 16) + 0x08E, "Block Load Error" },
	{ NULL, (0x0f << 16) + 0x08F, "Block Handle" },
	{ NULL, (0x0f << 16) + 0x090, "PID Block Free Report" },
	{ NULL, (0x0f << 16) + 0x091, "Type Specific Block Handle" },
	{ NULL, (0x0f << 16) + 0x092, "PID State Report" },
	{ NULL, (0x0f << 16) + 0x094, "Effect Playing" },
	{ NULL, (0x0f << 16) + 0x095, "PID Device Control Report" },
	{ NULL, (0x0f << 16) + 0x096, "PID Device Control" },
	{ NULL, (0x0f << 16) + 0x097, "DC Enable Actuators" },
	{ NULL, (0x0f << 16) + 0x098, "DC Disable Actuators" },
	{ NULL, (0x0f << 16) + 0x099, "DC Stop All Effects" },
	{ NULL, (0x0f << 16) + 0x09A, "DC Device Reset" },
	{ NULL, (0x0f << 16) + 0x09B, "DC Device Pause" },
	{ NULL, (0x0f << 16) + 0x09C, "DC Device Continue" },
	{ NULL, (0x0f << 16) + 0x09F, "Device Paused" },
	{ NULL, (0x0f << 16) + 0x0A0, "Actuators Enabled" },
	{ NULL, (0x0f << 16) + 0x0A4, "Safety Switch" },
	{ NULL, (0x0f << 16) + 0x0A5, "Actuator Override Switch" },
	{ NULL, (0x0f << 16) + 0x0A6, "Actuator Power" },
	{ NULL, (0x0f << 16) + 0x0A7, "Start Delay" },
	{ NULL, (0x0f << 16) + 0x0A8, "Parameter Block Size" },
	{ NULL, (0x0f << 16) + 0x0A9, "Device Managed Pool" },
	{ NULL, (0x0f << 16) + 0x0AA, "Shared Parameter Blocks" },
	{ NULL, (0x0f << 16) + 0x0AB, "Create New Effect Report" },
	{ NULL, (0x0f << 16) + 0x0AC, "RAM Pool Available" },
	{ NULL, (0x14 << 16) + 0x000, "Undefined" },
	{ NULL, (0x14 << 16) + 0x001, "Alphanumeric Display" },
	{ NULL, (0x14 << 16) + 0x020, "Display Attributes Report" },
	{ NULL, (0x14 << 16) + 0x021, "ASCII Character Set" },
	{ NULL, (0x14 << 16) + 0x022, "Data Read Back" },
	{ NULL, (0x14 << 16) + 0x023, "Font Read Back" },
	{ NULL, (0x14 << 16) + 0x024, "Display Control Report" },
	{ NULL, (0x14 << 16) + 0x025, "Clear Display" },
	{ NULL, (0x14 << 16) + 0x026, "Display Enable" },
	{ NULL, (0x14 << 16) + 0x027, "Screen Saver Delay" },
	{ NULL, (0x14 << 16) + 0x028, "Screen Saver Enable" },
	{ NULL, (0x14 << 16) + 0x029, "Vertical Scroll" },
	{ NULL, (0x14 << 16) + 0x02a, "Horizontal Scroll" },
	{ NULL, (0x14 << 16) + 0x02b, "Character Report" },
	{ NULL, (0x14 << 16) + 0x02c, "Display Data" },
	{ NULL, (0x14 << 16) + 0x02d, "Display Status" },
	{ NULL, (0x14 << 16) + 0x02e, "Stat Not Ready" },
	{ NULL, (0x14 << 16) + 0x02f, "Stat Ready" },
	{ NULL, (0x14 << 16) + 0x030, "Err Not a loadable Character" },
	{ NULL, (0x14 << 16) + 0x031, "Err Font Data Cannot Be Read" },
	{ NULL, (0x14 << 16) + 0x032, "Cursur Position Report" },
	{ NULL, (0x14 << 16) + 0x033, "Row" },
	{ NULL, (0x14 << 16) + 0x034, "Column" },
	{ NULL, (0x14 << 16) + 0x035, "Rows" },
	{ NULL, (0x14 << 16) + 0x036, "Columns" },
	{ NULL, (0x14 << 16) + 0x037, "Cursor Pixel Positioning" },
	{ NULL, (0x14 << 16) + 0x038, "Cursor Mode" },
	{ NULL, (0x14 << 16) + 0x039, "Cursor Enable" },
	{ NULL, (0x14 << 16) + 0x03a, "Cursor Blink" },
	{ NULL, (0x14 << 16) + 0x03b, "Font Report" },
	{ NULL, (0x14 << 16) + 0x03c, "Font Data" },
	{ NULL, (0x14 << 16) + 0x03d, "Character Width" },
	{ NULL, (0x14 << 16) + 0x03e, "Character Height" },
	{ NULL, (0x14 << 16) + 0x03f, "Character Spacing Horizontal" },
	{ NULL, (0x14 << 16) + 0x040, "Character Spacing Vertical" },
	{ NULL, (0x14 << 16) + 0x041, "Unicode Character Set" },
	{ NULL, (0x80 << 16) + 0x001, "Monitor Control" },
	{ NULL, (0x80 << 16) + 0x002, "EDID Information" },
	{ NULL, (0x80 << 16) + 0x003, "VDIF Information" },
	{ NULL, (0x80 << 16) + 0x004, "VESA Version" },
	{ NULL, (0x82 << 16) + 0x001, "Degauss" },
	{ NULL, (0x82 << 16) + 0x010, "Brightness" },
	{ NULL, (0x82 << 16) + 0x012, "Contrast" },
	{ NULL, (0x82 << 16) + 0x016, "Red Video Gain" },
	{ NULL, (0x82 << 16) + 0x018, "Green Video Gain" },
	{ NULL, (0x82 << 16) + 0x01a, "Blue Video Gain" },
	{ NULL, (0x82 << 16) + 0x01c, "Focus" },
	{ NULL, (0x82 << 16) + 0x020, "Horizontal Position" },
	{ NULL, (0x82 << 16) + 0x022, "Horizontal Size" },
	{ NULL, (0x82 << 16) + 0x024, "Horizontal Pincushion" },
	{ NULL, (0x82 << 16) + 0x026, "Horizontal Pincushion Balance" },
	{ NULL, (0x82 << 16) + 0x028, "Horizontal Misconvergence" },
	{ NULL, (0x82 << 16) + 0x02a, "Horizontal Linearity" },
	{ NULL, (0x82 << 16) + 0x02c, "Horizontal Linearity Balance" },
	{ NULL, (0x82 << 16) + 0x030, "Vertical Position" },
	{ NULL, (0x82 << 16) + 0x032, "Vertical Size" },
	{ NULL, (0x82 << 16) + 0x034, "Vertical Pincushion" },
	{ NULL, (0x82 << 16) + 0x036, "Vertical Pincushion Balance" },
	{ NULL, (0x82 << 16) + 0x038, "Vertical Misconvergence" },
	{ NULL, (0x82 << 16) + 0x03a, "Vertical Linearity" },
	{ NULL, (0x82 << 16) + 0x03c, "Vertical Linearity Balance" },
	{ NULL, (0x82 << 16) + 0x040, "Parallelogram Balance (Key Distortion)" },
	{ NULL, (0x82 << 16) + 0x042, "Trapezoidal Distortion (Key)" },
	{ NULL, (0x82 << 16) + 0x044, "Tilt (Rotation)" },
	{ NULL, (0x82 << 16) + 0x046, "Top Corner Distortion Control" },
	{ NULL, (0x82 << 16) + 0x048, "Top Corner Distortion Balance" },
	{ NULL, (0x82 << 16) + 0x04a, "Bottom Corner Distortion Control" },
	{ NULL, (0x82 << 16) + 0x04c, "Bottom Corner Distortion Balance" },
	{ NULL, (0x82 << 16) + 0x056, "Horizontal Moire" },
	{ NULL, (0x82 << 16) + 0x058, "Vertical Moire" },
	{ NULL, (0x82 << 16) + 0x05e, "Input Level Select" },
	{ NULL, (0x82 << 16) + 0x060, "Input Source Select" },
	{ NULL, (0x82 << 16) + 0x06c, "Red Video Black Level" },
	{ NULL, (0x82 << 16) + 0x06e, "Green Video Black Level" },
	{ NULL, (0x82 << 16) + 0x070, "Blue Video Black Level" },
	{ NULL, (0x82 << 16) + 0x0a2, "Auto Size Center" },
	{ NULL, (0x82 << 16) + 0x0a4, "Polarity Horizontal Sychronization" },
	{ NULL, (0x82 << 16) + 0x0a6, "Polarity Vertical Synchronization" },
	{ NULL, (0x82 << 16) + 0x0aa, "Screen Orientation" },
	{ NULL, (0x82 << 16) + 0x0ac, "Horizontal Frequency in Hz" },
	{ NULL, (0x82 << 16) + 0x0ae, "Vertical Frequency in 0.1 Hz" },
	{ NULL, (0x82 << 16) + 0x0b0, "Settings" },
	{ NULL, (0x82 << 16) + 0x0ca, "On Screen Display (OSD)" },
	{ NULL, (0x82 << 16) + 0x0d4, "Stereo Mode" },
	{ NULL, (0x84 << 16) + 0x000, "Undefined" },
	{ NULL, (0x84 << 16) + 0x001, "iName" },
	{ NULL, (0x84 << 16) + 0x002, "Present Status" },
	{ NULL, (0x84 << 16) + 0x003, "Changed Status" },
	{ NULL, (0x84 << 16) + 0x004, "UPS" },
	{ NULL, (0x84 << 16) + 0x005, "Power Supply" },
	{ NULL, (0x84 << 16) + 0x010, "Battery System" },
	{ NULL, (0x84 << 16) + 0x011, "Battery System ID" },
	{ NULL, (0x84 << 16) + 0x012, "Battery" },
	{ NULL, (0x84 << 16) + 0x013, "Battery ID" },
	{ NULL, (0x84 << 16) + 0x014, "Charger" },
	{ NULL, (0x84 << 16) + 0x015, "Charger ID" },
	{ NULL, (0x84 << 16) + 0x016, "Power Converter" },
	{ NULL, (0x84 << 16) + 0x017, "Power Converter ID" },
	{ NULL, (0x84 << 16) + 0x018, "Outlet System" },
	{ NULL, (0x84 << 16) + 0x019, "Outlet System ID" },
	{ NULL, (0x84 << 16) + 0x01a, "Input" },
	{ NULL, (0x84 << 16) + 0x01b, "Input ID" },
	{ NULL, (0x84 << 16) + 0x01c, "Output" },
	{ NULL, (0x84 << 16) + 0x01d, "Output ID" },
	{ NULL, (0x84 << 16) + 0x01e, "Flow" },
	{ NULL, (0x84 << 16) + 0x01f, "Flow ID" },
	{ NULL, (0x84 << 16) + 0x020, "Outlet" },
	{ NULL, (0x84 << 16) + 0x021, "Outlet ID" },
	{ NULL, (0x84 << 16) + 0x022, "Gang" },
	{ NULL, (0x84 << 16) + 0x023, "Gang ID" },
	{ NULL, (0x84 << 16) + 0x024, "Power Summary" },
	{ NULL, (0x84 << 16) + 0x025, "Power Summary ID" },
	{ NULL, (0x84 << 16) + 0x030, "Voltage" },
	{ NULL, (0x84 << 16) + 0x031, "Current" },
	{ NULL, (0x84 << 16) + 0x032, "Frequency" },
	{ NULL, (0x84 << 16) + 0x033, "Apparent Power" },
	{ NULL, (0x84 << 16) + 0x034, "Active Power" },
	{ NULL, (0x84 << 16) + 0x035, "Percent Load" },
	{ NULL, (0x84 << 16) + 0x036, "Temperature" },
	{ NULL, (0x84 << 16) + 0x037, "Humidity" },
	{ NULL, (0x84 << 16) + 0x038, "Bad Count" },
	{ NULL, (0x84 << 16) + 0x040, "Config Voltage" },
	{ NULL, (0x84 << 16) + 0x041, "Config Current" },
	{ NULL, (0x84 << 16) + 0x042, "Config Frequency" },
	{ NULL, (0x84 << 16) + 0x043, "Config Apparent Power" },
	{ NULL, (0x84 << 16) + 0x044, "Config Active Power" },
	{ NULL, (0x84 << 16) + 0x045, "Config Percent Load" },
	{ NULL, (0x84 << 16) + 0x046, "Config Temperature" },
	{ NULL, (0x84 << 16) + 0x047, "Config Humidity" },
	{ NULL, (0x84 << 16) + 0x050, "Switch On Control" },
	{ NULL, (0x84 << 16) + 0x051, "Switch Off Control" },
	{ NULL, (0x84 << 16) + 0x052, "Toggle Control" },
	{ NULL, (0x84 << 16) + 0x053, "Low Voltage Transfer" },
	{ NULL, (0x84 << 16) + 0x054, "High Voltage Transfer" },
	{ NULL, (0x84 << 16) + 0x055, "Delay Before Reboot" },
	{ NULL, (0x84 << 16) + 0x056, "Delay Before Startup" },
	{ NULL, (0x84 << 16) + 0x057, "Delay Before Shutdown" },
	{ NULL, (0x84 << 16) + 0x058, "Test" },
	{ NULL, (0x84 << 16) + 0x059, "Module Reset" },
	{ NULL, (0x84 << 16) + 0x05a, "Audible Alarm Control" },
	{ NULL, (0x84 << 16) + 0x060, "Present" },
	{ NULL, (0x84 << 16) + 0x061, "Good" },
	{ NULL, (0x84 << 16) + 0x062, "Internal Failure" },
	{ NULL, (0x84 << 16) + 0x063, "Voltage out of range" },
	{ NULL, (0x84 << 16) + 0x064, "Frequency out of range" },
	{ NULL, (0x84 << 16) + 0x065, "Overload" },
	{ NULL, (0x84 << 16) + 0x066, "Over Charged" },
	{ NULL, (0x84 << 16) + 0x067, "Over Temperature" },
	{ NULL, (0x84 << 16) + 0x068, "Shutdown Requested" },
	{ NULL, (0x84 << 16) + 0x069, "Shutdown  Imminent" },
	{ NULL, (0x84 << 16) + 0x06a, "Reserved" },
	{ NULL, (0x84 << 16) + 0x06b, "Switch On/Off" },
	{ NULL, (0x84 << 16) + 0x06c, "Switchable" },
	{ NULL, (0x84 << 16) + 0x06d, "Used" },
	{ NULL, (0x84 << 16) + 0x06e, "Boost" },
	{ NULL, (0x84 << 16) + 0x06f, "Buck" },
	{ NULL, (0x84 << 16) + 0x070, "Initialized" },
	{ NULL, (0x84 << 16) + 0x071, "Tested" },
	{ NULL, (0x84 << 16) + 0x072, "Awaiting Power" },
	{ NULL, (0x84 << 16) + 0x073, "Communication Lost" },
	{ NULL, (0x84 << 16) + 0x0fd, "iManufacturer" },
	{ NULL, (0x84 << 16) + 0x0fe, "iProduct" },
	{ NULL, (0x84 << 16) + 0x0ff, "iSerialNumber" },
	{ NULL, (0x85 << 16) + 0x000, "Undefined" },
	{ NULL, (0x85 << 16) + 0x001, "SMB Battery Mode" },
	{ NULL, (0x85 << 16) + 0x002, "SMB Battery Status" },
	{ NULL, (0x85 << 16) + 0x003, "SMB Alarm Warning" },
	{ NULL, (0x85 << 16) + 0x004, "SMB Charger Mode" },
	{ NULL, (0x85 << 16) + 0x005, "SMB Charger Status" },
	{ NULL, (0x85 << 16) + 0x006, "SMB Charger Spec Info" },
	{ NULL, (0x85 << 16) + 0x007, "SMB Selector State" },
	{ NULL, (0x85 << 16) + 0x008, "SMB Selector Presets" },
	{ NULL, (0x85 << 16) + 0x009, "SMB Selector Info" },
	{ NULL, (0x85 << 16) + 0x010, "Optional Mfg. Function 1" },
	{ NULL, (0x85 << 16) + 0x011, "Optional Mfg. Function 2" },
	{ NULL, (0x85 << 16) + 0x012, "Optional Mfg. Function 3" },
	{ NULL, (0x85 << 16) + 0x013, "Optional Mfg. Function 4" },
	{ NULL, (0x85 << 16) + 0x014, "Optional Mfg. Function 5" },
	{ NULL, (0x85 << 16) + 0x015, "Connection to SMBus" },
	{ NULL, (0x85 << 16) + 0x016, "Output Connection" },
	{ NULL, (0x85 << 16) + 0x017, "Charger Connection" },
	{ NULL, (0x85 << 16) + 0x018, "Battery Insertion" },
	{ NULL, (0x85 << 16) + 0x019, "Use Next" },
	{ NULL, (0x85 << 16) + 0x01a, "OK to use" },
	{ NULL, (0x85 << 16) + 0x01b, "Battery  Supported" },
	{ NULL, (0x85 << 16) + 0x01c, "SelectorRevision" },
	{ NULL, (0x85 << 16) + 0x01d, "Charging Indicator" },
	{ NULL, (0x85 << 16) + 0x028, "Manufacturer Access" },
	{ NULL, (0x85 << 16) + 0x029, "Remaining Capacity Limit" },
	{ NULL, (0x85 << 16) + 0x02a, "Remaining Time Limit" },
	{ NULL, (0x85 << 16) + 0x02b, "At Rate" },
	{ NULL, (0x85 << 16) + 0x02c, "Capacity Mode" },
	{ NULL, (0x85 << 16) + 0x02d, "Broadcast To Charger" },
	{ NULL, (0x85 << 16) + 0x02e, "Primary Battery" },
	{ NULL, (0x85 << 16) + 0x02f, "Charge Controller" },
	{ NULL, (0x85 << 16) + 0x040, "Terminate Charge" },
	{ NULL, (0x85 << 16) + 0x041, "Terminate Discharge" },
	{ NULL, (0x85 << 16) + 0x042, "Below Remaining Capacity Limit" },
	{ NULL, (0x85 << 16) + 0x043, "Remaining Time Limit Expired" },
	{ NULL, (0x85 << 16) + 0x044, "Charging" },
	{ NULL, (0x85 << 16) + 0x045, "Discharging" },
	{ NULL, (0x85 << 16) + 0x046, "Fully Charged" },
	{ NULL, (0x85 << 16) + 0x047, "Fully Discharged" },
	{ NULL, (0x85 << 16) + 0x048, "Conditioning Flag" },
	{ NULL, (0x85 << 16) + 0x049, "At Rate OK" },
	{ NULL, (0x85 << 16) + 0x04a, "SMB Error Code" },
	{ NULL, (0x85 << 16) + 0x04b, "Need Replacement" },
	{ NULL, (0x85 << 16) + 0x060, "At Rate Time To Full" },
	{ NULL, (0x85 << 16) + 0x061, "At Rate Time To Empty" },
	{ NULL, (0x85 << 16) + 0x062, "Average Current" },
	{ NULL, (0x85 << 16) + 0x063, "Max Error" },
	{ NULL, (0x85 << 16) + 0x064, "Relative State Of Charge" },
	{ NULL, (0x85 << 16) + 0x065, "Absolute State Of Charge" },
	{ NULL, (0x85 << 16) + 0x066, "Remaining Capacity" },
	{ NULL, (0x85 << 16) + 0x067, "Full Charge Capacity" },
	{ NULL, (0x85 << 16) + 0x068, "Run Time To Empty" },
	{ NULL, (0x85 << 16) + 0x069, "Average Time To Empty" },
	{ NULL, (0x85 << 16) + 0x06a, "Average Time To Full" },
	{ NULL, (0x85 << 16) + 0x06b, "Cycle Count" },
	{ NULL, (0x85 << 16) + 0x080, "Batt. Pack Model Level" },
	{ NULL, (0x85 << 16) + 0x081, "Internal Charge Controller" },
	{ NULL, (0x85 << 16) + 0x082, "Primary Battery Support" },
	{ NULL, (0x85 << 16) + 0x083, "Design Capacity" },
	{ NULL, (0x85 << 16) + 0x084, "Specification Info" },
	{ NULL, (0x85 << 16) + 0x085, "Manufacturer Date" },
	{ NULL, (0x85 << 16) + 0x086, "Serial Number" },
	{ NULL, (0x85 << 16) + 0x087, "iManufacturerName" },
	{ NULL, (0x85 << 16) + 0x088, "iDeviceName" },
	{ NULL, (0x85 << 16) + 0x089, "iDeviceChemistry" },
	{ NULL, (0x85 << 16) + 0x08a, "Manufacturer Data" },
	{ NULL, (0x85 << 16) + 0x08b, "Rechargeable" },
	{ NULL, (0x85 << 16) + 0x08c, "Warning Capacity Limit" },
	{ NULL, (0x85 << 16) + 0x08d, "Capacity Granularity 1" },
	{ NULL, (0x85 << 16) + 0x08e, "Capacity Granularity 2" },
	{ NULL, (0x85 << 16) + 0x08f, "iOEMInformation" },
	{ NULL, (0x85 << 16) + 0x0c0, "Inhibit Charge" },
	{ NULL, (0x85 << 16) + 0x0c1, "Enable Polling" },
	{ NULL, (0x85 << 16) + 0x0c2, "Reset To Zero" },
	{ NULL, (0x85 << 16) + 0x0d0, "AC Present" },
	{ NULL, (0x85 << 16) + 0x0d1, "Battery Present" },
	{ NULL, (0x85 << 16) + 0x0d2, "Power Fail" },
	{ NULL, (0x85 << 16) + 0x0d3, "Alarm Inhibited" },
	{ NULL, (0x85 << 16) + 0x0d4, "Thermistor Under Range" },
	{ NULL, (0x85 << 16) + 0x0d5, "Thermistor Hot" },
	{ NULL, (0x85 << 16) + 0x0d6, "Thermistor Cold" },
	{ NULL, (0x85 << 16) + 0x0d7, "Thermistor Over Range" },
	{ NULL, (0x85 << 16) + 0x0d8, "Voltage Out Of Range" },
	{ NULL, (0x85 << 16) + 0x0d9, "Current Out Of Range" },
	{ NULL, (0x85 << 16) + 0x0da, "Current Not Regulated" },
	{ NULL, (0x85 << 16) + 0x0db, "Voltage Not Regulated" },
	{ NULL, (0x85 << 16) + 0x0dc, "Master Mode" },
	{ NULL, (0x85 << 16) + 0x0f0, "Charger Selector Support" },
	{ NULL, (0x85 << 16) + 0x0f1, "Charger Spec" },
	{ NULL, (0x85 << 16) + 0x0f2, "Level 2" },
	{ NULL, (0x85 << 16) + 0x0f3, "Level 3" },
	{ NULL, (0xf0 << 16) + 0x0f1, "Cash Drawer" },
	{ NULL, (0xf0 << 16) + 0x0f2, "Cash Drawer Number" },
	{ NULL, (0xf0 << 16) + 0x0f3, "Cash Drawer Set" },
	{ NULL, (0xf0 << 16) + 0x0f4, "Cash Drawer Status" },
	{ NULL },
};

/* dialects are represented as:
 *	langid + (dialectid << 10)
 */
static struct genericstrtable langids[] =
{
	{ NULL, 0x0001 + (0x01 << 10), "Saudi Arabia" },
	{ NULL, 0x0001 + (0x02 << 10), "Iraq" },
	{ NULL, 0x0001 + (0x03 << 10), "Egypt" },
	{ NULL, 0x0001 + (0x04 << 10), "Libya" },
	{ NULL, 0x0001 + (0x05 << 10), "Algeria" },
	{ NULL, 0x0001 + (0x06 << 10), "Morocco" },
	{ NULL, 0x0001 + (0x07 << 10), "Tunesia" },
	{ NULL, 0x0001 + (0x08 << 10), "Oman" },
	{ NULL, 0x0001 + (0x09 << 10), "Yemen" },
	{ NULL, 0x0001 + (0x0a << 10), "Syria" },
	{ NULL, 0x0001 + (0x0b << 10), "Jordan" },
	{ NULL, 0x0001 + (0x0c << 10), "Lebanon" },
	{ NULL, 0x0001 + (0x0d << 10), "Kuwait" },
	{ NULL, 0x0001 + (0x0e << 10), "U.A.E" },
	{ NULL, 0x0001 + (0x0f << 10), "Bahrain" },
	{ NULL, 0x0001 + (0x10 << 10), "Qatar" },
	{ NULL, 0x0001, "Arabic" },
	{ NULL, 0x0002, "Bulgarian" },
	{ NULL, 0x0003, "Catalan" },
	{ NULL, 0x0004 + (0x01 << 10), "Traditional" },
	{ NULL, 0x0004 + (0x02 << 10), "Simplified" },
	{ NULL, 0x0004 + (0x03 << 10), "Hongkong SAR, PRC" },
	{ NULL, 0x0004 + (0x04 << 10), "Singapore" },
	{ NULL, 0x0004 + (0x05 << 10), "Macau SAR" },
	{ NULL, 0x0004, "Chinese" },
	{ NULL, 0x0005, "Czech" },
	{ NULL, 0x0006, "Danish" },
	{ NULL, 0x0007 + (0x01 << 10), "German" },
	{ NULL, 0x0007 + (0x02 << 10), "Swiss" },
	{ NULL, 0x0007 + (0x03 << 10), "Austrian" },
	{ NULL, 0x0007 + (0x04 << 10), "Luxembourg" },
	{ NULL, 0x0007 + (0x05 << 10), "Liechtenstein" },
	{ NULL, 0x0007, "German" },
	{ NULL, 0x0008, "Greek" },
	{ NULL, 0x0009 + (0x01 << 10), "US" },
	{ NULL, 0x0009 + (0x02 << 10), "UK" },
	{ NULL, 0x0009 + (0x03 << 10), "Australian" },
	{ NULL, 0x0009 + (0x04 << 10), "Canadian" },
	{ NULL, 0x0009 + (0x05 << 10), "New Zealand" },
	{ NULL, 0x0009 + (0x06 << 10), "Ireland" },
	{ NULL, 0x0009 + (0x07 << 10), "South Africa" },
	{ NULL, 0x0009 + (0x08 << 10), "Jamaica" },
	{ NULL, 0x0009 + (0x09 << 10), "Carribean" },
	{ NULL, 0x0009 + (0x0a << 10), "Belize" },
	{ NULL, 0x0009 + (0x0b << 10), "Trinidad" },
	{ NULL, 0x0009 + (0x0c << 10), "Zimbabwe" },
	{ NULL, 0x0009 + (0x0d << 10), "Philippines" },
	{ NULL, 0x0009, "English" },
	{ NULL, 0x000a + (0x01 << 10), "Castilian" },
	{ NULL, 0x000a + (0x02 << 10), "Mexican" },
	{ NULL, 0x000a + (0x03 << 10), "Modern" },
	{ NULL, 0x000a + (0x04 << 10), "Guatemala" },
	{ NULL, 0x000a + (0x05 << 10), "Costa Rica" },
	{ NULL, 0x000a + (0x06 << 10), "Panama" },
	{ NULL, 0x000a + (0x07 << 10), "Dominican Republic" },
	{ NULL, 0x000a + (0x08 << 10), "Venzuela" },
	{ NULL, 0x000a + (0x09 << 10), "Colombia" },
	{ NULL, 0x000a + (0x0a << 10), "Peru" },
	{ NULL, 0x000a + (0x0b << 10), "Argentina" },
	{ NULL, 0x000a + (0x0c << 10), "Ecuador" },
	{ NULL, 0x000a + (0x0d << 10), "Chile" },
	{ NULL, 0x000a + (0x0e << 10), "Uruguay" },
	{ NULL, 0x000a + (0x0f << 10), "Paraguay" },
	{ NULL, 0x000a + (0x10 << 10), "Bolivia" },
	{ NULL, 0x000a + (0x11 << 10), "El Salvador" },
	{ NULL, 0x000a + (0x12 << 10), "Honduras" },
	{ NULL, 0x000a + (0x13 << 10), "Nicaragua" },
	{ NULL, 0x000a + (0x14 << 10), "Puerto Rico" },
	{ NULL, 0x000a, "Spanish" },
	{ NULL, 0x000b, "Finnish" },
	{ NULL, 0x000c + (0x01 << 10), "French" },
	{ NULL, 0x000c + (0x02 << 10), "Belgian" },
	{ NULL, 0x000c + (0x03 << 10), "Canadian" },
	{ NULL, 0x000c + (0x04 << 10), "Swiss" },
	{ NULL, 0x000c + (0x05 << 10), "Luxembourg" },
	{ NULL, 0x000c + (0x06 << 10), "Monaco" },
	{ NULL, 0x000c, "French" },
	{ NULL, 0x000d, "Hebrew" },
	{ NULL, 0x000e, "Hungarian" },
	{ NULL, 0x000f, "Idelandic" },
	{ NULL, 0x0010 + (0x01 << 10), "Italian" },
	{ NULL, 0x0010 + (0x02 << 10), "Swiss" },
	{ NULL, 0x0010, "Italian" },
	{ NULL, 0x0011, "Japanese" },
	{ NULL, 0x0012 + (0x01 << 10), "Korean" },
	{ NULL, 0x0012, "Korean" },
	{ NULL, 0x0013 + (0x01 << 10), "Dutch" },
	{ NULL, 0x0013 + (0x02 << 10), "Belgian" },
	{ NULL, 0x0013, "Dutch" },
	{ NULL, 0x0014 + (0x01 << 10), "Bokmal" },
	{ NULL, 0x0014 + (0x02 << 10), "Nynorsk" },
	{ NULL, 0x0014, "Norwegian" },
	{ NULL, 0x0015, "Polish" },
	{ NULL, 0x0016 + (0x01 << 10), "Portuguese" },
	{ NULL, 0x0016 + (0x02 << 10), "Brazilian" },
	{ NULL, 0x0016, "Portuguese" },
	{ NULL, 0x0017, "forgotten" },
	{ NULL, 0x0018, "Romanian" },
	{ NULL, 0x0019, "Russian" },
	{ NULL, 0x001a + (0x01 << 10), "Croatian" },
	{ NULL, 0x001a + (0x02 << 10), "Latin" },
	{ NULL, 0x001a + (0x03 << 10), "Cyrillic" },
	{ NULL, 0x001a, "Serbian" },
	{ NULL, 0x001b, "Slovak" },
	{ NULL, 0x001c, "Albanian" },
	{ NULL, 0x001d + (0x01 << 10), "Swedish" },
	{ NULL, 0x001d + (0x02 << 10), "Finland" },
	{ NULL, 0x001d, "Swedish" },
	{ NULL, 0x001e, "Thai" },
	{ NULL, 0x001f, "Turkish" },
	{ NULL, 0x0020 + (0x01 << 10), "Pakistan" },
	{ NULL, 0x0020 + (0x02 << 10), "India" },
	{ NULL, 0x0020, "Urdu" },
	{ NULL, 0x0021, "Indonesian" },
	{ NULL, 0x0022, "Ukrainian" },
	{ NULL, 0x0023, "Belarusian" },
	{ NULL, 0x0024, "Slovenian" },
	{ NULL, 0x0025, "Estonian" },
	{ NULL, 0x0026, "Latvian" },
	{ NULL, 0x0027 + (0x01 << 10), "Lithuanian" },
	{ NULL, 0x0027, "Lithuanian" },
	{ NULL, 0x0028, "forgotten" },
	{ NULL, 0x0029, "Farsi" },
	{ NULL, 0x002a, "Vietnamese" },
	{ NULL, 0x002b, "Armenian" },
	{ NULL, 0x002c + (0x01 << 10), "Cyrillic" },
	{ NULL, 0x002c + (0x02 << 10), "Latin" },
	{ NULL, 0x002c, "Azeri" },
	{ NULL, 0x002d, "Basque" },
	{ NULL, 0x002e, "forgotten" },
	{ NULL, 0x002f, "Macedonian" },
	{ NULL, 0x0036, "Afrikaans" },
	{ NULL, 0x0037, "Georgian" },
	{ NULL, 0x0038, "Faeroese" },
	{ NULL, 0x0039, "Hindi" },
	{ NULL, 0x003e + (0x01 << 10), "Malaysia" },
	{ NULL, 0x003e + (0x02 << 10), "Brunei Darassalam" },
	{ NULL, 0x003e, "Malay" },
	{ NULL, 0x003f, "Kazak" },
	{ NULL, 0x0041, "Awahili" },
	{ NULL, 0x0043 + (0x01 << 10), "Latin" },
	{ NULL, 0x0043 + (0x02 << 10), "Cyrillic" },
	{ NULL, 0x0043, "Uzbek" },
	{ NULL, 0x0044, "Tatar" },
	{ NULL, 0x0045, "Bengali" },
	{ NULL, 0x0046, "Punjabi" },
	{ NULL, 0x0047, "Gujarati" },
	{ NULL, 0x0048, "Oriya" },
	{ NULL, 0x0049, "Tamil" },
	{ NULL, 0x004a, "Telugu" },
	{ NULL, 0x004b, "Kannada" },
	{ NULL, 0x004c, "Malayalam" },
	{ NULL, 0x004d, "Assamese" },
	{ NULL, 0x004e, "Marathi" },
	{ NULL, 0x004f, "Sanskrit" },
	{ NULL, 0x0057, "Konkani" },
	{ NULL, 0x0058, "Manipuri" },
	{ NULL, 0x0059, "Sindhi" },
	{ NULL, 0x0060 + (0x02 << 10), "India" },
	{ NULL, 0x0060, "Kashmiri" },
	{ NULL, 0x0061 + (0x02 << 10), "India" },
	{ NULL, 0x0061, "Nepali" },
	{ NULL },
};

static struct genericstrtable countrycodes[] =
{
        { NULL, 0, "Not supported" },
        { NULL, 1, "Arabic" },
        { NULL, 2, "Belgian" },
        { NULL, 3, "Canadian-Bilingual" },
        { NULL, 4, "Canadian-French" },
        { NULL, 5, "Czech Republic" },
        { NULL, 6, "Danish" },
        { NULL, 7, "Finnish" },
        { NULL, 8, "French" },
        { NULL, 9, "German" },
        { NULL, 10, "Greek" },
        { NULL, 11, "Hebrew" },
        { NULL, 12, "Hungary" },
        { NULL, 13, "International (ISO)" },
        { NULL, 14, "Italian" },
        { NULL, 15, "Japan (Katakana)" },
        { NULL, 16, "Korean" },
        { NULL, 17, "Latin American" },
        { NULL, 18, "Netherlands/Dutch" },
        { NULL, 19, "Norwegian" },
        { NULL, 20, "Persian (Farsi)" },
        { NULL, 21, "Poland" },
        { NULL, 22, "Portuguese" },
        { NULL, 23, "Russia" },
        { NULL, 24, "Slovakia" },
        { NULL, 25, "Spanish" },
        { NULL, 26, "Swedish" },
        { NULL, 27, "Swiss/French" },
        { NULL, 28, "Swiss/German" },
        { NULL, 29, "Switzerland" },
        { NULL, 30, "Taiwan" },
        { NULL, 31, "Turkish-Q" },
        { NULL, 32, "UK" },
        { NULL, 33, "US" },
        { NULL, 34, "Yugoslavia" },
        { NULL, 35, "Turkish-F" },
	{ NULL },
};

/* ---------------------------------------------------------------------- */
#endif /* _USB_SPEC_H */
