/*
   ** Copyright 1996 Thorsten Kukuk <kukuk@uni-paderborn.de>
   **
   ** This program is free software; you can redistribute it and/or modify
   ** it under the terms of the GNU General Public License as published by
   ** the Free Software Foundation; either version 2 of the License, or
   ** (at your option) any later version.
   **
   ** This program is distributed in the hope that it will be useful,
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   ** GNU General Public License for more details.
   **
   ** You should have received a copy of the GNU General Public License
   ** along with this program; if not, write to the Free Software
   ** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <ctype.h>
#include <unistd.h>

#include "pwdutils.h"

/*
   **  print_shell_list - prints /etc/shells to stdout.
 */
void 
print_shell_list ()
{
  char buf[FILENAME_MAX + 2];
  FILE *f;
  int len;

  f = fopen ("/etc/shells", "r");
  if (!f)
    {
      printf ("No known shells.\n");
      return;
    }

  while (fgets (buf, sizeof (buf), f) != NULL)
    {
      /* ignore comments */
      if (*buf == '#')
	continue;
      len = strlen (buf);
      /* strip the ending newline */
      if (buf[len - 1] == '\n')
	buf[len - 1] = 0;
      /* ignore lines that are too long */
      else
	continue;
      /* output the shell */
      printf ("%s\n", buf);
    }
  fclose (f);
}

/*
   ** check_shell - if the shell isn't valid, print an warning and return 0, 
   ** else return 1.
 */
int 
check_shell (char *shell)
{
  FILE *f;
  char buf[FILENAME_MAX + 2];
  unsigned int i, len;
  char c;

  if (*shell != '/')
    {
      printf ("%s: shell must be a full path name.\n", progname);
      return 1;
    }
  if (access (shell, F_OK) < 0)
    {
      printf ("%s: \"%s\" does not exist.\n", progname, shell);
      return 1;
    }
  if (access (shell, X_OK) < 0)
    {
      printf ("%s: \"%s\" is not executable.\n", progname, shell);
      return 1;
    }
  for (i = 0; i < strlen (shell); i++)
    {
      c = shell[i];
      if (c == ',' || c == ':' || c == '=' || c == '"' || c == '\n')
	{
	  printf ("%s: '%c' is not allowed.\n", progname, c);
	  return 1;
	}
      if (iscntrl (c))
	{
	  printf ("%s: Control characters are not allowed.\n", progname);
	  return 1;
	}
    }

  f = fopen ("/etc/shells", "r");
  if (!f)
    {
      if ((strcmp (shell, "/bin/sh") == 0) ||
	  (strcmp (shell, "/bin/csh") == 0))
	return 1;
      return 0;
    }

  while (fgets (buf, sizeof (buf), f) != NULL)
    {
      if (*buf == '#')
	continue;
      len = strlen (buf);
      if (buf[len - 1] == '\n')
	buf[len - 1] = 0;
      /* ignore lines that are too long */
      else
	continue;
      if (strcmp (shell, buf) == 0)
	{
	  fclose (f);
	  return 1;
	}
    }

  fclose (f);
  printf ("%s: \"%s\" is not listed as a valid shell.\n", progname, shell);
  return 0;
}
