/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin for monthly report.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgreportbank.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"
#include "skgoperationobject.h"
#include "skgrecurrentoperationobject.h"

#include <kstandarddirs.h>
#include <grantlee/engine.h>
#include <grantlee/templateloader.h>
#ifdef SKGGRANTLEE17ENABLED
#include <grantlee/metatype.h>
#endif
#include <kaboutdata.h>
#include <kcomponentdata.h>
#include <kcolorscheme.h>
#include <KUrl>

#include <QTextStream>
#include <QFile>
#include <QDir>


#ifdef SKGGRANTLEE17ENABLED
GRANTLEE_BEGIN_LOOKUP(SKGObjectBase)
Q_UNUSED(object);
Q_UNUSED(property);
GRANTLEE_END_LOOKUP
#endif

SKGReportBank::SKGReportBank(SKGDocument* iDocument)
    : SKGReport(iDocument)
{
    SKGTRACEIN(1, "SKGReportBank::SKGReportBank");

    //Grantlee initialization
#ifdef SKGGRANTLEE17ENABLED
    Grantlee::MetaType::init();
    Grantlee::registerMetaType<SKGObjectBase>();
#endif
}

SKGReportBank::~SKGReportBank()
{
    SKGTRACEIN(1, "SKGReportBank::~SKGReportBank");
}

QVariantList SKGReportBank::getBudgetTable()
{
    QVariantList table = m_cache["getBudgetTable"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getBudgetTable");
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
        table = doc ? doc->getBudget(getMonth()) : QVariantList();
        m_cache["getBudgetTable"] = table;
    }
    return table;
}

QVariantList SKGReportBank::getPortfolio()
{
    QVariantList table = m_cache["getPortfolio"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getPortfolio");
        QString month = getMonth();
        if (!month.isEmpty()) {
            QDate date = QDate::fromString(month, "yyyy-MM");
            QDate date2 = date.addMonths(1).addDays(-1);

            SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
            KLocale* locale = KGlobal::locale();
            if (doc && locale) {
                SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

                //Get list of operations
                SKGObjectBase::SKGListSKGObjectBase objs;
                SKGError err = doc->getObjects("v_operation_display", "d_date<'" % SKGServices::dateToSqlString(QDateTime(date2)) % "' AND rc_unit_id IN (SELECT id FROM v_unit_display WHERE t_type='S' AND f_QUANTITYOWNED>0.01) ORDER BY t_UNIT", objs);
                int nb = objs.count();
                if (!err && nb > 0) {
                    QVariantList line;
                    line << doc->getDisplay("t_symbol")
                         << doc->getDisplay("t_UNIT")
                         << i18nc("Column table title", "Quantity")
                         << i18nc("Column table title", "Purchase amount")
                         << i18nc("Column table title", "Initial amount")
                         << locale->formatDate(date2, KLocale::ShortDate)
                         << i18nc("Column table title", "Variation")
                         << i18nc("Column table title", "Variation %")
                         ;
                    table << QVariant(line);

                    QList<unitValues> listUnitValues;
                    unitValues current;
                    current.initalAmount = 0.0;
                    current.purchaseAmount = 0.0;
                    current.currentAmount = 0.0;
                    current.quantity = 0.0;
                    for (int i = 0; i < nb; ++i) {
                        SKGOperationObject obj(objs.at(i));
                        SKGUnitObject unit;
                        obj.getUnit(unit);
                        if (i != 0 && current.unit != unit) {
                            listUnitValues.push_back(current);
                            current.initalAmount = 0.0;
                            current.purchaseAmount = 0.0;
                            current.currentAmount = 0.0;
                            current.quantity = 0.0;
                        }

                        current.unit = unit;
                        current.initalAmount += obj.getAmount(obj.getDate());
                        current.currentAmount += obj.getAmount(date2);
                        SKGObjectBase::SKGListSKGObjectBase oGroupedOperations;
                        obj.getGroupedOperations(oGroupedOperations);
                        oGroupedOperations.removeAll(obj);
                        if (oGroupedOperations.count() == 1) {
                            SKGOperationObject obj2(oGroupedOperations.at(0));
                            current.purchaseAmount += obj2.getAmount(obj.getDate());
                        }
                        current.quantity += SKGServices::stringToDouble(obj.getAttribute("f_QUANTITY"));
                    }
                    if (!current.unit.getName().isEmpty()) listUnitValues.push_back(current);

                    nb = listUnitValues.count();
                    for (int i = 0; i < nb; ++i) {
                        unitValues current = listUnitValues.at(i);

                        SKGServices::SKGUnitInfo ui = current.unit.getUnitInfo();
                        ui.Value = 1;

                        QVariantList line;
                        line << current.unit.getSymbol()
                             << current.unit.getName()
                             << doc->formatMoney(current.quantity, ui, false)
                             << current.purchaseAmount
                             << current.initalAmount
                             << current.currentAmount
                             << current.currentAmount - current.initalAmount
                             << (current.initalAmount == 0.0 ? 0.0 : (current.currentAmount - current.initalAmount) / current.initalAmount);
                        table << QVariant(line);
                    }
                }
            }
        }
        m_cache["getPortfolio"] = table;
    }

    return table;
}

QVariantList SKGReportBank::getUnitTable()
{
    QVariantList table = m_cache["getUnitTable"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getUnitTable");
        QString month = getMonth();
        if (!month.isEmpty()) {
            QDate date = QDate::fromString(month, "yyyy-MM");
            QDate date1 = date.addDays(-1);
            QDate date2 = date.addMonths(1).addDays(-1);
            QString previousmonth = date1.toString("yyyy-MM");
            SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
            KLocale* locale = KGlobal::locale();
            if (doc && locale) {
                SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

                SKGObjectBase::SKGListSKGObjectBase units;
                SKGError err = doc->getObjects("v_unit_display", "1=1 ORDER BY t_TYPENLS", units);
                if (units.count()) {
                    QVariantList line;
                    line << "sum"
                         << doc->getDisplay("t_UNIT")
                         << locale->formatDate(date1, KLocale::ShortDate)
                         << locale->formatDate(date2, KLocale::ShortDate)
                         << "%"
                         << doc->getDisplay("t_symbol");
                    table << QVariant(line);
                    foreach(const SKGObjectBase & item, units) {
                        SKGUnitObject unit(item);
                        double v1 = unit.getAmount(date1);
                        double v2 = unit.getAmount(date2);
                        QVariantList line;
                        line << false << unit.getName() << v1 << v2 << (100.0 * (v2 - v1) / qAbs(v1)) << unit.getSymbol();
                        table << QVariant(line);
                    }
                }
            }
        }
        m_cache["getUnitTable"] = table;
    }

    return table;
}

QVariantList SKGReportBank::getAccountTable()
{
    QVariantList table = m_cache["getAccountTable"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getAccountTable");
        QString month = getMonth();
        if (!month.isEmpty()) {
            QDate date = QDate::fromString(month, "yyyy-MM");
            QDate date1 = date.addDays(-1);
            QDate date2 = date.addMonths(1).addDays(-1);
            QDate date3 = date2.addYears(-1);
            QString previousmonth = date1.toString("yyyy-MM");
            SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
            KLocale* locale = KGlobal::locale();
            if (doc && locale) {
                SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

                SKGObjectBase::SKGListSKGObjectBase accounts;
                SKGError err = doc->getObjects("v_account_display", "1=1 ORDER BY t_TYPENLS, t_BANK, t_name", accounts);
                if (!err) {
                    QVariantList line;
                    line << "sum" << doc->getDisplay("t_ACCOUNT") << locale->formatDate(date1, KLocale::ShortDate) << locale->formatDate(date2, KLocale::ShortDate) <<
                         "%" << locale->formatDate(date3, KLocale::ShortDate) << locale->formatDate(date2, KLocale::ShortDate) << "%";
                    table.push_back(line);
                    double sumTypeV1 = 0;
                    double sumTypeV2 = 0;
                    double sumTypeV3 = 0;
                    double sumV1 = 0;
                    double sumV2 = 0;
                    double sumV3 = 0;
                    QString currentType;
                    int nb = accounts.count();
                    for (int i = 0; !err && i < nb; ++i) {
                        SKGAccountObject account(accounts.at(i));
                        double v1 = account.getAmount(date1);
                        double v2 = account.getAmount(date2);
                        double v3 = account.getAmount(date3);
                        QString type = account.getAttribute("t_TYPENLS");
                        bool closed = account.isClosed();
                        if (type != currentType) {
                            if (!currentType.isEmpty()) {
                                QVariantList line;
                                line << true << i18nc("Noun",  "Total of %1", currentType) << sumTypeV1 << sumTypeV2 <<
                                     (100.0 * (sumTypeV2 - sumTypeV1) / qAbs(sumTypeV1)) << sumTypeV3 << sumTypeV2
                                     << (100.0 * (sumTypeV2 - sumTypeV3) / qAbs(sumTypeV3))
                                     << "" << "";
                                table.push_back(line);
                                sumTypeV1 = 0;
                                sumTypeV2 = 0;
                                sumTypeV3 = 0;
                            }
                            currentType = type;
                        }
                        if (!closed || qAbs(v1) > 0.01 || qAbs(v2) > 0.01 || qAbs(v3) > 0.01) {
                            QVariantList line;
                            QString icon = account.getAttribute("t_ICON");
                            if (!icon.isEmpty()) {
                                QDir dirLogo(KStandardDirs::locate("data", QString::fromLatin1("skrooge/images/logo/")));
                                icon = dirLogo.absoluteFilePath(icon);
                            }
                            line << false << account.getName() << v1 << v2 << (100.0 * (v2 - v1) / qAbs(v1)) << v3 << v2
                                 << (100.0 * (v2 - v3) / qAbs(v3)) << account.getAttribute("t_BANK") << icon ;
                            table.push_back(line);
                            ;
                        }
                        sumTypeV1 += v1;
                        sumTypeV2 += v2;
                        sumTypeV3 += v3;
                        sumV1 += v1;
                        sumV2 += v2;
                        sumV3 += v3;
                    }

                    {
                        QVariantList line;
                        line << true << i18nc("Noun",  "Total of %1", currentType) << sumTypeV1 << sumTypeV2
                             << (100.0 * (sumTypeV2 - sumTypeV1) / qAbs(sumTypeV1)) << sumTypeV3 << sumTypeV2 << (100.0 * (sumTypeV2 - sumTypeV3) / qAbs(sumTypeV3))
                             << "" << "";
                        table.push_back(line);
                    }

                    {
                        QVariantList line;
                        line << true << i18nc("Noun, the numerical total of a sum of values", "Total") << sumV1 << sumV2
                             << (100.0 * (sumV2 - sumV1) / qAbs(sumV1)) << sumV3 << sumV2 << (100.0 * (sumV2 - sumV3) / qAbs(sumV3))
                             << "" << "";
                        table.push_back(line);
                    }
                }
            }
        }
        m_cache["getAccountTable"] = table;
    }
    return table;
}

QVariantList SKGReportBank::getBankTable()
{
    QVariantList table = m_cache["getBankTable"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getBankTable");
        QString month = getMonth();
        if (!month.isEmpty()) {
            QDate date = QDate::fromString(month, "yyyy-MM");
            QDate date1 = date.addDays(-1);
            QDate date2 = date.addMonths(1).addDays(-1);
            QDate date3 = date2.addYears(-1);
            QString previousmonth = date1.toString("yyyy-MM");
            SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
            KLocale* locale = KGlobal::locale();
            if (doc && locale) {
                SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

                SKGObjectBase::SKGListSKGObjectBase accounts;
                SKGError err = doc->getObjects("v_account", "1=1 ORDER BY t_BANK", accounts);
                if (!err) {
                    QVariantList line;
                    line << "sum" << doc->getDisplay("t_BANK") << locale->formatDate(date1, KLocale::ShortDate) << locale->formatDate(date2, KLocale::ShortDate) <<
                         "%" << locale->formatDate(date3, KLocale::ShortDate) << locale->formatDate(date2, KLocale::ShortDate) << "%";
                    table.push_back(line);
                    double sumTypeV1 = 0;
                    double sumTypeV2 = 0;
                    double sumTypeV3 = 0;
                    double sumV1 = 0;
                    double sumV2 = 0;
                    double sumV3 = 0;
                    QString currentType;
                    QString currentIcon;
                    bool currentOpen = false;
                    int nb = accounts.count();
                    for (int i = 0; !err && i < nb; ++i) {
                        SKGAccountObject account(accounts.at(i));
                        double v1 = account.getAmount(date1);
                        double v2 = account.getAmount(date2);
                        double v3 = account.getAmount(date3);
                        QString type = account.getAttribute("t_BANK");
                        QString icon = account.getAttribute("t_ICON");
                        if (!icon.isEmpty()) {
                            QDir dirLogo(KStandardDirs::locate("data", QString::fromLatin1("skrooge/images/logo/")));
                            icon = dirLogo.absoluteFilePath(icon);
                        }
                        bool open = !account.isClosed();
                        if (type != currentType) {
                            if (!currentType.isEmpty() && currentOpen) {
                                QVariantList line;
                                line << false << currentType << sumTypeV1 << sumTypeV2 <<
                                     (100.0 * (sumTypeV2 - sumTypeV1) / qAbs(sumTypeV1)) << sumTypeV3 << sumTypeV2 << (100.0 * (sumTypeV2 - sumTypeV3) / qAbs(sumTypeV3)) <<
                                     currentIcon;
                                table.push_back(line);
                                sumTypeV1 = 0;
                                sumTypeV2 = 0;
                                sumTypeV3 = 0;
                                currentOpen = open;
                            }
                            currentType = type;
                            currentIcon = icon;
                        }
                        currentOpen = currentOpen || open;

                        sumTypeV1 += v1;
                        sumTypeV2 += v2;
                        sumTypeV3 += v3;
                        sumV1 += v1;
                        sumV2 += v2;
                        sumV3 += v3;
                    }

                    if (currentOpen) {
                        QVariantList line;
                        line << false << currentType << sumTypeV1 << sumTypeV2
                             << (100.0 * (sumTypeV2 - sumTypeV1) / qAbs(sumTypeV1)) << sumTypeV3
                             << sumTypeV2 << (100.0 * (sumTypeV2 - sumTypeV3) / qAbs(sumTypeV3))
                             << currentIcon;
                        table.push_back(line);
                    }

                    {
                        QVariantList line;
                        line << true << i18nc("Noun, the numerical total of a sum of values", "Total") << sumV1 << sumV2 << (100.0 * (sumV2 - sumV1) / qAbs(sumV1)) << sumV3 << sumV2 << (100.0 * (sumV2 - sumV3) / qAbs(sumV3));
                        table.push_back(line);
                    }
                }
            }
        }
        m_cache["getBankTable"] = table;
    }
    return table;
}

QVariantList SKGReportBank::getScheduledOperations()
{
    QVariantList table = m_cache["getScheduledOperations"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getScheduledOperations");
        SKGObjectBase::SKGListSKGObjectBase objs;
        SKGError err = m_document->getObjects("v_recurrentoperation_display", "i_nb_times!=0 ORDER BY d_date LIMIT 5", objs);
        if (!err) {
            int nb = objs.count();
            if (nb) {
                for (int i = 0; i < nb; ++i) {
                    SKGRecurrentOperationObject obj(objs.at(i));
                    bool bold = false;
                    if (obj.isWarnEnabled() &&  QDate::currentDate() >= obj.getDate().addDays(-obj.getWarnDays())) bold = true;
                    table.push_back(QVariantList() << bold << obj.getDisplayName());
                }
            }
            m_cache["getScheduledOperations"] = table;
        }
    }
    return table;
}

QVariantList SKGReportBank::getMainCategoriesMonth()
{
    QVariantList table = m_cache["getMainCategoriesMonth"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getMainCategoriesMonth");
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
        table = doc ? doc->getMainCategories(getMonth(), 5) : QVariantList();
        m_cache["getMainCategoriesMonth"] = table;
    }
    return table;
}

QVariantList SKGReportBank::getMainCategoriesPreviousMonth()
{
    QVariantList table = m_cache["getMainCategoriesPreviousMonth"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getMainCategoriesPreviousMonth");
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
        table = doc ? doc->getMainCategories(getPreviousMonth(), 5) : QVariantList();
        m_cache["getMainCategoriesPreviousMonth"] = table;
    }
    return table;
}

QStringList SKGReportBank::get5MainCategoriesVariation()
{
    QStringList table = m_cache["get5MainCategoriesVariation"].toStringList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::get5MainCategoriesVariation");
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
        table = doc ? doc->get5MainCategoriesVariationList(getMonth(), getPreviousMonth()) : QStringList();
        m_cache["get5MainCategoriesVariation"] = table;
    }
    return table;
}

QVariant SKGReportBank::getAdvice()
{
    SKGTRACEIN(10, "SKGReportBank::getAdvice");
    QVariant output = m_cache["getAdvice"];
    if (!output.isValid()) {
        //TODO output = qVariantFromValue(SKGMainPanel::getMainPanel()->getAdvice());
        m_cache["getAdvice"] = output;
    }
    return output;
}

QVariantList SKGReportBank::getIncomeVsExpenditure()
{
    QVariantList table = m_cache["getIncomeVsExpenditure"].toList();
    if (table.count() == 0) {
        SKGTRACEIN(10, "SKGReportBank::getIncomeVsExpenditure");
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(m_document);
        if (doc) {
            QString month = getMonth();
            QString previousmonth = getPreviousMonth();
            SKGObjectBase::SKGListSKGObjectBase ops;
            SKGError err = doc->getObjects("v_operation_display", "d_DATEMONTH IN ('" % month % "', '" % previousmonth % "') AND t_TRANSFER='N'", ops);
            int nbval = ops.count();
            if (!err && nbval > 0) {
                QVariantList line;
                line << "sum" << "" << previousmonth << month << "max";
                table.push_back(line);

                double income_previous_month = 0;
                double expense_previous_month = 0;
                double income_month = 0;
                double expense_month = 0;

                for (int i = 0; i < nbval; ++i) {
                    SKGOperationObject op(ops.at(i));
                    QString m = op.getAttribute("d_DATEMONTH");
                    QDate d = QDate::fromString(m, "yyyy-MM").addMonths(1).addDays(-1);
                    double v = op.getAmount(d);
                    if (v > 0 && m == month) income_month += v;
                    else if (v < 0 && m == month) expense_month += v;
                    else if (v > 0 && m == previousmonth) income_previous_month += v;
                    else if (v < 0 && m == previousmonth) expense_previous_month += v;
                }
                double saving_previous_month = income_previous_month + expense_previous_month;
                double saving_month = income_month + expense_month;

                {
                    QVariantList line;
                    line << false << doc->getDisplay("f_CURRENTAMOUNT_INCOME") << qAbs(income_previous_month) << qAbs(income_month);
                    table.push_back(line);
                }
                {
                    QVariantList line;
                    line << false << doc->getDisplay("f_CURRENTAMOUNT_EXPENSE") << qAbs(expense_previous_month) << qAbs(expense_month);
                    table.push_back(line);
                }
                {
                    QVariantList line;
                    line << true << i18nc("Noun",  "Savings possible") << saving_previous_month << saving_month;
                    table.push_back(line);
                }
                {
                    QVariantList line;
                    line << true << i18nc("Noun",  "Max") << qMax(qAbs(income_previous_month), qAbs(expense_previous_month)) << qMax(qAbs(income_month), qAbs(expense_month));
                    table.push_back(line);
                }
            }
        }
    }
    return table;
}

void SKGReportBank::addItemsInMapping(QVariantHash& iMapping)
{
    SKGReport::addItemsInMapping(iMapping);
    iMapping.insert("about_forumpage", "http://forum.kde.org/viewforum.php?f=210");
    iMapping.insert("about_newspage", "http://skrooge.org/news");
    iMapping.insert("about_operationpage", "skg://Skrooge_operation_plugin/");
    iMapping.insert("about_accountpage", "skg://Skrooge_bank_plugin/");
    iMapping.insert("about_importurl", "skg://import_operation/");
    iMapping.insert("about_maintext", i18nc("The main text of skrooge",
                                            "Skrooge allows you to keep a hold on your expenses, by tracking and budgeting them.<br/>"
                                            "What should you do now ?<br/>"
                                            "<ul>"
                                            "<li>Create at least one <a href=\"%1\">account</a></li>"
                                            "<li>Add some operations, using <a href=\"%3\">import</a> or <a href=\"%2\">manual input</a></li>"
                                            "<li>Categorize them</li>"
                                            "</ul>"
                                            "<p>You may come back to this page any time by closing all tabs.<br/>"
                                            "For more information about using Skrooge, check the <a href=\"http://skrooge.org\">Skrooge website</a>.</p>"
                                            "<p>We hope that you will enjoy Skrooge</p>"
                                            "    The Skrooge Team",
                                            iMapping["about_accountpage"].toString(), iMapping["about_operationpage"].toString(), iMapping["about_importurl"].toString()));
    iMapping.insert("logo", KUrl(KStandardDirs().findResource("icon", QString::fromLatin1("hicolor/128x128/apps/skrooge.png"))).url());
    iMapping.insert("logo_black", KUrl(KStandardDirs().findResource("icon", QString::fromLatin1("hicolor/128x128/apps/skrooge-black.png"))).url());

    iMapping.insert("title_main", i18nc("A monthly report title", "Report for %1", getMonth()));
    iMapping.insert("title_budget", i18nc("A monthly report title", "Budget"));
    iMapping.insert("title_main_categories", i18nc("A monthly report title", "5 main categories of expenditure"));
    iMapping.insert("title_variations", i18nc("A monthly report title", "5 main variations"));
    iMapping.insert("title_account", i18nc("A monthly report title", "Amounts in accounts"));
    iMapping.insert("title_unit", i18nc("A monthly report title", "Amounts of units"));
    iMapping.insert("title_advice", i18nc("A monthly report title", "Advice"));
    iMapping.insert("title_portfolio", i18nc("A monthly report title", "Stock portfolio"));
    iMapping.insert("title_highlighted", i18nc("A monthly report title", "Highlighted operations"));

    iMapping.insert("msg_no_variation", i18nc("A monthly report message", "No variation found."));
    iMapping.insert("msg_no_scheduled", i18nc("A monthly report message", "No scheduled operations defined on the \"<a href=\"%1\">Scheduled operations</a>\" page.", "skg://Skrooge_scheduled_plugin/"));
    iMapping.insert("msg_no_highlighted", i18nc("A monthly report message", "No highlighted operations defined on the \"<a href=\"%1\">Operations</a>\" page.", "skg://Skrooge_operation_plugin/"));
    iMapping.insert("msg_no_budget", i18nc("A monthly report message", "No budget defined on the \"<a href=\"%1\">Budget</a>\" page.", "skg://Skrooge_budget_plugin/"));
    iMapping.insert("msg_no_share", i18nc("A monthly report message", "No share defined on the \"<a href=\"%1\">Unit</a>\" page.", "skg://Skrooge_unit_plugin/"));
    iMapping.insert("msg_amount_unit_date", i18nc("A monthly report message", "All amounts are calculated using the unit rates of the last day of the corresponding month."));
}

#include "skgreportbank.moc"
