"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generatePreview = exports.displayQuickPreviewWindow = exports.canPossiblyPreviewExtension = void 0;
const child_process_1 = require("child_process");
const path_1 = __importDefault(require("path"));
let quickPreviewWindow = null;
let captureWindow = null;
const captureQueue = [];
const filesRoot = __dirname.replace('app.asar', 'app.asar.unpacked');
const FileSizeLimit = 5 * 1024 * 1024;
const ThumbnailWidth = 320 * (11 / 8.5);
const QuicklookIsAvailable = process.platform === 'darwin';
const PDFJSRoot = path_1.default.join(filesRoot, 'pdfjs-4.3.136');
const QuicklookBlacklist = [
    'jpg',
    'bmp',
    'gif',
    'png',
    'jpeg',
    'zip',
    'tar',
    'gz',
    'bz2',
    'dmg',
    'exe',
    'ics',
];
const CrossplatformStrategies = {
    pdfjs: ['pdf'],
    mammoth: ['docx'],
    snarkdown: ['md'],
    xlsx: ['xls', 'xlsx', 'csv', 'eth', 'ods', 'fods', 'uos1', 'uos2', 'dbf', 'prn', 'xlw', 'xlsb'],
    prism: [
        'html',
        'svg',
        'xml',
        'css',
        'c',
        'cc',
        'cpp',
        'js',
        'jsx',
        'tsx',
        'ts',
        'go',
        'cs',
        'patch',
        'swift',
        'java',
        'json',
        'jsonp',
        'tex',
        'mm',
        'm',
        'h',
        'py',
        'rb',
        'rs',
        'sql',
        'yml',
        'yaml',
        'txt',
        'log',
    ],
};
const CrossplatformStrategiesBetterThanQuicklook = ['snarkdown', 'prism'];
function strategyForPreviewing(ext) {
    if (ext.startsWith('.'))
        ext = ext.substr(1);
    const strategy = Object.keys(CrossplatformStrategies).find(strategy => CrossplatformStrategies[strategy].includes(ext));
    if (QuicklookIsAvailable && !QuicklookBlacklist.includes(ext)) {
        if (!strategy || !CrossplatformStrategiesBetterThanQuicklook.includes(strategy)) {
            return 'quicklook';
        }
    }
    return strategy;
}
const PreviewWindowMenuTemplate = [
    {
        label: 'File',
        role: 'window',
        submenu: [
            {
                label: 'Minimize Window',
                accelerator: 'CmdOrCtrl+M',
                role: 'minimize',
            },
            {
                label: 'Close Window',
                accelerator: 'CmdOrCtrl+W',
                role: 'close',
            },
        ],
    },
    {
        label: 'Edit',
        submenu: [
            {
                label: 'Undo',
                accelerator: 'CmdOrCtrl+Z',
                role: 'undo',
            },
            {
                label: 'Redo',
                accelerator: 'Shift+CmdOrCtrl+Z',
                role: 'redo',
            },
            {
                type: 'separator',
            },
            {
                label: 'Cut',
                accelerator: 'CmdOrCtrl+X',
                role: 'cut',
            },
            {
                label: 'Copy',
                accelerator: 'CmdOrCtrl+C',
                role: 'copy',
            },
            {
                label: 'Paste',
                accelerator: 'CmdOrCtrl+V',
                role: 'paste',
            },
            {
                label: 'Select All',
                accelerator: 'CmdOrCtrl+A',
                role: 'selectAll',
            },
        ],
    },
    {
        label: 'View',
        submenu: [
            {
                label: 'Reload',
                accelerator: 'CmdOrCtrl+R',
                click: function (item, focusedWindow) {
                    if (focusedWindow)
                        focusedWindow.reload();
                },
            },
            {
                label: 'Toggle Full Screen',
                accelerator: (function () {
                    if (process.platform === 'darwin')
                        return 'Ctrl+Command+F';
                    else
                        return 'F11';
                })(),
                click: function (item, focusedWindow) {
                    if (focusedWindow)
                        focusedWindow.setFullScreen(!focusedWindow.isFullScreen());
                },
            },
            {
                label: 'Toggle Developer Tools',
                accelerator: (function () {
                    if (process.platform === 'darwin')
                        return 'Alt+Command+I';
                    else
                        return 'Ctrl+Shift+I';
                })(),
                click: function (item, focusedWindow) {
                    if (focusedWindow)
                        focusedWindow.webContents.toggleDevTools();
                },
            },
        ],
    },
];
function canPossiblyPreviewExtension(file) {
    if (file.size > FileSizeLimit) {
        return false;
    }
    return !!strategyForPreviewing(file.displayExtension());
}
exports.canPossiblyPreviewExtension = canPossiblyPreviewExtension;
function displayQuickPreviewWindow(filePath) {
    const isPDF = filePath.endsWith('.pdf');
    const strategy = strategyForPreviewing(path_1.default.extname(filePath));
    if (strategy === 'quicklook') {
        const currentWin = AppEnv.getCurrentWindow();
        currentWin.previewFile(filePath);
        return;
    }
    if (quickPreviewWindow === null) {
        const { BrowserWindow } = require('@electron/remote');
        quickPreviewWindow = new BrowserWindow({
            width: 800,
            height: 600,
            center: true,
            skipTaskbar: true,
            backgroundColor: isPDF ? '#404040' : '#FFF',
            webPreferences: {
                preload: path_1.default.join(filesRoot, 'preload.js'),
                nodeIntegration: false,
                contextIsolation: true,
            },
        });
        quickPreviewWindow.once('closed', () => {
            quickPreviewWindow = null;
        });
        quickPreviewWindow.setMenu(require('@electron/remote').Menu.buildFromTemplate(PreviewWindowMenuTemplate));
    }
    else {
        quickPreviewWindow.show();
    }
    quickPreviewWindow.setTitle(path_1.default.basename(filePath));
    if (isPDF) {
        quickPreviewWindow.loadFile(path_1.default.join(PDFJSRoot, 'web/viewer.html'), {
            search: `file=${encodeURIComponent(`file://${filePath}`)}`,
        });
    }
    else {
        quickPreviewWindow.loadFile(path_1.default.join(filesRoot, 'renderer.html'), {
            search: JSON.stringify({ mode: 'display', filePath, strategy }),
        });
    }
}
exports.displayQuickPreviewWindow = displayQuickPreviewWindow;
async function generatePreview({ file, filePath, previewPath, }) {
    const strategy = strategyForPreviewing(file.displayExtension());
    if (strategy === 'quicklook') {
        return await _generateQuicklookPreview({ filePath });
    }
    else if (strategy) {
        return await _generateCrossplatformPreview({ file, filePath, previewPath, strategy });
    }
    else {
        return false;
    }
}
exports.generatePreview = generatePreview;
async function _generateCrossplatformPreview({ file, filePath, previewPath, strategy }) {
    return new Promise(resolve => {
        captureQueue.push({ file, filePath, previewPath, strategy, resolve });
        if (!captureWindow || captureWindow.isDestroyed()) {
            captureWindow = _createCaptureWindow();
            _generateNextCrossplatformPreview();
        }
    });
}
function _createCaptureWindow() {
    const { BrowserWindow } = require('@electron/remote');
    const win = new BrowserWindow({
        width: ThumbnailWidth,
        height: ThumbnailWidth,
        show: false,
        webPreferences: {
            preload: path_1.default.join(filesRoot, 'preload.js'),
            nodeIntegration: false,
            contextIsolation: true,
        },
    });
    win.webContents.on('crashed', () => {
        console.warn(`Thumbnail generation webcontents crashed.`);
        if (captureWindow === win)
            captureWindow = null;
        win.destroy();
    });
    win.once('closed', () => {
        if (captureWindow === win)
            captureWindow = null;
    });
    return win;
}
function _generateNextCrossplatformPreview() {
    if (captureQueue.length === 0) {
        if (captureWindow && !captureWindow.isDestroyed()) {
            captureWindow.destroy();
        }
        else {
            console.warn(`Thumbnail generation finished but window is already destroyed.`);
        }
        captureWindow = null;
        return;
    }
    const { strategy, filePath, previewPath, resolve } = captureQueue.pop();
    captureWindow.loadFile(path_1.default.join(filesRoot, 'renderer.html'), {
        search: JSON.stringify({ strategy, mode: 'capture', filePath, previewPath }),
    });
    let onFinalize = null;
    const timer = setTimeout(() => {
        console.warn(`Thumbnail generation timed out for ${filePath}`);
        onFinalize(false);
    }, 5000);
    const onRendererSuccess = () => {
        onFinalize(true);
    };
    onFinalize = success => {
        clearTimeout(timer);
        if (captureWindow) {
            captureWindow.removeListener('page-title-updated', onRendererSuccess);
        }
        process.nextTick(_generateNextCrossplatformPreview);
        resolve(success);
    };
    captureWindow.once('page-title-updated', onRendererSuccess);
}
async function _generateQuicklookPreview({ filePath }) {
    const dirQuoted = path_1.default.dirname(filePath).replace(/"/g, '\\"');
    const pathQuoted = filePath.replace(/"/g, '\\"');
    return new Promise(resolve => {
        const cmd = '/usr/bin/qlmanage';
        const args = [
            '-t',
            '-f',
            `${window.devicePixelRatio}`,
            '-s',
            `${ThumbnailWidth}`,
            '-o',
            dirQuoted,
            pathQuoted,
        ];
        child_process_1.execFile(cmd, args, (error, stdout, stderr) => {
            if (error ||
                stdout.match(/No thumbnail created/i) ||
                (stderr && !stdout.includes('produced one thumbnail'))) {
                resolve(false);
            }
            else {
                resolve(true);
            }
        });
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJmaWxlOi8vL3RtcC9ueWxhcy1idWlsZC9lbGVjdHJvbi1wYWNrYWdlci90bXAtcHZLSUxCL3Jlc291cmNlcy9hcHAvc3JjL3F1aWNrcHJldmlldy9pbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxpREFBeUM7QUFDekMsZ0RBQXdCO0FBR3hCLElBQUksa0JBQWtCLEdBQUcsSUFBSSxDQUFDO0FBQzlCLElBQUksYUFBYSxHQUFHLElBQUksQ0FBQztBQUN6QixNQUFNLFlBQVksR0FBRyxFQUFFLENBQUM7QUFFeEIsTUFBTSxTQUFTLEdBQUcsU0FBUyxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztBQUVyRSxNQUFNLGFBQWEsR0FBRyxDQUFDLEdBQUcsSUFBSSxHQUFHLElBQUksQ0FBQztBQUN0QyxNQUFNLGNBQWMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxFQUFFLEdBQUcsR0FBRyxDQUFDLENBQUM7QUFDeEMsTUFBTSxvQkFBb0IsR0FBRyxPQUFPLENBQUMsUUFBUSxLQUFLLFFBQVEsQ0FBQztBQUMzRCxNQUFNLFNBQVMsR0FBRyxjQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxlQUFlLENBQUMsQ0FBQztBQUV4RCxNQUFNLGtCQUFrQixHQUFHO0lBQ3pCLEtBQUs7SUFDTCxLQUFLO0lBQ0wsS0FBSztJQUNMLEtBQUs7SUFDTCxNQUFNO0lBQ04sS0FBSztJQUNMLEtBQUs7SUFDTCxJQUFJO0lBQ0osS0FBSztJQUNMLEtBQUs7SUFDTCxLQUFLO0lBQ0wsS0FBSztDQUNOLENBQUM7QUFFRixNQUFNLHVCQUF1QixHQUFHO0lBQzlCLEtBQUssRUFBRSxDQUFDLEtBQUssQ0FBQztJQUNkLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQztJQUNqQixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUM7SUFDakIsSUFBSSxFQUFFLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUM7SUFDL0YsS0FBSyxFQUFFO1FBQ0wsTUFBTTtRQUNOLEtBQUs7UUFDTCxLQUFLO1FBQ0wsS0FBSztRQUNMLEdBQUc7UUFDSCxJQUFJO1FBQ0osS0FBSztRQUNMLElBQUk7UUFDSixLQUFLO1FBQ0wsS0FBSztRQUNMLElBQUk7UUFDSixJQUFJO1FBQ0osSUFBSTtRQUNKLE9BQU87UUFDUCxPQUFPO1FBQ1AsTUFBTTtRQUNOLE1BQU07UUFDTixPQUFPO1FBQ1AsS0FBSztRQUNMLElBQUk7UUFDSixHQUFHO1FBQ0gsR0FBRztRQUNILElBQUk7UUFDSixJQUFJO1FBQ0osSUFBSTtRQUNKLEtBQUs7UUFDTCxLQUFLO1FBQ0wsTUFBTTtRQUNOLEtBQUs7UUFDTCxLQUFLO0tBQ047Q0FDRixDQUFDO0FBRUYsTUFBTSwwQ0FBMEMsR0FBRyxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztBQUUxRSxTQUFTLHFCQUFxQixDQUFDLEdBQUc7SUFDaEMsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQztRQUFFLEdBQUcsR0FBRyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRTdDLE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsdUJBQXVCLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FDcEUsdUJBQXVCLENBQUMsUUFBUSxDQUFDLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUNoRCxDQUFDO0lBRUYsSUFBSSxvQkFBb0IsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUM3RCxJQUFJLENBQUMsUUFBUSxJQUFJLENBQUMsMENBQTBDLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQy9FLE9BQU8sV0FBVyxDQUFDO1NBQ3BCO0tBQ0Y7SUFFRCxPQUFPLFFBQVEsQ0FBQztBQUNsQixDQUFDO0FBRUQsTUFBTSx5QkFBeUIsR0FBMEM7SUFDdkU7UUFDRSxLQUFLLEVBQUUsTUFBTTtRQUNiLElBQUksRUFBRSxRQUFRO1FBQ2QsT0FBTyxFQUFFO1lBQ1A7Z0JBQ0UsS0FBSyxFQUFFLGlCQUFpQjtnQkFDeEIsV0FBVyxFQUFFLGFBQWE7Z0JBQzFCLElBQUksRUFBRSxVQUFVO2FBQ2pCO1lBQ0Q7Z0JBQ0UsS0FBSyxFQUFFLGNBQWM7Z0JBQ3JCLFdBQVcsRUFBRSxhQUFhO2dCQUMxQixJQUFJLEVBQUUsT0FBTzthQUNkO1NBQ0Y7S0FDRjtJQUNEO1FBQ0UsS0FBSyxFQUFFLE1BQU07UUFDYixPQUFPLEVBQUU7WUFDUDtnQkFDRSxLQUFLLEVBQUUsTUFBTTtnQkFDYixXQUFXLEVBQUUsYUFBYTtnQkFDMUIsSUFBSSxFQUFFLE1BQU07YUFDYjtZQUNEO2dCQUNFLEtBQUssRUFBRSxNQUFNO2dCQUNiLFdBQVcsRUFBRSxtQkFBbUI7Z0JBQ2hDLElBQUksRUFBRSxNQUFNO2FBQ2I7WUFDRDtnQkFDRSxJQUFJLEVBQUUsV0FBVzthQUNsQjtZQUNEO2dCQUNFLEtBQUssRUFBRSxLQUFLO2dCQUNaLFdBQVcsRUFBRSxhQUFhO2dCQUMxQixJQUFJLEVBQUUsS0FBSzthQUNaO1lBQ0Q7Z0JBQ0UsS0FBSyxFQUFFLE1BQU07Z0JBQ2IsV0FBVyxFQUFFLGFBQWE7Z0JBQzFCLElBQUksRUFBRSxNQUFNO2FBQ2I7WUFDRDtnQkFDRSxLQUFLLEVBQUUsT0FBTztnQkFDZCxXQUFXLEVBQUUsYUFBYTtnQkFDMUIsSUFBSSxFQUFFLE9BQU87YUFDZDtZQUNEO2dCQUNFLEtBQUssRUFBRSxZQUFZO2dCQUNuQixXQUFXLEVBQUUsYUFBYTtnQkFDMUIsSUFBSSxFQUFFLFdBQVc7YUFDbEI7U0FDRjtLQUNGO0lBQ0Q7UUFDRSxLQUFLLEVBQUUsTUFBTTtRQUNiLE9BQU8sRUFBRTtZQUNQO2dCQUNFLEtBQUssRUFBRSxRQUFRO2dCQUNmLFdBQVcsRUFBRSxhQUFhO2dCQUMxQixLQUFLLEVBQUUsVUFBUyxJQUFJLEVBQUUsYUFBYTtvQkFDakMsSUFBSSxhQUFhO3dCQUFFLGFBQWEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztnQkFDNUMsQ0FBQzthQUNGO1lBQ0Q7Z0JBQ0UsS0FBSyxFQUFFLG9CQUFvQjtnQkFDM0IsV0FBVyxFQUFFLENBQUM7b0JBQ1osSUFBSSxPQUFPLENBQUMsUUFBUSxLQUFLLFFBQVE7d0JBQUUsT0FBTyxnQkFBZ0IsQ0FBQzs7d0JBQ3RELE9BQU8sS0FBSyxDQUFDO2dCQUNwQixDQUFDLENBQUMsRUFBRTtnQkFDSixLQUFLLEVBQUUsVUFBUyxJQUFJLEVBQUUsYUFBYTtvQkFDakMsSUFBSSxhQUFhO3dCQUFFLGFBQWEsQ0FBQyxhQUFhLENBQUMsQ0FBQyxhQUFhLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztnQkFDaEYsQ0FBQzthQUNGO1lBQ0Q7Z0JBQ0UsS0FBSyxFQUFFLHdCQUF3QjtnQkFDL0IsV0FBVyxFQUFFLENBQUM7b0JBQ1osSUFBSSxPQUFPLENBQUMsUUFBUSxLQUFLLFFBQVE7d0JBQUUsT0FBTyxlQUFlLENBQUM7O3dCQUNyRCxPQUFPLGNBQWMsQ0FBQztnQkFDN0IsQ0FBQyxDQUFDLEVBQUU7Z0JBQ0osS0FBSyxFQUFFLFVBQVMsSUFBSSxFQUFFLGFBQWE7b0JBQ2pDLElBQUksYUFBYTt3QkFBRSxhQUFhLENBQUMsV0FBVyxDQUFDLGNBQWMsRUFBRSxDQUFDO2dCQUNoRSxDQUFDO2FBQ0Y7U0FDRjtLQUNGO0NBQ0YsQ0FBQztBQUVGLFNBQWdCLDJCQUEyQixDQUFDLElBQUk7SUFDOUMsSUFBSSxJQUFJLENBQUMsSUFBSSxHQUFHLGFBQWEsRUFBRTtRQUM3QixPQUFPLEtBQUssQ0FBQztLQUNkO0lBQ0QsT0FBTyxDQUFDLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLENBQUMsQ0FBQztBQUMxRCxDQUFDO0FBTEQsa0VBS0M7QUFFRCxTQUFnQix5QkFBeUIsQ0FBQyxRQUFRO0lBQ2hELE1BQU0sS0FBSyxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDeEMsTUFBTSxRQUFRLEdBQUcscUJBQXFCLENBQUMsY0FBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBRS9ELElBQUksUUFBUSxLQUFLLFdBQVcsRUFBRTtRQUM1QixNQUFNLFVBQVUsR0FBRyxNQUFNLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztRQUM3QyxVQUFVLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ2pDLE9BQU87S0FDUjtJQUVELElBQUksa0JBQWtCLEtBQUssSUFBSSxFQUFFO1FBQy9CLE1BQU0sRUFBRSxhQUFhLEVBQUUsR0FBRyxPQUFPLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUN0RCxrQkFBa0IsR0FBRyxJQUFJLGFBQWEsQ0FBQztZQUNyQyxLQUFLLEVBQUUsR0FBRztZQUNWLE1BQU0sRUFBRSxHQUFHO1lBQ1gsTUFBTSxFQUFFLElBQUk7WUFDWixXQUFXLEVBQUUsSUFBSTtZQUNqQixlQUFlLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLE1BQU07WUFDM0MsY0FBYyxFQUFFO2dCQUNkLE9BQU8sRUFBRSxjQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxZQUFZLENBQUM7Z0JBQzNDLGVBQWUsRUFBRSxLQUFLO2dCQUN0QixnQkFBZ0IsRUFBRSxJQUFJO2FBQ3ZCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsa0JBQWtCLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxHQUFHLEVBQUU7WUFDckMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDO1FBQzVCLENBQUMsQ0FBQyxDQUFDO1FBQ0gsa0JBQWtCLENBQUMsT0FBTyxDQUN4QixPQUFPLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMseUJBQXlCLENBQUMsQ0FDOUUsQ0FBQztLQUNIO1NBQU07UUFDTCxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQztLQUMzQjtJQUNELGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxjQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7SUFFckQsSUFBSSxLQUFLLEVBQUU7UUFDVCxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsY0FBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsaUJBQWlCLENBQUMsRUFBRTtZQUNuRSxNQUFNLEVBQUUsUUFBUSxrQkFBa0IsQ0FBQyxVQUFVLFFBQVEsRUFBRSxDQUFDLEVBQUU7U0FDM0QsQ0FBQyxDQUFDO0tBQ0o7U0FBTTtRQUNMLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxjQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxlQUFlLENBQUMsRUFBRTtZQUNqRSxNQUFNLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxDQUFDO1NBQ2hFLENBQUMsQ0FBQztLQUNKO0FBQ0gsQ0FBQztBQTVDRCw4REE0Q0M7QUFFTSxLQUFLLFVBQVUsZUFBZSxDQUFDLEVBQ3BDLElBQUksRUFDSixRQUFRLEVBQ1IsV0FBVyxHQUtaO0lBQ0MsTUFBTSxRQUFRLEdBQUcscUJBQXFCLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLENBQUMsQ0FBQztJQUVoRSxJQUFJLFFBQVEsS0FBSyxXQUFXLEVBQUU7UUFDNUIsT0FBTyxNQUFNLHlCQUF5QixDQUFDLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztLQUN0RDtTQUFNLElBQUksUUFBUSxFQUFFO1FBQ25CLE9BQU8sTUFBTSw2QkFBNkIsQ0FBQyxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7S0FDdkY7U0FBTTtRQUNMLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7QUFDSCxDQUFDO0FBbEJELDBDQWtCQztBQUlELEtBQUssVUFBVSw2QkFBNkIsQ0FBQyxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLFFBQVEsRUFBRTtJQUNwRixPQUFPLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQzNCLFlBQVksQ0FBQyxJQUFJLENBQUMsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUV0RSxJQUFJLENBQUMsYUFBYSxJQUFJLGFBQWEsQ0FBQyxXQUFXLEVBQUUsRUFBRTtZQUNqRCxhQUFhLEdBQUcsb0JBQW9CLEVBQUUsQ0FBQztZQUN2QyxpQ0FBaUMsRUFBRSxDQUFDO1NBQ3JDO0lBQ0gsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsU0FBUyxvQkFBb0I7SUFDM0IsTUFBTSxFQUFFLGFBQWEsRUFBRSxHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3RELE1BQU0sR0FBRyxHQUFHLElBQUksYUFBYSxDQUFDO1FBQzVCLEtBQUssRUFBRSxjQUFjO1FBQ3JCLE1BQU0sRUFBRSxjQUFjO1FBQ3RCLElBQUksRUFBRSxLQUFLO1FBQ1gsY0FBYyxFQUFFO1lBQ2QsT0FBTyxFQUFFLGNBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFlBQVksQ0FBQztZQUMzQyxlQUFlLEVBQUUsS0FBSztZQUN0QixnQkFBZ0IsRUFBRSxJQUFJO1NBQ3ZCO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRTtRQUNqQyxPQUFPLENBQUMsSUFBSSxDQUFDLDJDQUEyQyxDQUFDLENBQUM7UUFDMUQsSUFBSSxhQUFhLEtBQUssR0FBRztZQUFFLGFBQWEsR0FBRyxJQUFJLENBQUM7UUFDaEQsR0FBRyxDQUFDLE9BQU8sRUFBRSxDQUFDO0lBQ2hCLENBQUMsQ0FBQyxDQUFDO0lBQ0gsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsR0FBRyxFQUFFO1FBQ3RCLElBQUksYUFBYSxLQUFLLEdBQUc7WUFBRSxhQUFhLEdBQUcsSUFBSSxDQUFDO0lBQ2xELENBQUMsQ0FBQyxDQUFDO0lBQ0gsT0FBTyxHQUFHLENBQUM7QUFDYixDQUFDO0FBRUQsU0FBUyxpQ0FBaUM7SUFDeEMsSUFBSSxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtRQUM3QixJQUFJLGFBQWEsSUFBSSxDQUFDLGFBQWEsQ0FBQyxXQUFXLEVBQUUsRUFBRTtZQUNqRCxhQUFhLENBQUMsT0FBTyxFQUFFLENBQUM7U0FDekI7YUFBTTtZQUNMLE9BQU8sQ0FBQyxJQUFJLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNoRjtRQUNELGFBQWEsR0FBRyxJQUFJLENBQUM7UUFDckIsT0FBTztLQUNSO0lBRUQsTUFBTSxFQUFFLFFBQVEsRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLE9BQU8sRUFBRSxHQUFHLFlBQVksQ0FBQyxHQUFHLEVBQUUsQ0FBQztJQUd4RSxhQUFhLENBQUMsUUFBUSxDQUFDLGNBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGVBQWUsQ0FBQyxFQUFFO1FBQzVELE1BQU0sRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSxDQUFDO0tBQzdFLENBQUMsQ0FBQztJQUlILElBQUksVUFBVSxHQUFHLElBQUksQ0FBQztJQUV0QixNQUFNLEtBQUssR0FBRyxVQUFVLENBQUMsR0FBRyxFQUFFO1FBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsc0NBQXNDLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDL0QsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3BCLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUVULE1BQU0saUJBQWlCLEdBQUcsR0FBRyxFQUFFO1FBQzdCLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNuQixDQUFDLENBQUM7SUFFRixVQUFVLEdBQUcsT0FBTyxDQUFDLEVBQUU7UUFDckIsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3BCLElBQUksYUFBYSxFQUFFO1lBQ2pCLGFBQWEsQ0FBQyxjQUFjLENBQUMsb0JBQW9CLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztTQUN2RTtRQUNELE9BQU8sQ0FBQyxRQUFRLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUNwRCxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDbkIsQ0FBQyxDQUFDO0lBRUYsYUFBYSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0FBQzlELENBQUM7QUFFRCxLQUFLLFVBQVUseUJBQXlCLENBQUMsRUFBRSxRQUFRLEVBQXdCO0lBQ3pFLE1BQU0sU0FBUyxHQUFHLGNBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM5RCxNQUFNLFVBQVUsR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUVqRCxPQUFPLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQzNCLE1BQU0sR0FBRyxHQUFHLG1CQUFtQixDQUFDO1FBQ2hDLE1BQU0sSUFBSSxHQUFHO1lBQ1gsSUFBSTtZQUNKLElBQUk7WUFDSixHQUFHLE1BQU0sQ0FBQyxnQkFBZ0IsRUFBRTtZQUM1QixJQUFJO1lBQ0osR0FBRyxjQUFjLEVBQUU7WUFDbkIsSUFBSTtZQUNKLFNBQVM7WUFDVCxVQUFVO1NBQ1gsQ0FBQztRQUVGLHdCQUFRLENBQUMsR0FBRyxFQUFFLElBQUksRUFBRSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsTUFBTSxFQUFFLEVBQUU7WUFHNUMsSUFDRSxLQUFLO2dCQUNMLE1BQU0sQ0FBQyxLQUFLLENBQUMsdUJBQXVCLENBQUM7Z0JBQ3JDLENBQUMsTUFBTSxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDLEVBQ3REO2dCQUNBLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUNoQjtpQkFBTTtnQkFDTCxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDZjtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDIn0=