//--------------------------------------------------------------//
// Wipes.fx
//
// Notes : Each wipe pattern has its own separate technique/shader.
// There are also separate techniques/shaders for wipes with borders
// and wipes without (eg. Circle and CircleWithBorder).  This is
// because a separate logic path is required for bordered patterns
// which typically causes shaders to approach the 64 instruction
// limit)
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------//

int _LwksEffectInfo
<
   string EffectGroup = "GenericPixelShader";
   string Description = "Wipe";
   string Category    = "Mixes";
   bool CanSize       = true;
> = 0;

//--------------------------------------------------------------//
// Params
//--------------------------------------------------------------//

const float CentreX = 0.5;
const float CentreY = 0.5;

float4 BorderColour
<
   string Description = "Colour";
   string Group       = "Border";
> = { 1.0, 1.0, 1.0, 1.0 };

float BorderWidth
<
   string Description = "Size";
   string Group       = "Border";
   float MinVal = 0.0;
   float MaxVal = 1.0;
> = 0.0;

float EdgeSoftness
<
   string Description = "Edge Softness";
   float MinVal = 0.0;
   float MaxVal = 1.0;
> = 0.2;

float Progress
<
   string Description = "Progress";
   float MinVal = 0.0;
   float MaxVal = 1.0;
   float KF0    = 0.0;
   float KF1    = 1.0;
> = 0.0;

float _OutputAspectRatio = 1.0;

//--------------------------------------------------------------//
// Inputs
//--------------------------------------------------------------//
texture fg;
texture bg;

//--------------------------------------------------------------//
// Samplers
//--------------------------------------------------------------//
sampler FgSampler = sampler_state
{
   Texture  = <fg>;
   AddressU  = Border;
   AddressV  = Border;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;
};

sampler BgSampler = sampler_state
{
   Texture  = <bg>;
   AddressU  = Border;
   AddressV  = Border;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;
};

const float _ProgressComplete  = 1.0;
const float _HalfProgress      = 0.5;
const float _MaxSoftEdgeSize   = 0.2;
const float _MaxBorderSize     = 0.1;
const float _Cos45             = 0.7071;
const float _45                = 0.7854;
const float _Sqrt2             = 1.4142;

//--------------------------------------------------------------//
// Pixel Shaders
//--------------------------------------------------------------//
float4 V_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( Progress * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   if ( xyNorm.x > edgeSoftEnd )
   {
      ret = fg;
   }
   else if ( xyNorm.x < edgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( bg, fg, ( xyNorm.x - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 H_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( Progress * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   if ( xyNorm.y > edgeSoftEnd )
   {
      ret = fg;
   }
   else if ( xyNorm.y < edgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( bg, fg, ( xyNorm.y - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 HShutter_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float maxWidth      = max( CentreX, 1.0 - CentreX );
   float r             = CentreX + ( ( 1.0 - Progress ) * maxWidth );
   float edgeSoftEnd   = r + ( ( 1.0 - Progress ) * edgeSoftness );
   float edgeSoftStart = edgeSoftEnd - edgeSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // If this pixel is left of the middle, pretend its right of the middle for simplicity
   if ( xyNorm.x < CentreX )
   {
      xyNorm.x = ( 2 * CentreX ) - xyNorm.x;
   }

   if ( xyNorm.x < edgeSoftStart )
   {
      ret = fg;
   }
   else if ( xyNorm.x >= edgeSoftEnd )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( fg, bg, ( xyNorm.x - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VShutter_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float maxHeight     = max( CentreY, 1.0 - CentreY );
   float b             = CentreY + ( ( 1.0 - Progress ) * maxHeight );
   float edgeSoftEnd   = b + ( ( 1.0 - Progress ) * edgeSoftness );
   float edgeSoftStart = edgeSoftEnd - edgeSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // If this pixel is left of the middle, pretend its right of the middle for simplicity
   if ( xyNorm.y < CentreY )
   {
      xyNorm.y = ( 2 * CentreY ) - xyNorm.y;
   }

   if ( xyNorm.y < edgeSoftStart )
   {
      ret = fg;
   }
   else if ( xyNorm.y >= edgeSoftEnd )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( fg, bg, ( xyNorm.y - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CornerTL_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float xEdgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float yEdgeSoftness  = xEdgeSoftness * _OutputAspectRatio;
   float xEdgeSoftStart = ( Progress * ( _ProgressComplete + xEdgeSoftness ) ) - xEdgeSoftness;
   float xEdgeSoftEnd   = xEdgeSoftStart + xEdgeSoftness;
   float yEdgeSoftStart = ( Progress * ( _ProgressComplete + yEdgeSoftness ) ) - yEdgeSoftness;
   float yEdgeSoftEnd   = yEdgeSoftStart + yEdgeSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can do the same code as for BR wipe..
   xyNorm = 1.0 - xyNorm;

   if ( xyNorm.x > xEdgeSoftEnd && xyNorm.y > yEdgeSoftEnd )
   {
      ret = fg;
   }
   else if ( xyNorm.x < xEdgeSoftStart || xyNorm.y < yEdgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      float pixelSoftness;

      if ( xyNorm.y < yEdgeSoftEnd )
      {
         if ( xyNorm.x < xEdgeSoftEnd )
         {
            pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
            pixelSoftness = min( pixelSoftness, ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness );
         }
         else
         {
            pixelSoftness = ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness;
         }
      }
      else
      {
         pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
      }

      ret = lerp( bg, fg, pixelSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CornerTR_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float xEdgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float yEdgeSoftness  = xEdgeSoftness * _OutputAspectRatio;
   float xEdgeSoftStart = ( Progress * ( _ProgressComplete + xEdgeSoftness ) ) - xEdgeSoftness;
   float xEdgeSoftEnd   = xEdgeSoftStart + xEdgeSoftness;
   float yEdgeSoftStart = ( Progress * ( _ProgressComplete + yEdgeSoftness ) ) - yEdgeSoftness;
   float yEdgeSoftEnd   = yEdgeSoftStart + yEdgeSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can do the same code as for BR wipe..
   xyNorm.y = 1.0 - xyNorm.y;

   if ( xyNorm.x > xEdgeSoftEnd && xyNorm.y > yEdgeSoftEnd )
   {
      ret = fg;
   }
   else if ( xyNorm.x < xEdgeSoftStart || xyNorm.y < yEdgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      float pixelSoftness;

      if ( xyNorm.y < yEdgeSoftEnd )
      {
         if ( xyNorm.x < xEdgeSoftEnd )
         {
            pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
            pixelSoftness = min( pixelSoftness, ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness );
         }
         else
         {
            pixelSoftness = ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness;
         }
      }
      else
      {
         pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
      }

      ret = lerp( bg, fg, pixelSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CornerBR_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float xEdgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float yEdgeSoftness  = xEdgeSoftness * _OutputAspectRatio;
   float xEdgeSoftStart = ( Progress * ( _ProgressComplete + xEdgeSoftness ) ) - xEdgeSoftness;
   float xEdgeSoftEnd   = xEdgeSoftStart + xEdgeSoftness;
   float yEdgeSoftStart = ( Progress * ( _ProgressComplete + yEdgeSoftness ) ) - yEdgeSoftness;
   float yEdgeSoftEnd   = yEdgeSoftStart + yEdgeSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   if ( xyNorm.x > xEdgeSoftEnd && xyNorm.y > yEdgeSoftEnd )
   {
      ret = fg;
   }
   else if ( xyNorm.x < xEdgeSoftStart || xyNorm.y < yEdgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      float pixelSoftness;

      if ( xyNorm.y < yEdgeSoftEnd )
      {
         if ( xyNorm.x < xEdgeSoftEnd )
         {
            pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
            pixelSoftness = min( pixelSoftness, ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness );
         }
         else
         {
            pixelSoftness = ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness;
         }
      }
      else
      {
         pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
      }

      ret = lerp( bg, fg, pixelSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CornerBL_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float xEdgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float yEdgeSoftness  = xEdgeSoftness * _OutputAspectRatio;
   float xEdgeSoftStart = ( Progress * ( _ProgressComplete + xEdgeSoftness ) ) - xEdgeSoftness;
   float xEdgeSoftEnd   = xEdgeSoftStart + xEdgeSoftness;
   float yEdgeSoftStart = ( Progress * ( _ProgressComplete + yEdgeSoftness ) ) - yEdgeSoftness;
   float yEdgeSoftEnd   = yEdgeSoftStart + yEdgeSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can do the same code as for BR wipe..
   xyNorm.x = 1.0 - xyNorm.x;

   if ( xyNorm.x > xEdgeSoftEnd && xyNorm.y > yEdgeSoftEnd )
   {
      ret = fg;
   }
   else if ( xyNorm.x < xEdgeSoftStart || xyNorm.y < yEdgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      float pixelSoftness;

      if ( xyNorm.y < yEdgeSoftEnd )
      {
         if ( xyNorm.x < xEdgeSoftEnd )
         {
            pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
            pixelSoftness = min( pixelSoftness, ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness );
         }
         else
         {
            pixelSoftness = ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness;
         }
      }
      else
      {
         pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
      }

      ret = lerp( bg, fg, pixelSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 Circle_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;
   float centreY = 1.0 - CentreY;
   float2 centrePos = float2( CentreX, centreY );
   float2 aspectAdjustedtexCoord = float2( xyNorm.x, ( ( xyNorm.y - centreY ) / _OutputAspectRatio ) + centreY );
   float distanceFromCentre = abs( distance( centrePos, aspectAdjustedtexCoord ) );

   // Calculate the radius of the circle when it is at its largest

   if ( centrePos.x < 0.5 )
   {
      centrePos.x = 1.0 - centrePos.x;
   }

   if ( centrePos.y < 0.5 )
   {
      centrePos.y = 1.0 - centrePos.y;
   }

   centrePos.y /= _OutputAspectRatio;

   float maxRadius = length( centrePos );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( ( 1.0 - Progress ) * maxRadius * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   if ( distanceFromCentre > edgeSoftEnd )
   {
      ret = bg;
   }
   else if ( distanceFromCentre < edgeSoftStart )
   {
      ret = fg;
   }
   else
   {
      ret = lerp( fg, bg, ( distanceFromCentre - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 Diag2_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Read the fg & bg pixels (before wwe mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( ( 1.0 - Progress ) * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm ) / _Sqrt2; // adjust by _Sqrt2 due to diagonal size of image being > 1.0
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > edgeSoftEnd )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < edgeSoftStart )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness = ( rotatedTexCoord.y - edgeSoftStart ) / edgeSoftness;
      ret = lerp( fg, bg, pixelSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 Diag1_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Read the fg & bg pixels (before wwe mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can do exactly the same code as for Diag1..
   xyNorm.y = 1 - xyNorm.y;

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( ( 1.0 - Progress ) * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm ) / _Sqrt2; // adjust by _Sqrt2 due to diagonal size of image being > 1.0
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > edgeSoftEnd )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < edgeSoftStart )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness = ( rotatedTexCoord.y - edgeSoftStart ) / edgeSoftness;
      ret = lerp( fg, bg, pixelSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VeeThree_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Read the fg & bg pixels (before we mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( Progress * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   if ( xyNorm.y > 0.5 )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm );
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > edgeSoftEnd )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.y < edgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( bg, fg, ( rotatedTexCoord.y - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VeeTwo_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Read the fg & bg pixels (before we mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( Progress * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   if ( xyNorm.y > 0.5 )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }
   xyNorm.x = 1.0 - xyNorm.x;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm );
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > edgeSoftEnd )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.y < edgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( bg, fg, ( rotatedTexCoord.y - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VeeFour_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Read the fg & bg pixels (before we mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( Progress * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   if ( xyNorm.x > 0.5 )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm );
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a - _45 ), d * sin( a - _45 ) );

   if ( rotatedTexCoord.x > edgeSoftEnd )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.x < edgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( bg, fg, ( rotatedTexCoord.x - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VeeOne_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Read the fg & bg pixels (before we mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( Progress * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   if ( xyNorm.x > 0.5 )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }
   xyNorm.y = 1.0 - xyNorm.y;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm );
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a - _45 ), d * sin( a - _45 ) );

   if ( rotatedTexCoord.x > edgeSoftEnd )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.x < edgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( bg, fg, ( rotatedTexCoord.x - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 Cross_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float xEdgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float yEdgeSoftness  = xEdgeSoftness * _OutputAspectRatio;
   float xEdgeSoftStart = CentreX + ( Progress * ( _HalfProgress + xEdgeSoftness ) ) - xEdgeSoftness;
   float xEdgeSoftEnd   = xEdgeSoftStart + xEdgeSoftness;
   float yEdgeSoftStart = CentreX + ( Progress * ( _HalfProgress + yEdgeSoftness ) ) - yEdgeSoftness;
   float yEdgeSoftEnd   = yEdgeSoftStart + yEdgeSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can pretend this is pixel is in the BR quadrant
   if ( xyNorm.x < CentreX )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }

   if ( xyNorm.y < CentreY )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }

   if ( xyNorm.x > xEdgeSoftEnd && xyNorm.y > yEdgeSoftEnd )
   {
      ret = fg;
   }
   else if ( xyNorm.x < xEdgeSoftStart || xyNorm.y < yEdgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      float pixelSoftness;

      if ( xyNorm.y < yEdgeSoftEnd )
      {
         if ( xyNorm.x < xEdgeSoftEnd )
         {
            pixelSoftness = ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness;
            pixelSoftness = min( pixelSoftness, ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness );
         }
         else
         {
            pixelSoftness = ( xyNorm.y - yEdgeSoftStart ) / yEdgeSoftness;
         }
      }
      else
      {
         pixelSoftness = ( xyNorm.x - xEdgeSoftStart ) / xEdgeSoftness;
      }

      ret = ( pixelSoftness * fg ) + ( ( 1.0 - pixelSoftness ) * bg );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 BowTie_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Read the fg & bg pixels (before wwe mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = 1.0 - ( ( Progress * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness );
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   // Each quadrant is some kind of reflection of the bottom-right quadrant :

   if ( xyNorm.x > CentreX )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }

   if ( xyNorm.y < CentreY )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }

   xyNorm.y /= _OutputAspectRatio;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm ) / _Sqrt2 * 2;
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > edgeSoftEnd )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < edgeSoftStart )
   {
      ret = fg;
   }
   else
   {
      ret = lerp( fg, bg, ( rotatedTexCoord.y - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 Square_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Figure out what we need to know
   float maxSquareW    = max( CentreX, 1.0 - CentreX );
   float maxSquareH    = max( CentreY, 1.0 - CentreY );
   float maxSquareSize = 2 * max( maxSquareW, maxSquareH );
   float xEdgeSoftness = EdgeSoftness * _MaxBorderSize;
   float yEdgeSoftness = xEdgeSoftness * _OutputAspectRatio;
   float sqW           = ( 1.0 - Progress ) * maxSquareSize;
   float sqH           = sqW * _OutputAspectRatio;
   float r             = CentreX + ( sqW / 2 );
   float b             = CentreY + ( sqH / 2 );

   float rSoftEnd   = r + ( ( 1.0 - Progress ) * xEdgeSoftness );
   float rSoftStart = rSoftEnd - xEdgeSoftness;
   float bSoftEnd   = b + ( ( 1.0 - Progress ) * yEdgeSoftness );
   float bSoftStart = bSoftEnd - yEdgeSoftness;

   // Read the source pixels (before we adjust xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Reflect pixel coords such that we appear to be dealing the the bottom-right
   // quadrant only (this enables us to use less shader instructions and hence
   // perform the wipe in a single pass)
   if ( xyNorm.x < CentreX )
   {
      xyNorm.x = ( 2 * CentreX ) - xyNorm.x;
   }
   if ( xyNorm.y < CentreY )
   {
      xyNorm.y = ( 2 * CentreY ) - xyNorm.y;
   }

   if ( xyNorm.x > rSoftEnd || xyNorm.y > bSoftEnd )
   {
      ret = bg;
   }
   else if ( xyNorm.x <= rSoftStart && xyNorm.y <= bSoftStart )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness;

      bool xSoft = xyNorm.x > rSoftStart;

      if ( xSoft && ( xyNorm.y > bSoftStart ) ) // Bottom right corner
      {
         pixelSoftness = ( xyNorm.x - rSoftStart ) / xEdgeSoftness;
         pixelSoftness = max( pixelSoftness, ( xyNorm.y - bSoftStart ) / yEdgeSoftness );
      }
      else
      {
         if ( xSoft )
         {
            pixelSoftness = ( xyNorm.x - rSoftStart ) / xEdgeSoftness;
         }
         else
         {
            pixelSoftness = ( xyNorm.y - bSoftStart ) / yEdgeSoftness;
         }
      }

      ret = lerp( fg, bg, pixelSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 Diamond_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float edgeSoftness  = EdgeSoftness * _MaxSoftEdgeSize;
   float edgeSoftStart = ( Progress * ( _ProgressComplete + edgeSoftness ) ) - edgeSoftness;
   float edgeSoftEnd   = edgeSoftStart + edgeSoftness;

   // Read the fg & bg pixels (before wwe mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Each quadrant is some kind of reflection of the bottom-right quadrant :

   if ( xyNorm.x > CentreX )
   {
      xyNorm.x = ( CentreX * 2 ) - xyNorm.x;
   }

   if ( xyNorm.y > CentreY )
   {
      xyNorm.y = ( CentreY * 2 ) - xyNorm.y;
   }

   xyNorm.y /= _OutputAspectRatio;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm ) / _Sqrt2 * 2;
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > edgeSoftEnd )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.y < edgeSoftStart )
   {
      ret = bg;
   }
   else
   {
      ret = lerp( bg, fg, ( rotatedTexCoord.y - edgeSoftStart ) / edgeSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float borderWidth  = BorderWidth * _MaxBorderSize;
   float borderStart = ( Progress * ( _ProgressComplete + borderWidth ) ) - borderWidth;
   float borderEnd   = borderStart + borderWidth;

   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( borderStart * ( _ProgressComplete + borderSoftness ) ) - borderSoftness;
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftStart2 = ( borderEnd * ( _ProgressComplete + borderSoftness ) ) - borderSoftness;
   float borderSoftEnd2   = borderSoftStart2 + borderSoftness;

   if ( xyNorm.x > borderSoftEnd2 )
   {
      ret = fg;
   }
   else if ( xyNorm.x < borderSoftStart1 )
   {
      ret = bg;
   }
   else if ( xyNorm.x > borderSoftEnd1 && xyNorm.x < borderSoftStart2 )
   {
      ret = BorderColour;
   }
   else if ( xyNorm.x > borderSoftStart2 )
   {
      ret = lerp( BorderColour, fg, ( xyNorm.x - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = lerp( bg, BorderColour, ( xyNorm.x - borderSoftStart1 ) / borderSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 HWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float borderWidth  = BorderWidth * _MaxBorderSize;
   float borderStart = ( Progress * ( _ProgressComplete + borderWidth ) ) - borderWidth;
   float borderEnd   = borderStart + borderWidth;

   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( borderStart * ( _ProgressComplete + borderSoftness ) ) - borderSoftness;
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftStart2 = ( borderEnd * ( _ProgressComplete + borderSoftness ) ) - borderSoftness;
   float borderSoftEnd2   = borderSoftStart2 + borderSoftness;

   if ( xyNorm.y > borderSoftEnd2 )
   {
      ret = fg;
   }
   else if ( xyNorm.y < borderSoftStart1 )
   {
      ret = bg;
   }
   else if ( xyNorm.y > borderSoftEnd1 && xyNorm.y < borderSoftStart2 )
   {
      ret = BorderColour;
   }
   else if ( xyNorm.y > borderSoftStart2 )
   {
      ret = lerp( BorderColour, fg, ( xyNorm.y - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = lerp( bg, BorderColour, ( xyNorm.y - borderSoftStart1 ) / borderSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 HShutterWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float maxWidth         = max( CentreX, 1.0 - CentreX );
   float r                = CentreX + ( ( 1.0 - Progress ) * maxWidth );
   float borderSoftEnd2   = r + ( ( 1.0 - Progress ) * ( borderWidth + borderSoftness ) );
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;
   float borderSoftEnd1   = borderSoftStart2 - borderWidth;
   float borderSoftStart1 = borderSoftEnd1 - borderSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // If this pixel is left of the middle, pretend its right of the middle for simplicity
   if ( xyNorm.x < CentreX )
   {
      xyNorm.x = ( 2 * CentreX ) - xyNorm.x;
   }

   if ( xyNorm.x >= borderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( xyNorm.x < borderSoftStart1 )
   {
      ret = fg;
   }
   else if ( xyNorm.x > borderSoftEnd1 && xyNorm.x < borderSoftStart2 )
   {
      ret = BorderColour;
   }
   else if ( xyNorm.x > borderSoftStart2 )
   {
      ret = lerp( BorderColour, bg, ( xyNorm.x - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = lerp( fg, BorderColour, ( xyNorm.x - borderSoftStart1 ) / borderSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VShutterWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float maxHeight        = max( CentreY, 1.0 - CentreY );
   float b                = CentreY + ( ( 1.0 - Progress ) * maxHeight );
   float borderSoftEnd2   = b + ( ( 1.0 - Progress ) * ( borderWidth + borderSoftness ) );
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;
   float borderSoftEnd1   = borderSoftStart2 - borderWidth;
   float borderSoftStart1 = borderSoftEnd1 - borderSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // If this pixel is above the middle, pretend its below the middle for simplicity
   if ( xyNorm.y < CentreY )
   {
      xyNorm.y = ( 2 * CentreY ) - xyNorm.y;
   }

   if ( xyNorm.y >= borderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( xyNorm.y < borderSoftStart1 )
   {
      ret = fg;
   }
   else if ( xyNorm.y > borderSoftEnd1 && xyNorm.y < borderSoftStart2 )
   {
      ret = BorderColour;
   }
   else if ( xyNorm.y > borderSoftStart2 )
   {
      ret = lerp( BorderColour, bg, ( xyNorm.y - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = lerp( fg, BorderColour, ( xyNorm.y - borderSoftStart1 ) / borderSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CornerTLWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Figure out what we need to know
   float xBorderWidth      = BorderWidth * _MaxBorderSize;
   float yBorderWidth      = xBorderWidth * _OutputAspectRatio;
   float xBorderSoftness   = xBorderWidth * EdgeSoftness;
   float yBorderSoftness   = xBorderSoftness * _OutputAspectRatio;
   float xBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _ProgressComplete + xBorderSoftness + xBorderWidth ) );
   float xBorderSoftEnd1   = xBorderSoftEnd2 - xBorderWidth;
   float xBorderSoftStart1 = xBorderSoftEnd1 - xBorderSoftness;
   float xBorderSoftStart2 = xBorderSoftStart1 + xBorderWidth;
   float yBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _ProgressComplete + yBorderSoftness + yBorderWidth ) );
   float yBorderSoftEnd1   = yBorderSoftEnd2 - yBorderWidth;
   float yBorderSoftStart1 = yBorderSoftEnd1 - yBorderSoftness;
   float yBorderSoftStart2 = yBorderSoftStart1 + yBorderWidth;

   // Read the source pixels (before we adjust xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   if ( xyNorm.x > xBorderSoftEnd2 || xyNorm.y > yBorderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( xyNorm.x <= xBorderSoftStart1 && xyNorm.y <= yBorderSoftStart1 )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness;

      // Is the current-pixel in an OUTER edge softness area?
      bool xSoft = xyNorm.x > xBorderSoftStart2;
      bool ySoft = xyNorm.y > yBorderSoftStart2;

      if ( xSoft || ySoft )
      {
         if ( xSoft && ySoft ) // Outer bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness );
         }
         else if ( xSoft )     // Outer right edge
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
         }
         else                  // Outer bottom edge
         {
            pixelSoftness = ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness;
         }

         ret = lerp( BorderColour, bg, pixelSoftness );
      }
      else if ( xyNorm.x > xBorderSoftEnd1 || xyNorm.y > yBorderSoftEnd1 ) // Solid border area
      {
         ret = BorderColour;
      }
      else // INNER soft edge area
      {
         bool xSoft = xyNorm.x > xBorderSoftStart1;
         bool ySoft = xyNorm.y > yBorderSoftStart1;

         if ( xSoft && ySoft )  // Inner bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness );
         }
         else
         {
            if ( xSoft ) // Inner right edge
            {
               pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            }
            else         // Outer right edge
            {
               pixelSoftness = ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness;
            }
         }

         ret = lerp( fg, BorderColour, pixelSoftness );
      }
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CornerTRWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Figure out what we need to know
   float xBorderWidth      = BorderWidth * _MaxBorderSize;
   float yBorderWidth      = xBorderWidth * _OutputAspectRatio;
   float xBorderSoftness   = xBorderWidth * EdgeSoftness;
   float yBorderSoftness   = xBorderSoftness * _OutputAspectRatio;
   float xBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _ProgressComplete + xBorderSoftness + xBorderWidth ) );
   float xBorderSoftEnd1   = xBorderSoftEnd2 - xBorderWidth;
   float xBorderSoftStart1 = xBorderSoftEnd1 - xBorderSoftness;
   float xBorderSoftStart2 = xBorderSoftStart1 + xBorderWidth;
   float yBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _ProgressComplete + yBorderSoftness + yBorderWidth ) );
   float yBorderSoftEnd1   = yBorderSoftEnd2 - yBorderWidth;
   float yBorderSoftStart1 = yBorderSoftEnd1 - yBorderSoftness;
   float yBorderSoftStart2 = yBorderSoftStart1 + yBorderWidth;

   // Read the source pixels (before we adjust xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can do the same code as for BR wipe..
   xyNorm.x = 1.0 - xyNorm.x;

   if ( xyNorm.x > xBorderSoftEnd2 || xyNorm.y > yBorderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( xyNorm.x <= xBorderSoftStart1 && xyNorm.y <= yBorderSoftStart1 )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness;

      // Is the current-pixel in an OUTER edge softness area?
      bool xSoft = xyNorm.x > xBorderSoftStart2;
      bool ySoft = xyNorm.y > yBorderSoftStart2;

      if ( xSoft || ySoft )
      {
         if ( xSoft && ySoft ) // Outer bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness );
         }
         else if ( xSoft )     // Outer right edge
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
         }
         else                  // Outer bottom edge
         {
            pixelSoftness = ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness;
         }

         ret = lerp( BorderColour, bg, pixelSoftness );
      }
      else if ( xyNorm.x > xBorderSoftEnd1 || xyNorm.y > yBorderSoftEnd1 ) // Solid border area
      {
         ret = BorderColour;
      }
      else // INNER soft edge area
      {
         bool xSoft = xyNorm.x > xBorderSoftStart1;
         bool ySoft = xyNorm.y > yBorderSoftStart1;

         if ( xSoft && ySoft )  // Inner bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness );
         }
         else
         {
            if ( xSoft ) // Inner right edge
            {
               pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            }
            else         // Outer right edge
            {
               pixelSoftness = ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness;
            }
         }

         ret = lerp( fg, BorderColour, pixelSoftness );
      }
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CornerBRWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Figure out what we need to know
   float xBorderWidth      = BorderWidth * _MaxBorderSize;
   float yBorderWidth      = xBorderWidth * _OutputAspectRatio;
   float xBorderSoftness   = xBorderWidth * EdgeSoftness;
   float yBorderSoftness   = xBorderSoftness * _OutputAspectRatio;
   float xBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _ProgressComplete + xBorderSoftness + xBorderWidth ) );
   float xBorderSoftEnd1   = xBorderSoftEnd2 - xBorderWidth;
   float xBorderSoftStart1 = xBorderSoftEnd1 - xBorderSoftness;
   float xBorderSoftStart2 = xBorderSoftStart1 + xBorderWidth;
   float yBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _ProgressComplete + yBorderSoftness + yBorderWidth ) );
   float yBorderSoftEnd1   = yBorderSoftEnd2 - yBorderWidth;
   float yBorderSoftStart1 = yBorderSoftEnd1 - yBorderSoftness;
   float yBorderSoftStart2 = yBorderSoftStart1 + yBorderWidth;

   // Read the source pixels (before we adjust xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can do the same code as for TR wipe..
   xyNorm = 1.0 - xyNorm;

   if ( xyNorm.x > xBorderSoftEnd2 || xyNorm.y > yBorderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( xyNorm.x <= xBorderSoftStart1 && xyNorm.y <= yBorderSoftStart1 )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness;

      // Is the current-pixel in an OUTER edge softness area?
      bool xSoft = xyNorm.x > xBorderSoftStart2;
      bool ySoft = xyNorm.y > yBorderSoftStart2;

      if ( xSoft || ySoft )
      {
         if ( xSoft && ySoft ) // Outer bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness );
         }
         else if ( xSoft )     // Outer right edge
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
         }
         else                  // Outer bottom edge
         {
            pixelSoftness = ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness;
         }

         ret = lerp( BorderColour, bg, pixelSoftness );
      }
      else if ( xyNorm.x > xBorderSoftEnd1 || xyNorm.y > yBorderSoftEnd1 ) // Solid border area
      {
         ret = BorderColour;
      }
      else // INNER soft edge area
      {
         bool xSoft = xyNorm.x > xBorderSoftStart1;
         bool ySoft = xyNorm.y > yBorderSoftStart1;

         if ( xSoft && ySoft )  // Inner bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness );
         }
         else
         {
            if ( xSoft ) // Inner right edge
            {
               pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            }
            else         // Outer right edge
            {
               pixelSoftness = ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness;
            }
         }

         ret = lerp( fg, BorderColour, pixelSoftness );
      }
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CornerBLWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Figure out what we need to know
   float xBorderWidth      = BorderWidth * _MaxBorderSize;
   float yBorderWidth      = xBorderWidth * _OutputAspectRatio;
   float xBorderSoftness   = xBorderWidth * EdgeSoftness;
   float yBorderSoftness   = xBorderSoftness * _OutputAspectRatio;
   float xBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _ProgressComplete + xBorderSoftness + xBorderWidth ) );
   float xBorderSoftEnd1   = xBorderSoftEnd2 - xBorderWidth;
   float xBorderSoftStart1 = xBorderSoftEnd1 - xBorderSoftness;
   float xBorderSoftStart2 = xBorderSoftStart1 + xBorderWidth;
   float yBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _ProgressComplete + yBorderSoftness + yBorderWidth ) );
   float yBorderSoftEnd1   = yBorderSoftEnd2 - yBorderWidth;
   float yBorderSoftStart1 = yBorderSoftEnd1 - yBorderSoftness;
   float yBorderSoftStart2 = yBorderSoftStart1 + yBorderWidth;

   // Read the source pixels (before we adjust xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can do the same code as for BR wipe..
   xyNorm.y = 1.0 - xyNorm.y;

   if ( xyNorm.x > xBorderSoftEnd2 || xyNorm.y > yBorderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( xyNorm.x <= xBorderSoftStart1 && xyNorm.y <= yBorderSoftStart1 )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness;

      // Is the current-pixel in an OUTER edge softness area?
      bool xSoft = xyNorm.x > xBorderSoftStart2;
      bool ySoft = xyNorm.y > yBorderSoftStart2;

      if ( xSoft || ySoft )
      {
         if ( xSoft && ySoft ) // Outer bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness );
         }
         else if ( xSoft )     // Outer right edge
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
         }
         else                  // Outer bottom edge
         {
            pixelSoftness = ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness;
         }

         ret = lerp( BorderColour, bg, pixelSoftness );
      }
      else if ( xyNorm.x > xBorderSoftEnd1 || xyNorm.y > yBorderSoftEnd1 ) // Solid border area
      {
         ret = BorderColour;
      }
      else // INNER soft edge area
      {
         bool xSoft = xyNorm.x > xBorderSoftStart1;
         bool ySoft = xyNorm.y > yBorderSoftStart1;

         if ( xSoft && ySoft )  // Inner bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness );
         }
         else
         {
            if ( xSoft ) // Inner right edge
            {
               pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            }
            else         // Outer right edge
            {
               pixelSoftness = ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness;
            }
         }

         ret = lerp( fg, BorderColour, pixelSoftness );
      }
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CircleWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;
   float centreY = 1.0 - CentreY;
   float2 centrePos = float2( CentreX, centreY );
   float2 aspectAdjustedtexCoord = float2( xyNorm.x, ( ( xyNorm.y - centreY ) / _OutputAspectRatio ) + centreY );
   float distanceFromCentre = abs( distance( centrePos, aspectAdjustedtexCoord ) );

   // Calculate the radius of the circle when it is at its largest

   if ( centrePos.x < 0.5 )
   {
      centrePos.x = 1.0 - centrePos.x;
   }

   if ( centrePos.y < 0.5 )
   {
      centrePos.y = 1.0 - centrePos.y;
   }

   centrePos.y /= _OutputAspectRatio;

   float maxRadius = length( centrePos );

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( ( 1.0 - Progress ) * maxRadius * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftStart1 + borderWidth + borderSoftness;;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   if ( distanceFromCentre >= borderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( distanceFromCentre <= borderSoftStart1 )
   {
      ret = fg;
   }
   else if ( distanceFromCentre >= borderSoftEnd1 && distanceFromCentre <= borderSoftStart2 )
   {
      ret = BorderColour;
   }
   else if ( distanceFromCentre > borderSoftStart2 )
   {
      ret = lerp( BorderColour, bg, (  distanceFromCentre - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = lerp( fg, BorderColour, ( distanceFromCentre - borderSoftStart1 ) / borderSoftness );
   }

   return ret;
}
//--------------------------------------------------------------//
float4 Diag2WithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( ( 1.0 - Progress ) * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftStart1 + borderWidth + borderSoftness;;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   // Read the fg & bg pixels (before wwe mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm ) / _Sqrt2; // adjust by _Sqrt2 due to diagonal size of image being > 1.0
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > borderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < borderSoftStart1 )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.y < borderSoftEnd1 )
   {
      ret = lerp( fg, BorderColour, ( rotatedTexCoord.y - borderSoftStart1 ) / borderSoftness );
   }
   else if ( rotatedTexCoord.y > borderSoftStart2 )
   {
      ret = lerp( BorderColour, bg, ( rotatedTexCoord.y - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = BorderColour;
   }

   return ret;
}
//--------------------------------------------------------------//
float4 Diag1WithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( ( 1.0 - Progress ) * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftStart1 + borderWidth + borderSoftness;;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   // Read the fg & bg pixels (before wwe mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can do exactly the same code as for Diag1..
   xyNorm.y = 1 - xyNorm.y;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm ) / _Sqrt2; // adjust by _Sqrt2 due to diagonal size of image being > 1.0
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > borderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < borderSoftStart1 )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.y < borderSoftEnd1 )
   {
      ret = lerp( fg, BorderColour, ( rotatedTexCoord.y - borderSoftStart1 ) / borderSoftness );
   }
   else if ( rotatedTexCoord.y > borderSoftStart2 )
   {
      ret = lerp( BorderColour, bg, ( rotatedTexCoord.y - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = BorderColour;
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VeeThreeWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( Progress * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftStart1 + borderWidth + borderSoftness;;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   // Read the fg & bg pixels (before we mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   if ( xyNorm.y > 0.5 )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm );
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > borderSoftEnd2 )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.y < borderSoftStart1 )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < borderSoftEnd1 )
   {
      ret = lerp( bg, BorderColour, ( rotatedTexCoord.y - borderSoftStart1 ) / borderSoftness );
   }
   else if ( rotatedTexCoord.y > borderSoftStart2 )
   {
      ret = lerp( BorderColour, fg, ( rotatedTexCoord.y - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = BorderColour;
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VeeTwoWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( Progress * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftStart1 + borderWidth + borderSoftness;;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   // Read the fg & bg pixels (before we mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   if ( xyNorm.y > 0.5 )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }
   xyNorm.x = 1.0 - xyNorm.x;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm );
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > borderSoftEnd2 )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.y < borderSoftStart1 )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < borderSoftEnd1 )
   {
      ret = lerp( bg, BorderColour, ( rotatedTexCoord.y - borderSoftStart1 ) / borderSoftness );
   }
   else if ( rotatedTexCoord.y > borderSoftStart2 )
   {
      ret = lerp( BorderColour, fg, ( rotatedTexCoord.y - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = BorderColour;
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VeeFourWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( Progress * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftStart1 + borderWidth + borderSoftness;;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   // Read the fg & bg pixels (before we mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   if ( xyNorm.x > 0.5 )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm );
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a - _45 ), d * sin( a - _45 ) );

   if ( rotatedTexCoord.x > borderSoftEnd2 )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.x < borderSoftStart1 )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.x < borderSoftEnd1 )
   {
      ret = lerp( bg, BorderColour, ( rotatedTexCoord.x - borderSoftStart1 ) / borderSoftness );
   }
   else if ( rotatedTexCoord.x > borderSoftStart2 )
   {
      ret = lerp( BorderColour, fg, ( rotatedTexCoord.x - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = BorderColour;
   }

   return ret;
}
//--------------------------------------------------------------//
float4 VeeOneWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( Progress * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftStart1 + borderWidth + borderSoftness;;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   // Read the fg & bg pixels (before we mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   if ( xyNorm.x > 0.5 )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }
   xyNorm.y = 1.0 - xyNorm.y;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm );
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a - _45 ), d * sin( a - _45 ) );

   if ( rotatedTexCoord.x > borderSoftEnd2 )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.x < borderSoftStart1 )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.x < borderSoftEnd1 )
   {
      ret = lerp( bg, BorderColour, ( rotatedTexCoord.x - borderSoftStart1 ) / borderSoftness );
   }
   else if ( rotatedTexCoord.x > borderSoftStart2 )
   {
      ret = lerp( BorderColour, fg, ( rotatedTexCoord.x - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = BorderColour;
   }

   return ret;
}
//--------------------------------------------------------------//
float4 CrossWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Figure out what we need to know
   float xBorderWidth      = BorderWidth * _MaxBorderSize;
   float yBorderWidth      = xBorderWidth * _OutputAspectRatio;
   float xBorderSoftness   = xBorderWidth * EdgeSoftness;
   float yBorderSoftness   = xBorderSoftness * _OutputAspectRatio;
   float xBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _HalfProgress + xBorderSoftness + xBorderWidth ) );
   float xBorderSoftEnd1   = xBorderSoftEnd2 - xBorderWidth;
   float xBorderSoftStart1 = xBorderSoftEnd1 - xBorderSoftness;
   float xBorderSoftStart2 = xBorderSoftStart1 + xBorderWidth;
   float yBorderSoftEnd2   = ( ( 1.0 - Progress ) * ( _HalfProgress + yBorderSoftness + yBorderWidth ) );
   float yBorderSoftEnd1   = yBorderSoftEnd2 - yBorderWidth;
   float yBorderSoftStart1 = yBorderSoftEnd1 - yBorderSoftness;
   float yBorderSoftStart2 = yBorderSoftStart1 + yBorderWidth;

   // Read the source pixels (before we adjust xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Fiddle with the xyNorm so that we can pretend this is pixel is in the TR quadrant
   if ( xyNorm.x > CentreX )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }

   if ( xyNorm.y > CentreY )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }

   if ( xyNorm.x > xBorderSoftEnd2 || xyNorm.y > yBorderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( xyNorm.x <= xBorderSoftStart1 && xyNorm.y <= yBorderSoftStart1 )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness;

      // Is the current-pixel in an OUTER edge softness area?
      bool xSoft = xyNorm.x > xBorderSoftStart2;
      bool ySoft = xyNorm.y > yBorderSoftStart2;

      if ( xSoft || ySoft )
      {
         if ( xSoft && ySoft ) // Outer bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness );
         }
         else if ( xSoft )     // Outer right edge
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
         }
         else                  // Outer bottom edge
         {
            pixelSoftness = ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness;
         }

         ret = lerp( BorderColour, bg, pixelSoftness );
      }
      else if ( xyNorm.x > xBorderSoftEnd1 || xyNorm.y > yBorderSoftEnd1 ) // Solid border area
      {
         ret = BorderColour;
      }
      else // INNER soft edge area
      {
         bool xSoft = xyNorm.x > xBorderSoftStart1;
         bool ySoft = xyNorm.y > yBorderSoftStart1;

         if ( xSoft && ySoft )  // Inner bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness );
         }
         else
         {
            if ( xSoft ) // Inner right edge
            {
               pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            }
            else         // Outer right edge
            {
               pixelSoftness = ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness;
            }
         }

         ret = lerp( fg, BorderColour, pixelSoftness );
      }
   }

   return ret;
}
//--------------------------------------------------------------//
float4 BowTieWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Read the fg & bg pixels (before wwe mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = 1.0 - ( ( Progress * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness ) );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftEnd1 + borderWidth;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   // Each quadrant is some kind of reflection of the bottom-right quadrant :

   if ( xyNorm.x > CentreX )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }

   if ( xyNorm.y < CentreY )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }

   xyNorm.y /= _OutputAspectRatio;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm ) / _Sqrt2 * 2;
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > borderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < borderSoftStart1 )
   {
      ret = fg;
   }
   else
   {
      ret = BorderColour;
   }

   return ret;
}
//--------------------------------------------------------------//
float4 SquareWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   // Figure out what we need to know
   float maxSquareW        = max( CentreX, 1.0 - CentreX );
   float maxSquareH        = max( CentreY, 1.0 - CentreY );
   float maxSquareSize     = 2 * max( maxSquareW, maxSquareH );
   float xBorderWidth      = BorderWidth * _MaxBorderSize;
   float yBorderWidth      = xBorderWidth * _OutputAspectRatio;
   float xBorderSoftness   = xBorderWidth * EdgeSoftness;
   float yBorderSoftness   = xBorderSoftness * _OutputAspectRatio;
   float sqW               = ( 1.0 - Progress ) * maxSquareSize;
   float sqH               = sqW * _OutputAspectRatio;
   float r                 = CentreX + ( sqW / 2 );
   float b                 = CentreY + ( sqH / 2 );
   float xBorderSoftEnd2   = r + ( ( 1.0 - Progress ) * ( xBorderSoftness + xBorderWidth ) );
   float xBorderSoftEnd1   = xBorderSoftEnd2 - xBorderWidth;
   float xBorderSoftStart1 = xBorderSoftEnd1 - xBorderSoftness;
   float xBorderSoftStart2 = xBorderSoftStart1 + xBorderWidth;
   float yBorderSoftEnd2   = b + ( ( 1.0 - Progress ) * ( yBorderSoftness + yBorderWidth ) );
   float yBorderSoftEnd1   = yBorderSoftEnd2 - yBorderWidth;
   float yBorderSoftStart1 = yBorderSoftEnd1 - yBorderSoftness;
   float yBorderSoftStart2 = yBorderSoftStart1 + yBorderWidth;

   // Read the source pixels (before we adjust xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Reflect pixel coords such that we appear to be dealing the the bottom-right
   // quadrant only (this enables us to use less shader instructions and hence
   // perform the wipe in a single pass)
   if ( xyNorm.x < CentreX )
   {
      xyNorm.x = ( 2 * CentreX ) - xyNorm.x;
   }
   if ( xyNorm.y < CentreY )
   {
      xyNorm.y = ( 2 * CentreY ) - xyNorm.y;
   }

   if ( xyNorm.x > xBorderSoftEnd2 || xyNorm.y > yBorderSoftEnd2 )
   {
      ret = bg;
   }
   else if ( xyNorm.x <= xBorderSoftStart1 && xyNorm.y <= yBorderSoftStart1 )
   {
      ret = fg;
   }
   else
   {
      float pixelSoftness;

      // Is the current-pixel in an OUTER edge softness area?
      bool xSoft = xyNorm.x > xBorderSoftStart2;
      bool ySoft = xyNorm.y > yBorderSoftStart2;

      if ( xSoft || ySoft )
      {
         if ( xSoft && ySoft ) // Outer bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness );
         }
         else if ( xSoft )     // Outer right edge
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart2 ) / xBorderSoftness;
         }
         else                  // Outer bottom edge
         {
            pixelSoftness = ( xyNorm.y - yBorderSoftStart2 ) / yBorderSoftness;
         }

         ret = lerp( BorderColour, bg, pixelSoftness );
      }
      else if ( xyNorm.x > xBorderSoftEnd1 || xyNorm.y > yBorderSoftEnd1 ) // Solid border area
      {
         ret = BorderColour;
      }
      else // INNER soft edge area
      {
         bool xSoft = xyNorm.x > xBorderSoftStart1;
         bool ySoft = xyNorm.y > yBorderSoftStart1;

         if ( xSoft && ySoft )  // Inner bottom right corner
         {
            pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            pixelSoftness = max( pixelSoftness, ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness );
         }
         else
         {
            if ( xSoft ) // Inner right edge
            {
               pixelSoftness = ( xyNorm.x - xBorderSoftStart1 ) / xBorderSoftness;
            }
            else         // Outer right edge
            {
               pixelSoftness = ( xyNorm.y - yBorderSoftStart1 ) / yBorderSoftness;
            }
         }

         ret = lerp( fg, BorderColour, pixelSoftness );
      }
   }

   return ret;

}
//--------------------------------------------------------------//
const float _Sqrt2Div2 = 1.4142 / 2;

float4 DiamondWithBorder_main( float2 xyNorm : TEXCOORD0, float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 ret;

   float borderWidth      = BorderWidth * _MaxBorderSize;
   float borderSoftness   = borderWidth * EdgeSoftness;
   float borderSoftStart1 = ( Progress * ( _ProgressComplete + borderWidth + borderSoftness ) ) - ( borderWidth + borderSoftness );
   float borderSoftEnd1   = borderSoftStart1 + borderSoftness;
   float borderSoftEnd2   = borderSoftStart1 + borderWidth + borderSoftness;;
   float borderSoftStart2 = borderSoftEnd2 - borderSoftness;

   // Read the fg & bg pixels (before wwe mess with xyNorm)
   float4 fg = tex2D( FgSampler, xy1 );
   float4 bg = tex2D( BgSampler, xy2 );

   // Each quadrant is some kind of reflection of the bottom-right quadrant :

   if ( xyNorm.x > CentreX )
   {
      xyNorm.x = 1.0 - xyNorm.x;
   }

   if ( xyNorm.y > CentreY )
   {
      xyNorm.y = 1.0 - xyNorm.y;
   }

   xyNorm.y /= _OutputAspectRatio;

   // Rotate current pixel pos through 45 degrees so that we can then just pretend
   // its a horizontal wipe..
   float d = length( xyNorm ) / _Sqrt2Div2;
   float a = atan( xyNorm.y / xyNorm.x );
   float2 rotatedTexCoord = float2( d * cos( a + _45 ), d * sin( a + _45 ) );

   if ( rotatedTexCoord.y > borderSoftEnd2 )
   {
      ret = fg;
   }
   else if ( rotatedTexCoord.y < borderSoftStart1 )
   {
      ret = bg;
   }
   else if ( rotatedTexCoord.y < borderSoftEnd1 )
   {
      ret = lerp( bg, BorderColour, ( rotatedTexCoord.y - borderSoftStart1 ) / borderSoftness );
   }
   else if ( rotatedTexCoord.y > borderSoftStart2 )
   {
      ret = lerp( BorderColour, fg, ( rotatedTexCoord.y - borderSoftStart2 ) / borderSoftness );
   }
   else
   {
      ret = BorderColour;
   }

   return ret;
}

//--------------------------------------------------------------//
// Techniques
//--------------------------------------------------------------//

//--------------------------------------------------------------//
// Vertical Wipe
//--------------------------------------------------------------//
technique V                  { pass Single_Pass { PixelShader = compile PROFILE V_main(); } }
technique VWithBorder        { pass Single_Pass { PixelShader = compile PROFILE VWithBorder_main(); } }

//--------------------------------------------------------------//
// Horinzontal Wipe
//--------------------------------------------------------------//
technique H                  { pass Single_Pass { PixelShader = compile PROFILE H_main(); } }
technique HWithBorder        { pass Single_Pass { PixelShader = compile PROFILE HWithBorder_main(); } }

//--------------------------------------------------------------//
// Diagonal Wipe
//--------------------------------------------------------------//
technique Diag1              { pass Single_Pass { PixelShader = compile PROFILE Diag1_main(); } }
technique Diag1WithBorder    { pass Single_Pass { PixelShader = compile PROFILE Diag1WithBorder_main(); } }

//--------------------------------------------------------------//
// Diagonal Wipe
//--------------------------------------------------------------//
technique Diag2              { pass Single_Pass { PixelShader = compile PROFILE Diag2_main(); } }
technique Diag2WithBorder    { pass Single_Pass { PixelShader = compile PROFILE Diag2WithBorder_main(); } }

//--------------------------------------------------------------//
// Vertical Shutter wipe
//--------------------------------------------------------------//
technique VShutter           { pass Single_Pass { PixelShader = compile PROFILE VShutter_main(); } }
technique VShutterWithBorder { pass Single_Pass { PixelShader = compile PROFILE VShutterWithBorder_main(); } }

//--------------------------------------------------------------//
// Horizontal shutter wipe
//--------------------------------------------------------------//
technique HShutter           { pass Single_Pass { PixelShader = compile PROFILE HShutter_main(); } }
technique HShutterWithBorder { pass Single_Pass { PixelShader = compile PROFILE HShutterWithBorder_main(); } }

//--------------------------------------------------------------//
// Cross wipe
//--------------------------------------------------------------//
technique Cross              { pass Single_Pass { PixelShader = compile PROFILE Cross_main(); } }
technique CrossWithBorder    { pass Single_Pass { PixelShader = compile PROFILE CrossWithBorder_main(); } }

//--------------------------------------------------------------//
// Bow-Tie wipe
//--------------------------------------------------------------//
technique BowTie             { pass Single_Pass { PixelShader = compile PROFILE BowTie_main(); } }
technique BowTieWithBorder   { pass Single_Pass { PixelShader = compile PROFILE BowTieWithBorder_main(); } }

//--------------------------------------------------------------//
// Corner Wipe (top left)
//--------------------------------------------------------------//
technique CornerTL           { pass Single_Pass { PixelShader = compile PROFILE CornerTL_main(); } }
technique CornerTLWithBorder { pass Single_Pass { PixelShader = compile PROFILE CornerTLWithBorder_main(); } }

//--------------------------------------------------------------//
// Corner Wipe (top right)
//--------------------------------------------------------------//
technique CornerTR           { pass Single_Pass { PixelShader = compile PROFILE CornerTR_main(); } }

technique CornerTRWithBorder { pass Single_Pass { PixelShader = compile PROFILE CornerTRWithBorder_main(); } }

//--------------------------------------------------------------//
// Corner Wipe (bottom right)
//--------------------------------------------------------------//
technique CornerBR           { pass Single_Pass { PixelShader = compile PROFILE CornerBR_main(); } }
technique CornerBRWithBorder { pass Single_Pass { PixelShader = compile PROFILE CornerBRWithBorder_main(); } }

//--------------------------------------------------------------//
// Corner Wipe (bottom left)
//--------------------------------------------------------------//
technique CornerBL           { pass Single_Pass { PixelShader = compile PROFILE CornerBL_main(); } }
technique CornerBLWithBorder { pass Single_Pass { PixelShader = compile PROFILE CornerBLWithBorder_main(); } }

//--------------------------------------------------------------//
// Vee Wipe
//--------------------------------------------------------------//
technique VeeOne             { pass Single_Pass { PixelShader = compile PROFILE VeeOne_main(); } }
technique VeeOneWithBorder   { pass Single_Pass { PixelShader = compile PROFILE VeeOneWithBorder_main(); } }

//--------------------------------------------------------------//
// Vee Wipe
//--------------------------------------------------------------//
technique VeeTwo             { pass Single_Pass { PixelShader = compile PROFILE VeeTwo_main(); } }
technique VeeTwoWithBorder   { pass Single_Pass { PixelShader = compile PROFILE VeeTwoWithBorder_main(); } }

//--------------------------------------------------------------//
// Vee Wipe
//--------------------------------------------------------------//
technique VeeThree           { pass Single_Pass { PixelShader = compile PROFILE VeeThree_main(); } }
technique VeeThreeWithBorder { pass Single_Pass { PixelShader = compile PROFILE VeeThreeWithBorder_main(); } }

//--------------------------------------------------------------//
// Vee wipe
//--------------------------------------------------------------//
technique VeeFour           { pass Single_Pass { PixelShader = compile PROFILE VeeFour_main(); } }
technique VeeFourWithBorder { pass Single_Pass { PixelShader = compile PROFILE VeeFourWithBorder_main(); } }

//--------------------------------------------------------------//
// Circle Wipe
//--------------------------------------------------------------//
technique Circle            { pass Single_Pass { PixelShader = compile PROFILE Circle_main(); } }
technique CircleWithBorder  { pass Single_Pass { PixelShader = compile PROFILE CircleWithBorder_main(); } }

//--------------------------------------------------------------//
// Square Wipe
//--------------------------------------------------------------//
technique Square            { pass Single_Pass { PixelShader = compile PROFILE Square_main(); } }
technique SquareWithBorder  { pass Single_Pass { PixelShader = compile PROFILE SquareWithBorder_main(); } }

//--------------------------------------------------------------//
// Diamond Wipe
//--------------------------------------------------------------//
technique Diamond           { pass Single_Pass { PixelShader = compile PROFILE Diamond_main(); } }
technique DiamondWithBorder { pass Single_Pass { PixelShader = compile PROFILE DiamondWithBorder_main(); } }
