//--------------------------------------------------------------
// TexWipe.fx
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------
DeclareLightworksEffect( "Texture Wipe", Mix, Internal, kNoNotes, CanSize );

//--------------------------------------------------------------
// Params
//--------------------------------------------------------------
DeclareFloatParam( XRepeat, "Across", "Repeats", kNoFlags, 1.0, 0.5, 50.0 );
DeclareFloatParam( YRepeat, "Across", "Repeats", kNoFlags, 1.0, 0.5, 50.0 );
DeclareFloatParam( BorderWidth, "Size", "Border", kNoFlags, 0.0, 0.0, 1.0 );
DeclareColourParam( BorderColour1, "Colour one", "Border", kNoFlags, 1.0, 1.0, 0.0, 1.0 );
DeclareColourParam( BorderColour2, "Colour two", "Border", kNoFlags, 1.0, 0.0, 0.0, 1.0 );
DeclareFloatParam( Softness, "Softness", kNoGroup, kNoFlags, 0.2, 0.0, 1.0 );
DeclareFloatParamAnimated( Progress, "Progress", kNoGroup, kNoFlags, 0.0, 0.0, 1.0 );
DeclareBoolParam( Invert, "Reverse", kNoGroup, false );
DeclareFloatParam( XPos, "Horizontal Position", "Position", kNoFlags, 0.5, -0.5, 1.5 );
DeclareFloatParam( YPos, "Vertical Position", "Position", kNoFlags, 0.5, -0.5, 1.5 );
DeclareFloatParam( _OutputWidth );
DeclareFloatParam( _OutputHeight );
DeclareFloatParam( _OutputAspectRatio );

//--------------------------------------------------------------
// Inputs
//--------------------------------------------------------------
DeclareInputs( fg, bg );

// Private constants
float  _PI         = 3.1415926535;
float  _TwoPI      = 2.0 * 3.1415926535;
float  _HalfPI     = 3.1415926535 / 2.0;
float  _QuarterPI  = 3.1415926535 / 4.0;
float  _PIAndAHalf = 1.5 * 3.1415926535;
float2 _Centre     = float2( 0.5, 0.5 );
float _255         = 255.0;
float _sqrt2       = sqrt( 2.0 );
float _EllipseSquish       = 0.6;
float _Tan60               = 1.7320; //tan( 2 * 3.14159 / 6 );
float _Tan30               = 0.5774; //tan( 2 * 3.14159 / 12 );
float _Cos30               = 0.8660; //cos( 2 * 3.14159 / 12 );
float _SixtyDegreeGradient = 0.5774; //1.0 / tan( 2 * 3.14159 / 6 );
float _Tan45 = 1; //tan( 2 * 3.14159 / 8 );
float _Tan72  =  3.0777; // tan( 2 * 3.14159 / 5 );
float _Tan144 = -0.7265; // tan( 2 * 3.14159 / 2.5 );
float _Tan36  =  0.7265; // tan( 2 * 3.14159 / 10 );
float _Cos36  =  0.8090; // cos( 2 * 3.14159 / 10 );
float _Cos72  =  0.3090; // cos( 2 * 3.14159 / 5 );

//--------------------------------------------------------------
// doWipe
//
// Main wipe rendering code (shared between all techniques)
//--------------------------------------------------------------
float4 doWipe( float2 xy1, float2 xy2, float luma )
{
   float totalEdgeSize = Softness + BorderWidth;
   float halfSoftness  = ( Softness / 2.0 );
   float edgeSoftStart = ( Progress * ( 1.0 + totalEdgeSize ) ) - totalEdgeSize;
   float borderStart   = edgeSoftStart + halfSoftness;
   float borderEnd     = borderStart + BorderWidth;
   float edgeSoftEnd   = borderEnd + halfSoftness;

   float4 ret;
   float4 fgPix = ReadPixel( fg, xy1 );
   float4 bgPix = ReadPixel( bg, xy2 );

   luma = lerp( luma, 1.0 - luma, Invert );

   if ( luma > edgeSoftEnd )
   {
      ret = fgPix;
   }
   else if ( luma < edgeSoftStart )
   {
      ret = bgPix;
   }
   else
   {
      float lumaMinusEdgeStart =  luma - edgeSoftStart;

      if ( BorderWidth == 0 )
      {
         ret = lerp( bgPix, fgPix, lumaMinusEdgeStart / Softness );
      }
      else
      {
         float4 borderColour = lerp( BorderColour1, BorderColour2, lumaMinusEdgeStart / totalEdgeSize );

         if ( luma < borderStart )
            ret = lerp( bgPix, borderColour, lumaMinusEdgeStart / halfSoftness );
         else if ( luma > borderEnd )
            ret = lerp( fgPix, borderColour, ( edgeSoftEnd - luma ) / halfSoftness );
         else
            ret = borderColour;
      }
   }

   return ret;
}

//--------------------------------------------------------------------------
// Pixel shaders for SMPTE wipes
//--------------------------------------------------------------------------
DeclareEntryPoint( Bar_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, xy0.x );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Bar_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, xy0.y );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Box_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( xy0.x, xy0.y ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Box_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( 1.0 - xy0.x, xy0.y ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Box_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( 1.0 - xy0.x, 1.0 - xy0.y ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Box_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( xy0.x, 1.0 - xy0.y ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Box_5 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( abs( xy0.x - float2(XPos, YPos).x ) * 2.0 , xy0.y ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Box_6 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( 1.0 - xy0.x, abs( xy0.y - float2(XPos, YPos).y ) * 2.0 ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Box_7 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( abs( xy0.x - float2(XPos, YPos).x ) * 2.0, 1.0 - xy0.y ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Box_8 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( abs( xy0.y - float2(XPos, YPos).y ) * 2.0, xy0.x ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Four_Box_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 = abs( float2(XPos, YPos) - xy0 ) * 2.0;
   return doWipe( uv1, uv2, 1.0 - min( xy0.x, xy0.y ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Four_Box_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, max( abs( frac( xy0.x * 2.0 ) - float2(XPos, YPos).x ) * 2.0,
                         abs( frac( xy0.y * 2.0 ) - float2(XPos, YPos).y ) * 2.0 ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Barn_Door_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, abs( xy0.x - float2(XPos, YPos).x ) * 2.0 );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Barn_Door_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, abs( xy0.y - float2(XPos, YPos).y ) * 2.0 );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Barn_Door_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float c = xy0.x + xy0.y;

   if ( c >= 1.0 )
      c = 1.0 - frac( c );

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Barn_Door_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float c = ( 1.0 - xy0.x ) + xy0.y;

   if ( c >= 1.0 )
      c = 1.0 - frac( c );

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Diagonal_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, ( xy0.x + xy0.y ) / 2.0 );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Diagonal_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, ( ( 1.0 - xy0.x ) + xy0.y ) / 2.0 );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Bow_Tie )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float c;

   if ( xy0.x < float2(XPos, YPos).x && xy0.y < float2(XPos, YPos).y )
      c = ( float2(XPos, YPos).x - xy0.x ) + xy0.y;

   else if ( xy0.x >= float2(XPos, YPos).x && xy0.y < float2(XPos, YPos).y )
      c = ( 1.0 - ( 1.0 - ( xy0.x - float2(XPos, YPos).x ) ) ) + xy0.y;

   else if ( xy0.x < float2(XPos, YPos).x && xy0.y >= float2(XPos, YPos).y )
      c = ( float2(XPos, YPos).x - xy0.x ) + ( 1.0 - xy0.y );
   else
      c = ( 1.0 - ( 1.0 - ( xy0.x - float2(XPos, YPos).x ) ) ) + ( 1.0 - xy0.y );

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Vee_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float c;

   if ( xy0.x < float2(XPos, YPos).x )
      c = ( ( ( float2(XPos, YPos).x - xy0.x ) * 2.0 ) + xy0.y ) / 2.0;

   else
      c = ( ( ( xy0.x - float2(XPos, YPos).x ) * 2.0 ) + xy0.y ) / 2.0;

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Vee_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0.x = 1.0 - xy0.x;

   float c;

   if ( xy0.y < float2(XPos, YPos).y )
      c = ( ( ( float2(XPos, YPos).y - xy0.y ) * 2.0 ) + xy0.x ) / 2.0;

   else
      c = ( ( ( xy0.y - float2(XPos, YPos).y ) * 2.0 ) + xy0.x ) / 2.0;

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Vee_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0.y = 1.0 - xy0.y;

   float c;

   if ( xy0.x < float2(XPos, YPos).x )
      c = ( ( ( float2(XPos, YPos).x - xy0.x ) * 2.0 ) + xy0.y ) / 2.0;

   else
      c = ( ( ( xy0.x - float2(XPos, YPos).x ) * 2.0 ) + xy0.y ) / 2.0;

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Vee_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float c;

   if ( xy0.y < float2(XPos, YPos).y )
      c = ( ( ( float2(XPos, YPos).y - xy0.y ) * 2.0 ) + xy0.x ) / 2.0;

   else
      c = ( ( ( xy0.y - float2(XPos, YPos).y ) * 2.0 ) + xy0.x ) / 2.0;

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Barn_Vee_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0.y = 1.0 - xy0.y;

   float c;

   if ( xy0.x < float2(XPos, YPos).x )
   {
      c = ( 2.0 * xy0.x ) + xy0.y;

      if ( c > 1.0 )
         c = 1.0 - frac( c );
   }
   else
   {
      c = ( ( 1.0 - xy0.x ) * 2.0 ) + xy0.y;

      if ( c > 1.0 )
         c = 1.0 - frac( c );
   }

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Barn_Vee_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float c;

   if ( xy0.y < float2(XPos, YPos).y )
   {
      c = ( 2.0 * xy0.y ) + xy0.x;

      if ( c > 1.0 )
         c = 1.0 - frac( c );
   }
   else
   {
      c = ( ( 1.0 - xy0.y ) * 2.0 ) + xy0.x;

      if ( c > 1.0 )
         c = 1.0 - frac( c );
   }

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Barn_Vee_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float c;

   if ( xy0.x < float2(XPos, YPos).x )
   {
      c = ( 2.0 * xy0.x ) + xy0.y;

      if ( c > 1.0 )
         c = 1.0 - frac( c );
   }
   else
   {
      c = ( ( 1.0 - xy0.x ) * 2.0 ) + xy0.y;

      if ( c > 1.0 )
         c = 1.0 - frac( c );
   }

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Barn_Vee_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0.x = 1.0 - xy0.x;

   float c;

   if ( xy0.y < float2(XPos, YPos).y )
   {
      c = ( 2.0 * xy0.y ) + xy0.x;

      if ( c > 1.0 )
         c = 1.0 - frac( c );
   }
   else
   {
      c = ( ( 1.0 - xy0.y ) * 2.0 ) + xy0.x;

      if ( c > 1.0 )
         c = 1.0 - frac( c );
   }

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Clock_12 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0-= float2(XPos, YPos);

   float angleRads = atan( abs( xy0.y / xy0.x ) );

   if ( xy0.x >= 0 )
   {
      if ( xy0.y < 0 )
         angleRads = ( _HalfPI - angleRads );
      else
         angleRads += _HalfPI;
   }
   else
   {
      if ( xy0.y < 0 )
         angleRads += _PIAndAHalf;
      else
         angleRads = _PIAndAHalf - angleRads;
   }

   return doWipe( uv1, uv2, angleRads / _TwoPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Clock_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0-= float2(XPos, YPos);

   float angleRads = atan( abs( xy0.y / xy0.x ) );

   if ( xy0.x >= 0 )
   {
      if ( xy0.y < 0 )
         angleRads = _TwoPI - angleRads;
   }
   else
   {
      if ( xy0.y >= 0 )
         angleRads = _PI - angleRads;
      else
         angleRads += _PI;
   }

   return doWipe( uv1, uv2, angleRads / _TwoPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Clock_6 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0-= float2(XPos, YPos);

   float angleRads = atan( abs( xy0.y / xy0.x ) );

   if ( xy0.x >= 0 )
   {
      if ( xy0.y < 0 )
         angleRads = _PIAndAHalf - angleRads;
      else
         angleRads += _PIAndAHalf;
   }
   else
   {
      if ( xy0.y < 0 )
         angleRads += _HalfPI;
      else
         angleRads = ( _HalfPI - angleRads );
   }

   return doWipe( uv1, uv2, angleRads / _TwoPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Clock_9 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0-= float2(XPos, YPos);

   float angleRads = atan( abs( xy0.y / xy0.x ) );

   if ( xy0.x >= 0 )
   {
      if ( xy0.y >= 0 )
         angleRads += _PI;
      else
         angleRads = _PI - angleRads;
   }
   else
   {
      if ( xy0.y >= 0 )
         angleRads = _TwoPI - angleRads;
   }

   return doWipe( uv1, uv2, angleRads / _TwoPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( PinWheel_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 -= float2(XPos, YPos);

   float angleRads = atan( abs( xy0.y / xy0.x ) );

   if ( ( xy0.y < 0.0 && xy0.x < 0.0 ) || ( xy0.x >= 0.0 && xy0.y >= 0.0 ) )
   {
      angleRads += _HalfPI;
   }
   else
   {
      angleRads = _HalfPI - angleRads;
   }

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( PinWheel_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0-= float2(XPos, YPos);

   float angleRads = atan( abs( xy0.x / xy0.y ) );

   if ( ( xy0.y >= 0.0 && xy0.x < 0.0 ) || ( xy0.x >= 0.0 && xy0.y < 0.0 ) )
   {
      angleRads += _HalfPI;
   }
   else
   {
      angleRads = _HalfPI - angleRads;
   }

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( PinWheel_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0-= float2(XPos, YPos);

   float angleRads = atan( abs( xy0.y / xy0.x ) );

   if ( ( xy0.y < 0.0 && xy0.x >= 0.0 ) || ( xy0.x < 0.0 && xy0.y >= 0.0 ) )
   {
      angleRads = _HalfPI - angleRads;
   }

   return doWipe( uv1, uv2, angleRads / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Single_Sweep_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float angleRads = abs( atan( xy0.y / ( xy0.x - float2(XPos, YPos).x ) ) );

   if ( xy0.x < float2(XPos, YPos).x )
      angleRads = _PI - angleRads;

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Single_Sweep_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 = 1.0 - xy0;

   float angleRads = _HalfPI - abs( atan( ( xy0.y - float2(XPos, YPos).y ) / xy0.x ) );

   if ( xy0.y >= float2(XPos, YPos).y )
      angleRads = _PI - angleRads;

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Single_Sweep_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 = 1.0 - xy0;

   float angleRads = abs( atan(  xy0.y / ( xy0.x - float2(XPos, YPos).x ) ) );

   if ( xy0.x < float2(XPos, YPos).x )
      angleRads = _PI - angleRads;

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Single_Sweep_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float angleRads = _HalfPI - abs( atan( ( xy0.y - float2(XPos, YPos).y ) / xy0.x ) );

   if ( xy0.y >= float2(XPos, YPos).y )
      angleRads = _PI - angleRads;

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Single_Sweep_5 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, abs( atan( xy0.x / xy0.y ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Single_Sweep_6 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, abs( atan( ( 1.0 - xy0.y ) / xy0.x ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Single_Sweep_7 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, abs( atan( ( 1.0 - xy0.y ) / ( 1.0 - xy0.x ) ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Single_Sweep_8 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, abs( atan( xy0.y / ( 1.0 - xy0.x ) ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Fan_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 -= float2(XPos, YPos);

   float angleRads = abs( atan( xy0.y / xy0.x ) );

   if ( xy0.y < 0.0 )
   {
      angleRads = _HalfPI - angleRads;
   }
   else
   {
      angleRads += _HalfPI;
   }

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Fan_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 -= float2(XPos, YPos);

   float angleRads = abs( atan( xy0.x / xy0.y ) );

   if ( xy0.x >= 0.0 )
   {
      angleRads = _HalfPI - angleRads;
   }
   else
   {
      angleRads += _HalfPI;
   }

   return doWipe( uv1, uv2,  angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Fan_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, abs( atan( ( xy0.x - float2(XPos, YPos).x ) / xy0.y ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Fan_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2,  1.0 - ( abs( atan( xy0.x / ( xy0.y - float2(XPos, YPos).y ) ) ) / _HalfPI ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Fan_5 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, abs( atan( ( xy0.x - float2(XPos, YPos).x ) / ( 1.0 - xy0.y ) ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Fan_6 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   return doWipe( uv1, uv2, 1.0 - ( abs( atan( ( 1.0 - xy0.x ) / ( xy0.y - float2(XPos, YPos).y ) ) ) / _HalfPI ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Double_Fan_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 -= float2(XPos, YPos);

   return doWipe( uv1, uv2, abs( atan( xy0.x / xy0.y ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Double_Sweep_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float xDistFromCentre = xy0.x - float2(XPos, YPos).x;
   float yDistFromCentre = xy0.x >= float2(XPos, YPos).x ? xy0.y : 1.0 - xy0.y;

   return doWipe( uv1, uv2, abs( atan( yDistFromCentre / xDistFromCentre ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Double_Sweep_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0.y = 1.0 - xy0.y;

   float xDistFromCentre = ( xy0.y >= float2(XPos, YPos).y ) ? xy0.x : 1.0 - xy0.x;
   float yDistFromCentre = xy0.y - float2(XPos, YPos).y;

   return doWipe( uv1, uv2, 1.0 - ( abs( atan( yDistFromCentre / xDistFromCentre ) ) / _HalfPI ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Double_Sweep_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0.x = 1.0 - xy0.x;

   float xDistFromCentre = xy0.x - float2(XPos, YPos).x;
   float yDistFromCentre = xy0.y < float2(XPos, YPos).y ? xy0.y : 1.0 - xy0.y;

   float angleRads = abs( atan( yDistFromCentre / xDistFromCentre ) );

   if ( xy0.x >= float2(XPos, YPos).x )
   {
      angleRads = _PI - angleRads;
   }

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Double_Sweep_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0.y = 1.0 - xy0.y;

   float yDistFromCentre = xy0.y - float2(XPos, YPos).y;
   float xDistFromCentre = xy0.x < float2(XPos, YPos).x ? xy0.x : 1.0 - xy0.x;

   float angleRads = atan( abs( yDistFromCentre / xDistFromCentre ) );

   if ( xy0.y >= float2(XPos, YPos).y )
   {
      angleRads = _HalfPI - angleRads;
   }
   else
   {
      angleRads += _HalfPI;
   }

   return doWipe( uv1, uv2, angleRads / _PI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Double_Sweep_5 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float f;

   if ( ( ( 1.0 - xy0.x ) + xy0.y ) > 1.0 )
      f = ( 1.0 - xy0.y ) / ( 1.0 - xy0.x );
   else
      f =  xy0.y / xy0.x;

   return doWipe( uv1, uv2, atan( abs( f ) ) / _QuarterPI );

}
//--------------------------------------------------------------------------
DeclareEntryPoint( Double_Sweep_6 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float f;

   if ( ( xy0.x + xy0.y ) > 1.0 )
      f = xy0.y / ( 1.0 - xy0.x );
   else
      f = ( 1.0 - xy0.y ) / xy0.x;

   return doWipe( uv1, uv2, ( _HalfPI - atan( abs( f ) ) ) / _QuarterPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Saloon_Door_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float xDistFromCentre = ( xy0.x < float2(XPos, YPos).x ? xy0.x : 1.0 - xy0.x ) * 2;

   return doWipe( uv1, uv2, abs( atan( xy0.y / xDistFromCentre ) ) / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Saloon_Door_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float yDistFromCentre = ( xy0.y >= float2(XPos, YPos).y ? 1.0 - xy0.y : xy0.y ) * 2;
   float angleRads = abs( atan( yDistFromCentre / xy0.x ) );

   return doWipe( uv1, uv2, 1.0 - ( angleRads / _HalfPI ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Saloon_Door_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float xDistFromCentre = ( xy0.x < float2(XPos, YPos).x ? xy0.x : 1.0 - xy0.x ) * 2;
   float angleRads = abs( atan( ( 1.0 - xy0.y ) / xDistFromCentre ) );

   return doWipe( uv1, uv2, angleRads / _HalfPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Saloon_Door_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float yDistFromCentre = ( xy0.y >= float2(XPos, YPos).y ? 1.0 - xy0.y : xy0.y ) * 2;
   float angleRads = abs( atan( yDistFromCentre / ( 1.0 - xy0.x ) ) );

   return doWipe( uv1, uv2, 1.0 - ( angleRads / _HalfPI ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Windshield_1 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float xx = ( float2(XPos, YPos).x - xy0.x );
   float yy = ( xy0.y < float2(XPos, YPos).y ? ( xy0.y - 0.25 ) : ( ( 1.0 - xy0.y ) - 0.25 ) );

   float xDistFromCentre = abs( xx );
   float yDistFromCentre = abs( yy );

   float angleRads = atan( yDistFromCentre / xDistFromCentre );

   if ( xx >= 0 )
   {
      if ( yy < 0 )
         angleRads += _HalfPI;
      else
         angleRads = ( _HalfPI - angleRads );
   }
   else if ( xx < 0 )
   {
      if ( yy < 0 )
         angleRads = ( _HalfPI - angleRads ) + _PI;
      else
         angleRads += _PI + _HalfPI;
   }

   return doWipe( uv1, uv2, 1.0 - ( angleRads / _TwoPI ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Windshield_2 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float xx = ( xy0.x < float2(XPos, YPos).x ) ? ( xy0.x - 0.25 ) : ( ( 1.0 - xy0.x ) - 0.25 );
   float yy = xy0.y - float2(XPos, YPos).y;

   float angleRads = atan( abs( yy / xx ) );

   if ( xx >= 0 )
   {
      if ( yy >= 0 )
         angleRads = _TwoPI - angleRads;
   }
   else
   {
      if ( yy < 0 )
         angleRads = ( _HalfPI - angleRads ) + _HalfPI;
      else
         angleRads += _PI;
   }

   return doWipe( uv1, uv2, angleRads / _TwoPI );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Windshield_3 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float xDistFromCentre = ( xy0.x - float2(XPos, YPos).x );
   float yDistFromCentre = xy0.y < float2(XPos, YPos).y ? ( xy0.y - 0.25 ) : ( ( 1.0 - xy0.y ) - 0.25 );

   float angleRads = atan( abs( yDistFromCentre / xDistFromCentre ) );

   if ( xy0.y < 0.25 || ( xy0.y >= float2(XPos, YPos).y && xy0.y < 0.75 ) )
   {
      angleRads = _HalfPI - angleRads;
   }
   else
   {
      angleRads += _HalfPI;
   }

   float c = angleRads / _PI;

   if ( xy0.y >= float2(XPos, YPos).y )
      c = 1.0 - c;

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Windshield_4 )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   float xDistFromCentre = xy0.x < float2(XPos, YPos).x ? ( xy0.x - 0.25 ) : ( ( 1.0 - xy0.x ) - 0.25 );
   float yDistFromCentre = ( xy0.y - float2(XPos, YPos).y );

   float angleRads = atan( abs( xDistFromCentre / yDistFromCentre ) );

   if ( ( xy0.x >= 0.25 && xy0.x < float2(XPos, YPos).x ) || xy0.x >= 0.75 )
   {
      angleRads = _HalfPI - angleRads;
   }
   else
   {
      angleRads += _HalfPI;
   }

   float c = angleRads / _PI;

   if ( xy0.x < float2(XPos, YPos).x )
      c = 1.0 - c;

   return doWipe( uv1, uv2, c );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Square )
{
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 -= float2(XPos, YPos);
   xy0.y /=_OutputAspectRatio; // Effectively makes it square pixels
   xy0 = abs( xy0 ) * 2.0;

   return doWipe( uv1, uv2, max( xy0.x, xy0.y ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Diamond )
{
   float maxVal = float2(XPos, YPos).x + ( float2(XPos, YPos).y / _OutputAspectRatio );

   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );
   xy0 -= float2(XPos, YPos);
   xy0.y /= _OutputAspectRatio; // Effectively makes it square pixels

   xy0 = abs( xy0 );

   return doWipe( uv1, uv2, ( xy0.x + xy0.y ) / maxVal );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Pentagon_1 )
{
   float scaleFactor = length( float2( 0.5, _Centre.y / _OutputAspectRatio ) );
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 -= float2(XPos, YPos);
   xy0.y /= _OutputAspectRatio; // Effectively makes it square pixels
   xy0.x = abs( xy0.x );

   float luma;

   if ( ( xy0.y - ( xy0.x / _Tan72 ) ) > 0 )
   {
      luma = ( ( _Tan36 * xy0.x ) + xy0.y ) * _Cos36;
   }
   else if ( ( xy0.y - ( xy0.x / _Tan144 ) ) > 0 )
   {
      luma = ( ( _Tan72 * xy0.x ) - xy0.y ) * _Cos72;
   }
   else
   {
      luma = -xy0.y;
   }

   return doWipe( uv1, uv2, luma / scaleFactor );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Pentagon_2 )
{
   float scaleFactor = length( float2( 0.5, _Centre.y / _OutputAspectRatio ) );
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0.y = 1.0 - xy0.y;
   xy0 -= float2(XPos, YPos);
   xy0.y /= _OutputAspectRatio; // Effectively makes it square pixels
   xy0.x = abs( xy0.x );

   float luma;

   if ( ( xy0.y - ( xy0.x / _Tan72 ) ) > 0 )
   {
      luma = ( ( _Tan36 * xy0.x ) + xy0.y ) * _Cos36;
   }
   else if ( ( xy0.y - ( xy0.x / _Tan144 ) ) > 0 )
   {
      luma = ( ( _Tan72 * xy0.x ) - xy0.y ) * _Cos72;
   }
   else
   {
      luma = -xy0.y;
   }

   return doWipe( uv1, uv2, luma / scaleFactor );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Hexagon_1 )
{
   float scaleFactor = length( float2( 0.5, _Centre.y / _OutputAspectRatio ) );

   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 -= float2(XPos, YPos);
   xy0.y /= _OutputAspectRatio; // Effectively makes it square pixels
   xy0 = abs( xy0 );

   float luma;

   if ( ( xy0.x - ( xy0.y / _Tan60 ) ) > 0 )
   {
      luma = ( ( ( _Tan30 * xy0.y ) + xy0.x ) * _Cos30 );
   }
   else
      luma = xy0.y;

   return doWipe( uv1, uv2, luma / scaleFactor );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Hexagon_2 )
{
   float scaleFactor = length( float2( _OutputAspectRatio, 1.0 ) ); // Seems to work, not sure why..
   float2 xy0 = frac( float2(XPos, YPos) - ( ( float2(XPos, YPos) - uv0 ) * float2( XRepeat, YRepeat ) ) );

   xy0 -= float2(XPos, YPos);
   xy0.y /= _OutputAspectRatio; // Effectively makes it square pixels
   xy0 = abs( xy0 );

   float luma;

   if ( ( xy0.y - ( xy0.x / _Tan60 ) ) > 0 )
   {
      luma = ( ( ( _Tan30 * xy0.x ) + xy0.y ) * _Cos30 );
   }
   else
      luma = xy0.x;

   return doWipe( uv1, uv2, luma * scaleFactor );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Circle )
{
   float scaleFactor = length( float2( 0.5, _Centre.y / _OutputAspectRatio ) );

   float2 xy0 = float2(XPos, YPos) - uv0;
   xy0.y /=_OutputAspectRatio; // Has the effect of making the ellipse circular
   xy0 = frac( float2(XPos, YPos) - ( xy0 * float2( XRepeat, YRepeat ) ) );
   xy0 = pow( xy0 - float2(XPos, YPos), float2( 2,2 ) );

   return doWipe( uv1, uv2, sqrt( xy0.x + xy0.y ) / scaleFactor );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Ellipse_1 )
{
   float halfWidth  = 0.5 * _EllipseSquish;
   float halfHeight = _Centre.y / _OutputAspectRatio;

   float2 xy = float2(XPos, YPos) - uv0;
   xy.y /=_OutputAspectRatio; // Has the effect of making the ellipse circular
   xy.x *= _EllipseSquish;    // Makes the circle elliptical again, but by a fixed amount which is aspect-ratio independent
   float2 xy0 = frac( float2(XPos, YPos) - ( xy * float2( XRepeat, YRepeat ) ) );

   xy0 = pow( xy0 - float2(XPos, YPos), float2( 2,2 ) );

   return doWipe( uv1, uv2, sqrt( xy0.x + xy0.y ) / length( float2( halfWidth, halfHeight ) ) );
}
//--------------------------------------------------------------------------
DeclareEntryPoint( Ellipse_2 )
{
   float halfWidth  = 0.5;
   float halfHeight = _Centre.y / _OutputAspectRatio * _EllipseSquish;

   float2 xy0 = float2(XPos, YPos) - uv0;
   xy0.y /=_OutputAspectRatio; // Has the effect of making the ellipse circular
   xy0.y *= _EllipseSquish;    // Makes the circle elliptical again, but by a fixed amount which is aspect-ratio independent
   xy0 = frac( float2(XPos, YPos) - ( xy0 * float2( XRepeat, YRepeat ) ) );

   xy0 = pow( xy0 - float2(XPos, YPos), float2( 2,2 ) );

   return doWipe( uv1, uv2, sqrt( xy0.x + xy0.y ) / length( float2( halfWidth, halfHeight ) ) );
}
