//--------------------------------------------------------------//
// Colour Tint SMH
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------//

int _LwksEffectInfo
<
   string EffectGroup = "GenericPixelShader";
   string Description = "Colour Correction";
   string Category    = "Colour Effects";
   bool CanSize       = true;
> = 0;

//--------------------------------------------------------------//
// Params
//--------------------------------------------------------------//
float4 BlackLevel
<
   string Description = "Black";
   string Group = "Balance";
> = { 0.0, 0.0, 0.0, 1.0 };

float4 WhiteLevel
<
   string Description = "White";
   string Group = "Balance";
> = { 1.0, 1.0, 1.0, 1.0 };

float4 GreyLevel
<
   string Description = "Grey";
   string Group = "Balance";
> = { 0.5, 0.5, 0.5, 1.0 };

float4 RangeExtents
<
   string Description = "Range definition";
   string Group       = "Tonal Ranges";
   string Flags       = "SpecifiesTonalRanges";
> = { 0.2, 0.4, 0.6, 0.8 };

float4 ShadowTintColour
<
   string Description = "Shadows";
   string Group       = "Tonal Ranges";
   string Flags       = "SpecifiesColourOffset";
> = { 1.0, 1.0, 1.0, 1.0 };

float4 MidTintColour
<
   string Description = "Midtones";
   string Group       = "Tonal Ranges";
   string Flags       = "SpecifiesColourOffset";
> = { 1.0, 1.0, 1.0, 1.0 };

float4 HighTintColour
<
   string Description = "Highlights";
   string Group       = "Tonal Ranges";
   string Flags       = "SpecifiesColourOffset";
> = { 1.0, 1.0, 1.0, 1.0 };

float MasterSaturation
<
   string Description = "Saturation";
   float MinVal = 0.0;
   float MaxVal = 4.00;
> = 1.00;

float MasterGamma
<
   string Description = "Gamma";
   float MinVal = 0.10;
   float MaxVal = 4.00;
> = 1.00;

float MasterContrast
<
   string Description = "Contrast";
   float MinVal = 0.00;
   float MaxVal = 5.00;
> = 1.0;

float MasterBrightness
<
   string Description = "Brightness";
   float MinVal = -1.00;
   float MaxVal = 1.00;
> = 0.0;

float MasterGain
<
   string Description = "Gain";
   float MinVal = 0.00;
   float MaxVal = 4.00;
> = 1.0;

float _SplitProportion = 0.5;

//--------------------------------------------------------------//
// Inputs
//--------------------------------------------------------------//
texture Input;

sampler InputSampler = sampler_state
{
   Texture  = <Input>;
   AddressU  = Border;
   AddressV  = Border;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;
};

//--------------------------------------------------------------//
float4 NoSplitScreen( float2 xy1 : TEXCOORD1 ) : COLOR
{
   float4 ret;

   // Pre-calc softness boundaries
   float soft1 = RangeExtents[ 1 ] - RangeExtents[ 0 ];
   float soft2 = RangeExtents[ 3 ] - RangeExtents[ 2 ];
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;

   // Read the source image pixel
   float4 src = tex2D( InputSampler, xy1 );
   float srcA = src.a;

   src -= BlackLevel;
   src /= ( WhiteLevel - BlackLevel );
   src -= GreyLevel;
   src += greyAverage;

   // Calc its brightness (so that it can be treated as a shadow/midtone/highlight as appropriate)
   float lum = ( src.r * 0.3 ) + ( src.g * 0.59 ) + ( src.b * 0.11 );

   if ( lum < RangeExtents[ 1 ] )
   {
      if ( lum < RangeExtents[ 0 ] ) // shadow
      {
         ret = src + ShadowTintColour;
      }
      else                           // Shadow/midtone
      {
         ret = lerp( src + ShadowTintColour, src + MidTintColour, ( lum - RangeExtents[ 0 ] ) / soft1 );
      }
   }
   else if ( lum > RangeExtents[ 2 ] )
   {
      if ( lum > RangeExtents[ 3 ] ) // Highlight
      {
         ret = src + HighTintColour;
      }
      else                           // Midtone/highlight
      {
         ret = lerp( src + MidTintColour, src + HighTintColour, ( lum - RangeExtents[ 2 ] ) / soft2 );
      }
   }
   else                              // Highlight
   {
      ret = src + MidTintColour;
   }

   float4 avg = ( ret.r + ret.g + ret.b ) / 3.0;
   ret = avg + ( ( ret - avg ) * MasterSaturation );

   // Do brightness, gamma & contrast logic
   ret = ( ( ( ( pow( saturate( ret ), 1 / MasterGamma ) * MasterGain ) + MasterBrightness ) - 0.5 ) * MasterContrast ) + 0.5;
   ret.a = srcA;

   return ret;
}

//--------------------------------------------------------------//
float4 LeftSplitScreen( float2 xy1 : TEXCOORD1 ) : COLOR
{
   float soft1 = RangeExtents[ 1 ] - RangeExtents[ 0 ];
   float soft2 = RangeExtents[ 3 ] - RangeExtents[ 2 ];
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;

   float4 ret;
   float4 src = tex2D( InputSampler, xy1 );
   float srcA = src.a;

   if ( xy1.x < _SplitProportion )
   {
      src -= BlackLevel;
      src /= ( WhiteLevel - BlackLevel );
      src -= GreyLevel;
      src += greyAverage;

      float lum = ( src.r * 0.3 ) + ( src.g * 0.59 ) + ( src.b * 0.11 );

      if ( lum < RangeExtents[ 1 ] )
      {
         if ( lum < RangeExtents[ 0 ] )
         {
            ret = src + ShadowTintColour;
         }
         else
         {
            ret = lerp( src + ShadowTintColour, src + MidTintColour, ( lum - RangeExtents[ 0 ] ) / soft1 );
         }
      }
      else if ( lum > RangeExtents[ 2 ] )
      {
         if ( lum > RangeExtents[ 3 ] )
         {
            ret = src + HighTintColour;
         }
         else
         {
            ret = lerp( src + MidTintColour, src + HighTintColour, ( lum - RangeExtents[ 2 ] ) / soft2 );
         }
      }
      else
      {
         ret = src + MidTintColour;
      }

      float4 avg = ( ret.r + ret.g + ret.b ) / 3.0;
      ret = avg + ( ( ret - avg ) * MasterSaturation );

      // Do brightness, gamma & contrast logic
      ret = ( ( ( ( pow( saturate( ret ), 1 / MasterGamma ) * MasterGain ) + MasterBrightness ) - 0.5 ) * MasterContrast ) + 0.5;
      ret.a = srcA;
   }
   else
      ret = src;

   return ret;
}

//--------------------------------------------------------------//
float4 BottomSplitScreen( float2 xy1 : TEXCOORD1 ) : COLOR
{
   float soft1 = RangeExtents[ 1 ] - RangeExtents[ 0 ];
   float soft2 = RangeExtents[ 3 ] - RangeExtents[ 2 ];
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;

   float4 ret;
   float4 src = tex2D( InputSampler, xy1 );
   float srcA = src.a;

   if ( xy1.y > _SplitProportion )
   {
      src -= BlackLevel;
      src /= ( WhiteLevel - BlackLevel );
      src -= GreyLevel;
      src += greyAverage;

      float lum = ( src.r * 0.3 ) + ( src.g * 0.59 ) + ( src.b * 0.11 );

      if ( lum < RangeExtents[ 1 ] )
      {
         if ( lum < RangeExtents[ 0 ] )
         {
            ret = src + ShadowTintColour;
         }
         else
         {
            ret = lerp( src + ShadowTintColour, src + MidTintColour, ( lum - RangeExtents[ 0 ] ) / soft1 );
         }
      }
      else if ( lum > RangeExtents[ 2 ] )
      {
         if ( lum > RangeExtents[ 3 ] )
         {
            ret = src + HighTintColour;
         }
         else
         {
            ret = lerp( src + MidTintColour, src + HighTintColour, ( lum - RangeExtents[ 2 ] ) / soft2 );
         }
      }
      else
      {
         ret = src + MidTintColour;
      }

      float4 avg = ( ret.r + ret.g + ret.b ) / 3.0;
      ret = avg + ( ( ret - avg ) * MasterSaturation );

      // Do brightness, gamma & contrast logic
      ret = ( ( ( ( pow( saturate( ret ), 1 / MasterGamma ) * MasterGain ) + MasterBrightness ) - 0.5 ) * MasterContrast ) + 0.5;
      ret.a = srcA;
   }
   else
      ret = src;

   return ret;
}

//--------------------------------------------------------------//
float4 RightSplitScreen( float2 xy1 : TEXCOORD1 ) : COLOR
{
   float soft1 = RangeExtents[ 1 ] - RangeExtents[ 0 ];
   float soft2 = RangeExtents[ 3 ] - RangeExtents[ 2 ];
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;

   float4 ret;
   float4 src = tex2D( InputSampler, xy1 );
   float srcA = src.a;

   if ( xy1.x > _SplitProportion )
   {
      src -= BlackLevel;
      src /= ( WhiteLevel - BlackLevel );
      src -= GreyLevel;
      src += greyAverage;

      float lum = ( src.r * 0.3 ) + ( src.g * 0.59 ) + ( src.b * 0.11 );

      if ( lum < RangeExtents[ 1 ] )
      {
         if ( lum < RangeExtents[ 0 ] )
         {
            ret = src + ShadowTintColour;
         }
         else
         {
            ret = lerp( src + ShadowTintColour, src + MidTintColour, ( lum - RangeExtents[ 0 ] ) / soft1 );
         }
      }
      else if ( lum > RangeExtents[ 2 ] )
      {
         if ( lum > RangeExtents[ 3 ] )
         {
            ret = src + HighTintColour;
         }
         else
         {
            ret = lerp( src + MidTintColour, src + HighTintColour, ( lum - RangeExtents[ 2 ] ) / soft2 );
         }
      }
      else
      {
         ret = src + MidTintColour;
      }

      float4 avg = ( ret.r + ret.g + ret.b ) / 3.0;
      ret = avg + ( ( ret - avg ) * MasterSaturation );

      // Do brightness, gamma & contrast logic
      ret = ( ( ( ( pow( saturate( ret ), 1 / MasterGamma ) * MasterGain ) + MasterBrightness ) - 0.5 ) * MasterContrast ) + 0.5;
      ret.a = srcA;
   }
   else
      ret = src;


   return ret;
}

//--------------------------------------------------------------//
float4 TopSplitScreen( float2 xy1 : TEXCOORD1 ) : COLOR
{
   float soft1 = RangeExtents[ 1 ] - RangeExtents[ 0 ];
   float soft2 = RangeExtents[ 3 ] - RangeExtents[ 2 ];
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;

   float4 ret;
   float4 src = tex2D( InputSampler, xy1 );
   float srcA = src.a;

   if ( xy1.y < _SplitProportion )
   {
      src -= BlackLevel;
      src /= ( WhiteLevel - BlackLevel );
      src -= GreyLevel;
      src += greyAverage;

      float lum = ( src.r * 0.3 ) + ( src.g * 0.59 ) + ( src.b * 0.11 );

      if ( lum < RangeExtents[ 1 ] )
      {
         if ( lum < RangeExtents[ 0 ] )
         {
            ret = src + ShadowTintColour;
         }
         else
         {
            ret = lerp( src + ShadowTintColour, src + MidTintColour, ( lum - RangeExtents[ 0 ] ) / soft1 );
         }
      }
      else if ( lum > RangeExtents[ 2 ] )
      {
         if ( lum > RangeExtents[ 3 ] )
         {
            ret = src + HighTintColour;
         }
         else
         {
            ret = lerp( src + MidTintColour, src + HighTintColour, ( lum - RangeExtents[ 2 ] ) / soft2 );
         }
      }
      else
      {
         ret = src + MidTintColour;
      }

      float4 avg = ( ret.r + ret.g + ret.b ) / 3.0;
      ret = avg + ( ( ret - avg ) * MasterSaturation );

      // Do brightness, gamma & contrast logic
      ret = ( ( ( ( pow( saturate( ret ), 1 / MasterGamma ) * MasterGain ) + MasterBrightness ) - 0.5 ) * MasterContrast ) + 0.5;
      ret.a = srcA;
   }
   else
      ret = src;

   return ret;
}


//--------------------------------------------------------------//
// Techniques
//--------------------------------------------------------------//
technique None   { pass Pass1 { PixelShader = compile PROFILE NoSplitScreen(); } }
technique Left   { pass Pass1 { PixelShader = compile PROFILE LeftSplitScreen(); } }
technique Right  { pass Pass1 { PixelShader = compile PROFILE RightSplitScreen(); } }
technique Top    { pass Pass1 { PixelShader = compile PROFILE TopSplitScreen(); } }
technique Bottom { pass Pass1 { PixelShader = compile PROFILE BottomSplitScreen(); } }
