/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       vip2win.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: Main file for the VIP system
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#ifndef PRD
char _vip_header[] =
"$Header$";
#endif

#include "vipinc.h"
#include <malloc.h>

#define FRAME_ID 1
#define VIP_ID   2


IMPORT void VipHandleKeys(VIPINFO *wip, MPARAM mp1, MPARAM mp2);

#ifdef OS2EMX
#pragma linkage (VipProc, system)         /* for ICC */
static MRESULT VipProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);
#else
MRESULT EXPENTRY VipProc(HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2);
#endif

PRIVAT void VipExit(void) ;
PRIVAT void VipCommandHandler(VIPINFO *wip, HWND frame, USHORT cmd);
PRIVAT void VipGarbageCollect(void);
PRIVAT void RaiseTimer(void);
PRIVAT void VipShowChild(VIPINFO *win);
        
PRIVAT void SetFrameAboutMenu(HWND frame);
PRIVAT void DrawBox1(HPS hps, RECTL *rect);
PRIVAT void DrawBox2(HPS hps, RECTL *rect, LONG a, LONG b);

#ifdef DEBUG
PRIVAT char * VipMsgText(USHORT msg);
PRIVAT void VipReadMessages(void);
#endif

PRIVAT CONST PSZ VIPClass="VIP";
EXPORT VIPINFO *vipList=NULL;
PRIVAT VIPINFO *freeList=NULL;
PRIVAT LONG     vipCount=0;
PRIVAT LONG     freeCount=0;
PRIVAT BOOLEAN  showing=FALSE,
                closing=FALSE;
PRIVAT VIPINFO *frameVip=NULL;
EXPORT LONG     vipDesktopW;
EXPORT LONG     vipDesktopH;

EXPORT HAB      vipHab=0;
EXPORT HMQ      vipHmq=0;


EXPORT TRANSCRIPT *vipLog=NULL;
EXPORT BOOLEAN assert=TRUE;

EXPORT LONG vipColors[VIP_COLORS] = { CLR_WHITE, CLR_BLACK, CLR_PALEGRAY,
               CLR_DARKGRAY, CLR_RED, CLR_BLUE, CLR_GREEN,
               CLR_PINK, CLR_CYAN, CLR_YELLOW, CLR_DARKRED, CLR_DARKBLUE,
               CLR_DARKGREEN, CLR_DARKPINK, CLR_DARKCYAN, CLR_BROWN };

IMPORT LONG    vipPrintScale;
IMPORT BOOLEAN vipPrinting;

/**************************************************************
** NAME:        VipInit                                   [API]
** SYNOPSIS:    BOOLEAN VipInit(int argc, char **argv)
** DESCRIPTION: Initializes the VIP system.
**              This function MUST be called before any other
**              VIP function can be used.
**              <argc> and <argv> aren't used in OS/2
**              environment, but they are in the X-Windows
**              version.
** RETURNS:     True, when initialized succeeded.
**              False, otherwise.
**************************************************************/
EXPORT BOOLEAN
VipInit(int argc, char **argv)
{
    RECTL rect;

    /* argc,argv; */
    DnpapAtExit(VipExit);
#ifdef DEBUG
    VipReadMessages();
#endif
    MessageConfig(VIP_ERROR,"VIP");
    if (!vipHab && !(vipHab = WinInitialize(0)))
    {
        VipError(VIPERR_INIT1,"Failed to initialize PM!");
        return FALSE;
    }
    if (!vipHmq && !(vipHmq = WinCreateMsgQueue(vipHab, 0)))
    {
        VipError(VIPERR_INIT2,"Failed to create the message queue!");
        return FALSE;
    }
    if (!WinRegisterClass(vipHab, VIPClass, VipProc, CS_SIZEREDRAW, sizeof(PVOID)))
    {
        VipError(VIPERR_INIT3,"Failed to register new window class!");
        return FALSE;
    }
    WinQueryWindowRect(HWND_DESKTOP, &rect);
    vipDesktopW=rect.xRight;
    vipDesktopH=rect.yTop;
    VipInitKeys();
#ifndef OS2EMX
    VipInitExceptions();
#endif
    return TRUE;
}


/**************************************************************
** NAME:        VipClose                                  [API]
** SYNOPSIS:    void VipClose(VIPINFO *frame) 
** DESCRIPTION: Closes the application.
**              <frame> must be a top window. (ie parentless)
**              This is the best way to close a VIP application
** RETURNS:     void
**************************************************************/
EXPORT void
VipClose(VIPINFO *frame) 
{
    VIPTEST(frame, return);
    WinPostMsg(frame->win, WM_CLOSE, 0L, 0L);
}


/**************************************************************
** NAME:        VipExit                             
** SYNOPSIS:    void VipExit() 
** DESCRIPTION: Destroys all windows and closes VIP.
**              While be called at exit (DnpapExit).
** RETURNS:     void
**************************************************************/
PRIVAT void
VipExit() 
{ 
    while (vipList) 
        VipDestroy(vipList);
    if (vipHmq) WinDestroyMsgQueue(vipHmq);
    if (vipHab) WinTerminate(vipHab);
    vipHab = vipHmq = NULL;
}


/**************************************************************
** NAME:        VipDestroy                                [API]
** SYNOPSIS:    void VipDestroy(VIPINFO *wip) 
** DESCRIPTION: Destroys the window and its children.
** RETURNS:     void
**************************************************************/
EXPORT void
VipDestroy(VIPINFO *wip) 
{ 
    VIPINFO **pp;

    if (!wip) return;

    VipSetFont(wip, NULL);      /* Unlink used font */

    /* Destroy all children first */
    while (wip->child)
        VipDestroy(wip->child);

    if (wip->destroy)
        wip->destroy(wip);

    /* Remove me from the parent */
    if (wip->parent)
        for (pp=&(wip->parent->child); *pp; pp = &((*pp)->sibling))
            if (*pp==wip) {
                *pp=(*pp)->sibling;
                break;
            }

    /* Remove me from the vipList */
    for (pp=&(vipList); *pp; pp = &((*pp)->next))
        if (*pp==wip) 
        {
            *pp=wip->next;
            vipCount--;
            wip->next=freeList;
            freeList=wip;
            freeCount++;
            break;
        }
}


/**************************************************************
** NAME:        VipBell                                   [API]
** SYNOPSIS:    void VipBell()
** DESCRIPTION: Rings a warning bell.
** RETURNS:     void
**************************************************************/
EXPORT void
VipBell()
{
    WinAlarm(HWND_DESKTOP, WA_WARNING);
}


/**************************************************************
** NAME:        VipShow                                   [API]
** SYNOPSIS:    void VipShow(VIPINFO *win)
** DESCRIPTION: Makes <win> visible. Also all the children
**              are made visible.
** RETURNS:     void
**************************************************************/
PRIVAT void
VipShowChild(VIPINFO *win)
{
    register VIPINFO *child;
    if (!win) return;
    for (child=win->child; child; child=child->sibling)
        VipShowChild(child);
    if (win->win)
        WinShowWindow(win->win, TRUE);
}

EXPORT void
VipShow(VIPINFO *win)
{
    if (!win) return;
    VipShowChild(win);
    if (win->win)
    {
        if (win->frame)
        {
            WinShowWindow(win->frame, TRUE);
            WinSetFocus(HWND_DESKTOP, win->win);
            WinSetActiveWindow(HWND_DESKTOP, win->win);
        }
        else
            WinSetFocus(HWND_DESKTOP, win->win);
    }
    showing=TRUE;  
}                  


/**************************************************************
** NAME:        VipHide                                   [API]
** SYNOPSIS:    void VipHide(VIPINFO *win)
** DESCRIPTION: Makes <win> invisible. Also all the children
**              are made invisible.
** RETURNS:     void
**************************************************************/
EXPORT void
VipHide(VIPINFO *win)
{
    VIPINFO *child;
    for (child=win->child; child; child=child->sibling)
        VipHide(child);
    if (win->win) WinShowWindow(win->win, FALSE);
    if (win->frame) WinShowWindow(win->frame, FALSE);
}


/**************************************************************
** NAME:        VipUpdate                                 [API]
** SYNOPSIS:    void VipUpdate(VIPINFO *win, BOOLEAN all)
** DESCRIPTION: Refreshes <win> and its children. if <all> is
**              true, then it redraws the windows completely.
** RETURNS:     void
**************************************************************/
EXPORT void       
VipUpdate(VIPINFO *win, BOOLEAN all)
{
    if (all)
    {
        if (win->win)
            WinInvalidateRect(win->win,NULL, TRUE);
    }
    else
    {
        VIPINFO *child;
        if (win->win && win->update)
        {
            HPS hps = WinGetPS(win->win);
	    	win->update(win,hps,0);
            WinReleasePS(hps);
        }
    	for (child=win->child;child; child=child->sibling) 
			VipUpdate(child, 0);
    }
}


/**************************************************************
** NAME:        VipCheck                                  [API]
** SYNOPSIS:    BOOLEAN VipCheck(BOOLEAN block)
** DESCRIPTION: Checks the message loop for VIP.
**              if <block> is true, then is it blocks in the
**              event loop, and returns only when the
**              program should exit.
**              Otherwise VipCheck will only check the queue,
**              and returns immediatly after handling all
**              queued messages.
**              This function must be called very regularly.
** RETURNS:     FALSE, when program should exit.
**              TRUE, when everything is oke.
**************************************************************/
EXPORT BOOLEAN
VipCheck(BOOLEAN block)
{
    QMSG qmsg;
    if (closing) return FALSE;  /* Please, please close down! */
    while (block || WinPeekMsg(vipHab, &qmsg, 0, 0, 0, PM_NOREMOVE))
    {
        if (!WinGetMsg(vipHab, &qmsg, 0, 0, 0))
        {
            closing=TRUE;
            return FALSE;       /* Please Close Down */
        }
        switch(qmsg.msg)
        {
        case WM_TIMER:
            RaiseTimer();
            break;
        default:
            WinDispatchMsg(vipHab, &qmsg);
        }
    }
    return TRUE;
}

/**************************************************************
** NAME:        VipSetTitle                               [API]
** SYNOPSIS:    void VipSetTitle(VIPINFO *win,
**                        const char *title)
** DESCRIPTION: Sets the title of the window <win>.
**              Not all types of windows use the title.
**              Types which use the title are: windows without
**              parent, input window (as input buffer), dial
**              window (legend) and meter window (legend).
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetTitle(VIPINFO *win, CONST char *title)
{
    VIPTEST(win,return);
    if (win->frame)
        WinSetWindowText(win->frame, (PSZ)(title?title:"VIP"));
    else
        SREDRAW(win);
    VipString(&(win->title), title);
}


/**************************************************************
** NAME:        VipSetAbout                               [API]
** SYNOPSIS:    void VipSetAbout(VIPINFO *win, VIP_CALLBACK *af);
** DESCRIPTION: Sets the about function of the window <win>.
**              This enables an 'About...' entry in the system
**              menu of the window. The window must be a
**              parentless window.
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetAbout(VIPINFO *wip, VIP_CALLBACK af)
{
    VIPTEST(wip,return);
    wip->aboutFun=af;
    if (wip->frame && af)
        SetFrameAboutMenu(wip->frame);
}

/**************************************************************
** NAME:        VipSetBorder                               [API]
** SYNOPSIS:    void VipSetBorder(VIPINFO *win,
**                    int width, int type);
** DESCRIPTION: Sets the border width of the window <win>.
**              <type> can be VIP_BOX, VIP_DEPTH,
**              VIP_LOW or VIP_RISE.
**              VIP_BOX is plain rectangle in black.
**              VIP_DEPTH draws a edge border.
**              VIP_RISE is a button border.
**              VIP_LOW is used in menus. 
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetBorder(VIPINFO *win, int width, int type)
{
    VIPTEST(win,return);
    win->border=width;
    win->btype=type;
}

/**************************************************************
** NAME:        VipSetBackground                          [API]
** SYNOPSIS:    void VipSetBackground(VIPINFO *win, int back)
** DESCRIPTION: Sets the main background of a window.
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetBackground(VIPINFO *win, int back)
{
    VIPTEST(win,return);
    BACKGROUND(win) = vipColors[back];
}                           


/**************************************************************
** NAME:        VipSetForeground                          [API]
** SYNOPSIS:    void VipSetForeground(VIPINFO *win, int fore)
** DESCRIPTION: Sets the main foreground of a window.
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetForeground(VIPINFO *win, int fore)
{
    VIPTEST(win,return);
    FOREGROUND(win) = vipColors[fore];
}                           


/**************************************************************
** NAME:        VipOpenSimple                             [API]
** SYNOPSIS:    VIPINFO *VipOpenSimple(VIPINFO *parent, 
**                  int x, int y, int w, int h)
** DESCRIPTION: Opens a simple, plain window.
**              This function is the base of all other
**              VipOpen... functions. 
** RETURNS:     A pointer to a window information block.
**************************************************************/
EXPORT VIPINFO *
VipOpenSimple(VIPINFO *parent, int x, int y, int w, int h)
{
    HWND frame, canvas;
    VIPINFO *wip;

    if (parent && !(parent->win))
    {
        VipError(VIPERR_PARENT, "Parent window is not valid!");
        return NULL;
    }
    wip = frame = canvas = NULL;
#ifdef TEST
    if (freeList)
    {
        wip=freeList;
        freeList=wip->next;
        freeCount--;
    }
    else
#endif
    if (!wip)
    {
        wip = VipMalloc(sizeof(VIPINFO));
        if (!wip) return NULL;
    }
    if (parent)
    {
        VIPINFO *p=parent->child;
        if (p)
        {
            while (p->sibling) p=p->sibling;
            p->sibling=wip;
        }
        else parent->child = wip;
    }
    memset(wip, 0, sizeof(VIPINFO));
    wip->next        = vipList;
    wip->parent      = parent;
    wip->handler     = VipSimpleHandler;
    wip->update      = VipSimpleUpdate;
    wip->destroy     = VipSimpleDestroy;
    wip->x           = x;
    wip->y           = y;
    wip->w           = w;
    wip->h           = h;
    wip->type        = T_SIMPLE;
    BACKGROUND(wip)  = CLR_PALEGRAY;
    BACKGROUND2(wip) = CLR_WHITE;
    FOREGROUND(wip)  = CLR_BLACK;
    FOREGROUND2(wip) = CLR_DARKGRAY;
    SHADOW1(wip)     = CLR_WHITE;
    SHADOW2(wip)     = CLR_DARKGRAY;
    RED(wip)         = CLR_RED;
    wip->btype       = VIP_NONE;
    wip->magic       = VipCookie;
    vipList=wip;
    vipCount++;    

    if (!parent)
    {
        ULONG creat = FCF_MINMAX  | FCF_SIZEBORDER | FCF_ICON |
                      FCF_SYSMENU | FCF_TASKLIST   | FCF_TITLEBAR ;
        frameVip = wip;
        frame = WinCreateStdWindow(HWND_DESKTOP,
                                   FS_ICON,
                                   &creat,
                                   VIPClass,
                                   VIPClass,
                                   0L,
                                   0,                                 
                                   1,                 
                                   &canvas);
        wip->frame=frame;
        wip->win=canvas;
        if (!frame||!canvas) /* Can't even display an error message now! */
        {
            DnpapMessage(DMC_FATAL, VIPERR_OPEN,
                "Failed to open VIP parent window!!");
            VipDestroy(wip);
            return NULL;
        }
    }          
    else          
    {
        LONG w=parent->cx-2*parent->border;
        LONG h=parent->cy-2*parent->border;

        wip->frame=NULL;
        wip->win = WinCreateWindow(
                        parent->win,
                        (PSZ)VIPClass,
                        NULL,
                        WS_CLIPCHILDREN|WS_CLIPSIBLINGS,
                        (SHORT)((w*wip->x+500L)/1000L)+parent->border,
                        (SHORT)((h*wip->y+500L)/1000L)+parent->border,
                        (SHORT)((w*wip->w+500L)/1000L),
                        (SHORT)((h*wip->h+500L)/1000L),
                        /* Same owner as the parent */
#ifdef OS2EMX
                        WinQueryWindow(parent->win, QW_OWNER),
#else
                        WinQueryWindow(parent->win, QW_OWNER, 0),
#endif
                        HWND_TOP, /* on top of other existing siblings */
                        VIP_ID,  /* window id! */
                        wip,
                        0);
        if (wip->win==NULL)
        {
            DnpapMessage(DMC_FATAL, VIPERR_OPEN, "Failed to open child window!!");
            VipDestroy(wip);
            return NULL;
        }
    }
    return wip;
}



PRIVAT void
SetFrameAboutMenu(HWND frame)
{
    PRIVAT const char *mt[2] = {NULL, "A~bout..."};
    PRIVAT MENUITEM mi[2]=
        {
            MIT_END,MIS_SEPARATOR,0,0,0,0,
            MIT_END,MIS_TEXT,0,MENU_ABOUT,0,0
        };

    HWND hwndSysMenu, hwndSub;
    SHORT sItem, idSysMenu;
    MENUITEM miSysMenu;
    hwndSysMenu = WinWindowFromID(frame,FID_SYSMENU);
    idSysMenu=SHORT1FROMMR(WinSendMsg(hwndSysMenu,
            MM_ITEMIDFROMPOSITION,NULL,NULL));
    WinSendMsg(hwndSysMenu, MM_QUERYITEM,
            MPFROM2SHORT(idSysMenu, FALSE),
            MPFROMP(&miSysMenu));
    hwndSub = miSysMenu.hwndSubMenu;
    for (sItem=0;sItem<2;sItem++)
        WinSendMsg(hwndSub, MM_INSERTITEM,
                MPFROMP(mi+sItem), MPFROMP(mt[sItem]));
}


MRESULT EXPENTRY
#ifdef OS2EMX
VipProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
#else
VipProc(HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
#endif
{
    VIPINFO *wip;
#ifdef DEBUG
    DnpapMessage(DMC_TRACE, VIP_ERROR, "VipProc(0x%08lx, %d (%s), 0x%08lx, 0x%08lx)", hwnd, msg, VipMsgText(msg), mp1, mp2);
#endif
    wip=(VIPINFO *)WinQueryWindowPtr(hwnd, 0);
    if (wip)
    {
#ifdef DEBUG
            if (wip->magic!=VipCookie)
                VipError(VIPERR_VIP, "VipProc: Window pointer doesn't point to a VIP window!");
            else if (wip->win==0L)
                VipError(VIPERR_WIN, "VipProc: Window is invalid!");
            else if (wip->handler)
                return (MRESULT)(wip->handler(wip, msg, mp1, mp2));
#else
            if (wip->handler && wip->win)
                return (MRESULT)(wip->handler(wip, msg, mp1, mp2));
#endif
    }

    /* Default Handing for unknown windows */
    switch(msg)
    {
    case WM_CREATE:
        if (frameVip)   /* Frame window is under creation ! */
        {
            wip = frameVip;
            wip->win = hwnd;
#ifdef OS2EMX
            wip->frame = WinQueryWindow(hwnd, QW_PARENT),
#else
            wip->frame = WinQueryWindow(hwnd, QW_PARENT, 0),
#endif
            /* The child of a frame has standard no owner! */
            WinSetOwner(hwnd, wip->frame);
            WinSetWindowPtr(hwnd, 0, wip);
            frameVip = NULL;  /* Not needed anymore */
#ifdef DEBUG
            DnpapMessage(DMC_TRACE, VIP_ERROR, "VipProc: Creating Frame window: %lP",wip);
#endif
            WinSetWindowText(wip->frame, "VIP");
            
            wip->cx = ((vipDesktopW*wip->w+500L)/1000L),
            wip->cy = ((vipDesktopH*wip->h+500L)/1000L),
            WinSetWindowPos(wip->frame, HWND_TOP,
                (SHORT)((vipDesktopW*wip->x+500L)/1000L),
                (SHORT)((vipDesktopH*wip->y+500L)/1000L),
                (SHORT)wip->cx, (SHORT)wip->cy, SWP_MOVE | SWP_SIZE);
            return FALSE;
        }
        wip=(VIPINFO*)PVOIDFROMMP(mp1);
        if (wip)
        {
            CREATESTRUCT *p=PVOIDFROMMP(mp2);
#ifdef DEBUG
            DnpapMessage(DMC_TRACE, VIP_ERROR, "VipProc: Creating Child window: %lP", wip);
#endif
            wip->cx = p->cx;
            wip->cy = p->cy;
            wip->win = hwnd;
            WinSetWindowPtr(hwnd, 0, wip);
            return FALSE;
        }
#ifdef DEBUG
        DnpapMessage(DMC_TRACE, VIP_ERROR, "VipProc: Failed to create window!");
#endif
        return (MPARAM)TRUE;

    case WM_COMMAND:
        VipCommandHandler(NULL, hwnd, SHORT1FROMMP(mp1));
        return FALSE;
    }
    return WinDefWindowProc(hwnd, msg, mp1, mp2);
}


void
VipAdjustChild(VIPINFO *wip, LONG w, LONG h, LONG x, LONG y)
{
    if (!wip->win) return;
    if (wip->w) wip->cx=(w*wip->w+500L)/1000L;
    if (wip->h) wip->cy=(h*wip->h+500L)/1000L;
    WinSetWindowPos(wip->win, 0,
        (SHORT)(x + ((w*wip->x+500L)/1000L)),
        (SHORT)(y + ((h*wip->y+500L)/1000L)),
        (SHORT)wip->cx,
        (SHORT)wip->cy,
        SWP_MOVE|SWP_SIZE);
}


/* Called to set a new window size */
void
VipMoveWindow(VIPINFO *wip, LONG sx, LONG sy)
{
    VIPINFO *parent;
    LONG pw,ph;

    parent=wip->parent;
    if (parent)
    {
        pw=parent->cx;
        ph=parent->cy;
    }
    else
    {
        pw=vipDesktopW;
        ph=vipDesktopH;
    }
    wip->x = (((sx*2000L)/pw) + 1) / 2;
    wip->y = (((sy*2000L)/ph) + 1) / 2;
    WinSetWindowPos(wip->win, NULL, (SHORT)sx, (SHORT)sy, 0, 0, SWP_MOVE);
}

EXPORT void
VipSizeWindow(VIPINFO *wip, LONG sx, LONG sy)
{
    WinSetWindowPos(wip->win, NULL, 0, 0, (SHORT)sx, (SHORT)sy, SWP_SIZE);
}


LONG
VipSimpleHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2)
{
    if (!wip->win)
        VipError(VIPERR_HANDLER,"VipHandleSimple called on invalid window!");
#ifdef DEBUG
    DnpapMessage(DMC_TRACE, VIP_ERROR, "VipSimpleHandler(0x%08lx, %d (%s), 0x%08lx, 0x%08lx)", wip, msg, VipMsgText(msg), mp1, mp2);
#endif
    switch(msg)
    {
        case WM_MINMAXFRAME:
        {
            SWP *pswp=(PSWP)PVOIDFROMMP(mp1);
            if (pswp->FS&SWP_MINIMIZE)          /*JvO fs changed in FS */
                wip->flags|=F_ICONIZED;
            else
                wip->flags&=~F_ICONIZED;
            break;
        } 
        case WM_PAINT:
            if (wip->update)
            {
                HPS hps = WinBeginPaint(wip->win, 0, 0);
                wip->update(wip, hps, 1);
                WinEndPaint(hps);
            }
            return FALSE;

        case WM_SIZE:
        {
	        VIPINFO *child;
            LONG w,h;
            if (SHORT2FROMMP(mp2)==0) break;
            wip->cx=SHORT1FROMMP(mp2);
            wip->cy=SHORT2FROMMP(mp2);
            w=wip->cx-2*wip->border;
            h=wip->cy-2*wip->border;
            for (child=wip->child; child ; child = child->sibling)
                VipAdjustChild(child, w, h, wip->border, wip->border);
            return FALSE;
        }
         
        case WM_BUTTON1DOWN:
        case WM_BUTTON2DOWN: 
        case WM_BUTTON3DOWN:
            if (wip->win)
                WinSetFocus(HWND_DESKTOP, wip->win); 
            return (LONG)TRUE;

        case WM_CHAR:
            VipHandleKeys(wip, mp1, mp2);
            return (LONG)TRUE;

        case WM_DESTROY:
            if (wip->win)
            {
                WinSetWindowPtr(wip->win, 0, NULL);
                wip->win=0L;
            }
            return FALSE;

        case WM_ACTIVATE:
            if (SHORT1FROMMP(mp1))
                wip->flags|=F_ACTIVE;
            else
                wip->flags&=~F_ACTIVE;
            return FALSE;
                     
        case WM_COMMAND:
            VipCommandHandler(wip, wip->win, SHORT1FROMMP(mp1));
            return FALSE;

        case WM_ADJUSTWINDOWPOS:
        {
            PSWP pswp = (PSWP)PVOIDFROMMP(mp1);
            if (vipPrinting) break;
            if (pswp->FS&SWP_MINIMIZE)       /*JvO fs changed in FS */
            {
                wip->flags|=F_ICONIZED;
                break;
            }
            else
            {
                wip->flags&=~F_ICONIZED;
                if ((wip->adjust) && ((wip->w==0)||(wip->h==0)))
                    return wip->adjust(wip,pswp);
            }
            break;
        }

        case UM_SCALE:
        case WM_SHOW:
            if (wip->flags&F_ICONIZED) break;
            if ( SHORT1FROMMP(mp1)
             && (wip->adjust)
             && ((wip->w==0)||(wip->h==0)))
            {
                SWP swp;
                WinQueryWindowPos(wip->win, &swp);
                if (wip->adjust(wip,&swp))
                    WinSetWindowPos(wip->win, HWND_TOP,
                        0, 0, swp.cx, swp.cy, SWP_SIZE);
            }
            break;
    }
    return (LONG)WinDefWindowProc(wip->win, msg, mp1, mp2);
}



PRIVAT void
VipCommandHandler(VIPINFO *win, HWND wnd, USHORT cmd)
{
    switch(cmd)
    {
    case MENU_ABOUT:
        if (win && win->aboutFun) win->aboutFun(win, win, 0);
        break;
    }
}


void
VipSimpleUpdate(VIPINFO *wip, HPS hps, BOOLEAN all)
{                                          
    if (all && wip->win)
        VipBorder(wip, hps);
}

void
VipSimpleDestroy(VIPINFO *wip)
{
    /* Destroy the window itself */
    if (wip->win)
    {
        WinDestroyWindow(wip->win);
        if (wip->frame)
            WinDestroyWindow(wip->frame);
        wip->frame=0L;
        wip->win=0L;
    }
}


PRIVAT void
MessageFun(VIPINFO *win, void *p, BOOLEAN flag)
{
    if (flag==FALSE) /* RELEASED */
        VipDestroy((VIPINFO *)p);
}


EXPORT void
VipError(LONG ec, const char *msg)
{
    DnpapMessage(DMC_ERROR, ec, "Error in VIP library: %s\n", msg);
}


EXPORT VOID *
VipMalloc(ULONG size)
{
    VOID *p=DnpapMalloc(size);
    if (!p)
        DnpapMessage(DMC_FATAL, VIPERR_MEMORY,
            "VIP: Failed to allocate memory!");
    return p;
}


/**************************************************************
** NAME:        VipString    
** SYNOPSIS:    void VipString(char **buf, CONST char *str)
** DESCRIPTION: Allocates space for the string <str>.
**              Only if the <buf> is smaller than the string
**              a new piece of memory is allocated.
** RETURNS:     void
**************************************************************/
EXPORT void
VipString(char **buf, CONST char *str)
{
    if (str==NULL)
    {
        if (*buf)
        {
            DnpapFree((*buf)-1);
            *buf=NULL;
        }
    }
    else
    {
#ifdef NEW
        static int two[]={8,16,32,64,128,256,1024,2048};
        register int i, len=strlen(str)+2;
        register char *p=*buf;

        for (i=0;i<NUMBEROF(two);i++)
            if (two[i]>=len)
                break;

        if (p)
            if (i!=p[0])
            {
                DnpapFree(p);
                p=NULL;
            }
        if (!p)
            p=VipMalloc(i==NUMBEROF(two)?len:two[len]);
        if (p)
        {
            *p++=(char)i;
            memcpy(p, str, len);
        }
        *buf=p;
#else
        register int len=strlen(str)+1;
        register char *p=*buf;
        if (p) DnpapFree(p);
        p=VipMalloc(len);
        if (p) memcpy(p, str, len);
        *buf=p;
#endif
    }
}


/**************************************************************
** NAME:        VipMessage                                [API]
** SYNOPSIS:    void VipMessage(VIPINFO *win,
**                  CONST char *title, CONST char *msg)
** DESCRIPTION: Displays the message <msg> on the screen.
**              It will remain until a <ok> button is pressed.
**              It opens a simple window with a title, textfield
**              and a 'Okay' button. When the user presses the
**              button the window is destroyed.
** RETURNS:     void
**************************************************************/
EXPORT void
VipMessage(VIPINFO *win, CONST char *title, CONST char *msg)
{
    VIPINFO *awin, *twin, *pwin, *nwin;

    if (!(awin = VipOpenFrame(win, 250,250,500,500)))
    {
        VipMessageBox(title, msg);
        return;
    }
    twin = VipOpenText(awin, 50,300,900,500);
    pwin = VipOpenButton(awin, 300,50,400,200);
    if (!twin || !pwin)
    {
        VipDestroy(awin);
        VipMessageBox(title, msg);
        return;
    }
    if (win)
    {
        VipSetBorder(awin, 3, VIP_RISE);
        if (title)
        {
            nwin=VipOpenText(awin, 50,900,1000,0);
            if (nwin)
            {
                VipSetBorder(nwin, 2, VIP_RISE);
                VipSetBackground(nwin, VIP_GREY);
                VipSetTextLine(nwin, title, 0, VIP_CENTER);
                VipSetFrameSplit(awin, nwin, VIP_HSPLIT|VIP_FIXTOP);
            }
        }
    }
    else
    {
        if (title)
            VipSetTitle(awin, title);
        else
            VipSetTitle(awin, "");
        VipSetSystemMenu(awin, FALSE);
    }
    VipSetTextLine(twin, msg, 0, VIP_WORDWRAP);
    VipSetTitle(pwin, "Okay");
    VipSetButtonKey(pwin, VIPKEY_ENTER);
    VipSetButtonCallBack(pwin, MessageFun, awin);
    if (win)
    {
        while (win->parent) win=win->parent;
        VipShow(win); /* Show always all message parents !! */
    }
    VipShow(awin); /* Show always !! */
}


/**************************************************************
** NAME:        VipMessageBox                             [API]
** SYNOPSIS:    void VipMessageBox(
**                  CONST char *title, CONST char *msg)
** DESCRIPTION: Displays the message <msg> on the screen.
**              It will remain until a <ok> button is pressed.
**              This is a failsafe version, for memory tight
**              situations.
** RETURNS:     void
**************************************************************/
EXPORT void
VipMessageBox(CONST char *title, CONST char *msg)
{
    WinMessageBox(HWND_DESKTOP, HWND_DESKTOP,
        (char*)msg, (char*)title, 0, MB_OK|MB_ICONHAND);
    /* Only the MB_ICONHAND is always safe to use! */
}


PRIVAT void DrawBorder(VIPINFO *wip, RECTL *prct, HPS hps, int type, int width);

EXPORT void
VipBorder(VIPINFO *wip, HPS hps)
{
    RECTL rect;
    POINTL pp;

    rect.xLeft=0;
    rect.yBottom=0;
    rect.xRight=wip->cx;
    rect.yTop=wip->cy;
    DrawBorder(wip, &rect, hps, wip->btype, wip->border);
    GpiSetColor(hps, BACKGROUND(wip));
    MOVE(rect.xLeft, rect.yBottom);
    BLOCK(rect.xRight-1, rect.yTop-1);
}

EXPORT void
VipDrawBorder(VIPINFO *wip, RECTL *prct, HPS hps, int type, int width)
{
    RECTL rect = *prct;
    DrawBorder(wip, &rect, hps, type, width);
}

PRIVAT void
DrawBorder(VIPINFO *wip, RECTL *rect, HPS hps, int type, int width)
{
    int i;

    GpiSetMix(hps, FM_OVERPAINT);
    if (vipPrinting)
        GpiSetLineWidth(hps, vipPrintScale>>16);
    else
        GpiSetLineWidth(hps, LINEWIDTH_NORMAL);
    switch(type)
    {
    case VIP_BOX:
        GpiSetColor(hps, FOREGROUND(wip));
        for (i=0;i<width;i++)
            DrawBox1(hps, rect);
        break;
    case VIP_DEPTH:
        GpiSetColor(hps, BACKGROUND(wip));
        for (i=0;i<width-2;i++)
            DrawBox1(hps, rect);
        if (width>=2) DrawBox2(hps, rect, SHADOW2(wip), SHADOW1(wip));
        if (width>=1) DrawBox2(hps, rect, SHADOW1(wip), SHADOW2(wip));
        break;
    case VIP_RISE:
        for (i=0;i<width;i++)
            DrawBox2(hps, rect, SHADOW1(wip), SHADOW2(wip));
        break;
    case VIP_LOW:
        for (i=0;i<width;i++)
            DrawBox2(hps, rect, SHADOW2(wip), SHADOW1(wip));
        break;
    case VIP_NONE:
        GpiSetColor(hps, BACKGROUND(wip));
        for (i=0;i<width;i++)
            DrawBox1(hps, rect);
        break;
    default:
        DnpapMessage(DMC_WARNING, VIPERR_BORDER,
            "VipBorder: Unknown border type: %d", wip->btype);
    }
}


PRIVAT void
DrawBox1(HPS hps, RECTL *rect)
{
    POINTL pp;

    rect->xRight--;
    rect->yTop--;
    MOVE(rect->xLeft, rect->yBottom);
    BOX(rect->xRight, rect->yTop);
    rect->xLeft++;   
    rect->yBottom++;
}

PRIVAT void
DrawBox2(HPS hps, RECTL *rect, LONG a, LONG b)
{
    POINTL pp;
    rect->xRight--;
    rect->yTop--;
    GpiSetColor(hps, a);
    MOVE(rect->xLeft,rect->yBottom+1);LINE(rect->xLeft,rect->yTop);LINE(rect->xRight-1,rect->yTop);
    GpiSetColor(hps, b);
    MOVE(rect->xRight,rect->yTop-1);LINE(rect->xRight,rect->yBottom);LINE(rect->xLeft+1,rect->yBottom);
    rect->xLeft++;
    rect->yBottom++;
}


/**************************************************************
** NAME:        VipSetSystemMenu                          [API]
** SYNOPSIS:    void VipSetSystemMenu(VIPINFO *win,
**                    BOOLEAN flag)
** DESCRIPTION: Enables the system menu entry of a main window.
**              If <flag> is TRUE, the menu is enabled,
**              otherwise the system menu is removed.
**              This function is handy for all non-parent
**              windows which are not the application main
**              window.
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetSystemMenu(VIPINFO *win, BOOLEAN flag)
{
    if (!win->frame)
        VipError(VIPERR_FRAME, "Window must be a frame window");
    else
    {
        WinSetParent(WinWindowFromID(win->frame, FID_SYSMENU),
            flag ? win->frame : HWND_OBJECT, TRUE);
        WinSendMsg(win->frame, WM_UPDATEFRAME, (MPARAM)FCF_SYSMENU, NULL);
    }              
}

/**************************************************************
** NAME:        VipIconize                                [API]
** SYNOPSIS:    void VipIconize(VIPINFO *win, BOOLEAN flag)
** DESCRIPTION: Iconizes the window. It should be a top-window.
**              In other words, it must not have any parents.
** RETURNS:     void
**************************************************************/
EXPORT void
VipIconize(VIPINFO *win, BOOLEAN flag)
{
    if (!win->frame)
        VipError(VIPERR_FRAME, "Only frame windows can be iconized");
    else
    {
        SWP swp;
        WinQueryWindowPos(win->frame, &swp);
        WinSetWindowPos(win->frame,
            NULL, swp.x, swp.y, swp.cx, swp.cy,
            SWP_MOVE|SWP_SIZE|SWP_MINIMIZE);
    }              
}


EXPORT void
VipPullMouseMoves()
{
    QMSG p;
    while (WinPeekMsg(vipHab,&p,NULL,WM_MOUSEMOVE,WM_MOUSEMOVE,PM_REMOVE));
}


PRIVAT void
VipGarbageCollect()
{
    VIPINFO *wip;
    while (freeList)
    {
        wip=freeList->next;
        DnpapFree(freeList);
        freeList=wip;
    }
}


EXPORT void
VipTypeError()
{
    VipError(VIPERR_TYPE, "VipSet function called on wrong type of VIP object");
}


EXPORT void
VipNullError()
{
    VipError(VIPERR_NULL, "VipSet: Window pointer is NULL");
}


EXPORT void
VipVipError()
{
    VipError(VIPERR_VIP, "VipSet: Window pointer doesn't point to a VIP window!");
}


IMPORT VIPTIMER *vipTimers;

PRIVAT void RaiseTimer()
{
    VIPTIMER *pt;
    ULONG now;
    QMSG p;

    while (WinPeekMsg(vipHab,&p,NULL,WM_TIMER,WM_TIMER,PM_REMOVE));
    for (pt=vipTimers;pt;pt=pt->next)
    {
        if (VipGetMiSecs() >= pt->alarm)
        {
            if (pt->callback) pt->callback(pt, pt->arg);
            now=VipGetMiSecs();
            while (now>=pt->alarm)
                pt->alarm+=pt->msecs;
        }
    }
}


/**************************************************************
** NAME:        VipSetBusy                                [API]
** SYNOPSIS:    void VipSetBusy(BOOLEAN on)
** DESCRIPTION: Changes the pointer into waiting pointer.
**              This function can be used to indicate the
**              a long operation is started.
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetBusy(BOOLEAN on)
{
#ifndef OS2EMX
    static HPOINTER hptrWait=0L;
    static HPOINTER hptrOld;

    if (!hptrWait)
        hptrWait = WinQuerySysPointer(HWND_DESKTOP, SPTR_WAIT, FALSE);
    if (on)
    {
        hptrOld = WinQueryPointer(HWND_DESKTOP);
        WinSetPointer(HWND_DESKTOP, hptrWait);
    }
    else
    {
        WinSetPointer(HWND_DESKTOP, hptrOld);
    }
#endif
}


#ifdef DEBUG
/*--------------------------- DEBUG ----------------------------*/

typedef struct _msginfo MSGINFO;
    struct _msginfo
    {
        USHORT msg;
        MSGINFO *next;
    };

PRIVAT MSGINFO *msgList=NULL;

PRIVAT void
VipReadMessages()
{
    FILE *fp;
    char line[80];
    char *p;
    USHORT msg;
    MSGINFO *mip;
    int l;

    fp=fopen("pm.msg","r");
    if (fp)
    {
        while (fgets(line,sizeof(line),fp))
        {
            p=strchr(line,' ');
            *p++='\0';
            l=p-line;
            p=strchr(p,'0');
            sscanf(p,"%x", &msg);
            mip=VipMalloc(sizeof(MSGINFO)+l);
            mip->msg=msg;
            mip->next=msgList;
            memcpy(((char*)mip)+sizeof(MSGINFO), line, l);
            msgList=mip;
        }
        fclose(fp);
    }
}

PRIVAT char *
VipMsgText(USHORT msg)
{
    MSGINFO *mip;
    
    for (mip=msgList; mip; mip=mip->next)
        if (mip->msg==msg)
            return ((char*)mip)+sizeof(MSGINFO);
    return "??";
}
#endif



