/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE	NAME:	        npspook.c
**     SYSTEM	NAME:	    tricklet
**     ORIGINAL AUTHOR(S):  Jan van Oorschot
**     VERSION	NUMBER:
**     CREATION DATE:
**
** DESCRIPTION: This module implements an OS/2 Named-Pipe
**              interface to Spook.
**
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION: $Revision$
** AUTHOR:   $Author$
** DATE:     $Date$
** LOG:	     $Log$
*************************************************************************/
#define         INCL_BASE
#include        <os2.h>
#include        <stdio.h>
#include        <stdlib.h>
#include        <stdarg.h>
#include        <string.h>
#include        <ctype.h>
#include        <time.h>

#include        <dnpap.h>
#include        <config.h>
#include        <snmp.h>
#include        <mibdf.h>
#include        <mgmt.h>
#include        "exec.h"
#include        "parse.h"
#include        "spook.h"

#define     SPOOK_HOST      "\\pipe\\beholder\\161"
#define     SPOOK_COMM      "public"
#define     SPOOK_TIMEOUT    5000000
#define     SPOOK_RETRIES    3

#define     PIPE_BUFLEN                 1514
#define     PIPE_MAX_CLIENTS            5               /** max pipe instances  */
#define     PIPE_IN_SIZE                1024            /* input buffer         */
#define     PIPE_OUT_SIZE               1024            /* output buffer        */
#define     PIPE_WSIZE                  0x1fff          /* work buffer size     */
#define     PIPE_SSIZE                  8192            /* stack size           */
#define     PIPE_STATE_UNUSED           0
#define     PIPE_STATE_CONNECTED        1
#define     PIPE_STATE_DISCONNECTED     2
#define     PIPE_SEM                    "\\SEM\\SPOOK0"

typedef struct _PIPE_TABLE PIPE_TABLE;

struct  _PIPE_TABLE
{
    int         state;              /* state of pipe instance       */
    unsigned    hdl;                /* pipe handle                  */
    char        *base;              /* base of write-data buffer    */
    char        *wbuf;              /* pending write date           */
    unsigned    num;                /* size of write                */
    SPOOK_OBJ   *spook;              /* ptr to spook info of this pip*/
};


BOOLEAN PipeInit(void);
static void FAR Thread(void);
static BOOLEAN  Check(void);
static int Handle(PIPE_TABLE *pt);

/* global data */
static HSYSSEM          pipeSem;
static ULONG            pipeMySem = 0;
static PIPE_TABLE       pipeTable[PIPE_MAX_CLIENTS] = {0};
static char	            pipeName[64] = "\\pipe\\spook\\161";
static TID              pipeTid;
static BYTE             *pipeStack = {0}; /* Thread Stack  */


/* eng */
static char             *pipeStateBuf;
static char             *pipeInBuf;
static char             *pipeOutBuf;

#ifdef _DEBUG
static  char DebugBuf[256];
#define DEBUGDUMP(x)    VioWrtTTY(x,strlen(x),(HVIO) NULL);
#endif


/* define the semaphore list to be used by the MuxSemWait */
DEFINEMUXSEMLIST(blockSemList,16);


int main(int argc,char **argv)
{
	USHORT i;
	int r;

	/* initialise everything */
	if (ConfigInit(argc, argv) == FALSE || ConfigLoad() == FALSE)
	{
		DnpapMessage(DMC_ERROR, TRICKLET_ERROR, "ConfigInit failed\n");
		DnpapExit(1);
	}
	if (MgmtInit() == FALSE)
	{
		DnpapMessage(DMC_ERROR, TRICKLET_ERROR, "MgmtInit failed\n");
		DnpapExit(1);
	}
	if (MibInit() != 0 || MibFullMib() != 0)
	{
		DnpapMessage(DMC_ERROR, TRICKLET_ERROR, "MibInit failed\n");
		DnpapExit(1);
	}

	if(!PipeInit())
	{
		DnpapMessage(DMC_ERROR, TRICKLET_ERROR, "PipeInit failed\n");
		DnpapExit(1);
	}

    /* loop and try to get questions */
	while(1)
	{
        /* block on the pipe semaphore */
		r = DosMuxSemWait(&i, &blockSemList, SEM_INDEFINITE_WAIT);
		Check();
	}

	/* end it all, shouldnt happen */
	return 0;
}

BOOLEAN PipeInit(void)
{
    SEL     Sel;
    USHORT  ret = 0;
    BYTE    *p;
    SHORT   s;
    BOOLEAN success = FALSE;
	int i;
    SPOOK_OBJ *spook;
    unsigned            hdl;


    /* increase your number of file handles, we'll need them */
    ret = DosSetMaxFH(64);
    if(ret != 0)
    {
	    DnpapMessage(DMC_ERROR, TRICKLET_ERROR, "DosSetMaxFH failed <%d>\n",
            ret);

    }
    
    if (ConfigGetString("pipe.hostname", &p))
        sprintf(pipeName,"\\pipe\\%s\\161", p);
    
    /* allocate */
    pipeStateBuf    = DnpapMalloc(PIPE_WSIZE);
    pipeInBuf       = DnpapMalloc(PIPE_BUFLEN);
    pipeOutBuf      = DnpapMalloc(PIPE_BUFLEN);
    if(pipeStateBuf == NULL || pipeInBuf == NULL || pipeOutBuf == NULL)
    {
	    DnpapMessage(DMC_ERROR, TRICKLET_ERROR, "DnpapMalloc failed\n");
    }

	/* allocate the named pipe handles */
	for (i=0; i<PIPE_MAX_CLIENTS;i++)
	{
        ret = DosMakeNmPipe(
            pipeName,   /* name of pipe */
            &hdl,             /* pointer to pipe handle */
            NP_ACCESS_DUPLEX |  /* access mode */
            NP_NOINHERIT |
            NP_NOWRITEBEHIND,
            NP_TYPE_MESSAGE |
            PIPE_MAX_CLIENTS+3,        /* maximum number of pipes */
            PIPE_IN_SIZE,            /* length input buffer */
            PIPE_OUT_SIZE,           /* length output buffer */
            0L);                /* timeout (ms) of DosWaitNmPipe */
        if(ret == 0)
        {
            pipeTable[i].hdl = hdl;
			pipeTable[i].state = PIPE_STATE_DISCONNECTED;
        }
		else
		{
		    DnpapMessage(DMC_ERROR, TRICKLET_ERROR, "DosMakeNmPipe %d failed\n",1);
			return FALSE;
		}
	}
    /* create all the Spook structures */
    for (i=0; i<PIPE_MAX_CLIENTS;++i)
    {
        spook = DnpapMalloc(sizeof(SPOOK_OBJ));
        if(spook == NULL)
        {
		    DnpapMessage(DMC_ERROR, TRICKLET_ERROR, "DnpapMalloc failed\n");
			return FALSE;
        }
        pipeTable[i].spook = spook;
        
        spook->have_list=0;
        spook->quit = 0;
        spook->repeat=0;
        if(ConfigGetString("spook.host",&p))
        {
            strcpy(spook->host,p);
        } else
        {
            strcpy(spook->host,SPOOK_HOST);
        }
        if(ConfigGetString("spook.medium",&p))
        {
            if(p[0]=='u' || p[0] == 'U')
                spook->medium = MGMT_TYPE_UDP;
            else
                spook->medium = MGMT_TYPE_PIPE;
        } else
        {
            spook->medium = MGMT_TYPE_DEFAULT;
        }
        if(ConfigGetString("spook.community",&p))
        {
            strcpy(spook->comm,p);
        } else
        {
            strcpy(spook->comm,SPOOK_COMM);
        }
        spook->commLen = strlen(spook->comm);
        if(ConfigGetShort("spook.timeout",&s))
        {
            spook->timeout = s;
        } else
        {
            spook->timeout = SPOOK_TIMEOUT;
        }
        if(ConfigGetShort("spook.retries",&s))
        {
            spook->retries = s;
        } else
        {
            spook->retries = SPOOK_RETRIES;
        }
        spook->handle = NULL;
    }


    /* create the system resources for the connect thread */
    ret = DosAllocSeg(PIPE_SSIZE,&Sel,SEG_NONSHARED);
    if (ret==0)
    {
        pipeStack = MAKEP(Sel,0);
        memset(pipeStack,0,PIPE_SSIZE);

        /* create resources */
        ret = DosCreateSem(CSEM_PUBLIC,&pipeSem,PIPE_SEM);
        if (ret==0)
        {
            ret = DosSemSet(pipeSem);
            if (ret==0)
            {
                ret = DosCreateThread(Thread,&pipeTid,pipeStack+PIPE_SSIZE);
                if (ret==0)
                {
        			blockSemList.cmxs = 0;
    				i=blockSemList.cmxs++;
    				blockSemList.amxs[i].zero=0;
    				blockSemList.amxs[i].hsem=pipeSem;
                    success = TRUE;
                }
            }
        }
    }
    return success;
}


static BOOLEAN Check(void)
{
    struct  npss    *np;            /* work structure */
    PIPE_TABLE      *pt;            /* instance table */
    USHORT          hdl;                /* pipe handle */
    USHORT          nrEvents;
    USHORT          ret;
    char            buf[128];

    /* initialise semphore */

#ifdef  _DEBUG
    DEBUGDUMP("-->\n\r");
#endif

    
    ret = DosSemSet(pipeSem);
    if (ret!=0)
    {
        return FALSE;
    }
    
    nrEvents = 1;

    while ((nrEvents > 0))
    {
        /* check what reason is for interruption of peace */
        ret = DosQNmPipeSemState(
            pipeSem,
            (PPIPESEMSTATE) pipeStateBuf,
            PIPE_WSIZE);
        if (ret !=0)
            return FALSE;

        np = (struct npss far *) pipeStateBuf;
        nrEvents = 0;

        /* do for each reason-record */
        while (np->npss_status != NPSS_EOI)
        {
            pt = &pipeTable[np->npss_key];
            hdl = pt->hdl;
#ifdef _DEBUG
            putchar('0'+np->npss_status);
#endif
            switch (np->npss_status)
                {
                case NPSS_CLOSE:
#ifdef _DEBUG
                    DEBUGDUMP("NPSS_CLOSE\n\r");
#endif
                    /* disconnect and mark the state */
                    DosDisConnectNmPipe(hdl);
                    pipeTable[np->npss_key].state = PIPE_STATE_DISCONNECTED;
                    if(pipeTable[np->npss_key].spook->handle!=NULL)
                    {
                        MgmtClose(pipeTable[np->npss_key].spook->handle);
                    }
                    /* Notify the Connect Loop */
                    DosSemClear(&pipeMySem);
                    nrEvents++;
                    break;

                case NPSS_RDATA:
#ifdef _DEBUG
                    DEBUGDUMP("NPSS_RDATA\n\r");
#endif
                    /* read available data */
                    ret = Handle(pt);
                    if (ret==FALSE)
                    {
#ifdef _DEBUG
                        DEBUGDUMP("Error handling RDATA\n\r");
#endif
                    }
                    nrEvents++;
                    break;

                case NPSS_WSPACE:
#ifdef _DEBUG
                    DEBUGDUMP("NPSS_WSPACE\n\r");
#endif
                    /* empty WSPACE in message mode */
                    break;

                default:
#ifdef _DEBUG
                    sprintf(buf,"unknown npss_status <%d>\n\r",
                        np->npss_status);
                    DEBUGDUMP(buf);
#endif
                    break;
                }
            ++np;
            }
    }
#ifdef  _DEBUG
    DEBUGDUMP("<--\n\r");
#endif
    return TRUE;
}

static int Handle(PIPE_TABLE *pt)
{
	USHORT n,num;
	int ret;

	/* this should handle a request through the pipe */
    ret = DosRead(pt->hdl,pipeInBuf,PIPE_BUFLEN,&num);
    if (ret!=0)
    {
            DnpapMessage(DMC_WARNING, TRICKLET_ERROR, "Pipe Read failed\n");
            return FALSE;
    }
    if(SpookMessage(pt->spook,pipeInBuf,pipeOutBuf) != 0)
    {
        DnpapMessage(DMC_WARNING,TRICKLET_ERROR,"Spook Message failed\n");
        return FALSE;
    }
    strcat(pipeOutBuf,"\n");
    ret = DosWrite(pt->hdl, pipeOutBuf, strlen(pipeOutBuf), &n);
    DosBufReset(pt->hdl);
    return (ret==0);
}
 
static void FAR Thread(void)
{
    int                 i;
    USHORT      ret;
    ret = 0;
    /* loop forever */
    while (ret == 0)
    {
        /* try to get a disconnected pipe instance */
        for (i=0; i<PIPE_MAX_CLIENTS;++i)
        {
            if (pipeTable[i].state == PIPE_STATE_DISCONNECTED)
                break;
        }
        /* if no instances available, wait for them */
        if (i == PIPE_MAX_CLIENTS)
        {
            DosSemSetWait(&pipeMySem,-1L);  /* Set and wait until it is cleared! */
            continue;                       /* start of loop please */
        }
		else
		{

	    	/* connect the pipeSem semaphore to the created named pipe
	       	so that the main routine can check that semaphore
	       	for any arriving data
	    	*/
        	/* Eigenlijk alleen doen als deze echt net 'geconnect' is */
        	ret |= DosSetNmPipeSem(pipeTable[i].hdl,pipeSem,i); 
#ifdef _DEBUG
           	if(ret!=0)
               	DEBUGDUMP("error in DosSetNmPipeSem\n\r");
#endif
	  
		/* wait for a client to connect to the current free pipe */
#ifdef _DEBUG
               	DEBUGDUMP("blocking for Connection\n\r");
#endif
        	ret |= DosConnectNmPipe(pipeTable[i].hdl);
#ifdef _DEBUG
           	if(ret!=0)
               	DEBUGDUMP("error in DosConnectNmPipe\n\r");
#endif
        	pipeTable[i].state = PIPE_STATE_CONNECTED;
		}
	    
    }
}
