/*
 *  Hash Table Functions
 *
 *  9 May 1992, R. Kooijman
 */


#include <dnpap.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>

#include "hash.h"


#define DEFAULT_HTSIZE  53



static HashKey DefaultHashFunc(BYTE *key, UINT n, UINT htsize);

static HashTable *NewHashTable(UINT size, HashFunction hf);
static VOID DelHashTable(HashTable *htp);

static HashKey GetHashKey(HashTable *ht, BYTE *key, UINT n);

static BucketEntry *NewBucketEntry(BYTE *key, UINT n);
static BucketEntry *ReturnBucketEntry(HashTable *ht, HashKey hk, BYTE *key, UINT n);
static BucketEntry *GetBucketEntry(HashTable *ht, HashKey hk, BYTE *key, UINT n);
static VOID DelBucketEntry(HashTable *ht, HashKey hk, BucketEntry *be);
static VOID DelBucketEntries(HashTable *ht, HashKey hk);

static VOID *GetEntryData(BucketEntry *be);
static VOID *SetEntryData(BucketEntry *be, VOID *data);

static UINT KeyEntryCmp(BucketEntry *be, BYTE *key, UINT n);
static UINT KeyCmp(BYTE *key1, UINT n1, BYTE *key2, UINT n2);

static VOID PrintHashStatistics(HashTable *ht);
static VOID PrintHashHead(HashTable *ht);
static VOID PrintHashTable(HashTable *ht);
static VOID PrintBucketEntry(HashTable *ht, HashKey hk, BucketEntry *be, UINT count);



HashTable *NewHash(UINT size, HashFunction hf)
{
   return NewHashTable(size, hf);
}


VOID DelHash(HashTable *ht)
{
   DelHashTable(ht);
}


VOID *HashAdd(HashTable *ht, BYTE *key, UINT n, VOID *data)
{
HashKey hk;
BucketEntry *be;

   hk = GetHashKey(ht, key, n);

   if ((be = ReturnBucketEntry(ht, hk, key, n)) == NULL)
      return NULL;

   if (GetEntryData(be) != NULL)
      return NULL;

   SetEntryData(be, data);

   return data;
}


VOID *HashRemove(HashTable *ht, BYTE *key, UINT n)
{
VOID *data;
HashKey hk;
BucketEntry *be;

   hk = GetHashKey(ht, key, n);

   if ((be = GetBucketEntry(ht, hk, key, n)) == NULL)
      return NULL;

   data = GetEntryData(be);

   DelBucketEntry(ht, hk, be);

   return data;
}


VOID *HashSearch(HashTable *ht, BYTE *key, UINT n)
{
HashKey hk;
BucketEntry *be;

   hk = GetHashKey(ht, key, n);

   if ((be = GetBucketEntry(ht, hk, key, n)) == NULL)
      return NULL;

   return GetEntryData(be);
}


static
HashKey DefaultHashFunc(BYTE *key, UINT n, UINT htsize)
{
HashKey hk = 0;
UINT i;

   if (key == NULL || n == 0)
      return -1;

   for (i = 0; i < n; i++)
      hk = (hk << 3) ^ key[i];

   return hk % htsize;
}


static
HashTable* NewHashTable(UINT size, HashFunction hf)
{
HashTable *htp;

   if (size > MAX_HASH_SIZE)
      return NULL;

   if ((htp = (HashTable *)DnpapMalloc(sizeof(HashTable))) == NULL)
      return NULL;

   if (size == 0)
      htp->Size = size = DEFAULT_HTSIZE;
   else
      htp->Size = size;

   if (hf == NULL)
      htp->HashFunc = hf = DefaultHashFunc;
   else
      htp->HashFunc = hf;

   htp->Occupied = 0;
   htp->Total = 0;
   
   htp->Table = NULL;

   if ((htp->Table = (BucketEntry **)DnpapMalloc(htp->Size*sizeof(BucketEntry *))) == NULL)
   {
      DnpapFree(htp);
      return NULL;
   }

   memset(htp->Table, 0, htp->Size*sizeof(BucketEntry *));

   return htp;
}


static
VOID DelHashTable(HashTable *htp)
{
HashKey hk;

   if (htp == NULL)
      return;

   for (hk = 0; hk < htp->Size && htp->Occupied > 0; hk++)
      DelBucketEntries(htp, hk);

   htp->Size = 0;   
   htp->Occupied = 0;   
   htp->HashFunc = NULL;
   
   DnpapFree(htp->Table);
   htp->Table = NULL;

   DnpapFree(htp);
   
   return;
}

   
static
BucketEntry* NewBucketEntry(BYTE *key, UINT n)
{
BucketEntry* bep;

   if ((bep = (BucketEntry *)DnpapMalloc(sizeof(BucketEntry))) == NULL)
      return NULL;

   bep->Key = key;
   bep->KeySize = n;
   bep->Contents = NULL;
   bep->Prev = NULL;
   bep->Next = NULL;

   return bep;
}


static
HashKey GetHashKey(HashTable *ht, BYTE *key, UINT n)
{
   if (ht == NULL)
      return -1;
   return ht->HashFunc(key, n, ht->Size);
}


static
BucketEntry *ReturnBucketEntry(HashTable *ht, HashKey hk, BYTE *key, UINT n)
{
BucketEntry *be;

   if (ht == NULL)
      return NULL;

   if (ht->Table[hk] == NULL)
   {
      be = NewBucketEntry(key, n);
      be ->Prev = NULL;
      be ->Next = NULL;
      ht->Table[hk] = be;
      ht->Occupied++;
      ht->Total++;

      return be;
   }   

   for (be = ht->Table[hk]; be != NULL; be = be->Next)
      if (!KeyEntryCmp(be, key, n))
         break;

   if (be != NULL)
      return be;

   be = NewBucketEntry(key, n);
   be ->Prev = NULL;
   be ->Next = ht->Table[hk];
   ht->Table[hk]->Prev = be;
   ht->Table[hk] = be;
   ht->Total++;

   return be;
}


static
BucketEntry *GetBucketEntry(HashTable *ht, HashKey hk, BYTE *key, UINT n)
{
BucketEntry *be;

   if (ht == NULL)
      return NULL;

   for (be = ht->Table[hk]; be != NULL; be = be->Next)
      if (!KeyEntryCmp(be, key, n))
         break;

   if (be != NULL)
      return be;

   return NULL;
}


static
VOID DelBucketEntry(HashTable *ht, HashKey hk, BucketEntry *be)
{
   if (be->Prev != NULL)
      be->Prev->Next = be->Next;
   else
      ht->Table[hk] = be->Next;
   if (be->Next != NULL)
      be->Next->Prev = be->Prev;

   DnpapFree(be);

   ht->Total--;
   if (ht->Table[hk] == NULL)
      ht->Occupied--;

   return;
}


static
VOID DelBucketEntries(HashTable *ht, HashKey hk)
{
BucketEntry *be1, *be2;

   if (ht == NULL)
      return;
      
   for (be2 = ht->Table[hk]; be2 != NULL; be2 = be1)
   {
      be1 = be2->Next;
      DnpapFree(be2);
   }
}


static
VOID *GetEntryData(BucketEntry *be)
{
   if (be == NULL)
      return NULL;
   return be->Contents;
}


static
VOID *SetEntryData(BucketEntry *be, VOID *data)
{
   if (be == NULL)
      return NULL;
   return be->Contents = data;
}


static
UINT KeyEntryCmp(BucketEntry *be, BYTE *key, UINT n)
{
   if (be == NULL)
      return 1;
   return KeyCmp(be->Key, be->KeySize, key, n);
}


static
UINT KeyCmp(BYTE *key1, UINT n1, BYTE *key2, UINT n2)
{
UINT i;

   i = 0;
   while (i < n1 && i < n2 && key1[i] == key2[i])
      i++;

   if (i == n1 || i == n2)
      if (n1 == n2)
         return 0;
      else
         return (n1 < n2) ? -1 : 1;
   else
      return (key1[i] < key2[i]) ? -1 : 1;
}


static
VOID PrintHashStatistics(HashTable *ht)
{
   PrintHashHead(ht);
   PrintHashTable(ht);
}


static
VOID PrintHashHead(HashTable *ht)
{
   printf("hash table size = %d\n", ht->Size);
   printf("hash table occupied = %d\n", ht->Occupied);
   printf("hash table total entries = %d\n", ht->Total);
   printf("hash function address = %p\n", ht->HashFunc);
   printf("bucket table address = %p\n", ht->Table);
}


static
VOID PrintHashTable(HashTable *ht)
{
BucketEntry *be;
HashKey hk;
UINT count;

   for (hk = 0; hk < ht->Size; hk++)
      if (ht->Table[hk] != NULL)
      {
         for (be = ht->Table[hk], count = 0; be != NULL; be = be->Next, count++)
            PrintBucketEntry(ht, hk, be, count);
         printf("%d places used in bucket[%d]\n", count, hk);
      }
}


static
VOID PrintBucketEntry(HashTable *ht, HashKey hk, BucketEntry *be, UINT count)
{
   printf("bucket entry[%d,%d] key = %.*s\n", hk, count, (INT)be->KeySize, be->Key);
   printf("bucket entry[%d,%d] key size = %d\n", hk, count, be->KeySize);
   printf("bucket entry[%d,%d] contents address = %p\n", hk, count, be->Contents);
   printf("bucket entry[%d,%d] prev address = %p\n", hk, count, be->Prev);
   printf("bucket entry[%d,%d] next address = %p\n", hk, count, be->Next);
}



#ifdef TEST

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


UINT main(UINT argc, char *argv[])
{
HashTable *ht;
UINT UINT1 = 1, UINT2 = 2, UINT3 = 3, UINT4 = 4, UINT5 = 5;
BYTE key[6], *keycp;
LONG count;
INT i;

   ht = NewHash(0, NULL);

   HashAdd(ht, (BYTE*)"one", strlen("one"), &UINT1);
   HashAdd(ht, (BYTE*)"two", strlen("two"), &UINT2);
   HashAdd(ht, (BYTE*)"three", strlen("three"), &UINT3);
   HashAdd(ht, (BYTE*)"four", strlen("four"), &UINT4);
   HashAdd(ht, (BYTE*)"five", strlen("five"), &UINT5);

   printf("UINT1 = %d\n", *(UINT *)HashSearch(ht, (BYTE*)"one", strlen("one")));
   printf("UINT2 = %d\n", *(UINT *)HashSearch(ht, (BYTE*)"two", strlen("two")));
   printf("UINT3 = %d\n", *(UINT *)HashSearch(ht, (BYTE*)"three", strlen("three")));
   printf("UINT4 = %d\n", *(UINT *)HashSearch(ht, (BYTE*)"four", strlen("four")));
   printf("UINT5 = %d\n", *(UINT *)HashSearch(ht, (BYTE*)"five", strlen("five")));

   PrintHashStatistics(ht);

   UINT1 = *(UINT *)(HashRemove(ht, (BYTE*)"one", strlen("one")));
   UINT2 = *(UINT *)(HashRemove(ht, (BYTE*)"two", strlen("two")));
   UINT3 = *(UINT *)(HashRemove(ht, (BYTE*)"three", strlen("three")));
   UINT4 = *(UINT *)(HashRemove(ht, (BYTE*)"four", strlen("four")));
   UINT5 = *(UINT *)(HashRemove(ht, (BYTE*)"five", strlen("five")));

   printf("UINT1 = %d\n", UINT1);
   printf("UINT2 = %d\n", UINT2);
   printf("UINT3 = %d\n", UINT3);
   printf("UINT4 = %d\n", UINT4);
   printf("UINT5 = %d\n", UINT5);

   PrintHashStatistics(ht);

   printf("1 = %p\n", HashSearch(ht, (BYTE*)"one", strlen("one")));
   printf("2 = %p\n", HashSearch(ht, (BYTE*)"two", strlen("two")));
   printf("3 = %p\n", HashSearch(ht, (BYTE*)"three", strlen("three")));
   printf("4 = %p\n", HashSearch(ht, (BYTE*)"four", strlen("four")));
   printf("5 = %p\n", HashSearch(ht, (BYTE*)"five", strlen("five")));

   DelHash(ht);
   
   PrintHashStatistics(ht);


   ht = NewHash(1011, NULL);

   for (count = 0; count < 100000; count++)
   {
      for (i = 0; i < 6; i++)
         key[i] = (BYTE)((rand() % 7) + 94);
      if ((keycp = HashSearch(ht, key, 6)) == NULL)
      {
         keycp = DnpapMalloc(6);
         memcpy(keycp, key, 6);
         if (HashAdd(ht, keycp, 6, keycp) == NULL)
         {
            printf("error in hash library");
            DnpapExit(1);
         }
         if (HashSearch(ht, keycp, 6) != keycp)
         {
            printf("error in hash library");
            DnpapExit(1);
         }
      }
      else
      {
         HashRemove(ht, key, 6);
         if (HashSearch(ht, key, 6) != NULL)
         {
            printf("error in hash library");
            DnpapExit(1);
         }
         DnpapFree(keycp);
      }
      printf("count = %ld, total = %u, occupied = %u\n", count, ht->Total, ht->Occupied);
      if (count % 10000 == 0)
         PrintHashStatistics(ht);
   }

   DelHash(ht);

   return 0;
}

#endif
