#include <limits.h>
#include <dnpap.h>
#include <dhostx.h>
#include <stdlib.h>
#include <memory.h>
#include <message.h>
#include <mibsup.h>

#include "dtopn_d.h"
#include "dtopn_e.h"
#include "dtopn_c.h"
#include "dtopn.h"



#define CONTROLINDEXSIZE   1
#define TOPNINDEXSIZE      2


static MIB_LOCAL    *TopNControlInstance = NULL;


static BOOLEAN RmonNext (SNMP_OBJECT *obj, MIB_LOCAL **local, WORD idlen, WORD indexsize, DNPAP_TOPN_ENTRY **elem);



BOOLEAN DnpapTopNMInit(VOID)
{
    MessageConfig(DNPAP_TOPN_ERROR, "DnpapTopN");
    return TRUE;
}


BOOLEAN RmonNext (SNMP_OBJECT *obj, MIB_LOCAL **local, WORD idlen, WORD indexsize, DNPAP_TOPN_ENTRY **elem)
{
    if (*local == NULL || indexsize < 1)
        return FALSE;

    if (obj->IdLen < idlen + indexsize)
    {
        switch (indexsize)
        {
        case TOPNINDEXSIZE:
            while (*local != NULL && 
                (((DNPAP_TOPN_CONTROL*)(*local)->Data)->Status != SNMP_VALID ||
                 ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TimeRemaining > 0 ||
                 ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TableSize == 0 ||
                 ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TopNArray == NULL))
                *local = (*local)->Next;
            if (*local == NULL)
                return FALSE;
            obj->Id[idlen] = (*local)->Index;
            obj->Id[idlen+1] = 1;
            obj->IdLen = idlen + TOPNINDEXSIZE;
            *elem = ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TopNArray[0];
            break;
        case CONTROLINDEXSIZE:
            obj->Id[idlen] = (*local)->Index;
            obj->IdLen = idlen + CONTROLINDEXSIZE;
            break;
        }
        return TRUE;
    }
    else
    {
        switch (indexsize)
        {            
        case TOPNINDEXSIZE:
            if (((DNPAP_TOPN_CONTROL*)(*local)->Data)->Status == SNMP_VALID &&
                ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TimeRemaining == 0 &&
                ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TableSize > 0 &&
                ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TopNArray != NULL &&
                obj->Id[idlen+1] < min(((DNPAP_TOPN_CONTROL*)(*local)->Data)->RequestedSize, ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TableSize))
            {
                obj->Id[idlen] = (*local)->Index;
                obj->Id[idlen+1]++;
                obj->IdLen = idlen + TOPNINDEXSIZE;
                *elem = ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TopNArray[obj->Id[idlen+1]-1];
                break;
            }
            *local = (*local)->Next;
            while (*local != NULL && 
                (((DNPAP_TOPN_CONTROL*)(*local)->Data)->Status != SNMP_VALID ||
                 ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TimeRemaining > 0 ||
                 ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TableSize == 0 ||
                 ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TopNArray == NULL))
                *local = (*local)->Next;
            if (*local == NULL)
                return FALSE;
            obj->Id[idlen] = (*local)->Index;
            obj->Id[idlen+1] = 1;
            obj->IdLen = idlen + TOPNINDEXSIZE;
            *elem = ((DNPAP_TOPN_CONTROL*)(*local)->Data)->TopNArray[0];
            break;
        case CONTROLINDEXSIZE:
            *local = (*local)->Next;
            if (*local == NULL)
                return FALSE;
            obj->Id[idlen] = (*local)->Index;
            obj->IdLen = idlen + CONTROLINDEXSIZE;
            break;
        }
        return TRUE;
    }
    return FALSE;
}


WORD HostTopNControlIndex(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL    *local = NULL;

    if ((local = MibRmon (obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        obj->Syntax.LngInt = local->Index;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HostTopNHostIndex(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL     *local = NULL;
DNPAP_TOPN_CONTROL  *data;

    if ((local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->HostIndex;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        if (data->Status != SNMP_UNDERCREATION)
            return SNMP_READONLY;
        data->HostIndex = obj->Syntax.LngInt;
        return SNMP_NOERROR;
    }
}


WORD HostTopNRateBase(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL     *local = NULL;
DNPAP_TOPN_CONTROL  *data;

    if ((local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->HostIndex;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        if (data->Status != SNMP_UNDERCREATION)
            return SNMP_READONLY;
        if (obj->Syntax.LngInt < DTOPN_RATEBASEMIN || obj->Syntax.LngInt < DTOPN_RATEBASEMAX)
            return SNMP_BADVALUE;
        data->HostIndex = obj->Syntax.LngInt;
        return SNMP_NOERROR;
    }
}


WORD HostTopNTimeRemaining(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL     *local = NULL;
DNPAP_TOPN_CONTROL  *data;

    if ((local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->TimeRemaining;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        if (obj->Syntax.LngInt < 1)
            return SNMP_BADVALUE;
        data->TimeRemaining = obj->Syntax.LngInt;
        data->Duration = data->TimeRemaining;
        if (data->Status == SNMP_VALID)
        {
            if (TopNCStop(data) == FALSE)
            {
                DnpapMessage(DMC_ERROR, DTOPN_STOP, "dtopN: collector %ld could not be stopped", local->Index);
                return SNMP_GENERROR;
            }
            if (TopNCStart(data) == FALSE)
            {
                DnpapMessage(DMC_ERROR, DTOPN_START, "dtopN: collector %ld could not be started", local->Index);
                return SNMP_GENERROR;
            }
        }
        return SNMP_NOERROR;
    }
}


WORD HostTopNDuration(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL     *local = NULL;
DNPAP_TOPN_CONTROL  *data;

    if ((local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->Duration;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HostTopNRequestedSize(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL     *local = NULL;
DNPAP_TOPN_CONTROL  *data;

    if ((local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->RequestedSize;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        if (obj->Syntax.LngInt < 0)
            return SNMP_BADVALUE;
        if (FillEntryTable(data, obj->Syntax.LngInt) != TRUE)
            return SNMP_GENERROR;
        data->RequestedSize = obj->Syntax.LngInt;
        return SNMP_NOERROR;
    }
}


WORD HostTopNGrantedSize(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL     *local = NULL;
DNPAP_TOPN_CONTROL  *data;

    if ((local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = min(data->RequestedSize, data->TableSize);
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HostTopNStartTime(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL     *local = NULL;
DNPAP_TOPN_CONTROL  *data;

    if ((local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->StartTime;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HostTopNOwner(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL    *local = NULL;
DNPAP_TOPN_CONTROL *data;

    if ((local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        memcpy (obj->Syntax.BufChr, data->Owner, data->OwnerLen);
        obj->SyntaxLen = data->OwnerLen;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        if (data->Status != SNMP_UNDERCREATION)
            return SNMP_READONLY;
        memcpy (data->Owner, obj->Syntax.BufChr, obj->SyntaxLen);
        data->OwnerLen = obj->SyntaxLen;
        return SNMP_NOERROR;
    }
}


WORD HostTopNStatus(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL    *local = NULL;
DNPAP_TOPN_CONTROL *data;

    local = MibRmon(obj, TopNControlInstance, idlen, CONTROLINDEXSIZE);

    switch (obj->Request)
    {
    case SNMP_PDU_GET:
        if (local == NULL)
        {
            obj->Syntax.LngInt = SNMP_INVALID;
            return SNMP_NOSUCHNAME;
        }
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->Status;
        return SNMP_NOERROR;
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
        {
            obj->Syntax.LngInt = SNMP_INVALID;
            return SNMP_NOSUCHNAME;
        }
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->Status;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        if (local == NULL)
        {
            switch (obj->Syntax.LngInt)
            {
            case SNMP_CREATEREQUEST:
                if ((local = MibInsert(obj, &TopNControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
                    return SNMP_GENERROR;
                if ((local->Data = DnpapMalloc(sizeof(DNPAP_TOPN_CONTROL))) == NULL)
                    return SNMP_GENERROR;
                data = (DNPAP_TOPN_CONTROL*) local->Data;
                if (TopNCInit(data) == TRUE)
                {
                    data->Status = SNMP_UNDERCREATION;
                    DnpapMessage(DMC_MESSAGE, DTOPN_CREATE, "dtopN: collector %ld created", local->Index);
                    return SNMP_NOERROR;
                }
                DnpapFree(local->Data);
                MibRemove (obj, &TopNControlInstance, idlen, CONTROLINDEXSIZE);
                return SNMP_GENERROR;
            default:
                return SNMP_NOSUCHNAME;
            }
        }
        data = (DNPAP_TOPN_CONTROL*) local->Data;
        switch (data->Status)
        {
        case SNMP_UNDERCREATION:
            switch (obj->Syntax.LngInt)
            {
            case SNMP_VALID:
                if (TopNCStart(data) == TRUE)
                {
                    data->Status = SNMP_VALID;
                    DnpapMessage(DMC_MESSAGE, DTOPN_START, "dtopN: collector %ld active", local->Index);
                    return SNMP_NOERROR;
                }
                return SNMP_GENERROR;
            case SNMP_INVALID:
                if (TopNCStop(data) == TRUE)
                {
                    DnpapFree(local->Data);
                    DnpapMessage(DMC_MESSAGE, DTOPN_DESTROY, "dtopN: collector %ld destroyed", local->Index);
                    if (MibRemove (obj, &TopNControlInstance, idlen, CONTROLINDEXSIZE) == TRUE)
                        return SNMP_NOERROR;
                }
                return SNMP_GENERROR;
            default:
                return SNMP_BADVALUE;
            }
        case SNMP_VALID:
            switch (obj->Syntax.LngInt)
            {
            case SNMP_VALID:
                return SNMP_NOERROR;
            case SNMP_INVALID:
                if (TopNCStop(data) == TRUE)
                {
                    DnpapFree(local->Data);
                    DnpapMessage(DMC_MESSAGE, DTOPN_DESTROY, "dtopN: collector %ld destroyed", local->Index);
                    if (MibRemove (obj, &TopNControlInstance, idlen, CONTROLINDEXSIZE) == TRUE)
                        return SNMP_NOERROR;
                }
                return SNMP_GENERROR;
            default:
                return SNMP_BADVALUE;
            }
        }
    }
}


WORD HostTopNReport(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL   *local = NULL;
DNPAP_TOPN_ENTRY  *data = NULL;

    if ((local = MibRmon (obj, TopNControlInstance, idlen, TOPNINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, TOPNINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:                                          
        if (data == NULL && (data = TopNEntrySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngInt = local->Index;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HostTopNIndex(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL   *local = NULL;
DNPAP_TOPN_ENTRY  *data = NULL;

    if ((local = MibRmon (obj, TopNControlInstance, idlen, TOPNINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, TOPNINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:                                          
        if (data == NULL && (data = TopNEntrySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngInt = obj->Id[idlen+1];
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HostTopNAddress(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL   *local = NULL;
DNPAP_TOPN_ENTRY  *data = NULL;

    if ((local = MibRmon (obj, TopNControlInstance, idlen, TOPNINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, TOPNINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:                                          
        if (data == NULL && (data = TopNEntrySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        memcpy(obj->Syntax.BufChr, data->Address, obj->SyntaxLen = HOST_SZEADDR);
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HostTopNRate(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL   *local = NULL;
DNPAP_TOPN_ENTRY  *data = NULL;

    if ((local = MibRmon (obj, TopNControlInstance, idlen, TOPNINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, TOPNINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:                                          
        if (data == NULL && (data = TopNEntrySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngInt = data->Rate;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


BOOLEAN TopNEntryReset(LONG hostcontrolindex, DNPAP_HOST *host)
{                                 
DNPAP_TOPN_CONTROL *topncontrol;
DNPAP_TOPN_ENTRY   *entry;
MIB_LOCAL    *local;

    for (local = TopNControlInstance; local != NULL; local = local->Next)
    {
        topncontrol = local->Data;
        if (topncontrol->Status == SNMP_VALID && topncontrol->HostIndex == hostcontrolindex &&
            topncontrol->TimeRemaining > 0)
        {
            if ((entry = HashSearch(topncontrol->Table, host->Address, HOST_SZEADDR)) != NULL)
                entry->PrevValue = 0;
        }
    }
    return TRUE;
}


BOOLEAN TopNEntryUpdate(LONG hostcontrolindex, DNPAP_HOST *host)
{                                 
DNPAP_TOPN_CONTROL *topncontrol;
DNPAP_TOPN_ENTRY   *entry;
MIB_LOCAL    *local;

    for (local = TopNControlInstance; local != NULL; local = local->Next)
    {
        topncontrol = local->Data;
        if (topncontrol->Status == SNMP_VALID && topncontrol->HostIndex == hostcontrolindex &&
            topncontrol->TimeRemaining > 0)
        {
            if ((entry = HashSearch(topncontrol->Table, host->Address, HOST_SZEADDR)) == NULL)
            {
                if ((entry = DnpapMalloc(sizeof(DNPAP_TOPN_ENTRY))) == NULL)
                    return FALSE;
                else
                {
                    memset(entry, 0, sizeof(DNPAP_TOPN_ENTRY));
                    memcpy(entry->Address, host->Address, HOST_SZEADDR);

                    entry->Rate = 0;
                    switch (topncontrol->RateBase)
                    {
                    case DTOPN_RATEINPKTS:
                        entry->PrevValue = host->InPkts;
                        break;
                    case DTOPN_RATEOUTPKTS:
                        entry->PrevValue = host->OutPkts;
                        break;
                    case DTOPN_RATEINOCTETS:
                        entry->PrevValue = host->InOctets;
                        break;
                    case DTOPN_RATEOUTOCTETS:
                        entry->PrevValue = host->OutOctets;
                        break;
                    case DTOPN_RATEOUTERRORS:
                        entry->PrevValue = host->OutErrors;
                        break;
                    case DTOPN_RATEOUTBROADCASTPKTS:
                        entry->PrevValue = host->OutBroadcastPkts;
                        break;
                    case DTOPN_RATEOUTMULTICASTPKTS:
                        entry->PrevValue = host->OutMulticastPkts;
                        break;
                    }

                    if (HashAdd(topncontrol->Table, entry->Address, HOST_SZEADDR, entry) == NULL)
                    {
                        DnpapFree(entry);
                        return FALSE;
                    }
                    else
                    {
                        EntryAddList(topncontrol, entry);
                        topncontrol->TableSize++;
                    }
                }
            }
            else
            {
                switch (topncontrol->RateBase)
                {
                case DTOPN_RATEINPKTS:
                    if (entry->PrevValue > host->InPkts)
                        entry->Rate += host->InPkts + (ULONG_MAX - entry->PrevValue);
                    else
                        entry->Rate += host->InPkts - entry->PrevValue;
                    entry->PrevValue = host->InPkts;
                    break;
                case DTOPN_RATEOUTPKTS:
                    if (entry->PrevValue > host->OutPkts)
                        entry->Rate += host->OutPkts + (ULONG_MAX - entry->PrevValue);
                    else
                        entry->Rate += host->OutPkts - entry->PrevValue;
                    entry->PrevValue = host->OutPkts;
                    break;
                case DTOPN_RATEINOCTETS:
                    if (entry->PrevValue > host->InOctets)
                        entry->Rate += host->InOctets + (ULONG_MAX - entry->PrevValue);
                    else
                        entry->Rate += host->InOctets - entry->PrevValue;
                    entry->PrevValue = host->InOctets;
                    break;
                case DTOPN_RATEOUTOCTETS:
                    if (entry->PrevValue > host->OutOctets)
                        entry->Rate += host->OutOctets + (ULONG_MAX - entry->PrevValue);
                    else
                        entry->Rate += host->OutOctets - entry->PrevValue;
                    entry->PrevValue = host->OutOctets;
                    break;
                case DTOPN_RATEOUTERRORS:
                    if (entry->PrevValue > host->OutErrors)
                        entry->Rate += host->OutErrors + (ULONG_MAX - entry->PrevValue);
                    else
                        entry->Rate += host->OutErrors - entry->PrevValue;
                    entry->PrevValue = host->OutErrors;
                    break;
                case DTOPN_RATEOUTBROADCASTPKTS:
                    if (entry->PrevValue > host->OutBroadcastPkts)
                        entry->Rate += host->OutBroadcastPkts + (ULONG_MAX - entry->PrevValue);
                    else
                        entry->Rate += host->OutBroadcastPkts - entry->PrevValue;
                    entry->PrevValue = host->OutBroadcastPkts;
                    break;
                case DTOPN_RATEOUTMULTICASTPKTS:
                    if (entry->PrevValue > host->OutMulticastPkts)
                        entry->Rate += host->OutMulticastPkts + (ULONG_MAX - entry->PrevValue);
                    else
                        entry->Rate += host->OutMulticastPkts - entry->PrevValue;
                    entry->PrevValue = host->OutMulticastPkts;
                    break;
                }
                EntryUpdateList(topncontrol, entry);
            }
        }
    }
    return TRUE;
}
