#include <memory.h>
#include <dnpap.h>
#include <maxmem.h>
#include <message.h>
#include <mac.h>
#include <snmp.h>
#include <mibsup.h>

#include "min.h"

#include "hist_e.h"
#include "hist_c.h"
#include "hist.h"



#define CONTROLINDEXSIZE        1
#define HISTORYINDEXSIZE        2


extern WORD HistoryMaxNrBuckets;


static MIB_LOCAL *HistoryControlInstance = NULL;
                                

static BOOLEAN RmonNext(SNMP_OBJECT *obj, MIB_LOCAL **local, WORD idlen, WORD indexsize, ETHER_HISTORY **elem);

static ETHER_HISTORY* HistoryGetFirstNext(HISTORY_CONTROL *historycontrol, SNMP_OBJECT *obj, WORD idlen, BOOLEAN first);



BOOLEAN HistoryMInit(VOID)
{
    MessageConfig(HISTORY_ERROR, "History");
    return HistoryConfigInit();
}


BOOLEAN RmonNext(SNMP_OBJECT *obj, MIB_LOCAL **local, WORD idlen, WORD indexsize, ETHER_HISTORY **elem)
{
BOOLEAN first = FALSE;
HISTORY_CONTROL *data;

    if (*local == NULL || indexsize < 1)
        return FALSE;

    if (obj->IdLen < idlen + indexsize)
    {
        switch (indexsize)
        {
        case HISTORYINDEXSIZE:
            while (*local != NULL && 
                    (((HISTORY_CONTROL*)(*local)->Data)->BucketsGranted == 0 ||
                     (((HISTORY_CONTROL*)(*local)->Data)->BucketsWrapped == FALSE &&
                      ((HISTORY_CONTROL*)(*local)->Data)->BucketsPos == 0)))
                *local = (*local)->Next;
            if (*local == NULL)
                return FALSE;
            obj->Id[idlen] = (*local)->Index;
            obj->Id[idlen+1] = 1;
            obj->IdLen = idlen + HISTORYINDEXSIZE;
            *elem = EtherHistorySearch((HISTORY_CONTROL*)(*local)->Data, obj, idlen);
            break;
        case CONTROLINDEXSIZE:
            obj->Id[idlen] = (*local)->Index;
            obj->IdLen = idlen + CONTROLINDEXSIZE;
            break;
        }
        return TRUE;
    }
    else
    {
        switch (indexsize)
        {            
        case HISTORYINDEXSIZE:      
            data = (HISTORY_CONTROL*)(*local)->Data;
            if ((data->BucketsWrapped == TRUE && obj->Id[idlen+1] < data->BucketsGranted) ||
                (data->BucketsWrapped == FALSE && obj->Id[idlen+1] < data->BucketsPos))
            {                   /*  BucketsWrapped == FALSE and BucketsPos == 0 if Status != VALID  */
                obj->Id[idlen] = (*local)->Index;
                obj->Id[idlen+1]++;
                obj->IdLen = idlen + HISTORYINDEXSIZE;
            	*elem = EtherHistorySearch((HISTORY_CONTROL*)(*local)->Data, obj, idlen);
                break;
            }
            *local = (*local)->Next;
            while (*local != NULL && 
                    (((HISTORY_CONTROL*)(*local)->Data)->BucketsGranted == 0 ||
                     (((HISTORY_CONTROL*)(*local)->Data)->BucketsWrapped == FALSE &&
                      ((HISTORY_CONTROL*)(*local)->Data)->BucketsPos == 0)))
                *local = (*local)->Next;
            if (*local == NULL)
                return FALSE;
            obj->Id[idlen] = (*local)->Index;
            obj->Id[idlen+1] = 1;
            obj->IdLen = idlen + HISTORYINDEXSIZE;
            *elem = EtherHistorySearch((HISTORY_CONTROL*)(*local)->Data, obj, idlen);
            break;
        case CONTROLINDEXSIZE:
            *local = (*local)->Next;
            if (*local == NULL)
                return FALSE;
            obj->Id[idlen] = (*local)->Index;
            obj->IdLen = idlen + CONTROLINDEXSIZE;
            break;
        }
        return TRUE;
    }
    return FALSE;
}


WORD HistoryControlIndex(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL        *local = NULL;

    if ((local = MibRmon(obj, HistoryControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        obj->Syntax.LngInt = local->Index;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HistoryControlDataSource(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL        *local = NULL;
HISTORY_CONTROL  *data;
MAC_IFACE        *iface;

    if ((local = MibRmon(obj, HistoryControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext(obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (HISTORY_CONTROL*) local->Data;
        memcpy(obj->Syntax.BufInt, data->Source, data->SourceLen*sizeof(data->Source[0]));
        obj->SyntaxLen = data->SourceLen;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (HISTORY_CONTROL*) local->Data;
        if (data->Status != SNMP_UNDERCREATION)
            return SNMP_READONLY;
        if (data->SourceLen != obj->SyntaxLen ||
            memcmp(data->Source, obj->Syntax.BufInt, (data->SourceLen-1)*sizeof(data->Source[0])) != 0)
            return SNMP_BADVALUE;
        if ((iface = MacIfaceGet((WORD) obj->Syntax.BufInt[obj->SyntaxLen-1])) == NULL)
            return SNMP_BADVALUE;
        memcpy(data->Source, obj->Syntax.BufInt, obj->SyntaxLen*sizeof(data->Source[0]));
        data->SourceLen = obj->SyntaxLen;
        data->Iface = iface;
        return SNMP_NOERROR;
    }
}


WORD HistoryControlBucketsRequested(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL        *local = NULL;
HISTORY_CONTROL  *data;
ETHER_HISTORY    *etherentries;
LONG             i, ni, j;

    if ((local = MibRmon(obj, HistoryControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (HISTORY_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->BucketsRequested;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (HISTORY_CONTROL*) local->Data;
        if (obj->Syntax.LngInt < 1)
            return SNMP_BADVALUE;
        data->BucketsGranted = Min(MemoryGetAvail(), MemoryGetMaxChunk())/sizeof(ETHER_HISTORY);
        if (data->BucketsGranted > Min(obj->Syntax.LngInt, HistoryMaxNrBuckets))
            data->BucketsGranted = Min(obj->Syntax.LngInt, HistoryMaxNrBuckets);
        if (data->BucketsGranted == data->BucketsAllocated)
        {
            data->BucketsRequested = obj->Syntax.LngInt;
            return SNMP_NOERROR;
        }
        if ((etherentries = NewEtherHistory(data->BucketsGranted)) == NULL)
        {   
            DelEtherHistory(data->EtherBuckets);
            data->BucketsPos = 0;
            data->BucketsGranted = 0;
            data->BucketsAllocated = 0;
            data->BucketsWrapped = FALSE;
        }
        else
        {
            if (data->BucketsWrapped == TRUE)
                ni = Min(data->BucketsGranted, data->BucketsAllocated);
            else
                ni = Min(data->BucketsGranted, data->BucketsPos);
            for (i = 0; i < ni; i++)                       
            {            
                j = data->BucketsPos - ni + i;
                if (j < 0)
                    j = j + data->BucketsAllocated;
                memcpy(etherentries+i, data->EtherBuckets+j, sizeof(ETHER_HISTORY));
            }
            DelEtherHistory(data->EtherBuckets);
            data->EtherBuckets = etherentries;
            data->BucketsPos = ni;
            data->BucketsAllocated = data->BucketsGranted;
            data->BucketsWrapped = FALSE;
        }
        data->BucketsRequested = obj->Syntax.LngInt;
        return SNMP_NOERROR;
    }
}


WORD HistoryControlBucketsGranted(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL        *local = NULL;
HISTORY_CONTROL  *data;

    if ((local = MibRmon(obj, HistoryControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (HISTORY_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->BucketsGranted;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD HistoryControlInterval(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL        *local = NULL;
HISTORY_CONTROL  *data;

    if ((local = MibRmon(obj, HistoryControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
        
    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (HISTORY_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->Interval;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (HISTORY_CONTROL*) local->Data;
        if (data->Status != SNMP_UNDERCREATION)
            return SNMP_READONLY;
        if (obj->Syntax.LngInt < 1)
            return SNMP_BADVALUE;
        data->Interval = obj->Syntax.LngInt;
        return SNMP_NOERROR;
    }
}


WORD HistoryControlOwner(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL        *local = NULL;
HISTORY_CONTROL  *data;

    if ((local = MibRmon(obj, HistoryControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext(obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (HISTORY_CONTROL*) local->Data;
        memcpy(obj->Syntax.BufChr, data->Owner, data->OwnerLen);
        obj->SyntaxLen = data->OwnerLen;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (HISTORY_CONTROL*) local->Data;
        if (data->Status != SNMP_UNDERCREATION)
            return SNMP_READONLY;
        memcpy(data->Owner, obj->Syntax.BufChr, obj->SyntaxLen);
        data->OwnerLen = obj->SyntaxLen;
        return SNMP_NOERROR;
    }
}


WORD HistoryControlStatus(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL        *local = NULL;
HISTORY_CONTROL  *data;

    local = MibRmon(obj, HistoryControlInstance, idlen, CONTROLINDEXSIZE);

    switch (obj->Request)
    {
    case SNMP_PDU_GET:
        if (local == NULL)
        {
            obj->Syntax.LngInt = SNMP_INVALID;
            return SNMP_NOSUCHNAME;
        }
        data = (HISTORY_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->Status;
        return SNMP_NOERROR;
    case SNMP_PDU_NEXT:
        if (RmonNext(obj, &local, idlen, CONTROLINDEXSIZE, NULL) == FALSE)
        {
            obj->Syntax.LngInt = SNMP_INVALID;
            return SNMP_NOSUCHNAME;
        }
        data = (HISTORY_CONTROL*) local->Data;
        obj->Syntax.LngInt = data->Status;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        if (local == NULL)
        {
            switch (obj->Syntax.LngInt)
            {
            case SNMP_CREATEREQUEST:
                if ((local = MibInsert(obj, &HistoryControlInstance, idlen, CONTROLINDEXSIZE)) == NULL)
                    return SNMP_GENERROR;
                if ((local->Data = DnpapMalloc(sizeof(HISTORY_CONTROL))) == NULL)
                    return SNMP_GENERROR;
                data = (HISTORY_CONTROL*) local->Data;
                memset(data, 0, sizeof(HISTORY_CONTROL));
                if (HistoryCInit(data) == TRUE)
                {
                    data->Status = SNMP_UNDERCREATION;
                    DnpapMessage(DMC_MESSAGE, HISTORY_CREATE, "history: collector %ld created", local->Index);
                    return SNMP_NOERROR;
                }
                DnpapFree(local->Data);
                MibRemove (obj, &HistoryControlInstance, idlen, CONTROLINDEXSIZE);
                return SNMP_GENERROR;
            default:
                return SNMP_NOSUCHNAME;
            }
        }
        data = (HISTORY_CONTROL*) local->Data;
        switch (data->Status)
        {
        case SNMP_UNDERCREATION:
            switch (obj->Syntax.LngInt)
            {
            case SNMP_VALID:
                if (HistoryCStart(data) == TRUE)
                {
                    data->Status = SNMP_VALID;
                    DnpapMessage(DMC_MESSAGE, HISTORY_ACTIVE, "history: collector %ld active", local->Index);
                    return SNMP_NOERROR;
                }
                return SNMP_GENERROR;
            case SNMP_INVALID:
                if (HistoryCStop(data) == TRUE)
                {
                    DnpapFree(local->Data);
                    DnpapMessage(DMC_MESSAGE, HISTORY_DESTROY, "history: collector %ld destroyed", local->Index);
                    if (MibRemove (obj, &HistoryControlInstance, idlen, CONTROLINDEXSIZE) == TRUE)
                        return SNMP_NOERROR;
                }
                return SNMP_GENERROR;
            default:
                return SNMP_BADVALUE;
            }
        case SNMP_VALID:
            switch (obj->Syntax.LngInt)
            {
            case SNMP_VALID:
                return SNMP_NOERROR;
            case SNMP_INVALID:
                if (HistoryCStop(data) == TRUE)
                {
                    DnpapFree(local->Data);
                    DnpapMessage(DMC_MESSAGE, HISTORY_DESTROY, "history: collector %ld destroyed", local->Index);
                    if (MibRemove (obj, &HistoryControlInstance, idlen, CONTROLINDEXSIZE) == TRUE)
                        return SNMP_NOERROR;
                }
                return SNMP_GENERROR;
            default:
                return SNMP_BADVALUE;
            }
        }
    }
}


WORD EtherHistoryIndex(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        obj->Syntax.LngInt = local->Index;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistorySampleIndex(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngInt = obj->Id[idlen+1];
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryIntervalStart(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->IntervalStart;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryDropEvents(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->DropEvents;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryOctets(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->Octets;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryPkts(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->Pkts;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryBroadcastPkts(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->BroadcastPkts;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryMulticastPkts(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->MulticastPkts;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryCRCAlignErrors(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->CRCAlignErrors;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryUndersizePkts(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->UndersizePkts;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryOversizePkts(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->OversizePkts;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryFragments(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->Fragments;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryJabbers(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->Jabbers;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryCollisions(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->Collisions;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


WORD EtherHistoryUtilization(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL       *local = NULL;
ETHER_HISTORY   *data = NULL;

    if ((local = MibRmon (obj, HistoryControlInstance, idlen, HISTORYINDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (obj, &local, idlen, HISTORYINDEXSIZE, &data) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        if (data == NULL && (data = EtherHistorySearch(local->Data, obj, idlen)) == NULL)
            return SNMP_NOSUCHNAME;
        obj->Syntax.LngUns = data->Utilization;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        return SNMP_READONLY;
    }
}


