/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          channelm.c
** SYSTEM NAME:        beholder
** MODULE NAME:        channel
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/9/3
** DESCRIPTION:        channel of the RMON filter group
**                     each function is preceded by channel
**                          Index, IfIndex, AcceptType, DataControl,
**                          TurnOnEventIndex, TurnOffEventIndex,
**                          EventIndex, EventStatus, Matches, Description
*************************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <mibsup.h>
#include <message.h>
#include <mac.h>

#include "channelc.h"
#include "channel.h"


#define INDEXSIZE   1


static MIB_LOCAL    *channel = NULL;

static BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, void *Elm);



BOOLEAN ChannelMInit(VOID)
{
    MessageConfig(CHANNEL_ERROR, "Channel");
    return TRUE;
}


/*****************************************************************
** NAME:        RmonNext
** SYNOPSIS:    BOOLEAN RmonNext (SNMP_OBJECT *Obj,
**                                  MIB_LOCAL **Local, WORD IdLen,
**                                  WORD IdSze, VOID **Elm)
** PARAMETERS:  Obj: requested object
**              Local: local datastructure: a collector
**              IdLen: identifier length known to the MIB
**              IdSze: size part application of identifier
**                     including the collector index !!
**              Elm: pointer to object if tables are used
** DESCRIPTION: application specific RMON NEXT function.
**              only called after MibRmon() for next requests 
**              searches the next object in the collector
** REMARKS:     ONLY FOR INTERNAL USE
** RETURNS:     TRUE: object found
**                    OPTIONAL: Elm -> pointer to object in table
*******************************************************************/

static BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, VOID **Elm)
{
    MIB_LOCAL    *local = *Local;

    if (local == NULL)
        return FALSE;
    if (Obj->IdLen == IdLen || local->Index > Obj->Id[IdLen])
    {
        /********************************************************************
        **  OPTIONAL: search function to find first obj in a table !! 
        **  if (not found)
        **      search next collector
        **  adjust object identifier to identifier first object
        **  Elm -> found table obj
        ********************************************************************/
        Obj->IdLen = IdLen + IdSze;
        Obj->Id[IdLen] = local->Index;
        return TRUE;
    }
    /********************************************************************
    **  OPTIONAL: search function to find next obj in a table !! 
    **  if (found)
    **      adjust object identifier to identifier next object
    **      Elm -> found table obj
    **      return TRUE
    ********************************************************************/
    *Local = local->Next;
    return (RmonNext(Obj, Local, IdLen, IdSze, Elm));
}


WORD channelIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);
    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            Obj->Syntax.LngInt = local->Index;
            return SNMP_NOERROR;
    }
}

WORD channelIfIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    MAC_IFACE   *iface;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->IfIndex;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            if ((iface = MacIfaceGet((WORD) Obj->Syntax.LngInt)) == NULL)
                return SNMP_BADVALUE;
            data->IfIndex = Obj->Syntax.LngInt;
            data->Iface = iface;
            return SNMP_NOERROR;
    }
}

WORD channelAcceptType (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->AcceptType;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            if (Obj->Syntax.LngInt != 1 && Obj->Syntax.LngInt != 2)
                return SNMP_BADVALUE;
            data->AcceptType = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD channelDataControl (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->DataControl;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            if (Obj->Syntax.LngInt != 1 && Obj->Syntax.LngInt != 2)
                return SNMP_BADVALUE;
            data->DataControl = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD channelTurnOnEventIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->TurnOnEventIndex;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            data->TurnOnEventIndex = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD channelTurnOffEventIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->TurnOffEventIndex;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            data->TurnOffEventIndex = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD channelEventIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->EventIndex;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            data->EventIndex = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}


WORD channelEventStatus (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->EventStatus;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            if (Obj->Syntax.LngInt != 1 && Obj->Syntax.LngInt != 3)
                return SNMP_BADVALUE;
            data->EventStatus = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}


WORD channelMatches (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->Matches;
            return SNMP_NOERROR;
    }
}


WORD channelDescription (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon (Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            memcpy (Obj->Syntax.BufChr, data->Description, data->DescriptionLen);
            Obj->SyntaxLen = data->DescriptionLen;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            memcpy (data->Description, Obj->Syntax.BufChr, Obj->SyntaxLen);
            data->DescriptionLen = Obj->SyntaxLen;
            return SNMP_NOERROR;
    }
}


WORD channelOwner (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    if ((local = MibRmon(Obj, channel, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (CHANNEL*) local->Data;
            memcpy (Obj->Syntax.BufChr, data->Owner, data->OwnerLen);
            Obj->SyntaxLen = data->OwnerLen;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (CHANNEL*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            memcpy (data->Owner, Obj->Syntax.BufChr, Obj->SyntaxLen);
            data->OwnerLen = Obj->SyntaxLen;
            return SNMP_NOERROR;
    }
}

WORD channelStatus (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CHANNEL     *data;

    local = MibRmon(Obj, channel, IdLen, INDEXSIZE);

    switch (Obj->Request)
    {
        case SNMP_PDU_GET:
            if (local == NULL)
            {
                Obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            {
                Obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (CHANNEL*) local->Data;
            Obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            if (local == NULL)
            {
                switch (Obj->Syntax.LngInt)
                {
                    case SNMP_CREATEREQUEST:
                        if ((local = MibInsert(Obj, &channel, IdLen, INDEXSIZE)) == NULL)
                            return SNMP_GENERROR;
                        if ((local->Data = DnpapMalloc(sizeof(CHANNEL))) == NULL)
                            return SNMP_GENERROR;
                        data = (CHANNEL*) local->Data;
                        memset(data, 0, sizeof(CHANNEL));
                        if (ChannelInit(data) == TRUE)
                        {
                            data->Index = local->Index;
                            data->Status = SNMP_UNDERCREATION;
                            DnpapMessage(DMC_MESSAGE, CHANNEL_ERROR, "channel: collector %ld created", local->Index);
                            return SNMP_NOERROR;
                        }
                        DnpapFree(local->Data);
                        MibRemove (Obj, &channel, IdLen, INDEXSIZE);
                        return SNMP_GENERROR;
                    default:
                        return SNMP_NOSUCHNAME;
                }
            }
            data = (CHANNEL*) local->Data;
            switch (data->Status)
            {
                case SNMP_UNDERCREATION:
                    switch (Obj->Syntax.LngInt)
                    {
                        case SNMP_VALID:
                            if (ChannelStart(data) == TRUE)
                            {
                                data->Status = SNMP_VALID;
                                DnpapMessage(DMC_MESSAGE, CHANNEL_ERROR, "channel: collector %ld active", local->Index);
                                return SNMP_NOERROR;
                            }
                            else
                                return SNMP_GENERROR;
                        case SNMP_INVALID:
                            if (ChannelStop(data) == TRUE)
                            {
                                DnpapFree(local->Data);
                                DnpapMessage(DMC_MESSAGE, CHANNEL_ERROR, "channel: collector %ld destroyed", local->Index);
                                if (MibRemove (Obj, &channel, IdLen, INDEXSIZE) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
                case SNMP_VALID:
                    switch (Obj->Syntax.LngInt)
                    {
                        case SNMP_INVALID:
                            if (ChannelStop(data) == TRUE)
                            {
                                DnpapFree(local->Data);
                                DnpapMessage(DMC_MESSAGE, CHANNEL_ERROR, "channel: collector %ld destroyed", local->Index);
                                if (MibRemove (Obj, &channel, IdLen, INDEXSIZE) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        case SNMP_VALID:
                            return SNMP_NOERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
            }
    }
}

VOID channelEventGenerated(LONG eventIndex)
{
    MIB_LOCAL   *local;
    CHANNEL     *data;

    for (local = channel; local != NULL; local = local->Next)
    {
        data = (CHANNEL*) local->Data;
        if (eventIndex == data->TurnOnEventIndex)
            data->DataControl = 1;
        if (eventIndex == data->TurnOffEventIndex)
            data->DataControl = 2;
    }
}
