/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          channelc.c
** SYSTEM NAME:        beholder
** MODULE NAME:        channel
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/9/16
** DESCRIPTION:        channel of the RMON MIB filter group
*************************************************************************/


#include <stdlib.h>
#include <dnpap.h>
#include <message.h>
#include <mac.h>
#include <filter.h>
#include <event.h>
#include <capture.h>
#include <trap.h>

#include "channelc.h"


static VOID ChannelMacCallback(MAC_COLL *collector, PROT_PKT *pkt);



/*****************************************************************
** NAME:        ChannelInit
** SYNOPSIS:    BOOLEAN ChannelInit (CHANNEL *channel)
** PARAMETERS:  pointer to data etherChannels entry
** DESCRIPTION: initializes collector.
** REMARKS:     called when status is CREATE_REQUEST: see channelm.c
** RETURNS:     TRUE: everything OK
**              FALSE: timer or net not initialized
*******************************************************************/

BOOLEAN ChannelInit (CHANNEL *channel)
{
    channel->IfIndex = 1;               /* first interface */
    channel->AcceptType = 1;            /* accept if filters TRUE */
    channel->DataControl = 2;           /* no flow through channel */
    channel->EventStatus = 1;           /* only one event is fired */

    if ((channel->Iface =
        MacIfaceGet((WORD) channel->IfIndex)) == NULL)
    {
        DnpapMessage(DMC_ERROR, CHANNEL_ERROR, "channel: network init");
        return (FALSE);
    }
    return (TRUE);
}


/*****************************************************************
** NAME:        ChannelStart
** SYNOPSIS:    BOOLEAN ChannelStart (CHANNEL *channel)
** PARAMETERS:  pointer to data etherChannels entry
** DESCRIPTION: starts collector.
** REMARKS:     called when status is VALID: see channelm.c
** RETURNS:     TRUE: everything OK
**              FALSE: timer or net not registered
*******************************************************************/

BOOLEAN ChannelStart (CHANNEL *channel)
{
    channel->Coll.Rcve       = ChannelMacCallback;
    channel->Coll.specific   = channel;
    if (!MacCollRegister(&(channel->Coll)))
    {
        DnpapMessage(DMC_ERROR, CHANNEL_ERROR, "channel: network init");
        return (FALSE);
    }
    return(TRUE);
}


/*****************************************************************
** NAME:        ChannelStop 
** SYNOPSIS:    BOOLEAN ChannelcStop (CHANNEL *etherChannels)
** PARAMETERS:  pointer to etherChannels entry
** DESCRIPTION: stops collector.
** REMARKS:     called when status is INVALID: see channelm.c
** RETURNS:     TRUE: timer & net registration removed 
*******************************************************************/

BOOLEAN ChannelStop (CHANNEL *channel)
{
    MacCollRemove (&(channel->Coll));
    return(TRUE);
}


/*****************************************************************
** NAME:        ChannelNetCallback
** PARAMETERS:  see Net-manual
** DESCRIPTION: analyses packets received from net
** REMARKS:     none
** RETURNS:     VOID
*******************************************************************/

VOID ChannelMacCallback(MAC_COLL *collector, PROT_PKT *pkt)
{
    BOOLEAN filterFlag = FALSE, packetMatch;
    CHANNEL *channel = collector->specific;

    filterFlag = FilterPkt (channel->Index, pkt);
    if ((channel->AcceptType == 1 && filterFlag == TRUE) ||
        (channel->AcceptType == 2 && filterFlag == FALSE))
    {
        channel->Matches++;
        packetMatch = TRUE;
    }
    else
        packetMatch = FALSE;

    if (channel->DataControl == 1)      /* channel on */
    {
        if (packetMatch == TRUE)
        {
            switch (channel->EventStatus)
            {
            case 1:     /* event ready */
                if (GenerateEvent("channel", channel->EventIndex, SNMP_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_PACKETMATCH, NULL, 0) == TRUE)
                    channel->EventStatus = 2;
                break;
            case 3:     /* event always ready */
                GenerateEvent("channel", channel->EventIndex, SNMP_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_PACKETMATCH, NULL, 0);
                break;
            case 2:     /* event fired once */
                break;
            }
            CapturePkt (channel->Index, pkt);
        }
    }
    return;
}
