/*
 * mvi - move interactive
 *      Original by TyeMcQueen, somewhere comp.sources.misc.
 *      Heavyly modified by akira@sra.co.jp, as OS/2 version.
 */

#include <stdio.h.>
#include <stdlib.h>
#include <string.h>
#include <process.h>

/*
 * Usage & Command Line Arguments
 */

static  char    *MyName = "mvi" ;
static  char    *Editor = NULL  ;
static  int     Verbose = 0     ;
static  int     Debug   = 0     ;

#define DEF_EDITOR  "vi"    /* USE non GUI editor */

static  void    usage(void)
{
    fprintf(stderr,
        "usage: %s [ -v ] [ -e editor ] wildcard(s)\n"
        " -v  Verbose - echo each name change as it is done.\n"
        " -e  Editor  - specifies an alternate editor to use.\n"
	"         (instead of that specified by environment variable \n"
	"         VISUAL or EDITOR or just `%s') and optionally specifies\n"
	"         command line arguments for it.\n",
	MyName, DEF_EDITOR) ;
        exit(1) ;
}

static  void    badarg(char *str)
{
    fprintf(stderr, "%s : %s\n", MyName, str) ;
    exit(1) ;
}

static  char    **flags(int *ac, char **av)
{
    int     i ;

    MyName = *av ;
    
    for (i = 1 ; i < *ac ; i++) {
        if (av[i][0] != '-') {
	    break ;
        }
	switch (av[i][1]) {
        case 'e' : case 'E' :
	    if ((Editor = av[i+=1]) == NULL) {
	        badarg("Name of editor must follow -e.") ;
	    }
            break ;

        case 'v' : case 'V' :
	    Verbose = !Verbose ;
	    break ;

	case 'd' : case 'D' :
	    Debug += 1 ;
	    break ;
	
        case 'h' : case 'H' : case '?' :
	    usage() ;
	    break ;

        default :
            {
	        char    msg[256] ;
		sprintf(msg, "Unknown switch `%s'.", av[i]) ;
		badarg(msg) ;
            }
        }
    }

    if (NULL == Editor) {
        Editor = getenv("VISUAL") ;
    }
    if (NULL == Editor) {
        Editor = getenv("EDITOR") ;
    }
    if (NULL == Editor) {
        Editor = DEF_EDITOR ;
    }

    *ac -= i ;
    return(&av[i]) ;
}

/*
 * Sort arguments, 'cause EMX does not sort expanded wildcards
 */

static  int     compare(const void *p1, const void *p2)
{
    char    **s1 = (char **) p1 ;
    char    **s2 = (char **) p2 ;
    
    return(strcmp(*s1, *s2)) ;
}

static  void    sort_args(int ac, char *av[])
{
    qsort(av, ac, sizeof(char *), compare) ;
}

/*
 * Writing Name List to Temporary File
 */

#define NAMSIZ  256

static  char    TempName[NAMSIZ] ;
static  int     TempExist = 0    ;

static  char    TempHead[] =
    "Edit file name to rename, \"D file\" to delete\n" ;

static  void    cleanup(int err)
{
    if (TempExist) {
        unlink(TempName) ;
    }
    exit(err) ;
}

static  int     write_temp(char **av)
{
    int     i, pid ;
    char    *tmpdir = NULL ;
    FILE    *fp ;

    if (Debug == 1 || Debug == 2) {
        fprintf(stderr, "write_temp\n") ;
    }

    pid = getpid() ;
    
    if (tmpdir == NULL) {
        tmpdir = getenv("TMP") ;
    }
    if (tmpdir == NULL) {
        tmpdir = getenv("TEMP") ;
    }
    if (tmpdir == NULL) {
        tmpdir =  "." ;
    }
    sprintf(TempName, "%s/mvi%05.5d", tmpdir, pid) ;

    if ((fp = fopen(TempName, "w")) == NULL) {
        fprintf(stderr,
	    "%s : cannot create temp. file `%s\n'", MyName, TempName) ;
        return(11) ;
    }
    TempExist = 1 ;
    
    fprintf(fp, TempHead) ;

    for (i = 0 ; av[i] != NULL ; i++) {
        fprintf(fp, "%s\n", av[i]) ;
    }    
    fclose(fp) ;
    return(0) ;
}

/*
 * Edit / Check Name List in Temp. File
 */

static  char    *patt[] = {
    "%s.EXE", "%s.CMD", "%s.COM", NULL 
} ;

static  int     edit_temp(void)
{
    int     i = 0 ;
    char    *tok  ;
    char    *delim = " \n\r" ;
    char    *args[32]   ;
    char    edt[NAMSIZ] ;
    char    exe[NAMSIZ] ;

    if (Debug == 1 || Debug == 3) {
        fprintf(stderr, "edit_temp\n") ;
    }

    tok = strtok(Editor, delim) ;
    while (tok != NULL && i < 30) {
        args[i++] = tok ;
	tok = strtok(NULL, delim) ;
    }
    args[i++] = TempName ;
    args[i] = NULL       ;

    for (i = 0 ; patt[i] != NULL ; i++) {
        sprintf(edt, patt[i], args[0]) ;
        if (_path(exe, edt) == 0) {
	    break ;
        }
    }
    if (patt[i] == NULL) {
        fprintf(stderr,
	    "%s : editor `%s' not found\n", MyName, args[0]) ;
        return(21) ;
    }
    if (spawnv(P_WAIT, exe, args) == -1) {
        fprintf(stderr,
	    "%s : cannot start editor `%s'\n", MyName, args[0]) ;
        return(22) ;
    } else {
        printf("\n") ; fflush(stdout) ;
        return(0) ;
    }
}

static  int     check_temp(char **av)
{
    int     i   ;
    FILE    *fp ;
    char    new[256] ;

    if (Debug == 1 || Debug == 4) {
        fprintf(stderr, "check_temp\n") ;
    }

    if ((fp = fopen(TempName, "r")) == NULL) {
        fprintf(stderr,
	    "%s : cannot open temp. file `%s'\n", MyName, TempName) ;
        return(31) ;
    }
    
    fgets(new, sizeof(new), fp) ;
    if (strcmp(new, TempHead) != 0) {
        fprintf(stderr,
	    "%s : First line modified. Abort.\n", MyName) ;
        return(32) ;
    }

    for (i = 0 ; fgets(new, sizeof(new), fp) != NULL ; i++) {
        if (av[i] == NULL) {
	    fprintf(stderr,
	        "%s : Extra line(s) added. Abort.\n", MyName) ;
            fclose(fp) ;
	    return(33) ;
        }
    }
    if (av[i] != NULL) {
        fprintf(stderr,
            "%s : Line(s) deleted. Abort.\n", MyName) ;
        fclose(fp) ;
        return(34) ;
    }
    fclose(fp) ;
    return(0)  ;
}

/*
 * Rename / Report
 */

static  int     numRename = 0 ;
static  int     numDelete = 0 ;
static  int     numLinked = 0 ;     /* not use for OS2 */

static  int     do_remove(char *org, char *new)
{
    if (strcmp(org, new) != 0) {
        fprintf(stderr,
	    "%s : Deleting name was modified `%s' ->  `%s'. Abort.\n",
	    MyName, org, new) ;
        return(0) ;
    }
    if (unlink(org) != 0) {
        fprintf(stderr,
	    "%s : Error deleting `%s'. \n", MyName, org) ;
        return(0) ;
    }
    if (Verbose) {
        fprintf(stdout, "rm %s\n", org) ; fflush(stdout) ;
    }

    numDelete += 1 ;
    
    return(0) ;
}

static  int     do_rename_special(char *org, char *new) ;

static  int     do_rename(char *org, char *new)
{
    if (stricmp(org, new) == 0) {   /* for FS ignore cases */
        return(do_rename_special(org, new)) ;
    }
    if (rename(org, new) != 0) {
        fprintf(stderr,
	    "%s : Error renaming `%s' -> `%s'.\n",
	    MyName, org, new) ;
        return(0) ;
    }
    if (Verbose) {
        fprintf(stdout, "mv %s %s\n", org, new) ; fflush(stdout) ;
    }
    
    numRename += 1 ;
    
    return(0) ;
}

static  int     do_rename_special(char *org, char *new)
{
    char    *tmpname = "mvi.tmp" ;
    
    if (rename(org, tmpname) != 0) {
        fprintf(stderr,
	    "%s : Error renaming `%s' -> `%s'.\n",
	    MyName, org, new) ;
        return(0) ;
    }
    if (rename(tmpname, new) != 0) {
        fprintf(stderr,
	    "%s : Error renaming `%s' -> `%s'.\n",
	    MyName, org, new) ;
        return(0) ;
	rename(tmpname, org) ;
    }
    if (Verbose) {
        fprintf(stdout, "mv %s %s\n", org, new) ; fflush(stdout) ;
    }
    
    numRename += 1 ;

    return(0) ;
}

static  int     do_move(char **av)
{
    int     i, err ;
    FILE    *fp ;
    char    *cp ;
    char    new[256] ;

    if (Debug == 1 || Debug == 5) {
        fprintf(stderr, "do_move\n") ;
    }

    if ((fp = fopen(TempName, "r")) == NULL) {
        fprintf(stderr,
	    "%s : cannot open temp. file `%s'\n", MyName, TempName) ;
        return(41) ;
    }
    
    fgets(new, sizeof(new), fp) ;
    if (strcmp(new, TempHead) != 0) {
        fprintf(stderr,
	    "%s : First line modified. Abort.\n", MyName) ;
        return(0) ;
    }

    for (i = 0 ; fgets(new, sizeof(new), fp) != NULL ; i++) {
        if (av[i] == NULL) {
	    fprintf(stderr,
	        "%s : Extra line(s) added. Abort.\n", MyName) ;
            fclose(fp) ;
	    return(0) ;
        }
	cp = new + strlen(new) - 1 ;
	if (*cp !=  '\n') {
	    fprintf(stderr,
	        "%s : New name is longer than %d characters. Abort.\n",
                MyName, sizeof(new)) ;
            fclose(fp) ;
            return(0) ;
        }
	*cp = '\0' ;
	
	if (strncmp(new, "D ", 2) == 0) {
	    err = do_remove(av[i], &new[2]) ;
        } else if (strcmp(av[i], new) != 0) {
	    err = do_rename(av[i], new) ;
        } else {
	    err = 0 ;
        }
	if (err != 0) {
	    fclose(fp)  ;
	    return(err) ;
        }
    }
    if (av[i] != NULL) {
        fprintf(stderr,
            "%s : Line(s) deleted. Abort.\n", MyName) ;
        fclose(fp) ;
        return(0) ;
    }
    fclose(fp) ;
    return(0)  ;
}

static  void    summarize(void)
{
    fprintf(stderr,
        "%s : remaned %d, deleted %d files\n", MyName, numRename, numDelete) ;
    fflush(stderr) ;
}

/*
 * the  M A I N
 */

int     main(int ac, char **av)
{
    int     err ;

    _wildcard(&ac, &av) ;       /* Expand Arguments on EMX */
    
    av = flags(&ac, av) ;

    if (ac <= 0) {
        usage() ;
    }
    
    sort_args(ac, av) ;

    if ((err = write_temp(av)) != 0) {
        cleanup(err) ;
    }
    if ((err = edit_temp()) != 0) {
        cleanup(err) ;
    }
    if ((err = check_temp(av)) != 0) {
        cleanup(err) ;
    }
    if ((err = do_move(av)) != 0) {
        cleanup(err) ;
    }
    summarize() ;
    cleanup(0) ;
}
