/*
program: file2dat.cmd
type:    REXXSAA-OS/2
purpose: allow transformations and calculations on date/time stamps of files.
version: 1.0
date:    Tue  95-10-31
changed:

author:  Rick Curry

Based on "datergf" work by:
         Rony G. Flatscher,
         Wirtschaftsuniversitt/Vienna
         Rony.Flatscher@wu-wien.ac.at

usage:   FILE2DAT(argument)

         Where argument = STREAM( FileName, 'C', 'query datetime' )

remark:  This program only works for date/time values which are possible on
         a DOS style machine: dates prior to 1980 (DOS beginning of time) are
         not handled.  Thus 1/1/1980 is trivialized to = 722814.  Note that the
         year 2000 is a leap year, thus trivializing the equation further.
         Therefore this algorithm is generally OK until the year 2100 at which
         time all matters will be settled religiously according to Heinlein.

         To compare the value generated by this algorithm with the REXX date
         and time functions, the following relationship should hold.  Note that
         the date and time stamps of the file named 'nul' is always now.
         Unfortunately, the STREAM function will not return this information
         for any device (such as nul), so that while the relationship
         does hold, the function below can not be used to test it :( --
         for testing I used the file 'c:\os2\system\swapper.dat'.

         evaluate 'now = ' Date('B')||'.'||Time('S')/86400
         FILE2DAT(STREAM( 'nul', 'C', 'query datetime' )) = now

procedures:
      DATE2DAYS:   calculate days based on 0000/01/01 (== 1)
      TIME2FRACT:  calculate decimal fraction for time

*/


IF ARG(1) = ''|ARG(1) = '?' THEN SIGNAL usage

/* invoked as a COMMAND (from command-line) or as a FUNCTION ? */
PARSE SOURCE . invocation .
invocation_as_function = (invocation = "FUNCTION")

IF \invocation_as_function THEN     /* called as COMMAND or SUBROUTINE and not as a FUNCTION! */
DO                                  /* only one argument-string, which has to be parsed ! */
   argument1 = ARG(1)
   argument1 = STRIP(argument1)     /* strip blanks */
   argument1 = STRIP(argument1,,"'")/* strip ' */
   argument1 = STRIP(argument1,,'"')/* strip " */
   argcount = 1

END
ELSE     /* invoked as a function */
DO
    argument1 = ARG(1)
    argcount  = ARG()      /* number of arguments */
END


NUMERIC DIGITS 14          /* set precision to 14 digits after comma */

PARSE UPPER VAR argument1 date1 time1

/* check sorted dates & numbers */

date1 = parse_date(date1)
time1 = parse_time(time1)

days1 = date2days(date1)
fraction1 = time2fract(time1)
result_file2dat = days1||SUBSTR(fraction1,2)


IF invocation_as_function THEN   /* invoked as function, therefore return the value */
   RETURN result_file2dat         /* return value */

/* invoked from the COMMAND-line or as a SUBROUTINE, both invocations must not return a value */
SAY "file2dat - result:" result_file2dat       /* show result on standard output */
RETURN
/* end of main routine */



parse_date: PROCEDURE
    PARSE ARG month '-' day '-' year

    RETURN year month day
/* end of parse_DATE */


/* parse & check time, return 24hour-Time (military time) */
parse_TIME: PROCEDURE
    PARSE ARG hours ':' minutes ':' seconds

    RETURN hours  minutes seconds
/* end of parse_TIME */



/* calculate days based on 0000/01/01 (= 1. day == 1) */
DATE2DAYS: PROCEDURE
    PARSE ARG year month day

    IF year < 80 THEN year = year + 100 /* account for > 2000 */
    year = year - 80                    /* Base days on 1980 */
    days_1    = year * 365
    days_1    = days_1 + 722813         /* Base calculations on 1-1-1980 */
    leap_days = year % 4

    IF year > 0 THEN
    DO
       leap_days = leap_days + 1        /* account for leap year in 0000 */
       leap_year = ((year // 4) = 0)    /* leap year in hand ? */
       leap_days = leap_days - leap_year
    END

    days_2 = date2julian(ARG(1))

    RETURN (days_1 + leap_days + days_2)
/* end of DATE2DAYS */



/* calculate decimal fraction from time */
TIME2FRACT: PROCEDURE                   /* calculate decimal value for time */
    PARSE ARG hours minutes seconds

    /* hour_div = 24      =   24           */
    /* min_div  = 1440    =   24 * 60      */
    /* sec_div  = 86400   =   24 * 60 * 60 */

    RETURN ((hours/24) + (minutes/1440) + (seconds/86400))
/* end of TIME2FRACT */


/* calculate time from fraction */
FRACT2TIME: PROCEDURE                   /* calculate time from given value */
    /* hours    = 24      =   24           */
    /* minutes  = 1440    =   24 * 60      */
    /* seconds  = 86400   =   24 * 60 * 60 */

    tmp = arg(1) + 0.0000001            /* account for possible precision error */

    hours   = (tmp * 24) % 1
    minutes = (tmp * 1440 - hours * 60) % 1
    seconds = (tmp * 86400 - hours * 3600 - minutes * 60) % 1

    RETURN RIGHT(hours,2,'0')':'RIGHT(minutes,2,'0')':'RIGHT(seconds,2,'0')

/* end of FRACT2TIME */


/* build Julian date from sorted date, result: yyyyddd */
DATE2JULIAN: PROCEDURE
    PARSE ARG year month day

     /* build monthdays array */
     monthdays.1 = 31
     monthdays.2 = 28
     monthdays.3 = 31
     monthdays.4 = 30
     monthdays.5 = 31
     monthdays.6 = 30
     monthdays.7 = 31
     monthdays.8 = 31
     monthdays.9 = 30
     monthdays.10 = 31
     monthdays.11 = 30
     monthdays.12 = 31

    /* is year a leap year ? */
    leap_year = (year // 4) = 0         /* Julian calender    */
    monthdays.2 = 28 + leap_year

    result_function = 0
    DO i = 1 TO month - 1
       result_function = result_function + monthdays.i
    END

    result_function = result_function + day

    RETURN result_function
/* end of DATE2JULIAN */








USAGE:
SAY
SAY 'usage as a function in REXX-programs:'
SAY '         FILE2DAT(argument)'
SAY
SAY 'usage from command-line:'
SAY '         FILE2DAT argument '
SAY
EXIT


ERROR:
/* 1994-03-03: produce an error-message on stderr even if called as a function
   /* invoked as a COMMAND (from command-line) or as a FUNCTION ? */
   PARSE SOURCE . invocation .
   invocation_as_function = (invocation = "FUNCTION")

   /* error message on device "STDERR", only if not called as function */
   IF \invocation_as_function THEN
*/

      '@ECHO file2dat:' errmsg '>&2'

   EXIT ''                                      /* error, therefore return empty string */
