/*
 * screen.c - screen management for Quick Pager
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <memory.h>

#define INCL_BASE
#include <os2.h>

#include "screen.h"

/*
 * Screen Control Data
 *
 *  |<------------------- nCols ------------------->|
 *  +-----------------------------------------------+---
 *  |                                               |  A
 *  |                                               |  |
 *  |                                               |  |
 *  |                                               | nLines
 *  |                                               |  |
 *  |                                               |  |
 *  |                                               |  V
 *  +-----------------------------------------------+---
 *  |                                               | Command/Status Line
 *  +-----------------------------------------------+
 */

int     nLines ;    /* Number of Lines      */
int     nCols  ;    /* Number of Columns    */

/*
 * Keyboard Control Data
 */

static  KBDINFO keySave ;       /* Saved   KDB Status   */
static  KBDINFO keyWork ;       /* Working KDB Status   */

/*
 * Variables for Controlling Logical Video Buffer
 */

static  ULONG   lvbBase ;       /* Base address of the LVB  */
static  SHORT   lvbLeng ;       /* Length of the LVB        */
static  SHORT   lvbTop  ;       /* Top of modified region   */
static  SHORT   lvbBot  ;       /* Bot of modified region   */

static  UCHAR   scrAttr = 0xf1 ;     /* Disp. Attributes    */
static	UCHAR	stsAttr = 0x1f ;

static  UCHAR   hexval(UCHAR ch)
{
    if ((ch >= '0') && (ch <= '9')) {
        return(ch - '0') ;
    } else if ((ch >= 'a') && (ch <= 'f')) {
        return(ch - 'a' + 10) ;
    } else if ((ch >= 'A') && (ch <= 'F')) {
        return(ch - 'A' + 10) ; 
    }
    return(0) ;
}

/*
 * scrInit - Initialize Screen Management
 */

int     scrInit(char *ename)
{
    VIOMODEINFO scrMode ;   /* Working Screen Mode  */
    char	*env ;
    int         i    ;

    /*
     * Get Screen Attributes to use
     */
     
    if ((env = getenv(ename)) != NULL) {
        for (i = 0 ; env[i] != '\0' ; i++) {
	    if (i < 2) {
	        scrAttr <<= 4 ;
		scrAttr += hexval(env[i]) ;
	    } else {
	        stsAttr <<= 4 ;
		stsAttr += hexval(env[i]) ;
	    }
	}
	scrAttr &= 0xff ;
	stsAttr &= 0xff ;
    }
    
    /* 
     * Get current screen size
     */
    scrMode.cb = sizeof(VIOMODEINFO) ;
    VioGetMode(&scrMode, 0) ;
    
    nLines = scrMode.row - 1 ;      /* Last Line for Command/Status */
    nCols  = scrMode.col     ;
    
    VioSetCurPos(nLines, 0, 0) ;
    
    /*
     * Get Logical Video Buffer
     */
    VioGetBuf(&lvbBase, &lvbLeng, 0) ;
    lvbBase = (ULONG) _emx_16to32((_far16ptr) lvbBase) ;
    lvbTop = lvbLeng ;
    lvbBot = 0       ;

    /*
     * Set Keyboard Mode
     */

    keySave.cb = keyWork.cb = sizeof(KBDINFO) ; 
    KbdGetStatus(&keySave, 0) ;
    KbdGetStatus(&keyWork, 0) ;

    keyWork.fsMask = 0x06 ;     /* no echo, raw */
    KbdSetStatus(&keyWork, 0) ;
	 
    return 0 ;
}
    
/*
 * scrDone - Terminate Screen Management
 */
 
void    scrDone(void)
{
    KbdSetStatus(&keySave, 0) ;
}

/*
 * scrShow - Refresh Screen (Display LVB Contents)
 */

void    scrShow(void)
{
    if (lvbBot > lvbTop) {
        VioShowBuf(lvbTop, (lvbBot - lvbTop), 0) ;
	lvbTop = lvbLeng ;
	lvbBot = 0       ;
    }
}

/*
 * scrClear - Clear Screen
 */

void    scrClear(void)
{
    PUCHAR  p ;
    INT     i ;

    p = (PUCHAR) lvbBase ;
    for (i = 0 ; i < lvbLeng ; i += 2) {
        *p++ = 0x20    ;
	*p++ = scrAttr ;
    }
    lvbTop = 0       ;
    lvbBot = lvbLeng ;
}

/*
 * scrPutLine - Put a Line to Screen
 */

void    scrPutLine(int lin, PUCHAR str)
{
    PUCHAR  p ;
    int     i, top, bot ;
    UCHAR   attr ;
    
    p = (PUCHAR) (lvbBase + lin * nCols * 2)  ;
    attr = (lin == nLines) ? stsAttr : scrAttr ;

    for (i = 0 ; *str && i < nCols ; i++) {
        *p++ = *str++ ;
	*p++ = attr   ;
    }
    for ( ; i < nCols ; i++) {
        *p++ = 0x20 ;
	*p++ = attr ;
    }
    
    top = lin * nCols * 2 ;
    bot = top + nCols * 2 ;
    
    if (top < lvbTop) lvbTop = top ;
    if (bot > lvbBot) lvbBot = bot ;
}
	
/*
 * scrScrollUp - Scroll Up Screen (except Command/Status line)
 */

void    scrScrollUp(int n)
{
    int     slin, dlin ;
    int     top, bot ;
    PUCHAR  sp, dp ;
    
    for (slin = n, dlin = 0 ; slin < nLines ; slin++, dlin++) {
        sp = (PUCHAR) (lvbBase + nCols * slin * 2) ;
        dp = (PUCHAR) (lvbBase + nCols * dlin * 2) ;
	memcpy(dp, sp, (nCols * 2)) ;
    }

    top = 0 ;
    bot = nCols * (nLines - n) * 2 ;
    
    if (top < lvbTop) lvbTop = top ;
    if (bot > lvbBot) lvbBot = bot ;
}

/*
 * scrScrollDn - Scroll Down Screen (except Command/Status line)
 */

void    scrScrollDn(int n)
{
    int     slin, dlin ;
    int     top, bot ;
    PUCHAR  sp, dp ;
    
    for (dlin = nLines - 1, slin = dlin - n ; slin >= 0 ; slin--, dlin--) {
        sp = (PUCHAR) (lvbBase + nCols * slin * 2) ;
        dp = (PUCHAR) (lvbBase + nCols * dlin * 2) ;
	memcpy(dp, sp, (nCols * 2)) ;
    }

    top = nCols * n * 2 ;
    bot = nCols * nLines * 2 ;
    
    if (top < lvbTop) lvbTop = top ;
    if (bot > lvbBot) lvbBot = bot ;
}

/*
 * scrEnter - prompt and get line
 */

static  UCHAR  	gBuff[256] ;

#define isdelim(x)      ( ispunct((x)) || isspace((x)) )

PUCHAR  scrEnter(PUCHAR prompt, PUCHAR s, int len)
{
    KBDKEYINFO  key ;
    UCHAR       ch, sc, last ;
    int         base, i, j ;
    BOOL        stat = TRUE ;

    /*
     * Prompt for Input
     */
    sprintf(gBuff, "%s%-128s", prompt, s) ;
    scrPutLine(nLines, gBuff) ; scrShow() ;

    base = strlen(prompt) ;
    i    = strlen(s)      ;
    VioSetCurPos(nLines, (base + i), 0) ;

    last = 0 ;

    while (i < len) {
        KbdCharIn(&key, 0, 0) ;
	ch = key.chChar ;
	sc = key.chScan ;
	
	if (last != 0) {
	    s[i++] = last ; s[i++] = ch ; s[i] = '\0' ; last = 0;
        } else if (ch >= 0x20 && ch < 0x7f) {
	    s[i++] = ch ; s[i] = '\0' ;
	} else if (ch > 0xa0 && ch <= 0xdf) {
	    s[i++] = ch ; s[i] = '\0' ;
        } else if (ch > 0x80 && ch <= 0xa0) {
	    last = ch ;
	} else if (ch >= 0xe0 && ch <= 0xff && sc == 0) {
	    last = ch ;
	} else if (ch == 0x03 || ch == 0x07 || ch == 0x1b) {
	    s[i = 0] = '\0' ; stat = FALSE ;
	    break ;
	} else if (ch == 0x0a || ch == 0x0d || ch == 0x09) {
	    s[i] = '\0' ; stat = TRUE ;
	    break ;
	} else if (ch == 0x08 || ch == 0x7f || sc == 0x4b) {
	    if (i > 0) {
	        i -= 1 ; s[i] = '\0' ;
	    }
	} else if (ch == 0x15 || ch == 0x18 || sc == 0x47) {
	    while (i > 0) {
	        i -= 1 ; s[i] = '\0' ;
	    }
        } else if (ch == 0x17 || sc == 0x73) {
	    if (i > 0 && ispunct(s[i-1])) {
	        while (i > 0 && ispunct(s[i-1])) {
		    i -= 1 ; s[i] = '\0' ;
		}
	    } else if (i > 0 && isspace(s[i-1])) {
	        while (i > 0 && isspace(s[i-1])) {
		    i -= 1 ; s[i] = '\0' ;
		}
            }
	    while (i > 0 && (!isdelim(s[i-1]))) {
	        i -= 1 ; s[i] = '\0' ;
	    }
	}
	
	/*
	 * For 2 bytes codes
	 */
	for (j = 0 ; j < i ; ) {
            if (s[j] >= 0x80 && s[j] < 0xa0) {
	        if (s[j+1] == '\0') {
	            i -= 1 ; s[i] = '\0' ;
		    break ;
		}
		j += 2 ;
	    } else if (s[j] >= 0xe0 && s[j] < 0xff) {
	        if (s[j+1] == '\0') {
	            i -= 1 ; s[i] = '\0' ;
		    break ;
		}
		j += 2 ;
	    } else {
	        j += 1 ;
	    }
	}
        sprintf(gBuff, "%s%-128s", prompt, s) ;
        scrPutLine(nLines, gBuff) ; scrShow() ;
        VioSetCurPos(nLines, (base + strlen(s)), 0) ;
    }

    VioSetCurPos(nLines, 0, 0) ;
    
    if (stat) {
        return s ;
    } else {
        return NULL ;
    }
}
