#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_DOSFILEMGR
#define INCL_DOSERRORS
#define INCL_DOSMISC
#include <os2.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "Defines.h"
#include "Structures.h"
#include "SupportFunctions.h"
#include "DiskFunctions.h"

USHORT setHeadCyl(ULONG secNum, PTRACKLAYOUT tl, USHORT trackPerCyl, USHORT secPerTrack)
{
    ULONG secPerCyl=trackPerCyl*secPerTrack;
    ULONG wSecNum=secNum+1;

    tl->usCylinder=wSecNum/secPerCyl;
    tl->usHead=(wSecNum%secPerCyl)/secPerTrack;

    return (USHORT)((wSecNum%secPerCyl)%secPerTrack);
}

PFUNCRETURN GetRealFreeSpace(PUCHAR driveString, PFREESPACEINFO fsi)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="GetRealFreeSpace";
    PFUNCRETURN pfr;
    FSALLOCATE fsa;
    ULONG driveNum, freeSectCount, hiddenSectors, bmpListLoc, bmpLoc,
          i, j, k, bmpListSectCount, bmpData, fsArray[32];
    APIRET rc;
    PUCHAR bmpList, bmpSect, tSect;
    PSUPERBLOCK sub;
    PSPAREBLOCK spb;
    HFILE driveHandle;

    memset(&fr, 0, sizeof(fr));

    pfr=OpenDrive(driveString, &driveHandle, 0L);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=GetDriveSpecs(driveHandle, &fsi->sectorSize, &hiddenSectors);
    if (!pfr->success)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    spb=(PSPAREBLOCK)malloc(fsi->sectorSize);
    if (spb==NULL)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSpareBlock(driveHandle, spb);
    if (!pfr->success)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    if (spb->partStatus & 0x10)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "badbitmapsexist");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    driveNum=(ULONG)(toupper(driveString[0])-64);

    memset(&fsa, 0, sizeof(fsa));

    rc=DosQueryFSInfo(driveNum, FSIL_ALLOC, &fsa, sizeof(fsa));
    if (rc!=NO_ERROR)
        {
        free(spb);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "DosQueryFSInfo");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=rc;
        return &fr;
        }

    fsi->totalSectors=fsa.cUnit;
    fsi->reportedFreeSectors=fsa.cUnitAvail;

    sub=(PSUPERBLOCK)malloc(fsi->sectorSize);
    if (sub==NULL)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSuperBlock(driveHandle, sub);
    if (!pfr->success)
        {
        free(spb);
        free(sub);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    bmpListLoc=sub->bitmapList;

    /*
        Each bitmap covers 16384 sectors, and each bitmap list sector
        covers 128 bitmaps.  This calculates how many list sectors are
        required for the number of sectors on the drive.  While this value
        has a granularity of one, since it's not necessary to read more
        sectors than this determins, the actual space reserved is a
        multiple of four sectors.
    */

    bmpListSectCount=fsi->totalSectors/16384+(fsi->totalSectors%16384>0?1:0);
    bmpListSectCount=bmpListSectCount*4;
    bmpListSectCount=bmpListSectCount/fsi->sectorSize+(bmpListSectCount%fsi->sectorSize>0?1:0);

    bmpList=(PUCHAR)malloc(bmpListSectCount*fsi->sectorSize);
    if (bmpList==NULL)
        {
        free(spb);
        free(sub);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    bmpSect=(PUCHAR)malloc(fsi->sectorSize*4);
    if (bmpList==NULL)
        {
        free(spb);
        free(sub);
        free(bmpList);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    tSect=(PUCHAR)malloc(fsi->sectorSize);
    if (tSect==NULL)
        {
        free(bmpSect);
        free(spb);
        free(sub);
        free(bmpList);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }


    for (i=0; i<bmpListSectCount; i++)
        {
        pfr=ReadSector(driveHandle, hiddenSectors+bmpListLoc+i, tSect);
        if (!pfr->success)
            {
            free(spb);
            free(bmpList);
            free(bmpSect);
            free(sub);
            DosClose(driveHandle);
            fr.success=FALSE;
            strcpy(fr.errorFunc, pfr->errorFunc);
            strcpy(fr.errorHome, pfr->errorHome);
            fr.errorCode=pfr->errorCode;
            return &fr;
            }
        memcpy(bmpList+(fsi->sectorSize*i), tSect, fsi->sectorSize);
        }

    freeSectCount=0;

    for (i=0; i<32; i++)
        {
        fsArray[i]=0x00000001 << i;
        }

    for (i=0; i<fsi->sectorSize*bmpListSectCount; i+=4)
        {
        memcpy(&bmpLoc, &bmpList[i], 4);
        if (bmpLoc==0x00000000)
            break;

        for (j=0; j<4; j++)
            {
            pfr=ReadSector(driveHandle, hiddenSectors+bmpLoc+j, tSect);
            if (!pfr->success)
                {
                free(spb);
                free(bmpList);
                free(bmpSect);
                free(sub);
                free(tSect);
                DosClose(driveHandle);
                fr.success=FALSE;
                strcpy(fr.errorFunc, pfr->errorFunc);
                strcpy(fr.errorHome, pfr->errorHome);
                fr.errorCode=pfr->errorCode;
                return &fr;
                }
            memcpy(bmpSect+(fsi->sectorSize*j), tSect, fsi->sectorSize);
            }

        for (j=0; j<fsi->sectorSize*4; j+=4)
            {
            memcpy(&bmpData, &bmpSect[j], 4);

            if (bmpData!=0x00000000)
                {
                if (bmpData==0xFFFFFFFF)
                    {
                    freeSectCount+=32;
                    }
                else
                    {
                    for (k=0; k<32; k++)
                        {
                        if ((bmpData & fsArray[k])>0)
                            freeSectCount++;
                        }
                    }
                }
            }
        }

    fsi->actualFreeSectors=freeSectCount;

    DosClose(driveHandle);

    free(bmpList);
    free(bmpSect);
    free(sub);
    free(spb);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN GetSystemUsage(PUCHAR driveString, PSYSTEMUSAGE suInfo)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="GetSystemUsage";
    PFUNCRETURN pfr;
    PSPAREBLOCK spb;
    PSUPERBLOCK sub;
    ULONG sectorSize;
    HFILE driveHandle;

    memset(&fr, 0, sizeof(fr));

    pfr=OpenDrive(driveString, &driveHandle, 0L);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=GetDriveSpecs(driveHandle, &sectorSize, NULL);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    spb=(PSPAREBLOCK)malloc(sectorSize);
    if (spb==NULL)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    sub=(PSUPERBLOCK)malloc(sectorSize);
    if (spb==NULL)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSpareBlock(driveHandle, spb);
    if (!pfr->success)
        {
        free(spb);
        free(sub);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=ReadSuperBlock(driveHandle, sub);
    if (!pfr->success)
        {
        free(spb);
        free(sub);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    DosClose(driveHandle);

    suInfo->sectorSize=sectorSize;
    suInfo->totalSectors=sub->sectorCount;
    suInfo->dirBandSectors=sub->dirBandSectorCount;
    suInfo->dirBandBitmapSectors=4;
    suInfo->frontStructures=20;
    suInfo->codePageSectors=2;
    suInfo->spareDirBlockSectors=spb->spareDirBlockCount*4;
    suInfo->hotfixSectors=spb->hotfixTotal;
    suInfo->hotfixListSectors=4;
    suInfo->bitmapListSectors=sub->sectorCount/16384+(sub->sectorCount%16384>0?1:0);
    suInfo->bitmapListSectors=suInfo->bitmapListSectors*4;
    suInfo->bitmapListSectors=suInfo->bitmapListSectors/sectorSize+
                              (suInfo->bitmapListSectors%sectorSize>0?1:0);
    suInfo->bitmapListSectors=(suInfo->bitmapListSectors/4*4)+
                              (suInfo->bitmapListSectors%4>0?4:0);
    suInfo->bitmapSectors=(sub->sectorCount/16384*4)+(sub->sectorCount%16384>0?4:0);
    suInfo->badSectorListSectors=((((sub->badSectorCount>>14)+2)*4)/sectorSize)+1;
    suInfo->badSectorListSectors=(suInfo->badSectorListSectors/4)+
                                 (suInfo->badSectorListSectors%4>0?4:0);
    suInfo->securityBlockSectors=8;
    suInfo->rootDirBlock=4;
    suInfo->rootFNode=1;

    suInfo->totalSystemSectors=
        suInfo->dirBandSectors+
        suInfo->dirBandBitmapSectors+
        suInfo->frontStructures+
        suInfo->codePageSectors+
        suInfo->spareDirBlockSectors+
        suInfo->hotfixSectors+
        suInfo->hotfixListSectors+
        suInfo->bitmapListSectors+
        suInfo->bitmapSectors+
        suInfo->badSectorListSectors+
        suInfo->rootDirBlock+
        suInfo->rootFNode+
        suInfo->securityBlockSectors;

    free(sub);
    free(spb);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN GetDirtyStatus(PUCHAR driveString, PBOOL dirtyFlag)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="GetDirtyStatus";
    ULONG sectorSize;
    PFUNCRETURN pfr;
    PSPAREBLOCK spb;
    HFILE driveHandle;
    BYTEBITFIELD partStatus;

    memset(&fr, 0, sizeof(fr));

    pfr=OpenDrive(driveString, &driveHandle, 0L);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=GetDriveSpecs(driveHandle, &sectorSize, NULL);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    spb=(PSPAREBLOCK)malloc(sectorSize);
    if (spb==NULL)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSpareBlock(driveHandle, spb);
    if (!pfr->success)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    DosClose(driveHandle);

    memcpy(&partStatus, &spb->partStatus, 1);

    *dirtyFlag=(BOOL)partStatus.bit0;

    free(spb);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN SetDirtyStatus(PUCHAR driveString, BOOL newDirtyFlag)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="SetDirtyStatus";
    ULONG sectorSize;
    PFUNCRETURN pfr;
    PSPAREBLOCK spb;
    HFILE driveHandle;
    BYTEBITFIELD partStatus;

    memset(&fr, 0, sizeof(fr));

    pfr=OpenDrive(driveString, &driveHandle, OPEN_ACCESS_READWRITE);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=LockDrive(driveHandle);
    if (!pfr->success)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "nolockdrive");
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=GetDriveSpecs(driveHandle, &sectorSize, NULL);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    spb=(PSPAREBLOCK)malloc(sectorSize);
    if (spb==NULL)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSpareBlock(driveHandle, spb);
    if (!pfr->success)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    memcpy(&partStatus, &spb->partStatus, 1);

    partStatus.bit0=newDirtyFlag;

    memcpy(&spb->partStatus, &partStatus, 1);

    pfr=WriteSpareBlock(driveHandle, spb);
    if (!pfr->success)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    DosClose(driveHandle);

    free(spb);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN ToggleDirtyStatus(PUCHAR driveString, PBOOL newStatus)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="ToggleDirtyStatus";
    ULONG sectorSize;
    PFUNCRETURN pfr;
    PSPAREBLOCK spb;
    HFILE driveHandle;
    BYTEBITFIELD partStatus;

    memset(&fr, 0, sizeof(fr));

    pfr=OpenDrive(driveString, &driveHandle, OPEN_ACCESS_READWRITE);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=LockDrive(driveHandle);
    if (!pfr->success)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "nolockdrive");
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=GetDriveSpecs(driveHandle, &sectorSize, NULL);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    spb=(PSPAREBLOCK)malloc(sectorSize);
    if (spb==NULL)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSpareBlock(driveHandle, spb);
    if (!pfr->success)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    memcpy(&partStatus, &spb->partStatus, 1);

    if (partStatus.bit0)
        {
        *newStatus=FALSE;
        partStatus.bit0=0;
        }
    else
        {
        *newStatus=TRUE;
        partStatus.bit0=1;
        }

    memcpy(&spb->partStatus, &partStatus, 1);

    pfr=WriteSpareBlock(driveHandle, spb);
    if (!pfr->success)
        {
        free(spb);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    DosClose(driveHandle);

    free(spb);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN GetBadSectorCount(PUCHAR driveString, PULONG bsCount)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="GetBadSectorCount";
    PFUNCRETURN pfr;
    PSUPERBLOCK sub;
    ULONG sectorSize;
    HFILE driveHandle;
    BYTEBITFIELD partStatus;

    memset(&fr, 0, sizeof(fr));

    pfr=OpenDrive(driveString, &driveHandle, 0L);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=GetDriveSpecs(driveHandle, &sectorSize, NULL);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    sub=(PSUPERBLOCK)malloc(sectorSize);
    if (sub==NULL)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSuperBlock(driveHandle, sub);
    if (!pfr->success)
        {
        free(sub);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    *bsCount=sub->badSectorCount;

    free(sub);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN DumpSectorData(PUCHAR driveString, ULONG begin, ULONG end, FILE *outFile, PULONG secSize)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="DumpSectorData";
    PFUNCRETURN pfr;
    ULONG sectorSize, hiddenSectors, secCount, i;
    PUCHAR sdata;
    PSUPERBLOCK sub;
    HFILE driveHandle;

    memset(&fr, 0, sizeof(fr));

    pfr=OpenDrive(driveString, &driveHandle, 0L);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=GetDriveSpecs(driveHandle, &sectorSize, &hiddenSectors);
    if (!pfr->success)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    if (secSize!=NULL)
        *secSize=sectorSize;

    sub=(PSUPERBLOCK)malloc(sectorSize);
    if (sub==NULL)
        {
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSuperBlock(driveHandle, sub);
    if (!pfr->success)
        {
        free(sub);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    if (end>sub->sectorCount)
        {
        free(sub);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "endpastend");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    secCount=end-begin+1;

    sdata=(PUCHAR)malloc(sectorSize);
    if (sdata==NULL)
        {
        free(sub);
        DosClose(driveHandle);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    printf("\nBytes dumped:                ");
    fflush(stdout);

    for (i=0; i<secCount; i++)
        {
        pfr=ReadSector(driveHandle, hiddenSectors+begin+i, sdata);
        if (!pfr->success)
            {
            free(sub);
            free(sdata);
            printf("\n");
            DosClose(driveHandle);
            fr.success=FALSE;
            strcpy(fr.errorFunc, pfr->errorFunc);
            strcpy(fr.errorHome, pfr->errorHome);
            fr.errorCode=pfr->errorCode;
            return &fr;
            }
        if (fwrite(sdata, sectorSize, 1, outFile)<1)
            {
            free(sub);
            free(sdata);
            printf("\n");
            DosClose(driveHandle);
            fr.success=FALSE;
            strcpy(fr.errorFunc, "fwrite");
            strcpy(fr.errorHome, currFunc);
            fr.errorCode=0;
            return &fr;
            }
        printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b%15s", fCodel((double)(i+1)*(double)sectorSize));
        fflush(stdout);
        }
    printf("\n");

    DosClose(driveHandle);

    free(sdata);
    free(sub);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN OpenDrive(PUCHAR driveString, PHFILE driveHandle, ULONG openFlags)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="OpenDrive";
    ULONG openAction, openMode;
    APIRET rc;

    openMode=0;
    openMode=openMode | OPEN_FLAGS_DASD | OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE
                      | OPEN_FLAGS_NO_CACHE;
    openMode=openMode | openFlags;
    rc=DosOpen(driveString, driveHandle, &openAction, 0L, 0L, FILE_OPEN, openMode, 0L);
    if (rc!=NO_ERROR)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "DosOpen");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=rc;
        return &fr;
        }

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN LockDrive(HFILE driveHandle)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="LockDrive";
    APIRET rc;

    memset(&fr, 0, sizeof(fr));

    rc=DosDevIOCtl(driveHandle, IOCTL_DISK, DSK_LOCKDRIVE, 0, 0, 0, 0, 0, 0);
    if (rc!=NO_ERROR)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "DosDevIOCtl");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=rc;
        return &fr;
        }

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN DriveIsHPFS(PUCHAR driveString, PBOOL isHPFS, PUCHAR fsInUse)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="DriveIsHPFS";
    PFSQBUFFER2 fsBuffer;
    ULONG bufLen;
    APIRET rc;
    PUCHAR fsStuff;

    memset(&fr, 0, sizeof(fr));

    fsBuffer=(PFSQBUFFER2)malloc(2048);
    if (fsBuffer==NULL)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }
    memset(fsBuffer, 0, 2048);
    bufLen=2048;
    DosError(FERR_DISABLEHARDERR);
    rc=DosQueryFSAttach(driveString, 0, FSAIL_QUERYNAME, fsBuffer, &bufLen);
    DosError(FERR_ENABLEHARDERR);
    if (rc!=NO_ERROR)
        {
        free(fsBuffer);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "DosQueryFSAttach");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=rc;
        return &fr;
        }
    fsStuff=(PUCHAR)(fsBuffer->szName+fsBuffer->cbName+1);

    strcpy(fsInUse, fsStuff);

    if (!strcmp(fsInUse, "HPFS"))
        *isHPFS=TRUE;
    else
        *isHPFS=FALSE;

    free(fsBuffer);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN GetBPB(HFILE driveHandle, PBIOSPARAMETERBLOCK bpb)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="GetBPB";
    PUCHAR bpParm, bpData;
    ULONG bpParmLen, bpDataLen;
    APIRET rc;

    memset(&fr, 0, sizeof(fr));

    bpParm=(PUCHAR)malloc(2);
    if (bpParm==NULL)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    bpData=(PUCHAR)malloc(40);
    if (bpData==NULL)
        {
        free(bpParm);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    memset(bpParm, 0, 2);
    memset(bpData, 0, 40);
    bpParmLen=2;
    bpDataLen=40;
    rc=DosDevIOCtl(driveHandle, IOCTL_DISK, DSK_GETDEVICEPARAMS,
                   bpParm, bpParmLen, &bpParmLen,
                   bpData, bpDataLen, &bpDataLen);

    if (rc!=NO_ERROR)
        {
        free(bpParm);
        free(bpData);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "DosDevIOCtl");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=rc;
        return &fr;
        }

    memset(bpb, 0, sizeof(BIOSPARAMETERBLOCK));
    memcpy(bpb, bpData, sizeof(BIOSPARAMETERBLOCK));

    free(bpParm);
    free(bpData);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN GetDriveSpecs(HFILE driveHandle, PULONG sectorSize, PULONG hiddenSectors)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="GetDriveSpecs";
    PFUNCRETURN pfr;
    ULONG i;
    PBIOSPARAMETERBLOCK bpb;

    memset(&fr, 0, sizeof(fr));

    bpb=(PBIOSPARAMETERBLOCK)malloc(sizeof(BIOSPARAMETERBLOCK));
    if (bpb==NULL)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=GetBPB(driveHandle, bpb);
    if (!pfr->success)
        {
        free(bpb);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=0;
        return &fr;
        }

    if (sectorSize!=NULL)
        *sectorSize=bpb->usBytesPerSector;

    if (hiddenSectors!=NULL)
        *hiddenSectors=bpb->cHiddenSectors;

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN ReadSuperBlock(HFILE driveHandle, PSUPERBLOCK sub)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="ReadSuperBlock";
    PFUNCRETURN pfr;
    PBIOSPARAMETERBLOCK bpb;

    bpb=(PBIOSPARAMETERBLOCK)malloc(sizeof(BIOSPARAMETERBLOCK));
    if (bpb==NULL)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=GetBPB(driveHandle, bpb);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSector(driveHandle, bpb->cHiddenSectors+SUPERLOC, (PUCHAR)sub);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorFunc);
        fr.errorCode=0;
        return &fr;
        }

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN ReadSpareBlock(HFILE driveHandle, PSPAREBLOCK spb)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="ReadSpareBlock";
    PFUNCRETURN pfr;
    PBIOSPARAMETERBLOCK bpb;

    bpb=(PBIOSPARAMETERBLOCK)malloc(sizeof(BIOSPARAMETERBLOCK));
    if (bpb==NULL)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=GetBPB(driveHandle, bpb);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=ReadSector(driveHandle, bpb->cHiddenSectors+SPARELOC, (PUCHAR)spb);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorFunc);
        fr.errorCode=0;
        return &fr;
        }

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN WriteSpareBlock(HFILE driveHandle, PSPAREBLOCK spb)
{
    static FUNCRETURN fr;
    PFUNCRETURN pfr;
    ULONG hiddenSectors;

    pfr=GetDriveSpecs(driveHandle, NULL, &hiddenSectors);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    pfr=WriteSector(driveHandle, hiddenSectors+SPARELOC, (PUCHAR)spb);
    if (!pfr->success)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN ReadSector(HFILE driveHandle, ULONG lsNum, PUCHAR secData)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="ReadSectorData";
    PFUNCRETURN pfr;
    PTRACKLAYOUT tl;
    PBIOSPARAMETERBLOCK bpb;
    ULONG bpParmLen, bpDataLen, i;
    APIRET rc;

    memset(&fr, 0, sizeof(fr));

    bpb=(PBIOSPARAMETERBLOCK)malloc(sizeof(BIOSPARAMETERBLOCK));
    if (bpb==NULL)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=GetBPB(driveHandle, bpb);
    if (!pfr->success)
        {
        free(bpb);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    bpParmLen=sizeof(TRACKLAYOUT)+sizeof(ULONG);
    bpDataLen=bpb->usBytesPerSector;

    tl=(PTRACKLAYOUT)malloc(bpParmLen);
    if (tl==NULL)
        {
        free(bpb);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    tl->bCommand=0;
    tl->usHead=0;
    tl->usCylinder=0;
    tl->usFirstSector=0;
    tl->cSectors=1;

    tl->TrackTable[0].usSectorNumber=setHeadCyl(lsNum, tl, bpb->cHeads,
                                                bpb->usSectorsPerTrack);
    tl->TrackTable[0].usSectorSize=bpb->usBytesPerSector;

    rc=DosDevIOCtl(driveHandle, IOCTL_DISK, DSK_READTRACK, tl,
                   bpParmLen, &bpParmLen, secData, bpDataLen, &bpDataLen);

    if (rc!=NO_ERROR)
        {
        free(bpb);
        free(tl);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "DosDevIOCtl");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=rc;
        return &fr;
        }

    free(tl);
    free(bpb);

    fr.success=TRUE;
    return &fr;
}

PFUNCRETURN WriteSector(HFILE driveHandle, ULONG lsNum, PUCHAR secData)
{
    static FUNCRETURN fr;
    UCHAR currFunc[]="WriteSectorData";
    PFUNCRETURN pfr;
    PTRACKLAYOUT tl;
    PBIOSPARAMETERBLOCK bpb;
    ULONG bpParmLen, bpDataLen, i;
    APIRET rc;

    memset(&fr, 0, sizeof(fr));

    bpb=(PBIOSPARAMETERBLOCK)malloc(sizeof(BIOSPARAMETERBLOCK));
    if (bpb==NULL)
        {
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    pfr=GetBPB(driveHandle, bpb);
    if (!pfr->success)
        {
        free(bpb);
        fr.success=FALSE;
        strcpy(fr.errorFunc, pfr->errorFunc);
        strcpy(fr.errorHome, pfr->errorHome);
        fr.errorCode=pfr->errorCode;
        return &fr;
        }

    bpParmLen=sizeof(TRACKLAYOUT)+sizeof(ULONG);
    bpDataLen=bpb->usBytesPerSector;

    tl=(PTRACKLAYOUT)malloc(bpParmLen);
    if (tl==NULL)
        {
        free(bpb);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "malloc");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=0;
        return &fr;
        }

    tl->bCommand=0;
    tl->usHead=0;
    tl->usCylinder=0;
    tl->usFirstSector=0;
    tl->cSectors=1;

    tl->TrackTable[0].usSectorNumber=setHeadCyl(lsNum, tl, bpb->cHeads,
                                                bpb->usSectorsPerTrack);
    tl->TrackTable[0].usSectorSize=bpb->usBytesPerSector;

    rc=DosDevIOCtl(driveHandle, IOCTL_DISK, DSK_WRITETRACK, tl,
                   bpParmLen, &bpParmLen, secData, bpDataLen, &bpDataLen);

    if (rc!=NO_ERROR)
        {
        free(bpb);
        free(tl);
        fr.success=FALSE;
        strcpy(fr.errorFunc, "DosDevIOCtl");
        strcpy(fr.errorHome, currFunc);
        fr.errorCode=rc;
        return &fr;
        }

    free(tl);
    free(bpb);

    fr.success=TRUE;
    return &fr;
}
