/*
 * SortHost
 *
 * This REXX program sorts the list and digest subscription files
 * for an InetMail Steward list server. The list is sorted by
 * domain name so that the mailer can be more efficient when
 * sending messages.
 *
 * Run the program without any parameters.
 *
 * This is based on a program originally written by ivanfm@ibm.net
 * that used an external program to perform the actual sort. In
 * this version, the sort is performed in REXX in order to eliminate
 * the overhead of loading another program.
 *
 * This program also sorts all the lists known to Steward. It does
 * this by looking at the steward.cf file, finding the Lists directory
 * and then checking each subdirectory in the Lists directory.
 *
 * Created: 15 Jun 1997 by Bill Schindler <bill@bitranch.com>
 *
 * Legal
 *   Copyright 1997 by Bill Schindler. You are granted a free and
 *   unlimited license to use, modify, and create derivatives of this code.
 *   There is no warranty that this code is fit for any purpose, whatsoever.
 *   By using this code, you agree that Bill Schindler is not responsible
 *   for anything this program may do to your computer, data, business,
 *   financial state, emotional state, intellect, sanity, or love life.
 */

SIGNAL ON Halt
CALL SetLocal

digestSuffix = '-digest'
steward. = ''
subsFile = ''

IF LoadRexxUtil() THEN EXIT

IF \ GetStewardInfo() THEN
  DO
  SAY '** Error getting Steward configuration information.'
  EXIT
  END

CALL Directory steward.!homeDir

CALL SysFileTree steward.!listDir || '\*', 'allLists.', 'DO'

DO x = 1 TO allLists.0
  listname = FileSpec('N', allLists.x)
  DO 2 /* once for normal list, once for digest */
    subsFile = allLists.x || '\' || listname
    /* Make sure file exists first */
    IF Stream(subsFile, 'C', 'Query Exists') >< '' THEN
      DO
      SAY 'Sorting' listname
      IF \ LockOpen(subsFile 'READ') THEN
        DO
        SAY '** Error locking subscription file:' subsFile
        ITERATE
        END

      list. = ''
      DO i = 1 WHILE Lines(subsFile) >< 0
        /* Reversing the mail id produces a slightly odd but effective sort */
        list.i = Reverse(LineIn(subsFile))
      END
      list.0 = i - 1
      CALL QuickSort 1, i - 1

      CALL LockClose subsFile

      rc = Lock(subsFile)
     '@copy' subsFile listname'.bak' '> NUL'

      CALL SysFileDelete subsFile

      DO i = 1 TO list.0
        CALL LineOut subsFile, Reverse(list.i)
      END

      CALL LockClose subsFile
      END
    listname = listname || digestSuffix
  END
END

SAY 'Lists now sorted.'
CALL EndLocal
EXIT

/*****************************************************************************
 * User pressed Ctrl+Break
 */
Halt:
  IF subsFile >< '' THEN
    CALL LockClose subsFile
  CALL EndLocal
  EXIT

/*****************************************************************************
 * Load the RexxUtil library
 * (Routine from "Teach Yourself REXX in 21 Days" by B & E Schindler.)
 */
LoadRexxUtil: PROCEDURE
IF RxFuncQuery('SysLoadFuncs') THEN
  DO
  IF RxFuncAdd('SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs') THEN
    DO
    SAY "Error: Couldn't load RexxUtil library."
    RETURN 1
    END
  CALL SysLoadFuncs
  END

RETURN 0

/*****************************************************************************
 * QuickSort algorithm borrowed from Mary2 StdLibs
 */
QuickSort: PROCEDURE EXPOSE list.
  ARG lower, upper
  choose = list.lower
  center = lower
  large = upper + 1
  DO WHILE (center + 1 < large)
    next = center + 1
    IF list.next <= choose THEN
      DO
      list.center = list.next
      center = center + 1
      list.next = choose
      END
    ELSE
      DO
      large = large - 1
      temp = list.large
      list.large = list.next
      list.next = temp
      END
  END
  IF center - 1 > lower THEN
    CALL QuickSort lower, center - 1
  IF center + 1 < upper THEN
    CALL QuickSort center + 1, upper
  RETURN

/*****************************************************************************
 * Get Steward configuration information
 */
GetStewardInfo: PROCEDURE EXPOSE steward.
  /* Set defaults */
  steward.!homeDir     = 'C:'
  steward.!logDir      = 'C:'
  steward.!listDir     = 'C:'
  steward.!mailer      = 'hmailer'
  steward.!whereAmI    = 'example.com'
  steward.!whoAmI      = 'Steward'
  steward.!whoAmIOwner = 'postmaster@' || steward.!whereAmI
  steward.!password    = 'steward'

  /* Find the configuration file */
  cfDir = Value('STEWARD_CF',, 'OS2ENVIRONMENT')
  IF cfDir = '' THEN
    cfDir = '.'
  steward.!cfFile = cfDir || '\steward.cf'
  IF \ LockOpen(steward.!cfFile 'READ') THEN
    RETURN 0

  /* parse the configuration file */
  DO WHILE Lines(steward.!cfFile) >< 0
    PARSE VALUE Space(Strip(LineIn(steward.!cfFile), 'B')) WITH line '#' .
    IF line = '' THEN
      ITERATE

    PARSE VAR line key '=' val
    val = Strip(val, 'B')
    SELECT
      WHEN key = 'HomeDir'        THEN steward.!homeDir     = val
      WHEN key = 'LogDir'         THEN steward.!logDir      = val
      WHEN key = 'ListDir'        THEN steward.!listDir     = val
      WHEN key = 'Mailer'         THEN steward.!mailer      = val
      WHEN key = 'WhereAmI'       THEN steward.!whereAmI    = val
      WHEN key = 'WhoAmI'         THEN steward.!whoAmI      = val
      WHEN key = 'WhoAmIOwner'    THEN steward.!whoAmIOwner = val
      WHEN key = 'MasterPassword' THEN steward.!password    = val

      OTHERWISE
        NOP
    END
  END

  CALL LockClose steward.!cfFile

  RETURN 1
