////////////////////////////////////////////////////////////////
// File - IOP480_LIB.C
//
// Library for 'WinDriver for PLX IOP 480' API. 
// The basic idea is to get a handle for the board
// with IOP480_Open() and use it in the rest of the program
// when calling WD functions.  Call IOP480_Close() when done.
// 
////////////////////////////////////////////////////////////////

#include "iop480_lib.h"
#include "../../../include/windrvr_int_thread.h"
#include <stdio.h>

// this string is set to an error message, if one occurs
CHAR IOP480_ErrorString[1024];

// internal data structures and enums
enum { IOP480_DMA_CHANNEL_SHIFT = 0x20 }; // shift in address between channels 0 and 1 of DMA

typedef struct IOP480_DMA_STRUCT{
    WD_DMA dma;
    WD_DMA dmaList;
    IOP480_DMA_CHANNEL dmaChannel;
} IOP480_DMA_STRUCT;

enum { IOP480_MODE_DESC       = 0xF9000140 };
enum { IOP480_MODE_DESC_BYTE  = 0x00000000 };
enum { IOP480_MODE_DESC_WORD  = 0x00010001 };
enum { IOP480_MODE_DESC_DWORD = 0x00030003 };

typedef struct
{
    WD_INTERRUPT Int;
    HANDLE hThread;
    WD_TRANSFER Trans[2];
    IOP480_INT_HANDLER funcIntHandler;
} IOP480_INTERRUPT;

typedef struct 
{
    DWORD dwLocalBase;
    DWORD dwMask;
    DWORD dwBytes;
    DWORD dwAddr;
    DWORD dwAddrDirect;
    BOOL  fIsMemory;
} IOP480_ADDR_DESC;

typedef struct IOP480_STRUCT
{
    HANDLE hWD;
    WD_CARD cardLock;
    WD_PCI_SLOT pciSlot;
    WD_CARD_REGISTER cardReg;
    IOP480_ADDR_DESC addrDesc[AD_PCI_BARS];
    DWORD  addrSpace;
    BOOL   fUseInt;
    IOP480_INTERRUPT Int;
} IOP480_STRUCT;


// internal function used by IOP480_Open()
BOOL IOP480_DetectCardElements(IOP480_HANDLE hPlx);
// internal function used by IOP480_Read... and IOP480_Write... functions
void IOP480_SetMode (IOP480_HANDLE hPlx, IOP480_MODE mode, DWORD dwLocalAddr);

DWORD IOP480_CountCards (DWORD dwVendorID, DWORD dwDeviceID)
{
    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    HANDLE hWD;

    IOP480_ErrorString[0] = '\0';
    hWD = WD_Open();
    // check if handle valid & version OK
    if (hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( IOP480_ErrorString, "Cannot open " WD_PROD_NAME " device\n");
        return 0;
    }

    BZERO(ver);
    WD_Version(hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( IOP480_ErrorString, "error - incorrect " WD_PROD_NAME " version\n");
        WD_Close (hWD);
        return 0;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hWD, &pciScan);
    WD_Close (hWD);
    if (pciScan.dwCards==0)
        sprintf( IOP480_ErrorString, "no cards found\n");
    return pciScan.dwCards;
}


BOOL IOP480_Open (IOP480_HANDLE *phPlx, DWORD dwVendorID, DWORD dwDeviceID, DWORD nCardNum, DWORD dwOptions)
{
    IOP480_HANDLE hPlx = (IOP480_HANDLE) malloc (sizeof (IOP480_STRUCT));

    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    WD_PCI_CARD_INFO pciCardInfo;
    DWORD dwIntEnb;

    *phPlx = NULL;
    IOP480_ErrorString[0] = '\0';
    BZERO(*hPlx);

    hPlx->cardReg.hCard = 0;
    hPlx->hWD = WD_Open();

    // check if handle valid & version OK
    if (hPlx->hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( IOP480_ErrorString, "Cannot open " WD_PROD_NAME " device\n");
        goto Exit;
    }

    BZERO(ver);
    WD_Version(hPlx->hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( IOP480_ErrorString, "error - incorrect " WD_PROD_NAME " version\n");
        goto Exit;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hPlx->hWD, &pciScan);
    if (pciScan.dwCards==0) // Found at least one card
    {
        sprintf( IOP480_ErrorString, "error - Cannot find PCI card\n");
        goto Exit;
    }
    if (pciScan.dwCards<=nCardNum)
    {
        sprintf( IOP480_ErrorString, "Card out of range of available cards\n");
        goto Exit;
    }

    BZERO(pciCardInfo);
    pciCardInfo.pciSlot = pciScan.cardSlot[nCardNum];
    WD_PciGetCardInfo (hPlx->hWD, &pciCardInfo);
    hPlx->pciSlot = pciCardInfo.pciSlot;
    hPlx->cardReg.Card = pciCardInfo.Card;

    hPlx->fUseInt = (dwOptions & IOP480_OPEN_USE_INT) ? TRUE : FALSE;
    if (!hPlx->fUseInt)
    {
        DWORD i;
        // Remove interrupt item if not needed
        for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->item = ITEM_NONE;
        }
    }
    else
    {
        DWORD i;
        // make interrupt resource sharable
        for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->fNotSharable = FALSE;
        }
    }

    hPlx->cardReg.fCheckLockOnly = FALSE;
    WD_CardRegister (hPlx->hWD, &hPlx->cardReg);
    if (hPlx->cardReg.hCard==0)
    {
        sprintf ( IOP480_ErrorString, "error - could not lock device\n");
        goto Exit;
    }

    if (!IOP480_DetectCardElements(hPlx))
    {
        sprintf ( IOP480_ErrorString, "error - card does not have all items expected for PLX IOP 480\n");
        goto Exit;
    }

    // this disables interrupts
    dwIntEnb = IOP480_ReadReg (hPlx, IOP480_PINTENB);
    IOP480_WriteReg (hPlx, IOP480_PINTENB, BIT0);

    // Open finished OK
    *phPlx = hPlx;
    return TRUE;

Exit:
    // Error durin Open
    if (hPlx->cardReg.hCard) 
        WD_CardUnregister(hPlx->hWD, &hPlx->cardReg);
    if (hPlx->hWD!=INVALID_HANDLE_VALUE)
        WD_Close(hPlx->hWD);
    free (hPlx);
    return FALSE;
}

void IOP480_GetPciSlot(IOP480_HANDLE hPlx, WD_PCI_SLOT *pPciSlot)
{
    *pPciSlot = hPlx->pciSlot;
}

DWORD IOP480_ReadPCIReg(IOP480_HANDLE hPlx, DWORD dwReg)
{
    WD_PCI_CONFIG_DUMP pciCnf;
    DWORD dwVal;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = &dwVal;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = TRUE;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
    return dwVal;
}

void IOP480_WritePCIReg(IOP480_HANDLE hPlx, DWORD dwReg, DWORD dwData)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = &dwData;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = FALSE;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
}

BOOL IOP480_DetectCardElements(IOP480_HANDLE hPlx)
{
    DWORD i;
    DWORD ad_sp;

    BZERO(hPlx->Int);
    BZERO(hPlx->addrDesc);

    for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
    {
        WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];

        switch (pItem->item)
        {
        case ITEM_MEMORY:
        case ITEM_IO:
            {
                DWORD dwBytes;
                DWORD dwAddr;
                DWORD dwPhysAddr;
                DWORD dwAddrDirect = 0;
                BOOL fIsMemory;
                if (pItem->item==ITEM_MEMORY)
                {
                    dwBytes = pItem->I.Mem.dwBytes;
                    dwAddr = pItem->I.Mem.dwTransAddr;
                    dwAddrDirect = pItem->I.Mem.dwUserDirectAddr;
                    dwPhysAddr = pItem->I.Mem.dwPhysicalAddr;
                    fIsMemory = TRUE;
                }
                else 
                {
                    dwBytes = pItem->I.IO.dwBytes;
                    dwAddr = pItem->I.IO.dwAddr;
                    dwPhysAddr = dwAddr;
                    fIsMemory = FALSE;
                }

                for (ad_sp=IOP480_ADDR_REG; ad_sp<=IOP480_ADDR_EPROM; ad_sp++)
                {
                    DWORD dwPCIAddr;
                    DWORD dwPCIReg;

                    if (hPlx->addrDesc[ad_sp].dwAddr) continue;
                    if (ad_sp==IOP480_ADDR_REG) dwPCIReg = PCI_BAR0;
                    else if (ad_sp<IOP480_ADDR_EPROM) 
                        dwPCIReg = PCI_BAR1 + 4*(ad_sp-IOP480_ADDR_SPACE0);
                    else dwPCIReg = PCI_ERBAR;
                    dwPCIAddr = IOP480_ReadPCIReg(hPlx, dwPCIReg);
                    if (dwPCIAddr & 1)
                    {
                        if (fIsMemory) continue;
                        dwPCIAddr &= ~0x3;
                    }
                    else
                    {
                        if (!fIsMemory) continue;
                        dwPCIAddr &= ~0xf;
                    }
                    if (dwPCIAddr==dwPhysAddr)
                        break;
                }
                if (ad_sp<=IOP480_ADDR_EPROM)
                {
                    DWORD j;
                    hPlx->addrDesc[ad_sp].dwBytes = dwBytes;
                    hPlx->addrDesc[ad_sp].dwAddr = dwAddr;
                    hPlx->addrDesc[ad_sp].dwAddrDirect = dwAddrDirect;
                    hPlx->addrDesc[ad_sp].fIsMemory = fIsMemory;
                    hPlx->addrDesc[ad_sp].dwMask = 0;
                    for (j=1; j<hPlx->addrDesc[ad_sp].dwBytes && j!=0x80000000; j *= 2)
                    {
                        hPlx->addrDesc[ad_sp].dwMask = 
                            (hPlx->addrDesc[ad_sp].dwMask << 1) | 1;
                    }
                }
            }
            break;
        case ITEM_INTERRUPT:
            if (hPlx->Int.Int.hInterrupt) return FALSE;
            hPlx->Int.Int.hInterrupt = pItem->I.Int.hInterrupt;
            break;
        }
    }

    // check that all the items needed were found
    // check if interrupt found
    if (hPlx->fUseInt && !hPlx->Int.Int.hInterrupt) 
    {
        return FALSE;
    }

    // check that the registers space was found
    if (!IOP480_IsAddrSpaceActive(hPlx, IOP480_ADDR_REG)) 
            //|| hPlx->addrDesc[IOP480_ADDR_REG].dwBytes!=IOP480_RANGE_REG)
        return FALSE;

    // use address space 0 for accessing local addresses
    hPlx->addrSpace = IOP480_ADDR_SPACE0;
    
    // check that address space 0 was found
    if (!IOP480_IsAddrSpaceActive(hPlx, hPlx->addrSpace)) return FALSE;

    return TRUE;
}

void IOP480_Close(IOP480_HANDLE hPlx)
{
    // disable interrupts
    if (IOP480_IntIsEnabled(hPlx))
        IOP480_IntDisable(hPlx);

    // unregister card
    if (hPlx->cardReg.hCard) 
        WD_CardUnregister(hPlx->hWD, &hPlx->cardReg);

    // close WinDriver
    WD_Close(hPlx->hWD);

    free (hPlx);
}

BOOL IOP480_IsAddrSpaceActive(IOP480_HANDLE hPlx, IOP480_ADDR addrSpace)
{
    return hPlx->addrDesc[addrSpace].dwAddr!=0;
}

DWORD IOP480_ReadReg (IOP480_HANDLE hPlx, DWORD dwReg)
{
    return IOP480_ReadDWord(hPlx, IOP480_ADDR_REG, dwReg);
}

void IOP480_WriteReg (IOP480_HANDLE hPlx, DWORD dwReg, DWORD dwData)
{
    IOP480_WriteDWord(hPlx, IOP480_ADDR_REG, dwReg, dwData);
}

BYTE IOP480_ReadByte (IOP480_HANDLE hPlx, IOP480_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        BYTE *pByte = (BYTE *) dwAddr;
        return *pByte;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_BYTE;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Byte;
    }
}

void IOP480_WriteByte (IOP480_HANDLE hPlx, IOP480_ADDR addrSpace, DWORD dwOffset, BYTE data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        BYTE *pByte = (BYTE *) dwAddr;
        *pByte = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_BYTE;
        trans.dwPort = dwAddr;
        trans.Data.Byte = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

WORD IOP480_ReadWord (IOP480_HANDLE hPlx, IOP480_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        WORD *pWord = (WORD *) dwAddr;
        return *pWord;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_WORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Word;
    }
}

void IOP480_WriteWord (IOP480_HANDLE hPlx, IOP480_ADDR addrSpace, DWORD dwOffset, WORD data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        WORD *pWord = (WORD *) dwAddr;
        *pWord = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_WORD;
        trans.dwPort = dwAddr;
        trans.Data.Word = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

DWORD IOP480_ReadDWord (IOP480_HANDLE hPlx, IOP480_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        return *pDword;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_DWORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Dword;
    }
}

void IOP480_WriteDWord (IOP480_HANDLE hPlx, IOP480_ADDR addrSpace, DWORD dwOffset, DWORD data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        *pDword = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_DWORD;
        trans.dwPort = dwAddr;
        trans.Data.Dword = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

void IOP480_ReadWriteBlock (IOP480_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, IOP480_ADDR addrSpace, IOP480_MODE mode)
{
    DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
    WD_TRANSFER trans;

    BZERO(trans);

    if (hPlx->addrDesc[addrSpace].fIsMemory) 
    {
        if (fIsRead) 
        {
            if (mode==IOP480_MODE_BYTE) trans.cmdTrans = RM_SBYTE;
            else if (mode==IOP480_MODE_WORD) trans.cmdTrans = RM_SWORD;
            else trans.cmdTrans = RM_SDWORD;
        }
        else 
        {
            if (mode==IOP480_MODE_BYTE) trans.cmdTrans = WM_SBYTE;
            else if (mode==IOP480_MODE_WORD) trans.cmdTrans = WM_SWORD;
            else trans.cmdTrans = WM_SDWORD;
        }
    }
    else 
    {
        if (fIsRead) 
        {
            if (mode==IOP480_MODE_BYTE) trans.cmdTrans = RP_SBYTE;
            else if (mode==IOP480_MODE_WORD) trans.cmdTrans = RP_SWORD;
            else trans.cmdTrans = RP_SDWORD;
        }
        else 
        {
            if (mode==IOP480_MODE_BYTE) trans.cmdTrans = WP_SBYTE;
            else if (mode==IOP480_MODE_WORD) trans.cmdTrans = WP_SWORD;
            else trans.cmdTrans = WP_SDWORD;
        }
    }
    trans.dwPort = dwAddr;
    trans.fAutoinc = TRUE;
    trans.dwBytes = dwBytes;
    trans.dwOptions = 0;
    trans.Data.pBuffer = buf;
    WD_Transfer (hPlx->hWD, &trans);
}

void IOP480_ReadBlock (IOP480_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, IOP480_ADDR addrSpace, IOP480_MODE mode)
{
    IOP480_ReadWriteBlock (hPlx, dwOffset, buf, dwBytes, TRUE, addrSpace, mode);
}

void IOP480_WriteBlock (IOP480_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                     DWORD dwBytes, IOP480_ADDR addrSpace, IOP480_MODE mode)
{
    IOP480_ReadWriteBlock (hPlx, dwOffset, buf, dwBytes, FALSE, addrSpace, mode);
}

void IOP480_SetMode (IOP480_HANDLE hPlx, IOP480_MODE mode, DWORD dwLocalAddr)
{
    hPlx->addrDesc[hPlx->addrSpace].dwLocalBase = dwLocalAddr & ~hPlx->addrDesc[hPlx->addrSpace].dwMask;
    hPlx->addrDesc[hPlx->addrSpace].dwLocalBase |= BIT0;
    IOP480_WriteReg (hPlx, IOP480_LAS0BA, hPlx->addrDesc[hPlx->addrSpace].dwLocalBase);
}

BYTE IOP480_ReadByteLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    IOP480_SetMode (hPlx, IOP480_MODE_BYTE, dwLocalAddr);
    return IOP480_ReadByte(hPlx, hPlx->addrSpace, dwOffset);
}

void IOP480_WriteByteLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr, BYTE data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    IOP480_SetMode (hPlx, IOP480_MODE_BYTE, dwLocalAddr);
    IOP480_WriteByte(hPlx, hPlx->addrSpace, dwOffset, data);
}

WORD IOP480_ReadWordLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    IOP480_SetMode (hPlx, IOP480_MODE_WORD, dwLocalAddr);
    return IOP480_ReadWord(hPlx, hPlx->addrSpace, dwOffset);
}

void IOP480_WriteWordLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr, WORD data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    IOP480_SetMode (hPlx, IOP480_MODE_WORD, dwLocalAddr);
    IOP480_WriteWord(hPlx, hPlx->addrSpace, dwOffset, data);
}

DWORD IOP480_ReadDWordLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    IOP480_SetMode (hPlx, IOP480_MODE_DWORD, dwLocalAddr);
    return IOP480_ReadDWord(hPlx, hPlx->addrSpace, dwOffset);
}

void IOP480_WriteDWordLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr, DWORD data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    IOP480_SetMode (hPlx, IOP480_MODE_DWORD, dwLocalAddr);
    IOP480_WriteDWord(hPlx, hPlx->addrSpace, dwOffset, data);
}

void IOP480_ReadWriteBlockLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, IOP480_MODE mode)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    IOP480_SetMode (hPlx, mode, dwLocalAddr);
    IOP480_ReadWriteBlock(hPlx, dwOffset, buf, dwBytes, fIsRead, hPlx->addrSpace, mode);
}

void IOP480_ReadBlockLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                    DWORD dwBytes, IOP480_MODE mode)
{
    IOP480_ReadWriteBlockLocal (hPlx, dwLocalAddr, buf, dwBytes, TRUE, mode);
}

void IOP480_WriteBlockLocal (IOP480_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                     DWORD dwBytes, IOP480_MODE mode)
{
    IOP480_ReadWriteBlockLocal (hPlx, dwLocalAddr, buf, dwBytes, FALSE, mode);
}

BOOL IOP480_IntIsEnabled (IOP480_HANDLE hPlx)
{
    if (!hPlx->fUseInt) return FALSE;
    if (!hPlx->Int.hThread) return FALSE;
    return TRUE;
}

VOID IOP480_IntHandler (PVOID pData)
{
    IOP480_HANDLE hPlx = (IOP480_HANDLE) pData;
    IOP480_INT_RESULT intResult;
    intResult.dwCounter = hPlx->Int.Int.dwCounter;
    intResult.dwLost = hPlx->Int.Int.dwLost;
    intResult.fStopped = hPlx->Int.Int.fStopped;
    intResult.dwStatusReg = hPlx->Int.Trans[0].Data.Dword;
    hPlx->Int.funcIntHandler(hPlx, &intResult);  
}

BOOL IOP480_IntEnable (IOP480_HANDLE hPlx, IOP480_INT_HANDLER funcIntHandler)
{
    DWORD dwIntStatus;
    DWORD dwIntEnbAddr, dwIntStatAddr;

    if (!hPlx->fUseInt) return FALSE;
    // check if interrupt is already enabled
    if (hPlx->Int.hThread) return FALSE;

    dwIntStatus = IOP480_ReadReg (hPlx, IOP480_PINTENB);

    BZERO(hPlx->Int.Trans);
    // This is a samlpe of handling interrupts:
    // Two transfer commands are issued. First the value of the interrrupt status
    // register is read.
    // The second will cancel interrupts after the first interrupt occurs.
    // When using interrupts, this section will have to change:
    // you must put transfer commands to CANCEL the source of the interrupt, otherwise, the 
    // PC will hang when an interrupt occurs!
    dwIntEnbAddr = hPlx->addrDesc[IOP480_ADDR_REG].dwAddr + IOP480_PINTENB;
    dwIntStatAddr = hPlx->addrDesc[IOP480_ADDR_REG].dwAddr + IOP480_PINTSTAT;
    hPlx->Int.Trans[0].cmdTrans = hPlx->addrDesc[IOP480_ADDR_REG].fIsMemory ? RM_DWORD : RP_DWORD;
    hPlx->Int.Trans[0].dwPort = dwIntStatAddr;
    hPlx->Int.Trans[1].cmdTrans = hPlx->addrDesc[IOP480_ADDR_REG].fIsMemory ? WM_DWORD : WP_DWORD;
    hPlx->Int.Trans[1].dwPort = dwIntEnbAddr;
    hPlx->Int.Trans[1].Data.Dword = dwIntStatus & ~BIT0; // put here the data to write to the control register
    hPlx->Int.Int.dwCmds = 2; 
    hPlx->Int.Int.Cmd = hPlx->Int.Trans;
    hPlx->Int.Int.dwOptions |= INTERRUPT_CMD_COPY;
 
    // this calls WD_IntEnable() and creates an interrupt handler thread
    hPlx->Int.funcIntHandler = funcIntHandler;
    if (!InterruptThreadEnable(&hPlx->Int.hThread, hPlx->hWD, &hPlx->Int.Int, IOP480_IntHandler, (PVOID) hPlx))
        return FALSE;

    // this physically enables interrupts
    IOP480_WriteReg (hPlx, IOP480_PINTENB, BIT0|BIT8|BIT9|BIT10|BIT11|BIT12|BIT13);

    return TRUE;
}

void IOP480_IntDisable (IOP480_HANDLE hPlx)
{
    DWORD dwIntStatus;

    if (!hPlx->fUseInt) return;
    if (!hPlx->Int.hThread) return;

    // this disables interrupts
    dwIntStatus = IOP480_ReadReg (hPlx, IOP480_PINTENB);
    IOP480_WriteReg (hPlx, IOP480_PINTENB, dwIntStatus & ~BIT0);

    // this calls WD_IntDisable()
    InterruptThreadDisable(hPlx->Int.hThread);
    hPlx->Int.hThread = NULL;
}

IOP480_DMA_HANDLE IOP480_DMAOpen (IOP480_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
    DWORD dwBytes, BOOL fIsRead, IOP480_DMA_CHANNEL dmaChannel)
{
    DWORD dwDMAMODE, dwDMADPR, dwDMALADR, dwDMACOUNT, dwDMAPLADR;
    DWORD dwChannelShift = (dmaChannel==IOP480_DMA_CHANNEL_0) ? 0 : IOP480_DMA_CHANNEL_SHIFT;
    BOOL fAutoinc = TRUE;
    IOP480_DMA_HANDLE hDma;
    
    hDma = malloc (sizeof(IOP480_DMA_STRUCT));
    if (hDma==NULL)
    {
        sprintf( IOP480_ErrorString, "could not allocate memory for dma handle!\n");
        goto Exit;
    }
    BZERO (*hDma);
    hDma->dmaChannel = dmaChannel;
    hDma->dma.dwBytes = dwBytes;
    hDma->dma.pUserAddr = buf; 
    hDma->dma.dwOptions = 0; 
    WD_DMALock (hPlx->hWD, &hDma->dma);
    if (!hDma->dma.hDma) 
    {
        sprintf( IOP480_ErrorString, "could not lock the buffer!\n");
        goto Exit;
    }
    if (hDma->dma.dwPages==1)
    {
        //dma of one page ==> direct dma
        dwDMAMODE = fAutoinc ? 0 : BIT11; 
        dwDMADPR = BIT0 | (fIsRead ? BIT3 : 0);
        dwDMALADR = dwLocalAddr;
        dwDMACOUNT = (hDma->dma.Page[0].dwBytes & 0x007fffff);
        dwDMAPLADR = (DWORD) hDma->dma.Page[0].pPhysicalAddr;

        IOP480_WriteReg (hPlx, IOP480_DMA_MODE    + dwChannelShift, dwDMAMODE);
        IOP480_WriteReg (hPlx, IOP480_DMA_PCILADR + dwChannelShift, dwDMAPLADR);
        IOP480_WriteReg (hPlx, IOP480_DMA_LOCADR  + dwChannelShift, dwDMALADR);
        IOP480_WriteReg (hPlx, IOP480_DMA_COUNT   + dwChannelShift, dwDMACOUNT);
        IOP480_WriteReg (hPlx, IOP480_DMA_DPR     + dwChannelShift, dwDMADPR);
    }
    else
    {
        DWORD dwAlignShift, dwPageNumber, dwMemoryCopied;
        typedef struct {
            DWORD dwSIZ;
            DWORD dwPADR;
            DWORD dwLADR;
            DWORD dwDPR;
        } DMA_LIST;
        DMA_LIST *pList;

        //dma of more then one page ==> chain dma
        hDma->dmaList.dwBytes = hDma->dma.dwPages * sizeof(DMA_LIST) + 0x10; // includes extra 0x10 bytes for quadword alignment
        hDma->dmaList.pUserAddr = NULL;
        hDma->dmaList.dwOptions = DMA_KERNEL_BUFFER_ALLOC;
        WD_DMALock (hPlx->hWD, &hDma->dmaList);
        if (!hDma->dmaList.hDma)
        {
            sprintf (IOP480_ErrorString, "could not lock the chain buffer!\n");
            goto Exit;
        }

        //setting chain of dma pages in the memory
        dwMemoryCopied = 0;
        dwAlignShift = 0x10 - ((DWORD) hDma->dmaList.pUserAddr & 0xf);
        // verifcation that bits 0-3 are zero (quadword aligned)
        pList = (DMA_LIST *) ((DWORD) hDma->dmaList.pUserAddr + dwAlignShift);
        for (dwPageNumber=0; dwPageNumber<hDma->dma.dwPages; dwPageNumber++)
        {
            pList[dwPageNumber].dwPADR = (DWORD) hDma->dma.Page[dwPageNumber].pPhysicalAddr;
            pList[dwPageNumber].dwLADR = dwLocalAddr + (fAutoinc ? dwMemoryCopied : 0);
            pList[dwPageNumber].dwSIZ = hDma->dma.Page[dwPageNumber].dwBytes;
            pList[dwPageNumber].dwDPR = 
                ((DWORD) hDma->dmaList.Page[0].pPhysicalAddr + dwAlignShift + sizeof(DMA_LIST)*(dwPageNumber+1))
                | BIT0 | (fIsRead ? BIT3 : 0);
            dwMemoryCopied += hDma->dma.Page[dwPageNumber].dwBytes;
        }
        pList[dwPageNumber - 1].dwDPR |= BIT1; // mark end of chain
    
        dwDMAMODE = (fAutoinc ? 0 : BIT11) 
                    | BIT9;        // chain transfer

        dwDMADPR = ((DWORD)hDma->dmaList.Page[0].pPhysicalAddr + dwAlignShift) | BIT0; 
        // starting the dma
        IOP480_WriteReg (hPlx, IOP480_DMA_MODE + dwChannelShift, dwDMAMODE);
        IOP480_WriteReg (hPlx, IOP480_DMA_DPR  + dwChannelShift, dwDMADPR);
    }

    return hDma;

Exit:
    if (hDma!=NULL)
        IOP480_DMAClose(hPlx,hDma);
    return NULL;
}

void IOP480_DMAClose (IOP480_HANDLE hPlx, IOP480_DMA_HANDLE hDma)
{
    if (hDma->dma.hDma)
        WD_DMAUnlock(hPlx->hWD, &hDma->dma);
    if (hDma->dmaList.hDma)
        WD_DMAUnlock(hPlx->hWD, &hDma->dmaList);
    free (hDma);
}

BOOL IOP480_DMAIsDone (IOP480_HANDLE hPlx, IOP480_DMA_HANDLE hDma)
{
    return (IOP480_ReadReg (hPlx, IOP480_DMA_CSR + hDma->dmaChannel*IOP480_DMA_CHANNEL_SHIFT) & BIT4)==BIT4;
}

void IOP480_DMAStart (IOP480_HANDLE hPlx, IOP480_DMA_HANDLE hDma, BOOL fBlocking)
{
    IOP480_WriteByte (hPlx, IOP480_ADDR_REG, IOP480_DMA_CSR + 
        hDma->dmaChannel * IOP480_DMA_CHANNEL_SHIFT, BIT0 | BIT1);

    //Busy wait for plx to finish transfer
    if (fBlocking) 
        while (!IOP480_DMAIsDone(hPlx, hDma));
}

BOOL IOP480_DMAReadWriteBlock (IOP480_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
    DWORD dwBytes, BOOL fIsRead, IOP480_DMA_CHANNEL dmaChannel)
{
    IOP480_DMA_HANDLE hDma;
    if (dwBytes==0) 
        return TRUE;

    hDma = IOP480_DMAOpen(hPlx, dwLocalAddr, buf, dwBytes, fIsRead, dmaChannel);
    if (!hDma) 
        return FALSE;

    IOP480_DMAStart(hPlx, hDma, TRUE);
    IOP480_DMAClose(hPlx, hDma);
    return TRUE;
}

BOOL IOP480_EEPROMValid(IOP480_HANDLE hPlx)
{
    return (IOP480_ReadReg(hPlx, IOP480_DEVINIT) & BIT5)==BIT5;
}

BOOL IOP480_EEPROMReadWord(IOP480_HANDLE hPlx, DWORD dwOffset, PWORD pwData)
{
    DWORD dwData;
    DWORD dwAddr;

    if (dwOffset % 2)
    {
        sprintf (IOP480_ErrorString, "The offset is not even\n");
        return FALSE;
    }
    dwAddr = dwOffset - (dwOffset % 4);

    if (!IOP480_EEPROMReadDWord(hPlx, dwAddr, &dwData))
        return FALSE;

    *pwData = (WORD) (dwData >> ((dwOffset % 4)*8));

    return TRUE;
}

BOOL IOP480_EEPROMWriteWord(IOP480_HANDLE hPlx, DWORD dwOffset, WORD wData)
{
    DWORD dwData;
    DWORD dwAddr;

    dwAddr = dwOffset - (dwOffset % 4);

    if (!IOP480_EEPROMReadDWord(hPlx, dwAddr, &dwData))
        return FALSE;

    switch (dwOffset % 4)
    {
    case 0:
        dwData = (dwData & 0xffff0000) | wData;
        break;
    case 2:
        dwData = (dwData & 0x0000ffff) | (wData << 16);
        break;
    default:
        sprintf (IOP480_ErrorString, "The offset is not even\n");
        return FALSE;
    }

    return IOP480_EEPROMWriteDWord(hPlx, dwAddr, dwData);
}

void IOP480_Sleep(IOP480_HANDLE hPlx, DWORD dwMicroSeconds)
{
    WD_SLEEP sleep;

    BZERO (sleep);
    sleep.dwMicroSeconds = dwMicroSeconds;
    WD_Sleep( hPlx->hWD, &sleep);
}

BYTE IOP480_EEPROMEnable(IOP480_HANDLE hPlx, WORD addr)
{
    DWORD dwVal ,dwAddr;
    BYTE bOldValue;
    DWORD mask = BIT8|BIT9|BIT10|BIT11|BIT12|BIT13|BIT14;

    dwVal = IOP480_ReadReg (hPlx, IOP480_DEVINIT);
    bOldValue = (BYTE) ((dwVal & mask) >> 8);
    dwVal = (dwVal & ~mask) ;
    dwAddr = addr / 4; // using LWord (4 bytes) address.
    dwAddr <<= 8;
    dwAddr &= mask;

    dwVal = dwVal | dwAddr ;

    IOP480_WriteReg (hPlx, IOP480_DEVINIT, dwVal) ;
    IOP480_Sleep(hPlx, 10000);

    return bOldValue * 4;
}

void IOP480_EEPROMDataReadWrite(IOP480_HANDLE hPlx, BOOL fIsRead, PDWORD pdwData)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = pdwData;
    pciCnf.dwOffset = IOP480_VPD_DATA;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = fIsRead;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
}

void IOP480_EEPROMAddrReadWrite(IOP480_HANDLE hPlx, BOOL fIsRead, PWORD pwAddr)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = pwAddr;
    pciCnf.dwOffset = IOP480_VPD_ADDR;
    pciCnf.dwBytes = 2;
    pciCnf.fIsRead = fIsRead;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
}

BOOL IOP480_EEPROMReadDWord(IOP480_HANDLE hPlx, DWORD dwOffset, PDWORD pdwData)
{
    WORD wVal;
    WORD wAddr;
    int i ;
    BOOL fEnd = FALSE ;

    if (dwOffset % 4)
    {
        sprintf (IOP480_ErrorString, "The offset is not a multiple of 4\n");
        return FALSE;
    }
    wAddr = (((WORD)dwOffset) & (~BIT15)) ;
 
    
    IOP480_EEPROMAddrReadWrite(hPlx, FALSE, &wAddr);

    IOP480_Sleep(hPlx, 10000);

    for (i=0; !fEnd && i<100; i++)
    {
        IOP480_EEPROMAddrReadWrite(hPlx, TRUE, &wVal);
        if (wVal & BIT15)
            fEnd = TRUE;
        IOP480_Sleep(hPlx, 10000);
    }
    
    if (i==100)
    {
        sprintf (IOP480_ErrorString, "Acknoledge to EEPROM read was not recived\n");
        return FALSE;
    }

    IOP480_EEPROMDataReadWrite(hPlx, TRUE, pdwData);

    return TRUE;
}

BOOL IOP480_EEPROMWriteDWord(IOP480_HANDLE hPlx, DWORD dwOffset, DWORD dwData)
{
    DWORD dwReadback;
    WORD wAddr;
    WORD wVal;
    int i;
    BOOL fRet;
    BOOL fEnd = FALSE ;
    BOOL fReadOk = FALSE;
    BYTE bEnableOffset;


    if (dwOffset % 4)
    {
        sprintf (IOP480_ErrorString, "The offset is not a multiple of 4\n");
        return FALSE;
    }
    wAddr = (WORD)dwOffset;
    bEnableOffset = IOP480_EEPROMEnable(hPlx, wAddr);
    wAddr |= BIT15;

    IOP480_EEPROMDataReadWrite(hPlx, FALSE, &dwData);

    IOP480_EEPROMAddrReadWrite(hPlx, FALSE, &wAddr);

    IOP480_Sleep(hPlx, 10000);
    
    for (i=0; !fEnd && i<100 ;i++)
    {
        IOP480_EEPROMAddrReadWrite(hPlx, TRUE, &wVal);
        if ((wVal & BIT15) == 0)
            fEnd = TRUE;
        IOP480_Sleep(hPlx, 10000);
    }

    fReadOk = IOP480_EEPROMReadDWord(hPlx, dwOffset, &dwReadback);

    if (fReadOk && dwReadback==dwData)
        fRet = TRUE;
    else
    {
        fRet = FALSE;
        if (fReadOk)
            sprintf (IOP480_ErrorString, "Write 0x%08x, Read 0x%08x\n",dwData, dwReadback);
        else
            sprintf (IOP480_ErrorString, "Error reading EEPROM\n");
    }
    IOP480_EEPROMEnable(hPlx, bEnableOffset); 
    return fRet;
}
