////////////////////////////////////////////////////////////////
// File - P9060_LIB.C
//
// Library for 'WinDriver for PLX 9060' API. 
// The basic idea is to get a handle for the board
// with P9060_Open() and use it in the rest of the program
// when calling WD functions.  Call P9060_Close() when done.
// 
////////////////////////////////////////////////////////////////

#include "../../../include/windrvr.h"
#include "../../../include/windrvr_int_thread.h"
#include "p9060_lib.h"
#include <stdio.h>

// this string is set to an error message, if one occurs
CHAR P9060_ErrorString[1024];

// internal function used by P9060_Open()
BOOL P9060_DetectCardElements(P9060_HANDLE hPlx);
// internal function used by P9060_Read... and P9060_Write... functions
void P9060_SetMode (P9060_HANDLE hPlx, P9060_MODE mode, DWORD dwLocalAddr);

DWORD P9060_CountCards (DWORD dwVendorID, DWORD dwDeviceID)
{
    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    HANDLE hWD;

    P9060_ErrorString[0] = '\0';
    hWD = WD_Open();
    // check if handle valid & version OK
    if (hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( P9060_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        return 0;
    }

    BZERO(ver);
    WD_Version(hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( P9060_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        WD_Close (hWD);
        return 0;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hWD, &pciScan);
    WD_Close (hWD);
    if (pciScan.dwCards==0)
        sprintf( P9060_ErrorString, "no cards found\n");
    return pciScan.dwCards;
}


BOOL P9060_Open (P9060_HANDLE *phPlx, DWORD dwVendorID, DWORD dwDeviceID, DWORD nCardNum, DWORD dwOptions)
{
    P9060_HANDLE hPlx = (P9060_HANDLE) malloc (sizeof (P9060_STRUCT));
    DWORD dwIntStatus;

    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    WD_PCI_CARD_INFO pciCardInfo;

    *phPlx = NULL;
    P9060_ErrorString[0] = '\0';
    BZERO(*hPlx);

    hPlx->cardReg.hCard = 0;
    hPlx->hWD = WD_Open();

    // check if handle valid & version OK
    if (hPlx->hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( P9060_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        goto Exit;
    }

    BZERO(ver);
    WD_Version(hPlx->hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( P9060_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        goto Exit;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hPlx->hWD, &pciScan);
    if (pciScan.dwCards==0) // Found at least one card
    {
        sprintf( P9060_ErrorString, "Could not find PCI card\n");
        goto Exit;
    }
    if (pciScan.dwCards<=nCardNum)
    {
        sprintf( P9060_ErrorString, "Card out of range of available cards\n");
        goto Exit;
    }

    BZERO(pciCardInfo);
    pciCardInfo.pciSlot = pciScan.cardSlot[nCardNum];
    WD_PciGetCardInfo (hPlx->hWD, &pciCardInfo);
    hPlx->pciSlot = pciCardInfo.pciSlot;
    hPlx->cardReg.Card = pciCardInfo.Card;

    hPlx->fUseInt = (dwOptions & P9060_OPEN_USE_INT) ? TRUE : FALSE;
    if (!hPlx->fUseInt)
    {
        DWORD i;
        // Remove interrupt item if not needed
        for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->item = ITEM_NONE;
        }
    }
    else
    {
        DWORD i;
        // make interrupt resource sharable
        for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->fNotSharable = FALSE;
        }
    }

    hPlx->cardReg.fCheckLockOnly = FALSE;
    WD_CardRegister (hPlx->hWD, &hPlx->cardReg);
    if (hPlx->cardReg.hCard==0)
    {
        sprintf ( P9060_ErrorString, "Failed locking device\n");
        goto Exit;
    }

    if (!P9060_DetectCardElements(hPlx))
    {
        sprintf ( P9060_ErrorString, "Card does not have all items expected for PLX 9060\n");
        goto Exit;
    }

    // set the values to be written to REGION_DESC register
    hPlx->modeDesc[P9060_MODE_BYTE] = (DWORD) (P9060_MODE_DESC | P9060_MODE_DESC_BYTE);
    hPlx->modeDesc[P9060_MODE_WORD] = (DWORD) (P9060_MODE_DESC | P9060_MODE_DESC_WORD);
    hPlx->modeDesc[P9060_MODE_DWORD] = (DWORD) (P9060_MODE_DESC | P9060_MODE_DESC_DWORD);

    // this enables target abort so it wont get stuck 
    dwIntStatus = P9060_ReadReg (hPlx, P9060_INT_STATUS);
    P9060_WriteReg (hPlx, P9060_INT_STATUS, dwIntStatus | BIT12);

    // Open finished OK
    *phPlx = hPlx;
    return TRUE;

Exit:
    // Error durin Open
    if (hPlx->cardReg.hCard) 
        WD_CardUnregister(hPlx->hWD, &hPlx->cardReg);
    if (hPlx->hWD!=INVALID_HANDLE_VALUE)
        WD_Close(hPlx->hWD);
    free (hPlx);
    return FALSE;
}

DWORD P9060_ReadPCIReg(P9060_HANDLE hPlx, DWORD dwReg)
{
    WD_PCI_CONFIG_DUMP pciCnf;
    DWORD dwVal;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = &dwVal;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = TRUE;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
    return dwVal;
}

void P9060_WritePCIReg(P9060_HANDLE hPlx, DWORD dwReg, DWORD dwData)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = &dwData;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = FALSE;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
}

BOOL P9060_DetectCardElements(P9060_HANDLE hPlx)
{
    DWORD i;
    DWORD ad_sp;

    BZERO(hPlx->Int);
    BZERO(hPlx->addrDesc);

    for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
    {
        WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];

        switch (pItem->item)
        {
        case ITEM_MEMORY:
        case ITEM_IO:
            {
                DWORD dwBytes;
                DWORD dwAddr;
                DWORD dwPhysAddr;
                DWORD dwAddrDirect = 0;
                BOOL fIsMemory;
                if (pItem->item==ITEM_MEMORY)
                {
                    dwBytes = pItem->I.Mem.dwBytes;
                    dwAddr = pItem->I.Mem.dwTransAddr;
                    dwAddrDirect = pItem->I.Mem.dwUserDirectAddr;
                    dwPhysAddr = pItem->I.Mem.dwPhysicalAddr;
                    fIsMemory = TRUE;
                }
                else 
                {
                    dwBytes = pItem->I.IO.dwBytes;
                    dwAddr = pItem->I.IO.dwAddr;
                    dwPhysAddr = dwAddr;
                    fIsMemory = FALSE;
                }

                for (ad_sp=P9060_ADDR_REG; ad_sp<=P9060_ADDR_EPROM; ad_sp++)
                {
                    DWORD dwPCIAddr;
                    DWORD dwPCIReg;

                    if (hPlx->addrDesc[ad_sp].dwAddr) continue;
                    if (ad_sp==P9060_ADDR_REG) dwPCIReg = PCI_BAR0;
                    else if (ad_sp<P9060_ADDR_EPROM) 
                        dwPCIReg = PCI_BAR2 + 4*(ad_sp-P9060_ADDR_SPACE0);
                    else dwPCIReg = PCI_ERBAR;
                    dwPCIAddr = P9060_ReadPCIReg(hPlx, dwPCIReg);
                    if (dwPCIAddr & 1)
                    {
                        if (fIsMemory) continue;
                        dwPCIAddr &= ~0x3;
                    }
                    else
                    {
                        if (!fIsMemory) continue;
                        dwPCIAddr &= ~0xf;
                    }
                    if (dwPCIAddr==dwPhysAddr)
                        break;
                }
                if (ad_sp<=P9060_ADDR_EPROM)
                {
                    DWORD j;
                    hPlx->addrDesc[ad_sp].dwBytes = dwBytes;
                    hPlx->addrDesc[ad_sp].dwAddr = dwAddr;
                    hPlx->addrDesc[ad_sp].dwAddrDirect = dwAddrDirect;
                    hPlx->addrDesc[ad_sp].fIsMemory = fIsMemory;
                    hPlx->addrDesc[ad_sp].dwMask = 0;
                    for (j=1; j<hPlx->addrDesc[ad_sp].dwBytes && j!=0x80000000; j *= 2)
                    {
                        hPlx->addrDesc[ad_sp].dwMask = 
                            (hPlx->addrDesc[ad_sp].dwMask << 1) | 1;
                    }
                }
            }
            break;
        case ITEM_INTERRUPT:
            if (hPlx->Int.Int.hInterrupt) return FALSE;
            hPlx->Int.Int.hInterrupt = pItem->I.Int.hInterrupt;
            break;
        }
    }

    // check that all the items needed were found
    // check if interrupt found
    if (hPlx->fUseInt && !hPlx->Int.Int.hInterrupt) 
    {
        return FALSE;
    }

    // check that the registers space was found
    if (!P9060_IsAddrSpaceActive(hPlx, P9060_ADDR_REG)) 
            //|| hPlx->addrDesc[P9060_ADDR_REG].dwBytes!=P9060_RANGE_REG)
        return FALSE;

    // check that address space 0 was found
    if (!P9060_IsAddrSpaceActive(hPlx, P9060_ADDR_SPACE0)) return FALSE;
    hPlx->addrSpace = P9060_ADDR_SPACE0;

    return TRUE;
}

void P9060_Close(P9060_HANDLE hPlx)
{
    // disable interrupts
    if (P9060_IntIsEnabled(hPlx))
        P9060_IntDisable(hPlx);

    // unregister card
    if (hPlx->cardReg.hCard) 
        WD_CardUnregister(hPlx->hWD, &hPlx->cardReg);

    // close WinDriver
    WD_Close(hPlx->hWD);

    free (hPlx);
}

BOOL P9060_IsAddrSpaceActive(P9060_HANDLE hPlx, P9060_ADDR addrSpace)
{
    return hPlx->addrDesc[addrSpace].dwAddr!=0;
}

DWORD P9060_ReadReg (P9060_HANDLE hPlx, DWORD dwReg)
{
    return P9060_ReadSpaceDWord(hPlx, P9060_ADDR_REG, dwReg);
}

void P9060_WriteReg (P9060_HANDLE hPlx, DWORD dwReg, DWORD dwData)
{
    P9060_WriteSpaceDWord(hPlx, P9060_ADDR_REG, dwReg, dwData);
}

BYTE P9060_ReadSpaceByte (P9060_HANDLE hPlx, P9060_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        BYTE *pByte = (BYTE *) dwAddr;
        return *pByte;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_BYTE;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Byte;
    }
}

void P9060_WriteSpaceByte (P9060_HANDLE hPlx, P9060_ADDR addrSpace, DWORD dwOffset, BYTE data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        BYTE *pByte = (BYTE *) dwAddr;
        *pByte = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_BYTE;
        trans.dwPort = dwAddr;
        trans.Data.Byte = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

WORD P9060_ReadSpaceWord (P9060_HANDLE hPlx, P9060_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        WORD *pWord = (WORD *) dwAddr;
        return *pWord;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_WORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Word;
    }
}

void P9060_WriteSpaceWord (P9060_HANDLE hPlx, P9060_ADDR addrSpace, DWORD dwOffset, WORD data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        WORD *pWord = (WORD *) dwAddr;
        *pWord = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_WORD;
        trans.dwPort = dwAddr;
        trans.Data.Word = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

DWORD P9060_ReadSpaceDWord (P9060_HANDLE hPlx, P9060_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        return *pDword;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_DWORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Dword;
    }
}

void P9060_WriteSpaceDWord (P9060_HANDLE hPlx, P9060_ADDR addrSpace, DWORD dwOffset, DWORD data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        *pDword = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_DWORD;
        trans.dwPort = dwAddr;
        trans.Data.Dword = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

void P9060_ReadWriteSpaceBlock (P9060_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, P9060_ADDR addrSpace, P9060_MODE mode)
{
    DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
    WD_TRANSFER trans;

    BZERO(trans);

    if (hPlx->addrDesc[addrSpace].fIsMemory) 
    {
        if (fIsRead) 
        {
            if (mode==P9060_MODE_BYTE) trans.cmdTrans = RM_SBYTE;
            else if (mode==P9060_MODE_WORD) trans.cmdTrans = RM_SWORD;
            else trans.cmdTrans = RM_SDWORD;
        }
        else 
        {
            if (mode==P9060_MODE_BYTE) trans.cmdTrans = WM_SBYTE;
            else if (mode==P9060_MODE_WORD) trans.cmdTrans = WM_SWORD;
            else trans.cmdTrans = WM_SDWORD;
        }
    }
    else 
    {
        if (fIsRead) 
        {
            if (mode==P9060_MODE_BYTE) trans.cmdTrans = RP_SBYTE;
            else if (mode==P9060_MODE_WORD) trans.cmdTrans = RP_SWORD;
            else trans.cmdTrans = RP_SDWORD;
        }
        else 
        {
            if (mode==P9060_MODE_BYTE) trans.cmdTrans = WP_SBYTE;
            else if (mode==P9060_MODE_WORD) trans.cmdTrans = WP_SWORD;
            else trans.cmdTrans = WP_SDWORD;
        }
    }
    trans.dwPort = dwAddr;
    trans.fAutoinc = TRUE;
    trans.dwBytes = dwBytes;
    trans.dwOptions = 0;
    trans.Data.pBuffer = buf;
    WD_Transfer (hPlx->hWD, &trans);
}

void P9060_ReadSpaceBlock (P9060_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, P9060_ADDR addrSpace, P9060_MODE mode)
{
    P9060_ReadWriteSpaceBlock (hPlx, dwOffset, buf, dwBytes, TRUE, addrSpace, mode);
}

void P9060_WriteSpaceBlock (P9060_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                     DWORD dwBytes, P9060_ADDR addrSpace, P9060_MODE mode)
{
    P9060_ReadWriteSpaceBlock (hPlx, dwOffset, buf, dwBytes, FALSE, addrSpace, mode);
}

void P9060_SetMode (P9060_HANDLE hPlx, P9060_MODE mode, DWORD dwLocalAddr)
{
    P9060_WriteReg (hPlx, P9060_REGION_DESC, hPlx->modeDesc[mode]);
    hPlx->addrDesc[hPlx->addrSpace].dwLocalBase = dwLocalAddr & ~hPlx->addrDesc[hPlx->addrSpace].dwMask;
    hPlx->addrDesc[hPlx->addrSpace].dwLocalBase |= BIT0;
    P9060_WriteReg (hPlx, P9060_REMAP_PTOL_0, hPlx->addrDesc[hPlx->addrSpace].dwLocalBase);
}

BYTE P9060_ReadByte (P9060_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9060_SetMode (hPlx, P9060_MODE_BYTE, dwLocalAddr);
    return P9060_ReadSpaceByte(hPlx, hPlx->addrSpace, dwOffset);
}

void P9060_WriteByte (P9060_HANDLE hPlx, DWORD dwLocalAddr, BYTE data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9060_SetMode (hPlx, P9060_MODE_BYTE, dwLocalAddr);
    P9060_WriteSpaceByte(hPlx, hPlx->addrSpace, dwOffset, data);
}

WORD P9060_ReadWord (P9060_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9060_SetMode (hPlx, P9060_MODE_WORD, dwLocalAddr);
    return P9060_ReadSpaceWord(hPlx, hPlx->addrSpace, dwOffset);
}

void P9060_WriteWord (P9060_HANDLE hPlx, DWORD dwLocalAddr, WORD data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9060_SetMode (hPlx, P9060_MODE_WORD, dwLocalAddr);
    P9060_WriteSpaceWord(hPlx, hPlx->addrSpace, dwOffset, data);
}

DWORD P9060_ReadDWord (P9060_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9060_SetMode (hPlx, P9060_MODE_DWORD, dwLocalAddr);
    return P9060_ReadSpaceDWord(hPlx, hPlx->addrSpace, dwOffset);
}

void P9060_WriteDWord (P9060_HANDLE hPlx, DWORD dwLocalAddr, DWORD data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9060_SetMode (hPlx, P9060_MODE_DWORD, dwLocalAddr);
    P9060_WriteSpaceDWord(hPlx, hPlx->addrSpace, dwOffset, data);
}

void P9060_ReadWriteBlock (P9060_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, P9060_MODE mode)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9060_SetMode (hPlx, mode, dwLocalAddr);
    P9060_ReadWriteSpaceBlock(hPlx, dwOffset, buf, dwBytes, fIsRead, hPlx->addrSpace, mode);
}

void P9060_ReadBlock (P9060_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                    DWORD dwBytes, P9060_MODE mode)
{
    P9060_ReadWriteBlock (hPlx, dwLocalAddr, buf, dwBytes, TRUE, mode);
}

void P9060_WriteBlock (P9060_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                     DWORD dwBytes, P9060_MODE mode)
{
    P9060_ReadWriteBlock (hPlx, dwLocalAddr, buf, dwBytes, FALSE, mode);
}

BOOL P9060_IntIsEnabled (P9060_HANDLE hPlx)
{
    if (!hPlx->fUseInt) return FALSE;
    if (!hPlx->Int.hThread) return FALSE;
    return TRUE;
}

VOID P9060_IntHandler (PVOID pData)
{
    P9060_HANDLE hPlx = (P9060_HANDLE) pData;
    P9060_INT_RESULT intResult;

    intResult.dwCounter = hPlx->Int.Int.dwCounter;
    intResult.dwLost = hPlx->Int.Int.dwLost;
    intResult.fStopped = hPlx->Int.Int.fStopped;
    intResult.dwStatusReg = hPlx->Int.Trans[0].Data.Dword;
    hPlx->Int.funcIntHandler(hPlx, &intResult);
}

BOOL P9060_IntEnable (P9060_HANDLE hPlx, P9060_INT_HANDLER funcIntHandler)
{
    DWORD dwIntStatus;
    DWORD dwAddr;

    if (!hPlx->fUseInt) return FALSE;
    // check if interrupt is already enabled
    if (hPlx->Int.hThread) return FALSE;

    dwIntStatus = P9060_ReadReg (hPlx, P9060_INT_STATUS);

    BZERO(hPlx->Int.Trans);
    // This is a samlpe of handling interrupts:
    // Two transfer commands are issued. First the value of the interrrupt control/status
    // register is read. Then, a value of ZERO is written.
    // This will cancel interrupts after the first interrupt occurs.
    // When using interrupts, this section will have to change:
    // you must put transfer commands to CANCEL the source of the interrupt, otherwise, the 
    // PC will hang when an interrupt occurs!
    dwAddr = hPlx->addrDesc[P9060_ADDR_REG].dwAddr + P9060_INT_STATUS;
    hPlx->Int.Trans[0].cmdTrans = hPlx->addrDesc[P9060_ADDR_REG].fIsMemory ? RM_DWORD : RP_DWORD;
    hPlx->Int.Trans[0].dwPort = dwAddr;
    hPlx->Int.Trans[1].cmdTrans = hPlx->addrDesc[P9060_ADDR_REG].fIsMemory ? WM_DWORD : WP_DWORD;
    hPlx->Int.Trans[1].dwPort = dwAddr;
    hPlx->Int.Trans[1].Data.Dword = dwIntStatus & ~(BIT8|BIT10); // put here the data to write to the control register
    hPlx->Int.Int.dwCmds = 2; 
    hPlx->Int.Int.Cmd = hPlx->Int.Trans;
    hPlx->Int.Int.dwOptions |= INTERRUPT_CMD_COPY;

    // create interrupt handler thread
    hPlx->Int.funcIntHandler = funcIntHandler;
    if (!InterruptThreadEnable(&hPlx->Int.hThread, hPlx->hWD, &hPlx->Int.Int, P9060_IntHandler, (PVOID) hPlx))
        return FALSE;

    // this enables interrupts
    P9060_WriteReg (hPlx, P9060_INT_STATUS, dwIntStatus | (BIT8|BIT10));

    return TRUE;
}

void P9060_IntDisable (P9060_HANDLE hPlx)
{
    DWORD dwIntStatus;

    if (!hPlx->fUseInt) return;
    if (!hPlx->Int.hThread) return;

    // this disables interrupts
    dwIntStatus = P9060_ReadReg (hPlx, P9060_INT_STATUS);
    P9060_WriteReg (hPlx, P9060_INT_STATUS, dwIntStatus & ~(BIT8|BIT10));

    // this calls WD_IntDisable()
    InterruptThreadDisable(hPlx->Int.hThread);
    hPlx->Int.hThread = NULL;
}

