////////////////////////////////////////////////////////////////
// File - AMCCLIB.C
//
// Library for 'WinDriver for AMCC 5933' API. 
// The basic idea is to get a handle for the board
// with AMCC_Open() and use it in the rest of the program
// when calling WD functions.  Call AMCC_Close() when done.
// 
////////////////////////////////////////////////////////////////

#include "../../include/windrvr.h"
#include "../../include/windrvr_int_thread.h"
#include "amcclib.h"
#include <stdio.h>
#if !defined(WINCE)
    #include <time.h>
#else
    extern time_t time();
#endif

// this string is set to an error message, if one occurs
CHAR AMCC_ErrorString[1024];

// internal function used by AMCC_Open()
BOOL AMCC_DetectCardElements(AMCCHANDLE hAmcc);

DWORD AMCC_CountCards (DWORD dwVendorID, DWORD dwDeviceID)
{
    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    HANDLE hWD;

    AMCC_ErrorString[0] = '\0';
    hWD = WD_Open();
    // check if handle valid & version OK
    if (hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( AMCC_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        return 0;
    }

    BZERO(ver);
    WD_Version(hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( AMCC_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        WD_Close (hWD);
        return 0;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hWD, &pciScan);
    WD_Close (hWD);
    if (pciScan.dwCards==0)
        sprintf( AMCC_ErrorString, "no cards found\n");
    return pciScan.dwCards;
}

BOOL AMCC_Open (AMCCHANDLE *phAmcc, DWORD dwVendorID, DWORD dwDeviceID, DWORD nCardNum, DWORD dwOptions)
{
    AMCCHANDLE hAmcc = (AMCCHANDLE) malloc (sizeof (AMCC_STRUCT));

    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    WD_PCI_CARD_INFO pciCardInfo;

    *phAmcc = NULL;
    AMCC_ErrorString[0] = '\0';
    BZERO(*hAmcc);

    hAmcc->cardReg.hCard = 0;
    hAmcc->hWD = WD_Open();

    // check if handle valid & version OK
    if (hAmcc->hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( AMCC_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        goto Exit;
    }

    BZERO(ver);
    WD_Version(hAmcc->hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( AMCC_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        goto Exit;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hAmcc->hWD, &pciScan);
    if (pciScan.dwCards==0) // Found at least one card
    {
        sprintf( AMCC_ErrorString, "Could not find PCI card\n");
        goto Exit;
    }
    if (pciScan.dwCards<=nCardNum)
    {
        sprintf( AMCC_ErrorString, "Card out of range of available cards\n");
        goto Exit;
    }

    BZERO(pciCardInfo);
    pciCardInfo.pciSlot = pciScan.cardSlot[nCardNum];
    WD_PciGetCardInfo (hAmcc->hWD, &pciCardInfo);
    hAmcc->pciSlot = pciCardInfo.pciSlot;
    hAmcc->cardReg.Card = pciCardInfo.Card;

    hAmcc->fUseInt = (dwOptions & AMCC_OPEN_USE_INT) ? TRUE : FALSE;
    if (!hAmcc->fUseInt)
    {
        DWORD i;
        // Remove interrupt item if not needed
        for (i=0; i<hAmcc->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hAmcc->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->item = ITEM_NONE;
        }
    }
    else
    {
        DWORD i;
        // make interrupt resource sharable
        for (i=0; i<hAmcc->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hAmcc->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->fNotSharable = FALSE;
        }
    }

    hAmcc->cardReg.fCheckLockOnly = FALSE;
    WD_CardRegister (hAmcc->hWD, &hAmcc->cardReg);
    if (hAmcc->cardReg.hCard==0)
    {
        sprintf ( AMCC_ErrorString, "Failed locking device\n");
        goto Exit;
    }

    if (!AMCC_DetectCardElements(hAmcc))
    {
        sprintf ( AMCC_ErrorString, "Card does not have all items expected for AMCC\n");
        goto Exit;
    }

    // Open finished OK
    *phAmcc = hAmcc;
    return TRUE;

Exit:
    // Error durin Open
    if (hAmcc->cardReg.hCard) 
        WD_CardUnregister(hAmcc->hWD, &hAmcc->cardReg);
    if (hAmcc->hWD!=INVALID_HANDLE_VALUE)
        WD_Close(hAmcc->hWD);
    free (hAmcc);
    return FALSE;
}

DWORD AMCC_ReadPCIReg(AMCCHANDLE hAmcc, DWORD dwReg)
{
    WD_PCI_CONFIG_DUMP pciCnf;
    DWORD dwVal;

    BZERO(pciCnf);
    pciCnf.pciSlot = hAmcc->pciSlot;
    pciCnf.pBuffer = &dwVal;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = TRUE;
    WD_PciConfigDump(hAmcc->hWD,&pciCnf);
    return dwVal;
}

void AMCC_WritePCIReg(AMCCHANDLE hAmcc, DWORD dwReg, DWORD dwData)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO (pciCnf);
    pciCnf.pciSlot = hAmcc->pciSlot;
    pciCnf.pBuffer = &dwData;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = FALSE;
    WD_PciConfigDump(hAmcc->hWD,&pciCnf);
}

BOOL AMCC_DetectCardElements(AMCCHANDLE hAmcc)
{
    DWORD i;
    DWORD ad_sp;

    BZERO(hAmcc->Int);
    BZERO(hAmcc->addrDesc);

    for (i=0; i<hAmcc->cardReg.Card.dwItems; i++)
    {
        WD_ITEMS *pItem = &hAmcc->cardReg.Card.Item[i];

        switch (pItem->item)
        {
        case ITEM_MEMORY:
        case ITEM_IO:
            {
                DWORD dwBytes;
                DWORD dwAddr;
                DWORD dwPhysAddr;
                DWORD dwAddrDirect = 0;
                BOOL fIsMemory;
                if (pItem->item==ITEM_MEMORY)
                {
                    dwBytes = pItem->I.Mem.dwBytes;
                    dwAddr = pItem->I.Mem.dwTransAddr;
                    dwAddrDirect = pItem->I.Mem.dwUserDirectAddr;
                    dwPhysAddr = pItem->I.Mem.dwPhysicalAddr;
                    fIsMemory = TRUE;
                }
                else 
                {
                    dwBytes = pItem->I.IO.dwBytes;
                    dwAddr = pItem->I.IO.dwAddr;
                    dwPhysAddr = dwAddr;
                    fIsMemory = FALSE;
                }

                for (ad_sp=AMCC_ADDR_REG; ad_sp<=AMCC_ADDR_NOT_USED; ad_sp++)
                {
                    DWORD dwPCIAddr;

                    if (hAmcc->addrDesc[ad_sp].dwAddr) continue;
                    dwPCIAddr = AMCC_ReadPCIReg(hAmcc, PCI_BAR0 + ad_sp*4);
                    if (dwPCIAddr & 1)
                    {
                        if (fIsMemory) continue;
                        dwPCIAddr &= ~0x3;
                    }
                    else
                    {
                        if (!fIsMemory) continue;
                        dwPCIAddr &= ~0xf;
                    }
                    if (dwPCIAddr==dwPhysAddr)
                        break;
                }
                if (ad_sp<=AMCC_ADDR_NOT_USED)
                {
                    DWORD j;
                    hAmcc->addrDesc[ad_sp].dwBytes = dwBytes;
                    hAmcc->addrDesc[ad_sp].dwAddr = dwAddr;
                    hAmcc->addrDesc[ad_sp].dwAddrDirect = dwAddrDirect;
                    hAmcc->addrDesc[ad_sp].fIsMemory = fIsMemory;
                    hAmcc->addrDesc[ad_sp].dwMask = 0;
                    for (j=1; j<hAmcc->addrDesc[ad_sp].dwBytes && j!=0x80000000; j *= 2)
                    {
                        hAmcc->addrDesc[ad_sp].dwMask = 
                            (hAmcc->addrDesc[ad_sp].dwMask << 1) | 1;
                    }
                }
            }
            break;
        case ITEM_INTERRUPT:
            if (hAmcc->Int.Int.hInterrupt) return FALSE;
            hAmcc->Int.Int.hInterrupt = pItem->I.Int.hInterrupt;
            break;
        }
    }

    // check that all the items needed were found
    // check if interrupt found
    if (hAmcc->fUseInt && !hAmcc->Int.Int.hInterrupt) 
    {
        return FALSE;
    }

    // check that the registers space was found
    if (!AMCC_IsAddrSpaceActive(hAmcc, AMCC_ADDR_REG))
        return FALSE;

    // check that at least one memory space was found
    //for (i = AMCC_ADDR_SPACE0; i<=AMCC_ADDR_NOT_USED; i++)
    //    if (AMCC_IsAddrSpaceActive(hAmcc, i)) break;
    //if (i>AMCC_ADDR_NOT_USED) return FALSE;

    return TRUE;
}

void AMCC_Close(AMCCHANDLE hAmcc)
{
    // disable interrupts
    if (AMCC_IntIsEnabled(hAmcc))
        AMCC_IntDisable(hAmcc);

    // unregister card
    if (hAmcc->cardReg.hCard) 
        WD_CardUnregister(hAmcc->hWD, &hAmcc->cardReg);

    // close WinDriver
    WD_Close(hAmcc->hWD);

    free (hAmcc);
}

BOOL AMCC_IsAddrSpaceActive(AMCCHANDLE hAmcc, AMCC_ADDR addrSpace)
{
    return hAmcc->addrDesc[addrSpace].dwAddr!=0;
}

void AMCC_WriteRegDWord (AMCCHANDLE hAmcc, DWORD dwReg, DWORD data)
{
    AMCC_WriteDWord (hAmcc, AMCC_ADDR_REG, dwReg, data);
}

DWORD AMCC_ReadRegDWord (AMCCHANDLE hAmcc, DWORD dwReg)
{
    return AMCC_ReadDWord (hAmcc, AMCC_ADDR_REG, dwReg);
}

void AMCC_WriteRegWord (AMCCHANDLE hAmcc, DWORD dwReg, WORD data)
{
    AMCC_WriteWord (hAmcc, AMCC_ADDR_REG, dwReg, data);
}

WORD AMCC_ReadRegWord (AMCCHANDLE hAmcc, DWORD dwReg)
{
    return AMCC_ReadWord (hAmcc, AMCC_ADDR_REG, dwReg);
}

void AMCC_WriteRegByte (AMCCHANDLE hAmcc, DWORD dwReg, BYTE data)
{
    AMCC_WriteByte (hAmcc, AMCC_ADDR_REG, dwReg, data);
}

BYTE AMCC_ReadRegByte (AMCCHANDLE hAmcc, DWORD dwReg)
{
    return AMCC_ReadByte (hAmcc, AMCC_ADDR_REG, dwReg);
}

// performs a single 32 bit write from address space
void AMCC_WriteDWord(AMCCHANDLE hAmcc, AMCC_ADDR addrSpace, DWORD dwLocalAddr, DWORD data)
{
    if (hAmcc->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddrDirect + dwLocalAddr;
        DWORD *pDword = (DWORD *) dwAddr;
        *pDword = data;
    }
    else
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddr + dwLocalAddr;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_DWORD;
        trans.dwPort = dwAddr;
        trans.Data.Dword = data;
        WD_Transfer (hAmcc->hWD, &trans);
    }
}

// performs a single 32 bit read from address space
DWORD AMCC_ReadDWord(AMCCHANDLE hAmcc, AMCC_ADDR addrSpace, DWORD dwLocalAddr)
{
    if (hAmcc->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddrDirect + dwLocalAddr;
        DWORD *pDword = (DWORD *) dwAddr;
        return *pDword;
    }
    else
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddr + dwLocalAddr;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_DWORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hAmcc->hWD, &trans);
        return trans.Data.Dword;
    }
}

// performs a single 16 bit write from address space
void AMCC_WriteWord(AMCCHANDLE hAmcc, AMCC_ADDR addrSpace, DWORD dwLocalAddr, WORD data)
{
    if (hAmcc->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddrDirect + dwLocalAddr;
        WORD *pWord = (WORD *) dwAddr;
        *pWord = data;
    }
    else
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddr + dwLocalAddr;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_WORD;
        trans.dwPort = dwAddr;
        trans.Data.Word = data;
        WD_Transfer (hAmcc->hWD, &trans);
    }
}

// performs a single 16 bit read from address space
WORD AMCC_ReadWord(AMCCHANDLE hAmcc, AMCC_ADDR addrSpace, DWORD dwLocalAddr)
{
    if (hAmcc->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddrDirect + dwLocalAddr;
        WORD *pWord = (WORD *) dwAddr;
        return *pWord;
    }
    else
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddr + dwLocalAddr;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_WORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hAmcc->hWD, &trans);
        return trans.Data.Word;
    }
}

// performs a single 8 bit write from address space
void AMCC_WriteByte(AMCCHANDLE hAmcc, AMCC_ADDR addrSpace, DWORD dwLocalAddr, BYTE data)
{
    if (hAmcc->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddrDirect + dwLocalAddr;
        BYTE *pByte = (BYTE *) dwAddr;
        *pByte = data;
    }
    else
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddr + dwLocalAddr;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_BYTE;
        trans.dwPort = dwAddr;
        trans.Data.Byte = data;
        WD_Transfer (hAmcc->hWD, &trans);
    }
}

// performs a single 8 bit read from address space
BYTE AMCC_ReadByte(AMCCHANDLE hAmcc, AMCC_ADDR addrSpace, DWORD dwLocalAddr)
{
    if (hAmcc->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddrDirect + dwLocalAddr;
        BYTE *pByte = (BYTE *) dwAddr;
        return *pByte;
    }
    else
    {
        DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddr + dwLocalAddr;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_BYTE;
        trans.dwPort = dwAddr;
        WD_Transfer (hAmcc->hWD, &trans);
        return trans.Data.Byte;
    }
}

void AMCC_ReadWriteSpaceBlock (AMCCHANDLE hAmcc, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, AMCC_ADDR addrSpace, AMCC_MODE mode)
{
    WD_TRANSFER trans;
    DWORD dwAddr = hAmcc->addrDesc[addrSpace].dwAddr +
        (hAmcc->addrDesc[addrSpace].dwMask & dwOffset);

    BZERO(trans);
    
    if (hAmcc->addrDesc[addrSpace].fIsMemory) 
    {
        if (fIsRead) 
        {
            if (mode==AMCC_MODE_BYTE) trans.cmdTrans = RM_SBYTE;
            else if (mode==AMCC_MODE_WORD) trans.cmdTrans = RM_SWORD;
            else trans.cmdTrans = RM_SDWORD;
        }
        else 
        {
            if (mode==AMCC_MODE_BYTE) trans.cmdTrans = WM_SBYTE;
            else if (mode==AMCC_MODE_WORD) trans.cmdTrans = WM_SWORD;
            else trans.cmdTrans = WM_SDWORD;
        }
    }
    else 
    {
        if (fIsRead) 
        {
            if (mode==AMCC_MODE_BYTE) trans.cmdTrans = RP_SBYTE;
            else if (mode==AMCC_MODE_WORD) trans.cmdTrans = RP_SWORD;
            else trans.cmdTrans = RP_SDWORD;
        }
        else 
        {
            if (mode==AMCC_MODE_BYTE) trans.cmdTrans = WP_SBYTE;
            else if (mode==AMCC_MODE_WORD) trans.cmdTrans = WP_SWORD;
            else trans.cmdTrans = WP_SDWORD;
        }
    }
    trans.dwPort = dwAddr;
    trans.fAutoinc = TRUE;
    trans.dwBytes = dwBytes;
    trans.dwOptions = 0;
    trans.Data.pBuffer = buf;
    WD_Transfer (hAmcc->hWD, &trans);
}

void AMCC_ReadSpaceBlock (AMCCHANDLE hAmcc, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, AMCC_ADDR addrSpace)
{
    AMCC_ReadWriteSpaceBlock (hAmcc, dwOffset, buf, dwBytes, TRUE, addrSpace, AMCC_MODE_DWORD);
}

void AMCC_WriteSpaceBlock (AMCCHANDLE hAmcc, DWORD dwOffset, PVOID buf, 
                     DWORD dwBytes, AMCC_ADDR addrSpace)
{
    AMCC_ReadWriteSpaceBlock (hAmcc, dwOffset, buf, dwBytes, FALSE, addrSpace, AMCC_MODE_DWORD);
}

//////////////////////////////////////////////////////////////////////////////
// Interrupts
//////////////////////////////////////////////////////////////////////////////

BOOL AMCC_IntIsEnabled (AMCCHANDLE hAmcc)
{
    if (!hAmcc->fUseInt) return FALSE;
    if (!hAmcc->Int.hThread) return FALSE;
    return TRUE;
}

VOID AMCC_IntHandler (PVOID pData)
{
    AMCCHANDLE hAmcc = (AMCCHANDLE) pData;
    AMCC_INT_RESULT intResult;

    intResult.dwCounter = hAmcc->Int.Int.dwCounter;
    intResult.dwLost = hAmcc->Int.Int.dwLost;
    intResult.fStopped = hAmcc->Int.Int.fStopped;
    intResult.dwStatusReg = hAmcc->Int.Trans[0].Data.Dword;
    hAmcc->Int.funcIntHandler(hAmcc, &intResult);
}

BOOL AMCC_IntEnable (AMCCHANDLE hAmcc, AMCC_INT_HANDLER funcIntHandler)
{
    DWORD dwAddr;

    if (!hAmcc->fUseInt) return FALSE;
    // check if interrupt is already enabled
    if (hAmcc->Int.hThread) return FALSE;

    BZERO(hAmcc->Int.Trans);
    // This is a samlpe of handling interrupts:
    // Two transfer commands are issued. First the value of the interrrupt control/status
    // register is read. Then, a value of ZERO is written.
    // This will cancel interrupts after the first interrupt occurs.
    // When using interrupts, this section will have to change:
    // you must put transfer commands to CANCEL the source of the interrupt, otherwise, the 
    // PC will hang when an interrupt occurs!
    dwAddr = hAmcc->addrDesc[AMCC_ADDR_REG].dwAddr + INTCSR_ADDR;
    hAmcc->Int.Trans[0].cmdTrans = hAmcc->addrDesc[AMCC_ADDR_REG].fIsMemory ? RM_DWORD : RP_DWORD;
    hAmcc->Int.Trans[0].dwPort = dwAddr;
    hAmcc->Int.Trans[1].cmdTrans = hAmcc->addrDesc[AMCC_ADDR_REG].fIsMemory ? WM_DWORD : WP_DWORD;
    hAmcc->Int.Trans[1].dwPort = dwAddr;
    hAmcc->Int.Trans[1].Data.Dword = 0x8cc000; // put here the data to write to the control register
    hAmcc->Int.Int.dwCmds = 2; 
    hAmcc->Int.Int.Cmd = hAmcc->Int.Trans;
    hAmcc->Int.Int.dwOptions |= INTERRUPT_CMD_COPY;

    // this calls WD_IntEnable() and creates an interrupt handler thread
    hAmcc->Int.funcIntHandler = funcIntHandler;
    if (!InterruptThreadEnable(&hAmcc->Int.hThread, hAmcc->hWD, &hAmcc->Int.Int, AMCC_IntHandler, (PVOID) hAmcc))
        return FALSE;
    
    // add here code to physically enable interrupts,
    // by setting bits in the INTCSR_ADDR register

    return TRUE;
}

void AMCC_IntDisable (AMCCHANDLE hAmcc)
{
    if (!hAmcc->fUseInt) return;
    if (!hAmcc->Int.hThread) return;

    // add here code to physically disable interrupts,
    // by clearing bits in the INTCSR_ADDR register

    // this calls WD_IntDisable()
    InterruptThreadDisable(hAmcc->Int.hThread);

    hAmcc->Int.hThread = NULL;
}

//////////////////////////////////////////////////////////////////////////////
// NVRam
//////////////////////////////////////////////////////////////////////////////

BOOL AMCC_WaitForNotBusy(AMCCHANDLE hAmcc)
{
    BOOL fReady = FALSE;
    time_t timeStart = time(NULL);

    for (; !fReady; ) 
    {
        if ((AMCC_ReadRegByte(hAmcc, BMCSR_NVCMD_ADDR) & NVRAM_BUSY_BITS) != NVRAM_BUSY_BITS) 
        {
            fReady = TRUE;
        }
        else 
        {
            if ((time(NULL) - timeStart) > 1) /* More than 1 second? */
                break;
        }
    } 

    return fReady;
}

BOOL AMCC_ReadNVByte(AMCCHANDLE hAmcc, DWORD dwAddr, BYTE *pbData)
{
    if (dwAddr >= AMCC_NVRAM_SIZE) return FALSE;
    /* Access non-volatile memory */

    /* Wait for nvRAM not busy */
    if (!AMCC_WaitForNotBusy(hAmcc)) return FALSE;

    /* Load Low address */
    AMCC_WriteRegByte(hAmcc, BMCSR_NVCMD_ADDR, NVCMD_LOAD_LOW_BITS);
    AMCC_WriteRegByte(hAmcc, BMCSR_NVDATA_ADDR, (BYTE) (dwAddr & 0xff));

    /* Load High address */
    AMCC_WriteRegByte(hAmcc, BMCSR_NVCMD_ADDR, NVCMD_LOAD_HIGH_BITS);
    AMCC_WriteRegByte(hAmcc, BMCSR_NVDATA_ADDR, (BYTE) (dwAddr >> 8));

    /* Send Begin Read command */
    AMCC_WriteRegByte(hAmcc, BMCSR_NVCMD_ADDR, NVCMD_BEGIN_READ_BITS);

    /* Wait for nvRAM not busy */
    if (!AMCC_WaitForNotBusy(hAmcc)) return FALSE;

    /* Get data from nvRAM Data register */
    *pbData = AMCC_ReadRegByte(hAmcc, BMCSR_NVDATA_ADDR);

    return TRUE;
}

//////////////////////////////////////////////////////////////////////////////
// DMA
//////////////////////////////////////////////////////////////////////////////

BOOL AMCC_DMAOpen(AMCCHANDLE hAmcc, WD_DMA *pDMA, DWORD dwBytes)
{
    AMCC_ErrorString[0] = '\0';
    BZERO(*pDMA);
    pDMA->pUserAddr = NULL; // the kernel will allocate the buffer
    pDMA->dwBytes = dwBytes; // size of buffer to allocate
    pDMA->dwOptions = DMA_KERNEL_BUFFER_ALLOC; 
    WD_DMALock (hAmcc->hWD, pDMA);
    if (!pDMA->hDma) 
    {
        sprintf ( AMCC_ErrorString, "Failed allocating the buffer!\n");
        return FALSE;
    }

    return TRUE;
}

void AMCC_DMAClose(AMCCHANDLE hAmcc, WD_DMA *pDMA)
{
    WD_DMAUnlock (hAmcc->hWD, pDMA);
}

BOOL AMCC_DMAStart(AMCCHANDLE hAmcc, WD_DMA *pDMA, BOOL fRead,
    BOOL fBlocking, DWORD dwBytes, DWORD dwOffset)
{
    DWORD dwBMCSR;

    // Important note:
    // fRead - if TRUE, data moved from the AMCC-card to the PC memory 
    // fRead - if FALSE, data moved from the PC memory to the AMCC card
    // the terms used by AMCC are oposite! 
    // in AMCC terms - read operation is from PC memory to AMCC card
    AMCC_WriteRegDWord(hAmcc, fRead ? MWAR_ADDR : MRAR_ADDR, (DWORD) pDMA->Page[0].pPhysicalAddr + dwOffset);
    AMCC_WriteRegDWord(hAmcc, fRead ? MWTC_ADDR : MRTC_ADDR, dwBytes);
    dwBMCSR = AMCC_ReadRegDWord(hAmcc, BMCSR_ADDR);
    dwBMCSR |= fRead ? BIT10 : BIT14;
    AMCC_WriteRegDWord(hAmcc, BMCSR_ADDR, dwBMCSR);
    // if blocking then wait for transfer to complete
    if (fBlocking)
        while (!AMCC_DMAIsDone(hAmcc, fRead));

    return TRUE;
}

BOOL AMCC_DMAIsDone(AMCCHANDLE hAmcc, BOOL fRead)
{
    DWORD dwBIT = fRead ? BIT18 : BIT19;
    DWORD dwINTCSR = AMCC_ReadRegDWord(hAmcc, INTCSR_ADDR);
    if (dwINTCSR & dwBIT)
    {
        AMCC_WriteRegDWord(hAmcc, INTCSR_ADDR, dwBIT);
        return TRUE;
    }
    return FALSE;
}

