(********************************************************************************************
** PROGRAM     : wDNS
** VERSION     : 1.0.0
** DESCRIPTION : Simple DNS and Reverse DNS lookup application.
** AUTHOR      : Stuart King
** COPYRIGHT   : Copyright (c) Irie Tools, 2002. All Rights Reserved.
** NOTES       :
**    This sample program is distributed with Irie Pascal, and was written to provide
** an example of how to write Windows Sockets API programs with Irie Pascal. To make best
** use of this sample you should have a basic understanding of Pascal as well as a basic
** understanding of the Sockets API. An effort was made, while writing this program, to
** stick to the general Sockets API as much as possible, and to avoid the extensions
** added in the Windows Sockets API. However in two cases this was not possible. The
** Windows Sockets API, requires you to call function "WSAStartUp" before calling any
** other function, and to call the function "WSACleanUp" when you are finished.
**
**    The file "windows.inc" is a system include file, that is distributed with
** Irie Pascal. Including "windows.inc" is an easy way to access hundreds of
** functions, procedures, types, and constants defined in the Windows API. In almost all
** cases, the names of entities included by "windows.inc" are identical to the names of
** the corresponding entity in the Windows API. However there are a few unavoidable
** cases where different names are used. For example the Windows API was created for the C
** programming language, and defines the type "SOCKET" and the function "socket". Although
** "SOCKET" and "socket" differ only in case, they refer to different entities. This is not
** a problem in C which is case sensitive. However since Pascal is not case sensitive,
** "SOCKET" and "socket" are the same identifier and can not refer to different entities. So
** "windows.inc" uses "SOCKET" for the type and "createsocket" for the function.
**
**   Notice that "windows.inc" is included before the keyword "program" which normally
** marks the start of a program. Starting in version 2.5, Irie Pascal allows declarations
** to occur before the keyword "program". When this is done, the declarations are placed in
** the same invisible outermost scope, as the declarations for the built-in identifiers
** (e.g. writeln). As you probably know, if you declare an identifier your declaration will
** override any other declarations of that identifier in outer scopes. This means that the
** program below can use any declarations in "windows.inc" it wants to, and pretend that the
** other declarations don't exist. For this reason, it is recommended that you also include
** system include files before the keyword "program".
**********************************************************************************************)

(*$I windows.inc *)
program wDNS;

	procedure Shutdown; forward;

	//PURPOSE: Initializes the application
	procedure Initialize;
	var
		data : WSADATA;
		iRet : integer;
	begin (* Initialize *)
		iRet := WSAStartUp($0202, data);
		if iRet <> 0 then
			begin
				writeln('WSAStartUp call failed. Return Code=', iRet);
				Shutdown;
			end;
	end; (* Initialize *)

	//PURPOSE: Perform DNS and Reverse DNS lookups.
	//NOTES:
	//DNS lookups are performed using the Socket function 'gethostbyname'.
	//Reverse DNS lookups are performed using the Socket function 'gethostbyaddr'.
	procedure DNSLookUps;
	var
		strEntry : string;

		//PURPOSE: Checks whether a string value looks like an IP address.
		//PARAMETER(s):
		//    1. s - contains the string value to check.
		//RETURNS:
		//    TRUE - if the string looks like an IP address
		//    FALSE - if the string does not look like an IP address
		//NOTES:
		//    This function actually checks whether the string value contains
		//only numbers and periods (.).
		function IsIPAddress(s : string) : boolean;
		var
			blnRet : boolean;
			i : 0..maxint;
		begin (* IsIPAddress *)
			blnRet := true;
			for i := 1 to length(s) do
				if (not isdigit(s[i])) and (s[i]<>'.') then
					blnRet := false;
			IsIPAddress := blnRet;
		end; (* IsIPAddress *)

		//PURPOSE: Checks whether a string value is a synonym for quit.
		//PARAMETER(s):
		//    1. s - contains the string value to check.
		//RETURNS:
		//    TRUE - if the string is a synonym for quit
		//    FALSE - if the string is not a synonym for quit
		function IsQuitCommand(s : string) : boolean;
		begin (* IsQuitCommand *)
			s := lowercase(s);
			IsQuitCommand := (s='quit') or (s='stop') or (s='exit') or (s='bye') or (s='done')
		end; (* IsQuitCommand *)

		//PURPOSE: Displays the information that was obtained from a DNS or Reverse DNS lookup
		//PARAMETER(s):
		//    1. pHostEnt - a pointer to the sockets API HOSTENT structure that contains
		//       the result of the lookup.
		//NOTES:
		//    This procedure uses the Windows API functions 'lstrlen' and 'lstrcpy' to
		//manipulate strings in C format (i.e. null-terminated arrays of char). These
		//functions are used because the strings in the HOSTENT structure are in C format.
		//Since these functions are not described in the online help (because they are
		//Windows API functions not Irie Pascal functions) a brief description is given
		//below for persons not already familiar with these functions:
		// 'lstrlen' returns the length of the string (not counting the null terminator).
		// 'lstrcpy' copies the string pointed to by it's second argument into the memory
		//   pointed to by it's first argument (the null terminator is also copied).
		procedure DisplayHostEntInfo(pHostEnt : p_hostent);
		const
			MAX_ASCII_ADDRESS = 15;
		var
			AddrTemp, AddrAddr, AddrRet : address;
			iRet : integer;
			strAddress : cstring[MAX_ASCII_ADDRESS];
			strName : cstring;

			//PURPOSE: Returns the address pointed to be another address.
			//PARAMETER(s):
			//    1. a - an address pointing to another address
			//RETURNS:
			//    The address pointed to by the parameter 'a'.
			//NOTES:
			//Irie Pascal pointer variables contain virtual addresses that are only
			//meaningful to the Irie Pascal Run Time Engine. However when using the
			//Windows API you occassionally need to manipulate non-virtual addresses.
			//To make this easier Irie Pascal supports a new type called 'address'.
			//However you can't use ^ to dereference an address, because addresses
			//are like untyped pointers (i.e. the compiler does not know the type of
			//the value pointed to by the address, so it can not generate code to return
			//the value pointed to be the address). The address type is assignment
			//compatible with all pointer types and vice/versa, so the solution to this
			//problem is to assign the address to suitable pointer variable and then
			//dereference the pointer variable.
			//In the case of the function below the address is known to be pointing
			// to another address. So the address to be deferenced is assigned to
			// a pointer to an address, and then this pointer is dereferenced.
			function DereferenceAddress(a : address) : address;
			var
				p : ^address;
			begin
				p := a;
				DereferenceAddress := p^;
			end;

		begin (* DisplayHostEntInfo *)
			//pHostEnt^.h_name is the address of a null-terminated string containing
			//the name of the host.
			writeln('NAME:');
			if pHostEnt^.h_name=NULL then
				writeln('NOT FOUND')
			else if (lstrlen(pHostEnt^.h_name)+1) > sizeof(strName) then
				writeln('TOO LONG')
			else
				begin
					iRet := lstrcpy(addr(strName), pHostEnt^.h_name);
					writeln(strName);
				end;
			writeln;

			//pHostEnt^.h_aliases is the address of a null terminated array of addresses
			//of null-terminated strings containing alternative names for the host.
			writeln('ALIASES:');
			AddrAddr := pHostEnt^.h_aliases;
			if AddrAddr = NULL then
				writeln('None')
			else
				begin
					AddrTemp := DereferenceAddress(AddrAddr);
					while AddrTemp <> NULL do
						begin
							if lstrlen(AddrTemp) > 0 then
								begin
									if (lstrlen(addrTemp)+1) > sizeof(strName) then
										writeln('TOO LONG')
									else
										begin
											iRet := lstrcpy(addr(strName), AddrTemp);
											writeln(strName);
										end;
								end
							else
								writeln('EMPTY');

							AddrAddr := AddrAddr + sizeof(address);
							AddrTemp := DereferenceAddress(AddrAddr);
						end;
				end;
			writeln;

			if pHostEnt^.h_addrtype <> AF_INET then
				writeln('Invalid address type')
			else if pHostEnt^.h_length <> sizeof(address) then
				writeln('Invalid address length')
			else
				begin
					//pHostEnt^.h_addr_list is the address of a null terminated array of
					//addresses of IP addresses of the host.
					writeln('ADDRESSES:');
					AddrAddr := pHostEnt^.h_addr_list;
					if AddrAddr = NULL then
						writeln('None')
					else
						begin
							//Get the first element of the array
							AddrTemp := DereferenceAddress(AddrAddr);
							while AddrTemp <> NULL do
								begin
									//Dereference the current array element to get the
									//IP address.
									AddrTemp := DereferenceAddress(AddrTemp);
									//Convert the IP address from binary format to a human
									//readable format (like nnnn.nnnn.nnnn.nnn)
									AddrRet := inet_ntoa(AddrTemp);
									if (AddrRet=null) or ((lstrlen(AddrRet)+1)>sizeof(strAddress)) then
										writeln('[ERROR]')
									else
										begin
											iRet := lstrcpy(addr(strAddress), AddrRet);
											writeln(strAddress);
										end;

									AddrAddr := AddrAddr + sizeof(address);
									//Get the next element of the array
									AddrTemp := DereferenceAddress(AddrAddr);
								end;
						end;
					writeln;
				end;
		end; (* DisplayHostEntInfo *)

		//PURPOSE: Given the name of a host performs a DNS lookup
		//PARAMETER(s):
		//    1. sName - contains the name of the host
		procedure DNSLookUp(strName : string);
		var
			pHostEnt : p_hostent;
			cstrName : cstring;
		begin
			//Convert the name to a cstring since 'gethostbyname' expects a cstring
			cstrName := strName;
			pHostEnt := gethostbyname(addr(cstrName));
			if pHostEnt = nil then
				writeln('Can not find host ''', cstrName, '''')
			else
				DisplayHostEntInfo(pHostEnt);
		end;

		//PUPOSE: Given the IP address of a host perform a reverse DNS lookup.
		//    1. strIPAddress - contains the name of the host
		procedure ReverseDNSLookUp(strIPAddress : cstring);
		var
			IPAddress : in_addr;
			pHostEnt : p_hostent;
		begin
			IPAddress := inet_addr(addr(strIPAddress));
			if IPAddress = INADDR_NONE then
				writeln('Invalid IP address')
			else
				begin
					pHostEnt := gethostbyaddr(addr(IPAddress), sizeof(IPAddress), AF_INET);
					if pHostEnt = nil then
						writeln('Can not find address ''', hex(IPAddress), '''')
					else
						DisplayHostEntInfo(pHostEnt);
				end;
		end;

	begin (* DNSLookUps *)
		repeat
			write('Enter IP address or Host Name: ');
			readln(strEntry);
			strEntry := trim(strEntry);
			if not IsQuitCommand(strEntry) then
				begin
					if IsIPAddress(strEntry) then
						ReverseDNSLookUp(strEntry)
					else
						DNSLookUp(strEntry)
				end
		until IsQuitCommand(strEntry)
	end; (* DNSLookUps *)

	//PURPOSE: Shuts down the application.
	procedure Shutdown;
	var
		iRet : integer;
	begin (* Shutdown *)
		iRet := WSACleanUp;
		halt
	end; (* Shutdown *)

begin
	Initialize;
	DNSLookUps;
	Shutdown;
end.
