(********************************************************************************************
** PROGRAM     : TestIrieDB
** VERSION     : 1.0.0
** DESCRIPTION : Example database program.
** AUTHOR      : Stuart King
** COPYRIGHT   : Copyright (c) Irie Tools, 2003. All Rights Reserved.
** NOTES       :
**    This sample program is distributed with Irie Pascal, and is an example of how to
** write database programs using Irie Pascal.
** When this program is run under Windows it attempts to use a Data Source Name (DSN),
** called 'TestIrieDB' to connect to an ODBC database engine. As a result you should be
** familiar with DSN's and basic database administration, if you want to run this sample
** under Windows.
** When this program is run under Linux, FreeBSD, Solaris/x86, or Solaris/Sparc it attempts
** to connect to a MySQL database called 'TestIrieDB', and user 'test', with password 'testpwd'.
** The connection is made through the socket '/tmp/mysql.sock'. As a result you will need to
** create the database and setup the user with the appropriate permissions, if you want to run
** this sample under those operating systems.
** NOTE: The only ODBC or MySQL specific code in this program is the code that gets the
** connection string.
**********************************************************************************************)
program TestIrieDB(fOut);
const
	MAX_SQL_COMMAND = 2000;
type
	SQLCommandType = string[MAX_SQL_COMMAND];
var
	fOut : text;
	strSQLCommand : SQLCommandType;
	bConnectionOpen : boolean;
	objConn : connection;
	objRec : recordset;

	procedure ErrorMsg(msg : string); forward;
	procedure Done; forward;
	procedure DisplayErrors; forward;

	procedure Init;
	begin (* Init *)
		rewrite(fOut);
		strSQLCommand := '';
		bConnectionOpen := false;
		new(objConn);
		new(objRec);
	end; (* Init *)

	procedure OpenConnection;
	var
		strConnectionString : string;

		function GetODBCConnectionString : string;
		begin (* GetODBCConnectionString *)
			GetODBCConnectionString := 'ODBC;DSN=TestIrieDB';
		end; (* GetODBCConnectionString *)

		function GetMySQLConnectionString : string;
		begin (* GetMySQLConnectionString *)
			GetMySQLConnectionString := 'MYSQL;user="test";password="testpwd";database="TestIrieDB";socket="/tmp/mysql.sock"'
		end; (* GetMySQLConnectionString *)

	begin (* OpenConnection *)
		if supported(feature_odbc) then
			strConnectionString := GetODBCConnectionString
		else if supported(feature_mysql) then
			strConnectionString := GetMySQLConnectionString
		else
			ErrorMsg('Databases are not supported on this platform');
		writeln(fOut, 'Opening connection with: ', strConnectionString);
		traperrors(false);
		objConn.open(strConnectionString);
		traperrors(true);
		if getlasterror <> 0 then
			begin
				DisplayErrors;
				Done;
			end;
		bConnectionOpen := true
	end; (* OpenConnection *)

	procedure CloseConnection;
	begin (* CloseConnection *)
		if bConnectionOpen then
			begin
				objConn.close;
				bConnectionOpen := false
			end;
	end; (* CloseConnection *)

	procedure ExecuteSQLCommand(bIgnoreErrors : boolean);
	begin (* ExecuteSQL *)
		if not bConnectionOpen then
			ErrorMsg('Connection is not open');
		if strSQLCommand = '' then
			ErrorMsg('SQL Command is empty');
		writeln(fOut, 'EXECUTE ', strSQLCommand);
		traperrors(false);
		objConn.execute(strSQLCommand);
		traperrors(true);
		if (getlasterror <> 0) and not bIgnoreErrors then
			begin
				DisplayErrors;
				Done;
			end;
		strSQLCommand := '';
	end; (* ExecuteSQL *)

	procedure ExecuteSQLCommandAndIgnoreErrors;
	begin (* ExecuteSQLCommandAndIgnoreErrors *)
		ExecuteSQLCommand(true);
	end; (* ExecuteSQLCommandAndIgnoreErrors *)

	procedure ExecuteSQLCommandAndDetectErrors;
	begin (* ExecuteSQLCommandAndDetectErrors *)
		ExecuteSQLCommand(false);
	end; (* ExecuteSQLCommandAndDetectErrors *)

	//************************************************************************
	// PURPOSE: Puts the database into a known initial state by dropping all
	//          the tables changed by this program.
	// NOTE: Errors are ignored while dropping the tables because the
	//       tables being dropped might not exist.
	//************************************************************************
	procedure DropTables;
	begin (* DropTables *)
		strSQLCommand := 'DROP TABLE score';
		ExecuteSQLCommandAndIgnoreErrors;

		strSQLCommand := 'DROP TABLE subject';
		ExecuteSQLCommandAndIgnoreErrors;

		strSQLCommand := 'DROP TABLE student';
		ExecuteSQLCommandAndIgnoreErrors;

		writeln(fOut);
	end; (* DropTables *)

	procedure CreateTables;
	begin (* CreateTables *)
		strSQLCommand := 'CREATE TABLE Student (';
		strSQLCommand := strSQLCommand + 'Code INT NOT NULL,';
		strSQLCommand := strSQLCommand + 'Id CHAR(8) NOT NULL,';
		strSQLCommand := strSQLCommand + 'LastName CHAR(20),';
		strSQLCommand := strSQLCommand + 'FirstName CHAR(20),';
		strSQLCommand := strSQLCommand + 'Address1 CHAR(40),';
		strSQLCommand := strSQLCommand + 'Address2 CHAR(40),';
		strSQLCommand := strSQLCommand + 'Phone CHAR(12),';
		strSQLCommand := strSQLCommand + 'DateOfBirth DATETIME,';
		strSQLCommand := strSQLCommand + 'PRIMARY KEY (Code)';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;
		
		strSQLCommand := 'CREATE TABLE Subject (';
		strSQLCommand := strSQLCommand + 'Code INT NOT NULL,';
		strSQLCommand := strSQLCommand + 'Id CHAR(8) NOT NULL,';
		strSQLCommand := strSQLCommand + 'Name CHAR(40),';
		strSQLCommand := strSQLCommand + 'PRIMARY KEY (Code)';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;
		
		strSQLCommand := 'CREATE TABLE Score (';
		strSQLCommand := strSQLCommand + 'StudentCode INT NOT NULL,';
		strSQLCommand := strSQLCommand + 'SubjectCode INT NOT NULL,';
		strSQLCommand := strSQLCommand + 'SubjectScore NUMERIC(5,2),';
		strSQLCommand := strSQLCommand + 'PRIMARY KEY (StudentCode, SubjectCode)';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;		

		writeln(fOut);
	end; (* CreateTables *)

	procedure InsertData;
	begin (* InsertData *)
		strSQLCommand := 'INSERT INTO Student (';
		strSQLCommand := strSQLCommand + 'Code, Id, LastName, FirstName, Address1, Address2, Phone, DateOfBirth';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + "1,'STUD001','SMITH','JOHN','21 First Street','','123-4567','1970-10-1'";
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Student (';
		strSQLCommand := strSQLCommand + 'Code, Id, LastName, FirstName, Address1, Address2, Phone, DateOfBirth';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + "2,'STUD002','BROWN','ROBERT','876 Second Street','','999-1234','1962-7-16'";
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Student (';
		strSQLCommand := strSQLCommand + 'Code, Id, LastName, FirstName, Address1, Address2, Phone, DateOfBirth';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + "3,'STUD003','BROWN','MARY','7A Third Street','','786-3455','1975-4-7'";
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Subject (';
		strSQLCommand := strSQLCommand + 'Code, Id, Name';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + "1,'MATH','MATHEMATICS'";
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Subject (';
		strSQLCommand := strSQLCommand + 'Code, Id, Name';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + "2,'ENG','ENGLISH LANGUAGE'";
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Subject (';
		strSQLCommand := strSQLCommand + 'Code, Id, Name';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + "3,'LIT','ENGLISH LITERATURE'";
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Score (';
		strSQLCommand := strSQLCommand + 'StudentCode, SubjectCode, SubjectScore';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + '1,2,56.86';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Score (';
		strSQLCommand := strSQLCommand + 'StudentCode, SubjectCode, SubjectScore';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + '1,1,87.2';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Score (';
		strSQLCommand := strSQLCommand + 'StudentCode, SubjectCode, SubjectScore';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + '1,3,72';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Score (';
		strSQLCommand := strSQLCommand + 'StudentCode, SubjectCode, SubjectScore';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + '2,2,45.81';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Score (';
		strSQLCommand := strSQLCommand + 'StudentCode, SubjectCode, SubjectScore';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + '2,1,55.92';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		strSQLCommand := 'INSERT INTO Score (';
		strSQLCommand := strSQLCommand + 'StudentCode, SubjectCode, SubjectScore';
		strSQLCommand := strSQLCommand + ')';
		strSQLCommand := strSQLCommand + 'VALUES (';
		strSQLCommand := strSQLCommand + '3,2,89.54';
		strSQLCommand := strSQLCommand + ')';
		ExecuteSQLCommandAndDetectErrors;

		writeln(fOut);
	end; (* InsertData *)

	procedure QueryData;
	begin (* QueryData *)
		strSQLCommand := 'SELECT Code, Id, LastName, FirstName, DateOfBirth ';
		strSQLCommand := strSQLCommand + 'FROM Student ';
		strSQLCommand := strSQLCommand + 'ORDER BY Code';
		writeln(fOut, 'QUERY ', strSQLCommand);
		objRec.open(objConn, strSQLCommand, rsforward);
		while not objRec.eof do
			begin
				writeln(fOut, objRec.('Code'), ' ', objRec.('Id'), ' ', objRec.('LastName'), ' ', objRec.('FirstName'), ' ', objRec.('DateOfBirth'));
				objRec.movenext;
			end;
		objRec.close;

		writeln(fOut);

		strSQLCommand := 'SELECT Student.Id, LastName, FirstName, DateOfBirth, Subject.Name, Score.SubjectScore ';
		strSQLCommand := strSQLCommand + 'FROM Student, Subject, Score ';
		strSQLCommand := strSQLCommand + 'WHERE Student.Code=Score.StudentCode ';
		strSQLCommand := strSQLCommand + 'AND Score.SubjectCode=Subject.Code ';
		strSQLCommand := strSQLCommand + 'ORDER BY Student.Id, Score.SubjectScore';
		writeln(fOut, 'QUERY ', strSQLCommand);
		objRec.open(objConn, strSQLCommand, rsforward);
		while not objRec.eof do
			begin
				writeln(fOut, objRec.('Id'), ' ', objRec.('LastName'), ' ', objRec.('FirstName'), objRec.('DateOfBirth'), ' ', objRec.('Name'), ' ', objRec.('SubjectScore'));
				objRec.movenext;
			end;
		objRec.close
	end; (* QueryData *)

	procedure ErrorMsg;
	begin (* ErrorMsg *)
		writeln(fOut, 'ERROR: ', msg);
		Done;
	end; (* ErrorMsg *)

	procedure DisplayErrors;
	var
		iNumErrors, iCurrError : integer;

		procedure DisplayAnError(e : error);
		begin (* DisplayAnError *)
			with e do
				writeln(fOut, 'ERROR: ', number:1, ' ', name, ' ', description);
		end; (* DisplayAnError *)

	begin (* DisplayErrors *)
		iNumErrors := length(errors);
		for iCurrError := 1 to iNumErrors do
			DisplayAnError(errors[iCurrError])
	end; (* DisplayErrors *)

	procedure Done;
	begin
		if bConnectionOpen then
			CloseConnection;
		dispose(objRec);
		dispose(objConn);
		halt
	end;

begin
	Init;
	OpenConnection;
	DropTables;
	CreateTables;
	InsertData;
	QueryData;
	Done
end.
