/*******************************************************************************
* FILE NAME: istattxt.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in istattxt.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

#define INCL_GPIPRIMITIVES
#define INCL_WINERRORS             // PMERR_HEAP_MAX_SIZE_REACHED
#define INCL_WINSTATICS
#define INCL_WINSYS
#define INCL_WINWINDOWMGR
extern "C" {
  #include <iwindefs.h>
}

#include <istattxt.hpp>
#include <icconst.h>
#include <icmnfun.hpp>
#include <icolor.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <ihandle.hpp>
#include <inotifev.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <istathdr.hpp>
#include <istring.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ISTATTXT_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| Public static text styles.                                                   |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLStaticConst)
const IStaticText::Style
  IStaticText::left              ( 0, IDT_LEFT ),
  IStaticText::top               ( 0, IDT_TOP ),
  IStaticText::underscore        ( 0, IDT_UNDERSCORE ),
  IStaticText::strikeout         ( 0, IDT_STRIKEOUT ),
  IStaticText::border3D          ( 0, IWS_BORDER3D ),
#ifdef IC_MOTIFPM
  IStaticText::center            = DT_CENTER,
  IStaticText::right             = DT_RIGHT,
  IStaticText::vertCenter        = DT_VCENTER,
  IStaticText::bottom            = DT_BOTTOM,
  IStaticText::wordBreak         = DT_WORDBREAK,
  IStaticText::mnemonic          = DT_MNEMONIC,
  IStaticText::halftone          = DT_HALFTONE,
  IStaticText::fillBackground    = DT_ERASERECT,
  IStaticText::classDefaultStyle ( WS_VISIBLE | DT_ERASERECT,
                                   IDT_TOP | IDT_LEFT );
#endif
#ifdef IC_WIN
  IStaticText::center            ( 0, IDT_CENTER ),
  IStaticText::right             ( 0, IDT_RIGHT ),
  IStaticText::vertCenter        ( 0, IDT_VCENTER ),
  IStaticText::bottom            ( 0, IDT_BOTTOM ),
  IStaticText::wordBreak         ( 0, IDT_WORDBREAK ),
  IStaticText::mnemonic          ( 0, IDT_MNEMONIC ),
  IStaticText::halftone          ( 0, IDT_HALFTONE ),
  IStaticText::fillBackground    ( 0, IDT_ERASERECT ),
  IStaticText::classDefaultStyle ( WS_VISIBLE | IDT_MNEMONIC,
                                   IDT_TOP | IDT_LEFT | IDT_ERASERECT );
#endif

#pragma data_seg()

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLNonConst)
#ifdef IC_MOTIFPM
  IStaticText::Style
      IStaticText::currentDefaultStyle ( WS_VISIBLE | DT_ERASERECT,
                                         IDT_TOP | IDT_LEFT );
#endif
#ifdef IC_WIN
  IStaticText::Style
      IStaticText::currentDefaultStyle ( WS_VISIBLE,
                                         IDT_MNEMONIC |
                                         IDT_TOP | IDT_LEFT | IDT_ERASERECT );
#endif
#pragma data_seg()

class IStaticTextData : public IBase {
/*******************************************************************************
* The IStaticTextData class encapsulates private data and functions            *
* used by the IStaticText class.  An object of this class is created in the    *
* IStaticText constructors.  This is the os2/windows version.                  *
*******************************************************************************/
public:
  IStaticTextData ( );

virtual
 ~IStaticTextData ( );

IColor
  clrfill;
Boolean
  clrset;
};

/*------------------------------------------------------------------------------
| IStaticTextData::IStaticTextData                                             |
|                                                                              |
------------------------------------------------------------------------------*/
IStaticTextData::IStaticTextData ( )
  : clrset( false ), clrfill( IColor::paleGray )
{ }

/*------------------------------------------------------------------------------
| IStaticTextData::~IStaticTextData                                            |
|                                                                              |
------------------------------------------------------------------------------*/
IStaticTextData::~IStaticTextData ( )
{ }

/*------------------------------------------------------------------------------
| IStaticTextHandler                                                           |
|                                                                              |
| This class wrappers an object pointer and destroys the pointed               |
| to object when the wrapper is destructed.                                    |
------------------------------------------------------------------------------*/
static struct DefaultHandler {
  operator IStaticTextHandler* ( );

IStaticTextHandler
 *operator -> ( )
  {
    return( *this );
  }

 ~DefaultHandler ( )
  {
    if ( ptr )
    {
       delete ptr;
       ptr = 0;
    }
  }

IStaticTextHandler
 *ptr;

} defaultHandler;

DefaultHandler::operator IStaticTextHandler* ( )
{
  if ( !ptr )
     ptr = new IStaticTextHandler;

  return ptr;
}

/*------------------------------------------------------------------------------
| IStaticText::IStaticText                                                     |
|                                                                              |
| Constructor to create a static text control on a standard window.            |
------------------------------------------------------------------------------*/
IStaticText::IStaticText ( unsigned long id,
                           IWindow* parent,
                           IWindow* owner,
                           const IRectangle& initial,
                           const Style& style )
  : fStaticTextData( new IStaticTextData ),
    ulClLimit( 0 )
{
  // Assertions on input parms.
  IASSERTPARM( parent != 0 );

  // Save the extended style to make sure we have a copy of it stored.
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

  // Create static text control.
  IWindowHandle ownerHandle = IWindowHandle( 0 );
  if ( owner )
     ownerHandle = owner->handle();
  IWindowHandle whStaticText =
      this->create( id,
                    0,
                    convertToGUIStyle( style ),
                    WC_STATIC,
                    parent->handle(),
                    ownerHandle,
                    initial,
                    0,
                    0 );

  this->startHandlingEventsFor( whStaticText );
  this->addHandler( defaultHandler );
}


/*------------------------------------------------------------------------------
| IStaticText::IStaticText                                                     |
|                                                                              |
| Constructor to create an instance for template window.                       |
------------------------------------------------------------------------------*/
IStaticText::IStaticText ( unsigned long id,
                           IWindow* parent )
  : fStaticTextData( new IStaticTextData ),
    ulClLimit( 0 )
{
  setAutoDestroyWindow( false );
  this->startHandlingEventsFor( id, parent );
  this->addHandler( defaultHandler );

  /*********************************************************/
  /* Set the style without causing a refresh (i.e. don't   */
  /* call enableFillBackground).                           */
  /*********************************************************/
#ifdef IC_PM
  setStyle( style() | fillBackground.asUnsignedLong() );
#endif
#ifdef IC_WIN
  setExtendedStyle( extendedStyle() | fillBackground.asExtendedUnsignedLong() );
#endif
}

/*------------------------------------------------------------------------------
| IStaticText::IStaticText                                                     |
|                                                                              |
| Constructor to instantiate from an existing static text control.             |
------------------------------------------------------------------------------*/
IStaticText::IStaticText ( const IWindowHandle& handle )
  : fStaticTextData( new IStaticTextData ),
    ulClLimit( 0 )
{
  this->setAutoDestroyWindow( false );
  this->startHandlingEventsFor( handle );
  this->addHandler( defaultHandler );

  /*********************************************************/
  /* Set the style without causing a refresh (i.e. don't   */
  /* call enableFillBackground).                           */
  /*********************************************************/
#ifdef IC_PM
  setStyle( style() | fillBackground.asUnsignedLong() );
#endif
#ifdef IC_WIN
  setExtendedStyle( extendedStyle() | fillBackground.asExtendedUnsignedLong() );
#endif
}

/*------------------------------------------------------------------------------
| IStaticText::~IStaticText                                                    |
------------------------------------------------------------------------------*/
IStaticText::~IStaticText ( )
{
  this->removeHandler( defaultHandler );   // Remove the private handler.
  if ( fStaticTextData )
     delete fStaticTextData;
}

/*------------------------------------------------------------------------------
| IStaticText::defaultStyle                                                    |
|                                                                              |
| Return the default style for new static text objects.                        |
------------------------------------------------------------------------------*/
IStaticText::Style IStaticText::defaultStyle ( )
{
  return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
| IStaticText::setDefaultStyle                                                 |
|                                                                              |
| Set the default style for new static text objects.                           |
------------------------------------------------------------------------------*/
void IStaticText::setDefaultStyle ( const Style& style )
{
  currentDefaultStyle = style;
}

/*------------------------------------------------------------------------------
| IStaticText::convertToGUIStyle                                               |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (extendedOnly) is set.                                         |
------------------------------------------------------------------------------*/
unsigned long IStaticText::convertToGUIStyle ( const IBitFlag& style,
                                               Boolean extendedOnly ) const
{
  // Obtain the style from the class (ITextControl) that we inherit from.
  unsigned long ulStyle = Inherited::convertToGUIStyle( style, extendedOnly );

  if ( extendedOnly )
  {
    // Use mask to only return extended styles in the user defined range.
    ulStyle |= extendedStyle() & IS_EXTMASK;

  }
  else
  {
    // SS_ styles have a one-to-one correspondence to our style bits, and
    // inhabit the lower word of the base GUI style.  Therefore, obtain
    // that portion asis, masking out the upper word.
    ulStyle |= style.asUnsignedLong() & IDT_MASK;

#ifdef IC_PM
    ulStyle |= SS_TEXT;               // Add required style.
#endif
#ifdef IC_WIN
    ulStyle |= SS_LEFT | WS_CHILD;    // Add required style ...
#endif
  }

  return ulStyle;
}

/*------------------------------------------------------------------------------
| IStaticText::alignment                                                       |
------------------------------------------------------------------------------*/
IStaticText::Alignment IStaticText::alignment ( ) const
{
  Alignment result = topLeft;     // (Can't test for top or left.)
#ifndef IC_WIN
  unsigned long ulStyle = this->style();

  if ( ulStyle & right.asUnsignedLong() )
  {
     if ( ulStyle & bottom.asUnsignedLong() )
        result = bottomRight;

     else if ( ulStyle & vertCenter.asUnsignedLong() )
        result = centerRight;

     else     // (Can't test for IStaticText::top.)
        result = topRight;
  }

  else if ( ulStyle & center.asUnsignedLong() )
  {
     if ( ulStyle & bottom.asUnsignedLong() )
        result = bottomCenter;

     else if ( ulStyle & vertCenter.asUnsignedLong() )
        result = centerCenter;

     else     // (Can't test for IStaticText::top.)
        result = topCenter;
  }

  else if ( ulStyle & bottom.asUnsignedLong() )
     result = bottomLeft;

  else if ( ulStyle & vertCenter.asUnsignedLong() )
     result = centerLeft;

  else if ( ulStyle & wordBreak.asUnsignedLong() )
     result = topLeftWrapped;
#endif
#ifdef IC_WIN
  unsigned long ulStyle = this->extendedStyle();

  if ( ulStyle & right.asExtendedUnsignedLong() )
  {
     if ( ulStyle & bottom.asExtendedUnsignedLong() )
        result = bottomRight;

     else if ( ulStyle & vertCenter.asExtendedUnsignedLong() )
        result = centerRight;

     else     // (Can't test for IStaticText::top.)
        result = topRight;
  }

  else if ( ulStyle & center.asExtendedUnsignedLong() )
  {
     if ( ulStyle & bottom.asExtendedUnsignedLong() )
        result = bottomCenter;

     else if ( ulStyle & vertCenter.asExtendedUnsignedLong() )
        result = centerCenter;

     else     // (Can't test for IStaticText::top.)
        result = topCenter;
  }

  else if ( ulStyle & bottom.asExtendedUnsignedLong() )
     result = bottomLeft;

  else if ( ulStyle & vertCenter.asExtendedUnsignedLong() )
     result = centerLeft;

  else if ( ulStyle & wordBreak.asExtendedUnsignedLong() )
     result = topLeftWrapped;
#endif

  return result;
}

/*------------------------------------------------------------------------------
| IStaticText::setAlignment                                                    |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::setAlignment ( Alignment alignType )
{
#ifndef IC_WIN
  unsigned long ulStyles = this->style();
#endif
#ifdef IC_WIN
  unsigned long ulStyles = this->extendedStyle();
#endif
  unsigned long ulOldStyles = ulStyles;

  /********************************************************************/
  /* Mask off only the alignment flags ...                            */
  /********************************************************************/
#ifndef IC_WIN
  ulStyles &= ~( center.asUnsignedLong() | right.asUnsignedLong() |
                 bottom.asUnsignedLong() | vertCenter.asUnsignedLong() |
                 wordBreak.asUnsignedLong() );
#endif
#ifdef IC_WIN
  ulStyles &= ~( left.asExtendedUnsignedLong() |
                 top.asExtendedUnsignedLong() |
                 center.asExtendedUnsignedLong() |
                 right.asExtendedUnsignedLong() |
                 bottom.asExtendedUnsignedLong() |
                 vertCenter.asExtendedUnsignedLong() |
                 wordBreak.asExtendedUnsignedLong() );
#endif

  /********************************************************************/
  /* NOTE:  The flag definitions for top and left are 0 for both PM   */
  /*        and Windows.                                              */
  /********************************************************************/
  switch ( alignType )
  {                                    // Add new alignment.
    case topLeft:
#ifdef IC_WIN
      ulStyles |= left.asExtendedUnsignedLong() | top.asExtendedUnsignedLong();
#endif
      break;

    case topLeftWrapped:
#ifndef IC_WIN
      ulStyles |= wordBreak.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= wordBreak.asExtendedUnsignedLong();
#endif
      break;

    case topCenter:
#ifndef IC_WIN
      ulStyles |= center.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= center.asExtendedUnsignedLong();
#endif
      break;

    case topRight:
#ifndef IC_WIN
      ulStyles |= right.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= right.asExtendedUnsignedLong();
#endif
      break;

    case centerLeft:
#ifndef IC_WIN
      ulStyles |= vertCenter.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= vertCenter.asExtendedUnsignedLong();
#endif
      break;

    case centerCenter:
#ifndef IC_WIN
      ulStyles |= vertCenter.asUnsignedLong() | center.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= vertCenter.asExtendedUnsignedLong() | center.asExtendedUnsignedLong();
#endif
      break;

    case centerRight:
#ifndef IC_WIN
      ulStyles |= vertCenter.asUnsignedLong() | right.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= vertCenter.asExtendedUnsignedLong() | right.asExtendedUnsignedLong();
#endif
      break;

    case bottomLeft:
#ifndef IC_WIN
      ulStyles |= bottom.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= bottom.asExtendedUnsignedLong();
#endif
      break;

    case bottomCenter:
#ifndef IC_WIN
      ulStyles |= bottom.asUnsignedLong() | center.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= bottom.asExtendedUnsignedLong() | center.asExtendedUnsignedLong();
#endif
      break;

    case bottomRight:
#ifndef IC_WIN
      ulStyles |= bottom.asUnsignedLong() | right.asUnsignedLong();
#endif
#ifdef IC_WIN
      ulStyles |= bottom.asExtendedUnsignedLong() | right.asExtendedUnsignedLong();
#endif
      break;

    default:
      ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                          IErrorInfo::invalidParameter,
                          IException::recoverable );
      break;
  }

  if ( ulStyles != ulOldStyles )
  {
#ifndef IC_WIN
     this->setStyle( ulStyles );
#endif
#ifdef IC_WIN
     this->setExtendedStyle( ulStyles );
#endif
     this->refresh();
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::disableFillBackground                                           |
|                                                                              |
| Turn off fill background style.                                              |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::disableFillBackground ( )
{
  return this->enableFillBackground( false );
}

/*------------------------------------------------------------------------------
| IStaticText:: disableHalftone                                                |
|                                                                              |
| Turn off halftone style.                                                     |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::disableHalftone ( )
{
  return this->enableHalftone( false );
}

/*------------------------------------------------------------------------------
| IStaticText::enableFillBackground                                            |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::enableFillBackground ( Boolean enable )
{
#ifdef IC_MOTIFPM
  unsigned long ulStyles = this->style();
#endif
#ifdef IC_WIN
  unsigned long ulStyles = this->extendedStyle();
#endif
  unsigned long ulOldStyles = ulStyles;

#ifdef IC_MOTIFPM
  if ( enable )
  {
     ulStyles |= fillBackground.asUnsignedLong();
  }
  else
  {
     ulStyles &= ~fillBackground.asUnsignedLong();
  }
#endif
#ifdef IC_MOTIFPM
  if ( enable )
  {
     ulStyles |= fillBackground.asExtendedUnsignedLong();
  }
  else
  {
     ulStyles &= ~fillBackground.asExtendedUnsignedLong();
  }
#endif

  if ( ulStyles != ulOldStyles )
  {
#ifdef IC_MOTIFPM
     this->setStyle( ulStyles );
#endif
#ifdef IC_WIN
     this->setExtendedStyle( ulStyles );
#endif
     this->refresh();
     this->notify( IStaticText::fillBackgroundId, enable );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::hasFillBackground                                               |
------------------------------------------------------------------------------*/
Boolean IStaticText::hasFillBackground ( ) const
{
#ifdef IC_MOTIFPM
  return( (this->style() & fillBackground.asUnsignedLong()) ? true : false );
#endif
#ifdef IC_WIN
  return( (extendedStyle() & fillBackground.asExtendedUnsignedLong()) ? true
                                                                      : false );
#endif
}

/*------------------------------------------------------------------------------
| IStaticText::enableHalftone                                                  |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::enableHalftone ( Boolean enable )
{
#ifdef IC_MOTIFPM
  unsigned long ulStyles = this->style();
#endif
#ifdef IC_WIN
  unsigned long ulStyles = this->extendedStyle();
#endif
  unsigned long ulOldStyles = ulStyles;

#ifdef IC_MOTIFPM
  if ( enable )
  {
     ulStyles |= halftone.asUnsignedLong();
  }
  else
  {
     ulStyles &= ~halftone.asUnsignedLong();
  }
#endif
#ifdef IC_WIN
  if ( enable )
  {
     ulStyles |= halftone.asExtendedUnsignedLong();
  }
  else
  {
     ulStyles &= ~halftone.asExtendedUnsignedLong();
  }
#endif

  if ( ulStyles != ulOldStyles )
  {
#ifdef IC_MOTIFPM
     this->setStyle( ulStyles );
#endif
#ifdef IC_WIN
     this->setExtendedStyle( ulStyles );
#endif
     this->refresh();
     this->notify( IStaticText::halftoneId, enable );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::isHalftone                                                      |
------------------------------------------------------------------------------*/
Boolean IStaticText::isHalftone ( ) const
{
#ifdef IC_MOTIFPM
  return( (style() & halftone.asUnsignedLong()) ? true : false );
#endif
#ifdef IC_WIN
  return( (extendedStyle() & halftone.asExtendedUnsignedLong()) ? true
                                                                : false );
#endif
}

#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| IStaticText:: disableUnderscore                                              |
|                                                                              |
| Turn off underscore style.                                                   |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::disableUnderscore ( )
{
  return this->enableUnderscore( false );
}

/*------------------------------------------------------------------------------
| IStaticText:: disableStrikeout                                               |
|                                                                              |
| Turn off strikeout style.                                                    |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::disableStrikeout ( )
{
  return this->enableStrikeout( false );
}

/*------------------------------------------------------------------------------
| IStaticText::enableUnderscore                                                |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::enableUnderscore ( Boolean enable )
{
  unsigned long ulExtStyles = this->extendedStyle();
  unsigned long ulOldExtStyles = ulExtStyles;

  if ( enable )
  {
     ulExtStyles |= underscore.asExtendedUnsignedLong();
  }
  else
  {
     ulExtStyles &= ~underscore.asExtendedUnsignedLong();
  }

  if ( ulExtStyles != ulOldExtStyles )
  {
     this->setExtendedStyle( ulExtStyles );
     this->refresh();
     this->notify( IStaticText::underscoreId, enable );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::isUnderscore                                                    |
------------------------------------------------------------------------------*/
Boolean IStaticText::isUnderscore ( ) const
{
  return( (extendedStyle() & underscore.asExtendedUnsignedLong()) ? true
                                                                  : false );
}

/*------------------------------------------------------------------------------
| IStaticText::enableStrikeout                                                 |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::enableStrikeout ( Boolean enable )
{
  unsigned long ulExtStyles = this->extendedStyle();
  unsigned long ulOldExtStyles = ulExtStyles;

  if ( enable )
  {
     ulExtStyles |= strikeout.asExtendedUnsignedLong();
  }
  else
  {
     ulExtStyles &= ~strikeout.asExtendedUnsignedLong();
  }

  if ( ulExtStyles != ulOldExtStyles )
  {
     this->setExtendedStyle( ulExtStyles );
     this->refresh();
     this->notify( IStaticText::strikeoutId, enable );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::isStrikeout                                                     |
------------------------------------------------------------------------------*/
Boolean IStaticText::isStrikeout ( ) const
{
  return( (extendedStyle() & strikeout.asExtendedUnsignedLong()) ? true
                                                                 : false );
}
#endif //IC_MOTIFPM

/*------------------------------------------------------------------------------
| IStaticText::calcMinimumSize                                                 |
------------------------------------------------------------------------------*/
ISize IStaticText::calcMinimumSize ( ) const
{
  ISize sizNew = this->displaySize( text() );

  /**********************************************************/
  /* Calculate a new minimum size if the text limit is set. */
  /**********************************************************/
  if ( this->ulClLimit )
  {
     ISize sizLimit = this->calcLimitSize();
     if ( sizLimit.height() > sizNew.height() )
        sizNew.setHeight( sizLimit.height() );   // Grow the height.

     if ( sizLimit.width() > sizNew.width() )
        sizNew.setWidth( sizLimit.width() );     // Grow the width.
  }

  return sizNew;
}

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IStaticText::foregroundColor                                                 |
|                                                                              |
| Returns the foreground color for the IStaticText control.                    |
------------------------------------------------------------------------------*/
IColor IStaticText::foregroundColor ( ) const
{
  return IWindow::color( PP_FOREGROUNDCOLOR,
                         IGUIColor( IGUIColor::windowStaticText ));
}

/*------------------------------------------------------------------------------
| IStaticText::backgroundColor                                                 |
|                                                                              |
| Returns the background color for the IStaticText control.                    |
------------------------------------------------------------------------------*/
IColor IStaticText::backgroundColor ( ) const
{
  return IWindow::color( PP_BACKGROUNDCOLOR,
                         defaultbackgroundcolor( this->handle() ));
}
#endif //IC_PM

/*------------------------------------------------------------------------------
| IStaticText::fillColor                                                       |
|                                                                              |
| Returns the fillColor for the IStaticText control.                           |
------------------------------------------------------------------------------*/
IColor IStaticText::fillColor ( ) const
{
  if ( this->fStaticTextData->clrset )
     return this->fStaticTextData->clrfill;

  return this->backgroundColor();
}

/*------------------------------------------------------------------------------
| IStaticText::setFillColor                                                    |
|                                                                              |
| Sets the fill color for the IStaticText control.                             |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::setFillColor ( const IColor &color )
{
  if ( this->fStaticTextData->clrfill != color )
  {
     this->fStaticTextData->clrset  = true;
     this->fStaticTextData->clrfill = color;
     this->refresh( );
     this->notify( IStaticText::fillColorId,
                   false );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::resetFillColor                                                  |
|                                                                              |
| Resets the fill color of the IStaticText control.                            |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::resetFillColor ( )
{
  if ( this->fStaticTextData->clrset )
  {
     this->fStaticTextData->clrset  = false;
     this->fStaticTextData->clrfill = this->fillColor( );
     this->refresh( );
     this->notify( IStaticText::fillColorId,
                   false );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::notify                                                          |
|                                                                              |
| Notifies observers of an event.                                              |
------------------------------------------------------------------------------*/
void IStaticText::notify ( INotificationId nId, Boolean enabled )
{
  if ( nId == IStaticText::fillColorId )
  {
     this->notifyObservers( INotificationEvent( nId, *this ));
  }
  else if ( nId == IStaticText::limitId )
  {
     unsigned long limit = this->limit();
     this->notifyObservers( INotificationEvent( nId, *this, true,
                                                (void*)limit ));
  }
#ifdef IC_PM
  else if (( nId == IStaticText::strikeoutId )      ||
           ( nId == IStaticText::underscoreId )     ||
           ( nId == IStaticText::fillBackgroundId ) ||
           ( nId == IStaticText::halftoneId ))
#endif
#ifdef IC_WIN
  else if (( nId == IStaticText::fillBackgroundId ) ||
           ( nId == IStaticText::halftoneId ))
#endif
  {
     this->notifyObservers( INotificationEvent( nId, *this, true,
                                                (void*)enabled ));
  }
}

/*------------------------------------------------------------------------------
| IStaticText::setText                                                         |
|                                                                              |
| Changes the text used by the static text control, and notifies the parent    |
| canvas.                                                                      |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::setText ( const char* pszText )
{
  ISize sizPreviousMinimum = this->minimumSize();

  /*********************************************************/
  /* Change the displayed text.                            */
  /*********************************************************/
  Inherited::setText( pszText );

#ifdef IC_PMWIN
  // Force IWindow::minimumSize to call calcMinimumSize.
  this->saveMinimumSize( ISize( 0, 0 ) );
#endif
  ISize sizNewMinimum = this->minimumSize();

  /*********************************************************/
  /* See if a significant change has occurred.             */
  /*********************************************************/
  if ( sizPreviousMinimum != sizNewMinimum )
  {           // Text change has caused a new minimum size.
     this->setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
              // Flag this for parent canvas.
#ifdef IC_PMWIN
     this->setLayoutDistorted( 0, IWindow::minimumSizeChanged );
              // Optimization to use just-calculated, cached min size.
#endif
  }

#ifdef IC_WIN
  // force repaint to avoid bogus text in upper left
  if (alignment() != topLeft)
     IINVALIDATERECT( handle(), 0, false );
#endif

  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::setText                                                         |
|                                                                              |
| Changes the text used by the static text control from a string table.        |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::setText ( const IResourceId & residText )
{
  Inherited::setText( residText );
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::setLimit                                                        |
------------------------------------------------------------------------------*/
IStaticText& IStaticText::setLimit ( unsigned long limit )
{
  /**********************************************************/
  /* Change the limit iff the new limit is different.       */
  /**********************************************************/
  if ( limit != this->ulClLimit )
  {                                    // Value change.
     this->ulClLimit = limit;

     /*******************************************************/
     /* Give canvas an opportunity to resize if a canvas is */
     /* the owner.                                          */
     /*******************************************************/
     this->setLayoutDistorted( IWindow::minimumSizeChanged |
                                 IWindow::immediateUpdate, 0 );
     this->notify( IStaticText::limitId, false );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IStaticText::calcLimitSize                                                   |
------------------------------------------------------------------------------*/
ISize IStaticText::calcLimitSize ( ) const
{
  ISize sizChar;

  /*********************************************************/
  /* Calculate iff the text limit has been specified.      */
  /*********************************************************/
  if ( this->ulClLimit )
  {
     sizChar = ((IStaticText*)this)->characterSize();

     /**************************************************************/
     /* The below block only exists because work-around logic in   */
     /* displaySize will cause its returned height to exceed the   */
     /* characterSize height (IFont::maxCharHeight value), and we  */
     /* want the tallest height that this font will yield.         */
     /**************************************************************/
     unsigned long ulBlankHeight = this->displaySize( " " ).height();
     if ( ulBlankHeight > sizChar.height() )
        sizChar.setHeight( ulBlankHeight );   // Have taller value.

     /**************************************************************/
     /* Use a step function to reflect the fact that you are more  */
     /* likely to exceed an average character width with an        */
     /* individual character, but a long string will likely be ok  */
     /* with this average value.                                   */
     /**************************************************************/
     sizChar *= ISize( this->ulClLimit, 1 );  // Convert avg chars to pels.
     IFont font(this);
     if ( !font.isFixed() )
     {
        // DEFECT 8842 : earlier algorithm gave uneven length increases;
        // simplify to a smooth curve of lengths over text limit range.
        if (this->ulClLimit <= 2)
           sizChar.setWidth( sizChar.width() * 3 );
        else if (this->ulClLimit <=16)
           sizChar.setWidth( sizChar.width() + 5 );
        else if (this->ulClLimit <=24)
           sizChar.setWidth( sizChar.width() + 4 );
        else if (this->ulClLimit <=32)
           sizChar.setWidth( sizChar.width() + 3 );
        else if (this->ulClLimit <=40)
           sizChar.setWidth( sizChar.width() + 2 );
        else if (this->ulClLimit <=48)
           sizChar.setWidth( sizChar.width() + 1 );
     }   // is proportional
  }

  return sizChar;
}
