/*******************************************************************************
* FILE NAME: imenuhdr.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imenuhdr.hpp                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
extern "C" {
   #define INCL_WINWINDOWMGR
   #define INCL_WININPUT
   #define INCL_WINPOINTERS
   #define INCL_WINFRAMEMGR
   #define INCL_WINSYS
   #include <iwindefs.h>
}

#include <imenuhdr.hpp>
#include <isubmenu.hpp>
#include <ipopmenu.hpp>
#include <itrace.hpp>
#include <iseq.h>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMENUHDR_CPP_
  #include <ipagetun.h>
#endif

// ISubmenu does not call startHandlingEvents()
// as part of its cstor. We do need to handle cascaded
// menus that will nest WM_INITMENU and WM_MENUENDED pairs.
// Will use a collection of SubmenuRec's to do this.

class SubmenuRec
{
public:
  SubmenuRec(unsigned long mnItem,
             IWindowHandle owner,
             ISubmenu* pSubmenu)
    : mnItemCl(mnItem), ownerCl(owner), pSubmenuCl(pSubmenu), nestedCnt(0) {}
  //--------------------------------------------------------------
  // DEFECT 7873 : memory leak resulted from use of temporary
  //     submenu tracked by this class; cleanup pointer on delete
  //--------------------------------------------------------------
  ~SubmenuRec() { delete pSubmenuCl; }
  //--------------------------------------------------------------
  unsigned long mnItemCl;
  IWindowHandle ownerCl;
  ISubmenu*     pSubmenuCl;
  unsigned long nestedCnt;
};

typedef ISequence<SubmenuRec*> SubmenuCol;
static SubmenuCol* pSubmenuCol = 0;

static Boolean
  IMenuDeleteElements (SubmenuRec* const& pMenuRec, void* evt)
{
  Boolean rc = false;
  IMenuEvent menuEvent(*((IEvent*)evt));

  if (pMenuRec->mnItemCl == menuEvent.menuItemId() &&
      pMenuRec->ownerCl == menuEvent.handle())
  {
    ISubmenu* pSubmenu = pMenuRec->pSubmenuCl;
    pSubmenu->undo();
    delete (SubmenuRec*)pMenuRec;
    rc = true;
  }
  return (rc);
}

// Define a static handler to delete popup menus when they disappear.
// This handler is only attached when the menu end message is received
// and autoDeleteObject is set.  This handler is needed to delay the
// deletion of the menu object until all of the menu handlers and/or PM
// have processed the menu end message (otherwise traps may occur).

class AutoDeleteHandler : public IHandler {

typedef IHandler
  Inherited;

public:

  AutoDeleteHandler  ( );
virtual
  ~AutoDeleteHandler ( );

protected:

Boolean
  dispatchHandlerEvent(IEvent& event);
};

/*------------------------------------------------------------------------------
| AutoDeleteHandler::AutoDeleteHandler                                         |
------------------------------------------------------------------------------*/
AutoDeleteHandler::AutoDeleteHandler( )
  : AutoDeleteHandler::Inherited( )
{ }

/*------------------------------------------------------------------------------
| AutoDeleteHandler::~AutoDeleteHandler                                        |
------------------------------------------------------------------------------*/
AutoDeleteHandler::~AutoDeleteHandler( )
{ }

/*------------------------------------------------------------------------------
| AutoDeleteHandler::dispatchHandlerEvent                                      |
------------------------------------------------------------------------------*/
Boolean AutoDeleteHandler::dispatchHandlerEvent(IEvent& event)
{
   if (event.eventId() == IC_UM_DESTROY_MENU)
   {
      stopHandlingEventsFor(event.window());
      // Use PM to destroy the window so that the handlers will
      // work correctly and the auto delete code in the dispatcher
      // is executed (when the WM_DESTROY msg is handled)
      IDESTROYWINDOW(event.window()->handle());
      return true;
   }

   return false;
}

static struct DefaultDeleteHandler {
  operator AutoDeleteHandler* ();
AutoDeleteHandler
 *operator -> ()
    {
    return *this;
    }
  ~DefaultDeleteHandler ( )
    {
    if (ptr)
      {
      delete ptr;
      ptr = 0;
      }
    }
AutoDeleteHandler
 *ptr;
} defaultDeleteHandler;

/*------------------------------------------------------------------------------
| DefaultDeleteHandler :: operator AutoDeleteHandler* ( )                      |
------------------------------------------------------------------------------*/
DefaultDeleteHandler :: operator AutoDeleteHandler* ( )
  {
  if ( !ptr )
    ptr = new AutoDeleteHandler;
  return ptr;
  }

/*------------------------------------------------------------------------------
| IMenuHandler::IMenuHandler                                                   |
------------------------------------------------------------------------------*/
IMenuHandler::IMenuHandler()
{
   if (pSubmenuCol == 0) {
      pSubmenuCol = new SubmenuCol(10);
   } /* endif */
}

/*------------------------------------------------------------------------------
| IMenuHandler::~IMenuHandler                                                  |
------------------------------------------------------------------------------*/
IMenuHandler::~IMenuHandler()
{
}

/*------------------------------------------------------------------------------
| IMenuHandler::dispatchHandlerEvent                                           |
------------------------------------------------------------------------------*/
Boolean IMenuHandler::dispatchHandlerEvent(IEvent& evt)
{
   switch (evt.eventId()) {
#ifndef IC_WIN
   case WM_TRANSLATEACCEL:
   {
        // Check for SV_CONTEXTMENUKB translation
        // Low word is the virtual key code (VK_*)
        // High word is the keyboard control code (KC_*)

        unsigned long
          svKeyPopUp = WinQuerySysValue(IWindow::desktopWindow()->handle(),
                                        SV_CONTEXTMENUKB);

        PQMSG pQmsg = (PQMSG) (unsigned long) evt.parameter1();

        // Because the keyboard does not really send the exact same
        // keyboard control codes, we need to do the checking in
        // several stages:
        //
        //  - bits 0 thru 2 must have the desired bits turned on, but
        //    may also have others (KC_CHAR, KC_VIRTUALKEY, KC_SCANCODE)
        //  - bits 3 thru 7 must match (KC_SHIFT, KC_CTRL, KC_ALT,
        //    KC_KEYUP, KC_KEYDOWN)
        //  - bits 8 thru 15 can be ignored
        //  - the virtual key (VK_*) values must match

        if (( SHORT1FROMMP(pQmsg->mp1) & SHORT2FROMMP(svKeyPopUp) & 0x07) &&
           (( SHORT1FROMMP(pQmsg->mp1) & 0xF8) ==
            ( SHORT2FROMMP(svKeyPopUp) & 0xF8) ) &&
            ( SHORT2FROMMP(pQmsg->mp2) == SHORT1FROMMP(svKeyPopUp)) )
        {
          // This is a popup menu request from the keyboard
          pQmsg->hwnd = evt.window()->handle();
          pQmsg->msg = WM_CONTEXTMENU;
          pQmsg->mp1 = 0;
          pQmsg->mp2 = (void *) 0x10000;
          evt.setResult(true);
          return true;
        }
   }
   break;
#endif
   case WM_CONTEXTMENU:
   {
      IMenuEvent mnEvt(evt);
      if (makePopUpMenu(mnEvt)) {
         addSourceEmphasis(mnEvt);
         evt.setResult(true);   //Tell PM action taken.
         return true;
      } /* endif */
      break;
   } /* endcase */

   case WM_INITMENU:
   {
      IMenuEvent mnEvt(evt);

      Boolean found = false;
      SubmenuCol::Cursor cursor(*pSubmenuCol);
      for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
      {
         SubmenuRec *pRec = pSubmenuCol->elementAt(cursor);
         if (pRec->mnItemCl == mnEvt.menuItemId())
         {
            // A submenu record already exists for this menu.  This
            // means it is a nested call
            pRec->nestedCnt++;
            found = true;
         }
      }

      if (!found)
      {
#ifdef IC_WIN
        ISubmenu* pSubmenu = new ISubmenu(IMenuHandle(evt.parameter1()));
#endif
#ifndef IC_WIN
        ISubmenu* pSubmenu = new ISubmenu(IWindowHandle(evt.parameter2()));
#endif
        if (menuShowing(mnEvt, *pSubmenu)) {
           // Add only if action taken
           pSubmenuCol->addAsFirst(new SubmenuRec(mnEvt.menuItemId(),
                                           mnEvt.handle(), pSubmenu));
           return true;
        } /* endif */
        delete pSubmenu; //No action taken - delete ISubmenu*
      }
      else
        return true;

      break;
   } /* endcase */

   case WM_MENUSELECT:
   {
#ifdef IC_WIN
      if (evt.parameter1().highNumber() != 0xFFFF)   //Don't process invalid
#endif
#ifndef IC_WIN
      if (evt.parameter1().lowNumber() != 0xFFFF)    //Don't process invalid
#endif
      {                                              //menu item ID
         IMenuEvent mnEvt(evt);
         mnEvt.setResult(true);             //default result to true to
         if (menuSelected(mnEvt)) {         //tell PM to post command
            evt.setResult(mnEvt.result());  //Apps can override with false.
            return true;
         } /* endif */
      } /* endif */
      break;
   }

   case WM_MENUEND:
   {
      IMenuEvent menuEvent(evt);

      Boolean nested = false;
      SubmenuCol::Cursor cursor(*pSubmenuCol);
      for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
      {
         SubmenuRec *pRec = pSubmenuCol->elementAt(cursor);
         if ((pRec->mnItemCl == menuEvent.menuItemId()) &&
             (pRec->nestedCnt > 0))
         {
            // This is a nested call (to match a nested initmenu)
            nested = true;
            pRec->nestedCnt--;
         }
      }

      if (!nested)
      {
        Boolean endReturn = (menuEnded(menuEvent));

        if (!pSubmenuCol->isEmpty())
        {
          pSubmenuCol->removeAll(&IMenuDeleteElements, &evt);
        }

        IWindow* popUpWindow =
                 IWindow::windowWithHandle(IWindowHandle(menuEvent.parameter2()));
        if (popUpWindow) {
           if (popUpWindow->isAutoDeleteObject())
           {
              ((AutoDeleteHandler*)defaultDeleteHandler)->
                                   handleEventsFor(popUpWindow);
              popUpWindow->postEvent(IC_UM_DESTROY_MENU);
           }
        } /* endif */
        return endReturn;
      }
      else
        return true;

   } /* endcase */

   } /* endswitch */

   return false;
}

/*------------------------------------------------------------------------------
| IMenuHandler::menuEnded                                                      |
------------------------------------------------------------------------------*/
Boolean IMenuHandler::menuEnded( IMenuEvent& menuEvent )
{
#ifndef IC_WIN   //IC_NOTYET - Don't know what this is supposed to do
   IWindow* popUpWindow =
            IWindow::windowWithHandle(IWindowHandle(menuEvent.parameter2()));
   if (popUpWindow && (popUpWindow->id() != FID_MENU)) {
      removeSourceEmphasis(menuEvent);
   } /* endif */
#endif
   return false;
}

/*------------------------------------------------------------------------------
| IMenuHandler::addSourceEmphasis                                              |
|  Call owner to show source emphasis                                          |
------------------------------------------------------------------------------*/
void IMenuHandler :: addSourceEmphasis(const IMenuEvent& menuEvent )
{
  menuEvent.window()->showSourceEmphasis();
}

/*------------------------------------------------------------------------------
| IMenuHandler::removeSourceEmphasis                                           |
|  Call owner to hide source emphasis                                          |
------------------------------------------------------------------------------*/
void IMenuHandler :: removeSourceEmphasis(const IMenuEvent& menuEvent)
{
  menuEvent.window()->hideSourceEmphasis();
}

/*------------------------------------------------------------------------------
   Do nothing virtual function implemenations.
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
| IMenuHandler::makePopUpMenu                                                  |
------------------------------------------------------------------------------*/
Boolean IMenuHandler::makePopUpMenu ( IMenuEvent& menuEvent )
{
  return false;
}

/*------------------------------------------------------------------------------
| IMenuHandler::menuShowing                                                    |
------------------------------------------------------------------------------*/
Boolean IMenuHandler::menuShowing   ( IMenuEvent& menuEvent,
                                      ISubmenu&   submenuAboutToShow )
{
  return false;
}

/*------------------------------------------------------------------------------
| IMenuHandler::menuSelected                                                   |
------------------------------------------------------------------------------*/
Boolean IMenuHandler:: menuSelected ( IMenuEvent& menuEvent )
{
  return false;
}

