/*******************************************************************************
* FILE NAME: icnrctl6.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the Object Manipulation functions *
*   for IContainerControl, declared in icnrctl.hpp.                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_WINSTDCNR
#define INCL_WINWINDOWMGR

#include <icnrrec.hpp>  // Must be first for OS flags


#include <itrace.hpp>
#include <irect.hpp>
#include <iexcept.hpp>
#include <icnrctl.hpp>
#include <icnrolst.hpp>
#include <iscroll.hpp>
#include <icoordsy.hpp>
#include <inotifev.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCTL6_CPP_
  #include <ipagetun.h>
#endif


ICnrObjectSet ::  ICnrObjectSet() {}
ICnrObjectSet ::  ICnrObjectSet(const ICnrObjectSet& objSet)
                    : ISequence<IContainerObject*>(objSet) {}
ICnrObjectSet :: ~ICnrObjectSet() {}
IContainerControl::Iterator :: ~Iterator    ( ) {}
IContainerControl::Iterator::Iterator() {}



/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: addObject

 Implementation: Add an object to the container.
------------------------------------------------------------------------------*/
IContainerControl&  IContainerControl :: addObject(const IContainerObject* newObject,
                                                         IContainerObject* pcnrobjParent)
{
   IMODTRACE_DEVELOP("CnrCtl::addObj");
   IASSERTPARM(newObject!=0);
   RECORDINSERT recordinsert;
   IMiniCnrRecord* pcnrrec = IRecFromObj(newObject);

   //Set up recordinsert structure
   recordinsert.cb = sizeof(RECORDINSERT);
   recordinsert.zOrder = (unsigned short)CMA_TOP;
   recordinsert.cRecordsInsert = (unsigned short)1;
   recordinsert.pRecordOrder = (PRECORDCORE)CMA_END;

   //If object is to be a child object, state the parent here.
   if(pcnrobjParent!=0)
   {
     recordinsert.pRecordParent = (PRECORDCORE)IRecFromObj(pcnrobjParent);
     pcnrobjParent->flClState |= IContainerObject::hasDescendents;

     // Note: the following is a patch to a WC_CONTAINER Bug
     // The bug was that a images were being displayed in icon view for
     // tree-view children.  Children objects should not be displayed in
     // non-tree views.
     pcnrrec->ptlIcon.x = -LONG_MAX;
     pcnrrec->ptlIcon.y = -LONG_MAX;
   }
   else
     recordinsert.pRecordParent = 0;

   //Refresh screen only if both container and object refresh states are on.
   if(newObject->isRefreshOn() && isRefreshOn() && isVisible() )
     recordinsert.fInvalidateRecord = true;
   else
     recordinsert.fInvalidateRecord = false;

   //Call OS/2 container to insert the container record.
   Boolean fSuccess = (Boolean)sendEvent(CM_INSERTRECORD,
                                         MPFROMP(pcnrrec),
                                         MPFROMP(&recordinsert));
   if(!fSuccess)
   {
     ITHROWGUIERROR("CM_INSERTRECORD");
   }

   notifyObservers(INotificationEvent(IContainerControl::addId,
                                      *this, true, (void*)1));
   ulClObjectChanges++;
   ((IContainerObject*)newObject)->incrementUseCount();
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: addObjectAfter

 Implementation: Add an object to the container after an object that is already
                 in the container.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: addObjectAfter(const IContainerObject* newObject,
                                                       const IContainerObject* pcnrobjAfter,
                                                             IContainerObject* pcnrobjParent)
{
   IMODTRACE_DEVELOP("CnrCtl::addObjAfter");
   IASSERTPARM(newObject!=0);
   RECORDINSERT recordinsert;
   IMiniCnrRecord* pcnrrec = IRecFromObj(newObject);

   //Set up recordinsert structure
   recordinsert.cb = sizeof(RECORDINSERT);
   recordinsert.zOrder = (unsigned short)CMA_TOP;
   recordinsert.cRecordsInsert = (unsigned short)1;

   //Place the new object after the given one. Default is the beginning
   if(pcnrobjAfter!=0)
     recordinsert.pRecordOrder = (PRECORDCORE)IRecFromObj(pcnrobjAfter);
   else
     recordinsert.pRecordOrder = (PRECORDCORE)CMA_FIRST;

   //If object is to be a child object, state the parent here.
   if(pcnrobjParent!=0)
   {
     recordinsert.pRecordParent = (PRECORDCORE)IRecFromObj(pcnrobjParent);
     pcnrobjParent->flClState |= IContainerObject::hasDescendents;

     // Note: the following is a patch to a WC_CONTAINER Bug
     // The bug was that a images were being displayed in icon view for
     // tree-view children.  Children objects should not be displayed in
     // non-tree views.
     pcnrrec->ptlIcon.x = -LONG_MAX;
     pcnrrec->ptlIcon.y = -LONG_MAX;
   }
   else
     recordinsert.pRecordParent = 0;

   //Refresh screen only if both container and object refresh states are on.
   if(newObject->isRefreshOn() && isRefreshOn() && isVisible() )
     recordinsert.fInvalidateRecord = true;
   else
     recordinsert.fInvalidateRecord = false;

   //Call OS/2 container to insert the container record.
   Boolean fSuccess = (Boolean)sendEvent(CM_INSERTRECORD,
                                         MPFROMP(pcnrrec),
                                         MPFROMP(&recordinsert));

   if(!fSuccess)
   {
      ITHROWGUIERROR("CM_INSERTRECORD");
   }

   notifyObservers(INotificationEvent(IContainerControl::addId,
                                      *this, true, (void*)1));
   ulClObjectChanges++;
   ((IContainerObject*)newObject)->incrementUseCount();
   return *this;
}



/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: removeObjectAt

 Implementation: Remove an object from the container using and updating a
 cursor.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: removeObjectAt( IContainerControl::ObjectCursor& cursor)
{
  IASSERTSTATE(cursor.isValid());
  IContainerObject* pcnrobjNewCurrent = 0;
  IContainerObject* pcnrobj = cursor.current();

  // If there is a next item in the container, we want to update the cursor
  // to point to it.
  cursor.setToNext();
  if(cursor.isValid())
  {
    pcnrobjNewCurrent = cursor.current();
  }
  removeObject(pcnrobj);
  // Note: remove will invalidate the cursor, setCurrent will revalidate
  if(pcnrobjNewCurrent)
    cursor.setCurrent(pcnrobjNewCurrent);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: removeObjectAt

 Implementation: Remove an object from the container using and updating a
 cursor.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: removeObjectAt( IContainerControl::TextCursor& cursor)
{
  IASSERTSTATE(cursor.isValid());
  IContainerObject* pcnrobjNewCurrent = 0;
  IContainerObject* pcnrobj = cursor.current();

  // If there is a next item in the container, we want to update the cursor
  // to point to it.
  cursor.setToNext();
  if(cursor.isValid())
  {
    pcnrobjNewCurrent = cursor.current();
  }
  removeObject(pcnrobj);
  // Note: remove will invalidate the cursor, setCurrent will revalidate
  if(pcnrobjNewCurrent)
    cursor.setCurrent(pcnrobjNewCurrent);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: removeSelectedObjects

 Implementation: Remove selected objects from the container.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: removeSelectedObjects()
{
   IMODTRACE_DEVELOP("CnrCtl::removeSelObj");
   IMiniCnrRecord* pcnrrec;
   IContainerObject* pcnrobj;
   Boolean fRefreshNeeded = false;

   do
   {
      pcnrrec = (IMiniCnrRecord*)(void*)sendEvent(CM_QUERYRECORDEMPHASIS,
                                           (IContainerObject*)CMA_FIRST,
                                           MPFROMLONG(CRA_SELECTED));
      if(pcnrrec!=0)
      {
         fRefreshNeeded = true;
         pcnrobj = IObjFromRec(pcnrrec);
         pcnrobj->setRefreshOff();
         removeObject(pcnrobj);

         /* Get out if in Single Selection Mode */
         if( isSingleSelection() || isTreeView() )
              break;
      }
 } while (pcnrrec != 0);

   if(fRefreshNeeded && isRefreshOn())
     refresh();

   return *this;

}



// Note: deleteAllObjects moved to icnrctl0 - trying to
//       reduce the instances of BB static functions


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: removeAllObjects

 Implementation: Remove all objects in the container.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: removeAllObjects()
{

  IMODTRACE_DEVELOP("CnrCtl::removeAllObj");

  /* decrement all the object's usecount*/
  IContainerControl::ObjectCursor  objCursor(*this);
  for (objCursor.setToFirst(); objCursor.isValid(); objCursor.setToNext())
  {
    ((IContainerObject*)(objectAt(objCursor)))->decrementUseCount();
  }

  /* remove all the objects, invalidate if refresh is on in the cnr  */
  unsigned short usInvalidate = 0;
  if(isRefreshOn())
    usInvalidate = CMA_INVALIDATE;

  removeRecords(0, 0, usInvalidate);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: containsObject

 Implementation: Determine if an object exists in the container.

------------------------------------------------------------------------------*/
IBase::Boolean IContainerControl :: containsObject(const IContainerObject* object) const
{
   IMODTRACE_DEVELOP("Cnr::containsObject");
   IASSERTPARM(object!=0);

   void* pbase= IRecFromObj(object);
   Boolean fSuccess = false;
   if(isValid())
      fSuccess = (Boolean)sendEvent(CM_QUERYRECORDINFO,
                                    MPFROMP(&pbase),
                                    MPFROMSHORT(1));

  return fSuccess;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: objectAt

 Implementation: Return the object at the (zero-based) index position.
------------------------------------------------------------------------------*/
IContainerObject* IContainerControl :: objectAt(unsigned long ulIndex) const
{
  IContainerControl::ObjectCursor cnrobjcur(*this);
  IContainerObject* pcnrobj;
  unsigned long ulCount=0;
  while((pcnrobj=cnrobjcur.next())!=0)
  {
    if(ulCount==ulIndex)
      return(pcnrobj);
    ulCount++;
  }
  return 0;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: objectAt

 Implementation: Determine the object at a cursor location.
------------------------------------------------------------------------------*/
IContainerObject* IContainerControl::objectAt(const IContainerControl::ObjectCursor& cursor) const
{
   IASSERTSTATE(cursor.isValid());
   return cursor.current();
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: objectAt

 Implementation: Determine the object at a cursor location.
------------------------------------------------------------------------------*/
IContainerObject* IContainerControl::objectAt(const IContainerControl::TextCursor& cursor) const
{
   IASSERTSTATE(cursor.isValid());
   return cursor.current();
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: parentObject

 Implementation: Query an objects parent.
------------------------------------------------------------------------------*/
IContainerObject* IContainerControl :: parentObject(const IContainerObject* childObject) const
{
  IASSERTPARM(childObject!=0);
  IMiniCnrRecord* pcnrrec = IRecFromObj(childObject);
  if (pcnrrec)
    pcnrrec = (IMiniCnrRecord*)(void*)sendEvent(CM_QUERYRECORD,
                                         MPFROMP(pcnrrec),
                                         MPFROM2SHORT((unsigned int)CMA_PARENT,
                                                      CMA_ITEMORDER));
  // DEFECT 22474
  // if we can't locate the specified object, then a parent is unavailable
  if ( pcnrrec==0 || (long)pcnrrec==-1L )
     return 0; 
  
  return IObjFromRec(pcnrrec);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: objectCount

 Implementation: Return the number of objects in the container.
------------------------------------------------------------------------------*/
unsigned long IContainerControl :: objectCount() const
{
   IContainerControl::ObjectCursor cursor(*this,IContainerObject::none, itemOrder);
   unsigned long ulCount = 0;
   for(cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
   {
      ulCount++;
   }
   return ulCount;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: descendentsOf

 Implementation: Return a Set of all descendents of an object.
------------------------------------------------------------------------------*/
ICnrObjectSet IContainerControl :: descendentsOf(
                                IContainerObject* parentObject) const
{
   IASSERTPARM(parentObject!=0);
   ICnrObjectSet cnrset;
   parentObject->pcnrobjClParent = 0;
   if(parentObject->flClState & IContainerObject::hasDescendents)
     collectDescendents(cnrset, parentObject);
   return cnrset;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: immediateDescendentsOf

 Implementation: Return a Set of the children (but not grandchildren) of an object.
------------------------------------------------------------------------------*/
ICnrObjectSet IContainerControl :: immediateDescendentsOf(
                                IContainerObject* parentObject) const
{
   IASSERTPARM(parentObject!=0);
   ICnrObjectSet cnrset;

   if(parentObject->flClState & IContainerObject::hasDescendents)
   {
      IContainerControl::ObjectCursor cursorChildren(*this, parentObject);
      IContainerObject* pcnrobjChild;

      while((pcnrobjChild = cursorChildren.next()) != 0)
      {
         cnrset.add(pcnrobjChild);
      }
   }
   return cnrset;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: objectList

 Implementation: Return a Set of all objects in the container.
------------------------------------------------------------------------------*/
ICnrObjectSet IContainerControl :: objectList() const
{
   ICnrObjectSet cnrset;

  IContainerControl::ObjectCursor iter(*this);
  IContainerObject* pcnrobj;

  while ((pcnrobj=iter.next())!=0)
  {
     cnrset.add(pcnrobj);
  }

  return cnrset;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: collectDescendents

 Implementation: Recursive function to collect all descendents.
------------------------------------------------------------------------------*/
void IContainerControl :: collectDescendents(ICnrObjectSet& cnrset,
                                        IContainerObject* pcnrobj) const
{
  IMODTRACE_DEVELOP("CnrCtl::collectDesc");
  IASSERT(pcnrobj!=0);
  IContainerControl::ObjectCursor cursorChildren(*this, pcnrobj);
  IContainerObject* pcnrobjChild;

  while((pcnrobjChild = cursorChildren.next()) != 0)
  {
     pcnrobjChild->pcnrobjClParent = pcnrobj;
     cnrset.add(pcnrobjChild);
     collectDescendents(cnrset, pcnrobjChild);
  }
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: objectUnderPoint

 Implementation: Determine the record under a point in window
   coordinates.
------------------------------------------------------------------------------*/
IContainerObject* IContainerControl :: objectUnderPoint(const IPoint& pt) const
{
  IMODTRACE_DEVELOP("CnrCtl :: objectUnderPoint");
  QUERYRECFROMRECT queryrect;
  queryrect.cb = sizeof(QUERYRECFROMRECT);

  IPoint nativePt = pt;
  if (ICoordinateSystem::applicationOrientation() ==
      ICoordinateSystem::originUpperLeft )
      {
      nativePt.setY( size().height() - pt.y() );
      }

  // develop a rect around point
  // Rectangle must have size, because we call CM_QUERYRECORDFROMRECT and
  // it calls WinIntersectRect, which requires a rectangle bigger than a point.
#ifdef IC_WIN
  queryrect.rect.left   = nativePt.x()-1;
  queryrect.rect.bottom = nativePt.y()-1;
  queryrect.rect.right  = nativePt.x();
  queryrect.rect.top    = nativePt.y();
#else
  queryrect.rect.xLeft   = nativePt.x()-1;
  queryrect.rect.yBottom = nativePt.y()-1;
  queryrect.rect.xRight  = nativePt.x();
  queryrect.rect.yTop    = nativePt.y();
#endif

  // The details view returns the object relative to the windows
  // on the container which are sitting above the horizontal
  // scrollbar.  So we will adjust by the scrollbar
  if(isDetailsView())
  {
    unsigned long cyHScroll = IScrollBar::systemScrollBarWidth(false);
#ifdef IC_WIN
    queryrect.rect.bottom -= cyHScroll;
    queryrect.rect.top    -= cyHScroll;
#else
    queryrect.rect.yBottom -= cyHScroll;
    queryrect.rect.yTop    -= cyHScroll;
#endif
  }
  queryrect.fsSearch = CMA_PARTIAL | CMA_ZORDER;
  IMiniCnrRecord* pcnrrec;

  pcnrrec = (IMiniCnrRecord*)(void*)sendEvent(CM_QUERYRECORDFROMRECT,
                                       (void*)CMA_FIRST,
                                       MPFROMP(&queryrect));
  if(((void*) pcnrrec) == (void*)-1)
  {
    ITHROWGUIERROR("CM_QUERYRECORDFROMRECT");
  }

  //
  // If it is iconview and not autoposition, we have to account for
  // the case that icons may be on top of each other.
  if (pcnrrec && isIconView() && (style() & CCS_AUTOPOSITION))
  {
    IMiniCnrRecord* pcnrnextrec=pcnrrec;
    while(pcnrnextrec)
    {
      pcnrrec = pcnrnextrec;
      pcnrnextrec = (IMiniCnrRecord*)(void*)sendEvent(CM_QUERYRECORDFROMRECT,
                                               (void*)pcnrnextrec,
                                               MPFROMP(&queryrect));
      if(((void*) pcnrnextrec) == (void*)-1)
      {
        ITHROWGUIERROR("CM_QUERYRECORDFROMRECT");
      }
    }
  }

  if(pcnrrec)
  {
    ITRACE_DEVELOP(IString("object is: ") + IObjFromRec(pcnrrec)->iconText());
    return (IObjFromRec(pcnrrec));
  }
  return 0;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isMoveValid

 Implementation:  Checks if an object and its descendents can be moved to a
                  new location.
------------------------------------------------------------------------------*/
IBase::Boolean IContainerControl :: isMoveValid(IContainerObject* moveObject,
                                         IContainerObject* pcnrobjNewParent,
                                         IContainerControl* pcnrctlNewContainer,
                                         IContainerObject* pcnrobjAfter )
{
  //Must have a valid object to move.  It must be in the source container.
  if ( (moveObject==0) || !containsObject(moveObject) )
    return false;

  IContainerControl* pcnrctlTarget;
  if(pcnrctlNewContainer==0)
     pcnrctlTarget = this;
  else
     pcnrctlTarget = pcnrctlNewContainer;

  //The target drop location must be valid.  The target object must exist
  //in the target container.  A non-zero afterObject and a non-zero newParent
  //must exist in the target container.
  //It must be different from moveObject.
  if ((pcnrobjAfter     && !pcnrctlTarget->containsObject(pcnrobjAfter)) ||
      (pcnrobjNewParent && !pcnrctlTarget->containsObject(pcnrobjNewParent)) ||
      (pcnrobjAfter == moveObject) )
    return false;

  //The afterObject must be a child of the NewParent if both are non-zero.
  if ( pcnrobjAfter && pcnrobjNewParent &&
       (pcnrctlTarget->parentObject(pcnrobjAfter) != pcnrobjNewParent) )
    return false;

  ICnrObjectSet cnrset;
  cnrset = descendentsOf(moveObject);
  IContainerObject* pcnrobj;

  if( pcnrctlTarget==this )
  {
    //Moving objects in the same container...
    //Make sure we are not moving a subtree to a child of our subtree
    ICnrObjectSet::Cursor cnrcur(cnrset);
    if( (pcnrobjNewParent==0) && (pcnrobjAfter!=0) )
      pcnrobjNewParent = parentObject(pcnrobjAfter);
    if(pcnrobjNewParent!=0)
    {
      for(cnrcur.setToFirst();
          cnrcur.isValid();
          cnrcur.setToNext())
      {
         pcnrobj=cnrset.elementAt(cnrcur);
         if(pcnrobj==pcnrobjNewParent)
           return false;
      }  /* endfor */
    } /* endif */
  }
  else
  {
    //Moving objects to another container, need to make sure that none of
    //the moving tree already exists in the target container.
    if( pcnrctlTarget->containsObject(moveObject) )
      return false;

    ICnrObjectSet::Cursor cnrcur(cnrset);
    for(cnrcur.setToFirst();
        cnrcur.isValid();
        cnrcur.setToNext())
    {
      pcnrobj=cnrset.elementAt(cnrcur);
      if( pcnrctlTarget->containsObject(pcnrobj) )
        return false;
    }  /* endfor */
  }

  return true;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: moveObjectTo

 Implementation: Effectively moves the leg of a tree to a new location.  Returns
  true if the action could be accomplished, returns false if the move would be
  recursive;  If NewParent is zero, the leg becomes a root leg.
------------------------------------------------------------------------------*/
IBase::Boolean IContainerControl :: moveObjectTo(IContainerObject* moveObject,
                                          IContainerObject* pcnrobjNewParent,
                                          IContainerControl* pcnrctlNewContainer,
                                          IContainerObject* pcnrobjAfter,
                                          const IPoint& ptIcon)
{
  IASSERTPARM(moveObject!=0);

  /* insure the move is a valid One - return false if not */
  if( !isMoveValid(moveObject, pcnrobjNewParent,
                   pcnrctlNewContainer, pcnrobjAfter) )
     return false;

  ICnrObjectSet cnrset;
  cnrset = descendentsOf(moveObject);
  IContainerObject* pcnrobj;
  IContainerControl* pcnrctlTarget;
  Boolean fRefreshThis = true;
  Boolean fRefreshTarget = true;
  if(pcnrctlNewContainer!=0)
  {
     pcnrctlTarget = pcnrctlNewContainer;
     if(!pcnrctlTarget->isRefreshOn())
        fRefreshTarget = false;
     else
        pcnrctlTarget->setRefreshOff();
  }
  else
  {
     pcnrctlTarget = this;
     fRefreshTarget = false;
  }

  if(!isRefreshOn())
     fRefreshThis = false;
  else
     setRefreshOff();

  // update the object for current container before removing it
  IMiniCnrRecord* pcnrrec = IRecFromObj(moveObject);
  sendEvent(CM_QUERYRECORDINFO, &pcnrrec, MPFROMSHORT(1));

  /* remove the object (removes the children as well) */
  removeObject(moveObject);

  /* Adjust the Icon position */
  pcnrrec->ptlIcon.x = ptIcon.x();
  pcnrrec->ptlIcon.y = ptIcon.y();

  // convert to application coordinates if needed
  if (ICoordinateSystem::applicationOrientation() ==
      ICoordinateSystem::originUpperLeft )
     {
     pcnrrec->ptlIcon.y = pcnrctlTarget->size().height() - ptIcon.y();
     }

  //Add moveObject to the target container.
  pcnrctlTarget->addObjectAfter(moveObject, pcnrobjAfter, pcnrobjNewParent );


  if(cnrset.numberOfElements()!=0)    /* add children if gathered   */
  {
    ICnrObjectSet::Cursor cnrcur(cnrset);
    for(cnrcur.setToFirst();
        cnrcur.isValid();
        cnrcur.setToNext())
    {
       pcnrobj=cnrset.elementAt(cnrcur);
       if(pcnrobj->pcnrobjClParent != 0)
           pcnrctlTarget->addObject(pcnrobj, pcnrobj->pcnrobjClParent);
       else
           pcnrctlTarget->addObject(pcnrobj, moveObject);
    }  /* end while descendents */
  } /* endif */

  /* Refresh if necessary */
  if (fRefreshThis)
  {
     refresh();
     setRefreshOn();
  }
  if (fRefreshTarget)
  {
     pcnrctlTarget->refresh();
     pcnrctlTarget->setRefreshOn();
  }
  return true;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: copyObjectTo

 Implementation: Effectively copies the leg of a tree to a new location.  Returns
  true if the action could be accomplished, returns false if the move would be
  recursive;  If NewParent is zero, the leg becomes a root leg.
------------------------------------------------------------------------------*/
IContainerObject* IContainerControl :: copyObjectTo(IContainerObject* copyObject,
                                          IContainerObject* pcnrobjParent,
                                          IContainerControl* pcnrctlNewContainer,
                                          IContainerObject* pcnrobjAfter,
                                          const IPoint& ptIcon)
{
  IASSERTPARM(copyObject!=0);

  /* Collect a set containing the offspring of this object */
  ICnrObjectSet cnrset;
  Boolean fRefreshThis = true;
  Boolean fRefreshTarget = true;
  cnrset = descendentsOf(copyObject);

  IContainerObject* pcnrobjCopied;
  IContainerControl* pcnrctlTarget;

  if(pcnrctlNewContainer!=0)
  {
     pcnrctlTarget = pcnrctlNewContainer;
     if(!pcnrctlTarget->isRefreshOn())
        fRefreshTarget = false;
     else
        pcnrctlTarget->setRefreshOff();
  }
  else
  {
     pcnrctlTarget = this;
     fRefreshTarget = false;
  }

  if(!isRefreshOn())
     fRefreshThis = false;
  else
     setRefreshOff();

  /* copy the object */
  IContainerObject* pcnrobjCopiedRoot = copyObject->objectCopy();

  /* Adjust the Icon position */
  IMiniCnrRecord* pcnrrec = IRecFromObj(pcnrobjCopiedRoot);
  pcnrrec->ptlIcon.x = ptIcon.x();
  pcnrrec->ptlIcon.y = ptIcon.y();
  // convert to application coordinates if needed
  if (ICoordinateSystem::applicationOrientation() ==
      ICoordinateSystem::originUpperLeft )
     {
     pcnrrec->ptlIcon.y = pcnrctlTarget->size().height() - ptIcon.y();
     }

  /* Add to it's new parent */
  if(pcnrobjParent!=0)
    pcnrctlTarget->addObject(pcnrobjCopiedRoot, pcnrobjParent);
  else
    pcnrctlTarget->addObjectAfter(pcnrobjCopiedRoot, pcnrobjAfter);

  if(cnrset.numberOfElements()!=0)    /* add children if gathered   */
  {
    /* Now construct another set with copies of all objects  */
    ICnrObjectSet cnrsetCopies;
    ICnrObjectSet::Cursor cnrcur(cnrset);
    ICnrObjectSet::Cursor cnrcurCopies(cnrsetCopies);
    IContainerObject* pcnrobj;
    for(cnrcur.setToFirst();
        cnrcur.isValid();
        cnrcur.setToNext())
    {
       pcnrobj=cnrset.elementAt(cnrcur);
       pcnrobjCopied = pcnrobj->objectCopy();
       cnrsetCopies.add(pcnrobjCopied);
    }

    long lIndexCurrent =1;
    /* Go through all objects in the source list  */
    for(cnrcur.setToFirst(); cnrcur.isValid(); cnrcur.setToNext())
    {
       pcnrobj = cnrset.elementAt(cnrcur);
       if(pcnrobj->pcnrobjClParent != copyObject)
       {
          /* Find index of Parent  */
          long lIndexParent;
          for(lIndexParent=1; lIndexParent<=cnrset.numberOfElements(); lIndexParent++)
          {
             if(pcnrobj->pcnrobjClParent == cnrset.elementAtPosition(lIndexParent))
             {
                /* found the index of the parent */
                pcnrctlTarget->addObject(
                                 cnrsetCopies.elementAtPosition(lIndexCurrent),
                                 cnrsetCopies.elementAtPosition(lIndexParent));
             }
          }  /* end for all parents  */
       }
       else  /* Add to the Copied Root Object  */
       {
           pcnrctlTarget->addObject(cnrsetCopies.elementAtPosition(lIndexCurrent),
                                      pcnrobjCopiedRoot);
       }
       lIndexCurrent++;
    } /* endfor */
  } /* endif numberOfElements>0 */

  /* Refresh if necessary */
  if (fRefreshThis)
  {
     refresh();
     setRefreshOn();
  }
  if (fRefreshTarget)
  {
     pcnrctlTarget->refresh();
     pcnrctlTarget->setRefreshOn();
  }

  return pcnrobjCopiedRoot;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl::ObjectCursor::allObjectsDo

 Implementation: Apply behavior to all objects in the container.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::allObjectsDo( IContainerControl::Iterator& iterator,
                                                    Boolean fIncludeDescendents)
{
  IContainerObject* pcnrobj;

  IContainerControl::ObjectCursor cursor(*this, 0, fIncludeDescendents);
  for(cursor.setToFirst();
      cursor.isValid();
      cursor.setToNext())
  {
     pcnrobj=this->objectAt(cursor);
     if (! iterator.applyTo(pcnrobj) )
        break;
  }

  return *this;
}

#ifdef IC_PMWIN
/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: addObjects

 Implementation: Add multiple objects to the container.  The objects to add
 are determined by the number of initialized objects in the provided
 ICnrAllocator.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::addObjects(ICnrAllocator& allocator,
                                          IContainerObject* pcnrobjParent)
{
  IMODTRACE_DEVELOP("CnrCtl::addObjects");
  RECORDINSERT recordinsert;
  Boolean isChild = false;

  //Set up recordinsert structure
  recordinsert.cb = sizeof(RECORDINSERT);
  recordinsert.zOrder = (unsigned short)CMA_TOP;
  recordinsert.cRecordsInsert = (unsigned short)allocator.initialized();
  recordinsert.pRecordOrder = (PRECORDCORE)CMA_END;

  //If object is to be a child object, state the parent here.
  if (pcnrobjParent != 0)
  {
    recordinsert.pRecordParent = (PRECORDCORE)IRecFromObj(pcnrobjParent);
    pcnrobjParent->flClState |= IContainerObject::hasDescendents;

    // Note: the following is a patch to a WC_CONTAINER Bug
    // The bug was that images were being displayed in icon view for
    // tree-view children.  Children objects should not be displayed in
    // non-tree views.
    isChild = true;
  }
  else
  {
    recordinsert.pRecordParent = 0;
  }

  if (isRefreshOn() && isVisible())
  {
    recordinsert.fInvalidateRecord = true;
  }
  else
  {
    recordinsert.fInvalidateRecord = false;
  }

  /*************************************************************/
  /* Increment each object's use count prior to inserting them */
  /* into the container.  Also, set ptlIcon if the isChild     */
  /* flag is set.                                              */
  /*************************************************************/
  void* pcnrRec = allocator.first();
  unsigned long count = allocator.initialized();
  for (int i = 0; i < count; i++)
  {
    if (isChild)
    {
      ((IMiniCnrRecord*)(void*)pcnrRec)->ptlIcon.x = -LONG_MAX;
      ((IMiniCnrRecord*)(void*)pcnrRec)->ptlIcon.y = -LONG_MAX;
    }
    (IObjFromRec((IMiniCnrRecord*)(void*)pcnrRec))->incrementUseCount();
    pcnrRec = allocator.next(pcnrRec);
  }

  //Call OS/2 container to insert the container records.
  Boolean fSuccess = (Boolean)sendEvent(CM_INSERTRECORD,
                                        MPFROMP(allocator.first()),
                                        MPFROMP(&recordinsert));
  if (!fSuccess)
  {
    ITHROWGUIERROR("CM_INSERTRECORD");
  }

  notifyObservers(INotificationEvent(IContainerControl::addId,
                                     *this, true,
                                     (void*)recordinsert.cRecordsInsert));
  ulClObjectChanges += allocator.initialized();
  allocator.updateForInsert();
  return *this;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: addObjectsAfter

 Implementation:  Add multiple objects to the container after an object
 that is already in the container.  The objects to add are determined by
 the number of initialized objects in the provided ICnrAllocator.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::addObjectsAfter(ICnrAllocator& allocator,
                                         const IContainerObject* pcnrobjAfter,
                                         IContainerObject* pcnrobjParent)
{
  IMODTRACE_DEVELOP("CnrCtl::addObjectsAfter");
  RECORDINSERT recordinsert;
  Boolean isChild = false;

  //Set up recordinsert structure
  recordinsert.cb = sizeof(RECORDINSERT);
  recordinsert.zOrder = (unsigned short)CMA_TOP;
  recordinsert.cRecordsInsert = (unsigned short)allocator.initialized();

  //Place the new object after the given one. Default is the beginning
  if (pcnrobjAfter != 0)
  {
    recordinsert.pRecordOrder = (PRECORDCORE)IRecFromObj(pcnrobjAfter);
  }
  else
  {
    recordinsert.pRecordOrder = (PRECORDCORE)CMA_FIRST;
  }

  //If object is to be a child object, state the parent here.
  if (pcnrobjParent != 0)
  {
    recordinsert.pRecordParent = (PRECORDCORE)IRecFromObj(pcnrobjParent);
    pcnrobjParent->flClState |= IContainerObject::hasDescendents;

    // Note: the following is a patch to a WC_CONTAINER Bug
    // The bug was that a images were being displayed in icon view for
    // tree-view children.  Children objects should not be displayed in
    // non-tree views.
    isChild = true;
  }
  else
  {
    recordinsert.pRecordParent = 0;
  }

  //Refresh screen only if both container and object refresh states are on.
  if (isRefreshOn() && isVisible())
  {
    recordinsert.fInvalidateRecord = true;
  }
  else
  {
    recordinsert.fInvalidateRecord = false;
  }

  /*************************************************************/
  /* Increment each object's use count prior to inserting them */
  /* into the container.  Also, set ptlIcon if the isChild     */
  /* flag is set.                                              */
  /*************************************************************/
  void* pcnrRec = allocator.first();
  unsigned long count = allocator.initialized();
  for (int i = 0; i < count; i++)
  {
    if (isChild)
    {
      ((IMiniCnrRecord*)(void*)pcnrRec)->ptlIcon.x = -LONG_MAX;
      ((IMiniCnrRecord*)(void*)pcnrRec)->ptlIcon.y = -LONG_MAX;
    }
    (IObjFromRec((IMiniCnrRecord*)(void*)pcnrRec))->incrementUseCount();
    pcnrRec = allocator.next(pcnrRec);
  }

  //Call OS/2 container to insert the container records.
  Boolean fSuccess = (Boolean)sendEvent(CM_INSERTRECORD,
                                        MPFROMP(allocator.first()),
                                        MPFROMP(&recordinsert));

  if (!fSuccess)
  {
    ITHROWGUIERROR("CM_INSERTRECORD");
  }

  notifyObservers(INotificationEvent(IContainerControl::addId,
                                     *this, true,
                                     (void*)recordinsert.cRecordsInsert));
  ulClObjectChanges += allocator.initialized();
  allocator.updateForInsert();
  return *this;
}
#endif // IC_PMWIN
