(*******************************************************************
 *
 *  TTMemory.Pas                                             1.1
 *
 *    Memory management component (specification)
 *
 *  Copyright 1996 David Turner, Robert Wilhelm and Werner Lemberg
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  NOTE:
 *
 *    The FreeType library uses a simple and fast growing heap
 *    known as a 'Font Pool' to manage all its data. This version
 *    does not manage multiple pools ( a feature that may reveal
 *    useful for a FreeType based font server to be able to manage
 *    several open fonts ).
 *
 *  Differences between 1.0 and 1.1 :
 *
 *   - Added 'Mem_Allocated' and 'Mem_Free' functions
 *
 ******************************************************************)

unit TTMemory;

interface

type
  TMarkRecord = record
                  Magic : longint;
                  Top   : integer;
                end;

const
  Font_Pool_Allocated : boolean = False;

  procedure Init_FontPool ( var Buff; size : longint );
  { Initialize the font pool with the memory block 'buff' of }
  { 'size' bytes                                             }

  function  Alloc  ( Size : longint; var P: Pointer ): boolean;
  { Allocate 'size' bytes from the font pool, and place the  }
  { address of the fresh block in 'P'. Return FALSE on       }
  { failure ( no storage space left ), or TRUE on success    }

  procedure Mark   ( var M : TMarkRecord );
  { Mark the font pool's current state into the mark record 'M' }

  function  Release( var M : TMarkRecord ) : boolean;
  { Recover a previous marked state. Return FALSE on failure }
  { NOTE : This call destroys the content of M !!            }

  function Mem_Allocated : Longint;
  { Return amount of allocated memory }

  function Mem_Free : Longint;
  { Return amount of free memory }

implementation

const
  Mark_Magic = $BABE0007;


type
  TByte = array[0..0] of Byte;
  PByte = ^TByte;

var
  Buffer   : PByte;
  SizeBuff : longint;
  CurBuff  : longint;

function Mem_Allocated : Longint;
begin
  Mem_Allocated := CurBuff;
end;

function Mem_Free : LongInt;
begin
  Mem_Free := sizeBuff - curBuff;
end;

procedure Init_FontPool;
begin
  Buffer   := PByte(@Buff);
  SizeBuff := Size;
  CurBuff  := 0;

  Font_Pool_Allocated := True;
end;


function Alloc( Size : longint; var P: Pointer ): boolean;
var
  L : longint;
begin
  P     := nil;
  Alloc := False;
  L     := CurBuff + ( Size+3 ) and -4;

  if L >= SizeBuff then Alloc:=False
   else
    begin
     P       := @Buffer^[CurBuff];
     CurBuff := L;
     Alloc   := True;
    end
end;

{**********}
{*  Mark  *}
{*        ***********************************************}
{*                                                      *}
{*  Gets the current heap top position and put it       *}
{*  into a mark record.                                 *}
{*                                                      *}
{********************************************************}

procedure Mark( var M : TMarkRecord );
begin
  M.Magic := Mark_Magic;
  M.Top   := CurBuff;
end;

{*************}
{*  Release  *}
{*           ********************************************}
{*                                                      *}
{*  Sets the heap top to a previously saved position    *}
{*  with 'Mark'. Returns FALSE is the mark is invalid   *}
{*  or has been already released.                       *}
{*                                                      *}
{********************************************************}

function Release( var M : TMarkRecord ) : boolean;
begin
  if M.Magic = Mark_Magic then
    begin
      M.Magic := 0;
      CurBuff := M.Top;
      M.Top   := -1;
      Release := True;
    end
  else
    Release := False;
end;


end.
